/* SPDX-License-Identifier: (GPL-2.0 OR MIT) */
/*
 * Driver for Microsemi VSC85xx PHYs
 *
 * Copyright (c) 2020 Microsemi Corporation
 */

#ifndef _MSCC_PHY_PTP_H_
#define _MSCC_PHY_PTP_H_

/* 1588 page Registers */
#define MSCC_PHY_TS_BIU_ADDR_CNTL	  16
#define BIU_ADDR_EXE			  0x8000
#define BIU_ADDR_READ			  0x4000
#define BIU_ADDR_WRITE			  0x0000
#define BIU_BLK_ID(x)			  ((x) << 11)
#define BIU_CSR_ADDR(x)			  (x)
#define BIU_ADDR_CNT_MAX		  8

#define MSCC_PHY_TS_CSR_DATA_LSB	  17
#define MSCC_PHY_TS_CSR_DATA_MSB	  18

#define MSCC_PHY_1588_INGR_VSC85XX_INT_STATUS  0x002d
#define MSCC_PHY_1588_VSC85XX_INT_STATUS  0x004d
#define VSC85XX_1588_INT_FIFO_ADD	  0x0004
#define VSC85XX_1588_INT_FIFO_OVERFLOW	  0x0001

#define MSCC_PHY_1588_INGR_VSC85XX_INT_MASK	  0x002e
#define MSCC_PHY_1588_VSC85XX_INT_MASK	  0x004e
#define VSC85XX_1588_INT_MASK_MASK	  (VSC85XX_1588_INT_FIFO_ADD | \
					   VSC85XX_1588_INT_FIFO_OVERFLOW)

/* TS CSR addresses */
#define MSCC_PHY_ANA_ETH1_NTX_PROT	  0x0000
#define ANA_ETH1_NTX_PROT_SIG_OFF_MASK	  GENMASK(20, 16)
#define ANA_ETH1_NTX_PROT_SIG_OFF(x)	  (((x) << 16) & ANA_ETH1_NTX_PROT_SIG_OFF_MASK)
#define ANA_ETH1_NTX_PROT_COMPARATOR_MASK GENMASK(2, 0)
#define ANA_ETH1_NTX_PROT_PTP_OAM	  0x0005
#define ANA_ETH1_NTX_PROT_MPLS		  0x0004
#define ANA_ETH1_NTX_PROT_IP_UDP_ACH_2	  0x0003
#define ANA_ETH1_NTX_PROT_IP_UDP_ACH_1	  0x0002
#define ANA_ETH1_NTX_PROT_ETH2		  0x0001

#define MSCC_PHY_PTP_IFACE_CTRL		  0x0000
#define PTP_IFACE_CTRL_CLK_ENA		  0x0040
#define PTP_IFACE_CTRL_INGR_BYPASS	  0x0008
#define PTP_IFACE_CTRL_EGR_BYPASS	  0x0004
#define PTP_IFACE_CTRL_MII_PROT		  0x0003
#define PTP_IFACE_CTRL_GMII_PROT	  0x0002
#define PTP_IFACE_CTRL_XGMII_64_PROT	  0x0000

#define MSCC_PHY_ANA_ETH1_NTX_PROT_VLAN_TPID	0x0001
#define ANA_ETH1_NTX_PROT_VLAN_TPID_MASK  GENMASK(31, 16)
#define ANA_ETH1_NTX_PROT_VLAN_TPID(x)	  (((x) << 16) & ANA_ETH1_NTX_PROT_VLAN_TPID_MASK)

#define MSCC_PHY_PTP_ANALYZER_MODE	  0x0001
#define PTP_ANA_SPLIT_ENCAP_FLOW	  0x1000000
#define PTP_ANA_EGR_ENCAP_FLOW_MODE_MASK  GENMASK(22, 20)
#define PTP_ANA_EGR_ENCAP_FLOW_MODE(x)	  (((x) << 20) & PTP_ANA_EGR_ENCAP_FLOW_MODE_MASK)
#define PTP_ANA_INGR_ENCAP_FLOW_MODE_MASK GENMASK(18, 16)
#define PTP_ANA_INGR_ENCAP_FLOW_MODE(x)	  (((x) << 16) & PTP_ANA_INGR_ENCAP_FLOW_MODE_MASK)
#define PTP_ANALYZER_MODE_EGR_ENA_MASK	  GENMASK(6, 4)
#define PTP_ANALYZER_MODE_EGR_ENA(x)	  (((x) << 4) & PTP_ANALYZER_MODE_EGR_ENA_MASK)
#define PTP_ANALYZER_MODE_INGR_ENA_MASK	  GENMASK(2, 0)
#define PTP_ANALYZER_MODE_INGR_ENA(x)	  ((x) & PTP_ANALYZER_MODE_INGR_ENA_MASK)

#define MSCC_PHY_ANA_ETH1_NXT_PROT_TAG	  0x0002
#define ANA_ETH1_NXT_PROT_TAG_ENA	  0x0001

#define MSCC_PHY_PTP_MODE_CTRL		  0x0002
#define PTP_MODE_CTRL_MODE_MASK		  GENMASK(2, 0)
#define PTP_MODE_CTRL_PKT_MODE		  0x0004

#define MSCC_PHY_ANA_ETH1_NXT_PROT_ETYPE_MATCH	0x0003
#define ANA_ETH1_NXT_PROT_ETYPE_MATCH_ENA 0x10000
#define ANA_ETH1_NXT_PROT_ETYPE_MATCH_MASK	GENMASK(15, 0)
#define ANA_ETH1_NXT_PROT_ETYPE_MATCH(x)  ((x) & ANA_ETH1_NXT_PROT_ETYPE_MATCH_MASK)

#define MSCC_PHY_PTP_VERSION_CODE	  0x0003
#define PTP_IP_VERSION_MASK		  GENMASK(7, 0)
#define PTP_IP_VERSION_2_1		  0x0021

#define MSCC_ANA_ETH1_FLOW_ENA(x)	  (0x0010 + ((x) << 4))
#define ETH1_FLOW_ENA_CHANNEL_MASK_MASK	  GENMASK(9, 8)
#define ETH1_FLOW_ENA_CHANNEL_MASK(x)	  (((x) << 8) & ETH1_FLOW_ENA_CHANNEL_MASK_MASK)
#define ETH1_FLOW_VALID_CH1	  ETH1_FLOW_ENA_CHANNEL_MASK(2)
#define ETH1_FLOW_VALID_CH0	  ETH1_FLOW_ENA_CHANNEL_MASK(1)
#define ETH1_FLOW_ENA			  0x0001

#define MSCC_ANA_ETH1_FLOW_MATCH_MODE(x)  (MSCC_ANA_ETH1_FLOW_ENA(x) + 1)
#define ANA_ETH1_FLOW_MATCH_VLAN_TAG_MASK GENMASK(7, 6)
#define ANA_ETH1_FLOW_MATCH_VLAN_TAG(x)	  (((x) << 6) & ANA_ETH1_FLOW_MATCH_VLAN_TAG_MASK)
#define ANA_ETH1_FLOW_MATCH_VLAN_TAG2	  0x0200
#define ANA_ETH1_FLOW_MATCH_VLAN_VERIFY	  0x0010

#define MSCC_ANA_ETH1_FLOW_ADDR_MATCH1(x) (MSCC_ANA_ETH1_FLOW_ENA(x) + 2)

#define MSCC_ANA_ETH1_FLOW_ADDR_MATCH2(x) (MSCC_ANA_ETH1_FLOW_ENA(x) + 3)
#define ANA_ETH1_FLOW_ADDR_MATCH2_MASK_MASK	GENMASK(22, 20)
#define ANA_ETH1_FLOW_ADDR_MATCH2_ANY_MULTICAST	0x400000
#define ANA_ETH1_FLOW_ADDR_MATCH2_ANY_UNICAST	0x200000
#define ANA_ETH1_FLOW_ADDR_MATCH2_FULL_ADDR	0x100000
#define ANA_ETH1_FLOW_ADDR_MATCH2_SRC_DEST_MASK	GENMASK(17, 16)
#define ANA_ETH1_FLOW_ADDR_MATCH2_SRC_DEST	0x020000
#define ANA_ETH1_FLOW_ADDR_MATCH2_SRC	  0x010000
#define ANA_ETH1_FLOW_ADDR_MATCH2_DEST	  0x000000

#define MSCC_ANA_ETH1_FLOW_VLAN_RANGE_I_TAG(x)	(MSCC_ANA_ETH1_FLOW_ENA(x) + 4)
#define MSCC_ANA_ETH1_FLOW_VLAN_TAG1(x)	  (MSCC_ANA_ETH1_FLOW_ENA(x) + 5)
#define MSCC_ANA_ETH1_FLOW_VLAN_TAG2_I_TAG(x)	(MSCC_ANA_ETH1_FLOW_ENA(x) + 6)

#define MSCC_PHY_PTP_LTC_CTRL		  0x0010
#define PTP_LTC_CTRL_CLK_SEL_MASK	  GENMASK(14, 12)
#define PTP_LTC_CTRL_CLK_SEL(x)		  (((x) << 12) & PTP_LTC_CTRL_CLK_SEL_MASK)
#define PTP_LTC_CTRL_CLK_SEL_INTERNAL_250 PTP_LTC_CTRL_CLK_SEL(5)
#define PTP_LTC_CTRL_AUTO_ADJ_UPDATE	  0x0010
#define PTP_LTC_CTRL_ADD_SUB_1NS_REQ	  0x0008
#define PTP_LTC_CTRL_ADD_1NS		  0x0004
#define PTP_LTC_CTRL_SAVE_ENA		  0x0002
#define PTP_LTC_CTRL_LOAD_ENA		  0x0001

#define MSCC_PHY_PTP_LTC_LOAD_SEC_MSB	  0x0011
#define PTP_LTC_LOAD_SEC_MSB(x)		  (((x) & GENMASK_ULL(47, 32)) >> 32)

#define MSCC_PHY_PTP_LTC_LOAD_SEC_LSB	  0x0012
#define PTP_LTC_LOAD_SEC_LSB(x)		  ((x) & GENMASK(31, 0))

#define MSCC_PHY_PTP_LTC_LOAD_NS	  0x0013
#define PTP_LTC_LOAD_NS(x)		  ((x) & GENMASK(31, 0))

#define MSCC_PHY_PTP_LTC_SAVED_SEC_MSB	  0x0014
#define MSCC_PHY_PTP_LTC_SAVED_SEC_LSB	  0x0015
#define MSCC_PHY_PTP_LTC_SAVED_NS	  0x0016

#define MSCC_PHY_PTP_LTC_SEQUENCE	  0x0017
#define PTP_LTC_SEQUENCE_A_MASK		  GENMASK(3, 0)
#define PTP_LTC_SEQUENCE_A(x)		  ((x) & PTP_LTC_SEQUENCE_A_MASK)

#define MSCC_PHY_PTP_LTC_SEQ		  0x0018
#define PTP_LTC_SEQ_ADD_SUB		  0x80000
#define PTP_LTC_SEQ_ERR_MASK		  GENMASK(18, 0)
#define PTP_LTC_SEQ_ERR(x)		  ((x) & PTP_LTC_SEQ_ERR_MASK)

#define MSCC_PHY_PTP_LTC_AUTO_ADJ	  0x001a
#define PTP_AUTO_ADJ_NS_ROLLOVER(x)	  ((x) & GENMASK(29, 0))
#define PTP_AUTO_ADJ_ADD_SUB_1NS_MASK	  GENMASK(31, 30)
#define PTP_AUTO_ADJ_SUB_1NS		  0x80000000
#define PTP_AUTO_ADJ_ADD_1NS		  0x40000000

#define MSCC_PHY_PTP_LTC_1PPS_WIDTH_ADJ	  0x001b
#define PTP_LTC_1PPS_WIDTH_ADJ_MASK	  GENMASK(29, 0)

#define MSCC_PHY_PTP_TSTAMP_FIFO_SI	  0x0020
#define PTP_TSTAMP_FIFO_SI_EN		  0x0001

#define MSCC_PHY_PTP_INGR_PREDICTOR	  0x0022
#define PTP_INGR_PREDICTOR_EN		  0x0001

#define MSCC_PHY_PTP_EGR_PREDICTOR	  0x0026
#define PTP_EGR_PREDICTOR_EN		  0x0001

#define MSCC_PHY_PTP_INGR_TSP_CTRL	  0x0035
#define PHY_PTP_INGR_TSP_CTRL_FRACT_NS	  0x0004
#define PHY_PTP_INGR_TSP_CTRL_LOAD_DELAYS 0x0001

#define MSCC_PHY_PTP_INGR_LOCAL_LATENCY	  0x0037
#define PTP_INGR_LOCAL_LATENCY_MASK	  GENMASK(22, 0)
#define PTP_INGR_LOCAL_LATENCY(x)	  ((x) & PTP_INGR_LOCAL_LATENCY_MASK)

#define MSCC_PHY_PTP_INGR_DELAY_FIFO	  0x003a
#define PTP_INGR_DELAY_FIFO_DEPTH_MACSEC  0x0013
#define PTP_INGR_DELAY_FIFO_DEPTH_DEFAULT 0x000f

#define MSCC_PHY_PTP_INGR_TS_FIFO(x)	  (0x005c + (x))
#define PTP_INGR_TS_FIFO_EMPTY		  0x80000000

#define MSCC_PHY_PTP_INGR_REWRITER_CTRL	  0x0044
#define PTP_INGR_REWRITER_REDUCE_PREAMBLE 0x0010
#define PTP_INGR_REWRITER_FLAG_VAL	  0x0008
#define PTP_INGR_REWRITER_FLAG_BIT_OFF_M  GENMASK(2, 0)
#define PTP_INGR_REWRITER_FLAG_BIT_OFF(x) ((x) & PTP_INGR_REWRITER_FLAG_BIT_OFF_M)

#define MSCC_PHY_PTP_EGR_STALL_LATENCY	  0x004f

#define MSCC_PHY_PTP_EGR_TSP_CTRL	  0x0055
#define PHY_PTP_EGR_TSP_CTRL_FRACT_NS	  0x0004
#define PHY_PTP_EGR_TSP_CTRL_LOAD_DELAYS  0x0001

#define MSCC_PHY_PTP_EGR_LOCAL_LATENCY	  0x0057
#define PTP_EGR_LOCAL_LATENCY_MASK	  GENMASK(22, 0)
#define PTP_EGR_LOCAL_LATENCY(x)	  ((x) & PTP_EGR_LOCAL_LATENCY_MASK)

#define MSCC_PHY_PTP_EGR_DELAY_FIFO	  0x005a
#define PTP_EGR_DELAY_FIFO_DEPTH_MACSEC	  0x0013
#define PTP_EGR_DELAY_FIFO_DEPTH_DEFAULT  0x000f

#define MSCC_PHY_PTP_EGR_TS_FIFO_CTRL	  0x005b
#define PTP_EGR_TS_FIFO_RESET		  0x10000
#define PTP_EGR_FIFO_LEVEL_LAST_READ_MASK GENMASK(15, 12)
#define PTP_EGR_FIFO_LEVEL_LAST_READ(x)	  (((x) & PTP_EGR_FIFO_LEVEL_LAST_READ_MASK) >> 12)
#define PTP_EGR_TS_FIFO_THRESH_MASK	  GENMASK(11, 8)
#define PTP_EGR_TS_FIFO_THRESH(x)	  (((x) << 8) & PTP_EGR_TS_FIFO_THRESH_MASK)
#define PTP_EGR_TS_FIFO_SIG_BYTES_MASK	  GENMASK(4, 0)
#define PTP_EGR_TS_FIFO_SIG_BYTES(x)	  ((x) & PTP_EGR_TS_FIFO_SIG_BYTES_MASK)

#define MSCC_PHY_PTP_EGR_TS_FIFO(x)	  (0x005c + (x))
#define PTP_EGR_TS_FIFO_EMPTY		  0x80000000
#define PTP_EGR_TS_FIFO_0_MASK		  GENMASK(15, 0)

#define MSCC_PHY_PTP_EGR_REWRITER_CTRL	  0x0064
#define PTP_EGR_REWRITER_REDUCE_PREAMBLE  0x0010
#define PTP_EGR_REWRITER_FLAG_VAL	  0x0008
#define PTP_EGR_REWRITER_FLAG_BIT_OFF_M   GENMASK(2, 0)
#define PTP_EGR_REWRITER_FLAG_BIT_OFF(x)  ((x) & PTP_EGR_REWRITER_FLAG_BIT_OFF_M)

#define MSCC_PHY_PTP_SERIAL_TOD_IFACE	  0x006e
#define PTP_SERIAL_TOD_IFACE_LS_AUTO_CLR  0x0004

#define MSCC_PHY_PTP_LTC_OFFSET		  0x0070
#define PTP_LTC_OFFSET_ADJ		  BIT(31)
#define PTP_LTC_OFFSET_ADD		  BIT(30)
#define PTP_LTC_OFFSET_VAL(x)		  (x)

#define MSCC_PHY_PTP_ACCUR_CFG_STATUS	  0x0074
#define PTP_ACCUR_PPS_OUT_CALIB_ERR	  0x20000
#define PTP_ACCUR_PPS_OUT_CALIB_DONE	  0x10000
#define PTP_ACCUR_PPS_IN_CALIB_ERR	  0x4000
#define PTP_ACCUR_PPS_IN_CALIB_DONE	  0x2000
#define PTP_ACCUR_EGR_SOF_CALIB_ERR	  0x1000
#define PTP_ACCUR_EGR_SOF_CALIB_DONE	  0x0800
#define PTP_ACCUR_INGR_SOF_CALIB_ERR	  0x0400
#define PTP_ACCUR_INGR_SOF_CALIB_DONE	  0x0200
#define PTP_ACCUR_LOAD_SAVE_CALIB_ERR	  0x0100
#define PTP_ACCUR_LOAD_SAVE_CALIB_DONE	  0x0080
#define PTP_ACCUR_CALIB_TRIGG		  0x0040
#define PTP_ACCUR_PPS_OUT_BYPASS	  0x0010
#define PTP_ACCUR_PPS_IN_BYPASS		  0x0008
#define PTP_ACCUR_EGR_SOF_BYPASS	  0x0004
#define PTP_ACCUR_INGR_SOF_BYPASS	  0x0002
#define PTP_ACCUR_LOAD_SAVE_BYPASS	  0x0001

#define MSCC_PHY_ANA_ETH2_NTX_PROT	  0x0090
#define ANA_ETH2_NTX_PROT_COMPARATOR_MASK GENMASK(2, 0)
#define ANA_ETH2_NTX_PROT_PTP_OAM	  0x0005
#define ANA_ETH2_NTX_PROT_MPLS		  0x0004
#define ANA_ETH2_NTX_PROT_IP_UDP_ACH_2	  0x0003
#define ANA_ETH2_NTX_PROT_IP_UDP_ACH_1	  0x0002
#define ANA_ETH2_NTX_PROT_ETH2		  0x0001

#define MSCC_PHY_ANA_ETH2_NXT_PROT_ETYPE_MATCH	0x0003
#define ANA_ETH2_NXT_PROT_ETYPE_MATCH_ENA 0x10000
#define ANA_ETH2_NXT_PROT_ETYPE_MATCH_MASK	GENMASK(15, 0)
#define ANA_ETH2_NXT_PROT_ETYPE_MATCH(x)  ((x) & ANA_ETH2_NXT_PROT_ETYPE_MATCH_MASK)

#define MSCC_ANA_ETH2_FLOW_ENA(x)	  (0x00a0 + ((x) << 4))
#define ETH2_FLOW_ENA_CHANNEL_MASK_MASK	  GENMASK(9, 8)
#define ETH2_FLOW_ENA_CHANNEL_MASK(x)	  (((x) << 8) & ETH2_FLOW_ENA_CHANNEL_MASK_MASK)
#define ETH2_FLOW_VALID_CH1	  ETH2_FLOW_ENA_CHANNEL_MASK(2)
#define ETH2_FLOW_VALID_CH0	  ETH2_FLOW_ENA_CHANNEL_MASK(1)

#define MSCC_PHY_ANA_MPLS_COMP_NXT_COMP	  0x0120
#define ANA_MPLS_NTX_PROT_COMPARATOR_MASK GENMASK(2, 0)
#define ANA_MPLS_NTX_PROT_PTP_OAM	  0x0005
#define ANA_MPLS_NTX_PROT_MPLS		  0x0004
#define ANA_MPLS_NTX_PROT_IP_UDP_ACH_2	  0x0003
#define ANA_MPLS_NTX_PROT_IP_UDP_ACH_1	  0x0002
#define ANA_MPLS_NTX_PROT_ETH2		  0x0001

#define MSCC_ANA_MPLS_FLOW_CTRL(x)	  (0x0130 + ((x) << 4))
#define MPLS_FLOW_CTRL_CHANNEL_MASK_MASK  GENMASK(25, 24)
#define MPLS_FLOW_CTRL_CHANNEL_MASK(x)	  (((x) << 24) & MPLS_FLOW_CTRL_CHANNEL_MASK_MASK)
#define MPLS_FLOW_VALID_CH1		  MPLS_FLOW_CTRL_CHANNEL_MASK(2)
#define MPLS_FLOW_VALID_CH0		  MPLS_FLOW_CTRL_CHANNEL_MASK(1)

#define MSCC_ANA_IP1_NXT_PROT_NXT_COMP	  0x01b0
#define ANA_IP1_NXT_PROT_NXT_COMP_BYTES_HDR_MASK	GENMASK(15, 8)
#define ANA_IP1_NXT_PROT_NXT_COMP_BYTES_HDR(x)	(((x) << 8) & ANA_IP1_NXT_PROT_NXT_COMP_BYTES_HDR_MASK)
#define ANA_IP1_NXT_PROT_NXT_COMP_PTP_OAM	0x0005
#define ANA_IP1_NXT_PROT_NXT_COMP_IP_UDP_ACH2	0x0003

#define MSCC_ANA_IP1_NXT_PROT_IP1_MODE	  0x01b1
#define ANA_IP1_NXT_PROT_FLOW_OFFSET_IPV4 0x0c00
#define ANA_IP1_NXT_PROT_FLOW_OFFSET_IPV6 0x0800
#define ANA_IP1_NXT_PROT_IPV6		  0x0001
#define ANA_IP1_NXT_PROT_IPV4		  0x0000

#define MSCC_ANA_IP1_NXT_PROT_IP_MATCH1	  0x01b2
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_OFF_MASK	GENMASK(20, 16)
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_OFF(x)	(((x) << 16) & ANA_IP1_NXT_PROT_IP_MATCH1_PROT_OFF_MASK)
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MASK_MASK	GENMASK(15, 8)
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MASK(x)	(((x) << 15) & ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MASK_MASK)
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MATCH_MASK	GENMASK(7, 0)
#define ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MATCH(x)	((x) & ANA_IP1_NXT_PROT_IP_MATCH1_PROT_MATCH_MASK)

#define MSCC_ANA_IP1_NXT_PROT_MATCH2_UPPER	0x01b3
#define MSCC_ANA_IP1_NXT_PROT_MATCH2_LOWER	0x01b4
#define MSCC_ANA_IP1_NXT_PROT_MASK2_UPPER	0x01b5
#define MSCC_ANA_IP1_NXT_PROT_MASK2_LOWER	0x01b6

#define MSCC_ANA_IP1_NXT_PROT_OFFSET2	  0x01b7
#define ANA_IP1_NXT_PROT_OFFSET2_MASK	  GENMASK(6, 0)
#define ANA_IP1_NXT_PROT_OFFSET2(x)	  ((x) & ANA_IP1_NXT_PROT_OFFSET2_MASK)

#define MSCC_ANA_IP1_NXT_PROT_UDP_CHKSUM  0x01b8
#define IP1_NXT_PROT_UDP_CHKSUM_OFF_MASK  GENMASK(15, 8)
#define IP1_NXT_PROT_UDP_CHKSUM_OFF(x)	  (((x) << 8) & IP1_NXT_PROT_UDP_CHKSUM_OFF_MASK)
#define IP1_NXT_PROT_UDP_CHKSUM_WIDTH_MASK	GENMASK(5, 4)
#define IP1_NXT_PROT_UDP_CHKSUM_WIDTH(x)  (((x) << 4) & IP1_NXT_PROT_UDP_CHKSUM_WIDTH_MASK)
#define IP1_NXT_PROT_UDP_CHKSUM_UPDATE	  0x0002
#define IP1_NXT_PROT_UDP_CHKSUM_CLEAR	  0x0001

#define MSCC_ANA_IP1_FLOW_ENA(x)	  (0x01c0 + ((x) << 4))
#define IP1_FLOW_MATCH_ADDR_MASK	  GENMASK(9, 8)
#define IP1_FLOW_MATCH_DEST_SRC_ADDR	  0x0200
#define IP1_FLOW_MATCH_DEST_ADDR	  0x0100
#define IP1_FLOW_MATCH_SRC_ADDR		  0x0000
#define IP1_FLOW_ENA_CHANNEL_MASK_MASK	  GENMASK(5, 4)
#define IP1_FLOW_ENA_CHANNEL_MASK(x)	  (((x) << 4) & IP1_FLOW_ENA_CHANNEL_MASK_MASK)
#define IP1_FLOW_VALID_CH1		  IP1_FLOW_ENA_CHANNEL_MASK(2)
#define IP1_FLOW_VALID_CH0		  IP1_FLOW_ENA_CHANNEL_MASK(1)
#define IP1_FLOW_ENA			  0x0001

#define MSCC_ANA_OAM_PTP_FLOW_ENA(x)	  (0x1e0 + ((x) << 4))
#define MSCC_ANA_OAM_PTP_FLOW_MATCH_LOWER(x)	(MSCC_ANA_OAM_PTP_FLOW_ENA(x) + 2)
#define MSCC_ANA_OAM_PTP_FLOW_MASK_LOWER(x)	(MSCC_ANA_OAM_PTP_FLOW_ENA(x) + 4)

#define MSCC_ANA_OAM_PTP_FLOW_PTP_0_FIELD(x)	(MSCC_ANA_OAM_PTP_FLOW_ENA(x) + 8)

#define MSCC_ANA_IP1_FLOW_MATCH_UPPER(x)  (MSCC_ANA_IP1_FLOW_ENA(x) + 1)
#define MSCC_ANA_IP1_FLOW_MATCH_UPPER_MID(x)  (MSCC_ANA_IP1_FLOW_ENA(x) + 2)
#define MSCC_ANA_IP1_FLOW_MATCH_LOWER_MID(x)  (MSCC_ANA_IP1_FLOW_ENA(x) + 3)
#define MSCC_ANA_IP1_FLOW_MATCH_LOWER(x)  (MSCC_ANA_IP1_FLOW_ENA(x) + 4)
#define MSCC_ANA_IP1_FLOW_MASK_UPPER(x)	  (MSCC_ANA_IP1_FLOW_ENA(x) + 5)
#define MSCC_ANA_IP1_FLOW_MASK_UPPER_MID(x)	  (MSCC_ANA_IP1_FLOW_ENA(x) + 6)
#define MSCC_ANA_IP1_FLOW_MASK_LOWER_MID(x)	  (MSCC_ANA_IP1_FLOW_ENA(x) + 7)
#define MSCC_ANA_IP1_FLOW_MASK_LOWER(x)	  (MSCC_ANA_IP1_FLOW_ENA(x) + 8)

#define MSCC_ANA_IP2_NXT_PROT_NXT_COMP	  0x0240
#define ANA_IP2_NXT_PROT_NXT_COMP_BYTES_HDR_MASK	GENMASK(15, 8)
#define ANA_IP2_NXT_PROT_NXT_COMP_BYTES_HDR(x)	(((x) << 8) & ANA_IP2_NXT_PROT_NXT_COMP_BYTES_HDR_MASK)
#define ANA_IP2_NXT_PROT_NXT_COMP_PTP_OAM	0x0005
#define ANA_IP2_NXT_PROT_NXT_COMP_IP_UDP_ACH2	0x0003

#define MSCC_ANA_IP2_NXT_PROT_UDP_CHKSUM  0x0248
#define IP2_NXT_PROT_UDP_CHKSUM_OFF_MASK  GENMASK(15, 8)
#define IP2_NXT_PROT_UDP_CHKSUM_OFF(x)	  (((x) << 8) & IP2_NXT_PROT_UDP_CHKSUM_OFF_MASK)
#define IP2_NXT_PROT_UDP_CHKSUM_WIDTH_MASK  GENMASK(5, 4)
#define IP2_NXT_PROT_UDP_CHKSUM_WIDTH(x)  (((x) << 4) & IP2_NXT_PROT_UDP_CHKSUM_WIDTH_MASK)

#define MSCC_ANA_IP2_FLOW_ENA(x)	  (0x0250 + ((x) << 4))
#define IP2_FLOW_ENA_CHANNEL_MASK_MASK	  GENMASK(5, 4)
#define IP2_FLOW_ENA_CHANNEL_MASK(x)	  (((x) << 4) & IP2_FLOW_ENA_CHANNEL_MASK_MASK)
#define IP2_FLOW_VALID_CH1	  IP2_FLOW_ENA_CHANNEL_MASK(2)
#define IP2_FLOW_VALID_CH0	  IP2_FLOW_ENA_CHANNEL_MASK(1)

#define MSCC_ANA_PTP_FLOW_ENA(x)	  (0x02d0 + ((x) << 4))
#define PTP_FLOW_ENA_CHANNEL_MASK_MASK	  GENMASK(5, 4)
#define PTP_FLOW_ENA_CHANNEL_MASK(x)	  (((x) << 4) & PTP_FLOW_ENA_CHANNEL_MASK_MASK)
#define PTP_FLOW_VALID_CH1	  PTP_FLOW_ENA_CHANNEL_MASK(2)
#define PTP_FLOW_VALID_CH0	  PTP_FLOW_ENA_CHANNEL_MASK(1)
#define PTP_FLOW_ENA			  0x0001

#define MSCC_ANA_PTP_FLOW_MATCH_UPPER(x)  (MSCC_ANA_PTP_FLOW_ENA(x) + 1)
#define PTP_FLOW_MSG_TYPE_MASK		  0x0F000000
#define PTP_FLOW_MSG_PDELAY_RESP	  0x04000000
#define PTP_FLOW_MSG_PDELAY_REQ		  0x02000000
#define PTP_FLOW_MSG_DELAY_REQ		  0x01000000
#define PTP_FLOW_MSG_SYNC		  0x00000000

#define MSCC_ANA_PTP_FLOW_MATCH_LOWER(x)  (MSCC_ANA_PTP_FLOW_ENA(x) + 2)
#define MSCC_ANA_PTP_FLOW_MASK_UPPER(x)	  (MSCC_ANA_PTP_FLOW_ENA(x) + 3)
#define MSCC_ANA_PTP_FLOW_MASK_LOWER(x)	  (MSCC_ANA_PTP_FLOW_ENA(x) + 4)

#define MSCC_ANA_PTP_FLOW_DOMAIN_RANGE(x) (MSCC_ANA_PTP_FLOW_ENA(x) + 5)
#define PTP_FLOW_DOMAIN_RANGE_ENA	   0x0001

#define MSCC_ANA_PTP_FLOW_PTP_ACTION(x)	  (MSCC_ANA_PTP_FLOW_ENA(x) + 6)
#define PTP_FLOW_PTP_ACTION_MOD_FRAME_STATUS_UPDATE	0x10000000
#define PTP_FLOW_PTP_ACTION_MOD_FRAME_STATUS_BYTE_OFFSET_MASK	GENMASK(26, 24)
#define PTP_FLOW_PTP_ACTION_MOD_FRAME_STATUS_BYTE_OFFSET(x)	(((x) << 24) & PTP_FLOW_PTP_ACTION_MOD_FRAME_STATUS_BYTE_OFFSET_MASK)
#define PTP_FLOW_PTP_ACTION_PTP_CMD_MASK  GENMASK(3, 0)
#define PTP_FLOW_PTP_ACTION_PTP_CMD(x)	  ((x) & PTP_FLOW_PTP_ACTION_PTP_CMD_MASK)
#define PTP_FLOW_PTP_ACTION_SUB_DELAY_ASYM	0x00200000
#define PTP_FLOW_PTP_ACTION_ADD_DELAY_ASYM	0x00100000
#define PTP_FLOW_PTP_ACTION_TIME_OFFSET_MASK	GENMASK(15, 10)
#define PTP_FLOW_PTP_ACTION_TIME_OFFSET(x)	(((x) << 10) & PTP_FLOW_PTP_ACTION_TIME_OFFSET_MASK)
#define PTP_FLOW_PTP_ACTION_CORR_OFFSET_MASK	GENMASK(9, 5)
#define PTP_FLOW_PTP_ACTION_CORR_OFFSET(x)	(((x) << 5) & PTP_FLOW_PTP_ACTION_CORR_OFFSET_MASK)
#define PTP_FLOW_PTP_ACTION_SAVE_LOCAL_TIME 0x00000010

#define MSCC_ANA_PTP_FLOW_PTP_ACTION2(x)  (MSCC_ANA_PTP_FLOW_ENA(x) + 7)
#define PTP_FLOW_PTP_ACTION2_REWRITE_OFFSET_MASK	GENMASK(15, 8)
#define PTP_FLOW_PTP_ACTION2_REWRITE_OFFSET(x)	(((x) << 8) & PTP_FLOW_PTP_ACTION2_REWRITE_OFFSET_MASK)
#define PTP_FLOW_PTP_ACTION2_REWRITE_BYTES_MASK	GENMASK(3, 0)
#define PTP_FLOW_PTP_ACTION2_REWRITE_BYTES(x)	((x) & PTP_FLOW_PTP_ACTION2_REWRITE_BYTES_MASK)

#define MSCC_ANA_PTP_FLOW_PTP_0_FIELD(x)  (MSCC_ANA_PTP_FLOW_ENA(x) + 8)
#define PTP_FLOW_PTP_0_FIELD_PTP_FRAME	  0x8000
#define PTP_FLOW_PTP_0_FIELD_RSVRD_CHECK  0x4000
#define PTP_FLOW_PTP_0_FIELD_OFFSET_MASK  GENMASK(13, 8)
#define PTP_FLOW_PTP_0_FIELD_OFFSET(x)	  (((x) << 8) & PTP_FLOW_PTP_0_FIELD_OFFSET_MASK)
#define PTP_FLOW_PTP_0_FIELD_BYTES_MASK	  GENMASK(3, 0)
#define PTP_FLOW_PTP_0_FIELD_BYTES(x)	  ((x) & PTP_FLOW_PTP_0_FIELD_BYTES_MASK)

#define MSCC_ANA_PTP_IP_CHKSUM_SEL	  0x0330
#define ANA_PTP_IP_CHKSUM_SEL_IP_COMP_2   0x0001
#define ANA_PTP_IP_CHKSUM_SEL_IP_COMP_1	  0x0000

#define MSCC_PHY_ANA_FSB_CFG		  0x331
#define ANA_FSB_ADDR_FROM_BLOCK_SEL_MASK  GENMASK(1, 0)
#define ANA_FSB_ADDR_FROM_IP2		  0x0003
#define ANA_FSB_ADDR_FROM_IP1		  0x0002
#define ANA_FSB_ADDR_FROM_ETH2		  0x0001
#define ANA_FSB_ADDR_FROM_ETH1		  0x0000

#define MSCC_PHY_ANA_FSB_REG(x)		  (0x332 + (x))

#define COMP_MAX_FLOWS			  8
#define PTP_COMP_MAX_FLOWS		  6

#define PPS_WIDTH_ADJ			  0x1dcd6500
#define STALL_EGR_LATENCY(x)		  (1536000 / (x))

/* PHC clock available frequencies. */
enum {
	PHC_CLK_125MHZ,
	PHC_CLK_156_25MHZ,
	PHC_CLK_200MHZ,
	PHC_CLK_250MHZ,
	PHC_CLK_500MHZ,
};

enum ptp_cmd {
	PTP_NOP = 0,
	PTP_WRITE_1588 = 5,
	PTP_WRITE_NS = 7,
	PTP_SAVE_IN_TS_FIFO = 11, /* invalid when writing in reg */
};

struct vsc85xx_ptphdr {
	u8 tsmt; /* transportSpecific | messageType */
	u8 ver;  /* reserved0 | versionPTP */
	__be16 msglen;
	u8 domain;
	u8 rsrvd1;
	__be16 flags;
	__be64 correction;
	__be32 rsrvd2;
	__be64 clk_identity;
	__be16 src_port_id;
	__be16 seq_id;
	u8 ctrl;
	u8 log_interval;
} __attribute__((__packed__));

/* Represents an entry in the timestamping FIFO */
struct vsc85xx_ts_fifo {
	u32 ns;
	u64 secs:48;
	u8 sig[16];
} __attribute__((__packed__));

struct vsc85xx_ptp {
	struct phy_device *phydev;
	struct ptp_clock *ptp_clock;
	struct ptp_clock_info caps;
	struct sk_buff_head tx_queue;
	enum hwtstamp_tx_types tx_type;
	enum hwtstamp_rx_filters rx_filter;
	u8 configured:1;
};

#endif /* _MSCC_PHY_PTP_H_ */
