﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/amplifybackend/AmplifyBackendRequest.h>
#include <aws/amplifybackend/AmplifyBackend_EXPORTS.h>
#include <aws/amplifybackend/model/ServiceName.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AmplifyBackend {
namespace Model {

/**
 * <p>The request body for ImportBackendStorage.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/ImportBackendStorageRequest">AWS
 * API Reference</a></p>
 */
class ImportBackendStorageRequest : public AmplifyBackendRequest {
 public:
  AWS_AMPLIFYBACKEND_API ImportBackendStorageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ImportBackendStorage"; }

  AWS_AMPLIFYBACKEND_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The app ID.</p>
   */
  inline const Aws::String& GetAppId() const { return m_appId; }
  inline bool AppIdHasBeenSet() const { return m_appIdHasBeenSet; }
  template <typename AppIdT = Aws::String>
  void SetAppId(AppIdT&& value) {
    m_appIdHasBeenSet = true;
    m_appId = std::forward<AppIdT>(value);
  }
  template <typename AppIdT = Aws::String>
  ImportBackendStorageRequest& WithAppId(AppIdT&& value) {
    SetAppId(std::forward<AppIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the backend environment.</p>
   */
  inline const Aws::String& GetBackendEnvironmentName() const { return m_backendEnvironmentName; }
  inline bool BackendEnvironmentNameHasBeenSet() const { return m_backendEnvironmentNameHasBeenSet; }
  template <typename BackendEnvironmentNameT = Aws::String>
  void SetBackendEnvironmentName(BackendEnvironmentNameT&& value) {
    m_backendEnvironmentNameHasBeenSet = true;
    m_backendEnvironmentName = std::forward<BackendEnvironmentNameT>(value);
  }
  template <typename BackendEnvironmentNameT = Aws::String>
  ImportBackendStorageRequest& WithBackendEnvironmentName(BackendEnvironmentNameT&& value) {
    SetBackendEnvironmentName(std::forward<BackendEnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the S3 bucket.</p>
   */
  inline const Aws::String& GetBucketName() const { return m_bucketName; }
  inline bool BucketNameHasBeenSet() const { return m_bucketNameHasBeenSet; }
  template <typename BucketNameT = Aws::String>
  void SetBucketName(BucketNameT&& value) {
    m_bucketNameHasBeenSet = true;
    m_bucketName = std::forward<BucketNameT>(value);
  }
  template <typename BucketNameT = Aws::String>
  ImportBackendStorageRequest& WithBucketName(BucketNameT&& value) {
    SetBucketName(std::forward<BucketNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the storage service.</p>
   */
  inline ServiceName GetServiceName() const { return m_serviceName; }
  inline bool ServiceNameHasBeenSet() const { return m_serviceNameHasBeenSet; }
  inline void SetServiceName(ServiceName value) {
    m_serviceNameHasBeenSet = true;
    m_serviceName = value;
  }
  inline ImportBackendStorageRequest& WithServiceName(ServiceName value) {
    SetServiceName(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_appId;
  bool m_appIdHasBeenSet = false;

  Aws::String m_backendEnvironmentName;
  bool m_backendEnvironmentNameHasBeenSet = false;

  Aws::String m_bucketName;
  bool m_bucketNameHasBeenSet = false;

  ServiceName m_serviceName{ServiceName::NOT_SET};
  bool m_serviceNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace AmplifyBackend
}  // namespace Aws
