# coding: utf-8
# -------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
# -------------------------------------------------------------------------
import pytest
from azure.core.exceptions import ResourceNotFoundError

from testcase import LoadtestingPowerShellPreparer
from testcase_async import LoadtestingAsyncTest
from devtools_testutils.aio import recorded_by_proxy_async
from devtools_testutils import set_bodiless_matcher, set_custom_default_matcher
import os
from pathlib import Path

non_existing_test_id = "0000-0000"
non_existing_test_run_id = "0000-0000"
non_existing_file_id = "000-000"
DISPLAY_NAME = "TestingResource"


class TestServerMetricsAsync(LoadtestingAsyncTest):

    async def prepare(self, endpoint, test_id, file_id, test_run_id, subscription_id):
        client = self.create_client(endpoint=endpoint)

        await client.load_test_administration.create_or_update_test(
            test_id,
            {
                "resourceId": f"/subscriptions/{subscription_id}/resourceGroups/yashika-rg/providers/Microsoft.LoadTestService/loadtests/loadtestsdk",
                "description": "",
                "displayName": DISPLAY_NAME,
                "loadTestConfig": {
                    "engineSize": "m",
                    "engineInstances": 1,
                    "splitAllCSVs": False,
                },
                "secrets": {},
                "environmentVariables": {},
                "passFailCriteria": {"passFailMetrics": {}},
                "keyvaultReferenceIdentityType": "SystemAssigned",
                "keyvaultReferenceIdentityId": None,
            }
        )

        await client.load_test_administration.upload_test_file(
            test_id,
            file_id,
            open(os.path.join(Path(__file__).resolve().parent, "sample.jmx"), "rb")
        )

        await client.load_test_runs.create_or_update_test(
            test_run_id,
            {
                "testId": test_id,
                "displayName": DISPLAY_NAME
            }
        )

    @LoadtestingPowerShellPreparer()
    @recorded_by_proxy_async
    async def test_create_or_update_server_metrics_config(
            self,
            loadtesting_endpoint,
            loadtesting_test_id,
            loadtesting_file_id,
            loadtesting_test_run_id,
            loadtesting_subscription_id
    ):
        set_custom_default_matcher(
            compare_bodies=False, excluded_headers="Authorization,Content-Type,x-ms-client-request-id,x-ms-request-id"
        )
        await self.prepare(endpoint=loadtesting_endpoint,
                           test_id=loadtesting_test_id,
                           file_id=loadtesting_file_id,
                           test_run_id=loadtesting_test_run_id,
                           subscription_id=loadtesting_subscription_id
                           )

        # positive test
        client = self.create_client(endpoint=loadtesting_endpoint)
        result = await client.load_test_administration.create_or_update_server_metrics_config(
            "some-unique-server-metrics-config-id",
            {
                "testRunId": loadtesting_test_run_id,
            }
        )
        assert result is not None

        # negative test
        with pytest.raises(ResourceNotFoundError):
            await client.load_test_administration.create_or_update_server_metrics_config(
                "some-unique-server-metrics-config-id",
                {
                    "testRunId": non_existing_test_run_id,
                }
            )

    @LoadtestingPowerShellPreparer()
    @recorded_by_proxy_async
    async def test_delete_server_metrics_config(
            self,
            loadtesting_endpoint,
            loadtesting_test_id,
            loadtesting_file_id,
            loadtesting_test_run_id,
            loadtesting_subscription_id
    ):
        set_custom_default_matcher(
            compare_bodies=False, excluded_headers="Authorization,Content-Type,x-ms-client-request-id,x-ms-request-id"
        )
        await self.prepare(endpoint=loadtesting_endpoint,
                           test_id=loadtesting_test_id,
                           file_id=loadtesting_file_id,
                           test_run_id=loadtesting_test_run_id,
                           subscription_id=loadtesting_subscription_id
                           )

        # positive test
        client = self.create_client(endpoint=loadtesting_endpoint)
        result = await client.load_test_administration.delete_server_metrics_config(
            "some-unique-server-metrics-config-id"
        )
        assert result is None

        # negative test
        with pytest.raises(ResourceNotFoundError):
            await client.load_test_administration.delete_server_metrics_config(
                "some-unique-server-metrics-config-id"
            )

    @LoadtestingPowerShellPreparer()
    @recorded_by_proxy_async
    async def test_get_server_metrics_config(
            self,
            loadtesting_endpoint,
            loadtesting_test_id,
            loadtesting_file_id,
            loadtesting_test_run_id,
            loadtesting_subscription_id
    ):
        set_custom_default_matcher(
            compare_bodies=False, excluded_headers="Authorization,Content-Type,x-ms-client-request-id,x-ms-request-id"
        )
        await self.prepare(endpoint=loadtesting_endpoint,
                           test_id=loadtesting_test_id,
                           file_id=loadtesting_file_id,
                           test_run_id=loadtesting_test_run_id,
                           subscription_id=loadtesting_subscription_id
                           )

        # positive test
        client = self.create_client(endpoint=loadtesting_endpoint)

        result = await client.load_test_administration.get_server_metrics_config(
            test_id=loadtesting_test_id,
        )
        assert result is not None

        # negative test
        with pytest.raises(ResourceNotFoundError):
            await client.load_test_administration.get_server_metrics_config(
                test_id=non_existing_test_id,
            )

    @LoadtestingPowerShellPreparer()
    @recorded_by_proxy_async
    async def test_get_server_default_metrics_config(
            self,
            loadtesting_endpoint,
            loadtesting_test_id,
            loadtesting_file_id,
            loadtesting_test_run_id,
            loadtesting_subscription_id
    ):
        set_custom_default_matcher(
            compare_bodies=False, excluded_headers="Authorization,Content-Type,x-ms-client-request-id,x-ms-request-id"
        )
        await self.prepare(endpoint=loadtesting_endpoint,
                           test_id=loadtesting_test_id,
                           file_id=loadtesting_file_id,
                           test_run_id=loadtesting_test_run_id,
                           subscription_id=loadtesting_subscription_id
                           )

        # positive test
        client = self.create_client(endpoint=loadtesting_endpoint)

        result = await client.load_test_administration.get_server_default_metrics_config()
        assert result is not None

    @LoadtestingPowerShellPreparer()
    @recorded_by_proxy_async
    async def test_list_supported_resource_types(
            self,
            loadtesting_endpoint,
            loadtesting_test_id,
            loadtesting_file_id,
            loadtesting_test_run_id,
            loadtesting_subscription_id
    ):
        set_custom_default_matcher(
            compare_bodies=False, excluded_headers="Authorization,Content-Type,x-ms-client-request-id,x-ms-request-id"
        )
        await self.prepare(endpoint=loadtesting_endpoint,
                           test_id=loadtesting_test_id,
                           file_id=loadtesting_file_id,
                           test_run_id=loadtesting_test_run_id,
                           subscription_id=loadtesting_subscription_id
                           )

        client = self.create_client(endpoint=loadtesting_endpoint)
        result = await client.load_test_administration.list_supported_resource_types()
        assert result is not None
