/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once


/////////////////////// Stdlib includes
#include <vector>


/////////////////////// Qt includes
#include <QDomElement>


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/PropListHolder.hpp"
#include "MsXpS/libXpertMassCore/Formula.hpp"
#include "MsXpS/libXpertMassCore/Modif.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{

class Monomer;

typedef Monomer *MonomerRPtr;
typedef const Monomer *MonomerCstRPtr;

typedef std::unique_ptr<Monomer> MonomerUPtr;
typedef std::unique_ptr<const Monomer> MonomerCstUPtr;

typedef std::shared_ptr<Monomer> MonomerSPtr;
typedef std::shared_ptr<const Monomer> MonomerCstSPtr;

typedef std::weak_ptr<Monomer> MonomerWPtr;
typedef std::weak_ptr<const Monomer> MonomerCstWPtr;


using UuidMonomerCstWPtrPair = std::pair<QString, MonomerCstWPtr>;
using UuidMonomerWPtrPair    = std::pair<QString, MonomerWPtr>;

class DECLSPEC Monomer : public PropListHolder
{


  public:
  Monomer(PolChemDefCstSPtr pol_chem_def_csp,
          const QDomElement &element,
          int version);
  Monomer(PolChemDefCstSPtr pol_chem_def_csp = nullptr,
          const QString &name                = QString(),
          const QString &code                = QString(),
          const QString &formula_string      = QString(),
          double mono                        = 0.0,
          double avg                         = 0.0);
  Monomer(const Monomer &other);
  virtual ~Monomer();

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolChemDefCstSPtr(PolChemDefCstSPtr &pol_chem_def_csp);
  const PolChemDefCstSPtr &getPolChemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  QString getName() const;

  //////////////// THE CODE /////////////////////
  void setCode(const QString &code);
  QString getCode() const;
  bool checkCodeSyntax(int code_length = -1) const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const QString &formula_string);
  const QString &getFormula() const;

  //////////////// THE MODIFICATIONS /////////////////////
  const std::vector<ModifSPtr> &getModifsCstRef() const;
  std::vector<QString> modifNamesInOrder() const;
  bool isModifTarget(const Modif &modif) const;
  QString modify(const Modif &modif, bool override, ErrorList *error_list_p);
  QString
  modify(const QString &modif_name, bool override, ErrorList *error_list_p);
  bool unmodify(const QString &uuid);
  bool unmodify();
  bool isModified() const;
  int countModifsByName(const QString &name);

  //////////////// OPERATORS /////////////////////
  using PropListHolder::operator=;
  Monomer &operator=(const Monomer &other);
  bool operator==(const Monomer &) const;
  bool operator!=(const Monomer &) const;

  //////////////// VALIDATIONS /////////////////////
  MonomerSPtr getFromPolChemDefByName() const;
  MonomerSPtr getFromPolChemDefByCode() const;
  Enums::PolChemDefEntityStatus isKnownByNameInPolChemDef() const;
  Enums::PolChemDefEntityStatus isKnownByCodeInPolChemDef() const;

  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// MASS OPERATIONS /////////////////////
  bool calculateMasses(
    const IsotopicDataCstSPtr &isotopic_data_csp,
    double &mono,
    double &avg,
    Enums::ChemicalEntity monomer_chemical_entities = Enums::ChemicalEntity::NONE) const;
  bool calculateMasses(
    const IsotopicDataCstSPtr &isotopic_data_csp,
    Enums::ChemicalEntity monomer_chemical_entities = Enums::ChemicalEntity::NONE);
  Monomer &calculateMasses(
    bool &ok,
    const IsotopicDataCstSPtr &isotopic_data_csp,
    Enums::ChemicalEntity monomer_chemical_entities = Enums::ChemicalEntity::NONE);
  const Monomer &accountMasses(double *mono_p = nullptr,
                               double *avg_p  = nullptr,
                               int times      = 1) const;
  const Monomer &accountMasses(double &mono, double &avg, int times = 1) const;
  double getMass(Enums::MassType mass_type) const;

  //////////////// FORMULA OPERATIONS /////////////////////
  QString calculateFormula(
    Enums::ChemicalEntity monomer_chemical_entities = Enums::ChemicalEntity::NONE) const;

  //////////////// XML DATA LOADING WRITING /////////////////////
  bool renderXmlMnmElement(const QDomElement &element, int version);
  QString
  formatXmlMnmElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken) const;

  bool renderXmlMonomerElement(const QDomElement &element, int version);
  QString formatXmlMonomerElement(
    int offset, const QString &indent = Utils::xmlIndentationToken) const;

  //////////////// UTILS /////////////////////
  QString storeModif(const ModifSPtr &modif_sp);
  QString storeModif(const Modif &modif);

  bool hasModif(const ModifSPtr &modif_sp) const;
  bool hasUuid(const ModifSPtr &modif_sp) const;

  ModifSPtr getModifForUuid(const QString &uuid) const;
  QString getUuidForModif(const ModifSPtr &modif_sp) const;

  std::vector<QString> getAllModifUuids() const;

  void clear();
  QString toString() const;

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name                    = "";
  QString m_code                    = "";
  QString m_formula                 = "";

  double m_mono = 0.0;
  double m_avg  = 0.0;

  std::vector<ModifSPtr> m_modifs;

  //  We need in code that uses this class,  to constantly keep at hand
  //  the Modif instances that are involved in Monomer_s. For example,
  //  we need to store the Modif pointers as strings in QListWidget items
  //  (Qt:UserRole).
  //  We thus make use of the QUuid class to craft a Uuid string that
  //  we associate to the Modif weak pointer that in turn relates
  //  to the Modif shared pointer.
  std::vector<UuidModifWPtrPair> m_uuidModifPairs;

  mutable bool m_isValid = false;

private:
  void cleanupModifs();
  bool removeModif(ModifSPtr modif_sp);
};

} // namespace libXpertMassCore
} // namespace MsXpS


Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::Monomer);
extern int monomerMetaTypeId;
