/**
 * \file pappsomspp/processing/cbor/psm/psmcbor2json.cpp
 * \date 19/07/2025
 * \author Olivier Langella
 * \brief convert CBOR PSM to JSON
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "psmcbor2json.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonValue>
#include "pappsomspp/core/pappsoexception.h"

pappso::cbor::psm::PsmCbor2Json::PsmCbor2Json(pappso::cbor::JsonStreamWriter *json_output_p)
{
  mp_jsonOutput = json_output_p;

  *mp_jsonOutput << "{" << Qt::endl;
}


pappso::cbor::psm::PsmCbor2Json::~PsmCbor2Json()
{
}

void
pappso::cbor::psm::PsmCbor2Json::close()
{

  mp_jsonOutput->endMap();
}

void
pappso::cbor::psm::PsmCbor2Json::fastaFilesReady(pappso::UiMonitorInterface &monitor)
{
  if(!m_targetFastaFiles.isEmpty())
    {
      mp_jsonOutput->appendKey("target_fasta_files");
      mp_jsonOutput->writeArray(m_targetFastaFiles);
    }
  if(!m_decoyFastaFiles.isEmpty())
    {
      mp_jsonOutput->appendKey("decoy_fasta_files");
      mp_jsonOutput->writeArray(m_decoyFastaFiles);
    }
}

void
pappso::cbor::psm::PsmCbor2Json::informationsReady(pappso::UiMonitorInterface &monitor)
{

  mp_jsonOutput->appendKey("informations");
  mp_jsonOutput->writeCborMap(m_cborInformations);
}

void
pappso::cbor::psm::PsmCbor2Json::logReady(pappso::UiMonitorInterface &monitor)
{
  mp_jsonOutput->appendKey("log");
  mp_jsonOutput->writeCborArray(m_cborLog);
}


void
pappso::cbor::psm::PsmCbor2Json::parameterMapReady(pappso::UiMonitorInterface &monitor)
{
  mp_jsonOutput->appendKey("parameter_map");
  mp_jsonOutput->writeCborMap(m_cborParameterMap);
}

void
pappso::cbor::psm::PsmCbor2Json::sampleListStarted(pappso::UiMonitorInterface &monitor)
{

  mp_jsonOutput->appendKey("sample_list");
  mp_jsonOutput->startArray();
}

void
pappso::cbor::psm::PsmCbor2Json::sampleListFinished(pappso::UiMonitorInterface &monitor)
{

  mp_jsonOutput->endArray();
}

void
pappso::cbor::psm::PsmCbor2Json::readProteinMap(pappso::UiMonitorInterface &monitor)
{
  mp_jsonOutput->appendKey("protein_map");
  mp_jsonOutput->startMap();
  if(!mpa_cborReader->enterContainer())
    {
      throw pappso::PappsoException(QObject::tr("enterContainer in protein map failed"));
    }
  QString accession;
  while(!mpa_cborReader->lastError() && mpa_cborReader->hasNext())
    {
      mpa_cborReader->decodeString(accession);
      mp_jsonOutput->appendKey(accession);

      QCborMap cbor_protein;
      mpa_cborReader->readCborMap(cbor_protein);
      mp_jsonOutput->writeCborMap(cbor_protein);
    }

  mpa_cborReader->leaveContainer();
  mp_jsonOutput->endMap();
}


void
pappso::cbor::psm::PsmCbor2Json::readScan(pappso::UiMonitorInterface &monitor)
{
  qDebug();
  m_cborScanId.clear();
  if(!mpa_cborReader->enterContainer())
    {
      throw pappso::PappsoException(QObject::tr("enterContainer in scan failed"));
    }
  mp_jsonOutput->startMap();
  //"id": {
  //"index": 1976
  //},
  qDebug() << "scan begin";

  if(!getExpectedString())
    {
      throw pappso::PappsoException(QObject::tr("getExpectedString in scan failed"));
    }
  qDebug() << m_expectedString;

  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "id")
    {
      if(!mpa_cborReader->readCborMap(m_cborScanId))
        {
          throw pappso::PappsoException(QObject::tr("id element in scan is not a cbor map"));
        }

      mp_jsonOutput->writeCborMap(m_cborScanId);
    }
  else
    {
      throw pappso::PappsoException(
        QObject::tr("ERROR: expecting id element in scan not %1").arg(m_expectedString));
    }
  //"precursor": {
  //"z": 2,
  //"mz": 1120.529471
  //},

  getExpectedString();
  m_cborScanPrecursor.clear();
  qDebug() << m_expectedString;
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "precursor")
    {
      if(!mpa_cborReader->readCborMap(m_cborScanPrecursor))
        {
          throw pappso::PappsoException(QObject::tr("precursor element in scan is not a cbor map"));
        }
      mp_jsonOutput->writeCborMap(m_cborScanPrecursor);
    }
  //"ms2": {PSM CBOR format documentation
  //"rt": 12648.87,
  //"mz" :[1,2,3,4],
  //"intensity" : [1,2,3,4]
  //},

  getExpectedString();
  qDebug() << m_expectedString;
  m_cborScanMs2.clear();
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "ms2")
    {
      if(!mpa_cborReader->readCborMap(m_cborScanMs2))
        {
          throw pappso::PappsoException(
            QObject::tr("ms2 element in scan is not a cbor map %1 %2:\n%3")
              .arg(m_currentSampleName)
              .arg(m_cborScanId.value("index").toInteger())
              .arg(mpa_cborReader->lastError().toString()));
        }
      mp_jsonOutput->writeCborMap(m_cborScanMs2);
    }
  //"psm_list": [
  scanStarted(monitor);

  getExpectedString();
  qDebug() << m_expectedString;
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "psm_list")
    {
      mpa_cborReader->enterContainer();
      mp_jsonOutput->startArray();
      while(!mpa_cborReader->lastError() && mpa_cborReader->hasNext())
        {
          readPsm(monitor);
        }
      mpa_cborReader->leaveContainer();
      mp_jsonOutput->endArray();
    }

  mpa_cborReader->leaveContainer();
  mp_jsonOutput->endMap();
  qDebug() << "scan end";
  scanFinished(monitor);
  qDebug();
}


void
pappso::cbor::psm::PsmCbor2Json::readSample(pappso::UiMonitorInterface &monitor)
{
  //"name": "tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1",
  qDebug();
  mpa_cborReader->enterContainer();
  mp_jsonOutput->startMap();
  getExpectedString();

  qDebug() << m_expectedString;
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "name")
    {
      if(!mpa_cborReader->decodeString(m_currentSampleName))
        {
          throw pappso::PappsoException("sample name is not a string");
        }
      mp_jsonOutput->appendValue(m_currentSampleName);
    }
  else
    {
      throw pappso::PappsoException("ERROR: expecting name element in sample");
    }
  //"identification_file_list": [{ "name":
  //"/home/langella/data1/tandem/tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1.xml",
  //}],

  getExpectedString();

  qDebug() << m_expectedString;
  m_currentIdentificationFileList.clear();
  if(m_expectedString == "identification_file_list")
    {
      bool is_ok;
      mpa_cborReader->enterContainer();

      while(!mpa_cborReader->lastError() && mpa_cborReader->hasNext())
        {
          m_currentIdentificationFileList.push_back(readPsmFile(is_ok));
        }
      mpa_cborReader->leaveContainer();

      getExpectedString();
    }
  //"peaklist_file": {"name": "tandem2017_nopatch_20120906_balliau_extract_1_A01_urnb-1.mzml"
  //},

  if(m_expectedString == "peaklist_file")
    {
      bool is_ok;
      m_currentPeaklistFile = readPsmFile(is_ok);
    }
  else
    {
      throw pappso::PappsoException("ERROR: expecting peaklist_file element in sample");
    }
  //"scan_list": [
  sampleStarted(monitor);
  getExpectedString();
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "scan_list")
    {
      mpa_cborReader->enterContainer();
      mp_jsonOutput->startArray();
      while(!mpa_cborReader->lastError() && mpa_cborReader->hasNext())
        {
          readScan(monitor);
        }
      mpa_cborReader->leaveContainer();
      mp_jsonOutput->endArray();
    }
  else
    {
      throw pappso::PappsoException("ERROR: expecting scan_list element in sample");
    }
  mpa_cborReader->leaveContainer();
  mp_jsonOutput->endMap();

  sampleFinished(monitor);
}

void
pappso::cbor::psm::PsmCbor2Json::readPsm(pappso::UiMonitorInterface &monitor)
{
  bool is_ok;
  mpa_cborReader->enterContainer();
  mp_jsonOutput->startMap();
  getExpectedString();
  mp_jsonOutput->appendKey(m_expectedString);
  //  "proforma": "AQEEM[+15.99491]AQVAK",
  if(m_expectedString == "proforma")
    {
      if(!mpa_cborReader->decodeString(m_currentPsmProforma))
        {
          throw pappso::PappsoException("ERROR: proforma element in psm-scan is not a string");
        }
      mp_jsonOutput->appendValue(m_currentPsmProforma);
    }
  else
    {
      throw pappso::PappsoException("ERROR: expecting proforma element in psm-scan");
    }
  //"protein_list" : [
  //{
  //"accession": "GRMZM2G083841_P01",
  //"position": [15,236]
  //}
  //],

  getExpectedString();
  mp_jsonOutput->appendKey(m_expectedString);
  m_currentPsmProteinRefList.clear();
  qDebug() << m_expectedString;
  if(m_expectedString == "protein_list")
    {
      QCborArray cbor_protein_list;
      mpa_cborReader->readCborArray(cbor_protein_list);
      mp_jsonOutput->writeCborArray(cbor_protein_list);
    }
  else
    {
      throw pappso::PappsoException("ERROR: expecting protein_list element in psm-scan");
    }
  //"eval": {
  qDebug();
  m_cborScanPsmProps.clear();
  m_cborScanPsmEval.clear();
  if(!getExpectedString())
    {
      throw pappso::PappsoException(
        QObject::tr("ERROR: expecting props or eval element in psm-scan %1")
          .arg(m_currentPsmProforma));
    }
  if(m_expectedString == "props")
    {
      is_ok = mpa_cborReader->readCborMap(m_cborScanPsmProps);
      if(!is_ok)
        {
          throw pappso::PappsoException("ERROR: props element in psm-scan is not well formed");
        }
      mp_jsonOutput->appendKey(m_expectedString);
      mp_jsonOutput->writeCborMap(m_cborScanPsmProps);
      if(!getExpectedString())
        {
          throw pappso::PappsoException(
            QObject::tr("ERROR: expecting eval element in psm-scan %1").arg(m_currentPsmProforma));
        }
    }
  mp_jsonOutput->appendKey(m_expectedString);
  if(m_expectedString == "eval")
    {
      is_ok = mpa_cborReader->readCborMap(m_cborScanPsmEval);
      if(!is_ok)
        {
          throw pappso::PappsoException("ERROR: eval element in psm-scan is not well formed");
        }
      mp_jsonOutput->writeCborMap(m_cborScanPsmEval);
    }

  qDebug() << m_expectedString;


  mpa_cborReader->leaveContainer();
  mp_jsonOutput->endMap();
  qDebug();
  psmReady(monitor);
}
