/*
 * SSA/ASS common functions
 * Copyright (c) 2010  Aurelien Jacobs <aurel@gnuage.org>
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "avcodec.h"
#include "ass.h"
#include "libavutil/avstring.h"
#include "libavutil/bprint.h"
#include "libavutil/mem.h"
#include "version.h"

int ff_ass_subtitle_header_full(AVCodecContext *avctx,
                                int play_res_x, int play_res_y,
                                const char *font, int font_size,
                                int primary_color, int secondary_color,
                                int outline_color, int back_color,
                                int bold, int italic, int underline,
                                int border_style, int alignment)
{
    avctx->subtitle_header = av_asprintf(
             "[Script Info]\n"
             "; Script generated by FFmpeg/Lavc%s\n"
             "ScriptType: v4.00+\n"
             "PlayResX: %d\n"
             "PlayResY: %d\n"
             "ScaledBorderAndShadow: yes\n"
             "YCbCr Matrix: None\n"
             "\n"
             "[V4+ Styles]\n"

             /* ASS (v4+) header */
             "Format: Name, "
             "Fontname, Fontsize, "
             "PrimaryColour, SecondaryColour, OutlineColour, BackColour, "
             "Bold, Italic, Underline, StrikeOut, "
             "ScaleX, ScaleY, "
             "Spacing, Angle, "
             "BorderStyle, Outline, Shadow, "
             "Alignment, MarginL, MarginR, MarginV, "
             "Encoding\n"

             "Style: "
             "Default,"             /* Name */
             "%s,%d,"               /* Font{name,size} */
             "&H%x,&H%x,&H%x,&H%x," /* {Primary,Secondary,Outline,Back}Colour */
             "%d,%d,%d,0,"          /* Bold, Italic, Underline, StrikeOut */
             "100,100,"             /* Scale{X,Y} */
             "0,0,"                 /* Spacing, Angle */
             "%d,1,0,"              /* BorderStyle, Outline, Shadow */
             "%d,10,10,10,"         /* Alignment, Margin[LRV] */
             "1\n"                  /* Encoding */

             "\n"
             "[Events]\n"
             "Format: Layer, Start, End, Style, Name, MarginL, MarginR, MarginV, Effect, Text\n",
             !(avctx->flags & AV_CODEC_FLAG_BITEXACT) ? AV_STRINGIFY(LIBAVCODEC_VERSION) : "",
             play_res_x, play_res_y, font, font_size,
             primary_color, secondary_color, outline_color, back_color,
             -bold, -italic, -underline, border_style, alignment);

    if (!avctx->subtitle_header)
        return AVERROR(ENOMEM);
    avctx->subtitle_header_size = strlen(avctx->subtitle_header);
    return 0;
}

int ff_ass_subtitle_header(AVCodecContext *avctx,
                           const char *font, int font_size,
                           int color, int back_color,
                           int bold, int italic, int underline,
                           int border_style, int alignment)
{
    return ff_ass_subtitle_header_full(avctx,
                               ASS_DEFAULT_PLAYRESX, ASS_DEFAULT_PLAYRESY,
                               font, font_size, color, color,
                               back_color, back_color,
                               bold, italic, underline,
                               border_style, alignment);
}

int ff_ass_subtitle_header_default(AVCodecContext *avctx)
{
    return ff_ass_subtitle_header(avctx, ASS_DEFAULT_FONT,
                               ASS_DEFAULT_FONT_SIZE,
                               ASS_DEFAULT_COLOR,
                               ASS_DEFAULT_BACK_COLOR,
                               ASS_DEFAULT_BOLD,
                               ASS_DEFAULT_ITALIC,
                               ASS_DEFAULT_UNDERLINE,
                               ASS_DEFAULT_BORDERSTYLE,
                               ASS_DEFAULT_ALIGNMENT);
}

char *ff_ass_get_dialog(int readorder, int layer, const char *style,
                        const char *speaker, const char *text)
{
    return av_asprintf("%d,%d,%s,%s,0,0,0,,%s",
                       readorder, layer, style ? style : "Default",
                       speaker ? speaker : "", text);
}

int ff_ass_add_rect2(AVSubtitle *sub, const char *dialog,
                    int readorder, int layer, const char *style,
                    const char *speaker, unsigned *nb_rect_allocated)
{
    AVSubtitleRect **rects = sub->rects, *rect;
    char *ass_str;
    uint64_t new_nb = 0;

    if (sub->num_rects >= UINT_MAX)
        return AVERROR(ENOMEM);

    if (nb_rect_allocated && *nb_rect_allocated <= sub->num_rects) {
        if (sub->num_rects < UINT_MAX / 17 * 16) {
            new_nb = sub->num_rects + sub->num_rects/16 + 1;
        } else
            new_nb = UINT_MAX;
    } else if (!nb_rect_allocated)
        new_nb = sub->num_rects + 1;

    if (new_nb) {
        rects = av_realloc_array(rects, new_nb, sizeof(*sub->rects));
        if (!rects)
            return AVERROR(ENOMEM);
        if (nb_rect_allocated)
            *nb_rect_allocated = new_nb;
        sub->rects = rects;
    }

    rect       = av_mallocz(sizeof(*rect));
    if (!rect)
        return AVERROR(ENOMEM);
    rects[sub->num_rects++] = rect;
    rect->type = SUBTITLE_ASS;
    ass_str = ff_ass_get_dialog(readorder, layer, style, speaker, dialog);
    if (!ass_str)
        return AVERROR(ENOMEM);
    rect->ass = ass_str;
    return 0;
}

int ff_ass_add_rect(AVSubtitle *sub, const char *dialog,
                    int readorder, int layer, const char *style,
                    const char *speaker)
{
    return ff_ass_add_rect2(sub, dialog, readorder, layer, style, speaker, NULL);
}

void ff_ass_decoder_flush(AVCodecContext *avctx)
{
    FFASSDecoderContext *s = avctx->priv_data;
    if (!(avctx->flags2 & AV_CODEC_FLAG2_RO_FLUSH_NOOP))
        s->readorder = 0;
}

void ff_ass_bprint_text_event(AVBPrint *buf, const char *p, int size,
                             const char *linebreaks, int keep_ass_markup)
{
    const char *p_end = p + size;

    for (; p < p_end && *p; p++) {

        /* forced custom line breaks, not accounted as "normal" EOL */
        if (linebreaks && strchr(linebreaks, *p)) {
            av_bprintf(buf, "\\N");

        /* cancel curly brackets to avoid bogus override tag blocks
         * hiding text. Standard ASS has no character escapes,
         * though (only) libass provides \{ and \}.
         * Unpaired closing brackets don't need escaping at all though and
         * to make the situation less bad in standard ASS insert an empty block
         */
        } else if (!keep_ass_markup && *p == '{') {
            av_bprintf(buf, "\\{{}");

        /* append word-joiner U+2060 as UTF-8 to break up sequences like \N */
        } else if (!keep_ass_markup && *p == '\\') {
            if (p_end - p <= 3 || strncmp(p + 1, "\xe2\x81\xa0", 3))
                av_bprintf(buf, "\\\xe2\x81\xa0");
            else
                av_bprintf(buf, "\\");

        /* some packets might end abruptly (no \0 at the end, like for example
         * in some cases of demuxing from a classic video container), some
         * might be terminated with \n or \r\n which we have to remove (for
         * consistency with those who haven't), and we also have to deal with
         * evil cases such as \r at the end of the buffer (and no \0 terminated
         * character) */
        } else if (p[0] == '\n') {
            /* some stuff left so we can insert a line break */
            if (p < p_end - 1)
                av_bprintf(buf, "\\N");
        } else if (p[0] == '\r' && p < p_end - 1 && p[1] == '\n') {
            /* \r followed by a \n, we can skip it. We don't insert the \N yet
             * because we don't know if it is followed by more text */
            continue;

        /* finally, a sane character */
        } else {
            av_bprint_chars(buf, *p, 1);
        }
    }
}
