package OpenInteract2::Brick::BaseUser;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'base_user-2.41.zip' => 'BASE_USER241ZIP',
);

sub get_name {
    return 'base_user';
}

sub get_resources {
    return (
        'base_user-2.41.zip' => [ 'pkg base_user-2.41.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::BaseUser - Base-64 encoded OI2 package 'base_user-2.41.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "base_user-2.41.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'base_user' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<base_user-2.41.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub BASE_USER241ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

