#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util::GMP qw/primorial pn_primorial factorial factorialmod
                              multifactorial subfactorial factorial_sum
                              falling_factorial rising_factorial
                              addmod/;

my $extra = defined $ENV{EXTENDED_TESTING} && $ENV{EXTENDED_TESTING};

my @small_primes = qw/
2 3 5 7 11 13 17 19 23 29 31 37 41 43 47 53 59 61 67 71
73 79 83 89 97 101 103 107 109 113 127 131 137 139 149 151 157 163 167 173
179 181 191 193 197 199 211 223 227 229 233 239 241 251 257 263 269 271 277 281
283 293 307 311 313 317 331 337 347 349 353 359 367 373 379 383 389 397 401 409
419 421 431 433 439 443 449 457 461 463 467 479 487 491 499 503 509 521 523 541
547 557 563 569 571 577 587 593 599 601 607 613 617 619 631 641 643 647 653 659
661 673 677 683 691 701 709 719 727 733 739 743 751 757 761 769 773 787 797 809
811 821 823 827 829 839 853 857 859 863 877 881 883 887 907 911 919 929 937 941
/;
sub nth_prime {
  my $n = shift;
  return 0 if $n <= 0;
  die "Out of range for fake nth_prime: $n" unless defined $small_primes[$n-1];
  $small_primes[$n-1];
}

my @pn_primorials = qw/
1
2
6
30
210
2310
30030
510510
9699690
223092870
6469693230
200560490130
7420738134810
304250263527210
13082761331670030
614889782588491410
32589158477190044730
1922760350154212639070
117288381359406970983270
7858321551080267055879090
557940830126698960967415390
40729680599249024150621323470
3217644767340672907899084554130
267064515689275851355624017992790
23768741896345550770650537601358310
2305567963945518424753102147331756070
232862364358497360900063316880507363070
23984823528925228172706521638692258396210
2566376117594999414479597815340071648394470
279734996817854936178276161872067809674997230
31610054640417607788145206291543662493274686990
/;

my @factorials = (qw/
1
1
2
6
24
120
720
5040
40320
362880
3628800
39916800
479001600
6227020800
87178291200
1307674368000
20922789888000
355687428096000
6402373705728000
121645100408832000
2432902008176640000
51090942171709440000
1124000727777607680000
25852016738884976640000
620448401733239439360000
15511210043330985984000000
403291461126605635584000000
10888869450418352160768000000
304888344611713860501504000000
8841761993739701954543616000000
265252859812191058636308480000000
/);

my @facmods = (
  [32,-73,50],
  #[37,0,undef],
  [37,1,0],
  [37,31,0],
  [17,503,73],
  [498,503,482],
  [502,503,502],
  [503,503,0],
);

if ($extra) {
  push @facmods, [1000000, "5000000002", "1657069652"];
  push @facmods, [6000001, "779783490222", "136326683526"];
}

plan tests =>   1    # factorial
              + 1    # factorialmod
              + scalar(@facmods) + 1
              + 2    # primorial and pn_primorial
              + 2    # extra primorial tests
              + 1    # subfactorial
              + 1    # factorial_sum
              + 4    # multifactorial
              + 2    # falling_factorial
              + 2;   # rising_factorial

{
  my @fact = map { factorial($_) }  0 .. $#factorials;
  is_deeply( \@fact, \@factorials, "factorial 0 .. $#factorials" );
}

is_deeply( [map { factorialmod($_>>2,$_) }        10000 .. 10100],
           [map { addmod(factorial($_>>2),0,$_) } 10000 .. 10100],
           "factorialmod" );

for my $d (@facmods) {
  my($n, $m, $expect) = @$d;
  is( factorialmod($n, $m), $expect, "factorialmod($n,$m) = $expect" );
}
is( factorialmod(37,0), undef, "factorialmod(37,0) = undef" );

{
  my @prim   = map { primorial(nth_prime($_)) }  0 .. $#pn_primorials;
  my @pnprim = map { pn_primorial($_)         }  0 .. $#pn_primorials;
  is_deeply(\@prim, \@pn_primorials, "primorial(nth(...)) 0 - $#pn_primorials");
  is_deeply(\@pnprim, \@pn_primorials, "pn_primorial(...) 0 - $#pn_primorials");
}

is( primorial(100), '2305567963945518424753102147331756070', "primorial(100)");

is(
    primorial(541),
    '4711930799906184953162487834760260422020574773409675520188634839616415335845034221205289256705544681972439104097777157991804380284218315038719444943990492579030720635990538452312528339864352999310398481791730017201031090',
    "primorial(541)"
  );

is_deeply( [ map { subfactorial($_) } 0..23 ],
           [qw/1 0 1 2 9 44 265 1854 14833 133496 1334961 14684570 176214841 2290792932 32071101049 481066515734 7697064251745 130850092279664 2355301661033953 44750731559645106 895014631192902121 18795307255050944540 413496759611120779881 9510425471055777937262/],
           "subfactoral(n) for 0..23" );

is_deeply( [ map { factorial_sum($_) } 0..22 ],
           [qw/0 1 2 4 10 34 154 874 5914 46234 409114 4037914 43954714 522956314 6749977114 93928268314 1401602636314 22324392524314 378011820620314 6780385526348314 128425485935180314 2561327494111820314 53652269665821260314/],
           "factorial_sum(n) for 0..22" );

is_deeply( [ map { multifactorial($_,0) } 0..22 ],
           [ map { 1 } 0..22 ],
           "multifactorial(n,0) for 0..22" );

is_deeply( [ map { multifactorial($_,1) } 0..22 ],
           [ map { factorial($_) } 0..22 ],
           "multifactorial(n,1) for 0..22" );

is_deeply( [ map { multifactorial($_,2) } 0..26 ],
           [qw/1 1 2 3 8 15 48 105 384 945 3840 10395 46080 135135 645120 2027025 10321920 34459425 185794560 654729075 3715891200 13749310575 81749606400 316234143225 1961990553600 7905853580625 51011754393600/],
           "multifactorial(n,2) for 0..26" );

is_deeply( [ map { multifactorial($_,3) } 0..29 ],
           [qw/1 1 2 3 4 10 18 28 80 162 280 880 1944 3640 12320 29160 58240 209440 524880 1106560 4188800 11022480 24344320 96342400 264539520 608608000 2504902400 7142567040 17041024000 72642169600/],
           "multifactorial(n,3) for 0..29" );

###### falling_factorial
{
  my @s;
  for my $k (0..10) { push @s, map { falling_factorial($_,$k) } -10..10; }
  is_deeply( \@s,
             [qw/1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 -10 -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 10 110 90 72 56 42 30 20 12 6 2 0 0 2 6 12 20 30 42 56 72 90 -1320 -990 -720 -504 -336 -210 -120 -60 -24 -6 0 0 0 6 24 60 120 210 336 504 720 17160 11880 7920 5040 3024 1680 840 360 120 24 0 0 0 0 24 120 360 840 1680 3024 5040 -240240 -154440 -95040 -55440 -30240 -15120 -6720 -2520 -720 -120 0 0 0 0 0 120 720 2520 6720 15120 30240 3603600 2162160 1235520 665280 332640 151200 60480 20160 5040 720 0 0 0 0 0 0 720 5040 20160 60480 151200 -57657600 -32432400 -17297280 -8648640 -3991680 -1663200 -604800 -181440 -40320 -5040 0 0 0 0 0 0 0 5040 40320 181440 604800 980179200 518918400 259459200 121080960 51891840 19958400 6652800 1814400 362880 40320 0 0 0 0 0 0 0 0 40320 362880 1814400 -17643225600 -8821612800 -4151347200 -1816214400 -726485760 -259459200 -79833600 -19958400 -3628800 -362880 0 0 0 0 0 0 0 0 0 362880 3628800 335221286400 158789030400 70572902400 29059430400 10897286400 3632428800 1037836800 239500800 39916800 3628800 0 0 0 0 0 0 0 0 0 0 3628800/],
  "falling_factorial(-10..10, 0..10)" );
}
is_deeply( [map { falling_factorial($_->[0],$_->[1]) } ([515,7],[516,7],[568,7],[89,10],[103,101])],
           [qw/9222879462222182400 9349716704335257600 18378924259448108160 18452514066426316800 49514503582430902037733576272908866745450829110572462415026402773499383329208111416070720536941769246326758192988646046611441067207574945792000000000000000000000000/],
           "falling_factorial selected values");

###### rising_factorial
{
  my @s;
  for my $k (0..10) { push @s, map { rising_factorial($_,$k) } -10..10; }
  is_deeply( \@s,
             [qw/1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 -10 -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 10 90 72 56 42 30 20 12 6 2 0 0 2 6 12 20 30 42 56 72 90 110 -720 -504 -336 -210 -120 -60 -24 -6 0 0 0 6 24 60 120 210 336 504 720 990 1320 5040 3024 1680 840 360 120 24 0 0 0 0 24 120 360 840 1680 3024 5040 7920 11880 17160 -30240 -15120 -6720 -2520 -720 -120 0 0 0 0 0 120 720 2520 6720 15120 30240 55440 95040 154440 240240 151200 60480 20160 5040 720 0 0 0 0 0 0 720 5040 20160 60480 151200 332640 665280 1235520 2162160 3603600 -604800 -181440 -40320 -5040 0 0 0 0 0 0 0 5040 40320 181440 604800 1663200 3991680 8648640 17297280 32432400 57657600 1814400 362880 40320 0 0 0 0 0 0 0 0 40320 362880 1814400 6652800 19958400 51891840 121080960 259459200 518918400 980179200 -3628800 -362880 0 0 0 0 0 0 0 0 0 362880 3628800 19958400 79833600 259459200 726485760 1816214400 4151347200 8821612800 17643225600 3628800 0 0 0 0 0 0 0 0 0 0 3628800 39916800 239500800 1037836800 3632428800 10897286400 29059430400 70572902400 158789030400 335221286400/],
  "rising_factorial(-10..10, 0..10)" );
}
is_deeply( [map { rising_factorial($_->[0],$_->[1]) } ([509,7],[510,7],[562,7],[80,10],[103,101])],
           [qw/9222879462222182400 9349716704335257600 18378924259448108160 18452514066426316800 6760937240727169751346751449031021029092236987417146776093364751481076175432048515956305908925637116481562056123160956910787676051553407749205364947724300581490631820332063331242347041889126973440000000000000000000000000/],
           "rising_factorial selected values");
