package Map::Tube::Rome;

use 5.010;
use strict;
use warnings;

use File::ShareDir qw(dist_file);
use Moo;
use namespace::autoclean;

$Map::Tube::Rome::VERSION   = '1.01';
$Map::Tube::Rome::AUTHORITY = 'cpan:GDT';

has json => (is => 'ro', default => sub { dist_file('Map-Tube-Rome', 'rome-map.json') });

with 'Map::Tube';

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Map::Tube::Rome - Interface to the Rome tube map

=head1 VERSION

version 1.00

=head1 SYNOPSIS

    use Map::Tube::Rome;
    my $tube = Map::Tube::Rome->new();
 
    my $route = $tube->get_shortest_route('Colosseo', 'Anagnina');
 
    print "Route: $route\n";

=head1 DESCRIPTION

This module allows to find the shortest route between any two given tube
stations in Rome. All interesting methods are provided by the role L<Map::Tube>.

=head1 METHODS

=head2 CONSTRUCTOR

    use Map::Tube::Rome;
    my $tube = Map::Tube::Rome->new();

The only argument, C<json>, is optional; if specified, it should be a code ref
to a function that returns either the path the JSON map file, or a string
containing this JSON content. The default is the path to F<rome.json>
that is a part of this distribution. For further information see L<Map::Tube>.

=head2 json()

This read-only accessor returns whatever was specified as the JSON source at
construction.
 
=head2 get_shortest_route($from, $to)
 
It expects C<$from> and C<$to> station name, required param. It returns an object
of type L<Map::Tube::Route>. On error it throws exception of type L<Map::Tube::Exception>.
 
    use strict; use warnings;
    use Map::Tube::Rome;
 
    my $tube  = Map::Tube::Rome->new;
    my $route = $tube->get_shortest_route('Colosseo', 'Anagnina');
 
    print "Route: $route\n";
 
=head2 as_image($line_name)
 
It expects the plugin  L<Map::Tube::Plugin::Graph> to be  installed. Returns line
image  as  base64  encoded string if C<$line_name> passed in otherwise it returns
base64 encoded string of the entire map.
 
    use strict; use warnings;
    use MIME::Base64;
    use Map::Tube::Rome;
 
    my $tube = Map::Tube::Rome->new;
    my $map  = $tube->name;
    open(my $IMAGE, ">", "$map.png");
       or die "ERROR: Can't open [$map.png]: $!";
    binmode($IMAGE);
    print $IMAGE decode_base64($tube->as_image);
    close($IMAGE);
 
The L<Rome Tube Map|https://raw.githubusercontent.com/giterlizzi/perl-Map-Tube-Rome/main/maps/Metropolitana-di-Roma.png>
as generated by plugin L<Map::Tube::Plugin::Graph>.
 
=begin html
 
<a href = "https://raw.githubusercontent.com/giterlizzi/perl-Map-Tube-Rome/main/maps/Metropolitana-di-Roma.png">
<img src = "https://raw.githubusercontent.com/giterlizzi/perl-Map-Tube-Rome/main/maps/Metropolitana-di-Roma.png"
     alt = "Rome Tube Map" />
</a>
 
=end html

=head1 ERRORS

If something goes wrong, maybe because the map information file was corrupted,
the constructor will die.

=head1 SEE ALSO

L<Map::Tube>, L<Map::Tube::GraphViz>.

=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/giterlizzi/perl-Map-Tube-Rome/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/giterlizzi/perl-Map-Tube-Rome>

    git clone https://github.com/giterlizzi/perl-Map-Tube-Rome.git

=head1 AUTHOR

Giuseppe Di Terlizzi <gdt@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2024 by Giuseppe Di Terlizzi <gdt@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
