"use strict";
// Copyright 2021-2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateDts = void 0;
const ecmascript_1 = require("@bufbuild/protoplugin/ecmascript");
const editions_js_1 = require("./editions.js");
const util_js_1 = require("./util.js");
function generateDts(schema) {
    for (const file of schema.files) {
        const f = schema.generateFile(file.name + "_pb.d.ts");
        f.preamble(file);
        for (const enumeration of file.enums) {
            generateEnum(schema, f, enumeration);
        }
        for (const message of file.messages) {
            generateMessage(schema, f, message);
        }
        for (const extension of file.extensions) {
            generateExtension(schema, f, extension);
        }
        // We do not generate anything for services
    }
}
exports.generateDts = generateDts;
// prettier-ignore
function generateEnum(schema, f, enumeration) {
    f.print(f.jsDoc(enumeration));
    f.print(f.exportDecl("declare enum", (0, ecmascript_1.localName)(enumeration)), " {");
    for (const value of enumeration.values) {
        if (enumeration.values.indexOf(value) > 0) {
            f.print();
        }
        f.print(f.jsDoc(value, "  "));
        f.print("  ", (0, ecmascript_1.localName)(value), " = ", value.number, ",");
    }
    f.print("}");
    f.print();
}
// prettier-ignore
function generateMessage(schema, f, message) {
    const protoN = (0, editions_js_1.getNonEditionRuntime)(schema, message.file);
    const { PartialMessage, FieldList, Message, PlainMessage, BinaryReadOptions, JsonReadOptions, JsonValue } = schema.runtime;
    const m = (0, ecmascript_1.localName)(message);
    f.print(f.jsDoc(message));
    f.print(f.exportDecl("declare class", m), " extends ", Message, "<", m, "> {");
    for (const member of message.members) {
        switch (member.kind) {
            case "oneof":
                generateOneof(schema, f, member);
                break;
            default:
                generateField(schema, f, member);
                break;
        }
        f.print();
    }
    f.print("  constructor(data?: ", PartialMessage, "<", m, ">);");
    f.print();
    generateWktMethods(schema, f, message);
    f.print("  static readonly runtime: typeof ", protoN, ";");
    f.print('  static readonly typeName = ', f.string(message.typeName), ';');
    f.print("  static readonly fields: ", FieldList, ";");
    // In case we start supporting options, we have to surface them here
    //f.print("  static readonly options: { readonly [extensionName: string]: ", rt.JsonValue, " } = {};")
    f.print();
    generateWktStaticMethods(schema, f, message);
    f.print("  static fromBinary(bytes: Uint8Array, options?: Partial<", BinaryReadOptions, ">): ", m, ";");
    f.print();
    f.print("  static fromJson(jsonValue: ", JsonValue, ", options?: Partial<", JsonReadOptions, ">): ", m, ";");
    f.print();
    f.print("  static fromJsonString(jsonString: string, options?: Partial<", JsonReadOptions, ">): ", m, ";");
    f.print();
    f.print("  static equals(a: ", m, " | ", PlainMessage, "<", m, "> | undefined, b: ", m, " | ", PlainMessage, "<", m, "> | undefined): boolean;");
    f.print("}");
    f.print();
    for (const nestedEnum of message.nestedEnums) {
        generateEnum(schema, f, nestedEnum);
    }
    for (const nestedMessage of message.nestedMessages) {
        generateMessage(schema, f, nestedMessage);
    }
    for (const nestedExtension of message.nestedExtensions) {
        generateExtension(schema, f, nestedExtension);
    }
}
// prettier-ignore
function generateOneof(schema, f, oneof) {
    f.print(f.jsDoc(oneof, "  "));
    f.print("  ", (0, ecmascript_1.localName)(oneof), ": {");
    for (const field of oneof.fields) {
        if (oneof.fields.indexOf(field) > 0) {
            f.print(`  } | {`);
        }
        f.print(f.jsDoc(field, "    "));
        const { typing } = (0, util_js_1.getFieldTypeInfo)(field);
        f.print(`    value: `, typing, `;`);
        f.print(`    case: "`, (0, ecmascript_1.localName)(field), `";`);
    }
    f.print(`  } | { case: undefined; value?: undefined };`);
}
// prettier-ignore
function generateField(schema, f, field) {
    f.print(f.jsDoc(field, "  "));
    const e = [];
    const { typing, optional } = (0, util_js_1.getFieldTypeInfo)(field);
    if (!optional) {
        e.push("  ", (0, ecmascript_1.localName)(field), ": ", typing, ";");
    }
    else {
        e.push("  ", (0, ecmascript_1.localName)(field), "?: ", typing, ";");
    }
    f.print(e);
}
// prettier-ignore
function generateExtension(schema, f, ext) {
    const { typing } = (0, util_js_1.getFieldTypeInfo)(ext);
    const e = f.import(ext.extendee).toTypeOnly();
    f.print(f.jsDoc(ext));
    f.print(f.exportDecl("declare const", (0, ecmascript_1.localName)(ext)), ": ", schema.runtime.Extension, "<", e, ", ", typing, ">;");
    f.print();
}
// prettier-ignore
function generateWktMethods(schema, f, message) {
    const ref = (0, ecmascript_1.reifyWkt)(message);
    if (ref === undefined) {
        return;
    }
    const { Message, MessageType, IMessageTypeRegistry } = schema.runtime;
    switch (ref.typeName) {
        case "google.protobuf.Any":
            f.print("  packFrom(message: ", Message, "): void;");
            f.print();
            f.print("  unpackTo(target: ", Message, "): boolean;");
            f.print();
            f.print("  unpack(registry: ", IMessageTypeRegistry, "): Message | undefined;");
            f.print();
            f.print("  is(type: ", MessageType, " | string): boolean;");
            f.print();
            f.print("  private typeNameToUrl(name: string): string;");
            f.print();
            f.print("  private typeUrlToName(url: string): string;");
            f.print();
            break;
        case "google.protobuf.Timestamp":
            f.print("  toDate(): Date;");
            f.print();
            break;
        case "google.protobuf.Duration":
        case "google.protobuf.Struct":
        case "google.protobuf.Value":
        case "google.protobuf.ListValue":
        case "google.protobuf.FieldMask":
        case "google.protobuf.DoubleValue":
        case "google.protobuf.FloatValue":
        case "google.protobuf.Int64Value":
        case "google.protobuf.UInt64Value":
        case "google.protobuf.Int32Value":
        case "google.protobuf.UInt32Value":
        case "google.protobuf.BoolValue":
        case "google.protobuf.StringValue":
        case "google.protobuf.BytesValue":
            break;
    }
}
// prettier-ignore
function generateWktStaticMethods(schema, f, message) {
    const ref = (0, ecmascript_1.reifyWkt)(message);
    if (ref === undefined) {
        return;
    }
    switch (ref.typeName) {
        case "google.protobuf.Any":
            f.print("  static pack(message: Message): ", message, ";");
            f.print();
            break;
        case "google.protobuf.Timestamp":
            f.print("  static now(): ", message, ";");
            f.print();
            f.print("  static fromDate(date: Date): ", message, ";");
            f.print();
            break;
        case "google.protobuf.DoubleValue":
        case "google.protobuf.FloatValue":
        case "google.protobuf.Int64Value":
        case "google.protobuf.UInt64Value":
        case "google.protobuf.Int32Value":
        case "google.protobuf.UInt32Value":
        case "google.protobuf.BoolValue":
        case "google.protobuf.StringValue":
        case "google.protobuf.BytesValue": {
            const { typing } = (0, util_js_1.getFieldTypeInfo)(ref.value);
            f.print("  static readonly fieldWrapper: {");
            f.print("    wrapField(value: ", typing, "): ", message, ",");
            f.print("    unwrapField(value: ", message, "): ", typing, ",");
            f.print("  };");
            f.print();
            break;
        }
        case "google.protobuf.Duration":
        case "google.protobuf.Struct":
        case "google.protobuf.Value":
        case "google.protobuf.ListValue":
        case "google.protobuf.FieldMask":
            break;
    }
}
