package WebService::Pixabay;

use Modern::Perl '2010';
use Moo;
use Function::Parameters;
use Data::Printer;

with 'WebService::Client';
# ABSTRACT: Perl 5 interface to Pixabay API.
our $VERSION = '1.0.0'; our $VERSION = '1.0.0'; # VERSION

# token key
has api_key =>
(
	is => 'ro',
	required => 1
);

# object for holding json data
has json_data => 
(
	is => 'rw'
);

has '+base_url' =>
(
	default => 'https://pixabay.com/api/'
);

# get image metadata from JSON
method image_search
(
	:$q = "yellow flower", :$lang = "en", :$id = "", :$response_group = "image_details",
	:$image_type = "all", :$orientation = "all", :$category = "", :$min_width = 0, :$min_height = 0,
	:$editors_choice = "false", :$safesearch = "false", :$order = "popular",
	:$page = 1, :$per_page = 20, :$callback = "", :$pretty = "false"
)
{
	$self->json_data($self->get
	(
		"?key=" . $self->api_key .
						 "&q=$q&lang=$lang&id=$id&response_group=$response_group&image_type=$image_type" .
						 "&orientation=$orientation&category=$category&min_width=$min_width&mind_height=$min_height" .
					     "&editors_choice=$editors_choice&safesearch=$safesearch&order=$order&page=$page" .
					     "&per_page=$per_page&callback=$callback&pretty=$pretty"
	));

	return $self;
}

# get image metadata from JSON
method video_search
(	:$q = "yellow flower", :$lang = "en", :$id = "", :$video_type = "all",
	:$category = "", :$min_width = 0, :$min_height = 0,
	:$editors_choice = "false", :$safesearch = "false", :$order = "popular",
	:$page = 1, :$per_page = 20, :$callback = "", :$pretty = "false"
)
{
	$self->json_data($self->get
	(
		"videos/?key=" . $self->api_key .
						 "&q=$q&lang=$lang&id=$id&video_type=$video_type" .
					     "&category=$category&min_width=$min_width&mind_height=$min_height" .
					     "&editors_choice=$editors_choice&safesearch=$safesearch&order=$order&page=$page" .
					     "&per_page=$per_page&callback=$callback&pretty=$pretty"
	));

	return $self;
}

# pretty print the JSON data structure
method show_data_structure()
{
	return p $self->json_data;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

WebService::Pixabay - Perl 5 interface to Pixabay API.

=head1 VERSION

version 1.0.0

=head1 SYNOPSIS

	use WebService::Pixabay;
	
	my $pix = WebService::Pixabay->new(api_key => 'secret');
	
	 default searches
	my $img_search = $pix->image_search();
	my $vid_search = $pix->video_search();
	
	# show data structure using Data::Printer pretty print.
	$img_search->show_data_structure;
	$vid_search->show_data_structure;
	
	###################################################
	# The parameters of the method have the same name #
	# and default values as in Pixabay API docs       #
	###################################################

	# example custom image search and printing
	my $nis = $pix->image_search(
		q => 'cats dog',
		lang => 'es',
		response_group => 'high_resolution',
		image_type => 'illustration',
		category => 'animals',
		safesearch => 'true',
		order => 'latest',
		page => 2,
		per_page => 5,
		pretty => 'true'
	);

	$nis->show_data_structure;

	# -or-

	# example custom video search and printing
	my $nvs = $pix->video_search(
		q => 'tree',
		lang => 'en',
		pretty => 'false',
		page => 3,
		order => 'popular'
	)->show_data_structure;

	# Handling specific hash and arrays of values from the JSON
	# example retrieving webformatURL from each arrays
	my @urls = undef;
	
	# note the 'json_data' object usage
	foreach my $url (@{$nis->json_data->{hits}}) {
		# now has link of photo urls (non-preview photos)
		push(@urls, $url->{webformatURL});		
	}
	
	say $urls[3]; # image URL

	# Getting a specific single key or array value
	say $nis->image_search(q => 'cloud')
		->json_data
		->{hits}[2]{webformatURL};

=head1 NAME

WebService::Pixabay - Perl 5 interface to Pixabay API.

=head1 VERSION

version 0.1.0

=head1 SEE ALSO

L<Pixabay API documentations|https://pixabay.com/api/docs>

L<Moo>

L<Function::Parameters>

L<Test::More>

L<WebService::Client>

L<LWP::Online>

L<Data::Printer>

=head1 AUTHOR

faraco <skelic3@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by faraco.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 AUTHOR

faraco <skelic3@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by faraco.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
