package Char;
use strict;
BEGIN { $INC{'warnings.pm'} = '' if $] < 5.006 } use warnings;
######################################################################
#
# Char - Native Encoding Support by Traditional Scripting
#
# http://search.cpan.org/dist/Char/
my $copyright =
'Copyright (c) 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2018, 2019 INABA Hitoshi <ina@cpan.org>';
######################################################################

use 5.00503;    # Galapagos Consensus 1998 for primetools
# use 5.008001; # Lancaster Consensus 2013 for toolchains

# Version numbers should be boring
# http://www.dagolden.com/index.php/369/version-numbers-should-be-boring/
# For the impatient, the disinterested or those who just want to follow
# a recipe, my advice for all modules is this:
# our $VERSION = "0.001"; # or "0.001_001" for a dev release
# $VERSION = CORE::eval $VERSION; # No!! because '1.10' makes '1.1'

use vars qw($VERSION);
$VERSION = '1.19';
$VERSION = $VERSION;

# P.714 29.2.39. flock
# in Chapter 29: Functions
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.863 flock
# in Chapter 27: Functions
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

# P.228 Inlining Constant Functions
# in Chapter 6: Subroutines
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.331 Inlining Constant Functions
# in Chapter 7: Subroutines
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

sub LOCK_SH() {1}
sub LOCK_EX() {2}
sub LOCK_UN() {8}
sub LOCK_NB() {4}

local $^W = 1;
$| = 1;

sub unimport {}

my($OSNAME, $LANG) = ();
($OSNAME, $LANG) = ($^O,       $ENV{'LANG'})  if 1;
($OSNAME, $LANG) = ('MSWin32', undef)         if 0;
($OSNAME, $LANG) = ('darwin',  'ja_JP.UTF-8') if 0;
($OSNAME, $LANG) = ('MacOS',   undef)         if 0;
($OSNAME, $LANG) = ('solaris', 'ja')          if 0;
($OSNAME, $LANG) = ('hpux',    'SJIS')        if 0;
($OSNAME, $LANG) = ('aix',     'Ja_JP')       if 0;

#
# instead of Symbol.pm
#
BEGIN {
    sub gensym () {
        if ($] < 5.006) {
            return \do { local *_ };
        }
        else {
            return undef;
        }
    }
}

#
# source code filter
#
sub import {
    my(undef,$filename) = caller 0;

    # when escaped script not exists or older
    my $e_mtime = (stat("$filename.e"))[9];
    my $mtime   = (stat($filename))[9];
    if ((not -e "$filename.e") or ($e_mtime < $mtime)) {

        # select filter software
        my $encoding = '';
        my $filter = '';
        if ($encoding = (from_magic_comment($filename) || from_chcp_lang())) {
            $filter = abspath($encoding);

            #----------------------------------------------------
            #  older >
            #  newer >>>>>
            #----------------------------------------------------
            # __FILE__ >>>>>
            # Filter   (nothing) needs making Filter
            #
            # __FILE__ >>>>>
            # Filter   >         needs re-make Filter (Filter is old)
            #----------------------------------------------------

            if (($filter eq '') or ((stat $filter)[9] < (stat __FILE__)[9])) {

                # make filter software
                make_filter_software($encoding);

                $filter = abspath($encoding);
                if ($filter eq '') {
                    die __FILE__, ": filter software '$encoding.pm' not found in \@INC(@INC)\n";
                }
            }
        }
        else {
            die __FILE__, ": don't know which encoding.\n";
        }

        # escape perl script
        my $fh1 = gensym();
        my $q_filter   = '';
        my $q_filename = '';
        if ($OSNAME =~ m/\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
            $q_filter   = _escapeshellcmd_MSWin32($filter);
            $q_filename = _escapeshellcmd_MSWin32($filename);
        }
        elsif ($OSNAME eq 'MacOS') {
            $q_filter   = _escapeshellcmd_MacOS($filter);
            $q_filename = _escapeshellcmd_MacOS($filename);
        }
        else {
            $q_filter   = _escapeshellcmd($filter);
            $q_filename = _escapeshellcmd($filename);
        }
        open($fh1, qq{$^X $q_filter $q_filename | }) or die __FILE__, ": Can't open pipe: $^X $q_filter $q_filename |\n";

        local *FH2;
        open(FH2,">$filename.e") or die __FILE__, ": Can't write open file: $filename.e\n";

        # 7.19. Flushing Output
        # in Chapter 7. File Access
        # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.

        select((select(FH2), $|=1)[0]);

        # flock script file
        if (0) {
        }
        elsif ($^O eq 'MacOS') {
            CORE::eval q{
                CORE::require Mac::Files;
                Mac::Files::FSpSetFLock("$filename.e");
            };
        }
        elsif (exists $ENV{'CHAR_NONBLOCK'}) {

            # P.419 File Locking
            # in Chapter 16: Interprocess Communication
            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

            # P.524 File Locking
            # in Chapter 15: Interprocess Communication
            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

            # P.571 Handling Race Conditions
            # in Chapter 23: Security
            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

            # P.663 Handling Race Conditions
            # in Chapter 20: Security
            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

            # (and so on)

            CORE::eval q{ flock(FH2, LOCK_EX | LOCK_NB) };
            if ($@) {
                die __FILE__, ": Can't immediately write-lock the file: $filename.e\n";
            }
        }
        else {
            CORE::eval q{ flock(FH2, LOCK_EX) };
        }

        CORE::eval q{ truncate(FH2, 0) };
        seek(FH2, 0, 0) or die __FILE__, ": Can't seek file: $filename.e\n";

        # escape perl script
        while (<$fh1>) {
            print FH2 $_;
        }
        close($fh1) or die "Can't close pipe: $^X $q_filter $q_filename | : $!";

        if ($^O eq 'MacOS') {
            CORE::eval q{
                CORE::require Mac::Files;
                Mac::Files::FSpRstFLock("$filename.e");
            };
        }
        close(FH2) or die "Can't close file: $filename.e: $!";

        # inherit file mode
        my $mode = (stat($filename))[2] & 0777;
        chmod $mode, "$filename.e";

        # close file and unlock
        if ($OSNAME eq 'MacOS') {
            CORE::eval q{
                require Mac::Files;
                Mac::Files::FSpRstFLock($filename);
            };
        }
    }

    my @switch = ();
    if ($^W) {
        push @switch, '-w';
    }
    if (defined $^I) {
        push @switch, '-i' . $^I;
        undef $^I;
    }

    my $system = 0;

    # execute escaped script
    if ($OSNAME =~ m/\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
        my $fh = gensym();
        _open_r($fh, $filename) or die __FILE__, ": Can't read open file: $filename\n";
        CORE::eval q{ flock($fh, LOCK_SH) };
        $system = _systemx(
            _escapeshellcmd_MSWin32($^X),

        # -I switch can not treat space included path
        #   (map { '-I' . _escapeshellcmd_MSWin32($_) } @INC),
            (map { '-I' .                         $_  } @INC),

            @switch,
            '--',
            map { _escapeshellcmd_MSWin32($_) } "$filename.e", @ARGV
        );
        close($fh) or die "Can't close file: $filename: $!";
    }
    elsif ($OSNAME eq 'MacOS') {
        CORE::eval q{
            require Mac::Files;
            Mac::Files::FSpSetFLock($filename);
        };
        $system = _systemx(
            _escapeshellcmd_MacOS($^X),
            (map { '-I' . _escapeshellcmd_MacOS($_) } @INC),
            @switch,
            '--',
            map { _escapeshellcmd_MacOS($_) } "$filename.e", @ARGV
        );
        CORE::eval q{
            require Mac::Files;
            Mac::Files::FSpRstFLock($filename);
        };
    }
    else {
        my $fh = gensym();
        _open_r($fh, $filename) or die __FILE__, ": Can't read open file: $filename\n";
        CORE::eval q{ flock($fh, LOCK_SH) };
        $system = _systemx(
            _escapeshellcmd($^X),
            (map { '-I' . _escapeshellcmd($_) } @INC),
            @switch,
            '--',
            map { _escapeshellcmd($_) } "$filename.e", @ARGV
        );
        close($fh) or die "Can't close file: $filename: $!";
    }

    # exit with actual exit value
    exit($system >> 8);
}

#
# open file in read mode
#
sub _open_r {
    my(undef,$file) = @_;
    use Fcntl qw(O_RDONLY);
    return CORE::sysopen($_[0], $file, &O_RDONLY);
}

#
# safe system
#
sub _systemx {
    $| = 1;
    local @ENV{qw(IFS CDPATH ENV BASH_ENV)}; # Make %ENV safer
#   return CORE::system { $_[0] } @_;
    return CORE::system           @_;
}

#
# escape shell command line on DOS-like system
#
sub _escapeshellcmd_MSWin32 {
    my($word) = @_;
    if ($word =~ / [ ] /oxms) {
        return qq{"$word"};
    }
    else {
        return $word;
    }
}

#
# escape shell command line on Mac OS
#
sub _escapeshellcmd_MacOS {
    my($word) = @_;
    return $word;
}

#
# escape shell command line on UNIX-like system
#
sub _escapeshellcmd {
    my($word) = @_;
    return $word;
}

#
# get encoding from magic comment
#
sub from_magic_comment {
    my($filename) = @_;
    my $encoding = '';

    my $fh = gensym();
    _open_r($fh, $filename) or die __FILE__, ": Can't read open file: $filename\n";
    while (<$fh>) {
        chomp;

        # P.72 2.15 Prevent Runaway Repetition
        # in Chapter 2: Basic Regular Expression Skills
        # of ISBN 978-0-596-52068-7 Regular Expressions Cookbook
        # (and so on)

        # P.81 2.15 Prevent Runaway Repetition
        # in Chapter 2: Basic Regular Expression Skills
        # of ISBN 978-1-4493-1943-4 Regular Expressions Cookbook, 2nd Edition
        # (and so on)

        if (($encoding) = m/\A(?>\s*)[#].*?coding[:=](?>\s*)(.+)/oxms) {
            last;
        }
    }
    close($fh) or die "Can't close file: $filename: $!";
    return '' unless $encoding;

    # resolve alias of encoding
    $encoding = lc $encoding;
    $encoding =~ tr/a-z0-9//cd;
    return { qw(

    ascii               USASCII
    usascii             USASCII

    shiftjis            Sjis
    shiftjisx0213       Sjis
    shiftjis2004        Sjis
    sjis                Sjis
    sjisx0213           Sjis
    sjis2004            Sjis
    cp932               Sjis
    windows31j          Sjis
    cswindows31j        Sjis
    sjiswin             Sjis
    macjapanese         Sjis
    macjapan            Sjis
    xsjis               Sjis
    mskanji             Sjis
    csshiftjis          Sjis
    windowscodepage932  Sjis
    ibmcp943            Sjis
    ms932               Sjis

    jisc6220            JIS8
    jisx0201            JIS8
    jis8                JIS8
    ank                 JIS8

    eucjp               EUCJP
    euc                 EUCJP
    ujis                EUCJP
    eucjpms             EUCJP
    eucjpwin            EUCJP
    cp51932             EUCJP

    euctw               EUCTW

    utf8                UTF2
    utf2                UTF2
    utffss              UTF2
    utf8mac             UTF2

    oldutf8             OldUTF8
    cesu8               OldUTF8
    modifiedutf8        OldUTF8

    hp15                HP15
    informixv6als       INFORMIXV6ALS

    gb18030             GB18030
    gbk                 GBK
    gb2312              GBK
    cp936               GBK
    euccn               GBK

    uhc                 UHC
    ksx1001             UHC
    ksc5601             UHC
    ksc56011987         UHC
    ks                  UHC
    cp949               UHC
    windows949          UHC

    kps9566             KPS9566
    kps95662003         KPS9566
    kps95662000         KPS9566
    kps95661997         KPS9566
    kps956697           KPS9566
    euckp               KPS9566

    big5plus            Big5Plus
    big5                Big5Plus
    big5et              Big5Plus
    big5eten            Big5Plus
    tcabig5             Big5Plus
    cp950               Big5Plus

    big5hk              Big5HKSCS
    big5hkscs           Big5HKSCS
    hkbig5              Big5HKSCS
    hkscsbig5           Big5HKSCS
    cp951               Big5HKSCS

    latin1              Latin1
    isoiec88591         Latin1
    iso88591            Latin1
    iec88591            Latin1

    latin2              Latin2
    isoiec88592         Latin2
    iso88592            Latin2
    iec88592            Latin2

    latin3              Latin3
    isoiec88593         Latin3
    iso88593            Latin3
    iec88593            Latin3

    latin4              Latin4
    isoiec88594         Latin4
    iso88594            Latin4
    iec88594            Latin4

    cyrillic            Cyrillic
    isoiec88595         Cyrillic
    iso88595            Cyrillic
    iec88595            Cyrillic

    koi8r               KOI8R
    koi8u               KOI8U

    arabic              Arabic
    isoiec88596         Arabic
    iso88596            Arabic
    iec88596            Arabic

    greek               Greek
    isoiec88597         Greek
    iso88597            Greek
    iec88597            Greek

    hebrew              Hebrew
    isoiec88598         Hebrew
    iso88598            Hebrew
    iec88598            Hebrew

    latin5              Latin5
    isoiec88599         Latin5
    iso88599            Latin5
    iec88599            Latin5

    latin6              Latin6
    isoiec885910        Latin6
    iso885910           Latin6
    iec885910           Latin6

    tis620              TIS620
    tis6202533          TIS620
    isoiec885911        TIS620
    iso885911           TIS620
    iec885911           TIS620

    latin7              Latin7
    isoiec885913        Latin7
    iso885913           Latin7
    iec885913           Latin7

    latin8              Latin8
    isoiec885914        Latin8
    iso885914           Latin8
    iec885914           Latin8

    latin9              Latin9
    isoiec885915        Latin9
    iso885915           Latin9
    iec885915           Latin9

    latin10             Latin10
    isoiec885916        Latin10
    iso885916           Latin10
    iec885916           Latin10

    windows1252         Windows1252

    windows1258         Windows1258

    )}->{$encoding} || $encoding;
}

#
# encoding from chcp or LANG environment variable
#
sub from_chcp_lang {
    my $encoding = '';

    # Microsoft Windows
    if ($OSNAME eq 'MSWin32') {
        $encoding = {

        # Code Page Identifiers (Windows)
        # Identifier .NET Name Additional information

          '037' => '', # IBM037 IBM EBCDIC US-Canada
          '437' => '', # IBM437 OEM United States
          '500' => '', # IBM500 IBM EBCDIC International
          '708' => 'Arabic', # ASMO-708 Arabic (ASMO 708)
          '709' => '', #  Arabic (ASMO-449+, BCON V4)
          '710' => '', #  Arabic - Transparent Arabic
          '720' => '', # DOS-720 Arabic (Transparent ASMO); Arabic (DOS)
          '737' => '', # ibm737 OEM Greek (formerly 437G); Greek (DOS)
          '775' => '', # ibm775 OEM Baltic; Baltic (DOS)
          '850' => '', # ibm850 OEM Multilingual Latin 1; Western European (DOS)
          '852' => '', # ibm852 OEM Latin 2; Central European (DOS)
          '855' => '', # IBM855 OEM Cyrillic (primarily Russian)
          '857' => '', # ibm857 OEM Turkish; Turkish (DOS)
          '858' => '', # IBM00858 OEM Multilingual Latin 1 + Euro symbol
          '860' => '', # IBM860 OEM Portuguese; Portuguese (DOS)
          '861' => '', # ibm861 OEM Icelandic; Icelandic (DOS)
          '862' => '', # DOS-862 OEM Hebrew; Hebrew (DOS)
          '863' => '', # IBM863 OEM French Canadian; French Canadian (DOS)
          '864' => '', # IBM864 OEM Arabic; Arabic (864)
          '865' => '', # IBM865 OEM Nordic; Nordic (DOS)
          '866' => '', # cp866 OEM Russian; Cyrillic (DOS)
          '869' => '', # ibm869 OEM Modern Greek; Greek, Modern (DOS)
          '870' => '', # IBM870 IBM EBCDIC Multilingual/ROECE (Latin 2); IBM EBCDIC Multilingual Latin 2
          '874' => 'TIS620', # windows-874 ANSI/OEM Thai (same as 28605, ISO 8859-15); Thai (Windows)
          '875' => '', # cp875 IBM EBCDIC Greek Modern
          '932' => 'Sjis', # shift_jis ANSI/OEM Japanese; Japanese (Shift-JIS)
          '936' => 'GBK', # gb2312 ANSI/OEM Simplified Chinese (PRC, Singapore); Chinese Simplified (GB2312)
          '949' => 'UHC', # ks_c_5601-1987 ANSI/OEM Korean (Unified Hangul Code)
          '950' => 'Big5Plus', # big5 ANSI/OEM Traditional Chinese (Taiwan; Hong Kong SAR, PRC); Chinese Traditional (Big5)
          '951' => 'Big5HKSCS', # HKSCS support on top of traditional Chinese Windows
         '1026' => '', # IBM1026 IBM EBCDIC Turkish (Latin 5)
         '1047' => '', # IBM01047 IBM EBCDIC Latin 1/Open System
         '1140' => '', # IBM01140 IBM EBCDIC US-Canada (037 + Euro symbol); IBM EBCDIC (US-Canada-Euro)
         '1141' => '', # IBM01141 IBM EBCDIC Germany (20273 + Euro symbol); IBM EBCDIC (Germany-Euro)
         '1142' => '', # IBM01142 IBM EBCDIC Denmark-Norway (20277 + Euro symbol); IBM EBCDIC (Denmark-Norway-Euro)
         '1143' => '', # IBM01143 IBM EBCDIC Finland-Sweden (20278 + Euro symbol); IBM EBCDIC (Finland-Sweden-Euro)
         '1144' => '', # IBM01144 IBM EBCDIC Italy (20280 + Euro symbol); IBM EBCDIC (Italy-Euro)
         '1145' => '', # IBM01145 IBM EBCDIC Latin America-Spain (20284 + Euro symbol); IBM EBCDIC (Spain-Euro)
         '1146' => '', # IBM01146 IBM EBCDIC United Kingdom (20285 + Euro symbol); IBM EBCDIC (UK-Euro)
         '1147' => '', # IBM01147 IBM EBCDIC France (20297 + Euro symbol); IBM EBCDIC (France-Euro)
         '1148' => '', # IBM01148 IBM EBCDIC International (500 + Euro symbol); IBM EBCDIC (International-Euro)
         '1149' => '', # IBM01149 IBM EBCDIC Icelandic (20871 + Euro symbol); IBM EBCDIC (Icelandic-Euro)
         '1200' => '', # utf-16 Unicode UTF-16, little endian byte order (BMP of ISO 10646); available only to managed applications
         '1201' => '', # unicodeFFFE Unicode UTF-16, big endian byte order; available only to managed applications
         '1250' => '', # windows-1250 ANSI Central European; Central European (Windows)
         '1251' => '', # windows-1251 ANSI Cyrillic; Cyrillic (Windows)
         '1252' => 'Windows1252', # windows-1252 ANSI Latin 1; Western European (Windows)
         '1253' => '', # windows-1253 ANSI Greek; Greek (Windows)
         '1254' => '', # windows-1254 ANSI Turkish; Turkish (Windows)
         '1255' => 'Hebrew', # windows-1255 ANSI Hebrew; Hebrew (Windows)
         '1256' => '', # windows-1256 ANSI Arabic; Arabic (Windows)
         '1257' => '', # windows-1257 ANSI Baltic; Baltic (Windows)
         '1258' => 'Windows1258', # windows-1258 ANSI/OEM Vietnamese; Vietnamese (Windows)
         '1361' => '', # Johab Korean (Johab)
        '10000' => '', # macintosh MAC Roman; Western European (Mac)
        '10001' => '', # x-mac-japanese Japanese (Mac)
        '10002' => '', # x-mac-chinesetrad MAC Traditional Chinese (Big5); Chinese Traditional (Mac)
        '10003' => '', # x-mac-korean Korean (Mac)
        '10004' => '', # x-mac-arabic Arabic (Mac)
        '10005' => '', # x-mac-hebrew Hebrew (Mac)
        '10006' => '', # x-mac-greek Greek (Mac)
        '10007' => '', # x-mac-cyrillic Cyrillic (Mac)
        '10008' => '', # x-mac-chinesesimp MAC Simplified Chinese (GB 2312); Chinese Simplified (Mac)
        '10010' => '', # x-mac-romanian Romanian (Mac)
        '10017' => '', # x-mac-ukrainian Ukrainian (Mac)
        '10021' => '', # x-mac-thai Thai (Mac)
        '10029' => '', # x-mac-ce MAC Latin 2; Central European (Mac)
        '10079' => '', # x-mac-icelandic Icelandic (Mac)
        '10081' => '', # x-mac-turkish Turkish (Mac)
        '10082' => '', # x-mac-croatian Croatian (Mac)
        '12000' => '', # utf-32 Unicode UTF-32, little endian byte order; available only to managed applications
        '12001' => '', # utf-32BE Unicode UTF-32, big endian byte order; available only to managed applications
        '20000' => '', # x-Chinese_CNS CNS Taiwan; Chinese Traditional (CNS)
        '20001' => '', # x-cp20001 TCA Taiwan
        '20002' => '', # x_Chinese-Eten Eten Taiwan; Chinese Traditional (Eten)
        '20003' => '', # x-cp20003 IBM5550 Taiwan
        '20004' => '', # x-cp20004 TeleText Taiwan
        '20005' => '', # x-cp20005 Wang Taiwan
        '20105' => '', # x-IA5 IA5 (IRV International Alphabet No. 5, 7-bit); Western European (IA5)
        '20106' => '', # x-IA5-German IA5 German (7-bit)
        '20107' => '', # x-IA5-Swedish IA5 Swedish (7-bit)
        '20108' => '', # x-IA5-Norwegian IA5 Norwegian (7-bit)
        '20127' => 'USASCII', # us-ascii US-ASCII (7-bit)
        '20261' => '', # x-cp20261 T.61
        '20269' => '', # x-cp20269 ISO 6937 Non-Spacing Accent
        '20273' => '', # IBM273 IBM EBCDIC Germany
        '20277' => '', # IBM277 IBM EBCDIC Denmark-Norway
        '20278' => '', # IBM278 IBM EBCDIC Finland-Sweden
        '20280' => '', # IBM280 IBM EBCDIC Italy
        '20284' => '', # IBM284 IBM EBCDIC Latin America-Spain
        '20285' => '', # IBM285 IBM EBCDIC United Kingdom
        '20290' => '', # IBM290 IBM EBCDIC Japanese Katakana Extended
        '20297' => '', # IBM297 IBM EBCDIC France
        '20420' => '', # IBM420 IBM EBCDIC Arabic
        '20423' => '', # IBM423 IBM EBCDIC Greek
        '20424' => '', # IBM424 IBM EBCDIC Hebrew
        '20833' => '', # x-EBCDIC-KoreanExtended IBM EBCDIC Korean Extended
        '20838' => '', # IBM-Thai IBM EBCDIC Thai
        '20866' => 'KOI8R', # koi8-r Russian (KOI8-R); Cyrillic (KOI8-R)
        '20871' => '', # IBM871 IBM EBCDIC Icelandic
        '20880' => '', # IBM880 IBM EBCDIC Cyrillic Russian
        '20905' => '', # IBM905 IBM EBCDIC Turkish
        '20924' => '', # IBM00924 IBM EBCDIC Latin 1/Open System (1047 + Euro symbol)
        '20932' => 'EUCJP', # EUC-JP Japanese (JIS 0208-1990 and 0121-1990)
        '20936' => '', # x-cp20936 Simplified Chinese (GB2312); Chinese Simplified (GB2312-80)
        '20949' => '', # x-cp20949 Korean Wansung
        '21025' => '', # cp1025 IBM EBCDIC Cyrillic Serbian-Bulgarian
        '21027' => '', #  (deprecated)
        '21866' => 'KOI8U', # koi8-u Ukrainian (KOI8-U); Cyrillic (KOI8-U)
        '28591' => 'Latin1', # iso-8859-1 ISO 8859-1 Latin 1; Western European (ISO)
        '28592' => 'Latin2', # iso-8859-2 ISO 8859-2 Central European; Central European (ISO)
        '28593' => 'Latin3', # iso-8859-3 ISO 8859-3 Latin 3
        '28594' => 'Latin4', # iso-8859-4 ISO 8859-4 Baltic
        '28595' => 'Cyrillic', # iso-8859-5 ISO 8859-5 Cyrillic
        '28596' => 'Arabic', # iso-8859-6 ISO 8859-6 Arabic
        '28597' => 'Greek', # iso-8859-7 ISO 8859-7 Greek
        '28598' => 'Hebrew', # iso-8859-8 ISO 8859-8 Hebrew; Hebrew (ISO-Visual)
        '28599' => 'Latin5', # iso-8859-9 ISO 8859-9 Turkish
        '28603' => 'Latin7', # iso-8859-13 ISO 8859-13 Estonian
        '28605' => 'Latin9', # iso-8859-15 ISO 8859-15 Latin 9
        '29001' => '', # x-Europa Europa 3
        '38598' => '', # iso-8859-8-i ISO 8859-8 Hebrew; Hebrew (ISO-Logical)
        '50220' => '', # iso-2022-jp ISO 2022 Japanese with no halfwidth Katakana; Japanese (JIS)
        '50221' => '', # csISO2022JP ISO 2022 Japanese with halfwidth Katakana; Japanese (JIS-Allow 1 byte Kana)
        '50222' => '', # iso-2022-jp ISO 2022 Japanese JIS X 0201-1989; Japanese (JIS-Allow 1 byte Kana - SO/SI)
        '50225' => '', # iso-2022-kr ISO 2022 Korean
        '50227' => '', # x-cp50227 ISO 2022 Simplified Chinese; Chinese Simplified (ISO 2022)
        '50229' => '', #  ISO 2022 Traditional Chinese
        '50930' => '', #  EBCDIC Japanese (Katakana) Extended
        '50931' => '', #  EBCDIC US-Canada and Japanese
        '50933' => '', #  EBCDIC Korean Extended and Korean
        '50935' => '', #  EBCDIC Simplified Chinese Extended and Simplified Chinese
        '50936' => '', #  EBCDIC Simplified Chinese
        '50937' => '', #  EBCDIC US-Canada and Traditional Chinese
        '50939' => '', #  EBCDIC Japanese (Latin) Extended and Japanese
        '51932' => 'EUCJP', # euc-jp EUC Japanese
        '51936' => '', # EUC-CN EUC Simplified Chinese; Chinese Simplified (EUC)
        '51949' => '', # euc-kr EUC Korean
        '51950' => '', #  EUC Traditional Chinese
        '52936' => '', # hz-gb-2312 HZ-GB2312 Simplified Chinese; Chinese Simplified (HZ)
        '54936' => 'GB18030', # GB18030 Windows XP and later: GB18030 Simplified Chinese (4 byte); Chinese Simplified (GB18030)
        '57002' => '', # x-iscii-de ISCII Devanagari
        '57003' => '', # x-iscii-be ISCII Bengali
        '57004' => '', # x-iscii-ta ISCII Tamil
        '57005' => '', # x-iscii-te ISCII Telugu
        '57006' => '', # x-iscii-as ISCII Assamese
        '57007' => '', # x-iscii-or ISCII Oriya
        '57008' => '', # x-iscii-ka ISCII Kannada
        '57009' => '', # x-iscii-ma ISCII Malayalam
        '57010' => '', # x-iscii-gu ISCII Gujarati
        '57011' => '', # x-iscii-pa ISCII Punjabi
        '65000' => '', # utf-7 Unicode (UTF-7)
        '65001' => 'UTF2', # utf-8 Unicode (UTF-8)

        }->{(qx{chcp} =~ m/([0-9]{3,5}) \Z/oxms)[0]};
    }

    # C or POSIX
    elsif (not defined($LANG) or ($LANG eq '')) {
        $encoding = 'USASCII';
    }
    elsif ($LANG =~ m/\A (?: C | POSIX ) \z/oxms) {
        $encoding = 'USASCII';
    }

    # Apple Mac OS X
    elsif ($OSNAME eq 'darwin') {
        $encoding = 'UTF2';
    }

    # Apple MacOS
    elsif ($OSNAME eq 'MacOS') {
        die __FILE__, ": $OSNAME requires magick comment.\n";
    }

    # Oracle Solaris
    elsif ($OSNAME eq 'solaris') {
        my $lang = {

        # Oracle Solaris 10 8/11 Information Library

        qw(

        ar                       ar_EG.ISO8859-6
        bg_BG                    bg_BG.ISO8859-5
        ca                       ca_ES.ISO8859-1
        ca_ES                    ca_ES.ISO8859-1
        cs                       cs_CZ.ISO8859-2
        cs_CZ                    cs_CZ.ISO8859-2
        da                       da_DK.ISO8859-1
        da_DK                    da_DK.ISO8859-1
        da.ISO8859-15            da_DK.ISO8859-15
        de                       de_DE.ISO8859-1
        de_AT                    de_AT.ISO8859-1
        de_CH                    de_CH.ISO8859-1
        de_DE                    de_DE.ISO8859-1
        de.ISO8859-15            de_DE.ISO8859-15
        de.UTF-8                 de_DE.UTF-8
        el                       el_GR.ISO8859-7
        el_GR                    el_GR.ISO8859-7
        el.sun_eu_greek          el_GR.ISO8859-7
        el.UTF-8                 el_CY.UTF-8
        en_AU                    en_AU.ISO8859-1
        en_CA                    en_CA.ISO8859-1
        en_GB                    en_GB.ISO8859-1
        en_IE                    en_IE.ISO8859-1
        en_NZ                    en_NZ.ISO8859-1
        en_US                    en_US.ISO8859-1
        es                       es_ES.ISO8859-1
        es_AR                    es_AR.ISO8859-1
        es_BO                    es_BO.ISO8859-1
        es_CL                    es_CL.ISO8859-1
        es_CO                    es_CO.ISO8859-1
        es_CR                    es_CR.ISO8859-1
        es_EC                    es_EC.ISO8859-1 
        es_ES                    es_ES.ISO8859-1
        es_GT                    es_GT.ISO8859-1
        es.ISO8859-15            es_ES.ISO8859-15
        es_MX                    es_MX.ISO8859-1
        es_NI                    es_NI.ISO8859-1 
        es_PA                    es_PA.ISO8859-1
        es_PE                    es_PE.ISO8859-1
        es_PY                    es_PY.ISO8859-1
        es_SV                    es_SV.ISO8859-1
        es.UTF-8                 es_ES.UTF-8
        es_UY                    es_UY.ISO8859-1
        es_VE                    es_VE.ISO8859-1
        et                       et_EE.ISO8859-15
        et_EE                    et_EE.ISO8859-15
        fi                       fi_FI.ISO8859-1
        fi_FI                    fi_FI.ISO8859-1
        fi.ISO8859-15            fi_FI.ISO8859-15
        fr                       fr_FR.ISO8859-1
        fr_BE                    fr_BE.ISO8859-1
        fr_CA                    fr_CA.ISO8859-1
        fr_CH                    fr_CH.ISO8859-1
        fr_FR                    fr_FR.ISO8859-1
        fr.ISO8859-15            fr_FR.ISO8859-15
        fr.UTF-8                 fr_FR.UTF-8
        he                       he_IL.ISO8859-8
        he_IL                    he_IL.ISO8859-8
        hr_HR                    hr_HR.ISO8859-2
        hu                       hu_HU.ISO8859-2
        hu_HU                    hu_HU.ISO8859-2
        is_IS                    is_IS.ISO8859-1
        it                       it_IT.ISO8859-1
        it.ISO8859-15            it_IT.ISO8859-15
        it_IT                    it_IT.ISO8859-1
        it.UTF-8                 it_IT.UTF-8
        ja                       ja_JP.eucJP
        ko                       ko_KR.EUC
        ko.UTF-8                 ko_KR.UTF-8
        lt                       lt_LT.ISO8859-13
        lt_LT                    lt_LT.ISO8859-13
        lu                       lu_LU.ISO8859-15
        lv                       lv_LV.ISO8859-13
        lv_LV                    lv_LV.ISO8859-13
        mk_MK                    mk_MK.ISO8859-5
        nl                       nl_NL.ISO8859-1
        nl_BE                    nl_BE.ISO8859-1
        nl.ISO8859-15            nl_NL.ISO8859-15
        nl_NL                    nl_NL.ISO8859-1
        no                       nb_NO.ISO8859-1
        no_NO                    nb_NO.ISO8859-1
        no_NO.ISO8859-1@bokmal   nb_NO.ISO8859-1
        no_NO.ISO8859-1@nynorsk  nn_NO.ISO8859-1
        no_NY                    nn_NO.ISO8859-1
        nr                       nr_NR.ISO8859-2
        pl                       pl_PL.ISO8859-2
        pl_PL                    pl_PL.ISO8859-2
        pl.UTF-8                 pl_PL.UTF-8
        pt                       pt_PT.ISO8859-1
        pt_BR                    pt_BR.ISO8859-1
        pt.ISO8859-15            pt_PT.ISO8859-15
        pt_PT                    pt_PT.ISO8859-1
        ro_RO                    ro_RO.ISO8859-2
        ru                       ru_RU.ISO8859-5
        ru.koi8-r                ru_RU.KOI8-R
        ru_RU                    ru_RU.ISO8859-5
        ru.UTF-8                 ru_RU.UTF-8
        sh                       bs_BA.ISO8859-2
        sh_BA                    bs_BA.ISO8859-2
        sh_BA.ISO8859-2@bosnia   bs_BA.ISO8859-2
        sh_BA.UTF-8              bs_BA.UTF-8
        sk_SK                    sk_SK.ISO8859-2
        sl_SI                    sl_SI.ISO8859-2
        sq_AL                    sq_AL.ISO8859-2
        sr_CS                    sr_ME.UTF-8
        sr_CS.UTF-8              sr_ME.UTF-8
        sr_SP                    sr_ME.ISO8859-5
        sr_YU                    sr_ME.ISO8859-5
        sr_YU.ISO8859-5          sr_ME.ISO8859-5
        sv                       sv_SE.ISO8859-1
        sv_SE                    sv_SE.ISO8859-1
        sv.ISO8859-15            sv_SE.ISO8859-15
        sv.UTF-8                 sv_SE.UTF-8
        th                       th_TH.TIS620
        th_TH                    th_TH.TIS620
        th_TH.ISO8859-11         th_TH.TIS620
        tr                       tr_TR.ISO8859-9
        tr_TR                    tr_TR.ISO8859-9
        zh                       zh_CN.EUC
        zh.GBK                   zh_CN.GBK
        zh_TW                    zh_TW.EUC
        zh.UTF-8                 zh_CN.UTF-8
        ca_ES.ISO8859-15@euro    ca_ES.ISO8859-15
        de_AT.ISO8859-15@euro    de_AT.ISO8859-15
        de_DE.ISO8859-15@euro    de_DE.ISO8859-15
        de_DE.UTF-8@euro         de_DE.UTF-8
        el_GR.ISO8859-7@euro     el_GR.ISO8859-7
        en_IE.ISO8859-15@euro    en_IE.ISO8859-15
        es_ES.ISO8859-15@euro    es_ES.ISO8859-15
        es_ES.UTF-8@euro         es_ES.UTF-8
        fi_FI.ISO8859-15@euro    fi_FI.ISO8859-15
        fr_BE.ISO8859-15@euro    fr_BE.ISO8859-15
        fr_BE.UTF-8@euro         fr_BE.UTF-8
        fr_FR.ISO8859-15@euro    fr_FR.ISO8859-15
        fr_FR.UTF-8@euro         fr_FR.UTF-8
        it_IT.ISO8859-15@euro    it_IT.ISO8859-15
        it_IT.UTF-8@euro         it_IT.UTF-8
        nl_BE.ISO8859-15@euro    nl_BE.ISO8859-15
        nl_NL.ISO8859-15@euro    nl_NL.ISO8859-15
        pt_PT.ISO8859-15@euro    pt_PT.ISO8859-15
        cz                       cs_CZ.ISO8859-2
        cs_CZ                    cs_CZ.ISO8859-2
        cs_CZ.ISO8859-2          cs_CZ.ISO8859-2
        cs_CZ.UTF-8              cs_CZ.UTF-8
        cs_CZ.UTF-8@euro         cs_CZ.UTF-8
        ko_KR.EUC                ko_KR.EUC
        ko_KR.UTF-8              ko_KR.UTF-8
        zh_CN.EUC                zh_CN.EUC
        zh_CN.GBK                zh_CN.GBK
        zh_CN.UTF-8              zh_CN.UTF-8
        zh_TW.EUC                zh_TW.EUC

        )}->{$LANG} || $LANG;

        if ($lang eq 'ko_KR.EUC') {
            $encoding = 'UHC';
        }
        elsif ($lang eq 'zh_CN.EUC') {
            $encoding = 'GBK';
        }
        elsif ($lang eq 'zh_TW.EUC') {
            $encoding = 'EUCTW';
        }
        elsif (my($codeset) = $lang =~ m/\A [^.]+ \. ([^@]+) /oxms) {
            $encoding = {qw(

            5601         UHC
            ANSI1251     N/A
            BIG5         Big5Plus
            BIG5HK       Big5HKSCS
            EUC          N/A
            GB18030      GB18030
            GBK          GBK
            ISO/IEC646   USASCII
            ISO8859-1    Latin1
            ISO8859-13   Latin7
            ISO8859-15   Latin9
            ISO8859-2    Latin2
            ISO8859-5    Cyrillic
            ISO8859-6    Arabic
            ISO8859-7    Greek
            ISO8859-8    Hebrew
            ISO8859-9    Latin5
            KOI8-R       KOI8R
            PCK          Sjis
            TIS620       TIS620
            TIS620-2533  TIS620
            UTF-8        UTF2
            cns11643     EUCTW
            eucJP        EUCJP
            gb2312       GBK

            )}->{$codeset};
        }
    }

    # HP HP-UX
    elsif ($OSNAME eq 'hpux') {

        # HP-UX 9.x
        if ($LANG =~ m/\A japanese \z/oxms) {
            $encoding = 'Sjis';
        }
        elsif ($LANG =~ m/\A japanese\.euc \z/oxms) {
            $encoding = 'EUCJP';
        }

        # HP-UX 10.x
        if ($LANG =~ m/\A ja_JP\.SJIS \z/oxms) {
            $encoding = 'Sjis';
        }
        elsif ($LANG =~ m/\A ja_JP\.eucJP \z/oxms) {
            $encoding = 'EUCJP';
        }

        # HP-UX 11.x
        elsif (my($codeset) = $LANG =~ m/\A [^.]+ \. ([^@]+) /oxms) {
            $encoding = {

            # HP-UX 11i v3 Internationalization Features
            # Appendix -- Summary of Locale and codeset Conversion Support in HP-UX 11i v3
            # Locales

            qw(

            SJIS       Sjis
            arabic8    N/A
            big5       Big5Plus
            ccdc       N/A
            cp1251     N/A
            eucJP      EUCJP
            eucKR      UHC
            eucTW      EUCTW
            gb18030    GB18030
            greek8     N/A
            hebrew8    N/A
            hkbig5     Big5HKSCS
            hp15CN     N/A
            iso88591   Latin1
            iso885910  Latin6
            iso885911  TIS620
            iso885913  Latin7
            iso885915  Latin9
            iso88592   Latin2
            iso88593   Latin3
            iso88594   Latin4
            iso88595   Cyrillic
            iso88596   Arabic
            iso88597   Greek
            iso88598   Hebrew
            iso88599   Latin5
            kana8      N/A
            koi8r      KOI8R
            roman8     N/A
            tis620     TIS620
            turkish8   N/A
            utf8       UTF2

            )}->{$codeset};
        }
    }

    # IBM AIX
    elsif ($OSNAME eq 'aix') {
        my $codeset = {

        # SC23-4902-03
        # AIX 5L Version 5.3
        # National Language Support Guide and Reference
        # (c) Copyright International Business Machines Corporation 2002, 2006. All rights reserved.

        qw(

        ar_AA           ISO8859-6
        AR_AA           UTF-8
        Ar_AA           IBM-1046
        ar_AE           ISO8859-6
        AR_AE           UTF-8
        ar_DZ           ISO8859-6
        AR_DZ           UTF-8
        ar_BH           ISO8859-6
        AR_BH           UTF-8
        ar_EG           ISO8859-6
        AR_EG           UTF-8
        ar_JO           ISO8859-6
        AR_JO           UTF-8
        ar_KW           ISO8859-6
        AR_KW           UTF-8
        ar_LB           ISO8859-6
        AR_LB           UTF-8
        ar_MA           ISO8859-6
        AR_MA           UTF-8
        ar_OM           ISO8859-6
        AR_OM           UTF-8
        ar_QA           ISO8859-6
        AR_QA           UTF-8
        ar_SA           ISO8859-6
        AR_SA           UTF-8
        ar_SY           ISO8859-6
        AR_SY           UTF-8
        ar_TN           ISO8859-6
        AR_TN           UTF-8
        ar_YE           ISO8859-6
        AR_YE           UTF-8
        sq_AL           ISO8859-1
        sq_AL.8859-15   ISO8859-15
        SQ_AL           UTF-8
        be_BY           ISO8859-5
        BE_BY           UTF-8
        bg_BG           ISO8859-5
        BG_BG           UTF-8
        ca_ES.IBM-1252  IBM-1252
        ca_ES           ISO8859-1
        ca_ES.8859-15   ISO8859-15
        CA_ES           UTF-8
        Ca_ES           IBM-850
        zh_TW           IBM-eucTW
        ZH_TW           UTF-8
        Zh_TW           big5
        zh_CN           IBM-eucCN
        ZH_CN           UTF-8
        Zh_CN           GBK/GB18030
        ZH_HK           UTF-8
        ZH_SG           UTF-8
        hr_HR           ISO8859-2
        HR_HR           UTF-8
        cs_CZ           ISO8859-2
        CS_CZ           UTF-8
        da_DK           ISO8859-1
        da_DK.8859-15   ISO8859-15
        DA_DK           UTF-8
        nl_BE.IBM-1252  IBM-1252
        nl_BE           ISO8859-1
        nl_BE.8859-15   ISO8859-15
        NL_BE           UTF-8
        nl_NL.IBM-1252  IBM-1252
        nl_NL           ISO8859-1
        nl_NL.8859-15   ISO8859-15
        NL_NL           UTF-8
        en_AU.8859-15   ISO8859-15
        EN_AU           UTF-8
        en_BE.8859-15   ISO8859-15
        EN_BE           UTF-8
        en_CA.8859-15   ISO8859-15
        EN_CA           UTF-8
        en_GB.IBM-1252  IBM-1252
        en_GB           ISO8859-1
        en_GB.8859-15   ISO8859-15
        EN_GB           UTF-8
        en_HK           ISO8859-15
        EN_HK           UTF-8
        en_IE.8859-15   ISO8859-15
        EN_IE           UTF-8
        en_IN.8859-15   ISO8859-15
        EN_IN           UTF-8
        en_NZ.8859-15   ISO8859-15
        EN_NZ           UTF-8
        en_PH           ISO8859-15
        EN_PH           UTF-8
        en_SG           ISO8859-15
        EN_SG           UTF-8
        en_US           ISO8859-1
        en_US.8859-15   ISO8859-15
        EN_US           UTF-8
        en_ZA.8859-15   ISO8859-15
        EN_ZA           UTF-8
        Et_EE           IBM-922
        et_EE           ISO8859-4
        ET_EE           UTF-8
        fi_FI.IBM-1252  IBM-1252
        fi_FI           ISO8859-1
        fi_FI.8859-15   ISO8859-15
        FI_FI           UTF-8
        fr_BE.IBM-1252  IBM-1252
        fr_BE           ISO8859-1
        fr_BE.8859-15   ISO8859-15
        FR_BE           UTF-8
        fr_CA           ISO8859-1
        fr_CA.8859-15   ISO8859-15
        FR_CA           UTF-8
        fr_FR.IBM-1252  IBM-1252
        fr_FR           ISO8859-1
        fr_FR.8859-15   ISO8859-15
        FR_FR           UTF-8
        fr_LU.8859-15   ISO8859-1
        FR_LU           ISO8859-1
        fr_CH           ISO8859-1
        fr_CH.8859-15   ISO8859-15
        FR_CH           UTF-8
        de_AT.8859-15   ISO8859-15
        DE_AT           UTF-8
        de_CH           ISO8859-1
        de_CH.8859-15   ISO8859-15
        DE_CH           UTF-8
        de_DE.IBM-1252  IBM-1252
        de_DE           ISO8859-1
        de_DE.8859-15   ISO8859-15
        DE_DE           UTF-8
        de_LU.8859-15   ISO8859-15
        DE_LU           UTF-8
        el_GR           ISO8859-7
        EL_GR           UTF-8
        iw_IL           ISO8859-8
        HE_IL           UTF-8
        Iw_IL           IBM-856
        hu_HU           ISO8859-2
        HU_HU           UTF-8
        is_IS           ISO8859-1
        is_IS.8859-15   ISO8859-15
        IS_IS           UTF-8
        AS_IN           UTF-8
        BN_IN           UTF-8
        GU_IN           UTF-8
        HI_IN           UTF-8
        KN_IN           UTF-8
        ML_IN           UTF-8
        MR_IN           UTF-8
        OR_IN           UTF-8
        PA_IN           UTF-8
        TA_IN           UTF-8
        TE_IN           UTF-8
        it_IT.IBM-1252  IBM-1252
        it_IT           ISO8859-1
        it_IT.8859-15   ISO8859-15
        IT_IT           UTF-8
        it_CH.8859-15   ISO8859-15
        IT_CH           UTF-8
        ja_JP           IBM-eucJP
        JA_JP           UTF-8
        Ja_JP           IBM-943
        KK_KZ           UTF-8
        ko_KR           IBM-eucKR
        KO_KR           UTF-8
        id_ID           ISO8859-15
        ID_ID           UTF-8
        Lv_LV           IBM-921
        lv_LV           ISO8859-4
        LV_LV           UTF-8
        Lt_LT           IBM-921
        lt_LT           ISO8859-4
        LT_LT           UTF-8
        mk_MK           ISO8859-5
        MK_MK           UTF-8
        ms_MY           ISO8859-15
        MS_MY           UTF-8
        no_NO           ISO8859-1
        no_NO.8859-15   ISO8859-15
        NO_NO           UTF-8
        pl_PL           ISO8859-2
        PL_PL           UTF-8
        pt_BR           ISO8859-1
        pt_BR.8859-15   ISO8859-15
        PT_BR           UTF-8
        pt_PT.IBM-1252  IBM-1252
        pt_PT           ISO8859-1
        pt_PT.8859-15   ISO8859-15
        PT_PT           UTF-8
        ro_RO           ISO8859-2
        RO_RO           UTF-8
        ru_RU           ISO8859-5
        RU_RU           UTF-8
        sr_SP           ISO8859-5
        SR_SP           UTF-8
        sr_YU           ISO8859-5
        SR_YU           UTF-8
        sh_SP           ISO8859-2
        SH_SP           UTF-8
        sh_YU           ISO8859-2
        SH_YU           UTF-8
        sk_SK           ISO8859-2
        SK_SK           UTF-8
        sl_SI           ISO8859-2
        SL_SI           UTF-8
        es_AR.8859-15   ISO8859-15
        ES_AR           UTF-8
        es_BO           ISO8859-15
        ES_BO           UTF-8
        es_CL.8859-15   ISO8859-15
        ES_CL           UTF-8
        es_CO.8859-15   ISO8859-15
        ES_CO           UTF-8
        es_CR           ISO8859-15
        ES_CR           UTF-8
        es_DO           ISO8859-15
        ES_DO           UTF-8
        es_EC           ISO8859-15
        ES_EC           UTF-8
        es_GT           ISO8859-15
        ES_GT           UTF-8
        es_HN           ISO8859-15
        ES_HN           UTF-8
        es_ES.IBM-1252  IBM-1252
        es_ES           ISO8859-1
        es_ES.8859-15   ISO8859-15
        ES_ES           UTF-8
        es_MX.8859-15   ISO8859-15
        ES_MX           UTF-8
        es_NI           ISO8859-15
        ES_NI           UTF-8
        es_PA           ISO8859-15
        ES_PA           UTF-8
        es_PY           ISO8859-15
        ES_PY           UTF-8
        es_PE.8859-15   ISO8859-15
        ES_PE           UTF-8
        es_PR.8859-15   ISO8859-15
        ES_PR           UTF-8
        es_US           ISO8859-15
        ES_US           UTF-8
        es_UY.8859-15   ISO8859-15
        ES_UY           UTF-8
        es_VE.8859-15   ISO8859-15
        ES_VE           UTF-8
        sv_SE           ISO8859-1
        sv_SE.8859-15   ISO8859-15
        SV_SE           UTF-8
        th_TH           TIS-620
        TH_TH           UTF-8
        tr_TR           ISO8859-9
        TR_TR           UTF-8
        Uk_UA           IBM-1124
        UK_UA           UTF-8
        Vi_VN           IBM-1129
        VI_VN           UTF-8

        )}->{$LANG};

        $encoding = {qw(

        GBK/GB18030  GB18030
        IBM-1046     N/A
        IBM-1124     N/A
        IBM-1129     N/A
        IBM-1252     N/A
        IBM-850      N/A
        IBM-856      N/A
        IBM-921      N/A
        IBM-922      N/A
        IBM-943      Sjis
        IBM-eucCN    GBK
        IBM-eucJP    EUCJP
        IBM-eucKR    UHC
        IBM-eucTW    EUCTW
        ISO8859-1    Latin1
        ISO8859-15   Latin9
        ISO8859-2    Latin2
        ISO8859-4    Latin4
        ISO8859-5    Cyrillic
        ISO8859-6    Arabic
        ISO8859-7    Greek
        ISO8859-8    Hebrew
        ISO8859-9    Latin5
        TIS-620      TIS620
        UTF-8        UTF2
        big5         Big5Plus

        )}->{$codeset};
    }

    # Other Systems
    if ($encoding eq '') {
        if ($encoding = {qw(

            ja            EUCJP
            ja_JP         EUCJP
            ja_JP.ujis    EUCJP
            ja_JP.eucJP   EUCJP
            Jp_JP         EUCJP
            ja_JP.AJEC    EUCJP
            ja_JP.EUC     EUCJP
            ja_JP.mscode  Sjis
            ja_JP.SJIS    Sjis
            ja_JP.PCK     Sjis
            ja_JP.UTF-8   UTF2
            ja_JP.utf8    UTF2
            japanese      Sjis
            japanese.euc  EUCJP
            japan         EUCJP
            Japanese-EUC  EUCJP

            )}->{$LANG}) {
        }
        elsif (my($codeset) = $LANG =~ m/\A [^.]+ \. ([^@]+) /oxms) {
            $encoding = {qw(

            UTF-8        UTF2
            UTF8         UTF2
            ISO_8859-1   Latin1
            ISO_8859-2   Latin2
            ISO_8859-3   Latin3
            ISO_8859-4   Latin4
            ISO_8859-5   Cyrillic
            ISO_8859-6   Arabic
            ISO_8859-7   Greek
            ISO_8859-8   Hebrew
            ISO_8859-9   Latin5
            ISO_8859-10  Latin6
            ISO_8859-11  TIS620
            ISO_8859-13  Latin7
            ISO_8859-14  Latin8
            ISO_8859-15  Latin9
            ISO_8859-16  Latin10
            ISO-8859-1   Latin1
            ISO-8859-2   Latin2
            ISO-8859-3   Latin3
            ISO-8859-4   Latin4
            ISO-8859-5   Cyrillic
            ISO-8859-6   Arabic
            ISO-8859-7   Greek
            ISO-8859-8   Hebrew
            ISO-8859-9   Latin5
            ISO-8859-10  Latin6
            ISO-8859-11  TIS620
            ISO-8859-13  Latin7
            ISO-8859-14  Latin8
            ISO-8859-15  Latin9
            ISO-8859-16  Latin10
            ISO8859-1    Latin1
            ISO8859-2    Latin2
            ISO8859-3    Latin3
            ISO8859-4    Latin4
            ISO8859-5    Cyrillic
            ISO8859-6    Arabic
            ISO8859-7    Greek
            ISO8859-8    Hebrew
            ISO8859-9    Latin5
            ISO8859-10   Latin6
            ISO8859-11   TIS620
            ISO8859-13   Latin7
            ISO8859-14   Latin8
            ISO8859-15   Latin9
            ISO8859-16   Latin10
            ISO88591     Latin1
            ISO88592     Latin2
            ISO88593     Latin3
            ISO88594     Latin4
            ISO88595     Cyrillic
            ISO88596     Arabic
            ISO88597     Greek
            ISO88598     Hebrew
            ISO88599     Latin5
            ISO885910    Latin6
            ISO885911    TIS620
            ISO885913    Latin7
            ISO885914    Latin8
            ISO885915    Latin9
            ISO885916    Latin10
            KOI8-R       KOI8R
            KOI8-U       KOI8U

            )}->{uc $codeset};
        }
    }

    return $encoding;
}

#
# get absolute path to filter software
#
sub abspath {
    my($encoding) = @_;
    for my $path (@INC) {
        if ($OSNAME eq 'MacOS') {
            if (-e "$path$encoding.pm") {
                return "$path$encoding.pm";
            }
        }
        else {
            if (-e "$path/$encoding.pm") {
                return "$path/$encoding.pm";
            }
        }
    }
    return '';
}

#
# make filter software
#
sub make_filter_software {
    my($dist) = @_;

    my %compile_options = (
    #-----------------------------------------------------------------------------------
    #                        #if define macros
    #                        +---------------- FILE_SYSTEM_SAFE
    #                        | +-------------- MULTIBYTE_ENCODING
    #                        | | +------------ ESCAPE_SECOND_OCTET
    #                        | | | +---------- MULTIBYTE_ANCHORING
    #                        | | | | +-------- LONG_STRING_FOR_RE
    #                        | | | | | +------ MACPERL
    #dist                    v v v v v v     encoding_name       rewrite_leading_byte
    #-----------------------------------------------------------------------------------
    'Arabic'        => [[qw( 1 0 0 0 0 0 )], q(Arabic),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Big5HKSCS'     => [[qw( 0 1 1 1 1 0 )], q(Big5-HKSCS),      sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE]/g; }],
    'Big5Plus'      => [[qw( 0 1 1 1 1 0 )], q(Big5Plus),        sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE]/g; }],
    'Cyrillic'      => [[qw( 1 0 0 0 0 0 )], q(Cyrillic),        sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'EUCJP'         => [[qw( 1 1 0 1 1 0 )], q(EUC-JP),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x8E\\x8F\\xA1-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x8E\\xA1-\\xFE][\\xA1-\\xFE]|\\x8F[\\xA1-\\xFE]/g; }],
    'EUCTW'         => [[qw( 1 1 0 1 1 0 )], q(EUC-TW),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x8E\\xA1-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\xA1-\\xFE][\\xA1-\\xFE]|\\x8E[\\xA1-\\xB0][\\xA1-\\xFE]/g; }],
    'GB18030'       => [[qw( 0 1 1 1 1 0 )], q(GB18030),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE][^\\x30-\\x39]|[\\x81-\\xFE][\\x30-\\x39][\\x81-\\xFE]/g; }],
    'GBK'           => [[qw( 0 1 1 1 1 0 )], q(GBK),             sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE]/g; }],
    'Greek'         => [[qw( 1 0 0 0 0 0 )], q(Greek),           sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'HP15'          => [[qw( 0 1 1 1 1 0 )], q(HP-15),           sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x80-\\xA0\\xE0-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x80-\\xA0\\xE0-\\xFE]/g; }],
    'Hebrew'        => [[qw( 1 0 0 0 0 0 )], q(Hebrew),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'INFORMIXV6ALS' => [[qw( 0 1 1 1 1 0 )], q(INFORMIX V6 ALS), sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\x9F\\xE0-\\xFD/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\x9F\\xE0-\\xFC][\\x00-\\xFF]|\\xFD[\\xA1-\\xFE]/g; }],
    'JIS8'          => [[qw( 1 0 0 0 0 0 )], q(JIS8),            sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'KOI8R'         => [[qw( 1 0 0 0 0 0 )], q(KOI8-R),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'KOI8U'         => [[qw( 1 0 0 0 0 0 )], q(KOI8-U),          sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'KPS9566'       => [[qw( 0 1 1 1 1 0 )], q(KPS9566),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE]/g; }],
    'Latin1'        => [[qw( 1 0 0 0 0 0 )], q(Latin-1),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin10'       => [[qw( 1 0 0 0 0 0 )], q(Latin-10),        sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin2'        => [[qw( 1 0 0 0 0 0 )], q(Latin-2),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin3'        => [[qw( 1 0 0 0 0 0 )], q(Latin-3),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin4'        => [[qw( 1 0 0 0 0 0 )], q(Latin-4),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin5'        => [[qw( 1 0 0 0 0 0 )], q(Latin-5),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin6'        => [[qw( 1 0 0 0 0 0 )], q(Latin-6),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin7'        => [[qw( 1 0 0 0 0 0 )], q(Latin-7),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin8'        => [[qw( 1 0 0 0 0 0 )], q(Latin-8),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Latin9'        => [[qw( 1 0 0 0 0 0 )], q(Latin-9),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'OldUTF8'       => [[qw( 1 1 0 0 0 0 )], q(old UTF-8),       sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x80-\\xFF/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/(?:[\\xC0-\\xDF]|[\\xE0-\\xEF][\\x80-\\xBF]|[\\xF0-\\xF4][\\x80-\\xBF][\\x80-\\xBF])/g; s{\(\?:\[\\xC0-\\xDF\]\|\[\\xE0-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF4\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\)\[\\x00-\\xFF\]}{\(\?:\[\\xC0-\\xDF\]\|\[\\xE0-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF4\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]}g; s{\(\?:\[\\xC0-\\xDF\]\|\[\\xE0-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF4\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]\|\[\\x00-\\xFF\]}{\(\?:\[\\xC0-\\xDF\]\|\[\\xE0-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF4\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]\|\[\\x00-\\x7F\\xF5-\\xFF\]}g; s/\\x82\\xa0/\\xe3\\x81\\x82/g; }],
    'Sjis'          => [[qw( 0 1 1 1 1 1 )], q(ShiftJIS),        sub{ }],
    'TIS620'        => [[qw( 1 0 0 0 0 0 )], q(TIS-620),         sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'UHC'           => [[qw( 0 1 1 1 1 0 )], q(UHC),             sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x81-\\xFE/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/[\\x81-\\xFE]/g; }],
    'USASCII'       => [[qw( 1 0 0 0 0 0 )], q(US-ASCII),        sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'UTF2'          => [[qw( 1 1 0 0 0 0 )], q(UTF-8),           sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^\\x80-\\xFF/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]/(?:[\\xC2-\\xDF]|[\\xE0-\\xE0][\\xA0-\\xBF]|[\\xE1-\\xEC][\\x80-\\xBF]|[\\xED-\\xED][\\x80-\\x9F]|[\\xEE-\\xEF][\\x80-\\xBF]|[\\xF0-\\xF0][\\x90-\\xBF][\\x80-\\xBF]|[\\xF1-\\xF3][\\x80-\\xBF][\\x80-\\xBF]|[\\xF4-\\xF4][\\x80-\\x8F][\\x80-\\xBF])/g; s{\(\?:\[\\xC2-\\xDF\]\|\[\\xE0-\\xE0\]\[\\xA0-\\xBF\]\|\[\\xE1-\\xEC\]\[\\x80-\\xBF\]\|\[\\xED-\\xED\]\[\\x80-\\x9F\]\|\[\\xEE-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF0\]\[\\x90-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF1-\\xF3\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF4-\\xF4\]\[\\x80-\\x8F\]\[\\x80-\\xBF\]\)\[\\x00-\\xFF\]}{\(\?:\[\\xC2-\\xDF\]\|\[\\xE0-\\xE0\]\[\\xA0-\\xBF\]\|\[\\xE1-\\xEC\]\[\\x80-\\xBF\]\|\[\\xED-\\xED\]\[\\x80-\\x9F\]\|\[\\xEE-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF0\]\[\\x90-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF1-\\xF3\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF4-\\xF4\]\[\\x80-\\x8F\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]}g; s{\(\?:\[\\xC2-\\xDF\]\|\[\\xE0-\\xE0\]\[\\xA0-\\xBF\]\|\[\\xE1-\\xEC\]\[\\x80-\\xBF\]\|\[\\xED-\\xED\]\[\\x80-\\x9F\]\|\[\\xEE-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF0\]\[\\x90-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF1-\\xF3\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF4-\\xF4\]\[\\x80-\\x8F\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]\|\[\\x00-\\xFF\]}{\(\?:\[\\xC2-\\xDF\]\|\[\\xE0-\\xE0\]\[\\xA0-\\xBF\]\|\[\\xE1-\\xEC\]\[\\x80-\\xBF\]\|\[\\xED-\\xED\]\[\\x80-\\x9F\]\|\[\\xEE-\\xEF\]\[\\x80-\\xBF\]\|\[\\xF0-\\xF0\]\[\\x90-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF1-\\xF3\]\[\\x80-\\xBF\]\[\\x80-\\xBF\]\|\[\\xF4-\\xF4\]\[\\x80-\\x8F\]\[\\x80-\\xBF\]\)\[\\x80-\\xBF\]\|\[\\x00-\\x7F\\xF5-\\xFF\]}g; s/\\x82\\xa0/\\xe3\\x81\\x82/g; }],
    'Windows1252'   => [[qw( 1 0 0 0 0 0 )], q(Windows-1252),    sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    'Windows1258'   => [[qw( 1 0 0 0 0 0 )], q(Windows-1258),    sub{ s/\[\^\\x81-\\x9F\\xE0-\\xFC\]/[\\x00-\\xFF]/g; s/\[\^\\x81-\\x9F\\xE0-\\xFC/[^/g; s/\[\\x81-\\x9F\\xE0-\\xFC\]//g; s/\|\[\\x00\-\\xFF\]//g; }],
    #-----------------------------------------------------------------------------------
    );

    my @class = split(/\n/,<<'END_OF_CLASS_SCRIPT');
package Sjis;
use strict;
BEGIN { $INC{'warnings.pm'} = '' if $] < 5.006 } use warnings;
######################################################################
#
# Sjis - Source code filter to escape ShiftJIS script
#
# http://search.cpan.org/dist/Char-Sjis/
#
# $copyright$
######################################################################

use 5.00503;    # Galapagos Consensus 1998 for primetools
# use 5.008001; # Lancaster Consensus 2013 for toolchains

# 12.3. Delaying use Until Runtime
# in Chapter 12. Packages, Libraries, and Modules
# of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
# (and so on)

# Version numbers should be boring
# http://www.dagolden.com/index.php/369/version-numbers-should-be-boring/
# For the impatient, the disinterested or those who just want to follow
# a recipe, my advice for all modules is this:
# our $VERSION = "0.001"; # or "0.001_001" for a dev release
# $VERSION = CORE::eval $VERSION; # No!! because '1.10' makes '1.1'

use vars qw($VERSION);
$VERSION = q$Revision$;
$VERSION = $VERSION;

BEGIN {
    if ($^X =~ / jperl /oxmsi) {
        die __FILE__, ": needs perl(not jperl) 5.00503 or later. (\$^X==$^X)\n";
    }
    if (CORE::ord('A') == 193) {
        die __FILE__, ": is not US-ASCII script (may be EBCDIC or EBCDIK script).\n";
    }
    if (CORE::ord('A') != 0x41) {
        die __FILE__, ": is not US-ASCII script (must be US-ASCII script).\n";
    }
}

BEGIN {
    (my $dirname = __FILE__) =~ s{^(.+)/[^/]*$}{$1};
    unshift @INC, $dirname;
    CORE::require Esjis;
}

# instead of Symbol.pm
BEGIN {
    sub gensym () {
        if ($] < 5.006) {
            return \do { local *_ };
        }
        else {
            return undef;
        }
    }
}

# P.714 29.2.39. flock
# in Chapter 29: Functions
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.863 flock
# in Chapter 27: Functions
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

# P.228 Inlining Constant Functions
# in Chapter 6: Subroutines
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.331 Inlining Constant Functions
# in Chapter 7: Subroutines
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

sub LOCK_SH() {1}
sub LOCK_EX() {2}
sub LOCK_UN() {8}
sub LOCK_NB() {4}

sub unimport {}
sub Sjis::escape_script;

# 6.18. Matching Multiple-Byte Characters
# in Chapter 6. Pattern Matching
# of ISBN 978-1-56592-243-3 Perl Perl Cookbook.
# (and so on)

# regexp of character
my $qq_char   = qr/(?> \\c[\x40-\x5F] | \\? (?:[\x81-\x9F\xE0-\xFC][\x00-\xFF] | [\x00-\xFF]) )/oxms;
my  $q_char   = qr/(?> [\x81-\x9F\xE0-\xFC][\x00-\xFF] | [\x00-\xFF] )/oxms;

# when this script is main program
if ($0 eq __FILE__) {

    # show usage
    unless (@ARGV) {
        die <<END;
$0: usage

perl $0 ShiftJIS_script.pl > Escaped_script.pl.e
END
    }

    print Sjis::escape_script($ARGV[0]);
    exit 0;
}

my($package,$filename,$line,$subroutine,$hasargs,$wantarray,$evaltext,$is_require,$hints,$bitmask) = caller 0;

# called any package not main
if ($package ne 'main') {
    die <<END;
@{[__FILE__]}: escape by manually command '$^X @{[__FILE__]} "$filename" > "@{[__PACKAGE__]}::$filename"'
and rewrite "use $package;" to "use @{[__PACKAGE__]}::$package;" of script "$0".
END
}

# P.302 Module Privacy and the Exporter
# in Chapter 11: Modules
# of ISBN 0-596-00027-8 Programming Perl Third Edition.
#
# A module can do anything it jolly well pleases when it's used, since use just
# calls the ordinary import method for the module, and you can define that
# method to do anything you like.

# P.406 Module Privacy and the Exporter
# in Chapter 11: Modules
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
#
# A module can do anything it jolly well pleases when it's used, since use just
# calls the ordinary import method for the module, and you can define that
# method to do anything you like.

sub import {

    if (Esjis::e("$filename.e")) {
        if (exists $ENV{'CHAR_DEBUG'}) {
            Esjis::unlink "$filename.e";
        }
        elsif (Esjis::z("$filename.e")) {
            Esjis::unlink "$filename.e";
        }
        else {

            #----------------------------------------------------
            #  older >
            #  newer >>>>>
            #----------------------------------------------------
            # Filter >
            # Source >>>>>
            # Escape >>>   needs re-escape (Source was changed)
            #
            # Filter >>>
            # Source >>>>>
            # Escape >     needs re-escape (Source was changed)
            #
            # Filter >>>>>
            # Source >>>
            # Escape >     needs re-escape (Source was changed)
            #
            # Filter >>>>>
            # Source >
            # Escape >>>   needs re-escape (Filter was changed)
            #
            # Filter >
            # Source >>>
            # Escape >>>>> executable without re-escape
            #
            # Filter >>>
            # Source >
            # Escape >>>>> executable without re-escape
            #----------------------------------------------------

            my $mtime_filter = (Esjis::stat(__FILE__     ))[9];
            my $mtime_source = (Esjis::stat($filename    ))[9];
            my $mtime_escape = (Esjis::stat("$filename.e"))[9];
            if (($mtime_escape < $mtime_source) or ($mtime_escape < $mtime_filter)) {
                Esjis::unlink "$filename.e";
            }
        }
    }

    if (not Esjis::e("$filename.e")) {
        my $fh = gensym();
        Esjis::_open_a($fh, "$filename.e") or die __FILE__, ": Can't write open file: $filename.e\n";

        # 7.19. Flushing Output
        # in Chapter 7. File Access
        # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.

        select((select($fh), $|=1)[0]);

        if (0) {
        }
#if MACPERL
        elsif ($^O eq 'MacOS') {
            CORE::eval q{
                CORE::require Mac::Files;
                Mac::Files::FSpSetFLock("$filename.e");
            };
        }
#endif
        elsif (exists $ENV{'CHAR_NONBLOCK'}) {

            # P.419 File Locking
            # in Chapter 16: Interprocess Communication
            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

            # P.524 File Locking
            # in Chapter 15: Interprocess Communication
            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

            # P.571 Handling Race Conditions
            # in Chapter 23: Security
            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

            # P.663 Handling Race Conditions
            # in Chapter 20: Security
            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

            # (and so on)

            CORE::eval q{ flock($fh, LOCK_EX | LOCK_NB) };
            if ($@) {
                die __FILE__, ": Can't immediately write-lock the file: $filename.e\n";
            }
        }
        else {
            CORE::eval q{ flock($fh, LOCK_EX) };
        }

        CORE::eval q{ truncate($fh, 0) };
        seek($fh, 0, 0) or die __FILE__, ": Can't seek file: $filename.e\n";

        my $e_script = Sjis::escape_script($filename);
        print {$fh} $e_script;

        my $mode = (Esjis::stat($filename))[2] & 0777;
        chmod $mode, "$filename.e";

#if MACPERL
        if ($^O eq 'MacOS') {
            CORE::eval q{
                CORE::require Mac::Files;
                Mac::Files::FSpRstFLock("$filename.e");
            };
        }

#endif
        close($fh) or die "Can't close file: $filename.e: $!";
    }

    my $fh = gensym();
    Esjis::_open_r($fh, "$filename.e") or die __FILE__, ": Can't read open file: $filename.e\n";

    if (0) {
    }
#if MACPERL
    elsif ($^O eq 'MacOS') {
        CORE::eval q{
            CORE::require Mac::Files;
            Mac::Files::FSpSetFLock("$filename.e");
        };
    }
#endif
    elsif (exists $ENV{'CHAR_NONBLOCK'}) {
        CORE::eval q{ flock($fh, LOCK_SH | LOCK_NB) };
        if ($@) {
            die __FILE__, ": Can't immediately read-lock the file: $filename.e\n";
        }
    }
    else {
        CORE::eval q{ flock($fh, LOCK_SH) };
    }

    my @switch = ();
    if ($^W) {
        push @switch, '-w';
    }
    if (defined $^I) {
        push @switch, '-i' . $^I;
        undef $^I;
    }

    # P.707 29.2.33. exec
    # in Chapter 29: Functions
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.
    #
    # If there is more than one argument in LIST, or if LIST is an array with more
    # than one value, the system shell will never be used. This also bypasses any
    # shell processing of the command. The presence or absence of metacharacters in
    # the arguments doesn't affect this list-triggered behavior, which makes it the
    # preferred from in security-conscious programs that do not with to expose
    # themselves to potential shell escapes.
    # Environment variable PERL5SHELL(Microsoft ports only) will never be used, too.

    # P.855 exec
    # in Chapter 27: Functions
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
    #
    # If there is more than one argument in LIST, or if LIST is an array with more
    # than one value, the system shell will never be used. This also bypasses any
    # shell processing of the command. The presence or absence of metacharacters in
    # the arguments doesn't affect this list-triggered behavior, which makes it the
    # preferred from in security-conscious programs that do not wish to expose
    # themselves to injection attacks via shell escapes.
    # Environment variable PERL5SHELL(Microsoft ports only) will never be used, too.

    # P.489 #! and Quoting on Non-Unix Systems
    # in Chapter 19: The Command-Line Interface
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.578 #! and Quoting on Non-Unix Systems
    # in Chapter 17: The Command-Line Interface
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    my $system = 0;

    # DOS-like system
    if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
        $system = Esjis::_systemx(
            _escapeshellcmd_MSWin32($^X),

        # -I switch can not treat space included path
        #   (map { '-I' . _escapeshellcmd_MSWin32($_) } @INC),
            (map { '-I' .                         $_  } @INC),

            @switch,
            '--',
            map { _escapeshellcmd_MSWin32($_) } "$filename.e", @ARGV
        );
    }

#if MACPERL
    # MacOS system
    elsif ($^O eq 'MacOS') {
        $system = Esjis::_systemx(
            _escapeshellcmd_MacOS($^X),
            (map { '-I' . _escapeshellcmd_MacOS($_) } @INC),
            @switch,
            '--',
            map { _escapeshellcmd_MacOS($_) } "$filename.e", @ARGV
        );
        CORE::eval q{
            CORE::require Mac::Files;
            Mac::Files::FSpRstFLock("$filename.e");
        };
    }

#endif
    # UNIX-like system
    else {
        $system = Esjis::_systemx(
            _escapeshellcmd($^X),
            (map { '-I' . _escapeshellcmd($_) } @INC),
            @switch,
            '--',
            map { _escapeshellcmd($_) } "$filename.e", @ARGV
        );
    }

    # exit with actual exit value
    exit($system >> 8);
}

# escape shell command line on DOS-like system
sub _escapeshellcmd_MSWin32 {
    my($word) = @_;
    if ($word =~ / [ ] /oxms) {
        return qq{"$word"};
    }
    else {
        return $word;
    }
}

#if MACPERL
# escape shell command line on Mac OS
sub _escapeshellcmd_MacOS {
    my($word) = @_;
    return $word;
}

#endif
# escape shell command line on UNIX-like system
sub _escapeshellcmd {
    my($word) = @_;
    return $word;
}

# P.619 Source Filters
# in Chapter 24: Common Practices
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.718 Source Filters
# in Chapter 21: Common Practices
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

# escape ShiftJIS script
sub Sjis::escape_script {
    my($script) = @_;
    my $e_script = '';

    # read ShiftJIS script
    my $fh = gensym();
    Esjis::_open_r($fh, $script) or die __FILE__, ": Can't open file: $script\n";
    local $/ = undef; # slurp mode
    $_ = <$fh>;
    close($fh) or die "Can't close file: $script: $!";

    if (/^ use Esjis(?:(?>\s+)(?>[0-9\.]*))?(?>\s*); $/oxms) {
        return $_;
    }
    else {

        # #! shebang line
        if (s/\A(#!.+?\n)//oms) {
            my $head = $1;
            $head =~ s/\bjperl\b/perl/gi;
            $e_script .= $head;
        }

        # DOS-like system header
        if (s/\A(\@rem(?>\s*)=(?>\s*)'.*?'(?>\s*);\s*\n)//oms) {
            my $head = $1;
            $head =~ s/\bjperl\b/perl/gi;
            $e_script .= $head;
        }

        # P.618 Generating Perl in Other Languages
        # in Chapter 24: Common Practices
        # of ISBN 0-596-00027-8 Programming Perl Third Edition.

        # P.717 Generating Perl in Other Languages
        # in Chapter 21: Common Practices
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        if (s/(.*^#(?>\s*)line(?>\s+)(?>[0-9]+)(?:(?>\s+)"(?:$q_char)+?")?\s*\n)//oms) {
            my $head = $1;
            $head =~ s/\bjperl\b/perl/gi;
            $e_script .= $head;
        }

        # P.210 5.10.3.3. Match-time code evaluation
        # in Chapter 5: Pattern Matching
        # of ISBN 0-596-00027-8 Programming Perl Third Edition.

        # P.255 Match-time code evaluation
        # in Chapter 5: Pattern Matching
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # '...' quote to avoid "Octal number in vector unsupported" on perl 5.6

        $e_script .= sprintf("use Esjis '%s.0'; # 'quote' for perl5.6\n", $Sjis::VERSION); # require run-time routines version

        # use Sjis version qw(ord reverse getc);
        if (s/^ (?>\s*) use (?>\s+) (?: Char | Sjis ) (?>\s*) ([^\x81-\x9F\xE0-\xFC;]*) ; \s* \n? $//oxms) {

            # require version
            my $list = $1;
            if ($list =~ s/\A ((?>[0-9]+)\.(?>[0-9]+)) \.0 (?>\s*) //oxms) {
                my $version = $1;
                if ($version ne $Sjis::VERSION) {
                    my @file = grep -e, map {qq{$_/Sjis.pm}} @INC;
                    my %file = map { $_ => 1 } @file;
                    if (scalar(keys %file) >= 2) {
                        my $file = join "\n", sort keys %file;
                        warn <<END;
****************************************************
                   C A U T I O N

              CONFLICT Sjis.pm FILE

$file
****************************************************

END
                    }
                    die "Script $0 expects Sjis.pm $version, but @{[__FILE__]} is version $Sjis::VERSION\n";
                }
                $e_script .= qq{die "Script \$0 expects Esjis.pm $version, but \\\$Esjis::VERSION is \$Esjis::VERSION" if \$Esjis::VERSION ne '$version';\n};
            }
            elsif ($list =~ s/\A ((?>[0-9]+)(?>\.[0-9]*)) (?>\s*) //oxms) {
                my $version = $1;
                if ($version > $Sjis::VERSION) {
                    die "Script $0 required Sjis.pm $version, but @{[__FILE__]} is only version $Sjis::VERSION\n";
                }
            }

            # demand ord, reverse, and getc
            if ($list !~ /\A (?>\s*) \z/oxms) {
                local $@;
                my @list = CORE::eval $list;
                for (@list) {
                    $Esjis::function_ord     = 'Sjis::ord'     if /\A ord \z/oxms;
                    $Esjis::function_ord_    = 'Sjis::ord_'    if /\A ord \z/oxms;
                    $Esjis::function_reverse = 'Sjis::reverse' if /\A reverse \z/oxms;
                    $Esjis::function_getc    = 'Sjis::getc'    if /\A getc \z/oxms;

                    # avoid: used only once: possible typo at ...
                    $Esjis::function_ord     = $Esjis::function_ord;
                    $Esjis::function_ord_    = $Esjis::function_ord_;
                    $Esjis::function_reverse = $Esjis::function_reverse;
                    $Esjis::function_getc    = $Esjis::function_getc;
                }
            }
        }
    }

    $e_script .= Sjis::escape();

    return $e_script;
}

1;

__END__

=__PLAIN_OLD_DOCUMENT__

=head1 NAME

Char - Native Encoding Support by Traditional Scripting

=head1 Install and Usage

There are two steps there:

=over 2

=item * You'll have to download Sjis.pm and Esjis.pm and put it in your perl lib directory.

=item * You'll need to write "use Sjis;" at head of the script.

=back

=head1 SYNOPSIS

  use Sjis;
  use Sjis ver.sion;             --- require minimum version
  use Sjis ver.sion.0;           --- expects version (match or die)
#if MULTIBYTE_ENCODING
  use Sjis qw(ord reverse getc); --- demand enhanced feature of ord, reverse, and getc
  use Sjis ver.sion qw(ord reverse getc);
  use Sjis ver.sion.0 qw(ord reverse getc);
#endif

  # "no Sjis;" not supported

  or

  $ perl Sjis.pm ShiftJIS_script.pl > Escaped_script.pl.e

  then

  $ perl Escaped_script.pl.e

  ShiftJIS_script.pl  --- script written in ShiftJIS
  Escaped_script.pl.e --- escaped script

  subroutines:
#if MULTIBYTE_ENCODING
    Sjis::ord(...);
    Sjis::reverse(...);
    Sjis::getc(...);
    Sjis::length(...);
    Sjis::substr(...);
    Sjis::index(...);
    Sjis::rindex(...);
#endif
    Sjis::eval(...);
#if MULTIBYTE_ENCODING
  functions:
    <*>
    glob(...);
    CORE::chop(...);
    CORE::ord(...);
    CORE::reverse(...);
    CORE::getc(...);
    CORE::index(...);
    CORE::rindex(...);
#endif
  dummy functions:
    utf8::upgrade(...);
    utf8::downgrade(...);
    utf8::encode(...);
    utf8::decode(...);
    utf8::is_utf8(...);
    utf8::valid(...);
    bytes::chr(...);
    bytes::index(...);
    bytes::length(...);
    bytes::ord(...);
    bytes::rindex(...);
    bytes::substr(...);

=head1 ABSTRACT

Sjis software is "middleware" between perl interpreter and your Perl script
written in ShiftJIS.

Perl is optimized for problems which are about 90% working with text and about
10% everything else. Even if this "text" doesn't contain ShiftJIS, Perl3 or later
can treat ShiftJIS as binary data.

By "use Sjis;", it automatically interpret your script as ShiftJIS. The various
functions of perl including a regular expression can treat ShiftJIS now.
The function length treats length per byte. This software does not use UTF8
flag.

=head1 Yet Another Future Of

JPerl is very useful software. -- Oops, note, this "JPerl" means "Japanized Perl"
or "Japanese Perl". Therefore, it is unrelated to JPerl of the following.

 JPerl is an implementation of Perl written in Java.
 http://www.javainc.com/projects/jperl/
 
 jPerl - Perl on the JVM
 http://www.dzone.com/links/175948.html
 
 Jamie's PERL scripts for bioinformatics
 http://code.google.com/p/jperl/
 
 jperl (Jonathan Perl)
 https://github.com/jperl

Now, the last version of JPerl is 5.005_04 and is not maintained now.

Japanization modifier WATANABE Hirofumi said,

  "Because WATANABE am tired I give over maintaing JPerl."

at Slide #15: "The future of JPerl" of

L<ftp://ftp.oreilly.co.jp/pcjp98/watanabe/jperlconf.ppt>

in The Perl Confernce Japan 1998.

When I heard it, I thought that someone excluding me would maintain JPerl.
And I slept every night hanging a sock. Night and day, I kept having hope.
After 10 years, I noticed that white beard exists in the sock :-)

This software is a source code filter to escape Perl script encoded by ShiftJIS
given from STDIN or command line parameter. The character code is never converted
by escaping the script. Neither the value of the character nor the length of the
character string change even if it escapes.

I learned the following things from the successful software.

=over 2

=item * Upper Compatibility like Perl4 to Perl5

=item * Maximum Portability like jcode.pl

=item * Remains One Language Handling Raw ShiftJIS, Doesn't Use UTF8 flag like JPerl

=item * Remains One Interpreter like Encode module

=item * Code Set Independent like Ruby

=item * Monolithic Script like cpanminus

=item * There's more than one way to do it like Perl itself

=back

I am excited about this software and Perl's future --- I hope you are too.

=head1 JRE: JPerl Runtime Environment

  +---------------------------------------+
  |        JPerl Application Script       | Your Script
  +---------------------------------------+
  |  Source Code Filter, Runtime Routine  | ex. Sjis.pm, Esjis.pm
  +---------------------------------------+
  |          PVM 5.00503 or later         | ex. perl 5.00503
  +---------------------------------------+

A Perl Virtual Machine (PVM) enables a set of computer software programs and
data structures to use a virtual machine model for the execution of other
computer programs and scripts. The model used by a PVM accepts a form of
computer intermediate language commonly referred to as Perl byteorientedcode.
This language conceptually represents the instruction set of a byte-oriented,
capability architecture.

=head1 Basic Idea of Source Code Filter

I discovered this mail again recently.

[Tokyo.pm] jus Benkyoukai

http://mail.pm.org/pipermail/tokyo-pm/1999-September/001854.html

save as: SJIS.pm

  package SJIS;
  use Filter::Util::Call;
  sub multibyte_filter {
      my $status;
      if (($status = filter_read()) > 0 ) {
          s/([\x81-\x9f\xe0-\xef])([\x40-\x7e\x80-\xfc])/
              sprintf("\\x%02x\\x%02x",ord($1),ord($2))
          /eg;
      }
      $status;
  }
  sub import {
      filter_add(\&multibyte_filter);
  }
  1;

I am glad that I could confirm my idea is not so wrong.

=head1 Command-line Wildcard Expansion on DOS-like Systems

The default command shells on DOS-like systems (COMMAND.COM or cmd.exe or
Win95Cmd.exe) do not expand wildcard arguments supplied to programs. Instead,
import of Esjis.pm works well.

   in Esjis.pm
   #
   # @ARGV wildcard globbing
   #
   sub import {

       if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
           my @argv = ();
           for (@ARGV) {

               # has space
               if (/\A (?:$q_char)*? [ ] /oxms) {
                   if (my @glob = Esjis::glob(qq{"$_"})) {
                       push @argv, @glob;
                   }
                   else {
                       push @argv, $_;
                   }
               }

               # has wildcard metachar
               elsif (/\A (?:$q_char)*? [*?] /oxms) {
                   if (my @glob = Esjis::glob($_)) {
                       push @argv, @glob;
                   }
                   else {
                       push @argv, $_;
                   }
               }

               # no wildcard globbing
               else {
                   push @argv, $_;
               }
           }
           @ARGV = @argv;
       }
   }

=head1 Software Composition

   Sjis.pm               --- source code filter to escape ShiftJIS
   Esjis.pm              --- run-time routines for Sjis.pm

=head1 Upper Compatibility by Escaping

This software adds the function by 'Escaping' it always, and nothing of the
past is broken. Therefore, 'Possible job' never becomes 'Impossible job'.
This approach is effective in the field where the retreat is never permitted.
It means incompatible upgrade of Perl should be rewound.

=head1 Escaping Your Script (You do)

You need write 'use Sjis;' in your script.

  ---------------------
  You do
  ---------------------
  use Sjis;
  ---------------------

#if ESCAPE_SECOND_OCTET
=head1 Escaping Multiple-Octet Code (Sjis software provides)

Insert chr(0x5c) before  @  [  \  ]  ^  `  {  |  and  }  in multiple-octet of

=over 2

=item * string in single quote ('', q{}, <<'END', and qw{})

=item * string in double quote ("", qq{}, <<END, <<"END", ``, qx{}, and <<`END`)

=item * regexp in single quote (m'', s''', split(''), split(m''), and qr'')

=item * regexp in double quote (//, m//, ??, s///, split(//), split(m//), and qr//)

=item * character in tr/// (tr/// and y///)

=back

  ex. Japanese Katakana "SO" like [ `/ ] code is "\x83\x5C" in SJIS
 
                  see     hex dump
  -----------------------------------------
  source script   "`/"    [83 5c]
  -----------------------------------------
 
  Here, use SJIS;
                          hex dump
  -----------------------------------------
  escaped script  "`\/"   [83 [5c] 5c]
  -----------------------------------------
                    ^--- escape by SJIS software
 
  by the by       see     hex dump
  -----------------------------------------
  your eye's      "`/\"   [83 5c] [5c]
  -----------------------------------------
  perl eye's      "`\/"   [83] \[5c]
  -----------------------------------------
 
                          hex dump
  -----------------------------------------
  in the perl     "`/"    [83] [5c]
  -----------------------------------------

#endif
#if MULTIBYTE_ANCHORING
=head1 Multiple-Octet Anchoring of Regular Expression (Sjis software provides)

Sjis software applies multiple-octet anchoring at beginning of regular expression.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  m/regexp/               m/${Esjis::anchor}(?:regexp).../
  --------------------------------------------------------------------------------

#endif
#if DIST_UTF2
=head1 Multiple-Octet Anchoring of Regular Expression

This software requires valid UTF8-encoded Unicode instead of using a
multi-octet anchoring.

#endif
#if DIST_OLDUTF8
=head1 Multiple-Octet Anchoring of Regular Expression

This software requires valid Modified UTF8-encoded Unicode instead of
using a multi-octet anchoring.

#endif
#if ESCAPE_SECOND_OCTET
=head1 Escaping Second Octet (Sjis software provides)

Sjis software escapes second octet of multiple-octet character in regular
expression.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  m<...`/...>             m<...`/\...>
  --------------------------------------------------------------------------------

#endif
#if MULTIBYTE_ENCODING
=head1 Multiple-Octet Character Regular Expression (Sjis software provides)

Sjis software clusters multiple-octet character with quantifier, makes cluster from
multiple-octet custom character classes. And makes multiple-octet version metasymbol
from classic Perl character class shortcuts and POSIX-style character classes.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  m/...MULTIOCT+.../      m/...(?:MULTIOCT)+.../
  m/...[AN-EM].../        m/...(?:A[N-Z]|[B-D][A-Z]|E[A-M]).../
  m/...\D.../             m/...(?:${Esjis::eD}).../
  m/...[[:^digit:]].../   m/...(?:${Esjis::not_digit}).../
  --------------------------------------------------------------------------------

#endif
=head1 Calling 'Esjis::ignorecase()' (Sjis software provides)

Sjis software applies calling 'Esjis::ignorecase()' instead of /i modifier.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  m/...$var.../i          m/...@{[Esjis::ignorecase($var)]}.../
  --------------------------------------------------------------------------------

#if MULTIBYTE_ENCODING
=head1 Character-Oriented Regular Expression

Regular expression works as character-oriented that has no /b modifier.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  /regexp/                /ditto$Esjis::matched/
  m/regexp/               m/ditto$Esjis::matched/
  ?regexp?                m?ditto$Esjis::matched?
  m?regexp?               m?ditto$Esjis::matched?
 
  $_ =~                   ($_ =~ m/ditto$Esjis::matched/) ?
  s/regexp/replacement/   CORE::eval{ Esjis::s_matched(); local $^W=0; my $__r=qq/replacement/; $_="${1}$__r$'"; 1 } :
                          undef
 
  $_ !~                   ($_ !~ m/ditto$Esjis::matched/) ?
  s/regexp/replacement/   1 :
                          CORE::eval{ Esjis::s_matched(); local $^W=0; my $__r=qq/replacement/; $_="${1}$__r$'"; undef }
 
  split(/regexp/)         Esjis::split(qr/regexp/)
  split(m/regexp/)        Esjis::split(qr/regexp/)
  split(qr/regexp/)       Esjis::split(qr/regexp/)
  qr/regexp/              qr/ditto$Esjis::matched/
  --------------------------------------------------------------------------------

=head1 Byte-Oriented Regular Expression

Regular expression works as byte-oriented that has /b modifier.

  --------------------------------------------------------------------------------
  Before                  After
  --------------------------------------------------------------------------------
  /regexp/b               /(?:regexp)$Esjis::matched/
  m/regexp/b              m/(?:regexp)$Esjis::matched/
  ?regexp?b               m?regexp$Esjis::matched?
  m?regexp?b              m?regexp$Esjis::matched?
 
  $_ =~                   ($_ =~ m/(\G[\x00-\xFF]*?)(?:regexp)$Esjis::matched/) ?
  s/regexp/replacement/b  CORE::eval{ Esjis::s_matched(); local $^W=0; my $__r=qq/replacement/; $_="${1}$__r$'"; 1 } :
                          undef
 
  $_ !~                   ($_ !~ m/(\G[\x00-\xFF]*?)(?:regexp)$Esjis::matched/) ?
  s/regexp/replacement/b  1 :
                          CORE::eval{ Esjis::s_matched(); local $^W=0; my $__r=qq/replacement/; $_="${1}$__r$'"; undef }
 
  split(/regexp/b)        split(qr/regexp/)
  split(m/regexp/b)       split(qr/regexp/)
  split(qr/regexp/b)      split(qr/regexp/)
  qr/regexp/b             qr/(?:regexp)$Esjis::matched/
  --------------------------------------------------------------------------------

#endif
=head1 Escaping Character Classes (Esjis.pm provides)

The character classes are redefined as follows to backward compatibility.

  ---------------------------------------------------------------
  Before        After
  ---------------------------------------------------------------
   .            ${Esjis::dot}
                ${Esjis::dot_s}    (/s modifier)
  \d            [0-9]              (universally)
  \s            \s
  \w            [0-9A-Z_a-z]       (universally)
  \D            ${Esjis::eD}
  \S            ${Esjis::eS}
  \W            ${Esjis::eW}
  \h            [\x09\x20]
  \v            [\x0A\x0B\x0C\x0D]
  \H            ${Esjis::eH}
  \V            ${Esjis::eV}
  \C            [\x00-\xFF]
  \X            X                  (so, just 'X')
  \R            ${Esjis::eR}
  \N            ${Esjis::eN}
  ---------------------------------------------------------------

Also POSIX-style character classes.

  ---------------------------------------------------------------
  Before        After
  ---------------------------------------------------------------
  [:alnum:]     [\x30-\x39\x41-\x5A\x61-\x7A]
  [:alpha:]     [\x41-\x5A\x61-\x7A]
  [:ascii:]     [\x00-\x7F]
  [:blank:]     [\x09\x20]
  [:cntrl:]     [\x00-\x1F\x7F]
  [:digit:]     [\x30-\x39]
  [:graph:]     [\x21-\x7F]
  [:lower:]     [\x61-\x7A]
                [\x41-\x5A\x61-\x7A]     (/i modifier)
  [:print:]     [\x20-\x7F]
  [:punct:]     [\x21-\x2F\x3A-\x3F\x40\x5B-\x5F\x60\x7B-\x7E]
  [:space:]     [\s\x0B]
  [:upper:]     [\x41-\x5A]
                [\x41-\x5A\x61-\x7A]     (/i modifier)
  [:word:]      [\x30-\x39\x41-\x5A\x5F\x61-\x7A]
  [:xdigit:]    [\x30-\x39\x41-\x46\x61-\x66]
  [:^alnum:]    ${Esjis::not_alnum}
  [:^alpha:]    ${Esjis::not_alpha}
  [:^ascii:]    ${Esjis::not_ascii}
  [:^blank:]    ${Esjis::not_blank}
  [:^cntrl:]    ${Esjis::not_cntrl}
  [:^digit:]    ${Esjis::not_digit}
  [:^graph:]    ${Esjis::not_graph}
  [:^lower:]    ${Esjis::not_lower}
                ${Esjis::not_lower_i}    (/i modifier)
  [:^print:]    ${Esjis::not_print}
  [:^punct:]    ${Esjis::not_punct}
  [:^space:]    ${Esjis::not_space}
  [:^upper:]    ${Esjis::not_upper}
                ${Esjis::not_upper_i}    (/i modifier)
  [:^word:]     ${Esjis::not_word}
  [:^xdigit:]   ${Esjis::not_xdigit}
  ---------------------------------------------------------------

\b and \B are redefined as follows to backward compatibility.

  ---------------------------------------------------------------
  Before      After
  ---------------------------------------------------------------
  \b          ${Esjis::eb}
  \B          ${Esjis::eB}
  ---------------------------------------------------------------

Definitions in Esjis.pm.

  ---------------------------------------------------------------------------------------------------------------------------------------------------------
  After                    Definition
  ---------------------------------------------------------------------------------------------------------------------------------------------------------
#if MULTIBYTE_ANCHORING
  ${Esjis::anchor}         qr{\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?}
#endif
#if DIST_GB18030
                           for over 32766 octets string on ActivePerl5.6 and Perl5.10 or later
                           qr{\G(?(?=.{0,32766}\z)\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[\x00-\x80\xFF]+\z).*?|(?:.*?[^\x30-\x39\x81-\xFE](?>[\x30-\x39]|[\x81-\xFE][\x81-\xFE]|[\x81-\xFE][\x30-\x39][\x81-\xFE][\x30-\x39])*?)))}oxms
#endif
#if DIST_EUCTW
                           for over 32766 octets string on ActivePerl5.6 and Perl5.10 or later
                           qr{\G(?(?=.{0,32766}\z)\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[\x00-\x8D\x8F-\xA0\xFF]+\z).*?|(?:.*?[^\x8E\xA1-\xFE](?>[\xA1-\xFE][\xA1-\xFE]|\x8E[\xA1-\xB0][\xA1-\xFE][\xA1-\xFE])*?)))}oxms
#endif
#if DIST_EUCJP
                           for over 32766 octets string on ActivePerl5.6 and Perl5.10 or later
                           qr{\G(?(?=.{0,32766}\z)\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[\x00-\x8D\x90-\xA0\xFF]+\z).*?|(?:.*?[^\x8E\x8F\xA1-\xFE](?>[\x8E\xA1-\xFE][\xA1-\xFE]|\x8F[\xA1-\xFE][\xA1-\xFE])*?)))}oxms
#endif
#if DIST_INFORMIXV6ALS
                           for over 32766 octets string on ActivePerl5.6 and Perl5.10 or later
                           qr{\G(?(?=.{0,32766}\z)\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[\x00-\x80\xA0-\xDF\xFE]+\z).*?|(?:.*?[\x00-\x80\xA0-\xDF\xFF](?>[\x81-\x9F\xE0-\xFC][\x40-\x7E\x80-\xFC]|\xFD[\xA1-\xFE][\xA1-\xFE]|\xFE)*?)))}oxms
#endif
#if LONG_STRING_FOR_RE
                           for over 32766 octets string on ActivePerl5.6 and Perl5.10 or later
                           qr{\G(?(?=.{0,32766}\z)\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[$sbcs]+\z).*?|(?:.*?[$sbcs](?:[^$sbcs][^$sbcs])*?)))}oxms
#endif
  ${Esjis::dot}            qr{(?>[^\x81-\x9F\xE0-\xFC\x0A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::dot_s}          qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eD}             qr{(?>[^\x81-\x9F\xE0-\xFC0-9]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eS}             qr{(?>[^\x81-\x9F\xE0-\xFC\s]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eW}             qr{(?>[^\x81-\x9F\xE0-\xFC0-9A-Z_a-z]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eH}             qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eV}             qr{(?>[^\x81-\x9F\xE0-\xFC\x0A\x0B\x0C\x0D]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::eR}             qr{(?>\x0D\x0A|[\x0A\x0D])};
  ${Esjis::eN}             qr{(?>[^\x81-\x9F\xE0-\xFC\x0A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_alnum}      qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_alpha}      qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_ascii}      qr{(?>[^\x81-\x9F\xE0-\xFC\x00-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_blank}      qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_cntrl}      qr{(?>[^\x81-\x9F\xE0-\xFC\x00-\x1F\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_digit}      qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_graph}      qr{(?>[^\x81-\x9F\xE0-\xFC\x21-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_lower}      qr{(?>[^\x81-\x9F\xE0-\xFC\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_lower_i}    qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])}; # Perl 5.16 compatible
# ${Esjis::not_lower_i}    qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};                   # older Perl compatible
  ${Esjis::not_print}      qr{(?>[^\x81-\x9F\xE0-\xFC\x20-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_punct}      qr{(?>[^\x81-\x9F\xE0-\xFC\x21-\x2F\x3A-\x3F\x40\x5B-\x5F\x60\x7B-\x7E]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_space}      qr{(?>[^\x81-\x9F\xE0-\xFC\s\x0B]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_upper}      qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_upper_i}    qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])}; # Perl 5.16 compatible
# ${Esjis::not_upper_i}    qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};                   # older Perl compatible
  ${Esjis::not_word}       qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x5A\x5F\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  ${Esjis::not_xdigit}     qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x46\x61-\x66]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
  
  # This solution is not perfect. I beg better solution from you who are reading this.
  ${Esjis::eb}             qr{(?:\A(?=[0-9A-Z_a-z])|(?<=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF])(?=[0-9A-Z_a-z])|(?<=[0-9A-Z_a-z])(?=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF]|\z))};
  ${Esjis::eB}             qr{(?:(?<=[0-9A-Z_a-z])(?=[0-9A-Z_a-z])|(?<=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF])(?=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF]))};
  ---------------------------------------------------------------------------------------------------------------------------------------------------------

=head1 Un-Escaping \ of \b{}, \B{}, \N{}, \p{}, \P{}, and \X (Sjis software provides)

Sjis software removes '\' at head of alphanumeric regexp metasymbols \b{}, \B{},
\N{}, \p{}, \P{} and \X. By this method, you can avoid the trap of the abstraction.

See also,
Deprecate literal unescaped "{" in regexes.
http://perl5.git.perl.org/perl.git/commit/2a53d3314d380af5ab5283758219417c6dfa36e9

  ------------------------------------
  Before           After
  ------------------------------------
  \b{...}          b\{...}
  \B{...}          B\{...}
  \N{CHARNAME}     N\{CHARNAME}
  \p{L}            p\{L}
  \p{^L}           p\{^L}
  \p{\^L}          p\{\^L}
  \pL              pL
  \P{L}            P\{L}
  \P{^L}           P\{^L}
  \P{\^L}          P\{\^L}
  \PL              PL
  \X               X
  ------------------------------------

=head1 Escaping Built-in Functions (Sjis software provides)

Insert 'Esjis::' at head of function name. Esjis.pm provides your script Esjis::*
subroutines.

  -------------------------------------------
  Before      After            Works as
  -------------------------------------------
#if MULTIBYTE_ENCODING
  length      length           Byte
  substr      substr           Byte
  pos         pos              Byte
  split       Esjis::split     Character
  tr///       Esjis::tr        Character
  tr///b      tr///            Byte
  tr///B      tr///            Byte
  y///        Esjis::tr        Character
  y///b       tr///            Byte
  y///B       tr///            Byte
  chop        Esjis::chop      Character
#endif
#if MULTIBYTE_ANCHORING
  index       Esjis::index     Character
  rindex      Esjis::rindex    Character
#endif
  lc          Esjis::lc        Character
  lcfirst     Esjis::lcfirst   Character
  uc          Esjis::uc        Character
  ucfirst     Esjis::ucfirst   Character
  fc          Esjis::fc        Character
  chr         Esjis::chr       Character
  glob        Esjis::glob      Character
#if ESCAPE_SECOND_OCTET
  lstat       Esjis::lstat     Character
  opendir     Esjis::opendir   Character
  stat        Esjis::stat      Character
  unlink      Esjis::unlink    Character
  chdir       Esjis::chdir     Character
  do          Esjis::do        Character
  require     Esjis::require   Character
#endif
  -------------------------------------------

  ------------------------------------------------------------------------------------------------------------------------
  Before                   After
  ------------------------------------------------------------------------------------------------------------------------
  use Perl::Module;        BEGIN { Esjis::require 'Perl/Module.pm'; Perl::Module->import() if Perl::Module->can('import'); }
  use Perl::Module @list;  BEGIN { Esjis::require 'Perl/Module.pm'; Perl::Module->import(@list) if Perl::Module->can('import'); }
  use Perl::Module ();     BEGIN { Esjis::require 'Perl/Module.pm'; }
  no Perl::Module;         BEGIN { Esjis::require 'Perl/Module.pm'; Perl::Module->unimport() if Perl::Module->can('unimport'); }
  no Perl::Module @list;   BEGIN { Esjis::require 'Perl/Module.pm'; Perl::Module->unimport(@list) if Perl::Module->can('unimport'); }
  no Perl::Module ();      BEGIN { Esjis::require 'Perl/Module.pm'; }
  ------------------------------------------------------------------------------------------------------------------------

#if ESCAPE_SECOND_OCTET
=head1 Escaping File Test Operators (Sjis software provides)

Insert 'Esjis::' instead of '-' of operator.

  Available in MSWin32, MacOS, and UNIX-like systems
  --------------------------------------------------------------------------
  Before   After      Meaning
  --------------------------------------------------------------------------
  -r       Esjis::r   File or directory is readable by this (effective) user or group
  -w       Esjis::w   File or directory is writable by this (effective) user or group
  -e       Esjis::e   File or directory name exists
  -x       Esjis::x   File or directory is executable by this (effective) user or group
  -z       Esjis::z   File exists and has zero size (always false for directories)
  -f       Esjis::f   Entry is a plain file
  -d       Esjis::d   Entry is a directory
  -t       -t         The filehandle is a TTY (as reported by the isatty() system function;
                      filenames can't be tested by this test)
  -T       Esjis::T   File looks like a "text" file
  -B       Esjis::B   File looks like a "binary" file
  -M       Esjis::M   Modification age (measured in days)
  -A       Esjis::A   Access age (measured in days)
  -C       Esjis::C   Inode-modification age (measured in days)
  -s       Esjis::s   File or directory exists and has nonzero size
                      (the value is the size in bytes)
  --------------------------------------------------------------------------
  
  Available in MacOS and UNIX-like systems
  --------------------------------------------------------------------------
  Before   After      Meaning
  --------------------------------------------------------------------------
  -R       Esjis::R   File or directory is readable by this real user or group
  -W       Esjis::W   File or directory is writable by this real user or group
  -X       Esjis::X   File or directory is executable by this real user or group
  -l       Esjis::l   Entry is a symbolic link
  -S       Esjis::S   Entry is a socket
  --------------------------------------------------------------------------
  
  Not available in MSWin32 and MacOS
  --------------------------------------------------------------------------
  Before   After      Meaning
  --------------------------------------------------------------------------
  -o       Esjis::o   File or directory is owned by this (effective) user
  -O       Esjis::O   File or directory is owned by this real user
  -p       Esjis::p   Entry is a named pipe (a "fifo")
  -b       Esjis::b   Entry is a block-special file (like a mountable disk)
  -c       Esjis::c   Entry is a character-special file (like an I/O device)
  -u       Esjis::u   File or directory is setuid
  -g       Esjis::g   File or directory is setgid
  -k       Esjis::k   File or directory has the sticky bit set
  --------------------------------------------------------------------------

-w only inspects the read-only file attribute (FILE_ATTRIBUTE_READONLY), which
determines whether the directory can be deleted, not whether it can be written
to. Directories always have read and write access unless denied by
discretionary access control lists (DACLs). (MSWin32)
-R, -W, -X, -O are indistinguishable from -r, -w, -x, -o. (MSWin32)
-g, -k, -l, -u, -A are not particularly meaningful. (MSWin32)
-x (or -X) determine if a file ends in one of the executable suffixes. -S is
meaningless. (MSWin32)

As of Perl 5.00503, as a form of purely syntactic sugar, you can stack file
test operators, in a way that -w -x $file is equivalent to -x $file && -w _ .

  if ( -w -r $file ) {
      print "The file is both readable and writable!\n";
  }

#endif
#if MULTIBYTE_ENCODING
=head1 Escaping Function Name (You do)

You need write 'Sjis::' at head of function name when you want character-
oriented subroutine. See 'Character-Oriented Subroutines'.

  --------------------------------------------------------
  Function   Character-Oriented   Description
  --------------------------------------------------------
  ord        Sjis::ord
  reverse    Sjis::reverse
  getc       Sjis::getc
  length     Sjis::length
  substr     Sjis::substr
  index      Sjis::index          See 'About Indexes'
  rindex     Sjis::rindex         See 'About Rindexes'
  eval       Sjis::eval
  --------------------------------------------------------

  About Indexes
  -------------------------------------------------------------------------
  Function       Works as    Returns as   Description
  -------------------------------------------------------------------------
  index          Character   Byte         JPerl semantics (most useful)
  (same as Esjis::index)
  Sjis::index    Character   Character    Character-oriented semantics
  CORE::index    Byte        Byte         Byte-oriented semantics
  (nothing)      Byte        Character    (most useless)
  -------------------------------------------------------------------------

  About Rindexes
  -------------------------------------------------------------------------
  Function       Works as    Returns as   Description
  -------------------------------------------------------------------------
  rindex         Character   Byte         JPerl semantics (most useful)
  (same as Esjis::rindex)
  Sjis::rindex   Character   Character    Character-oriented semantics
  CORE::rindex   Byte        Byte         Byte-oriented semantics
  (nothing)      Byte        Character    (most useless)
  -------------------------------------------------------------------------

=head1 Character-Oriented Subsroutines

=over 2

=item * Ordinal Value of Character

  $ord = Sjis::ord($string);

  This subroutine returns the numeric value (ASCII or ShiftJIS character) of the
  first character of $string, not Unicode. If $string is omitted, it uses $_.
  The return value is always unsigned.

  If you import ord "use Sjis qw(ord);", ord of your script will be rewritten in
  Sjis::ord. Sjis::ord is not compatible with ord of JPerl.

=item * Reverse List or String

  @reverse = Sjis::reverse(@list);
  $reverse = Sjis::reverse(@list);

  In list context, this subroutine returns a list value consisting of the elements
  of @list in the opposite order.

  In scalar context, the subroutine concatenates all the elements of @list and
  then returns the reverse of that resulting string, character by character.

  If you import reverse "use Sjis qw(reverse);", reverse of your script will be
  rewritten in Sjis::reverse. Sjis::reverse is not compatible with reverse of
  JPerl.

  Even if you do not know this subroutine, there is no problem. This subroutine
  can be created with

  $rev = join('', reverse(split(//, $jstring)));

  as before.

  See:
  P.558 JPerl (Japanese Perl)
  Appendix C Supplement the Japanese version
  ISBN 4-89052-384-7 PERL PUROGURAMINGU

=item * Returns Next Character

  $getc = Sjis::getc(FILEHANDLE);
  $getc = Sjis::getc($filehandle);
  $getc = Sjis::getc;

  This subroutine returns the next character from the input file attached to
  FILEHANDLE. It returns undef at end-of-file, or if an I/O error was encountered.
  If FILEHANDLE is omitted, the subroutine reads from STDIN.

  This subroutine is somewhat slow, but it's occasionally useful for
  single-character input from the keyboard -- provided you manage to get your
  keyboard input unbuffered. This subroutine requests unbuffered input from the
  standard I/O library. Unfortunately, the standard I/O library is not so standard
  as to provide a portable way to tell the underlying operating system to supply
  unbuffered keyboard input to the standard I/O system. To do that, you have to
  be slightly more clever, and in an operating-system-dependent fashion. Under
  Unix you might say this:

  if ($BSD_STYLE) {
      system "stty cbreak </dev/tty >/dev/tty 2>&1";
  }
  else {
      system "stty", "-icanon", "eol", "\001";
  }

  $key = Sjis::getc;

  if ($BSD_STYLE) {
      system "stty -cbreak </dev/tty >/dev/tty 2>&1";
  }
  else {
      system "stty", "icanon", "eol", "^@"; # ASCII NUL
  }
  print "\n";

  This code puts the next character typed on the terminal in the string $key. If
  your stty program has options like cbreak, you'll need to use the code where
  $BSD_STYLE is true. Otherwise, you'll need to use the code where it is false.

  If you import getc "use Sjis qw(getc);", getc of your script will be rewritten
  in Sjis::getc. Sjis::getc is not compatible with getc of JPerl.

=item * Length by ShiftJIS Character

  $length = Sjis::length($string);
  $length = Sjis::length();

  This subroutine returns the length in characters (programmer-visible characters)
  of the scalar value $string. If $string is omitted, it returns the Sjis::length
  of $_.

  Do not try to use Sjis::length to find the size of an array or hash. Use scalar
  @array for the size of an array, and scalar keys %hash for the number of key/value
  pairs in a hash. (The scalar is typically omitted when redundant.)

  To find the length of a string in bytes rather than characters, say simply:

  $bytes = length($string);

  Even if you do not know this subroutine, there is no problem. This subroutine
  can be created with

  $len = split(//, $jstring);

  as before.

  See:
  P.558 JPerl (Japanese Perl)
  Appendix C Supplement the Japanese version
  ISBN 4-89052-384-7 PERL PUROGURAMINGU

=item * Substr by ShiftJIS Character

  $substr = Sjis::substr($string,$offset,$length,$replacement);
  $substr = Sjis::substr($string,$offset,$length);
  $substr = Sjis::substr($string,$offset);

  This subroutine extracts a substring out of the string given by $string and returns
  it. The substring is extracted starting at $offset characters from the front of
  the string. First character is at offset zero. If $offset is negative, starts that
  far back from the end of the string.
  If $length is omitted, returns everything through the end of the string. If $length
  is negative, leaves that many characters off the end of the string. Otherwise,
  $length indicates the length of the substring to extract, which is sort of what
  you'd expect.

  my $s = "The black cat climbed the green tree";
  my $color  = Sjis::substr $s, 4, 5;      # black
  my $middle = Sjis::substr $s, 4, -11;    # black cat climbed the
  my $end    = Sjis::substr $s, 14;        # climbed the green tree
  my $tail   = Sjis::substr $s, -4;        # tree
  my $z      = Sjis::substr $s, -4, 2;     # tr

  If Perl version 5.14 or later, you can use the Sjis::substr() subroutine as an
  lvalue. In its case $string must itself be an lvalue. If you assign something
  shorter than $length, the string will shrink, and if you assign something longer
  than $length, the string will grow to accommodate it. To keep the string the
  same length, you may need to pad or chop your value using sprintf.

  If $offset and $length specify a substring that is partly outside the string,
  only the part within the string is returned. If the substring is beyond either
  end of the string, Sjis::substr() returns the undefined value and produces a
  warning. When used as an lvalue, specifying a substring that is entirely outside
  the string raises an exception. Here's an example showing the behavior for
  boundary cases:

  my $name = 'fred';
  Sjis::substr($name, 4) = 'dy';         # $name is now 'freddy'
  my $null = Sjis::substr $name, 6, 2;   # returns "" (no warning)
  my $oops = Sjis::substr $name, 7;      # returns undef, with warning
  Sjis::substr($name, 7) = 'gap';        # raises an exception

  An alternative to using Sjis::substr() as an lvalue is to specify the replacement
  string as the 4th argument. This allows you to replace parts of the $string and
  return what was there before in one operation, just as you can with splice().

  my $s = "The black cat climbed the green tree";
  my $z = Sjis::substr $s, 14, 7, "jumped from";    # climbed
  # $s is now "The black cat jumped from the green tree"

  Note that the lvalue returned by the three-argument version of Sjis::substr() acts
  as a 'magic bullet'; each time it is assigned to, it remembers which part of the
  original string is being modified; for example:

  $x = '1234';
  for (Sjis::substr($x,1,2)) {
      $_ = 'a';   print $x,"\n";    # prints 1a4
      $_ = 'xyz'; print $x,"\n";    # prints 1xyz4
      $x = '56789';
      $_ = 'pq';  print $x,"\n";    # prints 5pq9
  }

  With negative offsets, it remembers its position from the end of the string when
  the target string is modified:

  $x = '1234';
  for (Sjis::substr($x, -3, 2)) {
      $_ = 'a';   print $x,"\n";    # prints 1a4, as above
      $x = 'abcdefg';
      print $_,"\n";                # prints f
  }

  Prior to Perl version 5.10, the result of using an lvalue multiple times was
  unspecified. Prior to 5.16, the result with negative offsets was unspecified.

=item * Index by ShiftJIS Character

  $index = Sjis::index($string,$substring,$offset);
  $index = Sjis::index($string,$substring);

  This subroutine searches for one string within another. It returns the character
  position of the first occurrence of $substring in $string. The $offset, if
  specified, says how many characters from the start to skip before beginning to
  look. Positions are based at 0. If the substring is not found, the subroutine
  returns one less than the base, ordinarily -1. To work your way through a string,
  you might say:

  $pos = -1;
  while (($pos = Sjis::index($string, $lookfor, $pos)) > -1) {
      print "Found at $pos\n";
      $pos++;
  }

=item * Rindex by ShiftJIS Character

  $rindex = Sjis::rindex($string,$substring,$offset);
  $rindex = Sjis::rindex($string,$substring);

  This subroutine works just like Sjis::index except that it returns the character
  position of the last occurrence of $substring in $string (a reverse Sjis::index).
  The subroutine returns -1 if $substring is not found. $offset, if specified, is
  the rightmost character position that may be returned. To work your way through
  a string backward, say:

  $pos = Sjis::length($string);
  while (($pos = Sjis::rindex($string, $lookfor, $pos)) >= 0) {
      print "Found at $pos\n";
      $pos--;
  }

=item * Eval ShiftJIS Script

  $eval = Sjis::eval { block };
  $eval = Sjis::eval $expr;
  $eval = Sjis::eval;

  The Sjis::eval keyword serves two distinct but related purposes in JPerl.
  These purposes are represented by two forms of syntax, Sjis::eval { block }
  and Sjis::eval $expr. The first form traps runtime exceptions (errors)
  that would otherwise prove fatal, similar to the "try block" construct in
  C++ or Java. The second form compiles and executes little bits of code on
  the fly at runtime, and also (conveniently) traps any exceptions just like
  the first form. But the second form runs much slower than the first form,
  since it must parse the string every time. On the other hand, it is also
  more general. Whichever form you use, Sjis::eval is the preferred way to do
  all exception handling in JPerl.

  For either form of Sjis::eval, the value returned from an Sjis::eval is
  the value of the last expression evaluated, just as with subroutines.
  Similarly, you may use the return operator to return a value from the
  middle of the eval. The expression providing the return value is evaluated
  in void, scalar, or list context, depending on the context of the
  Sjis::eval itself. See wantarray for more on how the evaluation context
  can be determined.

  If there is a trappable error (including any produced by the die operator),
  Sjis::eval returns undef and puts the error message (or object) in $@. If
  there is no error, $@ is guaranteed to be set to the null string, so you
  can test it reliably afterward for errors. A simple Boolean test suffices:

      Sjis::eval { ... }; # trap runtime errors
      if ($@) { ... }     # handle error

  (Prior to Perl 5.16, a bug caused undef to be returned in list context for
  syntax errors, but not for runtime errors.)

  The Sjis::eval { block } form is syntax checked and compiled at compile time,
  so it is just as efficient at runtime as any other block. (People familiar
  with the slow Sjis::eval $expr form are occasionally confused on this issue.)
  Because the { block } is compiled when the surrounding code is, this form of
  Sjis::eval cannot trap syntax errors.

  The Sjis::eval $expr form can trap syntax errors because it parses the code
  at runtime. (If the parse is unsuccessful, it places the parse error in $@,
  as usual.) If $expr is omitted, evaluates $_ .

  Otherwise, it executes the value of $expr as though it were a little JPerl
  script. The code is executed in the context of the current of the current
  JPerl script, which means that it can see any enclosing lexicals from a
  surrounding scope, and that any nonlocal variable settings remain in effect
  after the Sjis::eval is complete, as do any subroutine or format definitions.
  The code of the Sjis::eval is treated as a block, so any locally scoped
  variables declared within the Sjis::eval last only until the Sjis::eval is
  done. (See my and local.) As with any code in a block, a final semicolon is
  not required.

  Sjis::eval will be escaped as follows:

  -------------------------------------------------
  Before                  After
  -------------------------------------------------
  Sjis::eval { block }    eval { block }
  Sjis::eval $expr        eval Sjis::escape $expr
  Sjis::eval              eval Sjis::escape
  -------------------------------------------------

  To tell the truth, the subroutine Sjis::eval does not exist. If it exists,
  you will troubled, when Sjis::eval has a parameter that is single quoted
  string included my variables. Sjis::escape is a subroutine that makes Perl
  script from JPerl script.

  Here is a simple JPerl shell. It prompts the user to enter a string of
  arbitrary JPerl code, compiles and executes that string, and prints whatever
  error occurred:

      #!/usr/bin/perl
      # jperlshell.pl - simple JPerl shell
      use Sjis;
      print "\nEnter some JPerl code: ";
      while (<STDIN>) {
          Sjis::eval;
          print $@;
          print "\nEnter some more JPerl code: ";
      }

  Here is a rename.pl script to do a mass renaming of files using a JPerl
  expression:

      #!/usr/bin/perl
      # rename.pl - change filenames
      use Sjis;
      $op = shift;
      for (@ARGV) {
          $was = $_;
          Sjis::eval $op;
          die if $@;
          # next line calls the built-in function, not
          # the script by the same name
          if ($was ne $_) {
              print STDERR "rename $was --> $_\n";
              rename($was,$_);
          }
      }

  You'd use that script like this:

      C:\WINDOWS> perl rename.pl 's/\.orig$//' *.orig
      C:\WINDOWS> perl rename.pl 'y/A-Z/a-z/ unless /^Make/' *
      C:\WINDOWS> perl rename.pl '$_ .= ".bad"' *.f

  Since Sjis::eval traps errors that would otherwise prove fatal, it is useful
  for determining whether particular features (such as fork or symlink) are
  implemented.

  Because Sjis::eval { block } is syntax checked at compile time, any syntax
  error is reported earlier. Therefore, if your code is invariant and both
  Sjis::eval $expr and Sjis::eval { block } will suit your purposes equally
  well, the { block } form is preferred. For example:

      # make divide-by-zero nonfatal
      Sjis::eval { $answer = $a / $b; };
      warn $@ if $@;

      # same thing, but less efficient if run multiple times
      Sjis::eval '$answer = $a / $b';
      warn $@ if $@;

      # a compile-time syntax error (not trapped)
      Sjis::eval { $answer = }; # WRONG

      # a runtime syntax error
      Sjis::eval '$answer =';   # sets $@

  Here, the code in the { block } has to be valid JPerl code to make it past
  the compile phase. The code in the $expr doesn't get examined until runtime,
  so it doesn't cause an error until runtime.

  Using the Sjis::eval { block } form as an exception trap in libraries does
  have some issues. Due to the current arguably broken state of __DIE__ hooks,
  you may wish not to trigger any __DIE__ hooks that user code may have
  installed. You can use the local $SIG{__DIE__} construct for this purpose,
  as this example shows:

      # a private exception trap for divide-by-zero
      Sjis::eval { local $SIG{'__DIE__'}; $answer = $a / $b; };
      warn $@ if $@;

  This is especially significant, given that __DIE__ hooks can call die again,
  which has the effect of changing their error messages:

      # __DIE__ hooks may modify error messages
      {
          local $SIG{'__DIE__'} =
              sub { (my $x = $_[0]) =~ s/foo/bar/g; die $x };
          Sjis::eval { die "foo lives here" };
          print $@ if $@;                # prints "bar lives here"
      }

  Because this promotes action at a distance, this counterintuitive behavior
  may be fixed in a future release.

  With an Sjis::eval, you should be especially careful to remember what's being
  looked at when:

      Sjis::eval $x;        # CASE 1
      Sjis::eval "$x";      # CASE 2

      Sjis::eval '$x';      # CASE 3
      Sjis::eval { $x };    # CASE 4

      Sjis::eval "\$$x++";  # CASE 5
      $$x++;                # CASE 6

  CASEs 1 and 2 above behave identically: they run the code contained in the
  variable $x. (Although CASE 2 has misleading double quotes making the reader
  wonder what else might be happening (nothing is).) CASEs 3 and 4 likewise
  behave in the same way: they run the code '$x' , which does nothing but return
  the value of $x. (CASE 4 is preferred for purely visual reasons, but it also
  has the advantage of compiling at compile-time instead of at run-time.) CASE 5
  is a place where normally you would like to use double quotes, except that in
  this particular situation, you can just use symbolic references instead, as
  in CASE 6.

  Before Perl 5.14, the assignment to $@ occurred before restoration of
  localized variables, which means that for your code to run on older versions,
  a temporary is required if you want to mask some but not all errors:

      # alter $@ on nefarious repugnancy only
      {
          my $e;
          {
              local $@; # protect existing $@
              Sjis::eval { test_repugnancy() };
              # $@ =~ /nefarious/ and die $@; # Perl 5.14 and higher only
              $@ =~ /nefarious/ and $e = $@;
          }
          die $e if defined $e
      }

  The block of Sjis::eval { block } does not count as a loop, so the loop
  control statements next, last, or redo cannot be used to leave or restart the
  block.

=item * Filename Globbing

  @glob = glob($expr);
  $glob = glob($expr);
  @glob = glob;
  $glob = glob;
  @glob = <*>;
  $glob = <*>;

  Performs filename expansion (globbing) on $expr, returning the next successive
  name on each call. If $expr is omitted, $_ is globbed instead.

  This operator is implemented via the Esjis::glob() subroutine. See Esjis::glob
  of Esjis.pm for details.

=back

=head1 Byte-Oriented Functions

=over 2

=item * Chop Byte String

  $byte = CORE::chop($string);
  $byte = CORE::chop(@list);
  $byte = CORE::chop;

  This function chops off the last byte of a string variable and returns the
  byte chopped. The CORE::chop operator is used primarily to remove the newline
  from the end of an input record, and is more efficient than using a
  substitution (s/\n$//). If that's all you're doing, then it would be safer to
  use chomp, since CORE::chop always shortens the string no matter what's there,
  and chomp is more selective.

  You cannot CORE::chop a literal, only a variable.

  If you CORE::chop a @list of variables, each string in the list is chopped:

  @lines = `cat myfile`;
  CORE::chop @lines;

  You can CORE::chop anything that is an lvalue, including an assignment:

  CORE::chop($cwd = `pwd`);
  CORE::chop($answer = <STDIN>);

  This is different from:

  $answer = CORE::chop($temp = <STDIN>); # WRONG

  which puts a newline into $answer because CORE::chop returns the byte chopped,
  not the remaining string (which is in $tmp). One way to get the result
  intended here is with substr:

  $answer = substr <STDIN>, 0, -1;

  But this is more commonly written as:

  CORE::chop($answer = <STDIN>);

  In the most general case, CORE::chop can be expressed in terms of substr:

  $last_byte = CORE::chop($var);
  $last_byte = substr($var, -1, 1, ""); # same thing

  Once you understand this equivalence, you can use it to do bigger chops. To
  CORE::chop more than one byte, use substr as an lvalue, assigning a null
  string. The following removes the last five bytes of $caravan:

  substr($caravan, -5) = "";

  The negative subscript causes substr to count from the end of the string
  instead of the beginning. If you wanted to save the bytes so removed, you
  could use the four-argument form of substr, creating something of a quintuple
  CORE::chop:

  $tail = substr($caravan, -5, 5, "");

  If no argument is given, the function chops the $_ variable.

=item * Ordinal Value of Byte

  $ord = CORE::ord($expr);

  This function returns the numeric value of the first byte of $expr, regardless
  of "use Sjis qw(ord);" exists or not. If $expr is omitted, it uses $_.
  The return value is always unsigned.

  If you want a signed value, use unpack('c',$expr). If you want all the bytes of
  the string converted to a list of numbers, use unpack('C*',$expr) instead.

=item * Reverse List or Byte String

  @reverse = CORE::reverse(@list);
  $reverse = CORE::reverse(@list);

  In list context, this function returns a list value consisting of the elements
  of @list in the opposite order.

  In scalar context, the function concatenates all the elements of @list and then
  returns the reverse of that resulting string, byte by byte, regardless of
  "use Sjis qw(reverse);" exists or not.

=item * Returns Next Byte

  $getc = CORE::getc(FILEHANDLE);
  $getc = CORE::getc($filehandle);
  $getc = CORE::getc;

  This function returns the next byte from the input file attached to FILEHANDLE.
  It returns undef at end-of-file, or if an I/O error was encountered. If
  FILEHANDLE is omitted, the function reads from STDIN.

  This function is somewhat slow, but it's occasionally useful for single-byte
  input from the keyboard -- provided you manage to get your keyboard input
  unbuffered. This function requests unbuffered input from the standard I/O library.
  Unfortunately, the standard I/O library is not so standard as to provide a portable
  way to tell the underlying operating system to supply unbuffered keyboard input to
  the standard I/O system. To do that, you have to be slightly more clever, and in
  an operating-system-dependent fashion. Under Unix you might say this:

  if ($BSD_STYLE) {
      system "stty cbreak </dev/tty >/dev/tty 2>&1";
  }
  else {
      system "stty", "-icanon", "eol", "\001";
  }

  $key = CORE::getc;

  if ($BSD_STYLE) {
      system "stty -cbreak </dev/tty >/dev/tty 2>&1";
  }
  else {
      system "stty", "icanon", "eol", "^@"; # ASCII NUL
  }
  print "\n";

  This code puts the next single-byte typed on the terminal in the string $key.
  If your stty program has options like cbreak, you'll need to use the code where
  $BSD_STYLE is true. Otherwise, you'll need to use the code where it is false.

=item * Index by Byte String

  $index = CORE::index($string,$substring,$offset);
  $index = CORE::index($string,$substring);

  This function searches for one byte string within another. It returns the position
  of the first occurrence of $substring in $string. The $offset, if specified, says
  how many bytes from the start to skip before beginning to look. Positions are based
  at 0. If the substring is not found, the function returns one less than the base,
  ordinarily -1. To work your way through a string, you might say:

  $pos = -1;
  while (($pos = CORE::index($string, $lookfor, $pos)) > -1) {
      print "Found at $pos\n";
      $pos++;
  }

=item * Rindex by Byte String

  $rindex = CORE::rindex($string,$substring,$offset);
  $rindex = CORE::rindex($string,$substring);

  This function works just like CORE::index except that it returns the position of
  the last occurrence of $substring in $string (a reverse CORE::index). The function
  returns -1 if not $substring is found. $offset, if specified, is the rightmost
  position that may be returned. To work your way through a string backward, say:

  $pos = CORE::length($string);
  while (($pos = CORE::rindex($string, $lookfor, $pos)) >= 0) {
      print "Found at $pos\n";
      $pos--;
  }

=back

#endif
=head1 Yada Yada Operator (Sjis software provides)

  The yada yada operator (noted ...) is a placeholder for code. Perl parses it
  without error, but when you try to execute a yada yada, it throws an exception
  with the text Unimplemented:

  sub unimplemented { ... }
  eval { unimplemented() };
  if ( $@ eq 'Unimplemented' ) {
      print "I found the yada yada!\n";
  }

  You can only use the yada yada to stand in for a complete statement. These
  examples of the yada yada work:

  { ... }
  sub foo { ... }
  ...;
  eval { ... };
  sub foo {
      my( $self ) = shift;
      ...;
  }
  do { my $n; ...; print 'Hurrah!' };

  The yada yada cannot stand in for an expression that is part of a larger statement
  since the ... is also the three-dot version of the range operator
  (see "Range Operators"). These examples of the yada yada are still syntax errors:

  print ...;
  open my($fh), '>', '/dev/passwd' or ...;
  if ( $condition && ... ) { print "Hello\n" };

  There are some cases where Perl can't immediately tell the difference between an
  expression and a statement. For instance, the syntax for a block and an anonymous
  hash reference constructor look the same unless there's something in the braces that
  give Perl a hint. The yada yada is a syntax error if Perl doesn't guess that the
  { ... } is a block. In that case, it doesn't think the ... is the yada yada because
  it's expecting an expression instead of a statement:

  my @transformed = map { ... } @input;  # syntax error

  You can use a ; inside your block to denote that the { ... } is a block and not a
  hash reference constructor. Now the yada yada works:

  my @transformed = map {; ... } @input; # ; disambiguates
  my @transformed = map { ...; } @input; # ; disambiguates

=head1 Un-Escaping bytes::* Subroutines (Sjis software provides)

Sjis software removes 'bytes::' at head of subroutine name.

  ---------------------------------------
  Before           After     Works as
  ---------------------------------------
  bytes::chr       chr       Byte
  bytes::index     index     Byte
  bytes::length    length    Byte
  bytes::ord       ord       Byte
  bytes::rindex    rindex    Byte
  bytes::substr    substr    Byte
  ---------------------------------------

=head1 Ignore Pragmas and Modules

  -----------------------------------------------------------
  Before                    After
  -----------------------------------------------------------
  use strict;               use strict; no strict qw(refs);
  use 5.12.0;               use 5.12.0; no strict qw(refs);
  require utf8;             # require utf8;
  require bytes;            # require bytes;
  require charnames;        # require charnames;
  require I18N::Japanese;   # require I18N::Japanese;
  require I18N::Collate;    # require I18N::Collate;
  require I18N::JExt;       # require I18N::JExt;
  require File::DosGlob;    # require File::DosGlob;
  require Wild;             # require Wild;
  require Wildcard;         # require Wildcard;
  require Japanese;         # require Japanese;
  use utf8;                 # use utf8;
  use bytes;                # use bytes;
  use charnames;            # use charnames;
  use I18N::Japanese;       # use I18N::Japanese;
  use I18N::Collate;        # use I18N::Collate;
  use I18N::JExt;           # use I18N::JExt;
  use File::DosGlob;        # use File::DosGlob;
  use Wild;                 # use Wild;
  use Wildcard;             # use Wildcard;
  use Japanese;             # use Japanese;
  no utf8;                  # no utf8;
  no bytes;                 # no bytes;
  no charnames;             # no charnames;
  no I18N::Japanese;        # no I18N::Japanese;
  no I18N::Collate;         # no I18N::Collate;
  no I18N::JExt;            # no I18N::JExt;
  no File::DosGlob;         # no File::DosGlob;
  no Wild;                  # no Wild;
  no Wildcard;              # no Wildcard;
  no Japanese;              # no Japanese;
  -----------------------------------------------------------

  Comment out pragma to ignore utf8 environment, and Esjis.pm provides these
  functions.

=over 2

=item * Dummy utf8::upgrade

  $num_octets = utf8::upgrade($string);

  Returns the number of octets necessary to represent the string.

=item * Dummy utf8::downgrade

  $success = utf8::downgrade($string[, FAIL_OK]);

  Returns true always.

=item * Dummy utf8::encode

  utf8::encode($string);

  Returns nothing.

=item * Dummy utf8::decode

  $success = utf8::decode($string);

  Returns true always.

=item * Dummy utf8::is_utf8

  $flag = utf8::is_utf8(STRING);

  Returns false always.

=item * Dummy utf8::valid

  $flag = utf8::valid(STRING);

  Returns true always.

=item * Dummy bytes::chr

  This subroutine is same as chr.

=item * Dummy bytes::index

  This subroutine is same as index.

=item * Dummy bytes::length

  This subroutine is same as length.

=item * Dummy bytes::ord

  This subroutine is same as ord.

=item * Dummy bytes::rindex

  This subroutine is same as rindex.

=item * Dummy bytes::substr

  This subroutine is same as substr.

=back

=head1 Environment Variable

 This software uses the flock function for exclusive control. The execution of the
 program is blocked until it becomes possible to read or write the file.
 You can have it not block in the flock function by defining environment variable
 CHAR_NONBLOCK.
 
 Example:
 
   SET CHAR_NONBLOCK=1
 
 (The value '1' doesn't have the meaning)

#if MACPERL
=head1 MacJPerl on The MacOS

 The functions of MacJPerl was mimicked referring to the books and web.
 It is welcome if there is a bug report.
 
 The following software is necessary to execute Sjis software.
 1. MacPerl module
 2. Mac::Files module
 3. ToolServer
 4. MPW(Macintosh Programmer's Workshop)

#endif
=head1 BUGS, LIMITATIONS, and COMPATIBILITY

I have tested and verified this software using the best of my ability.
However, a software containing much regular expression is bound to contain
some bugs. Thus, if you happen to find a bug that's in Sjis software and
not your own program, you can try to reduce it to a minimal test case and
then report it to the following author's address. If you have an idea that
could make this a more useful tool, please let everyone share it.

=over 2

=item * (dummy item to avoid Test::Pod error)

#if MULTIBYTE_ENCODING
=item * format

Function "format" can't handle multiple-octet code same as original Perl.

#endif
=item * cloister of regular expression

The cloister (?s) and (?i) of a regular expression will not be implemented for
the time being. Cloister (?s) can be substituted with the .(dot) and \N on /s
modifier. Cloister (?i) can be substituted with \F...\E.

#if ESCAPE_SECOND_OCTET
=item * chdir

Function chdir() can always be executed with perl5.005.

There are the following limitations for DOS-like system(any of MSWin32, NetWare,
symbian, dos).

On perl5.006 or perl5.00800, if path is ended by chr(0x5C), it needs jacode.pl
library.

On perl5.008001 or later, perl5.010, perl5.012, perl5.014, perl5.016, perl5.018,
perl5.020, perl5.022, perl5.024, perl5.026, perl5.028 if path is ended by
chr(0x5C), chdir succeeds when a short path name (8dot3name) can be acquired
according to COMMAND.COM or cmd.exe or Win95Cmd.exe. However, leaf-subdirectory
of the current directory is a short path name (8dot3name).

  see also,
  Bug #81839
  chdir does not work with chr(0x5C) at end of path
  http://bugs.activestate.com/show_bug.cgi?id=81839

#endif
#if MULTIBYTE_ENCODING
=item * Sjis::substr as Lvalue

If Perl version is older than 5.14, Sjis::substr differs from CORE::substr, and
cannot be used as a lvalue. To change part of a string, you need use the optional
fourth argument which is the replacement string.

Sjis::substr($string, 13, 4, "JPerl");

#endif
#if MULTIBYTE_ANCHORING
=item * Special Variables $` and $& need /( Capture All )/

  Because $` and $& use $1.

  -------------------------------------------------------------------------------------------
  Before          After                Works as
  -------------------------------------------------------------------------------------------
  $`              Esjis::PREMATCH()    CORE::substr($&,0,CORE::length($&)-CORE::length($1))
  ${`}            Esjis::PREMATCH()    CORE::substr($&,0,CORE::length($&)-CORE::length($1))
  $PREMATCH       Esjis::PREMATCH()    CORE::substr($&,0,CORE::length($&)-CORE::length($1))
  ${^PREMATCH}    Esjis::PREMATCH()    CORE::substr($&,0,CORE::length($&)-CORE::length($1))
  $&              Esjis::MATCH()       $1
  ${&}            Esjis::MATCH()       $1
  $MATCH          Esjis::MATCH()       $1
  ${^MATCH}       Esjis::MATCH()       $1
  $'              $'                   $'
  ${'}            ${'}                 $'
  $POSTMATCH      Esjis::POSTMATCH()   $'
  ${^POSTMATCH}   Esjis::POSTMATCH()   $'
  -------------------------------------------------------------------------------------------

=item * Limitation of Regular Expression

This software has limitation from \G in multibyte anchoring. Only the following
Perl can treat the character string which exceeds 32766 octets with a regular
expression.

perl 5.6  or later --- ActivePerl on MSWin32

perl 5.10.1 or later --- other Perl

  see also,
  
  In 5.10.0, the * quantifier in patterns was sometimes treated as {0,32767}
  http://perldoc.perl.org/perl5101delta.html
  
  [perl #116379] \G can't treat over 32767 octet
  http://www.nntp.perl.org/group/perl.perl5.porters/2013/01/msg197320.html
  
  perlre - Perl regular expressions
  http://perldoc.perl.org/perlre.html
  
  perlre length limit
  http://stackoverflow.com/questions/4592467/perlre-length-limit
  
  Japanese Document
  Sjis/JA.pm

#endif
#if MULTIBYTE_ENCODING
=item * Empty Variable in Regular Expression

Unlike literal null string, an interpolated variable evaluated to the empty string
can't use the most recent pattern from a previous successful regular expression.

=item * Limitation of ?? and m??

Multibyte character needs ( ) which is before {n,m}, {n,}, {n}, *, and + in ?? or m??.
As a result, you need to rewrite a script about $1,$2,$3,... You cannot use (?: )
?, {n,m}?, {n,}?, and {n}? in ?? and m??, because delimiter of m?? is '?'.

#endif
#if ESCAPE_SECOND_OCTET
=item * Look-behind Assertion

The look-behind assertion like (?<=[A-Z]) is not prevented from matching trail
octet of the previous multiple-octet code.

#endif
=item * Modifier /a /d /l and /u of Regular Expression

The concept of this software is not to use two or more encoding methods as
literal string and literal of regexp in one Perl script. Therefore, modifier
/a, /d, /l, and /u are not supported.
\d means [0-9] universally.

=item * Named Character

A named character, such \N{GREEK SMALL LETTER EPSILON}, \N{greek:epsilon}, or
\N{epsilon} is not supported.

=item * Unicode Properties (aka Character Properties) of Regular Expression

Unicode properties (aka character properties) of regexp are not available.
Also (?[]) in regexp of Perl 5.18 is not available. There is no plans to currently
support these.

#if ESCAPE_SECOND_OCTET
=item * ${^WIN32_SLOPPY_STAT} is ignored

Even if ${^WIN32_SLOPPY_STAT} is set to a true value, file test functions Esjis::*(),
Esjis::lstat(), and Esjis::stat() on Microsoft Windows open the file for the path
which has chr(0x5c) at end.

#endif
=item * Delimiter of String and Regexp

qq//, q//, qw//, qx//, qr//, m//, s///, tr///, and y/// can't use a wide character
as the delimiter.

=item * \b{...} Boundaries in Regular Expressions

Following \b{...} available starting in v5.22 are not supported.

  \b{gcb} or \b{g}   Unicode "Grapheme Cluster Boundary"
  \b{sb}             Unicode "Sentence Boundary"
  \b{wb}             Unicode "Word Boundary"
  \B{gcb} or \B{g}   Unicode "Grapheme Cluster Boundary" doesn't match
  \B{sb}             Unicode "Sentence Boundary" doesn't match
  \B{wb}             Unicode "Word Boundary" doesn't match

=back

=head1 AUTHOR

INABA Hitoshi E<lt>ina@cpan.orgE<gt>

This project was originated by INABA Hitoshi.

=head1 LICENSE AND COPYRIGHT

This software is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.

This software is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

=head1 My Goal

P.401 See chapter 15: Unicode
of ISBN 0-596-00027-8 Programming Perl Third Edition.

Before the introduction of Unicode support in perl, The eq operator
just compared the byte-strings represented by two scalars. Beginning
with perl 5.8, eq compares two byte-strings with simultaneous
consideration of the UTF8 flag.

/* You are not expected to understand this */

  Information processing model beginning with perl 5.8
 
    +----------------------+---------------------+
    |     Text strings     |                     |
    +----------+-----------|    Binary strings   |
    |  UTF-8   |  Latin-1  |                     |
    +----------+-----------+---------------------+
    | UTF8     |            Not UTF8             |
    | Flagged  |            Flagged              |
    +--------------------------------------------+
    http://perl-users.jp/articles/advent-calendar/2010/casual/4

  Confusion of Perl string model is made from double meanings of
  "Binary string."
  Meanings of "Binary string"
  1. Non-Text string
  2. Digital octet string

  Let's draw again using those term.
 
    +----------------------+---------------------+
    |     Text strings     |                     |
    +----------+-----------|   Non-Text strings  |
    |  UTF-8   |  Latin-1  |                     |
    +----------+-----------+---------------------+
    | UTF8     |            Not UTF8             |
    | Flagged  |            Flagged              |
    +--------------------------------------------+
    |            Digital octet string            |
    +--------------------------------------------+

There are people who don't agree to change in the character string
processing model of Perl 5.8. It is impossible to get to agree it to
majority of Perl user who hardly ever use Perl.
How to solve it by returning to a original method, let's drag out
page 402 of the old dusty Programming Perl, 3rd ed. again.

  Information processing model beginning with perl3 or this software
  of UNIX/C-ism.

    +--------------------------------------------+
    |    Text string as Digital octet string     |
    |    Digital octet string as Text string     |
    +--------------------------------------------+
    |       Not UTF8 Flagged, No Mojibake        |
    +--------------------------------------------+

  In UNIX Everything is a File
  - In UNIX everything is a stream of bytes
  - In UNIX the filesystem is used as a universal name space

  Native Encoding Scripting
  - native encoding of file contents
  - native encoding of file name on filesystem
  - native encoding of command line
  - native encoding of environment variable
  - native encoding of API
  - native encoding of network packet
  - native encoding of database

Ideally, I'd like to achieve these five Goals:

=over 2

=item * Goal #1:

Old byte-oriented programs should not spontaneously break on the old
byte-oriented data they used to work on.

This goal has been achieved by that this software is additional code
for perl like utf8 pragma. Perl should work same as past Perl if added
nothing.

=item * Goal #2:

Old byte-oriented programs should magically start working on the new
character-oriented data when appropriate.

Still now, 1 octet is counted with 1 by built-in functions length,
substr, index, rindex, and pos that handle length and position of string.
In this part, there is no change. The length of 1 character of 2 octet
code is 2.

On the other hand, the regular expression in the script is added the
multibyte anchoring processing with this software, instead of you.

figure of Goal #1 and Goal #2.

                               GOAL#1  GOAL#2
                        (a)     (b)     (c)     (d)     (e)
      +--------------+-------+-------+-------+-------+-------+
      | data         |  Old  |  Old  |  New  |  Old  |  New  |
      +--------------+-------+-------+-------+-------+-------+
      | script       |  Old  |      Old      |      New      |
      +--------------+-------+---------------+---------------+
      | interpreter  |  Old  |              New              |
      +--------------+-------+-------------------------------+
      Old --- Old byte-oriented
      New --- New character-oriented

There is a combination from (a) to (e) in data, script, and interpreter
of old and new. Let's add the Encode module and this software did not
exist at time of be written this document and JPerl did exist.

                        (a)     (b)     (c)     (d)     (e)
                                      JPerl,japerl    Encode,Sjis
      +--------------+-------+-------+-------+-------+-------+
      | data         |  Old  |  Old  |  New  |  Old  |  New  |
      +--------------+-------+-------+-------+-------+-------+
      | script       |  Old  |      Old      |      New      |
      +--------------+-------+---------------+---------------+
      | interpreter  |  Old  |              New              |
      +--------------+-------+-------------------------------+
      Old --- Old byte-oriented
      New --- New character-oriented

The reason why JPerl is very excellent is that it is at the position of
(c). That is, it is not necessary to do a special description to the
script to process new character-oriented string.
(May the japerl take over JPerl!)

=item * Goal #3:

Programs should run just as fast in the new character-oriented mode
as in the old byte-oriented mode.

It is impossible. Because the following time is necessary.

(1) Time of escape script for old byte-oriented perl.

#if MULTIBYTE_ANCHORING
(2) Time of processing regular expression by escaped script while
    multibyte anchoring.

#endif
Someday, I want to ask Larry Wall about this goal in the elevator.

=item * Goal #4:

Perl should remain one language, rather than forking into a
byte-oriented Perl and a character-oriented Perl.

JPerl remains one Perl language by forking to two interpreters.
However, the Perl core team did not desire fork of the interpreter.
As a result, Perl language forked contrary to goal #4.

A character-oriented perl is not necessary to make it specially,
because a byte-oriented perl can already treat the binary data.
This software is only an application program of byte-oriented Perl,
a filter program.

And you will get support from the Perl community, when you solve the
problem by the Perl script.

Sjis software remains one language and one interpreter.

=item * Goal #5:

JPerl users will be able to maintain JPerl by Perl.

May the JPerl be with you, always.

=back

Back when Programming Perl, 3rd ed. was written, UTF8 flag was not born
and Perl is designed to make the easy jobs easy. This software provides
programming environment like at that time.

=head1 Perl's motto

   Some computer scientists (the reductionists, in particular) would
  like to deny it, but people have funny-shaped minds. Mental geography
  is not linear, and cannot be mapped onto a flat surface without
  severe distortion. But for the last score years or so, computer
  reductionists have been first bowing down at the Temple of Orthogonality,
  then rising up to preach their ideas of ascetic rectitude to any who
  would listen.
 
   Their fervent but misguided desire was simply to squash your mind to
  fit their mindset, to smush your patterns of thought into some sort of
  Hyperdimensional Flatland. It's a joyless existence, being smushed.
  --- Learning Perl on Win32 Systems

  If you think this is a big headache, you're right. No one likes
  this situation, but Perl does the best it can with the input and
  encodings it has to deal with. If only we could reset history and
  not make so many mistakes next time.
  --- Learning Perl 6th Edition

   The most important thing for most people to know about handling
  Unicode data in Perl, however, is that if you don't ever use any Uni-
  code data -- if none of your files are marked as UTF-8 and you don't
  use UTF-8 locales -- then you can happily pretend that you're back in
  Perl 5.005_03 land; the Unicode features will in no way interfere with
  your code unless you're explicitly using them. Sometimes the twin
  goals of embracing Unicode but not disturbing old-style byte-oriented
  scripts has led to compromise and confusion, but it's the Perl way to
  silently do the right thing, which is what Perl ends up doing.
  --- Advanced Perl Programming, 2nd Edition

=head1 SEE ALSO

 PERL PUROGURAMINGU
 Larry Wall, Randal L.Schwartz, Yoshiyuki Kondo
 December 1997
 ISBN 4-89052-384-7
 http://www.context.co.jp/~cond/books/old-books.html

 Programming Perl, Second Edition
 By Larry Wall, Tom Christiansen, Randal L. Schwartz
 October 1996
 Pages: 670
 ISBN 10: 1-56592-149-6 | ISBN 13: 9781565921498
 http://shop.oreilly.com/product/9781565921498.do

 Programming Perl, Third Edition
 By Larry Wall, Tom Christiansen, Jon Orwant
 Third Edition  July 2000
 Pages: 1104
 ISBN 10: 0-596-00027-8 | ISBN 13: 9780596000271
 http://shop.oreilly.com/product/9780596000271.do

 The Perl Language Reference Manual (for Perl version 5.12.1)
 by Larry Wall and others
 Paperback (6"x9"), 724 pages
 Retail Price: $39.95 (pound 29.95 in UK)
 ISBN-13: 978-1-906966-02-7
 http://www.network-theory.co.uk/perl/language/

 Perl Pocket Reference, 5th Edition
 By Johan Vromans
 Publisher: O'Reilly Media
 Released: July 2011
 Pages: 102
 http://shop.oreilly.com/product/0636920018476.do

 Programming Perl, 4th Edition
 By: Tom Christiansen, brian d foy, Larry Wall, Jon Orwant
 Publisher: O'Reilly Media
 Formats: Print, Ebook, Safari Books Online
 Released: March 2012
 Pages: 1130
 Print ISBN: 978-0-596-00492-7 | ISBN 10: 0-596-00492-3
 Ebook ISBN: 978-1-4493-9890-3 | ISBN 10: 1-4493-9890-1
 http://shop.oreilly.com/product/9780596004927.do

 Perl Cookbook
 By Tom Christiansen, Nathan Torkington
 August 1998
 Pages: 800
 ISBN 10: 1-56592-243-3 | ISBN 13: 978-1-56592-243-3
 http://shop.oreilly.com/product/9781565922433.do

 Perl Cookbook, Second Edition
 By Tom Christiansen, Nathan Torkington
 Second Edition  August 2003
 Pages: 964
 ISBN 10: 0-596-00313-7 | ISBN 13: 9780596003135
 http://shop.oreilly.com/product/9780596003135.do

 Perl in a Nutshell, Second Edition
 By Stephen Spainhour, Ellen Siever, Nathan Patwardhan
 Second Edition  June 2002
 Pages: 760
 Series: In a Nutshell
 ISBN 10: 0-596-00241-6 | ISBN 13: 9780596002411
 http://shop.oreilly.com/product/9780596002411.do

 Learning Perl on Win32 Systems
 By Randal L. Schwartz, Erik Olson, Tom Christiansen
 August 1997
 Pages: 306
 ISBN 10: 1-56592-324-3 | ISBN 13: 9781565923249
 http://shop.oreilly.com/product/9781565923249.do

 Learning Perl, Fifth Edition
 By Randal L. Schwartz, Tom Phoenix, brian d foy
 June 2008
 Pages: 352
 Print ISBN:978-0-596-52010-6 | ISBN 10: 0-596-52010-7
 Ebook ISBN:978-0-596-10316-3 | ISBN 10: 0-596-10316-6
 http://shop.oreilly.com/product/9780596520113.do

 Learning Perl, 6th Edition
 By Randal L. Schwartz, brian d foy, Tom Phoenix
 June 2011
 Pages: 390
 ISBN-10: 1449303587 | ISBN-13: 978-1449303587
 http://shop.oreilly.com/product/0636920018452.do

 Advanced Perl Programming, 2nd Edition
 By Simon Cozens
 June 2005
 Pages: 300
 ISBN-10: 0-596-00456-7 | ISBN-13: 978-0-596-00456-9
 http://shop.oreilly.com/product/9780596004569.do

 Perl RESOURCE KIT UNIX EDITION
 Futato, Irving, Jepson, Patwardhan, Siever
 ISBN 10: 1-56592-370-7
 http://shop.oreilly.com/product/9781565923706.do

 Perl Resource Kit -- Win32 Edition
 Erik Olson, Brian Jepson, David Futato, Dick Hardt
 ISBN 10:1-56592-409-6
 http://shop.oreilly.com/product/9781565924093.do

 MODAN Perl NYUMON
 By Daisuke Maki
 2009/2/10
 Pages: 344
 ISBN 10: 4798119172 | ISBN 13: 978-4798119175
 http://www.seshop.com/product/detail/10250/

 Understanding Japanese Information Processing
 By Ken Lunde
 January 1900
 Pages: 470
 ISBN 10: 1-56592-043-0 | ISBN 13: 9781565920439
 http://shop.oreilly.com/product/9781565920439.do

 CJKV Information Processing
 Chinese, Japanese, Korean & Vietnamese Computing
 By Ken Lunde
 First Edition  January 1999
 Pages: 1128
 ISBN 10: 1-56592-224-7 | ISBN 13: 9781565922242
 http://shop.oreilly.com/product/9781565922242.do

 Mastering Regular Expressions, Second Edition
 By Jeffrey E. F. Friedl
 Second Edition  July 2002
 Pages: 484
 ISBN 10: 0-596-00289-0 | ISBN 13: 9780596002893
 http://shop.oreilly.com/product/9780596002893.do

 Mastering Regular Expressions, Third Edition
 By Jeffrey E. F. Friedl
 Third Edition  August 2006
 Pages: 542
 ISBN 10: 0-596-52812-4 | ISBN 13:9780596528126
 http://shop.oreilly.com/product/9780596528126.do

 Regular Expressions Cookbook
 By Jan Goyvaerts, Steven Levithan
 May 2009
 Pages: 512
 ISBN 10:0-596-52068-9 | ISBN 13: 978-0-596-52068-7
 http://shop.oreilly.com/product/9780596520694.do

 Regular Expressions Cookbook, 2nd Edition
 By Jan Goyvaerts, Steven Levithan
 Final Release Date: August 2012
 Pages: 612
 ISBN: 978-1-4493-1943-4 | ISBN 10:1-4493-1943-2

 JIS KANJI JITEN
 By Kouji Shibano
 Pages: 1456
 ISBN 4-542-20129-5
 http://www.webstore.jsa.or.jp/lib/lib.asp?fn=/manual/mnl01_12.htm

 UNIX MAGAZINE
 1993 Aug
 Pages: 172
 T1008901080816 ZASSHI 08901-8
 http://ascii.asciimw.jp/books/books/detail/978-4-7561-5008-0.shtml

 LINUX NIHONGO KANKYO
 By YAMAGATA Hiroo, Stephen J. Turnbull, Craig Oda, Robert J. Bickel
 June, 2000
 Pages: 376
 ISBN 4-87311-016-5
 http://www.oreilly.co.jp/books/4873110165/

 MacPerl Power and Ease
 By Vicki Brown, Chris Nandor
 April 1998
 Pages: 350
 ISBN 10: 1881957322 | ISBN 13: 978-1881957324
 http://www.amazon.com/Macperl-Power-Ease-Vicki-Brown/dp/1881957322

 Windows NT Shell Scripting
 By Timothy Hill
 April 27, 1998
 Pages: 400
 ISBN 10: 1578700477 | ISBN 13: 9781578700479
 http://search.barnesandnoble.com/Windows-NT-Shell-Scripting/Timothy-Hill/e/9781578700479/

 Windows(R) Command-Line Administrators Pocket Consultant, 2nd Edition
 By William R. Stanek
 February 2009
 Pages: 594
 ISBN 10: 0-7356-2262-0 | ISBN 13: 978-0-7356-2262-3
 http://shop.oreilly.com/product/9780735622623.do

 Kaoru Maeda, Perl's history Perl 1,2,3,4
 http://www.slideshare.net/KaoruMaeda/perl-perl-1234

 nurse, What is "string"
 http://d.hatena.ne.jp/nurse/20141107#1415355181

 NISHIO Hirokazu, What's meant "string as a sequence of characters"?
 http://d.hatena.ne.jp/nishiohirokazu/20141107/1415286729

 nurse, History of Japanese EUC 22:00
 http://d.hatena.ne.jp/nurse/20090308/1236517235

 Mike Whitaker, Perl And Unicode
 http://www.slideshare.net/Penfold/perl-and-unicode

 Ricardo Signes, Perl 5.14 for Pragmatists
 http://www.slideshare.net/rjbs/perl-514-8809465

 Ricardo Signes, What's New in Perl? v5.10 - v5.16 #'
 http://www.slideshare.net/rjbs/whats-new-in-perl-v510-v516

 YAP(achimon)C::Asia Hachioji 2016 mid in Shinagawa
 Kenichi Ishigaki (@charsbar) July 3, 2016 YAP(achimon)C::Asia Hachioji 2016mid
 https://www.slideshare.net/charsbar/cpan-63708689

 CPAN Directory INABA Hitoshi
 http://search.cpan.org/~ina/

 BackPAN
 http://backpan.perl.org/authors/id/I/IN/INA/

 Recent Perl packages by "INABA Hitoshi"
 http://code.activestate.com/ppm/author:INABA-Hitoshi/

=head1 ACKNOWLEDGEMENTS

This software was made referring to software and the document that the
following hackers or persons had made. 
I am thankful to all persons.

 Rick Yamashita, Shift_JIS
 ttp://furukawablog.spaces.live.com/Blog/cns!1pmWgsL289nm7Shn7cS0jHzA!2225.entry (dead link)
 ttp://shino.tumblr.com/post/116166805/1981-us-jis
 (add 'h' at head)
 http://www.wdic.org/w/WDIC/%E3%82%B7%E3%83%95%E3%83%88JIS

 Larry Wall, Perl
 http://www.perl.org/

 Kazumasa Utashiro, jcode.pl
 http://search.cpan.org/~utashiro/
 ftp://ftp.iij.ad.jp/pub/IIJ/dist/utashiro/perl/
 http://log.utashiro.com/pub/2006/07/jkondo_a580.html

 Jeffrey E. F. Friedl, Mastering Regular Expressions
 http://regex.info/

 SADAHIRO Tomoyuki, The right way of using Shift_JIS
 http://homepage1.nifty.com/nomenclator/perl/shiftjis.htm
 http://search.cpan.org/~sadahiro/

 Yukihiro "Matz" Matsumoto, YAPC::Asia2006 Ruby on Perl(s)
 http://www.rubyist.net/~matz/slides/yapc2006/

 jscripter, For jperl users
 http://homepage1.nifty.com/kazuf/jperl.html

 Bruce., Unicode in Perl
 http://www.rakunet.org/tsnet/TSabc/18/546.html

 Hiroaki Izumi, Perl5.8/Perl5.10 is not useful on the Windows.
 http://dl.dropbox.com/u/23756062/perlwin.html
 https://sites.google.com/site/hiroa63iz/perlwin

 TSUKAMOTO Makio, Perl memo/file path of Windows
 http://digit.que.ne.jp/work/wiki.cgi?Perl%E3%83%A1%E3%83%A2%2FWindows%E3%81%A7%E3%81%AE%E3%83%95%E3%82%A1%E3%82%A4%E3%83%AB%E3%83%91%E3%82%B9

 chaichanPaPa, Matching Shift_JIS file name
 http://d.hatena.ne.jp/chaichanPaPa/20080802/1217660826

 SUZUKI Norio, Jperl
 http://homepage2.nifty.com/kipp/perl/jperl/

 WATANABE Hirofumi, Jperl
 http://www.cpan.org/src/5.0/jperl/
 http://search.cpan.org/~watanabe/
 ftp://ftp.oreilly.co.jp/pcjp98/watanabe/jperlconf.ppt

 Chuck Houpt, Michiko Nozu, MacJPerl
 http://habilis.net/macjperl/index.j.html

 Kenichi Ishigaki, Pod-PerldocJp, Welcome to modern Perl world
 http://search.cpan.org/dist/Pod-PerldocJp/
 http://gihyo.jp/dev/serial/01/modern-perl/0031
 http://gihyo.jp/dev/serial/01/modern-perl/0032
 http://gihyo.jp/dev/serial/01/modern-perl/0033

 Fuji, Goro (gfx), Perl Hackers Hub No.16
 http://gihyo.jp/dev/serial/01/perl-hackers-hub/001602

 Dan Kogai, Encode module
 http://search.cpan.org/dist/Encode/
 http://www.archive.org/details/YAPCAsia2006TokyoPerl58andUnicodeMythsFactsandChanges (video)
 http://yapc.g.hatena.ne.jp/jkondo/ (audio)

 Takahashi Masatuyo, JPerl Wiki
 http://ja.jperl.wikia.com/wiki/JPerl_Wiki

 Juerd, Perl Unicode Advice
 http://juerd.nl/site.plp/perluniadvice

 daily dayflower, 2008-06-25 perluniadvice
 http://d.hatena.ne.jp/dayflower/20080625/1214374293

 Unicode issues in Perl
 http://www.i-programmer.info/programming/other-languages/1973-unicode-issues-in-perl.html

 Jesse Vincent, Compatibility is a virtue
 http://www.nntp.perl.org/group/perl.perl5.porters/2010/05/msg159825.html

 Tokyo-pm archive
 http://mail.pm.org/pipermail/tokyo-pm/
 http://mail.pm.org/pipermail/tokyo-pm/1999-September/001844.html
 http://mail.pm.org/pipermail/tokyo-pm/1999-September/001854.html

 Error: Runtime exception on jperl 5.005_03
 http://www.rakunet.org/tsnet/TSperl/12/374.html
 http://www.rakunet.org/tsnet/TSperl/12/375.html
 http://www.rakunet.org/tsnet/TSperl/12/376.html
 http://www.rakunet.org/tsnet/TSperl/12/377.html
 http://www.rakunet.org/tsnet/TSperl/12/378.html
 http://www.rakunet.org/tsnet/TSperl/12/379.html
 http://www.rakunet.org/tsnet/TSperl/12/380.html
 http://www.rakunet.org/tsnet/TSperl/12/382.html

 ruby-list
 http://blade.nagaokaut.ac.jp/ruby/ruby-list/index.shtml
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/2440
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/2446
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/2569
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/9427
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/9431
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/10500
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/10501
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/10502
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/12385
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/12392
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/12393
 http://blade.nagaokaut.ac.jp/cgi-bin/scat.rb/ruby/ruby-list/19156

 Object-oriented with Perl
 http://www.freeml.com/perl-oo/486
 http://www.freeml.com/perl-oo/487
 http://www.freeml.com/perl-oo/490
 http://www.freeml.com/perl-oo/491
 http://www.freeml.com/perl-oo/492
 http://www.freeml.com/perl-oo/494
 http://www.freeml.com/perl-oo/514

=cut

END_OF_CLASS_SCRIPT

    my @eclass = split(/\n/,<<'END_OF_ECLASS_SCRIPT');
package Esjis;
use strict;
BEGIN { $INC{'warnings.pm'} = '' if $] < 5.006 } use warnings;
######################################################################
#
# Esjis - Run-time routines for Sjis.pm
#
# http://search.cpan.org/dist/Char-Sjis/
#
# $copyright$
######################################################################

use 5.00503;    # Galapagos Consensus 1998 for primetools
# use 5.008001; # Lancaster Consensus 2013 for toolchains

# 12.3. Delaying use Until Runtime
# in Chapter 12. Packages, Libraries, and Modules
# of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
# (and so on)

# Version numbers should be boring
# http://www.dagolden.com/index.php/369/version-numbers-should-be-boring/
# For the impatient, the disinterested or those who just want to follow
# a recipe, my advice for all modules is this:
# our $VERSION = "0.001"; # or "0.001_001" for a dev release
# $VERSION = eval $VERSION; # No!! because '1.10' makes '1.1'

use vars qw($VERSION);
$VERSION = q$Revision$;
$VERSION = $VERSION;

BEGIN {
    if ($^X =~ / jperl /oxmsi) {
        die __FILE__, ": needs perl(not jperl) 5.00503 or later. (\$^X==$^X)\n";
    }
    if (CORE::ord('A') == 193) {
        die __FILE__, ": is not US-ASCII script (may be EBCDIC or EBCDIK script).\n";
    }
    if (CORE::ord('A') != 0x41) {
        die __FILE__, ": is not US-ASCII script (must be US-ASCII script).\n";
    }
}

BEGIN {

    # instead of utf8.pm
    CORE::eval q{
        no warnings qw(redefine);
        *utf8::upgrade   = sub { CORE::length $_[0] };
        *utf8::downgrade = sub { 1 };
        *utf8::encode    = sub {   };
        *utf8::decode    = sub { 1 };
        *utf8::is_utf8   = sub {   };
        *utf8::valid     = sub { 1 };
    };
    if ($@) {
        *utf8::upgrade   = sub { CORE::length $_[0] };
        *utf8::downgrade = sub { 1 };
        *utf8::encode    = sub {   };
        *utf8::decode    = sub { 1 };
        *utf8::is_utf8   = sub {   };
        *utf8::valid     = sub { 1 };
    }
}

# instead of Symbol.pm
BEGIN {
    sub gensym () {
        if ($] < 5.006) {
            return \do { local *_ };
        }
        else {
            return undef;
        }
    }

    sub qualify ($$) {
        my($name) = @_;

        if (ref $name) {
            return $name;
        }
        elsif (Esjis::index($name,'::') >= 0) {
            return $name;
        }
        elsif (Esjis::index($name,"'") >= 0) {
            return $name;
        }

        # special character, "^xyz"
        elsif ($name =~ /\A \^ [ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_]+ \z/x) {

            # RGS 2001-11-05 : translate leading ^X to control-char
            $name =~ s{\A \^ ([ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_]) }{'qq(\c'.$1.')'}xee;
            return 'main::' . $name;
        }

        # Global names
        elsif ($name =~ /\A (?: ARGV | ARGVOUT | ENV | INC | SIG | STDERR | STDIN | STDOUT ) \z/x) {
            return 'main::' . $name;
        }

        # or other
        elsif ($name =~ /\A [^ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz] \z/x) {
            return 'main::' . $name;
        }

        elsif (defined $_[1]) {
            return $_[1] . '::' . $name;
        }
        else {
            return (caller)[0] . '::' . $name;
        }
    }

    sub qualify_to_ref ($;$) {
        if (defined $_[1]) {
            no strict qw(refs);
            return \*{ qualify $_[0], $_[1] };
        }
        else {
            no strict qw(refs);
            return \*{ qualify $_[0], (caller)[0] };
        }
    }
}

# P.714 29.2.39. flock
# in Chapter 29: Functions
# of ISBN 0-596-00027-8 Programming Perl Third Edition.

# P.863 flock
# in Chapter 27: Functions
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

sub LOCK_SH() {1}
sub LOCK_EX() {2}
sub LOCK_UN() {8}
sub LOCK_NB() {4}

# instead of Carp.pm
sub carp;
sub croak;
sub cluck;
sub confess;

# 6.18. Matching Multiple-Byte Characters
# in Chapter 6. Pattern Matching
# of ISBN 978-1-56592-243-3 Perl Perl Cookbook.
# (and so on)

# regexp of character
my $your_char = q{[\x81-\x9F\xE0-\xFC][\x00-\xFF]|[\x00-\xFF]};
use vars qw($qq_char); $qq_char = qr/\\c[\x40-\x5F]|\\?(?:$your_char)/oxms;
use vars qw($q_char);  $q_char  = qr/$your_char/oxms;

#
# ShiftJIS character range per length
#
my %range_tr = ();

#
# ShiftJIS case conversion
#
my %lc = ();
@lc{qw(A B C D E F G H I J K L M N O P Q R S T U V W X Y Z)} =
    qw(a b c d e f g h i j k l m n o p q r s t u v w x y z);
my %uc = ();
@uc{qw(a b c d e f g h i j k l m n o p q r s t u v w x y z)} =
    qw(A B C D E F G H I J K L M N O P Q R S T U V W X Y Z);
my %fc = ();
@fc{qw(A B C D E F G H I J K L M N O P Q R S T U V W X Y Z)} =
    qw(a b c d e f g h i j k l m n o p q r s t u v w x y z);

if (0) {
}

#if DIST_BIG5HKSCS
elsif (__PACKAGE__ =~ / \b Ebig5hkscs \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x40..0x7E],
               [0x81..0xFE],[0xA1..0xFE],
             ],
    );
}

#endif
#if DIST_BIG5PLUS
elsif (__PACKAGE__ =~ / \b Ebig5plus \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x40..0x7E],
               [0x81..0xFE],[0x80..0xFE],
             ],
    );
}

#endif
#if DIST_GB18030
elsif (__PACKAGE__ =~ / \b Egb18030 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x40..0x7E],
               [0x81..0xFE],[0x80..0xFE],
             ],
        4 => [ [0x81..0xFE],[0x30..0x39],[0x81..0xFE],[0x30..0x39],
             ],
    );
}

#endif
#if DIST_GBK
elsif (__PACKAGE__ =~ / \b Egbk \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x40..0x7E],
               [0x81..0xFE],[0x80..0xFE],
             ],
    );
}

#endif
#if DIST_HP15
elsif (__PACKAGE__ =~ / \b Ehp15 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x7F],
               [0xA1..0xDF],
               [0xFF..0xFF],
             ],
        2 => [ [0x80..0xA0],[0x21..0x7E],
               [0x80..0xA0],[0x80..0xFF],
               [0xE0..0xFE],[0x21..0x7E],
               [0xE0..0xFE],[0x80..0xFF],
             ],
    );
}

#endif
#if DIST_INFORMIXV6ALS
elsif (__PACKAGE__ =~ / \b Einformixv6als \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xA0..0xDF],
               [0xFE..0xFF],
             ],
        2 => [ [0x81..0x9F],[0x40..0x7E],
               [0x81..0x9F],[0x80..0xFC],
               [0xE0..0xFC],[0x40..0x7E],
               [0xE0..0xFC],[0x80..0xFC],
             ],
        3 => [ [0xFD..0xFD],[0xA1..0xFE],[0xA1..0xFE],
             ],
    );
}

#endif
#if DIST_SJIS
elsif (__PACKAGE__ =~ / \b E s j i s \z/oxms) { # escape from build system
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xA0..0xDF],
               [0xFD..0xFF],
             ],
        2 => [ [0x81..0x9F],[0x40..0x7E],
               [0x81..0x9F],[0x80..0xFC],
               [0xE0..0xFC],[0x40..0x7E],
               [0xE0..0xFC],[0x80..0xFC],
             ],
    );
}

#endif
#if DIST_UHC
elsif (__PACKAGE__ =~ / \b Euhc \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x41..0x5A],
               [0x81..0xFE],[0x61..0x7A],
               [0x81..0xFE],[0x81..0xFE],
             ],
    );
}

#endif
#if DIST_KPS9566
elsif (__PACKAGE__ =~ / \b Ekps9566 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x80],
               [0xFF..0xFF],
             ],
        2 => [ [0x81..0xFE],[0x41..0x5A],
               [0x81..0xFE],[0x61..0x7A],
               [0x81..0xFE],[0x81..0xFE],
             ],
    );
}

#endif
#if DIST_USASCII
elsif (__PACKAGE__ =~ / \b Eusascii \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );
}

#endif
#if DIST_JIS8
elsif (__PACKAGE__ =~ / \b Ejis8 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );
}

#endif
#if DIST_LATIN1
elsif (__PACKAGE__ =~ / \b Elatin1 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER ETH (Icelandic)
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER THORN (Icelandic)
    );

    %uc = (%uc,
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER ETH (Icelandic)
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER THORN (Icelandic)
    );

    %fc = (%fc,
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER ETH               --> LATIN SMALL LETTER ETH
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE      --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER THORN             --> LATIN SMALL LETTER THORN
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN2
elsif (__PACKAGE__ =~ / \b Elatin2 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xB1",     # LATIN LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN LETTER L WITH STROKE
        "\xA5" => "\xB5",     # LATIN LETTER L WITH CARON
        "\xA6" => "\xB6",     # LATIN LETTER S WITH ACUTE
        "\xA9" => "\xB9",     # LATIN LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN LETTER S WITH CEDILLA
        "\xAB" => "\xBB",     # LATIN LETTER T WITH CARON
        "\xAC" => "\xBC",     # LATIN LETTER Z WITH ACUTE
        "\xAE" => "\xBE",     # LATIN LETTER Z WITH CARON
        "\xAF" => "\xBF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xC0" => "\xE0",     # LATIN LETTER R WITH ACUTE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER L WITH ACUTE
        "\xC6" => "\xE6",     # LATIN LETTER C WITH ACUTE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER E WITH CARON
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER D WITH CARON
        "\xD0" => "\xF0",     # LATIN LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN LETTER N WITH CARON
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH DOUBLE ACUTE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER R WITH CARON
        "\xD9" => "\xF9",     # LATIN LETTER U WITH RING ABOVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH DOUBLE ACUTE
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER T WITH CEDILLA
    );

    %uc = (%uc,
        "\xB1" => "\xA1",     # LATIN LETTER A WITH OGONEK
        "\xB3" => "\xA3",     # LATIN LETTER L WITH STROKE
        "\xB5" => "\xA5",     # LATIN LETTER L WITH CARON
        "\xB6" => "\xA6",     # LATIN LETTER S WITH ACUTE
        "\xB9" => "\xA9",     # LATIN LETTER S WITH CARON
        "\xBA" => "\xAA",     # LATIN LETTER S WITH CEDILLA
        "\xBB" => "\xAB",     # LATIN LETTER T WITH CARON
        "\xBC" => "\xAC",     # LATIN LETTER Z WITH ACUTE
        "\xBE" => "\xAE",     # LATIN LETTER Z WITH CARON
        "\xBF" => "\xAF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xE0" => "\xC0",     # LATIN LETTER R WITH ACUTE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH BREVE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER L WITH ACUTE
        "\xE6" => "\xC6",     # LATIN LETTER C WITH ACUTE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER C WITH CARON
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH OGONEK
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER E WITH CARON
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER D WITH CARON
        "\xF0" => "\xD0",     # LATIN LETTER D WITH STROKE
        "\xF1" => "\xD1",     # LATIN LETTER N WITH ACUTE
        "\xF2" => "\xD2",     # LATIN LETTER N WITH CARON
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH DOUBLE ACUTE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER R WITH CARON
        "\xF9" => "\xD9",     # LATIN LETTER U WITH RING ABOVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH DOUBLE ACUTE
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER T WITH CEDILLA
    );

    %fc = (%fc,
        "\xA1" => "\xB1",     # LATIN CAPITAL LETTER A WITH OGONEK       --> LATIN SMALL LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN CAPITAL LETTER L WITH STROKE       --> LATIN SMALL LETTER L WITH STROKE
        "\xA5" => "\xB5",     # LATIN CAPITAL LETTER L WITH CARON        --> LATIN SMALL LETTER L WITH CARON
        "\xA6" => "\xB6",     # LATIN CAPITAL LETTER S WITH ACUTE        --> LATIN SMALL LETTER S WITH ACUTE
        "\xA9" => "\xB9",     # LATIN CAPITAL LETTER S WITH CARON        --> LATIN SMALL LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER S WITH CEDILLA      --> LATIN SMALL LETTER S WITH CEDILLA
        "\xAB" => "\xBB",     # LATIN CAPITAL LETTER T WITH CARON        --> LATIN SMALL LETTER T WITH CARON
        "\xAC" => "\xBC",     # LATIN CAPITAL LETTER Z WITH ACUTE        --> LATIN SMALL LETTER Z WITH ACUTE
        "\xAE" => "\xBE",     # LATIN CAPITAL LETTER Z WITH CARON        --> LATIN SMALL LETTER Z WITH CARON
        "\xAF" => "\xBF",     # LATIN CAPITAL LETTER Z WITH DOT ABOVE    --> LATIN SMALL LETTER Z WITH DOT ABOVE
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER R WITH ACUTE        --> LATIN SMALL LETTER R WITH ACUTE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE        --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX   --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH BREVE        --> LATIN SMALL LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS    --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER L WITH ACUTE        --> LATIN SMALL LETTER L WITH ACUTE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER C WITH ACUTE        --> LATIN SMALL LETTER C WITH ACUTE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA      --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER C WITH CARON        --> LATIN SMALL LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE        --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH OGONEK       --> LATIN SMALL LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS    --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER E WITH CARON        --> LATIN SMALL LETTER E WITH CARON
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE        --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX   --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER D WITH CARON        --> LATIN SMALL LETTER D WITH CARON
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER D WITH STROKE       --> LATIN SMALL LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH ACUTE        --> LATIN SMALL LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER N WITH CARON        --> LATIN SMALL LETTER N WITH CARON
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE        --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX   --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH DOUBLE ACUTE --> LATIN SMALL LETTER O WITH DOUBLE ACUTE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS    --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER R WITH CARON        --> LATIN SMALL LETTER R WITH CARON
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH RING ABOVE   --> LATIN SMALL LETTER U WITH RING ABOVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE        --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH DOUBLE ACUTE --> LATIN SMALL LETTER U WITH DOUBLE ACUTE
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS    --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE        --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER T WITH CEDILLA      --> LATIN SMALL LETTER T WITH CEDILLA
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S               --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN3
elsif (__PACKAGE__ =~ / \b Elatin3 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xB1",     # LATIN LETTER H WITH STROKE
        "\xA6" => "\xB6",     # LATIN LETTER H WITH CIRCUMFLEX
        "\xAA" => "\xBA",     # LATIN LETTER S WITH CEDILLA
        "\xAB" => "\xBB",     # LATIN LETTER G WITH BREVE
        "\xAC" => "\xBC",     # LATIN LETTER J WITH CIRCUMFLEX
        "\xAF" => "\xBF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER C WITH DOT ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER C WITH CIRCUMFLEX
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER G WITH DOT ABOVE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER G WITH CIRCUMFLEX
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER U WITH BREVE
        "\xDE" => "\xFE",     # LATIN LETTER S WITH CIRCUMFLEX
    );

    %uc = (%uc,
        "\xB1" => "\xA1",     # LATIN LETTER H WITH STROKE
        "\xB6" => "\xA6",     # LATIN LETTER H WITH CIRCUMFLEX
        "\xBA" => "\xAA",     # LATIN LETTER S WITH CEDILLA
        "\xBB" => "\xAB",     # LATIN LETTER G WITH BREVE
        "\xBC" => "\xAC",     # LATIN LETTER J WITH CIRCUMFLEX
        "\xBF" => "\xAF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER C WITH DOT ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER C WITH CIRCUMFLEX
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER G WITH DOT ABOVE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER G WITH CIRCUMFLEX
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER U WITH BREVE
        "\xFE" => "\xDE",     # LATIN LETTER S WITH CIRCUMFLEX
    );

    %fc = (%fc,
        "\xA1" => "\xB1",     # LATIN CAPITAL LETTER H WITH STROKE --> LATIN SMALL LETTER H WITH STROKE
        "\xA6" => "\xB6",     # LATIN CAPITAL LETTER H WITH CIRCUMFLEX --> LATIN SMALL LETTER H WITH CIRCUMFLEX

        # CaseFolding-6.1.0.txt
        # Date: 2011-07-25, 21:21:56 GMT [MD]
        #
        # T: special case for uppercase I and dotted uppercase I
        #    - For non-Turkic languages, this mapping is normally not used.
        #    - For Turkic languages (tr, az), this mapping can be used instead of the normal mapping for these characters.
        #      Note that the Turkic mappings do not maintain canonical equivalence without additional processing.
        #      See the discussions of case mapping in the Unicode Standard for more information.

        #-------------------------------------------------------------------------------
        "\xA9" => "\x69",     # LATIN CAPITAL LETTER I WITH DOT ABOVE
                              # --> LATIN SMALL LETTER I (without COMBINING DOT ABOVE)
        #-------------------------------------------------------------------------------

        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER S WITH CEDILLA    --> LATIN SMALL LETTER S WITH CEDILLA
        "\xAB" => "\xBB",     # LATIN CAPITAL LETTER G WITH BREVE      --> LATIN SMALL LETTER G WITH BREVE
        "\xAC" => "\xBC",     # LATIN CAPITAL LETTER J WITH CIRCUMFLEX --> LATIN SMALL LETTER J WITH CIRCUMFLEX
        "\xAF" => "\xBF",     # LATIN CAPITAL LETTER Z WITH DOT ABOVE  --> LATIN SMALL LETTER Z WITH DOT ABOVE
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER C WITH DOT ABOVE  --> LATIN SMALL LETTER C WITH DOT ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER C WITH CIRCUMFLEX --> LATIN SMALL LETTER C WITH CIRCUMFLEX
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER G WITH DOT ABOVE  --> LATIN SMALL LETTER G WITH DOT ABOVE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER G WITH CIRCUMFLEX --> LATIN SMALL LETTER G WITH CIRCUMFLEX
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER U WITH BREVE      --> LATIN SMALL LETTER U WITH BREVE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER S WITH CIRCUMFLEX --> LATIN SMALL LETTER S WITH CIRCUMFLEX
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN4
elsif (__PACKAGE__ =~ / \b Elatin4 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xB1",     # LATIN LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN LETTER R WITH CEDILLA
        "\xA5" => "\xB5",     # LATIN LETTER I WITH TILDE
        "\xA6" => "\xB6",     # LATIN LETTER L WITH CEDILLA
        "\xA9" => "\xB9",     # LATIN LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN LETTER E WITH MACRON
        "\xAB" => "\xBB",     # LATIN LETTER G WITH CEDILLA
        "\xAC" => "\xBC",     # LATIN LETTER T WITH STROKE
        "\xAE" => "\xBE",     # LATIN LETTER Z WITH CARON
        "\xBD" => "\xBF",     # LATIN LETTER ENG
        "\xC0" => "\xE0",     # LATIN LETTER A WITH MACRON
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER I WITH OGONEK
        "\xC8" => "\xE8",     # LATIN LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER E WITH DOT ABOVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH MACRON
        "\xD0" => "\xF0",     # LATIN LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN LETTER N WITH CEDILLA
        "\xD2" => "\xF2",     # LATIN LETTER O WITH MACRON
        "\xD3" => "\xF3",     # LATIN LETTER K WITH CEDILLA
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH OGONEK
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER U WITH TILDE
        "\xDE" => "\xFE",     # LATIN LETTER U WITH MACRON
    );

    %uc = (%uc,
        "\xB1" => "\xA1",     # LATIN LETTER A WITH OGONEK
        "\xB3" => "\xA3",     # LATIN LETTER R WITH CEDILLA
        "\xB5" => "\xA5",     # LATIN LETTER I WITH TILDE
        "\xB6" => "\xA6",     # LATIN LETTER L WITH CEDILLA
        "\xB9" => "\xA9",     # LATIN LETTER S WITH CARON
        "\xBA" => "\xAA",     # LATIN LETTER E WITH MACRON
        "\xBB" => "\xAB",     # LATIN LETTER G WITH CEDILLA
        "\xBC" => "\xAC",     # LATIN LETTER T WITH STROKE
        "\xBE" => "\xAE",     # LATIN LETTER Z WITH CARON
        "\xBF" => "\xBD",     # LATIN LETTER ENG
        "\xE0" => "\xC0",     # LATIN LETTER A WITH MACRON
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER I WITH OGONEK
        "\xE8" => "\xC8",     # LATIN LETTER C WITH CARON
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH OGONEK
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER E WITH DOT ABOVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH MACRON
        "\xF0" => "\xD0",     # LATIN LETTER D WITH STROKE
        "\xF1" => "\xD1",     # LATIN LETTER N WITH CEDILLA
        "\xF2" => "\xD2",     # LATIN LETTER O WITH MACRON
        "\xF3" => "\xD3",     # LATIN LETTER K WITH CEDILLA
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH OGONEK
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER U WITH TILDE
        "\xFE" => "\xDE",     # LATIN LETTER U WITH MACRON
    );

    %fc = (%fc,
        "\xA1" => "\xB1",     # LATIN CAPITAL LETTER A WITH OGONEK     --> LATIN SMALL LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN CAPITAL LETTER R WITH CEDILLA    --> LATIN SMALL LETTER R WITH CEDILLA
        "\xA5" => "\xB5",     # LATIN CAPITAL LETTER I WITH TILDE      --> LATIN SMALL LETTER I WITH TILDE
        "\xA6" => "\xB6",     # LATIN CAPITAL LETTER L WITH CEDILLA    --> LATIN SMALL LETTER L WITH CEDILLA
        "\xA9" => "\xB9",     # LATIN CAPITAL LETTER S WITH CARON      --> LATIN SMALL LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER E WITH MACRON     --> LATIN SMALL LETTER E WITH MACRON
        "\xAB" => "\xBB",     # LATIN CAPITAL LETTER G WITH CEDILLA    --> LATIN SMALL LETTER G WITH CEDILLA
        "\xAC" => "\xBC",     # LATIN CAPITAL LETTER T WITH STROKE     --> LATIN SMALL LETTER T WITH STROKE
        "\xAE" => "\xBE",     # LATIN CAPITAL LETTER Z WITH CARON      --> LATIN SMALL LETTER Z WITH CARON
        "\xBD" => "\xBF",     # LATIN CAPITAL LETTER ENG               --> LATIN SMALL LETTER ENG
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH MACRON     --> LATIN SMALL LETTER A WITH MACRON
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER I WITH OGONEK     --> LATIN SMALL LETTER I WITH OGONEK
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER C WITH CARON      --> LATIN SMALL LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH OGONEK     --> LATIN SMALL LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER E WITH DOT ABOVE  --> LATIN SMALL LETTER E WITH DOT ABOVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH MACRON     --> LATIN SMALL LETTER I WITH MACRON
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER D WITH STROKE     --> LATIN SMALL LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH CEDILLA    --> LATIN SMALL LETTER N WITH CEDILLA
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH MACRON     --> LATIN SMALL LETTER O WITH MACRON
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER K WITH CEDILLA    --> LATIN SMALL LETTER K WITH CEDILLA
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH OGONEK     --> LATIN SMALL LETTER U WITH OGONEK
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER U WITH TILDE      --> LATIN SMALL LETTER U WITH TILDE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER U WITH MACRON     --> LATIN SMALL LETTER U WITH MACRON
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_CYRILLIC
elsif (__PACKAGE__ =~ / \b Ecyrillic \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xF1",     # CYRILLIC LETTER IO
        "\xA2" => "\xF2",     # CYRILLIC LETTER DJE
        "\xA3" => "\xF3",     # CYRILLIC LETTER GJE
        "\xA4" => "\xF4",     # CYRILLIC LETTER UKRAINIAN IE
        "\xA5" => "\xF5",     # CYRILLIC LETTER DZE
        "\xA6" => "\xF6",     # CYRILLIC LETTER BYELORUSSIAN-UKRAINIAN I
        "\xA7" => "\xF7",     # CYRILLIC LETTER YI
        "\xA8" => "\xF8",     # CYRILLIC LETTER JE
        "\xA9" => "\xF9",     # CYRILLIC LETTER LJE
        "\xAA" => "\xFA",     # CYRILLIC LETTER NJE
        "\xAB" => "\xFB",     # CYRILLIC LETTER TSHE
        "\xAC" => "\xFC",     # CYRILLIC LETTER KJE
        "\xAE" => "\xFE",     # CYRILLIC LETTER SHORT U
        "\xAF" => "\xFF",     # CYRILLIC LETTER DZHE
        "\xB0" => "\xD0",     # CYRILLIC LETTER A
        "\xB1" => "\xD1",     # CYRILLIC LETTER BE
        "\xB2" => "\xD2",     # CYRILLIC LETTER VE
        "\xB3" => "\xD3",     # CYRILLIC LETTER GHE
        "\xB4" => "\xD4",     # CYRILLIC LETTER DE
        "\xB5" => "\xD5",     # CYRILLIC LETTER IE
        "\xB6" => "\xD6",     # CYRILLIC LETTER ZHE
        "\xB7" => "\xD7",     # CYRILLIC LETTER ZE
        "\xB8" => "\xD8",     # CYRILLIC LETTER I
        "\xB9" => "\xD9",     # CYRILLIC LETTER SHORT I
        "\xBA" => "\xDA",     # CYRILLIC LETTER KA
        "\xBB" => "\xDB",     # CYRILLIC LETTER EL
        "\xBC" => "\xDC",     # CYRILLIC LETTER EM
        "\xBD" => "\xDD",     # CYRILLIC LETTER EN
        "\xBE" => "\xDE",     # CYRILLIC LETTER O
        "\xBF" => "\xDF",     # CYRILLIC LETTER PE
        "\xC0" => "\xE0",     # CYRILLIC LETTER ER
        "\xC1" => "\xE1",     # CYRILLIC LETTER ES
        "\xC2" => "\xE2",     # CYRILLIC LETTER TE
        "\xC3" => "\xE3",     # CYRILLIC LETTER U
        "\xC4" => "\xE4",     # CYRILLIC LETTER EF
        "\xC5" => "\xE5",     # CYRILLIC LETTER HA
        "\xC6" => "\xE6",     # CYRILLIC LETTER TSE
        "\xC7" => "\xE7",     # CYRILLIC LETTER CHE
        "\xC8" => "\xE8",     # CYRILLIC LETTER SHA
        "\xC9" => "\xE9",     # CYRILLIC LETTER SHCHA
        "\xCA" => "\xEA",     # CYRILLIC LETTER HARD SIGN
        "\xCB" => "\xEB",     # CYRILLIC LETTER YERU
        "\xCC" => "\xEC",     # CYRILLIC LETTER SOFT SIGN
        "\xCD" => "\xED",     # CYRILLIC LETTER E
        "\xCE" => "\xEE",     # CYRILLIC LETTER YU
        "\xCF" => "\xEF",     # CYRILLIC LETTER YA
    );

    %uc = (%uc,
        "\xD0" => "\xB0",     # CYRILLIC LETTER A
        "\xD1" => "\xB1",     # CYRILLIC LETTER BE
        "\xD2" => "\xB2",     # CYRILLIC LETTER VE
        "\xD3" => "\xB3",     # CYRILLIC LETTER GHE
        "\xD4" => "\xB4",     # CYRILLIC LETTER DE
        "\xD5" => "\xB5",     # CYRILLIC LETTER IE
        "\xD6" => "\xB6",     # CYRILLIC LETTER ZHE
        "\xD7" => "\xB7",     # CYRILLIC LETTER ZE
        "\xD8" => "\xB8",     # CYRILLIC LETTER I
        "\xD9" => "\xB9",     # CYRILLIC LETTER SHORT I
        "\xDA" => "\xBA",     # CYRILLIC LETTER KA
        "\xDB" => "\xBB",     # CYRILLIC LETTER EL
        "\xDC" => "\xBC",     # CYRILLIC LETTER EM
        "\xDD" => "\xBD",     # CYRILLIC LETTER EN
        "\xDE" => "\xBE",     # CYRILLIC LETTER O
        "\xDF" => "\xBF",     # CYRILLIC LETTER PE
        "\xE0" => "\xC0",     # CYRILLIC LETTER ER
        "\xE1" => "\xC1",     # CYRILLIC LETTER ES
        "\xE2" => "\xC2",     # CYRILLIC LETTER TE
        "\xE3" => "\xC3",     # CYRILLIC LETTER U
        "\xE4" => "\xC4",     # CYRILLIC LETTER EF
        "\xE5" => "\xC5",     # CYRILLIC LETTER HA
        "\xE6" => "\xC6",     # CYRILLIC LETTER TSE
        "\xE7" => "\xC7",     # CYRILLIC LETTER CHE
        "\xE8" => "\xC8",     # CYRILLIC LETTER SHA
        "\xE9" => "\xC9",     # CYRILLIC LETTER SHCHA
        "\xEA" => "\xCA",     # CYRILLIC LETTER HARD SIGN
        "\xEB" => "\xCB",     # CYRILLIC LETTER YERU
        "\xEC" => "\xCC",     # CYRILLIC LETTER SOFT SIGN
        "\xED" => "\xCD",     # CYRILLIC LETTER E
        "\xEE" => "\xCE",     # CYRILLIC LETTER YU
        "\xEF" => "\xCF",     # CYRILLIC LETTER YA
        "\xF1" => "\xA1",     # CYRILLIC LETTER IO
        "\xF2" => "\xA2",     # CYRILLIC LETTER DJE
        "\xF3" => "\xA3",     # CYRILLIC LETTER GJE
        "\xF4" => "\xA4",     # CYRILLIC LETTER UKRAINIAN IE
        "\xF5" => "\xA5",     # CYRILLIC LETTER DZE
        "\xF6" => "\xA6",     # CYRILLIC LETTER BYELORUSSIAN-UKRAINIAN I
        "\xF7" => "\xA7",     # CYRILLIC LETTER YI
        "\xF8" => "\xA8",     # CYRILLIC LETTER JE
        "\xF9" => "\xA9",     # CYRILLIC LETTER LJE
        "\xFA" => "\xAA",     # CYRILLIC LETTER NJE
        "\xFB" => "\xAB",     # CYRILLIC LETTER TSHE
        "\xFC" => "\xAC",     # CYRILLIC LETTER KJE
        "\xFE" => "\xAE",     # CYRILLIC LETTER SHORT U
        "\xFF" => "\xAF",     # CYRILLIC LETTER DZHE
    );

    %fc = (%fc,
        "\xA1" => "\xF1",     # CYRILLIC CAPITAL LETTER IO                       --> CYRILLIC SMALL LETTER IO
        "\xA2" => "\xF2",     # CYRILLIC CAPITAL LETTER DJE                      --> CYRILLIC SMALL LETTER DJE
        "\xA3" => "\xF3",     # CYRILLIC CAPITAL LETTER GJE                      --> CYRILLIC SMALL LETTER GJE
        "\xA4" => "\xF4",     # CYRILLIC CAPITAL LETTER UKRAINIAN IE             --> CYRILLIC SMALL LETTER UKRAINIAN IE
        "\xA5" => "\xF5",     # CYRILLIC CAPITAL LETTER DZE                      --> CYRILLIC SMALL LETTER DZE
        "\xA6" => "\xF6",     # CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I --> CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        "\xA7" => "\xF7",     # CYRILLIC CAPITAL LETTER YI                       --> CYRILLIC SMALL LETTER YI
        "\xA8" => "\xF8",     # CYRILLIC CAPITAL LETTER JE                       --> CYRILLIC SMALL LETTER JE
        "\xA9" => "\xF9",     # CYRILLIC CAPITAL LETTER LJE                      --> CYRILLIC SMALL LETTER LJE
        "\xAA" => "\xFA",     # CYRILLIC CAPITAL LETTER NJE                      --> CYRILLIC SMALL LETTER NJE
        "\xAB" => "\xFB",     # CYRILLIC CAPITAL LETTER TSHE                     --> CYRILLIC SMALL LETTER TSHE
        "\xAC" => "\xFC",     # CYRILLIC CAPITAL LETTER KJE                      --> CYRILLIC SMALL LETTER KJE
        "\xAE" => "\xFE",     # CYRILLIC CAPITAL LETTER SHORT U                  --> CYRILLIC SMALL LETTER SHORT U
        "\xAF" => "\xFF",     # CYRILLIC CAPITAL LETTER DZHE                     --> CYRILLIC SMALL LETTER DZHE
        "\xB0" => "\xD0",     # CYRILLIC CAPITAL LETTER A                        --> CYRILLIC SMALL LETTER A
        "\xB1" => "\xD1",     # CYRILLIC CAPITAL LETTER BE                       --> CYRILLIC SMALL LETTER BE
        "\xB2" => "\xD2",     # CYRILLIC CAPITAL LETTER VE                       --> CYRILLIC SMALL LETTER VE
        "\xB3" => "\xD3",     # CYRILLIC CAPITAL LETTER GHE                      --> CYRILLIC SMALL LETTER GHE
        "\xB4" => "\xD4",     # CYRILLIC CAPITAL LETTER DE                       --> CYRILLIC SMALL LETTER DE
        "\xB5" => "\xD5",     # CYRILLIC CAPITAL LETTER IE                       --> CYRILLIC SMALL LETTER IE
        "\xB6" => "\xD6",     # CYRILLIC CAPITAL LETTER ZHE                      --> CYRILLIC SMALL LETTER ZHE
        "\xB7" => "\xD7",     # CYRILLIC CAPITAL LETTER ZE                       --> CYRILLIC SMALL LETTER ZE
        "\xB8" => "\xD8",     # CYRILLIC CAPITAL LETTER I                        --> CYRILLIC SMALL LETTER I
        "\xB9" => "\xD9",     # CYRILLIC CAPITAL LETTER SHORT I                  --> CYRILLIC SMALL LETTER SHORT I
        "\xBA" => "\xDA",     # CYRILLIC CAPITAL LETTER KA                       --> CYRILLIC SMALL LETTER KA
        "\xBB" => "\xDB",     # CYRILLIC CAPITAL LETTER EL                       --> CYRILLIC SMALL LETTER EL
        "\xBC" => "\xDC",     # CYRILLIC CAPITAL LETTER EM                       --> CYRILLIC SMALL LETTER EM
        "\xBD" => "\xDD",     # CYRILLIC CAPITAL LETTER EN                       --> CYRILLIC SMALL LETTER EN
        "\xBE" => "\xDE",     # CYRILLIC CAPITAL LETTER O                        --> CYRILLIC SMALL LETTER O
        "\xBF" => "\xDF",     # CYRILLIC CAPITAL LETTER PE                       --> CYRILLIC SMALL LETTER PE
        "\xC0" => "\xE0",     # CYRILLIC CAPITAL LETTER ER                       --> CYRILLIC SMALL LETTER ER
        "\xC1" => "\xE1",     # CYRILLIC CAPITAL LETTER ES                       --> CYRILLIC SMALL LETTER ES
        "\xC2" => "\xE2",     # CYRILLIC CAPITAL LETTER TE                       --> CYRILLIC SMALL LETTER TE
        "\xC3" => "\xE3",     # CYRILLIC CAPITAL LETTER U                        --> CYRILLIC SMALL LETTER U
        "\xC4" => "\xE4",     # CYRILLIC CAPITAL LETTER EF                       --> CYRILLIC SMALL LETTER EF
        "\xC5" => "\xE5",     # CYRILLIC CAPITAL LETTER HA                       --> CYRILLIC SMALL LETTER HA
        "\xC6" => "\xE6",     # CYRILLIC CAPITAL LETTER TSE                      --> CYRILLIC SMALL LETTER TSE
        "\xC7" => "\xE7",     # CYRILLIC CAPITAL LETTER CHE                      --> CYRILLIC SMALL LETTER CHE
        "\xC8" => "\xE8",     # CYRILLIC CAPITAL LETTER SHA                      --> CYRILLIC SMALL LETTER SHA
        "\xC9" => "\xE9",     # CYRILLIC CAPITAL LETTER SHCHA                    --> CYRILLIC SMALL LETTER SHCHA
        "\xCA" => "\xEA",     # CYRILLIC CAPITAL LETTER HARD SIGN                --> CYRILLIC SMALL LETTER HARD SIGN
        "\xCB" => "\xEB",     # CYRILLIC CAPITAL LETTER YERU                     --> CYRILLIC SMALL LETTER YERU
        "\xCC" => "\xEC",     # CYRILLIC CAPITAL LETTER SOFT SIGN                --> CYRILLIC SMALL LETTER SOFT SIGN
        "\xCD" => "\xED",     # CYRILLIC CAPITAL LETTER E                        --> CYRILLIC SMALL LETTER E
        "\xCE" => "\xEE",     # CYRILLIC CAPITAL LETTER YU                       --> CYRILLIC SMALL LETTER YU
        "\xCF" => "\xEF",     # CYRILLIC CAPITAL LETTER YA                       --> CYRILLIC SMALL LETTER YA
    );
}

#endif
#if DIST_KOI8R
elsif (__PACKAGE__ =~ / \b Ekoi8r \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xB3" => "\xA3",     # CYRILLIC LETTER IO
        "\xE0" => "\xC0",     # CYRILLIC LETTER IU
        "\xE1" => "\xC1",     # CYRILLIC LETTER A
        "\xE2" => "\xC2",     # CYRILLIC LETTER BE
        "\xE3" => "\xC3",     # CYRILLIC LETTER TSE
        "\xE4" => "\xC4",     # CYRILLIC LETTER DE
        "\xE5" => "\xC5",     # CYRILLIC LETTER IE
        "\xE6" => "\xC6",     # CYRILLIC LETTER EF
        "\xE7" => "\xC7",     # CYRILLIC LETTER GE
        "\xE8" => "\xC8",     # CYRILLIC LETTER KHA
        "\xE9" => "\xC9",     # CYRILLIC LETTER II
        "\xEA" => "\xCA",     # CYRILLIC LETTER SHORT II
        "\xEB" => "\xCB",     # CYRILLIC LETTER KA
        "\xEC" => "\xCC",     # CYRILLIC LETTER EL
        "\xED" => "\xCD",     # CYRILLIC LETTER EM
        "\xEE" => "\xCE",     # CYRILLIC LETTER EN
        "\xEF" => "\xCF",     # CYRILLIC LETTER O
        "\xF0" => "\xD0",     # CYRILLIC LETTER PE
        "\xF1" => "\xD1",     # CYRILLIC LETTER IA
        "\xF2" => "\xD2",     # CYRILLIC LETTER ER
        "\xF3" => "\xD3",     # CYRILLIC LETTER ES
        "\xF4" => "\xD4",     # CYRILLIC LETTER TE
        "\xF5" => "\xD5",     # CYRILLIC LETTER U
        "\xF6" => "\xD6",     # CYRILLIC LETTER ZHE
        "\xF7" => "\xD7",     # CYRILLIC LETTER VE
        "\xF8" => "\xD8",     # CYRILLIC LETTER SOFT SIGN
        "\xF9" => "\xD9",     # CYRILLIC LETTER YERI
        "\xFA" => "\xDA",     # CYRILLIC LETTER ZE
        "\xFB" => "\xDB",     # CYRILLIC LETTER SHA
        "\xFC" => "\xDC",     # CYRILLIC LETTER REVERSED E
        "\xFD" => "\xDD",     # CYRILLIC LETTER SHCHA
        "\xFE" => "\xDE",     # CYRILLIC LETTER CHE
        "\xFF" => "\xDF",     # CYRILLIC LETTER HARD SIGN
    );

    %uc = (%uc,
        "\xA3" => "\xB3",     # CYRILLIC LETTER IO
        "\xC0" => "\xE0",     # CYRILLIC LETTER IU
        "\xC1" => "\xE1",     # CYRILLIC LETTER A
        "\xC2" => "\xE2",     # CYRILLIC LETTER BE
        "\xC3" => "\xE3",     # CYRILLIC LETTER TSE
        "\xC4" => "\xE4",     # CYRILLIC LETTER DE
        "\xC5" => "\xE5",     # CYRILLIC LETTER IE
        "\xC6" => "\xE6",     # CYRILLIC LETTER EF
        "\xC7" => "\xE7",     # CYRILLIC LETTER GE
        "\xC8" => "\xE8",     # CYRILLIC LETTER KHA
        "\xC9" => "\xE9",     # CYRILLIC LETTER II
        "\xCA" => "\xEA",     # CYRILLIC LETTER SHORT II
        "\xCB" => "\xEB",     # CYRILLIC LETTER KA
        "\xCC" => "\xEC",     # CYRILLIC LETTER EL
        "\xCD" => "\xED",     # CYRILLIC LETTER EM
        "\xCE" => "\xEE",     # CYRILLIC LETTER EN
        "\xCF" => "\xEF",     # CYRILLIC LETTER O
        "\xD0" => "\xF0",     # CYRILLIC LETTER PE
        "\xD1" => "\xF1",     # CYRILLIC LETTER IA
        "\xD2" => "\xF2",     # CYRILLIC LETTER ER
        "\xD3" => "\xF3",     # CYRILLIC LETTER ES
        "\xD4" => "\xF4",     # CYRILLIC LETTER TE
        "\xD5" => "\xF5",     # CYRILLIC LETTER U
        "\xD6" => "\xF6",     # CYRILLIC LETTER ZHE
        "\xD7" => "\xF7",     # CYRILLIC LETTER VE
        "\xD8" => "\xF8",     # CYRILLIC LETTER SOFT SIGN
        "\xD9" => "\xF9",     # CYRILLIC LETTER YERI
        "\xDA" => "\xFA",     # CYRILLIC LETTER ZE
        "\xDB" => "\xFB",     # CYRILLIC LETTER SHA
        "\xDC" => "\xFC",     # CYRILLIC LETTER REVERSED E
        "\xDD" => "\xFD",     # CYRILLIC LETTER SHCHA
        "\xDE" => "\xFE",     # CYRILLIC LETTER CHE
        "\xDF" => "\xFF",     # CYRILLIC LETTER HARD SIGN
    );

    %fc = (%fc,
        "\xB3" => "\xA3",     # CYRILLIC CAPITAL LETTER IO        --> CYRILLIC SMALL LETTER IO
        "\xE0" => "\xC0",     # CYRILLIC CAPITAL LETTER YU        --> CYRILLIC SMALL LETTER YU
        "\xE1" => "\xC1",     # CYRILLIC CAPITAL LETTER A         --> CYRILLIC SMALL LETTER A
        "\xE2" => "\xC2",     # CYRILLIC CAPITAL LETTER BE        --> CYRILLIC SMALL LETTER BE
        "\xE3" => "\xC3",     # CYRILLIC CAPITAL LETTER TSE       --> CYRILLIC SMALL LETTER TSE
        "\xE4" => "\xC4",     # CYRILLIC CAPITAL LETTER DE        --> CYRILLIC SMALL LETTER DE
        "\xE5" => "\xC5",     # CYRILLIC CAPITAL LETTER IE        --> CYRILLIC SMALL LETTER IE
        "\xE6" => "\xC6",     # CYRILLIC CAPITAL LETTER EF        --> CYRILLIC SMALL LETTER EF
        "\xE7" => "\xC7",     # CYRILLIC CAPITAL LETTER GHE       --> CYRILLIC SMALL LETTER GHE
        "\xE8" => "\xC8",     # CYRILLIC CAPITAL LETTER HA        --> CYRILLIC SMALL LETTER HA
        "\xE9" => "\xC9",     # CYRILLIC CAPITAL LETTER I         --> CYRILLIC SMALL LETTER I
        "\xEA" => "\xCA",     # CYRILLIC CAPITAL LETTER SHORT I   --> CYRILLIC SMALL LETTER SHORT I
        "\xEB" => "\xCB",     # CYRILLIC CAPITAL LETTER KA        --> CYRILLIC SMALL LETTER KA
        "\xEC" => "\xCC",     # CYRILLIC CAPITAL LETTER EL        --> CYRILLIC SMALL LETTER EL
        "\xED" => "\xCD",     # CYRILLIC CAPITAL LETTER EM        --> CYRILLIC SMALL LETTER EM
        "\xEE" => "\xCE",     # CYRILLIC CAPITAL LETTER EN        --> CYRILLIC SMALL LETTER EN
        "\xEF" => "\xCF",     # CYRILLIC CAPITAL LETTER O         --> CYRILLIC SMALL LETTER O
        "\xF0" => "\xD0",     # CYRILLIC CAPITAL LETTER PE        --> CYRILLIC SMALL LETTER PE
        "\xF1" => "\xD1",     # CYRILLIC CAPITAL LETTER YA        --> CYRILLIC SMALL LETTER YA
        "\xF2" => "\xD2",     # CYRILLIC CAPITAL LETTER ER        --> CYRILLIC SMALL LETTER ER
        "\xF3" => "\xD3",     # CYRILLIC CAPITAL LETTER ES        --> CYRILLIC SMALL LETTER ES
        "\xF4" => "\xD4",     # CYRILLIC CAPITAL LETTER TE        --> CYRILLIC SMALL LETTER TE
        "\xF5" => "\xD5",     # CYRILLIC CAPITAL LETTER U         --> CYRILLIC SMALL LETTER U
        "\xF6" => "\xD6",     # CYRILLIC CAPITAL LETTER ZHE       --> CYRILLIC SMALL LETTER ZHE
        "\xF7" => "\xD7",     # CYRILLIC CAPITAL LETTER VE        --> CYRILLIC SMALL LETTER VE
        "\xF8" => "\xD8",     # CYRILLIC CAPITAL LETTER SOFT SIGN --> CYRILLIC SMALL LETTER SOFT SIGN
        "\xF9" => "\xD9",     # CYRILLIC CAPITAL LETTER YERU      --> CYRILLIC SMALL LETTER YERU
        "\xFA" => "\xDA",     # CYRILLIC CAPITAL LETTER ZE        --> CYRILLIC SMALL LETTER ZE
        "\xFB" => "\xDB",     # CYRILLIC CAPITAL LETTER SHA       --> CYRILLIC SMALL LETTER SHA
        "\xFC" => "\xDC",     # CYRILLIC CAPITAL LETTER E         --> CYRILLIC SMALL LETTER E
        "\xFD" => "\xDD",     # CYRILLIC CAPITAL LETTER SHCHA     --> CYRILLIC SMALL LETTER SHCHA
        "\xFE" => "\xDE",     # CYRILLIC CAPITAL LETTER CHE       --> CYRILLIC SMALL LETTER CHE
        "\xFF" => "\xDF",     # CYRILLIC CAPITAL LETTER HARD SIGN --> CYRILLIC SMALL LETTER HARD SIGN
    );
}

#endif
#if DIST_KOI8U
elsif (__PACKAGE__ =~ / \b Ekoi8u \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xB3" => "\xA3",     # CYRILLIC LETTER IO
        "\xB4" => "\xA4",     # CYRILLIC LETTER UKRAINIAN IE
        "\xB6" => "\xA6",     # CYRILLIC LETTER BYELORUSSIAN-UKRAINIAN I
        "\xB7" => "\xA7",     # CYRILLIC LETTER YI (UKRAINIAN)
        "\xBD" => "\xAD",     # CYRILLIC LETTER GHE WITH UPTURN
        "\xE0" => "\xC0",     # CYRILLIC LETTER YU
        "\xE1" => "\xC1",     # CYRILLIC LETTER A
        "\xE2" => "\xC2",     # CYRILLIC LETTER BE
        "\xE3" => "\xC3",     # CYRILLIC LETTER TSE
        "\xE4" => "\xC4",     # CYRILLIC LETTER DE
        "\xE5" => "\xC5",     # CYRILLIC LETTER IE
        "\xE6" => "\xC6",     # CYRILLIC LETTER EF
        "\xE7" => "\xC7",     # CYRILLIC LETTER GHE
        "\xE8" => "\xC8",     # CYRILLIC LETTER KHA
        "\xE9" => "\xC9",     # CYRILLIC LETTER I
        "\xEA" => "\xCA",     # CYRILLIC LETTER SHORT I
        "\xEB" => "\xCB",     # CYRILLIC LETTER KA
        "\xEC" => "\xCC",     # CYRILLIC LETTER EL
        "\xED" => "\xCD",     # CYRILLIC LETTER EM
        "\xEE" => "\xCE",     # CYRILLIC LETTER EN
        "\xEF" => "\xCF",     # CYRILLIC LETTER O
        "\xF0" => "\xD0",     # CYRILLIC LETTER PE
        "\xF1" => "\xD1",     # CYRILLIC LETTER YA
        "\xF2" => "\xD2",     # CYRILLIC LETTER ER
        "\xF3" => "\xD3",     # CYRILLIC LETTER ES
        "\xF4" => "\xD4",     # CYRILLIC LETTER TE
        "\xF5" => "\xD5",     # CYRILLIC LETTER U
        "\xF6" => "\xD6",     # CYRILLIC LETTER ZHE
        "\xF7" => "\xD7",     # CYRILLIC LETTER VE
        "\xF8" => "\xD8",     # CYRILLIC LETTER SOFT SIGN
        "\xF9" => "\xD9",     # CYRILLIC LETTER YERU
        "\xFA" => "\xDA",     # CYRILLIC LETTER ZE
        "\xFB" => "\xDB",     # CYRILLIC LETTER SHA
        "\xFC" => "\xDC",     # CYRILLIC LETTER E
        "\xFD" => "\xDD",     # CYRILLIC LETTER SHCHA
        "\xFE" => "\xDE",     # CYRILLIC LETTER CHE
        "\xFF" => "\xDF",     # CYRILLIC LETTER HARD SIGN
    );

    %uc = (%uc,
        "\xA3" => "\xB3",     # CYRILLIC LETTER IO
        "\xA4" => "\xB4",     # CYRILLIC LETTER UKRAINIAN IE
        "\xA6" => "\xB6",     # CYRILLIC LETTER BYELORUSSIAN-UKRAINIAN I
        "\xA7" => "\xB7",     # CYRILLIC LETTER YI (UKRAINIAN)
        "\xAD" => "\xBD",     # CYRILLIC LETTER GHE WITH UPTURN
        "\xC0" => "\xE0",     # CYRILLIC LETTER YU
        "\xC1" => "\xE1",     # CYRILLIC LETTER A
        "\xC2" => "\xE2",     # CYRILLIC LETTER BE
        "\xC3" => "\xE3",     # CYRILLIC LETTER TSE
        "\xC4" => "\xE4",     # CYRILLIC LETTER DE
        "\xC5" => "\xE5",     # CYRILLIC LETTER IE
        "\xC6" => "\xE6",     # CYRILLIC LETTER EF
        "\xC7" => "\xE7",     # CYRILLIC LETTER GHE
        "\xC8" => "\xE8",     # CYRILLIC LETTER KHA
        "\xC9" => "\xE9",     # CYRILLIC LETTER I
        "\xCA" => "\xEA",     # CYRILLIC LETTER SHORT I
        "\xCB" => "\xEB",     # CYRILLIC LETTER KA
        "\xCC" => "\xEC",     # CYRILLIC LETTER EL
        "\xCD" => "\xED",     # CYRILLIC LETTER EM
        "\xCE" => "\xEE",     # CYRILLIC LETTER EN
        "\xCF" => "\xEF",     # CYRILLIC LETTER O
        "\xD0" => "\xF0",     # CYRILLIC LETTER PE
        "\xD1" => "\xF1",     # CYRILLIC LETTER YA
        "\xD2" => "\xF2",     # CYRILLIC LETTER ER
        "\xD3" => "\xF3",     # CYRILLIC LETTER ES
        "\xD4" => "\xF4",     # CYRILLIC LETTER TE
        "\xD5" => "\xF5",     # CYRILLIC LETTER U
        "\xD6" => "\xF6",     # CYRILLIC LETTER ZHE
        "\xD7" => "\xF7",     # CYRILLIC LETTER VE
        "\xD8" => "\xF8",     # CYRILLIC LETTER SOFT SIGN
        "\xD9" => "\xF9",     # CYRILLIC LETTER YERU
        "\xDA" => "\xFA",     # CYRILLIC LETTER ZE
        "\xDB" => "\xFB",     # CYRILLIC LETTER SHA
        "\xDC" => "\xFC",     # CYRILLIC LETTER E
        "\xDD" => "\xFD",     # CYRILLIC LETTER SHCHA
        "\xDE" => "\xFE",     # CYRILLIC LETTER CHE
        "\xDF" => "\xFF",     # CYRILLIC LETTER HARD SIGN
    );

    %fc = (%fc,
        "\xB3" => "\xA3",     # CYRILLIC CAPITAL LETTER IO                       --> CYRILLIC SMALL LETTER IO
        "\xB4" => "\xA4",     # CYRILLIC CAPITAL LETTER UKRAINIAN IE             --> CYRILLIC SMALL LETTER UKRAINIAN IE
        "\xB6" => "\xA6",     # CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I --> CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
        "\xB7" => "\xA7",     # CYRILLIC CAPITAL LETTER YI                       --> CYRILLIC SMALL LETTER YI
        "\xBD" => "\xAD",     # CYRILLIC CAPITAL LETTER GHE WITH UPTURN          --> CYRILLIC SMALL LETTER GHE WITH UPTURN
        "\xE0" => "\xC0",     # CYRILLIC CAPITAL LETTER YU                       --> CYRILLIC SMALL LETTER YU
        "\xE1" => "\xC1",     # CYRILLIC CAPITAL LETTER A                        --> CYRILLIC SMALL LETTER A
        "\xE2" => "\xC2",     # CYRILLIC CAPITAL LETTER BE                       --> CYRILLIC SMALL LETTER BE
        "\xE3" => "\xC3",     # CYRILLIC CAPITAL LETTER TSE                      --> CYRILLIC SMALL LETTER TSE
        "\xE4" => "\xC4",     # CYRILLIC CAPITAL LETTER DE                       --> CYRILLIC SMALL LETTER DE
        "\xE5" => "\xC5",     # CYRILLIC CAPITAL LETTER IE                       --> CYRILLIC SMALL LETTER IE
        "\xE6" => "\xC6",     # CYRILLIC CAPITAL LETTER EF                       --> CYRILLIC SMALL LETTER EF
        "\xE7" => "\xC7",     # CYRILLIC CAPITAL LETTER GHE                      --> CYRILLIC SMALL LETTER GHE
        "\xE8" => "\xC8",     # CYRILLIC CAPITAL LETTER HA                       --> CYRILLIC SMALL LETTER HA
        "\xE9" => "\xC9",     # CYRILLIC CAPITAL LETTER I                        --> CYRILLIC SMALL LETTER I
        "\xEA" => "\xCA",     # CYRILLIC CAPITAL LETTER SHORT I                  --> CYRILLIC SMALL LETTER SHORT I
        "\xEB" => "\xCB",     # CYRILLIC CAPITAL LETTER KA                       --> CYRILLIC SMALL LETTER KA
        "\xEC" => "\xCC",     # CYRILLIC CAPITAL LETTER EL                       --> CYRILLIC SMALL LETTER EL
        "\xED" => "\xCD",     # CYRILLIC CAPITAL LETTER EM                       --> CYRILLIC SMALL LETTER EM
        "\xEE" => "\xCE",     # CYRILLIC CAPITAL LETTER EN                       --> CYRILLIC SMALL LETTER EN
        "\xEF" => "\xCF",     # CYRILLIC CAPITAL LETTER O                        --> CYRILLIC SMALL LETTER O
        "\xF0" => "\xD0",     # CYRILLIC CAPITAL LETTER PE                       --> CYRILLIC SMALL LETTER PE
        "\xF1" => "\xD1",     # CYRILLIC CAPITAL LETTER YA                       --> CYRILLIC SMALL LETTER YA
        "\xF2" => "\xD2",     # CYRILLIC CAPITAL LETTER ER                       --> CYRILLIC SMALL LETTER ER
        "\xF3" => "\xD3",     # CYRILLIC CAPITAL LETTER ES                       --> CYRILLIC SMALL LETTER ES
        "\xF4" => "\xD4",     # CYRILLIC CAPITAL LETTER TE                       --> CYRILLIC SMALL LETTER TE
        "\xF5" => "\xD5",     # CYRILLIC CAPITAL LETTER U                        --> CYRILLIC SMALL LETTER U
        "\xF6" => "\xD6",     # CYRILLIC CAPITAL LETTER ZHE                      --> CYRILLIC SMALL LETTER ZHE
        "\xF7" => "\xD7",     # CYRILLIC CAPITAL LETTER VE                       --> CYRILLIC SMALL LETTER VE
        "\xF8" => "\xD8",     # CYRILLIC CAPITAL LETTER SOFT SIGN                --> CYRILLIC SMALL LETTER SOFT SIGN
        "\xF9" => "\xD9",     # CYRILLIC CAPITAL LETTER YERU                     --> CYRILLIC SMALL LETTER YERU
        "\xFA" => "\xDA",     # CYRILLIC CAPITAL LETTER ZE                       --> CYRILLIC SMALL LETTER ZE
        "\xFB" => "\xDB",     # CYRILLIC CAPITAL LETTER SHA                      --> CYRILLIC SMALL LETTER SHA
        "\xFC" => "\xDC",     # CYRILLIC CAPITAL LETTER E                        --> CYRILLIC SMALL LETTER E
        "\xFD" => "\xDD",     # CYRILLIC CAPITAL LETTER SHCHA                    --> CYRILLIC SMALL LETTER SHCHA
        "\xFE" => "\xDE",     # CYRILLIC CAPITAL LETTER CHE                      --> CYRILLIC SMALL LETTER CHE
        "\xFF" => "\xDF",     # CYRILLIC CAPITAL LETTER HARD SIGN                --> CYRILLIC SMALL LETTER HARD SIGN
    );
}

#endif
#if DIST_ARABIC
elsif (__PACKAGE__ =~ / \b Earabic \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );
}

#endif
#if DIST_GREEK
elsif (__PACKAGE__ =~ / \b Egreek \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xB6" => "\xDC",     # GREEK LETTER ALPHA WITH TONOS
        "\xB8" => "\xDD",     # GREEK LETTER EPSILON WITH TONOS
        "\xB9" => "\xDE",     # GREEK LETTER ETA WITH TONOS
        "\xBA" => "\xDF",     # GREEK LETTER IOTA WITH TONOS
        "\xBC" => "\xFC",     # GREEK LETTER OMICRON WITH TONOS
        "\xBE" => "\xFD",     # GREEK LETTER UPSILON WITH TONOS
        "\xBF" => "\xFE",     # GREEK LETTER OMEGA WITH TONOS
        "\xC1" => "\xE1",     # GREEK LETTER ALPHA
        "\xC2" => "\xE2",     # GREEK LETTER BETA
        "\xC3" => "\xE3",     # GREEK LETTER GAMMA
        "\xC4" => "\xE4",     # GREEK LETTER DELTA
        "\xC5" => "\xE5",     # GREEK LETTER EPSILON
        "\xC6" => "\xE6",     # GREEK LETTER ZETA
        "\xC7" => "\xE7",     # GREEK LETTER ETA
        "\xC8" => "\xE8",     # GREEK LETTER THETA
        "\xC9" => "\xE9",     # GREEK LETTER IOTA
        "\xCA" => "\xEA",     # GREEK LETTER KAPPA
        "\xCB" => "\xEB",     # GREEK LETTER LAMDA
        "\xCC" => "\xEC",     # GREEK LETTER MU
        "\xCD" => "\xED",     # GREEK LETTER NU
        "\xCE" => "\xEE",     # GREEK LETTER XI
        "\xCF" => "\xEF",     # GREEK LETTER OMICRON
        "\xD0" => "\xF0",     # GREEK LETTER PI
        "\xD1" => "\xF1",     # GREEK LETTER RHO
        "\xD3" => "\xF3",     # GREEK LETTER SIGMA
        "\xD4" => "\xF4",     # GREEK LETTER TAU
        "\xD5" => "\xF5",     # GREEK LETTER UPSILON
        "\xD6" => "\xF6",     # GREEK LETTER PHI
        "\xD7" => "\xF7",     # GREEK LETTER CHI
        "\xD8" => "\xF8",     # GREEK LETTER PSI
        "\xD9" => "\xF9",     # GREEK LETTER OMEGA
        "\xDA" => "\xFA",     # GREEK LETTER IOTA WITH DIALYTIKA
        "\xDB" => "\xFB",     # GREEK LETTER UPSILON WITH DIALYTIKA
    );

    %uc = (%uc,
        "\xDC" => "\xB6",     # GREEK LETTER ALPHA WITH TONOS
        "\xDD" => "\xB8",     # GREEK LETTER EPSILON WITH TONOS
        "\xDE" => "\xB9",     # GREEK LETTER ETA WITH TONOS
        "\xDF" => "\xBA",     # GREEK LETTER IOTA WITH TONOS
        "\xE1" => "\xC1",     # GREEK LETTER ALPHA
        "\xE2" => "\xC2",     # GREEK LETTER BETA
        "\xE3" => "\xC3",     # GREEK LETTER GAMMA
        "\xE4" => "\xC4",     # GREEK LETTER DELTA
        "\xE5" => "\xC5",     # GREEK LETTER EPSILON
        "\xE6" => "\xC6",     # GREEK LETTER ZETA
        "\xE7" => "\xC7",     # GREEK LETTER ETA
        "\xE8" => "\xC8",     # GREEK LETTER THETA
        "\xE9" => "\xC9",     # GREEK LETTER IOTA
        "\xEA" => "\xCA",     # GREEK LETTER KAPPA
        "\xEB" => "\xCB",     # GREEK LETTER LAMDA
        "\xEC" => "\xCC",     # GREEK LETTER MU
        "\xED" => "\xCD",     # GREEK LETTER NU
        "\xEE" => "\xCE",     # GREEK LETTER XI
        "\xEF" => "\xCF",     # GREEK LETTER OMICRON
        "\xF0" => "\xD0",     # GREEK LETTER PI
        "\xF1" => "\xD1",     # GREEK LETTER RHO
        "\xF3" => "\xD3",     # GREEK LETTER SIGMA
        "\xF4" => "\xD4",     # GREEK LETTER TAU
        "\xF5" => "\xD5",     # GREEK LETTER UPSILON
        "\xF6" => "\xD6",     # GREEK LETTER PHI
        "\xF7" => "\xD7",     # GREEK LETTER CHI
        "\xF8" => "\xD8",     # GREEK LETTER PSI
        "\xF9" => "\xD9",     # GREEK LETTER OMEGA
        "\xFA" => "\xDA",     # GREEK LETTER IOTA WITH DIALYTIKA
        "\xFB" => "\xDB",     # GREEK LETTER UPSILON WITH DIALYTIKA
        "\xFC" => "\xBC",     # GREEK LETTER OMICRON WITH TONOS
        "\xFD" => "\xBE",     # GREEK LETTER UPSILON WITH TONOS
        "\xFE" => "\xBF",     # GREEK LETTER OMEGA WITH TONOS
    );

    %fc = (%fc,
        "\xB6" => "\xDC",     # GREEK CAPITAL LETTER ALPHA WITH TONOS       --> GREEK SMALL LETTER ALPHA WITH TONOS
        "\xB8" => "\xDD",     # GREEK CAPITAL LETTER EPSILON WITH TONOS     --> GREEK SMALL LETTER EPSILON WITH TONOS
        "\xB9" => "\xDE",     # GREEK CAPITAL LETTER ETA WITH TONOS         --> GREEK SMALL LETTER ETA WITH TONOS
        "\xBA" => "\xDF",     # GREEK CAPITAL LETTER IOTA WITH TONOS        --> GREEK SMALL LETTER IOTA WITH TONOS
        "\xBC" => "\xFC",     # GREEK CAPITAL LETTER OMICRON WITH TONOS     --> GREEK SMALL LETTER OMICRON WITH TONOS
        "\xBE" => "\xFD",     # GREEK CAPITAL LETTER UPSILON WITH TONOS     --> GREEK SMALL LETTER UPSILON WITH TONOS
        "\xBF" => "\xFE",     # GREEK CAPITAL LETTER OMEGA WITH TONOS       --> GREEK SMALL LETTER OMEGA WITH TONOS
        "\xC1" => "\xE1",     # GREEK CAPITAL LETTER ALPHA                  --> GREEK SMALL LETTER ALPHA
        "\xC2" => "\xE2",     # GREEK CAPITAL LETTER BETA                   --> GREEK SMALL LETTER BETA
        "\xC3" => "\xE3",     # GREEK CAPITAL LETTER GAMMA                  --> GREEK SMALL LETTER GAMMA
        "\xC4" => "\xE4",     # GREEK CAPITAL LETTER DELTA                  --> GREEK SMALL LETTER DELTA
        "\xC5" => "\xE5",     # GREEK CAPITAL LETTER EPSILON                --> GREEK SMALL LETTER EPSILON
        "\xC6" => "\xE6",     # GREEK CAPITAL LETTER ZETA                   --> GREEK SMALL LETTER ZETA
        "\xC7" => "\xE7",     # GREEK CAPITAL LETTER ETA                    --> GREEK SMALL LETTER ETA
        "\xC8" => "\xE8",     # GREEK CAPITAL LETTER THETA                  --> GREEK SMALL LETTER THETA
        "\xC9" => "\xE9",     # GREEK CAPITAL LETTER IOTA                   --> GREEK SMALL LETTER IOTA
        "\xCA" => "\xEA",     # GREEK CAPITAL LETTER KAPPA                  --> GREEK SMALL LETTER KAPPA
        "\xCB" => "\xEB",     # GREEK CAPITAL LETTER LAMDA                  --> GREEK SMALL LETTER LAMDA
        "\xCC" => "\xEC",     # GREEK CAPITAL LETTER MU                     --> GREEK SMALL LETTER MU
        "\xCD" => "\xED",     # GREEK CAPITAL LETTER NU                     --> GREEK SMALL LETTER NU
        "\xCE" => "\xEE",     # GREEK CAPITAL LETTER XI                     --> GREEK SMALL LETTER XI
        "\xCF" => "\xEF",     # GREEK CAPITAL LETTER OMICRON                --> GREEK SMALL LETTER OMICRON
        "\xD0" => "\xF0",     # GREEK CAPITAL LETTER PI                     --> GREEK SMALL LETTER PI
        "\xD1" => "\xF1",     # GREEK CAPITAL LETTER RHO                    --> GREEK SMALL LETTER RHO
        "\xD3" => "\xF3",     # GREEK CAPITAL LETTER SIGMA                  --> GREEK SMALL LETTER SIGMA
        "\xD4" => "\xF4",     # GREEK CAPITAL LETTER TAU                    --> GREEK SMALL LETTER TAU
        "\xD5" => "\xF5",     # GREEK CAPITAL LETTER UPSILON                --> GREEK SMALL LETTER UPSILON
        "\xD6" => "\xF6",     # GREEK CAPITAL LETTER PHI                    --> GREEK SMALL LETTER PHI
        "\xD7" => "\xF7",     # GREEK CAPITAL LETTER CHI                    --> GREEK SMALL LETTER CHI
        "\xD8" => "\xF8",     # GREEK CAPITAL LETTER PSI                    --> GREEK SMALL LETTER PSI
        "\xD9" => "\xF9",     # GREEK CAPITAL LETTER OMEGA                  --> GREEK SMALL LETTER OMEGA
        "\xDA" => "\xFA",     # GREEK CAPITAL LETTER IOTA WITH DIALYTIKA    --> GREEK SMALL LETTER IOTA WITH DIALYTIKA
        "\xDB" => "\xFB",     # GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA --> GREEK SMALL LETTER UPSILON WITH DIALYTIKA
        "\xF2" => "\xF3",     # GREEK SMALL LETTER FINAL SIGMA              --> GREEK SMALL LETTER SIGMA
    );
}

#endif
#if DIST_HEBREW
elsif (__PACKAGE__ =~ / \b Ehebrew \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );
}

#endif
#if DIST_LATIN5
elsif (__PACKAGE__ =~ / \b Elatin5 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER G WITH BREVE
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDE" => "\xFE",     # LATIN LETTER S WITH CEDILLA
    );

    %uc = (%uc,
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER G WITH BREVE
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFE" => "\xDE",     # LATIN LETTER S WITH CEDILLA
    );

    %fc = (%fc,
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER G WITH BREVE      --> LATIN SMALL LETTER G WITH BREVE
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS

        # CaseFolding-6.1.0.txt
        # Date: 2011-07-25, 21:21:56 GMT [MD]
        #
        # T: special case for uppercase I and dotted uppercase I
        #    - For non-Turkic languages, this mapping is normally not used.
        #    - For Turkic languages (tr, az), this mapping can be used instead of the normal mapping for these characters.
        #      Note that the Turkic mappings do not maintain canonical equivalence without additional processing.
        #      See the discussions of case mapping in the Unicode Standard for more information.

        #-------------------------------------------------------------------------------
        "\xDD" => "\x69",     # LATIN CAPITAL LETTER I WITH DOT ABOVE
                              # --> LATIN SMALL LETTER I (without COMBINING DOT ABOVE)
        #-------------------------------------------------------------------------------

        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER S WITH CEDILLA --> LATIN SMALL LETTER S WITH CEDILLA
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN6
elsif (__PACKAGE__ =~ / \b Elatin6 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xB1",     # LATIN LETTER A WITH OGONEK
        "\xA2" => "\xB2",     # LATIN LETTER E WITH MACRON
        "\xA3" => "\xB3",     # LATIN LETTER G WITH CEDILLA
        "\xA4" => "\xB4",     # LATIN LETTER I WITH MACRON
        "\xA5" => "\xB5",     # LATIN LETTER I WITH TILDE
        "\xA6" => "\xB6",     # LATIN LETTER K WITH CEDILLA
        "\xA8" => "\xB8",     # LATIN LETTER L WITH CEDILLA
        "\xA9" => "\xB9",     # LATIN LETTER D WITH STROKE
        "\xAA" => "\xBA",     # LATIN LETTER S WITH CARON
        "\xAB" => "\xBB",     # LATIN LETTER T WITH STROKE
        "\xAC" => "\xBC",     # LATIN LETTER Z WITH CARON
        "\xAE" => "\xBE",     # LATIN LETTER U WITH MACRON
        "\xAF" => "\xBF",     # LATIN LETTER ENG
        "\xC0" => "\xE0",     # LATIN LETTER A WITH MACRON
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER I WITH OGONEK
        "\xC8" => "\xE8",     # LATIN LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER E WITH DOT ABOVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER ETH (Icelandic)
        "\xD1" => "\xF1",     # LATIN LETTER N WITH CEDILLA
        "\xD2" => "\xF2",     # LATIN LETTER O WITH MACRON
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN LETTER U WITH TILDE
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH OGONEK
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER THORN (Icelandic)
    );

    %uc = (%uc,
        "\xB1" => "\xA1",     # LATIN LETTER A WITH OGONEK
        "\xB2" => "\xA2",     # LATIN LETTER E WITH MACRON
        "\xB3" => "\xA3",     # LATIN LETTER G WITH CEDILLA
        "\xB4" => "\xA4",     # LATIN LETTER I WITH MACRON
        "\xB5" => "\xA5",     # LATIN LETTER I WITH TILDE
        "\xB6" => "\xA6",     # LATIN LETTER K WITH CEDILLA
        "\xB8" => "\xA8",     # LATIN LETTER L WITH CEDILLA
        "\xB9" => "\xA9",     # LATIN LETTER D WITH STROKE
        "\xBA" => "\xAA",     # LATIN LETTER S WITH CARON
        "\xBB" => "\xAB",     # LATIN LETTER T WITH STROKE
        "\xBC" => "\xAC",     # LATIN LETTER Z WITH CARON
        "\xBE" => "\xAE",     # LATIN LETTER U WITH MACRON
        "\xBF" => "\xAF",     # LATIN LETTER ENG
        "\xE0" => "\xC0",     # LATIN LETTER A WITH MACRON
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER I WITH OGONEK
        "\xE8" => "\xC8",     # LATIN LETTER C WITH CARON
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH OGONEK
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER E WITH DOT ABOVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER ETH (Icelandic)
        "\xF1" => "\xD1",     # LATIN LETTER N WITH CEDILLA
        "\xF2" => "\xD2",     # LATIN LETTER O WITH MACRON
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF7" => "\xD7",     # LATIN LETTER U WITH TILDE
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH OGONEK
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER THORN (Icelandic)
    );

    %fc = (%fc,
        "\xA1" => "\xB1",     # LATIN CAPITAL LETTER A WITH OGONEK     --> LATIN SMALL LETTER A WITH OGONEK
        "\xA2" => "\xB2",     # LATIN CAPITAL LETTER E WITH MACRON     --> LATIN SMALL LETTER E WITH MACRON
        "\xA3" => "\xB3",     # LATIN CAPITAL LETTER G WITH CEDILLA    --> LATIN SMALL LETTER G WITH CEDILLA
        "\xA4" => "\xB4",     # LATIN CAPITAL LETTER I WITH MACRON     --> LATIN SMALL LETTER I WITH MACRON
        "\xA5" => "\xB5",     # LATIN CAPITAL LETTER I WITH TILDE      --> LATIN SMALL LETTER I WITH TILDE
        "\xA6" => "\xB6",     # LATIN CAPITAL LETTER K WITH CEDILLA    --> LATIN SMALL LETTER K WITH CEDILLA
        "\xA8" => "\xB8",     # LATIN CAPITAL LETTER L WITH CEDILLA    --> LATIN SMALL LETTER L WITH CEDILLA
        "\xA9" => "\xB9",     # LATIN CAPITAL LETTER D WITH STROKE     --> LATIN SMALL LETTER D WITH STROKE
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER S WITH CARON      --> LATIN SMALL LETTER S WITH CARON
        "\xAB" => "\xBB",     # LATIN CAPITAL LETTER T WITH STROKE     --> LATIN SMALL LETTER T WITH STROKE
        "\xAC" => "\xBC",     # LATIN CAPITAL LETTER Z WITH CARON      --> LATIN SMALL LETTER Z WITH CARON
        "\xAE" => "\xBE",     # LATIN CAPITAL LETTER U WITH MACRON     --> LATIN SMALL LETTER U WITH MACRON
        "\xAF" => "\xBF",     # LATIN CAPITAL LETTER ENG               --> LATIN SMALL LETTER ENG
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH MACRON     --> LATIN SMALL LETTER A WITH MACRON
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER I WITH OGONEK     --> LATIN SMALL LETTER I WITH OGONEK
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER C WITH CARON      --> LATIN SMALL LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH OGONEK     --> LATIN SMALL LETTER E WITH OGONEK
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER E WITH DOT ABOVE  --> LATIN SMALL LETTER E WITH DOT ABOVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER ETH               --> LATIN SMALL LETTER ETH
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH CEDILLA    --> LATIN SMALL LETTER N WITH CEDILLA
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH MACRON     --> LATIN SMALL LETTER O WITH MACRON
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN CAPITAL LETTER U WITH TILDE      --> LATIN SMALL LETTER U WITH TILDE
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH OGONEK     --> LATIN SMALL LETTER U WITH OGONEK
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE      --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER THORN             --> LATIN SMALL LETTER THORN
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_TIS620
elsif (__PACKAGE__ =~ / \b Etis620 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );
}

#endif
#if DIST_LATIN7
elsif (__PACKAGE__ =~ / \b Elatin7 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA8" => "\xB8",     # LATIN LETTER O WITH STROKE
        "\xAA" => "\xBA",     # LATIN LETTER R WITH CEDILLA
        "\xAF" => "\xBF",     # LATIN LETTER AE
        "\xC0" => "\xE0",     # LATIN LETTER A WITH OGONEK
        "\xC1" => "\xE1",     # LATIN LETTER I WITH OGONEK
        "\xC2" => "\xE2",     # LATIN LETTER A WITH MACRON
        "\xC3" => "\xE3",     # LATIN LETTER C WITH ACUTE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER E WITH OGONEK
        "\xC7" => "\xE7",     # LATIN LETTER E WITH MACRON
        "\xC8" => "\xE8",     # LATIN LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER Z WITH ACUTE
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DOT ABOVE
        "\xCC" => "\xEC",     # LATIN LETTER G WITH CEDILLA
        "\xCD" => "\xED",     # LATIN LETTER K WITH CEDILLA
        "\xCE" => "\xEE",     # LATIN LETTER I WITH MACRON
        "\xCF" => "\xEF",     # LATIN LETTER L WITH CEDILLA
        "\xD0" => "\xF0",     # LATIN LETTER S WITH CARON
        "\xD1" => "\xF1",     # LATIN LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN LETTER N WITH CEDILLA
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH MACRON
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER U WITH OGONEK
        "\xD9" => "\xF9",     # LATIN LETTER L WITH STROKE
        "\xDA" => "\xFA",     # LATIN LETTER S WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH MACRON
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Z WITH DOT ABOVE
        "\xDE" => "\xFE",     # LATIN LETTER Z WITH CARON
    );

    %uc = (%uc,
        "\xB8" => "\xA8",     # LATIN LETTER O WITH STROKE
        "\xBA" => "\xAA",     # LATIN LETTER R WITH CEDILLA
        "\xBF" => "\xAF",     # LATIN LETTER AE
        "\xE0" => "\xC0",     # LATIN LETTER A WITH OGONEK
        "\xE1" => "\xC1",     # LATIN LETTER I WITH OGONEK
        "\xE2" => "\xC2",     # LATIN LETTER A WITH MACRON
        "\xE3" => "\xC3",     # LATIN LETTER C WITH ACUTE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER E WITH OGONEK
        "\xE7" => "\xC7",     # LATIN LETTER E WITH MACRON
        "\xE8" => "\xC8",     # LATIN LETTER C WITH CARON
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER Z WITH ACUTE
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DOT ABOVE
        "\xEC" => "\xCC",     # LATIN LETTER G WITH CEDILLA
        "\xED" => "\xCD",     # LATIN LETTER K WITH CEDILLA
        "\xEE" => "\xCE",     # LATIN LETTER I WITH MACRON
        "\xEF" => "\xCF",     # LATIN LETTER L WITH CEDILLA
        "\xF0" => "\xD0",     # LATIN LETTER S WITH CARON
        "\xF1" => "\xD1",     # LATIN LETTER N WITH ACUTE
        "\xF2" => "\xD2",     # LATIN LETTER N WITH CEDILLA
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH MACRON
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER U WITH OGONEK
        "\xF9" => "\xD9",     # LATIN LETTER L WITH STROKE
        "\xFA" => "\xDA",     # LATIN LETTER S WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH MACRON
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Z WITH DOT ABOVE
        "\xFE" => "\xDE",     # LATIN LETTER Z WITH CARON
    );

    %fc = (%fc,
        "\xA8" => "\xB8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER R WITH CEDILLA    --> LATIN SMALL LETTER R WITH CEDILLA
        "\xAF" => "\xBF",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH OGONEK     --> LATIN SMALL LETTER A WITH OGONEK
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER I WITH OGONEK     --> LATIN SMALL LETTER I WITH OGONEK
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH MACRON     --> LATIN SMALL LETTER A WITH MACRON
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER C WITH ACUTE      --> LATIN SMALL LETTER C WITH ACUTE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER E WITH OGONEK     --> LATIN SMALL LETTER E WITH OGONEK
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER E WITH MACRON     --> LATIN SMALL LETTER E WITH MACRON
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER C WITH CARON      --> LATIN SMALL LETTER C WITH CARON
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER Z WITH ACUTE      --> LATIN SMALL LETTER Z WITH ACUTE
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DOT ABOVE  --> LATIN SMALL LETTER E WITH DOT ABOVE
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER G WITH CEDILLA    --> LATIN SMALL LETTER G WITH CEDILLA
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER K WITH CEDILLA    --> LATIN SMALL LETTER K WITH CEDILLA
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH MACRON     --> LATIN SMALL LETTER I WITH MACRON
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER L WITH CEDILLA    --> LATIN SMALL LETTER L WITH CEDILLA
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER S WITH CARON      --> LATIN SMALL LETTER S WITH CARON
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH ACUTE      --> LATIN SMALL LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER N WITH CEDILLA    --> LATIN SMALL LETTER N WITH CEDILLA
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH MACRON     --> LATIN SMALL LETTER O WITH MACRON
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER U WITH OGONEK     --> LATIN SMALL LETTER U WITH OGONEK
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER L WITH STROKE     --> LATIN SMALL LETTER L WITH STROKE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER S WITH ACUTE      --> LATIN SMALL LETTER S WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH MACRON     --> LATIN SMALL LETTER U WITH MACRON
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Z WITH DOT ABOVE  --> LATIN SMALL LETTER Z WITH DOT ABOVE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER Z WITH CARON      --> LATIN SMALL LETTER Z WITH CARON
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN8
elsif (__PACKAGE__ =~ / \b Elatin8 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xA2",     # LATIN LETTER B WITH DOT ABOVE
        "\xA4" => "\xA5",     # LATIN LETTER C WITH DOT ABOVE
        "\xA6" => "\xAB",     # LATIN LETTER D WITH DOT ABOVE
        "\xA8" => "\xB8",     # LATIN LETTER W WITH GRAVE
        "\xAA" => "\xBA",     # LATIN LETTER W WITH ACUTE
        "\xAC" => "\xBC",     # LATIN LETTER Y WITH GRAVE
        "\xAF" => "\xFF",     # LATIN LETTER Y WITH DIAERESIS
        "\xB0" => "\xB1",     # LATIN LETTER F WITH DOT ABOVE
        "\xB2" => "\xB3",     # LATIN LETTER G WITH DOT ABOVE
        "\xB4" => "\xB5",     # LATIN LETTER M WITH DOT ABOVE
        "\xB7" => "\xB9",     # LATIN LETTER P WITH DOT ABOVE
        "\xBB" => "\xBF",     # LATIN LETTER S WITH DOT ABOVE
        "\xBD" => "\xBE",     # LATIN LETTER W WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER W WITH CIRCUMFLEX
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN LETTER T WITH DOT ABOVE
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER Y WITH CIRCUMFLEX
    );

    %uc = (%uc,
        "\xA2" => "\xA1",     # LATIN LETTER B WITH DOT ABOVE
        "\xA5" => "\xA4",     # LATIN LETTER C WITH DOT ABOVE
        "\xAB" => "\xA6",     # LATIN LETTER D WITH DOT ABOVE
        "\xB1" => "\xB0",     # LATIN LETTER F WITH DOT ABOVE
        "\xB3" => "\xB2",     # LATIN LETTER G WITH DOT ABOVE
        "\xB5" => "\xB4",     # LATIN LETTER M WITH DOT ABOVE
        "\xB8" => "\xA8",     # LATIN LETTER W WITH GRAVE
        "\xB9" => "\xB7",     # LATIN LETTER P WITH DOT ABOVE
        "\xBA" => "\xAA",     # LATIN LETTER W WITH ACUTE
        "\xBC" => "\xAC",     # LATIN LETTER Y WITH GRAVE
        "\xBE" => "\xBD",     # LATIN LETTER W WITH DIAERESIS
        "\xBF" => "\xBB",     # LATIN LETTER S WITH DOT ABOVE
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER W WITH CIRCUMFLEX
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF7" => "\xD7",     # LATIN LETTER T WITH DOT ABOVE
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER Y WITH CIRCUMFLEX
        "\xFF" => "\xAF",     # LATIN LETTER Y WITH DIAERESIS
    );

    %fc = (%fc,
        "\xA1" => "\xA2",     # LATIN CAPITAL LETTER B WITH DOT ABOVE  --> LATIN SMALL LETTER B WITH DOT ABOVE
        "\xA4" => "\xA5",     # LATIN CAPITAL LETTER C WITH DOT ABOVE  --> LATIN SMALL LETTER C WITH DOT ABOVE
        "\xA6" => "\xAB",     # LATIN CAPITAL LETTER D WITH DOT ABOVE  --> LATIN SMALL LETTER D WITH DOT ABOVE
        "\xA8" => "\xB8",     # LATIN CAPITAL LETTER W WITH GRAVE      --> LATIN SMALL LETTER W WITH GRAVE
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER W WITH ACUTE      --> LATIN SMALL LETTER W WITH ACUTE
        "\xAC" => "\xBC",     # LATIN CAPITAL LETTER Y WITH GRAVE      --> LATIN SMALL LETTER Y WITH GRAVE
        "\xAF" => "\xFF",     # LATIN CAPITAL LETTER Y WITH DIAERESIS  --> LATIN SMALL LETTER Y WITH DIAERESIS
        "\xB0" => "\xB1",     # LATIN CAPITAL LETTER F WITH DOT ABOVE  --> LATIN SMALL LETTER F WITH DOT ABOVE
        "\xB2" => "\xB3",     # LATIN CAPITAL LETTER G WITH DOT ABOVE  --> LATIN SMALL LETTER G WITH DOT ABOVE
        "\xB4" => "\xB5",     # LATIN CAPITAL LETTER M WITH DOT ABOVE  --> LATIN SMALL LETTER M WITH DOT ABOVE
        "\xB7" => "\xB9",     # LATIN CAPITAL LETTER P WITH DOT ABOVE  --> LATIN SMALL LETTER P WITH DOT ABOVE
        "\xBB" => "\xBF",     # LATIN CAPITAL LETTER S WITH DOT ABOVE  --> LATIN SMALL LETTER S WITH DOT ABOVE
        "\xBD" => "\xBE",     # LATIN CAPITAL LETTER W WITH DIAERESIS  --> LATIN SMALL LETTER W WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER W WITH CIRCUMFLEX --> LATIN SMALL LETTER W WITH CIRCUMFLEX
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN CAPITAL LETTER T WITH DOT ABOVE  --> LATIN SMALL LETTER T WITH DOT ABOVE
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE      --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER Y WITH CIRCUMFLEX --> LATIN SMALL LETTER Y WITH CIRCUMFLEX
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN9
elsif (__PACKAGE__ =~ / \b Elatin9 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA6" => "\xA8",     # LATIN LETTER S WITH CARON
        "\xB4" => "\xB8",     # LATIN LETTER Z WITH CARON
        "\xBC" => "\xBD",     # LATIN LIGATURE OE
        "\xBE" => "\xFF",     # LATIN LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER ETH
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER THORN
    );

    %uc = (%uc,
        "\xA8" => "\xA6",     # LATIN LETTER S WITH CARON
        "\xB8" => "\xB4",     # LATIN LETTER Z WITH CARON
        "\xBD" => "\xBC",     # LATIN LIGATURE OE
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER ETH
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER THORN
        "\xFF" => "\xBE",     # LATIN LETTER Y WITH DIAERESIS
    );

    %fc = (%fc,
        "\xA6" => "\xA8",     # LATIN CAPITAL LETTER S WITH CARON      --> LATIN SMALL LETTER S WITH CARON
        "\xB4" => "\xB8",     # LATIN CAPITAL LETTER Z WITH CARON      --> LATIN SMALL LETTER Z WITH CARON
        "\xBC" => "\xBD",     # LATIN CAPITAL LIGATURE OE              --> LATIN SMALL LIGATURE OE
        "\xBE" => "\xFF",     # LATIN CAPITAL LETTER Y WITH DIAERESIS  --> LATIN SMALL LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER ETH               --> LATIN SMALL LETTER ETH
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE      --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER THORN             --> LATIN SMALL LETTER THORN
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_LATIN10
elsif (__PACKAGE__ =~ / \b Elatin10 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\xA1" => "\xA2",     # LATIN LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN LETTER L WITH STROKE
        "\xA6" => "\xA8",     # LATIN LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN LETTER S WITH COMMA BELOW
        "\xAC" => "\xAE",     # LATIN LETTER Z WITH ACUTE
        "\xAF" => "\xBF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xB2" => "\xB9",     # LATIN LETTER C WITH CARON
        "\xB4" => "\xB8",     # LATIN LETTER Z WITH CARON
        "\xBC" => "\xBD",     # LATIN LIGATURE OE
        "\xBE" => "\xFF",     # LATIN LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER C WITH ACUTE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH DOUBLE ACUTE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN LETTER S WITH ACUTE
        "\xD8" => "\xF8",     # LATIN LETTER U WITH DOUBLE ACUTE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER E WITH OGONEK
        "\xDE" => "\xFE",     # LATIN LETTER T WITH COMMA BELOW
    );

    %uc = (%uc,
        "\xA2" => "\xA1",     # LATIN LETTER A WITH OGONEK
        "\xA8" => "\xA6",     # LATIN LETTER S WITH CARON
        "\xAE" => "\xAC",     # LATIN LETTER Z WITH ACUTE
        "\xB3" => "\xA3",     # LATIN LETTER L WITH STROKE
        "\xB8" => "\xB4",     # LATIN LETTER Z WITH CARON
        "\xB9" => "\xB2",     # LATIN LETTER C WITH CARON
        "\xBA" => "\xAA",     # LATIN LETTER S WITH COMMA BELOW
        "\xBD" => "\xBC",     # LATIN LIGATURE OE
        "\xBF" => "\xAF",     # LATIN LETTER Z WITH DOT ABOVE
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH BREVE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER C WITH ACUTE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER D WITH STROKE
        "\xF1" => "\xD1",     # LATIN LETTER N WITH ACUTE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH DOUBLE ACUTE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF7" => "\xD7",     # LATIN LETTER S WITH ACUTE
        "\xF8" => "\xD8",     # LATIN LETTER U WITH DOUBLE ACUTE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER E WITH OGONEK
        "\xFE" => "\xDE",     # LATIN LETTER T WITH COMMA BELOW
        "\xFF" => "\xBE",     # LATIN LETTER Y WITH DIAERESIS
    );

    %fc = (%fc,
        "\xA1" => "\xA2",     # LATIN CAPITAL LETTER A WITH OGONEK       --> LATIN SMALL LETTER A WITH OGONEK
        "\xA3" => "\xB3",     # LATIN CAPITAL LETTER L WITH STROKE       --> LATIN SMALL LETTER L WITH STROKE
        "\xA6" => "\xA8",     # LATIN CAPITAL LETTER S WITH CARON        --> LATIN SMALL LETTER S WITH CARON
        "\xAA" => "\xBA",     # LATIN CAPITAL LETTER S WITH COMMA BELOW  --> LATIN SMALL LETTER S WITH COMMA BELOW
        "\xAC" => "\xAE",     # LATIN CAPITAL LETTER Z WITH ACUTE        --> LATIN SMALL LETTER Z WITH ACUTE
        "\xAF" => "\xBF",     # LATIN CAPITAL LETTER Z WITH DOT ABOVE    --> LATIN SMALL LETTER Z WITH DOT ABOVE
        "\xB2" => "\xB9",     # LATIN CAPITAL LETTER C WITH CARON        --> LATIN SMALL LETTER C WITH CARON
        "\xB4" => "\xB8",     # LATIN CAPITAL LETTER Z WITH CARON        --> LATIN SMALL LETTER Z WITH CARON
        "\xBC" => "\xBD",     # LATIN CAPITAL LIGATURE OE                --> LATIN SMALL LIGATURE OE
        "\xBE" => "\xFF",     # LATIN CAPITAL LETTER Y WITH DIAERESIS    --> LATIN SMALL LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE        --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE        --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX   --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH BREVE        --> LATIN SMALL LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS    --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER C WITH ACUTE        --> LATIN SMALL LETTER C WITH ACUTE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                  --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA      --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE        --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE        --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX   --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS    --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE        --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE        --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX   --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS    --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER D WITH STROKE       --> LATIN SMALL LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH ACUTE        --> LATIN SMALL LETTER N WITH ACUTE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE        --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE        --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX   --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH DOUBLE ACUTE --> LATIN SMALL LETTER O WITH DOUBLE ACUTE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS    --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD7" => "\xF7",     # LATIN CAPITAL LETTER S WITH ACUTE        --> LATIN SMALL LETTER S WITH ACUTE
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER U WITH DOUBLE ACUTE --> LATIN SMALL LETTER U WITH DOUBLE ACUTE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE        --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE        --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX   --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS    --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER E WITH OGONEK       --> LATIN SMALL LETTER E WITH OGONEK
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER T WITH COMMA BELOW  --> LATIN SMALL LETTER T WITH COMMA BELOW
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S               --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_WINDOWS1252
elsif (__PACKAGE__ =~ / \b Ewindows1252 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\x8A" => "\x9A",     # LATIN LETTER S WITH CARON
        "\x8C" => "\x9C",     # LATIN LIGATURE OE
        "\x8E" => "\x9E",     # LATIN LETTER Z WITH CARON
        "\x9F" => "\xFF",     # LATIN LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER ETH
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN LETTER THORN
    );

    %uc = (%uc,
        "\x9A" => "\x8A",     # LATIN LETTER S WITH CARON
        "\x9C" => "\x8C",     # LATIN LIGATURE OE
        "\x9E" => "\x8E",     # LATIN LETTER Z WITH CARON
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH TILDE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xEC" => "\xCC",     # LATIN LETTER I WITH GRAVE
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER ETH
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF2" => "\xD2",     # LATIN LETTER O WITH GRAVE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH TILDE
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER Y WITH ACUTE
        "\xFE" => "\xDE",     # LATIN LETTER THORN
        "\xFF" => "\x9F",     # LATIN LETTER Y WITH DIAERESIS
    );

    %fc = (%fc,
        "\x8A" => "\x9A",     # LATIN CAPITAL LETTER S WITH CARON      --> LATIN SMALL LETTER S WITH CARON
        "\x8C" => "\x9C",     # LATIN CAPITAL LIGATURE OE              --> LATIN SMALL LIGATURE OE
        "\x8E" => "\x9E",     # LATIN CAPITAL LETTER Z WITH CARON      --> LATIN SMALL LETTER Z WITH CARON
        "\x9F" => "\xFF",     # LATIN CAPITAL LETTER Y WITH DIAERESIS  --> LATIN SMALL LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH TILDE      --> LATIN SMALL LETTER A WITH TILDE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCC" => "\xEC",     # LATIN CAPITAL LETTER I WITH GRAVE      --> LATIN SMALL LETTER I WITH GRAVE
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER ETH               --> LATIN SMALL LETTER ETH
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD2" => "\xF2",     # LATIN CAPITAL LETTER O WITH GRAVE      --> LATIN SMALL LETTER O WITH GRAVE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH TILDE      --> LATIN SMALL LETTER O WITH TILDE
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER Y WITH ACUTE      --> LATIN SMALL LETTER Y WITH ACUTE
        "\xDE" => "\xFE",     # LATIN CAPITAL LETTER THORN             --> LATIN SMALL LETTER THORN
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_WINDOWS1258
elsif (__PACKAGE__ =~ / \b Ewindows1258 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xFF],
             ],
    );

    %lc = (%lc,
        "\x8C" => "\x9C",     # LATIN LIGATURE OE
        "\x9F" => "\xFF",     # LATIN LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN LETTER AE
        "\xC7" => "\xE7",     # LATIN LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN LETTER E WITH DIAERESIS
        "\xCD" => "\xED",     # LATIN LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN LETTER N WITH TILDE
        "\xD3" => "\xF3",     # LATIN LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN LETTER O WITH HORN
        "\xD6" => "\xF6",     # LATIN LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN LETTER U WITH HORN
    );

    %uc = (%uc,
        "\x9C" => "\x8C",     # LATIN LIGATURE OE
        "\xFF" => "\x9F",     # LATIN LETTER Y WITH DIAERESIS
        "\xE0" => "\xC0",     # LATIN LETTER A WITH GRAVE
        "\xE1" => "\xC1",     # LATIN LETTER A WITH ACUTE
        "\xE2" => "\xC2",     # LATIN LETTER A WITH CIRCUMFLEX
        "\xE3" => "\xC3",     # LATIN LETTER A WITH BREVE
        "\xE4" => "\xC4",     # LATIN LETTER A WITH DIAERESIS
        "\xE5" => "\xC5",     # LATIN LETTER A WITH RING ABOVE
        "\xE6" => "\xC6",     # LATIN LETTER AE
        "\xE7" => "\xC7",     # LATIN LETTER C WITH CEDILLA
        "\xE8" => "\xC8",     # LATIN LETTER E WITH GRAVE
        "\xE9" => "\xC9",     # LATIN LETTER E WITH ACUTE
        "\xEA" => "\xCA",     # LATIN LETTER E WITH CIRCUMFLEX
        "\xEB" => "\xCB",     # LATIN LETTER E WITH DIAERESIS
        "\xED" => "\xCD",     # LATIN LETTER I WITH ACUTE
        "\xEE" => "\xCE",     # LATIN LETTER I WITH CIRCUMFLEX
        "\xEF" => "\xCF",     # LATIN LETTER I WITH DIAERESIS
        "\xF0" => "\xD0",     # LATIN LETTER D WITH STROKE
        "\xF1" => "\xD1",     # LATIN LETTER N WITH TILDE
        "\xF3" => "\xD3",     # LATIN LETTER O WITH ACUTE
        "\xF4" => "\xD4",     # LATIN LETTER O WITH CIRCUMFLEX
        "\xF5" => "\xD5",     # LATIN LETTER O WITH HORN
        "\xF6" => "\xD6",     # LATIN LETTER O WITH DIAERESIS
        "\xF8" => "\xD8",     # LATIN LETTER O WITH STROKE
        "\xF9" => "\xD9",     # LATIN LETTER U WITH GRAVE
        "\xFA" => "\xDA",     # LATIN LETTER U WITH ACUTE
        "\xFB" => "\xDB",     # LATIN LETTER U WITH CIRCUMFLEX
        "\xFC" => "\xDC",     # LATIN LETTER U WITH DIAERESIS
        "\xFD" => "\xDD",     # LATIN LETTER U WITH HORN
    );

    %fc = (%fc,
        "\x8C" => "\x9C",     # LATIN CAPITAL LIGATURE OE --> LATIN SMALL LIGATURE OE
        "\x9F" => "\xFF",     # LATIN CAPITAL LETTER Y WITH DIAERESIS  --> LATIN SMALL LETTER Y WITH DIAERESIS
        "\xC0" => "\xE0",     # LATIN CAPITAL LETTER A WITH GRAVE      --> LATIN SMALL LETTER A WITH GRAVE
        "\xC1" => "\xE1",     # LATIN CAPITAL LETTER A WITH ACUTE      --> LATIN SMALL LETTER A WITH ACUTE
        "\xC2" => "\xE2",     # LATIN CAPITAL LETTER A WITH CIRCUMFLEX --> LATIN SMALL LETTER A WITH CIRCUMFLEX
        "\xC3" => "\xE3",     # LATIN CAPITAL LETTER A WITH BREVE      --> LATIN SMALL LETTER A WITH BREVE
        "\xC4" => "\xE4",     # LATIN CAPITAL LETTER A WITH DIAERESIS  --> LATIN SMALL LETTER A WITH DIAERESIS
        "\xC5" => "\xE5",     # LATIN CAPITAL LETTER A WITH RING ABOVE --> LATIN SMALL LETTER A WITH RING ABOVE
        "\xC6" => "\xE6",     # LATIN CAPITAL LETTER AE                --> LATIN SMALL LETTER AE
        "\xC7" => "\xE7",     # LATIN CAPITAL LETTER C WITH CEDILLA    --> LATIN SMALL LETTER C WITH CEDILLA
        "\xC8" => "\xE8",     # LATIN CAPITAL LETTER E WITH GRAVE      --> LATIN SMALL LETTER E WITH GRAVE
        "\xC9" => "\xE9",     # LATIN CAPITAL LETTER E WITH ACUTE      --> LATIN SMALL LETTER E WITH ACUTE
        "\xCA" => "\xEA",     # LATIN CAPITAL LETTER E WITH CIRCUMFLEX --> LATIN SMALL LETTER E WITH CIRCUMFLEX
        "\xCB" => "\xEB",     # LATIN CAPITAL LETTER E WITH DIAERESIS  --> LATIN SMALL LETTER E WITH DIAERESIS
        "\xCD" => "\xED",     # LATIN CAPITAL LETTER I WITH ACUTE      --> LATIN SMALL LETTER I WITH ACUTE
        "\xCE" => "\xEE",     # LATIN CAPITAL LETTER I WITH CIRCUMFLEX --> LATIN SMALL LETTER I WITH CIRCUMFLEX
        "\xCF" => "\xEF",     # LATIN CAPITAL LETTER I WITH DIAERESIS  --> LATIN SMALL LETTER I WITH DIAERESIS
        "\xD0" => "\xF0",     # LATIN CAPITAL LETTER D WITH STROKE     --> LATIN SMALL LETTER D WITH STROKE
        "\xD1" => "\xF1",     # LATIN CAPITAL LETTER N WITH TILDE      --> LATIN SMALL LETTER N WITH TILDE
        "\xD3" => "\xF3",     # LATIN CAPITAL LETTER O WITH ACUTE      --> LATIN SMALL LETTER O WITH ACUTE
        "\xD4" => "\xF4",     # LATIN CAPITAL LETTER O WITH CIRCUMFLEX --> LATIN SMALL LETTER O WITH CIRCUMFLEX
        "\xD5" => "\xF5",     # LATIN CAPITAL LETTER O WITH HORN       --> LATIN SMALL LETTER O WITH HORN
        "\xD6" => "\xF6",     # LATIN CAPITAL LETTER O WITH DIAERESIS  --> LATIN SMALL LETTER O WITH DIAERESIS
        "\xD8" => "\xF8",     # LATIN CAPITAL LETTER O WITH STROKE     --> LATIN SMALL LETTER O WITH STROKE
        "\xD9" => "\xF9",     # LATIN CAPITAL LETTER U WITH GRAVE      --> LATIN SMALL LETTER U WITH GRAVE
        "\xDA" => "\xFA",     # LATIN CAPITAL LETTER U WITH ACUTE      --> LATIN SMALL LETTER U WITH ACUTE
        "\xDB" => "\xFB",     # LATIN CAPITAL LETTER U WITH CIRCUMFLEX --> LATIN SMALL LETTER U WITH CIRCUMFLEX
        "\xDC" => "\xFC",     # LATIN CAPITAL LETTER U WITH DIAERESIS  --> LATIN SMALL LETTER U WITH DIAERESIS
        "\xDD" => "\xFD",     # LATIN CAPITAL LETTER U WITH HORN       --> LATIN SMALL LETTER U WITH HORN
        "\xDF" => "\x73\x73", # LATIN SMALL LETTER SHARP S             --> LATIN SMALL LETTER S, LATIN SMALL LETTER S
    );
}

#endif
#if DIST_EUCJP
elsif (__PACKAGE__ =~ / \b Eeucjp \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x8D],
               [0x90..0xA0],
               [0xFF..0xFF],
             ],
        2 => [ [0x8E..0x8E],[0xA1..0xFE],
               [0xA1..0xFE],[0xA1..0xFE],
             ],
        3 => [ [0x8F..0x8F],[0xA1..0xFE],[0xA1..0xFE],
             ],
    );
}

#endif
#if DIST_EUCTW
elsif (__PACKAGE__ =~ / \b Eeuctw \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x8D],
               [0x8F..0xA0],
               [0xFF..0xFF],
             ],
        2 => [ [0xA1..0xFE],[0xA1..0xFE],
             ],
        4 => [ [0x8E..0x8E],[0xA1..0xB0],[0xA1..0xFE],[0xA1..0xFE],
             ],
    );
}

#endif
#if DIST_SDECKANJI
elsif (__PACKAGE__ =~ / \b Esdeckanji \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x8D],
               [0x90..0xA0],
               [0xFF..0xFF],
             ],
        2 => [ [0x8E..0x8E],[0xA1..0xFE],
               [0xA1..0xFE],[0xA1..0xFE],
               [0xA1..0xFE],[0x21..0x7E],
             ],
        3 => [ [0x8F..0x8F],[0xA1..0xFE],[0xA1..0xFE],
             ],
    );
}

#endif
#if DIST_HP16
elsif (__PACKAGE__ =~ / \b Ehp16 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0xA0],
               [0xFF..0xFF],
             ],
        2 => [ [0xA1..0xFE],[0xA1..0xFE],
               [0xA1..0xC2],[0x21..0x7E],
               [0xC3..0xE3],[0x21..0x3F],
               [0xC3..0xE1],[0x40..0x64],
             ],
    );
}

#endif
#if DIST_UTF2
elsif (__PACKAGE__ =~ / \b Eutf2 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x7F],
               [0xF5..0xFF], # malformed octet
             ],
        2 => [ [0xC2..0xDF],[0x80..0xBF],
             ],
        3 => [ [0xE0..0xE0],[0xA0..0xBF],[0x80..0xBF],
               [0xE1..0xEC],[0x80..0xBF],[0x80..0xBF],
               [0xED..0xED],[0x80..0x9F],[0x80..0xBF],
               [0xEE..0xEF],[0x80..0xBF],[0x80..0xBF],
             ],
        4 => [ [0xF0..0xF0],[0x90..0xBF],[0x80..0xBF],[0x80..0xBF],
               [0xF1..0xF3],[0x80..0xBF],[0x80..0xBF],[0x80..0xBF],
               [0xF4..0xF4],[0x80..0x8F],[0x80..0xBF],[0x80..0xBF],
             ],
    );

    # CaseFolding-12.0.0.txt
    # Date: 2019-01-22, 08:18:22 GMT
    # c 2019 UnicodeR, Inc.
    # Unicode and the Unicode Logo are registered trademarks of Unicode, Inc. in the U.S. and other countries.
    # For terms of use, see http://www.unicode.org/terms_of_use.html
    #
    # Unicode Character Database
    #   For documentation, see http://www.unicode.org/reports/tr44/

    # you can use "make_CaseFolding.pl" to update this hash

    %fc = (
        "\x41"             => "\x61",                     # LATIN CAPITAL LETTER A
        "\x42"             => "\x62",                     # LATIN CAPITAL LETTER B
        "\x43"             => "\x63",                     # LATIN CAPITAL LETTER C
        "\x44"             => "\x64",                     # LATIN CAPITAL LETTER D
        "\x45"             => "\x65",                     # LATIN CAPITAL LETTER E
        "\x46"             => "\x66",                     # LATIN CAPITAL LETTER F
        "\x47"             => "\x67",                     # LATIN CAPITAL LETTER G
        "\x48"             => "\x68",                     # LATIN CAPITAL LETTER H
        "\x49"             => "\x69",                     # LATIN CAPITAL LETTER I
        "\x4A"             => "\x6A",                     # LATIN CAPITAL LETTER J
        "\x4B"             => "\x6B",                     # LATIN CAPITAL LETTER K
        "\x4C"             => "\x6C",                     # LATIN CAPITAL LETTER L
        "\x4D"             => "\x6D",                     # LATIN CAPITAL LETTER M
        "\x4E"             => "\x6E",                     # LATIN CAPITAL LETTER N
        "\x4F"             => "\x6F",                     # LATIN CAPITAL LETTER O
        "\x50"             => "\x70",                     # LATIN CAPITAL LETTER P
        "\x51"             => "\x71",                     # LATIN CAPITAL LETTER Q
        "\x52"             => "\x72",                     # LATIN CAPITAL LETTER R
        "\x53"             => "\x73",                     # LATIN CAPITAL LETTER S
        "\x54"             => "\x74",                     # LATIN CAPITAL LETTER T
        "\x55"             => "\x75",                     # LATIN CAPITAL LETTER U
        "\x56"             => "\x76",                     # LATIN CAPITAL LETTER V
        "\x57"             => "\x77",                     # LATIN CAPITAL LETTER W
        "\x58"             => "\x78",                     # LATIN CAPITAL LETTER X
        "\x59"             => "\x79",                     # LATIN CAPITAL LETTER Y
        "\x5A"             => "\x7A",                     # LATIN CAPITAL LETTER Z
        "\xC2\xB5"         => "\xCE\xBC",                 # MICRO SIGN
        "\xC3\x80"         => "\xC3\xA0",                 # LATIN CAPITAL LETTER A WITH GRAVE
        "\xC3\x81"         => "\xC3\xA1",                 # LATIN CAPITAL LETTER A WITH ACUTE
        "\xC3\x82"         => "\xC3\xA2",                 # LATIN CAPITAL LETTER A WITH CIRCUMFLEX
        "\xC3\x83"         => "\xC3\xA3",                 # LATIN CAPITAL LETTER A WITH TILDE
        "\xC3\x84"         => "\xC3\xA4",                 # LATIN CAPITAL LETTER A WITH DIAERESIS
        "\xC3\x85"         => "\xC3\xA5",                 # LATIN CAPITAL LETTER A WITH RING ABOVE
        "\xC3\x86"         => "\xC3\xA6",                 # LATIN CAPITAL LETTER AE
        "\xC3\x87"         => "\xC3\xA7",                 # LATIN CAPITAL LETTER C WITH CEDILLA
        "\xC3\x88"         => "\xC3\xA8",                 # LATIN CAPITAL LETTER E WITH GRAVE
        "\xC3\x89"         => "\xC3\xA9",                 # LATIN CAPITAL LETTER E WITH ACUTE
        "\xC3\x8A"         => "\xC3\xAA",                 # LATIN CAPITAL LETTER E WITH CIRCUMFLEX
        "\xC3\x8B"         => "\xC3\xAB",                 # LATIN CAPITAL LETTER E WITH DIAERESIS
        "\xC3\x8C"         => "\xC3\xAC",                 # LATIN CAPITAL LETTER I WITH GRAVE
        "\xC3\x8D"         => "\xC3\xAD",                 # LATIN CAPITAL LETTER I WITH ACUTE
        "\xC3\x8E"         => "\xC3\xAE",                 # LATIN CAPITAL LETTER I WITH CIRCUMFLEX
        "\xC3\x8F"         => "\xC3\xAF",                 # LATIN CAPITAL LETTER I WITH DIAERESIS
        "\xC3\x90"         => "\xC3\xB0",                 # LATIN CAPITAL LETTER ETH
        "\xC3\x91"         => "\xC3\xB1",                 # LATIN CAPITAL LETTER N WITH TILDE
        "\xC3\x92"         => "\xC3\xB2",                 # LATIN CAPITAL LETTER O WITH GRAVE
        "\xC3\x93"         => "\xC3\xB3",                 # LATIN CAPITAL LETTER O WITH ACUTE
        "\xC3\x94"         => "\xC3\xB4",                 # LATIN CAPITAL LETTER O WITH CIRCUMFLEX
        "\xC3\x95"         => "\xC3\xB5",                 # LATIN CAPITAL LETTER O WITH TILDE
        "\xC3\x96"         => "\xC3\xB6",                 # LATIN CAPITAL LETTER O WITH DIAERESIS
        "\xC3\x98"         => "\xC3\xB8",                 # LATIN CAPITAL LETTER O WITH STROKE
        "\xC3\x99"         => "\xC3\xB9",                 # LATIN CAPITAL LETTER U WITH GRAVE
        "\xC3\x9A"         => "\xC3\xBA",                 # LATIN CAPITAL LETTER U WITH ACUTE
        "\xC3\x9B"         => "\xC3\xBB",                 # LATIN CAPITAL LETTER U WITH CIRCUMFLEX
        "\xC3\x9C"         => "\xC3\xBC",                 # LATIN CAPITAL LETTER U WITH DIAERESIS
        "\xC3\x9D"         => "\xC3\xBD",                 # LATIN CAPITAL LETTER Y WITH ACUTE
        "\xC3\x9E"         => "\xC3\xBE",                 # LATIN CAPITAL LETTER THORN
        "\xC3\x9F"         => "\x73\x73",                 # LATIN SMALL LETTER SHARP S
        "\xC4\x80"         => "\xC4\x81",                 # LATIN CAPITAL LETTER A WITH MACRON
        "\xC4\x82"         => "\xC4\x83",                 # LATIN CAPITAL LETTER A WITH BREVE
        "\xC4\x84"         => "\xC4\x85",                 # LATIN CAPITAL LETTER A WITH OGONEK
        "\xC4\x86"         => "\xC4\x87",                 # LATIN CAPITAL LETTER C WITH ACUTE
        "\xC4\x88"         => "\xC4\x89",                 # LATIN CAPITAL LETTER C WITH CIRCUMFLEX
        "\xC4\x8A"         => "\xC4\x8B",                 # LATIN CAPITAL LETTER C WITH DOT ABOVE
        "\xC4\x8C"         => "\xC4\x8D",                 # LATIN CAPITAL LETTER C WITH CARON
        "\xC4\x8E"         => "\xC4\x8F",                 # LATIN CAPITAL LETTER D WITH CARON
        "\xC4\x90"         => "\xC4\x91",                 # LATIN CAPITAL LETTER D WITH STROKE
        "\xC4\x92"         => "\xC4\x93",                 # LATIN CAPITAL LETTER E WITH MACRON
        "\xC4\x94"         => "\xC4\x95",                 # LATIN CAPITAL LETTER E WITH BREVE
        "\xC4\x96"         => "\xC4\x97",                 # LATIN CAPITAL LETTER E WITH DOT ABOVE
        "\xC4\x98"         => "\xC4\x99",                 # LATIN CAPITAL LETTER E WITH OGONEK
        "\xC4\x9A"         => "\xC4\x9B",                 # LATIN CAPITAL LETTER E WITH CARON
        "\xC4\x9C"         => "\xC4\x9D",                 # LATIN CAPITAL LETTER G WITH CIRCUMFLEX
        "\xC4\x9E"         => "\xC4\x9F",                 # LATIN CAPITAL LETTER G WITH BREVE
        "\xC4\xA0"         => "\xC4\xA1",                 # LATIN CAPITAL LETTER G WITH DOT ABOVE
        "\xC4\xA2"         => "\xC4\xA3",                 # LATIN CAPITAL LETTER G WITH CEDILLA
        "\xC4\xA4"         => "\xC4\xA5",                 # LATIN CAPITAL LETTER H WITH CIRCUMFLEX
        "\xC4\xA6"         => "\xC4\xA7",                 # LATIN CAPITAL LETTER H WITH STROKE
        "\xC4\xA8"         => "\xC4\xA9",                 # LATIN CAPITAL LETTER I WITH TILDE
        "\xC4\xAA"         => "\xC4\xAB",                 # LATIN CAPITAL LETTER I WITH MACRON
        "\xC4\xAC"         => "\xC4\xAD",                 # LATIN CAPITAL LETTER I WITH BREVE
        "\xC4\xAE"         => "\xC4\xAF",                 # LATIN CAPITAL LETTER I WITH OGONEK
        "\xC4\xB0"         => "\x69\xCC\x87",             # LATIN CAPITAL LETTER I WITH DOT ABOVE
        "\xC4\xB2"         => "\xC4\xB3",                 # LATIN CAPITAL LIGATURE IJ
        "\xC4\xB4"         => "\xC4\xB5",                 # LATIN CAPITAL LETTER J WITH CIRCUMFLEX
        "\xC4\xB6"         => "\xC4\xB7",                 # LATIN CAPITAL LETTER K WITH CEDILLA
        "\xC4\xB9"         => "\xC4\xBA",                 # LATIN CAPITAL LETTER L WITH ACUTE
        "\xC4\xBB"         => "\xC4\xBC",                 # LATIN CAPITAL LETTER L WITH CEDILLA
        "\xC4\xBD"         => "\xC4\xBE",                 # LATIN CAPITAL LETTER L WITH CARON
        "\xC4\xBF"         => "\xC5\x80",                 # LATIN CAPITAL LETTER L WITH MIDDLE DOT
        "\xC5\x81"         => "\xC5\x82",                 # LATIN CAPITAL LETTER L WITH STROKE
        "\xC5\x83"         => "\xC5\x84",                 # LATIN CAPITAL LETTER N WITH ACUTE
        "\xC5\x85"         => "\xC5\x86",                 # LATIN CAPITAL LETTER N WITH CEDILLA
        "\xC5\x87"         => "\xC5\x88",                 # LATIN CAPITAL LETTER N WITH CARON
        "\xC5\x89"         => "\xCA\xBC\x6E",             # LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        "\xC5\x8A"         => "\xC5\x8B",                 # LATIN CAPITAL LETTER ENG
        "\xC5\x8C"         => "\xC5\x8D",                 # LATIN CAPITAL LETTER O WITH MACRON
        "\xC5\x8E"         => "\xC5\x8F",                 # LATIN CAPITAL LETTER O WITH BREVE
        "\xC5\x90"         => "\xC5\x91",                 # LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
        "\xC5\x92"         => "\xC5\x93",                 # LATIN CAPITAL LIGATURE OE
        "\xC5\x94"         => "\xC5\x95",                 # LATIN CAPITAL LETTER R WITH ACUTE
        "\xC5\x96"         => "\xC5\x97",                 # LATIN CAPITAL LETTER R WITH CEDILLA
        "\xC5\x98"         => "\xC5\x99",                 # LATIN CAPITAL LETTER R WITH CARON
        "\xC5\x9A"         => "\xC5\x9B",                 # LATIN CAPITAL LETTER S WITH ACUTE
        "\xC5\x9C"         => "\xC5\x9D",                 # LATIN CAPITAL LETTER S WITH CIRCUMFLEX
        "\xC5\x9E"         => "\xC5\x9F",                 # LATIN CAPITAL LETTER S WITH CEDILLA
        "\xC5\xA0"         => "\xC5\xA1",                 # LATIN CAPITAL LETTER S WITH CARON
        "\xC5\xA2"         => "\xC5\xA3",                 # LATIN CAPITAL LETTER T WITH CEDILLA
        "\xC5\xA4"         => "\xC5\xA5",                 # LATIN CAPITAL LETTER T WITH CARON
        "\xC5\xA6"         => "\xC5\xA7",                 # LATIN CAPITAL LETTER T WITH STROKE
        "\xC5\xA8"         => "\xC5\xA9",                 # LATIN CAPITAL LETTER U WITH TILDE
        "\xC5\xAA"         => "\xC5\xAB",                 # LATIN CAPITAL LETTER U WITH MACRON
        "\xC5\xAC"         => "\xC5\xAD",                 # LATIN CAPITAL LETTER U WITH BREVE
        "\xC5\xAE"         => "\xC5\xAF",                 # LATIN CAPITAL LETTER U WITH RING ABOVE
        "\xC5\xB0"         => "\xC5\xB1",                 # LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
        "\xC5\xB2"         => "\xC5\xB3",                 # LATIN CAPITAL LETTER U WITH OGONEK
        "\xC5\xB4"         => "\xC5\xB5",                 # LATIN CAPITAL LETTER W WITH CIRCUMFLEX
        "\xC5\xB6"         => "\xC5\xB7",                 # LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
        "\xC5\xB8"         => "\xC3\xBF",                 # LATIN CAPITAL LETTER Y WITH DIAERESIS
        "\xC5\xB9"         => "\xC5\xBA",                 # LATIN CAPITAL LETTER Z WITH ACUTE
        "\xC5\xBB"         => "\xC5\xBC",                 # LATIN CAPITAL LETTER Z WITH DOT ABOVE
        "\xC5\xBD"         => "\xC5\xBE",                 # LATIN CAPITAL LETTER Z WITH CARON
        "\xC5\xBF"         => "\x73",                     # LATIN SMALL LETTER LONG S
        "\xC6\x81"         => "\xC9\x93",                 # LATIN CAPITAL LETTER B WITH HOOK
        "\xC6\x82"         => "\xC6\x83",                 # LATIN CAPITAL LETTER B WITH TOPBAR
        "\xC6\x84"         => "\xC6\x85",                 # LATIN CAPITAL LETTER TONE SIX
        "\xC6\x86"         => "\xC9\x94",                 # LATIN CAPITAL LETTER OPEN O
        "\xC6\x87"         => "\xC6\x88",                 # LATIN CAPITAL LETTER C WITH HOOK
        "\xC6\x89"         => "\xC9\x96",                 # LATIN CAPITAL LETTER AFRICAN D
        "\xC6\x8A"         => "\xC9\x97",                 # LATIN CAPITAL LETTER D WITH HOOK
        "\xC6\x8B"         => "\xC6\x8C",                 # LATIN CAPITAL LETTER D WITH TOPBAR
        "\xC6\x8E"         => "\xC7\x9D",                 # LATIN CAPITAL LETTER REVERSED E
        "\xC6\x8F"         => "\xC9\x99",                 # LATIN CAPITAL LETTER SCHWA
        "\xC6\x90"         => "\xC9\x9B",                 # LATIN CAPITAL LETTER OPEN E
        "\xC6\x91"         => "\xC6\x92",                 # LATIN CAPITAL LETTER F WITH HOOK
        "\xC6\x93"         => "\xC9\xA0",                 # LATIN CAPITAL LETTER G WITH HOOK
        "\xC6\x94"         => "\xC9\xA3",                 # LATIN CAPITAL LETTER GAMMA
        "\xC6\x96"         => "\xC9\xA9",                 # LATIN CAPITAL LETTER IOTA
        "\xC6\x97"         => "\xC9\xA8",                 # LATIN CAPITAL LETTER I WITH STROKE
        "\xC6\x98"         => "\xC6\x99",                 # LATIN CAPITAL LETTER K WITH HOOK
        "\xC6\x9C"         => "\xC9\xAF",                 # LATIN CAPITAL LETTER TURNED M
        "\xC6\x9D"         => "\xC9\xB2",                 # LATIN CAPITAL LETTER N WITH LEFT HOOK
        "\xC6\x9F"         => "\xC9\xB5",                 # LATIN CAPITAL LETTER O WITH MIDDLE TILDE
        "\xC6\xA0"         => "\xC6\xA1",                 # LATIN CAPITAL LETTER O WITH HORN
        "\xC6\xA2"         => "\xC6\xA3",                 # LATIN CAPITAL LETTER OI
        "\xC6\xA4"         => "\xC6\xA5",                 # LATIN CAPITAL LETTER P WITH HOOK
        "\xC6\xA6"         => "\xCA\x80",                 # LATIN LETTER YR
        "\xC6\xA7"         => "\xC6\xA8",                 # LATIN CAPITAL LETTER TONE TWO
        "\xC6\xA9"         => "\xCA\x83",                 # LATIN CAPITAL LETTER ESH
        "\xC6\xAC"         => "\xC6\xAD",                 # LATIN CAPITAL LETTER T WITH HOOK
        "\xC6\xAE"         => "\xCA\x88",                 # LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
        "\xC6\xAF"         => "\xC6\xB0",                 # LATIN CAPITAL LETTER U WITH HORN
        "\xC6\xB1"         => "\xCA\x8A",                 # LATIN CAPITAL LETTER UPSILON
        "\xC6\xB2"         => "\xCA\x8B",                 # LATIN CAPITAL LETTER V WITH HOOK
        "\xC6\xB3"         => "\xC6\xB4",                 # LATIN CAPITAL LETTER Y WITH HOOK
        "\xC6\xB5"         => "\xC6\xB6",                 # LATIN CAPITAL LETTER Z WITH STROKE
        "\xC6\xB7"         => "\xCA\x92",                 # LATIN CAPITAL LETTER EZH
        "\xC6\xB8"         => "\xC6\xB9",                 # LATIN CAPITAL LETTER EZH REVERSED
        "\xC6\xBC"         => "\xC6\xBD",                 # LATIN CAPITAL LETTER TONE FIVE
        "\xC7\x84"         => "\xC7\x86",                 # LATIN CAPITAL LETTER DZ WITH CARON
        "\xC7\x85"         => "\xC7\x86",                 # LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
        "\xC7\x87"         => "\xC7\x89",                 # LATIN CAPITAL LETTER LJ
        "\xC7\x88"         => "\xC7\x89",                 # LATIN CAPITAL LETTER L WITH SMALL LETTER J
        "\xC7\x8A"         => "\xC7\x8C",                 # LATIN CAPITAL LETTER NJ
        "\xC7\x8B"         => "\xC7\x8C",                 # LATIN CAPITAL LETTER N WITH SMALL LETTER J
        "\xC7\x8D"         => "\xC7\x8E",                 # LATIN CAPITAL LETTER A WITH CARON
        "\xC7\x8F"         => "\xC7\x90",                 # LATIN CAPITAL LETTER I WITH CARON
        "\xC7\x91"         => "\xC7\x92",                 # LATIN CAPITAL LETTER O WITH CARON
        "\xC7\x93"         => "\xC7\x94",                 # LATIN CAPITAL LETTER U WITH CARON
        "\xC7\x95"         => "\xC7\x96",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
        "\xC7\x97"         => "\xC7\x98",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
        "\xC7\x99"         => "\xC7\x9A",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
        "\xC7\x9B"         => "\xC7\x9C",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
        "\xC7\x9E"         => "\xC7\x9F",                 # LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
        "\xC7\xA0"         => "\xC7\xA1",                 # LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
        "\xC7\xA2"         => "\xC7\xA3",                 # LATIN CAPITAL LETTER AE WITH MACRON
        "\xC7\xA4"         => "\xC7\xA5",                 # LATIN CAPITAL LETTER G WITH STROKE
        "\xC7\xA6"         => "\xC7\xA7",                 # LATIN CAPITAL LETTER G WITH CARON
        "\xC7\xA8"         => "\xC7\xA9",                 # LATIN CAPITAL LETTER K WITH CARON
        "\xC7\xAA"         => "\xC7\xAB",                 # LATIN CAPITAL LETTER O WITH OGONEK
        "\xC7\xAC"         => "\xC7\xAD",                 # LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
        "\xC7\xAE"         => "\xC7\xAF",                 # LATIN CAPITAL LETTER EZH WITH CARON
        "\xC7\xB0"         => "\x6A\xCC\x8C",             # LATIN SMALL LETTER J WITH CARON
        "\xC7\xB1"         => "\xC7\xB3",                 # LATIN CAPITAL LETTER DZ
        "\xC7\xB2"         => "\xC7\xB3",                 # LATIN CAPITAL LETTER D WITH SMALL LETTER Z
        "\xC7\xB4"         => "\xC7\xB5",                 # LATIN CAPITAL LETTER G WITH ACUTE
        "\xC7\xB6"         => "\xC6\x95",                 # LATIN CAPITAL LETTER HWAIR
        "\xC7\xB7"         => "\xC6\xBF",                 # LATIN CAPITAL LETTER WYNN
        "\xC7\xB8"         => "\xC7\xB9",                 # LATIN CAPITAL LETTER N WITH GRAVE
        "\xC7\xBA"         => "\xC7\xBB",                 # LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
        "\xC7\xBC"         => "\xC7\xBD",                 # LATIN CAPITAL LETTER AE WITH ACUTE
        "\xC7\xBE"         => "\xC7\xBF",                 # LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
        "\xC8\x80"         => "\xC8\x81",                 # LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
        "\xC8\x82"         => "\xC8\x83",                 # LATIN CAPITAL LETTER A WITH INVERTED BREVE
        "\xC8\x84"         => "\xC8\x85",                 # LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
        "\xC8\x86"         => "\xC8\x87",                 # LATIN CAPITAL LETTER E WITH INVERTED BREVE
        "\xC8\x88"         => "\xC8\x89",                 # LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
        "\xC8\x8A"         => "\xC8\x8B",                 # LATIN CAPITAL LETTER I WITH INVERTED BREVE
        "\xC8\x8C"         => "\xC8\x8D",                 # LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
        "\xC8\x8E"         => "\xC8\x8F",                 # LATIN CAPITAL LETTER O WITH INVERTED BREVE
        "\xC8\x90"         => "\xC8\x91",                 # LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
        "\xC8\x92"         => "\xC8\x93",                 # LATIN CAPITAL LETTER R WITH INVERTED BREVE
        "\xC8\x94"         => "\xC8\x95",                 # LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
        "\xC8\x96"         => "\xC8\x97",                 # LATIN CAPITAL LETTER U WITH INVERTED BREVE
        "\xC8\x98"         => "\xC8\x99",                 # LATIN CAPITAL LETTER S WITH COMMA BELOW
        "\xC8\x9A"         => "\xC8\x9B",                 # LATIN CAPITAL LETTER T WITH COMMA BELOW
        "\xC8\x9C"         => "\xC8\x9D",                 # LATIN CAPITAL LETTER YOGH
        "\xC8\x9E"         => "\xC8\x9F",                 # LATIN CAPITAL LETTER H WITH CARON
        "\xC8\xA0"         => "\xC6\x9E",                 # LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
        "\xC8\xA2"         => "\xC8\xA3",                 # LATIN CAPITAL LETTER OU
        "\xC8\xA4"         => "\xC8\xA5",                 # LATIN CAPITAL LETTER Z WITH HOOK
        "\xC8\xA6"         => "\xC8\xA7",                 # LATIN CAPITAL LETTER A WITH DOT ABOVE
        "\xC8\xA8"         => "\xC8\xA9",                 # LATIN CAPITAL LETTER E WITH CEDILLA
        "\xC8\xAA"         => "\xC8\xAB",                 # LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
        "\xC8\xAC"         => "\xC8\xAD",                 # LATIN CAPITAL LETTER O WITH TILDE AND MACRON
        "\xC8\xAE"         => "\xC8\xAF",                 # LATIN CAPITAL LETTER O WITH DOT ABOVE
        "\xC8\xB0"         => "\xC8\xB1",                 # LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
        "\xC8\xB2"         => "\xC8\xB3",                 # LATIN CAPITAL LETTER Y WITH MACRON
        "\xC8\xBA"         => "\xE2\xB1\xA5",             # LATIN CAPITAL LETTER A WITH STROKE
        "\xC8\xBB"         => "\xC8\xBC",                 # LATIN CAPITAL LETTER C WITH STROKE
        "\xC8\xBD"         => "\xC6\x9A",                 # LATIN CAPITAL LETTER L WITH BAR
        "\xC8\xBE"         => "\xE2\xB1\xA6",             # LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
        "\xC9\x81"         => "\xC9\x82",                 # LATIN CAPITAL LETTER GLOTTAL STOP
        "\xC9\x83"         => "\xC6\x80",                 # LATIN CAPITAL LETTER B WITH STROKE
        "\xC9\x84"         => "\xCA\x89",                 # LATIN CAPITAL LETTER U BAR
        "\xC9\x85"         => "\xCA\x8C",                 # LATIN CAPITAL LETTER TURNED V
        "\xC9\x86"         => "\xC9\x87",                 # LATIN CAPITAL LETTER E WITH STROKE
        "\xC9\x88"         => "\xC9\x89",                 # LATIN CAPITAL LETTER J WITH STROKE
        "\xC9\x8A"         => "\xC9\x8B",                 # LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
        "\xC9\x8C"         => "\xC9\x8D",                 # LATIN CAPITAL LETTER R WITH STROKE
        "\xC9\x8E"         => "\xC9\x8F",                 # LATIN CAPITAL LETTER Y WITH STROKE
        "\xCD\x85"         => "\xCE\xB9",                 # COMBINING GREEK YPOGEGRAMMENI
        "\xCD\xB0"         => "\xCD\xB1",                 # GREEK CAPITAL LETTER HETA
        "\xCD\xB2"         => "\xCD\xB3",                 # GREEK CAPITAL LETTER ARCHAIC SAMPI
        "\xCD\xB6"         => "\xCD\xB7",                 # GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
        "\xCD\xBF"         => "\xCF\xB3",                 # GREEK CAPITAL LETTER YOT
        "\xCE\x86"         => "\xCE\xAC",                 # GREEK CAPITAL LETTER ALPHA WITH TONOS
        "\xCE\x88"         => "\xCE\xAD",                 # GREEK CAPITAL LETTER EPSILON WITH TONOS
        "\xCE\x89"         => "\xCE\xAE",                 # GREEK CAPITAL LETTER ETA WITH TONOS
        "\xCE\x8A"         => "\xCE\xAF",                 # GREEK CAPITAL LETTER IOTA WITH TONOS
        "\xCE\x8C"         => "\xCF\x8C",                 # GREEK CAPITAL LETTER OMICRON WITH TONOS
        "\xCE\x8E"         => "\xCF\x8D",                 # GREEK CAPITAL LETTER UPSILON WITH TONOS
        "\xCE\x8F"         => "\xCF\x8E",                 # GREEK CAPITAL LETTER OMEGA WITH TONOS
        "\xCE\x90"         => "\xCE\xB9\xCC\x88\xCC\x81", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
        "\xCE\x91"         => "\xCE\xB1",                 # GREEK CAPITAL LETTER ALPHA
        "\xCE\x92"         => "\xCE\xB2",                 # GREEK CAPITAL LETTER BETA
        "\xCE\x93"         => "\xCE\xB3",                 # GREEK CAPITAL LETTER GAMMA
        "\xCE\x94"         => "\xCE\xB4",                 # GREEK CAPITAL LETTER DELTA
        "\xCE\x95"         => "\xCE\xB5",                 # GREEK CAPITAL LETTER EPSILON
        "\xCE\x96"         => "\xCE\xB6",                 # GREEK CAPITAL LETTER ZETA
        "\xCE\x97"         => "\xCE\xB7",                 # GREEK CAPITAL LETTER ETA
        "\xCE\x98"         => "\xCE\xB8",                 # GREEK CAPITAL LETTER THETA
        "\xCE\x99"         => "\xCE\xB9",                 # GREEK CAPITAL LETTER IOTA
        "\xCE\x9A"         => "\xCE\xBA",                 # GREEK CAPITAL LETTER KAPPA
        "\xCE\x9B"         => "\xCE\xBB",                 # GREEK CAPITAL LETTER LAMDA
        "\xCE\x9C"         => "\xCE\xBC",                 # GREEK CAPITAL LETTER MU
        "\xCE\x9D"         => "\xCE\xBD",                 # GREEK CAPITAL LETTER NU
        "\xCE\x9E"         => "\xCE\xBE",                 # GREEK CAPITAL LETTER XI
        "\xCE\x9F"         => "\xCE\xBF",                 # GREEK CAPITAL LETTER OMICRON
        "\xCE\xA0"         => "\xCF\x80",                 # GREEK CAPITAL LETTER PI
        "\xCE\xA1"         => "\xCF\x81",                 # GREEK CAPITAL LETTER RHO
        "\xCE\xA3"         => "\xCF\x83",                 # GREEK CAPITAL LETTER SIGMA
        "\xCE\xA4"         => "\xCF\x84",                 # GREEK CAPITAL LETTER TAU
        "\xCE\xA5"         => "\xCF\x85",                 # GREEK CAPITAL LETTER UPSILON
        "\xCE\xA6"         => "\xCF\x86",                 # GREEK CAPITAL LETTER PHI
        "\xCE\xA7"         => "\xCF\x87",                 # GREEK CAPITAL LETTER CHI
        "\xCE\xA8"         => "\xCF\x88",                 # GREEK CAPITAL LETTER PSI
        "\xCE\xA9"         => "\xCF\x89",                 # GREEK CAPITAL LETTER OMEGA
        "\xCE\xAA"         => "\xCF\x8A",                 # GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
        "\xCE\xAB"         => "\xCF\x8B",                 # GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
        "\xCE\xB0"         => "\xCF\x85\xCC\x88\xCC\x81", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
        "\xCF\x82"         => "\xCF\x83",                 # GREEK SMALL LETTER FINAL SIGMA
        "\xCF\x8F"         => "\xCF\x97",                 # GREEK CAPITAL KAI SYMBOL
        "\xCF\x90"         => "\xCE\xB2",                 # GREEK BETA SYMBOL
        "\xCF\x91"         => "\xCE\xB8",                 # GREEK THETA SYMBOL
        "\xCF\x95"         => "\xCF\x86",                 # GREEK PHI SYMBOL
        "\xCF\x96"         => "\xCF\x80",                 # GREEK PI SYMBOL
        "\xCF\x98"         => "\xCF\x99",                 # GREEK LETTER ARCHAIC KOPPA
        "\xCF\x9A"         => "\xCF\x9B",                 # GREEK LETTER STIGMA
        "\xCF\x9C"         => "\xCF\x9D",                 # GREEK LETTER DIGAMMA
        "\xCF\x9E"         => "\xCF\x9F",                 # GREEK LETTER KOPPA
        "\xCF\xA0"         => "\xCF\xA1",                 # GREEK LETTER SAMPI
        "\xCF\xA2"         => "\xCF\xA3",                 # COPTIC CAPITAL LETTER SHEI
        "\xCF\xA4"         => "\xCF\xA5",                 # COPTIC CAPITAL LETTER FEI
        "\xCF\xA6"         => "\xCF\xA7",                 # COPTIC CAPITAL LETTER KHEI
        "\xCF\xA8"         => "\xCF\xA9",                 # COPTIC CAPITAL LETTER HORI
        "\xCF\xAA"         => "\xCF\xAB",                 # COPTIC CAPITAL LETTER GANGIA
        "\xCF\xAC"         => "\xCF\xAD",                 # COPTIC CAPITAL LETTER SHIMA
        "\xCF\xAE"         => "\xCF\xAF",                 # COPTIC CAPITAL LETTER DEI
        "\xCF\xB0"         => "\xCE\xBA",                 # GREEK KAPPA SYMBOL
        "\xCF\xB1"         => "\xCF\x81",                 # GREEK RHO SYMBOL
        "\xCF\xB4"         => "\xCE\xB8",                 # GREEK CAPITAL THETA SYMBOL
        "\xCF\xB5"         => "\xCE\xB5",                 # GREEK LUNATE EPSILON SYMBOL
        "\xCF\xB7"         => "\xCF\xB8",                 # GREEK CAPITAL LETTER SHO
        "\xCF\xB9"         => "\xCF\xB2",                 # GREEK CAPITAL LUNATE SIGMA SYMBOL
        "\xCF\xBA"         => "\xCF\xBB",                 # GREEK CAPITAL LETTER SAN
        "\xCF\xBD"         => "\xCD\xBB",                 # GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
        "\xCF\xBE"         => "\xCD\xBC",                 # GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
        "\xCF\xBF"         => "\xCD\xBD",                 # GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
        "\xD0\x80"         => "\xD1\x90",                 # CYRILLIC CAPITAL LETTER IE WITH GRAVE
        "\xD0\x81"         => "\xD1\x91",                 # CYRILLIC CAPITAL LETTER IO
        "\xD0\x82"         => "\xD1\x92",                 # CYRILLIC CAPITAL LETTER DJE
        "\xD0\x83"         => "\xD1\x93",                 # CYRILLIC CAPITAL LETTER GJE
        "\xD0\x84"         => "\xD1\x94",                 # CYRILLIC CAPITAL LETTER UKRAINIAN IE
        "\xD0\x85"         => "\xD1\x95",                 # CYRILLIC CAPITAL LETTER DZE
        "\xD0\x86"         => "\xD1\x96",                 # CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
        "\xD0\x87"         => "\xD1\x97",                 # CYRILLIC CAPITAL LETTER YI
        "\xD0\x88"         => "\xD1\x98",                 # CYRILLIC CAPITAL LETTER JE
        "\xD0\x89"         => "\xD1\x99",                 # CYRILLIC CAPITAL LETTER LJE
        "\xD0\x8A"         => "\xD1\x9A",                 # CYRILLIC CAPITAL LETTER NJE
        "\xD0\x8B"         => "\xD1\x9B",                 # CYRILLIC CAPITAL LETTER TSHE
        "\xD0\x8C"         => "\xD1\x9C",                 # CYRILLIC CAPITAL LETTER KJE
        "\xD0\x8D"         => "\xD1\x9D",                 # CYRILLIC CAPITAL LETTER I WITH GRAVE
        "\xD0\x8E"         => "\xD1\x9E",                 # CYRILLIC CAPITAL LETTER SHORT U
        "\xD0\x8F"         => "\xD1\x9F",                 # CYRILLIC CAPITAL LETTER DZHE
        "\xD0\x90"         => "\xD0\xB0",                 # CYRILLIC CAPITAL LETTER A
        "\xD0\x91"         => "\xD0\xB1",                 # CYRILLIC CAPITAL LETTER BE
        "\xD0\x92"         => "\xD0\xB2",                 # CYRILLIC CAPITAL LETTER VE
        "\xD0\x93"         => "\xD0\xB3",                 # CYRILLIC CAPITAL LETTER GHE
        "\xD0\x94"         => "\xD0\xB4",                 # CYRILLIC CAPITAL LETTER DE
        "\xD0\x95"         => "\xD0\xB5",                 # CYRILLIC CAPITAL LETTER IE
        "\xD0\x96"         => "\xD0\xB6",                 # CYRILLIC CAPITAL LETTER ZHE
        "\xD0\x97"         => "\xD0\xB7",                 # CYRILLIC CAPITAL LETTER ZE
        "\xD0\x98"         => "\xD0\xB8",                 # CYRILLIC CAPITAL LETTER I
        "\xD0\x99"         => "\xD0\xB9",                 # CYRILLIC CAPITAL LETTER SHORT I
        "\xD0\x9A"         => "\xD0\xBA",                 # CYRILLIC CAPITAL LETTER KA
        "\xD0\x9B"         => "\xD0\xBB",                 # CYRILLIC CAPITAL LETTER EL
        "\xD0\x9C"         => "\xD0\xBC",                 # CYRILLIC CAPITAL LETTER EM
        "\xD0\x9D"         => "\xD0\xBD",                 # CYRILLIC CAPITAL LETTER EN
        "\xD0\x9E"         => "\xD0\xBE",                 # CYRILLIC CAPITAL LETTER O
        "\xD0\x9F"         => "\xD0\xBF",                 # CYRILLIC CAPITAL LETTER PE
        "\xD0\xA0"         => "\xD1\x80",                 # CYRILLIC CAPITAL LETTER ER
        "\xD0\xA1"         => "\xD1\x81",                 # CYRILLIC CAPITAL LETTER ES
        "\xD0\xA2"         => "\xD1\x82",                 # CYRILLIC CAPITAL LETTER TE
        "\xD0\xA3"         => "\xD1\x83",                 # CYRILLIC CAPITAL LETTER U
        "\xD0\xA4"         => "\xD1\x84",                 # CYRILLIC CAPITAL LETTER EF
        "\xD0\xA5"         => "\xD1\x85",                 # CYRILLIC CAPITAL LETTER HA
        "\xD0\xA6"         => "\xD1\x86",                 # CYRILLIC CAPITAL LETTER TSE
        "\xD0\xA7"         => "\xD1\x87",                 # CYRILLIC CAPITAL LETTER CHE
        "\xD0\xA8"         => "\xD1\x88",                 # CYRILLIC CAPITAL LETTER SHA
        "\xD0\xA9"         => "\xD1\x89",                 # CYRILLIC CAPITAL LETTER SHCHA
        "\xD0\xAA"         => "\xD1\x8A",                 # CYRILLIC CAPITAL LETTER HARD SIGN
        "\xD0\xAB"         => "\xD1\x8B",                 # CYRILLIC CAPITAL LETTER YERU
        "\xD0\xAC"         => "\xD1\x8C",                 # CYRILLIC CAPITAL LETTER SOFT SIGN
        "\xD0\xAD"         => "\xD1\x8D",                 # CYRILLIC CAPITAL LETTER E
        "\xD0\xAE"         => "\xD1\x8E",                 # CYRILLIC CAPITAL LETTER YU
        "\xD0\xAF"         => "\xD1\x8F",                 # CYRILLIC CAPITAL LETTER YA
        "\xD1\xA0"         => "\xD1\xA1",                 # CYRILLIC CAPITAL LETTER OMEGA
        "\xD1\xA2"         => "\xD1\xA3",                 # CYRILLIC CAPITAL LETTER YAT
        "\xD1\xA4"         => "\xD1\xA5",                 # CYRILLIC CAPITAL LETTER IOTIFIED E
        "\xD1\xA6"         => "\xD1\xA7",                 # CYRILLIC CAPITAL LETTER LITTLE YUS
        "\xD1\xA8"         => "\xD1\xA9",                 # CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
        "\xD1\xAA"         => "\xD1\xAB",                 # CYRILLIC CAPITAL LETTER BIG YUS
        "\xD1\xAC"         => "\xD1\xAD",                 # CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
        "\xD1\xAE"         => "\xD1\xAF",                 # CYRILLIC CAPITAL LETTER KSI
        "\xD1\xB0"         => "\xD1\xB1",                 # CYRILLIC CAPITAL LETTER PSI
        "\xD1\xB2"         => "\xD1\xB3",                 # CYRILLIC CAPITAL LETTER FITA
        "\xD1\xB4"         => "\xD1\xB5",                 # CYRILLIC CAPITAL LETTER IZHITSA
        "\xD1\xB6"         => "\xD1\xB7",                 # CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
        "\xD1\xB8"         => "\xD1\xB9",                 # CYRILLIC CAPITAL LETTER UK
        "\xD1\xBA"         => "\xD1\xBB",                 # CYRILLIC CAPITAL LETTER ROUND OMEGA
        "\xD1\xBC"         => "\xD1\xBD",                 # CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
        "\xD1\xBE"         => "\xD1\xBF",                 # CYRILLIC CAPITAL LETTER OT
        "\xD2\x80"         => "\xD2\x81",                 # CYRILLIC CAPITAL LETTER KOPPA
        "\xD2\x8A"         => "\xD2\x8B",                 # CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
        "\xD2\x8C"         => "\xD2\x8D",                 # CYRILLIC CAPITAL LETTER SEMISOFT SIGN
        "\xD2\x8E"         => "\xD2\x8F",                 # CYRILLIC CAPITAL LETTER ER WITH TICK
        "\xD2\x90"         => "\xD2\x91",                 # CYRILLIC CAPITAL LETTER GHE WITH UPTURN
        "\xD2\x92"         => "\xD2\x93",                 # CYRILLIC CAPITAL LETTER GHE WITH STROKE
        "\xD2\x94"         => "\xD2\x95",                 # CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
        "\xD2\x96"         => "\xD2\x97",                 # CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
        "\xD2\x98"         => "\xD2\x99",                 # CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
        "\xD2\x9A"         => "\xD2\x9B",                 # CYRILLIC CAPITAL LETTER KA WITH DESCENDER
        "\xD2\x9C"         => "\xD2\x9D",                 # CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
        "\xD2\x9E"         => "\xD2\x9F",                 # CYRILLIC CAPITAL LETTER KA WITH STROKE
        "\xD2\xA0"         => "\xD2\xA1",                 # CYRILLIC CAPITAL LETTER BASHKIR KA
        "\xD2\xA2"         => "\xD2\xA3",                 # CYRILLIC CAPITAL LETTER EN WITH DESCENDER
        "\xD2\xA4"         => "\xD2\xA5",                 # CYRILLIC CAPITAL LIGATURE EN GHE
        "\xD2\xA6"         => "\xD2\xA7",                 # CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
        "\xD2\xA8"         => "\xD2\xA9",                 # CYRILLIC CAPITAL LETTER ABKHASIAN HA
        "\xD2\xAA"         => "\xD2\xAB",                 # CYRILLIC CAPITAL LETTER ES WITH DESCENDER
        "\xD2\xAC"         => "\xD2\xAD",                 # CYRILLIC CAPITAL LETTER TE WITH DESCENDER
        "\xD2\xAE"         => "\xD2\xAF",                 # CYRILLIC CAPITAL LETTER STRAIGHT U
        "\xD2\xB0"         => "\xD2\xB1",                 # CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
        "\xD2\xB2"         => "\xD2\xB3",                 # CYRILLIC CAPITAL LETTER HA WITH DESCENDER
        "\xD2\xB4"         => "\xD2\xB5",                 # CYRILLIC CAPITAL LIGATURE TE TSE
        "\xD2\xB6"         => "\xD2\xB7",                 # CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
        "\xD2\xB8"         => "\xD2\xB9",                 # CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
        "\xD2\xBA"         => "\xD2\xBB",                 # CYRILLIC CAPITAL LETTER SHHA
        "\xD2\xBC"         => "\xD2\xBD",                 # CYRILLIC CAPITAL LETTER ABKHASIAN CHE
        "\xD2\xBE"         => "\xD2\xBF",                 # CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
        "\xD3\x80"         => "\xD3\x8F",                 # CYRILLIC LETTER PALOCHKA
        "\xD3\x81"         => "\xD3\x82",                 # CYRILLIC CAPITAL LETTER ZHE WITH BREVE
        "\xD3\x83"         => "\xD3\x84",                 # CYRILLIC CAPITAL LETTER KA WITH HOOK
        "\xD3\x85"         => "\xD3\x86",                 # CYRILLIC CAPITAL LETTER EL WITH TAIL
        "\xD3\x87"         => "\xD3\x88",                 # CYRILLIC CAPITAL LETTER EN WITH HOOK
        "\xD3\x89"         => "\xD3\x8A",                 # CYRILLIC CAPITAL LETTER EN WITH TAIL
        "\xD3\x8B"         => "\xD3\x8C",                 # CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
        "\xD3\x8D"         => "\xD3\x8E",                 # CYRILLIC CAPITAL LETTER EM WITH TAIL
        "\xD3\x90"         => "\xD3\x91",                 # CYRILLIC CAPITAL LETTER A WITH BREVE
        "\xD3\x92"         => "\xD3\x93",                 # CYRILLIC CAPITAL LETTER A WITH DIAERESIS
        "\xD3\x94"         => "\xD3\x95",                 # CYRILLIC CAPITAL LIGATURE A IE
        "\xD3\x96"         => "\xD3\x97",                 # CYRILLIC CAPITAL LETTER IE WITH BREVE
        "\xD3\x98"         => "\xD3\x99",                 # CYRILLIC CAPITAL LETTER SCHWA
        "\xD3\x9A"         => "\xD3\x9B",                 # CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
        "\xD3\x9C"         => "\xD3\x9D",                 # CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
        "\xD3\x9E"         => "\xD3\x9F",                 # CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
        "\xD3\xA0"         => "\xD3\xA1",                 # CYRILLIC CAPITAL LETTER ABKHASIAN DZE
        "\xD3\xA2"         => "\xD3\xA3",                 # CYRILLIC CAPITAL LETTER I WITH MACRON
        "\xD3\xA4"         => "\xD3\xA5",                 # CYRILLIC CAPITAL LETTER I WITH DIAERESIS
        "\xD3\xA6"         => "\xD3\xA7",                 # CYRILLIC CAPITAL LETTER O WITH DIAERESIS
        "\xD3\xA8"         => "\xD3\xA9",                 # CYRILLIC CAPITAL LETTER BARRED O
        "\xD3\xAA"         => "\xD3\xAB",                 # CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
        "\xD3\xAC"         => "\xD3\xAD",                 # CYRILLIC CAPITAL LETTER E WITH DIAERESIS
        "\xD3\xAE"         => "\xD3\xAF",                 # CYRILLIC CAPITAL LETTER U WITH MACRON
        "\xD3\xB0"         => "\xD3\xB1",                 # CYRILLIC CAPITAL LETTER U WITH DIAERESIS
        "\xD3\xB2"         => "\xD3\xB3",                 # CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
        "\xD3\xB4"         => "\xD3\xB5",                 # CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
        "\xD3\xB6"         => "\xD3\xB7",                 # CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
        "\xD3\xB8"         => "\xD3\xB9",                 # CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
        "\xD3\xBA"         => "\xD3\xBB",                 # CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
        "\xD3\xBC"         => "\xD3\xBD",                 # CYRILLIC CAPITAL LETTER HA WITH HOOK
        "\xD3\xBE"         => "\xD3\xBF",                 # CYRILLIC CAPITAL LETTER HA WITH STROKE
        "\xD4\x80"         => "\xD4\x81",                 # CYRILLIC CAPITAL LETTER KOMI DE
        "\xD4\x82"         => "\xD4\x83",                 # CYRILLIC CAPITAL LETTER KOMI DJE
        "\xD4\x84"         => "\xD4\x85",                 # CYRILLIC CAPITAL LETTER KOMI ZJE
        "\xD4\x86"         => "\xD4\x87",                 # CYRILLIC CAPITAL LETTER KOMI DZJE
        "\xD4\x88"         => "\xD4\x89",                 # CYRILLIC CAPITAL LETTER KOMI LJE
        "\xD4\x8A"         => "\xD4\x8B",                 # CYRILLIC CAPITAL LETTER KOMI NJE
        "\xD4\x8C"         => "\xD4\x8D",                 # CYRILLIC CAPITAL LETTER KOMI SJE
        "\xD4\x8E"         => "\xD4\x8F",                 # CYRILLIC CAPITAL LETTER KOMI TJE
        "\xD4\x90"         => "\xD4\x91",                 # CYRILLIC CAPITAL LETTER REVERSED ZE
        "\xD4\x92"         => "\xD4\x93",                 # CYRILLIC CAPITAL LETTER EL WITH HOOK
        "\xD4\x94"         => "\xD4\x95",                 # CYRILLIC CAPITAL LETTER LHA
        "\xD4\x96"         => "\xD4\x97",                 # CYRILLIC CAPITAL LETTER RHA
        "\xD4\x98"         => "\xD4\x99",                 # CYRILLIC CAPITAL LETTER YAE
        "\xD4\x9A"         => "\xD4\x9B",                 # CYRILLIC CAPITAL LETTER QA
        "\xD4\x9C"         => "\xD4\x9D",                 # CYRILLIC CAPITAL LETTER WE
        "\xD4\x9E"         => "\xD4\x9F",                 # CYRILLIC CAPITAL LETTER ALEUT KA
        "\xD4\xA0"         => "\xD4\xA1",                 # CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
        "\xD4\xA2"         => "\xD4\xA3",                 # CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
        "\xD4\xA4"         => "\xD4\xA5",                 # CYRILLIC CAPITAL LETTER PE WITH DESCENDER
        "\xD4\xA6"         => "\xD4\xA7",                 # CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
        "\xD4\xA8"         => "\xD4\xA9",                 # CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
        "\xD4\xAA"         => "\xD4\xAB",                 # CYRILLIC CAPITAL LETTER DZZHE
        "\xD4\xAC"         => "\xD4\xAD",                 # CYRILLIC CAPITAL LETTER DCHE
        "\xD4\xAE"         => "\xD4\xAF",                 # CYRILLIC CAPITAL LETTER EL WITH DESCENDER
        "\xD4\xB1"         => "\xD5\xA1",                 # ARMENIAN CAPITAL LETTER AYB
        "\xD4\xB2"         => "\xD5\xA2",                 # ARMENIAN CAPITAL LETTER BEN
        "\xD4\xB3"         => "\xD5\xA3",                 # ARMENIAN CAPITAL LETTER GIM
        "\xD4\xB4"         => "\xD5\xA4",                 # ARMENIAN CAPITAL LETTER DA
        "\xD4\xB5"         => "\xD5\xA5",                 # ARMENIAN CAPITAL LETTER ECH
        "\xD4\xB6"         => "\xD5\xA6",                 # ARMENIAN CAPITAL LETTER ZA
        "\xD4\xB7"         => "\xD5\xA7",                 # ARMENIAN CAPITAL LETTER EH
        "\xD4\xB8"         => "\xD5\xA8",                 # ARMENIAN CAPITAL LETTER ET
        "\xD4\xB9"         => "\xD5\xA9",                 # ARMENIAN CAPITAL LETTER TO
        "\xD4\xBA"         => "\xD5\xAA",                 # ARMENIAN CAPITAL LETTER ZHE
        "\xD4\xBB"         => "\xD5\xAB",                 # ARMENIAN CAPITAL LETTER INI
        "\xD4\xBC"         => "\xD5\xAC",                 # ARMENIAN CAPITAL LETTER LIWN
        "\xD4\xBD"         => "\xD5\xAD",                 # ARMENIAN CAPITAL LETTER XEH
        "\xD4\xBE"         => "\xD5\xAE",                 # ARMENIAN CAPITAL LETTER CA
        "\xD4\xBF"         => "\xD5\xAF",                 # ARMENIAN CAPITAL LETTER KEN
        "\xD5\x80"         => "\xD5\xB0",                 # ARMENIAN CAPITAL LETTER HO
        "\xD5\x81"         => "\xD5\xB1",                 # ARMENIAN CAPITAL LETTER JA
        "\xD5\x82"         => "\xD5\xB2",                 # ARMENIAN CAPITAL LETTER GHAD
        "\xD5\x83"         => "\xD5\xB3",                 # ARMENIAN CAPITAL LETTER CHEH
        "\xD5\x84"         => "\xD5\xB4",                 # ARMENIAN CAPITAL LETTER MEN
        "\xD5\x85"         => "\xD5\xB5",                 # ARMENIAN CAPITAL LETTER YI
        "\xD5\x86"         => "\xD5\xB6",                 # ARMENIAN CAPITAL LETTER NOW
        "\xD5\x87"         => "\xD5\xB7",                 # ARMENIAN CAPITAL LETTER SHA
        "\xD5\x88"         => "\xD5\xB8",                 # ARMENIAN CAPITAL LETTER VO
        "\xD5\x89"         => "\xD5\xB9",                 # ARMENIAN CAPITAL LETTER CHA
        "\xD5\x8A"         => "\xD5\xBA",                 # ARMENIAN CAPITAL LETTER PEH
        "\xD5\x8B"         => "\xD5\xBB",                 # ARMENIAN CAPITAL LETTER JHEH
        "\xD5\x8C"         => "\xD5\xBC",                 # ARMENIAN CAPITAL LETTER RA
        "\xD5\x8D"         => "\xD5\xBD",                 # ARMENIAN CAPITAL LETTER SEH
        "\xD5\x8E"         => "\xD5\xBE",                 # ARMENIAN CAPITAL LETTER VEW
        "\xD5\x8F"         => "\xD5\xBF",                 # ARMENIAN CAPITAL LETTER TIWN
        "\xD5\x90"         => "\xD6\x80",                 # ARMENIAN CAPITAL LETTER REH
        "\xD5\x91"         => "\xD6\x81",                 # ARMENIAN CAPITAL LETTER CO
        "\xD5\x92"         => "\xD6\x82",                 # ARMENIAN CAPITAL LETTER YIWN
        "\xD5\x93"         => "\xD6\x83",                 # ARMENIAN CAPITAL LETTER PIWR
        "\xD5\x94"         => "\xD6\x84",                 # ARMENIAN CAPITAL LETTER KEH
        "\xD5\x95"         => "\xD6\x85",                 # ARMENIAN CAPITAL LETTER OH
        "\xD5\x96"         => "\xD6\x86",                 # ARMENIAN CAPITAL LETTER FEH
        "\xD6\x87"         => "\xD5\xA5\xD6\x82",         # ARMENIAN SMALL LIGATURE ECH YIWN
        "\xE1\x82\xA0"     => "\xE2\xB4\x80",             # GEORGIAN CAPITAL LETTER AN
        "\xE1\x82\xA1"     => "\xE2\xB4\x81",             # GEORGIAN CAPITAL LETTER BAN
        "\xE1\x82\xA2"     => "\xE2\xB4\x82",             # GEORGIAN CAPITAL LETTER GAN
        "\xE1\x82\xA3"     => "\xE2\xB4\x83",             # GEORGIAN CAPITAL LETTER DON
        "\xE1\x82\xA4"     => "\xE2\xB4\x84",             # GEORGIAN CAPITAL LETTER EN
        "\xE1\x82\xA5"     => "\xE2\xB4\x85",             # GEORGIAN CAPITAL LETTER VIN
        "\xE1\x82\xA6"     => "\xE2\xB4\x86",             # GEORGIAN CAPITAL LETTER ZEN
        "\xE1\x82\xA7"     => "\xE2\xB4\x87",             # GEORGIAN CAPITAL LETTER TAN
        "\xE1\x82\xA8"     => "\xE2\xB4\x88",             # GEORGIAN CAPITAL LETTER IN
        "\xE1\x82\xA9"     => "\xE2\xB4\x89",             # GEORGIAN CAPITAL LETTER KAN
        "\xE1\x82\xAA"     => "\xE2\xB4\x8A",             # GEORGIAN CAPITAL LETTER LAS
        "\xE1\x82\xAB"     => "\xE2\xB4\x8B",             # GEORGIAN CAPITAL LETTER MAN
        "\xE1\x82\xAC"     => "\xE2\xB4\x8C",             # GEORGIAN CAPITAL LETTER NAR
        "\xE1\x82\xAD"     => "\xE2\xB4\x8D",             # GEORGIAN CAPITAL LETTER ON
        "\xE1\x82\xAE"     => "\xE2\xB4\x8E",             # GEORGIAN CAPITAL LETTER PAR
        "\xE1\x82\xAF"     => "\xE2\xB4\x8F",             # GEORGIAN CAPITAL LETTER ZHAR
        "\xE1\x82\xB0"     => "\xE2\xB4\x90",             # GEORGIAN CAPITAL LETTER RAE
        "\xE1\x82\xB1"     => "\xE2\xB4\x91",             # GEORGIAN CAPITAL LETTER SAN
        "\xE1\x82\xB2"     => "\xE2\xB4\x92",             # GEORGIAN CAPITAL LETTER TAR
        "\xE1\x82\xB3"     => "\xE2\xB4\x93",             # GEORGIAN CAPITAL LETTER UN
        "\xE1\x82\xB4"     => "\xE2\xB4\x94",             # GEORGIAN CAPITAL LETTER PHAR
        "\xE1\x82\xB5"     => "\xE2\xB4\x95",             # GEORGIAN CAPITAL LETTER KHAR
        "\xE1\x82\xB6"     => "\xE2\xB4\x96",             # GEORGIAN CAPITAL LETTER GHAN
        "\xE1\x82\xB7"     => "\xE2\xB4\x97",             # GEORGIAN CAPITAL LETTER QAR
        "\xE1\x82\xB8"     => "\xE2\xB4\x98",             # GEORGIAN CAPITAL LETTER SHIN
        "\xE1\x82\xB9"     => "\xE2\xB4\x99",             # GEORGIAN CAPITAL LETTER CHIN
        "\xE1\x82\xBA"     => "\xE2\xB4\x9A",             # GEORGIAN CAPITAL LETTER CAN
        "\xE1\x82\xBB"     => "\xE2\xB4\x9B",             # GEORGIAN CAPITAL LETTER JIL
        "\xE1\x82\xBC"     => "\xE2\xB4\x9C",             # GEORGIAN CAPITAL LETTER CIL
        "\xE1\x82\xBD"     => "\xE2\xB4\x9D",             # GEORGIAN CAPITAL LETTER CHAR
        "\xE1\x82\xBE"     => "\xE2\xB4\x9E",             # GEORGIAN CAPITAL LETTER XAN
        "\xE1\x82\xBF"     => "\xE2\xB4\x9F",             # GEORGIAN CAPITAL LETTER JHAN
        "\xE1\x83\x80"     => "\xE2\xB4\xA0",             # GEORGIAN CAPITAL LETTER HAE
        "\xE1\x83\x81"     => "\xE2\xB4\xA1",             # GEORGIAN CAPITAL LETTER HE
        "\xE1\x83\x82"     => "\xE2\xB4\xA2",             # GEORGIAN CAPITAL LETTER HIE
        "\xE1\x83\x83"     => "\xE2\xB4\xA3",             # GEORGIAN CAPITAL LETTER WE
        "\xE1\x83\x84"     => "\xE2\xB4\xA4",             # GEORGIAN CAPITAL LETTER HAR
        "\xE1\x83\x85"     => "\xE2\xB4\xA5",             # GEORGIAN CAPITAL LETTER HOE
        "\xE1\x83\x87"     => "\xE2\xB4\xA7",             # GEORGIAN CAPITAL LETTER YN
        "\xE1\x83\x8D"     => "\xE2\xB4\xAD",             # GEORGIAN CAPITAL LETTER AEN
        "\xE1\x8F\xB8"     => "\xE1\x8F\xB0",             # CHEROKEE SMALL LETTER YE
        "\xE1\x8F\xB9"     => "\xE1\x8F\xB1",             # CHEROKEE SMALL LETTER YI
        "\xE1\x8F\xBA"     => "\xE1\x8F\xB2",             # CHEROKEE SMALL LETTER YO
        "\xE1\x8F\xBB"     => "\xE1\x8F\xB3",             # CHEROKEE SMALL LETTER YU
        "\xE1\x8F\xBC"     => "\xE1\x8F\xB4",             # CHEROKEE SMALL LETTER YV
        "\xE1\x8F\xBD"     => "\xE1\x8F\xB5",             # CHEROKEE SMALL LETTER MV
        "\xE1\xB2\x80"     => "\xD0\xB2",                 # CYRILLIC SMALL LETTER ROUNDED VE
        "\xE1\xB2\x81"     => "\xD0\xB4",                 # CYRILLIC SMALL LETTER LONG-LEGGED DE
        "\xE1\xB2\x82"     => "\xD0\xBE",                 # CYRILLIC SMALL LETTER NARROW O
        "\xE1\xB2\x83"     => "\xD1\x81",                 # CYRILLIC SMALL LETTER WIDE ES
        "\xE1\xB2\x84"     => "\xD1\x82",                 # CYRILLIC SMALL LETTER TALL TE
        "\xE1\xB2\x85"     => "\xD1\x82",                 # CYRILLIC SMALL LETTER THREE-LEGGED TE
        "\xE1\xB2\x86"     => "\xD1\x8A",                 # CYRILLIC SMALL LETTER TALL HARD SIGN
        "\xE1\xB2\x87"     => "\xD1\xA3",                 # CYRILLIC SMALL LETTER TALL YAT
        "\xE1\xB2\x88"     => "\xEA\x99\x8B",             # CYRILLIC SMALL LETTER UNBLENDED UK
        "\xE1\xB2\x90"     => "\xE1\x83\x90",             # GEORGIAN MTAVRULI CAPITAL LETTER AN
        "\xE1\xB2\x91"     => "\xE1\x83\x91",             # GEORGIAN MTAVRULI CAPITAL LETTER BAN
        "\xE1\xB2\x92"     => "\xE1\x83\x92",             # GEORGIAN MTAVRULI CAPITAL LETTER GAN
        "\xE1\xB2\x93"     => "\xE1\x83\x93",             # GEORGIAN MTAVRULI CAPITAL LETTER DON
        "\xE1\xB2\x94"     => "\xE1\x83\x94",             # GEORGIAN MTAVRULI CAPITAL LETTER EN
        "\xE1\xB2\x95"     => "\xE1\x83\x95",             # GEORGIAN MTAVRULI CAPITAL LETTER VIN
        "\xE1\xB2\x96"     => "\xE1\x83\x96",             # GEORGIAN MTAVRULI CAPITAL LETTER ZEN
        "\xE1\xB2\x97"     => "\xE1\x83\x97",             # GEORGIAN MTAVRULI CAPITAL LETTER TAN
        "\xE1\xB2\x98"     => "\xE1\x83\x98",             # GEORGIAN MTAVRULI CAPITAL LETTER IN
        "\xE1\xB2\x99"     => "\xE1\x83\x99",             # GEORGIAN MTAVRULI CAPITAL LETTER KAN
        "\xE1\xB2\x9A"     => "\xE1\x83\x9A",             # GEORGIAN MTAVRULI CAPITAL LETTER LAS
        "\xE1\xB2\x9B"     => "\xE1\x83\x9B",             # GEORGIAN MTAVRULI CAPITAL LETTER MAN
        "\xE1\xB2\x9C"     => "\xE1\x83\x9C",             # GEORGIAN MTAVRULI CAPITAL LETTER NAR
        "\xE1\xB2\x9D"     => "\xE1\x83\x9D",             # GEORGIAN MTAVRULI CAPITAL LETTER ON
        "\xE1\xB2\x9E"     => "\xE1\x83\x9E",             # GEORGIAN MTAVRULI CAPITAL LETTER PAR
        "\xE1\xB2\x9F"     => "\xE1\x83\x9F",             # GEORGIAN MTAVRULI CAPITAL LETTER ZHAR
        "\xE1\xB2\xA0"     => "\xE1\x83\xA0",             # GEORGIAN MTAVRULI CAPITAL LETTER RAE
        "\xE1\xB2\xA1"     => "\xE1\x83\xA1",             # GEORGIAN MTAVRULI CAPITAL LETTER SAN
        "\xE1\xB2\xA2"     => "\xE1\x83\xA2",             # GEORGIAN MTAVRULI CAPITAL LETTER TAR
        "\xE1\xB2\xA3"     => "\xE1\x83\xA3",             # GEORGIAN MTAVRULI CAPITAL LETTER UN
        "\xE1\xB2\xA4"     => "\xE1\x83\xA4",             # GEORGIAN MTAVRULI CAPITAL LETTER PHAR
        "\xE1\xB2\xA5"     => "\xE1\x83\xA5",             # GEORGIAN MTAVRULI CAPITAL LETTER KHAR
        "\xE1\xB2\xA6"     => "\xE1\x83\xA6",             # GEORGIAN MTAVRULI CAPITAL LETTER GHAN
        "\xE1\xB2\xA7"     => "\xE1\x83\xA7",             # GEORGIAN MTAVRULI CAPITAL LETTER QAR
        "\xE1\xB2\xA8"     => "\xE1\x83\xA8",             # GEORGIAN MTAVRULI CAPITAL LETTER SHIN
        "\xE1\xB2\xA9"     => "\xE1\x83\xA9",             # GEORGIAN MTAVRULI CAPITAL LETTER CHIN
        "\xE1\xB2\xAA"     => "\xE1\x83\xAA",             # GEORGIAN MTAVRULI CAPITAL LETTER CAN
        "\xE1\xB2\xAB"     => "\xE1\x83\xAB",             # GEORGIAN MTAVRULI CAPITAL LETTER JIL
        "\xE1\xB2\xAC"     => "\xE1\x83\xAC",             # GEORGIAN MTAVRULI CAPITAL LETTER CIL
        "\xE1\xB2\xAD"     => "\xE1\x83\xAD",             # GEORGIAN MTAVRULI CAPITAL LETTER CHAR
        "\xE1\xB2\xAE"     => "\xE1\x83\xAE",             # GEORGIAN MTAVRULI CAPITAL LETTER XAN
        "\xE1\xB2\xAF"     => "\xE1\x83\xAF",             # GEORGIAN MTAVRULI CAPITAL LETTER JHAN
        "\xE1\xB2\xB0"     => "\xE1\x83\xB0",             # GEORGIAN MTAVRULI CAPITAL LETTER HAE
        "\xE1\xB2\xB1"     => "\xE1\x83\xB1",             # GEORGIAN MTAVRULI CAPITAL LETTER HE
        "\xE1\xB2\xB2"     => "\xE1\x83\xB2",             # GEORGIAN MTAVRULI CAPITAL LETTER HIE
        "\xE1\xB2\xB3"     => "\xE1\x83\xB3",             # GEORGIAN MTAVRULI CAPITAL LETTER WE
        "\xE1\xB2\xB4"     => "\xE1\x83\xB4",             # GEORGIAN MTAVRULI CAPITAL LETTER HAR
        "\xE1\xB2\xB5"     => "\xE1\x83\xB5",             # GEORGIAN MTAVRULI CAPITAL LETTER HOE
        "\xE1\xB2\xB6"     => "\xE1\x83\xB6",             # GEORGIAN MTAVRULI CAPITAL LETTER FI
        "\xE1\xB2\xB7"     => "\xE1\x83\xB7",             # GEORGIAN MTAVRULI CAPITAL LETTER YN
        "\xE1\xB2\xB8"     => "\xE1\x83\xB8",             # GEORGIAN MTAVRULI CAPITAL LETTER ELIFI
        "\xE1\xB2\xB9"     => "\xE1\x83\xB9",             # GEORGIAN MTAVRULI CAPITAL LETTER TURNED GAN
        "\xE1\xB2\xBA"     => "\xE1\x83\xBA",             # GEORGIAN MTAVRULI CAPITAL LETTER AIN
        "\xE1\xB2\xBD"     => "\xE1\x83\xBD",             # GEORGIAN MTAVRULI CAPITAL LETTER AEN
        "\xE1\xB2\xBE"     => "\xE1\x83\xBE",             # GEORGIAN MTAVRULI CAPITAL LETTER HARD SIGN
        "\xE1\xB2\xBF"     => "\xE1\x83\xBF",             # GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
        "\xE1\xB8\x80"     => "\xE1\xB8\x81",             # LATIN CAPITAL LETTER A WITH RING BELOW
        "\xE1\xB8\x82"     => "\xE1\xB8\x83",             # LATIN CAPITAL LETTER B WITH DOT ABOVE
        "\xE1\xB8\x84"     => "\xE1\xB8\x85",             # LATIN CAPITAL LETTER B WITH DOT BELOW
        "\xE1\xB8\x86"     => "\xE1\xB8\x87",             # LATIN CAPITAL LETTER B WITH LINE BELOW
        "\xE1\xB8\x88"     => "\xE1\xB8\x89",             # LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
        "\xE1\xB8\x8A"     => "\xE1\xB8\x8B",             # LATIN CAPITAL LETTER D WITH DOT ABOVE
        "\xE1\xB8\x8C"     => "\xE1\xB8\x8D",             # LATIN CAPITAL LETTER D WITH DOT BELOW
        "\xE1\xB8\x8E"     => "\xE1\xB8\x8F",             # LATIN CAPITAL LETTER D WITH LINE BELOW
        "\xE1\xB8\x90"     => "\xE1\xB8\x91",             # LATIN CAPITAL LETTER D WITH CEDILLA
        "\xE1\xB8\x92"     => "\xE1\xB8\x93",             # LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
        "\xE1\xB8\x94"     => "\xE1\xB8\x95",             # LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
        "\xE1\xB8\x96"     => "\xE1\xB8\x97",             # LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
        "\xE1\xB8\x98"     => "\xE1\xB8\x99",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
        "\xE1\xB8\x9A"     => "\xE1\xB8\x9B",             # LATIN CAPITAL LETTER E WITH TILDE BELOW
        "\xE1\xB8\x9C"     => "\xE1\xB8\x9D",             # LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
        "\xE1\xB8\x9E"     => "\xE1\xB8\x9F",             # LATIN CAPITAL LETTER F WITH DOT ABOVE
        "\xE1\xB8\xA0"     => "\xE1\xB8\xA1",             # LATIN CAPITAL LETTER G WITH MACRON
        "\xE1\xB8\xA2"     => "\xE1\xB8\xA3",             # LATIN CAPITAL LETTER H WITH DOT ABOVE
        "\xE1\xB8\xA4"     => "\xE1\xB8\xA5",             # LATIN CAPITAL LETTER H WITH DOT BELOW
        "\xE1\xB8\xA6"     => "\xE1\xB8\xA7",             # LATIN CAPITAL LETTER H WITH DIAERESIS
        "\xE1\xB8\xA8"     => "\xE1\xB8\xA9",             # LATIN CAPITAL LETTER H WITH CEDILLA
        "\xE1\xB8\xAA"     => "\xE1\xB8\xAB",             # LATIN CAPITAL LETTER H WITH BREVE BELOW
        "\xE1\xB8\xAC"     => "\xE1\xB8\xAD",             # LATIN CAPITAL LETTER I WITH TILDE BELOW
        "\xE1\xB8\xAE"     => "\xE1\xB8\xAF",             # LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
        "\xE1\xB8\xB0"     => "\xE1\xB8\xB1",             # LATIN CAPITAL LETTER K WITH ACUTE
        "\xE1\xB8\xB2"     => "\xE1\xB8\xB3",             # LATIN CAPITAL LETTER K WITH DOT BELOW
        "\xE1\xB8\xB4"     => "\xE1\xB8\xB5",             # LATIN CAPITAL LETTER K WITH LINE BELOW
        "\xE1\xB8\xB6"     => "\xE1\xB8\xB7",             # LATIN CAPITAL LETTER L WITH DOT BELOW
        "\xE1\xB8\xB8"     => "\xE1\xB8\xB9",             # LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
        "\xE1\xB8\xBA"     => "\xE1\xB8\xBB",             # LATIN CAPITAL LETTER L WITH LINE BELOW
        "\xE1\xB8\xBC"     => "\xE1\xB8\xBD",             # LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
        "\xE1\xB8\xBE"     => "\xE1\xB8\xBF",             # LATIN CAPITAL LETTER M WITH ACUTE
        "\xE1\xB9\x80"     => "\xE1\xB9\x81",             # LATIN CAPITAL LETTER M WITH DOT ABOVE
        "\xE1\xB9\x82"     => "\xE1\xB9\x83",             # LATIN CAPITAL LETTER M WITH DOT BELOW
        "\xE1\xB9\x84"     => "\xE1\xB9\x85",             # LATIN CAPITAL LETTER N WITH DOT ABOVE
        "\xE1\xB9\x86"     => "\xE1\xB9\x87",             # LATIN CAPITAL LETTER N WITH DOT BELOW
        "\xE1\xB9\x88"     => "\xE1\xB9\x89",             # LATIN CAPITAL LETTER N WITH LINE BELOW
        "\xE1\xB9\x8A"     => "\xE1\xB9\x8B",             # LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
        "\xE1\xB9\x8C"     => "\xE1\xB9\x8D",             # LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
        "\xE1\xB9\x8E"     => "\xE1\xB9\x8F",             # LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
        "\xE1\xB9\x90"     => "\xE1\xB9\x91",             # LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
        "\xE1\xB9\x92"     => "\xE1\xB9\x93",             # LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
        "\xE1\xB9\x94"     => "\xE1\xB9\x95",             # LATIN CAPITAL LETTER P WITH ACUTE
        "\xE1\xB9\x96"     => "\xE1\xB9\x97",             # LATIN CAPITAL LETTER P WITH DOT ABOVE
        "\xE1\xB9\x98"     => "\xE1\xB9\x99",             # LATIN CAPITAL LETTER R WITH DOT ABOVE
        "\xE1\xB9\x9A"     => "\xE1\xB9\x9B",             # LATIN CAPITAL LETTER R WITH DOT BELOW
        "\xE1\xB9\x9C"     => "\xE1\xB9\x9D",             # LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
        "\xE1\xB9\x9E"     => "\xE1\xB9\x9F",             # LATIN CAPITAL LETTER R WITH LINE BELOW
        "\xE1\xB9\xA0"     => "\xE1\xB9\xA1",             # LATIN CAPITAL LETTER S WITH DOT ABOVE
        "\xE1\xB9\xA2"     => "\xE1\xB9\xA3",             # LATIN CAPITAL LETTER S WITH DOT BELOW
        "\xE1\xB9\xA4"     => "\xE1\xB9\xA5",             # LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
        "\xE1\xB9\xA6"     => "\xE1\xB9\xA7",             # LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
        "\xE1\xB9\xA8"     => "\xE1\xB9\xA9",             # LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
        "\xE1\xB9\xAA"     => "\xE1\xB9\xAB",             # LATIN CAPITAL LETTER T WITH DOT ABOVE
        "\xE1\xB9\xAC"     => "\xE1\xB9\xAD",             # LATIN CAPITAL LETTER T WITH DOT BELOW
        "\xE1\xB9\xAE"     => "\xE1\xB9\xAF",             # LATIN CAPITAL LETTER T WITH LINE BELOW
        "\xE1\xB9\xB0"     => "\xE1\xB9\xB1",             # LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
        "\xE1\xB9\xB2"     => "\xE1\xB9\xB3",             # LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
        "\xE1\xB9\xB4"     => "\xE1\xB9\xB5",             # LATIN CAPITAL LETTER U WITH TILDE BELOW
        "\xE1\xB9\xB6"     => "\xE1\xB9\xB7",             # LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
        "\xE1\xB9\xB8"     => "\xE1\xB9\xB9",             # LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
        "\xE1\xB9\xBA"     => "\xE1\xB9\xBB",             # LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
        "\xE1\xB9\xBC"     => "\xE1\xB9\xBD",             # LATIN CAPITAL LETTER V WITH TILDE
        "\xE1\xB9\xBE"     => "\xE1\xB9\xBF",             # LATIN CAPITAL LETTER V WITH DOT BELOW
        "\xE1\xBA\x80"     => "\xE1\xBA\x81",             # LATIN CAPITAL LETTER W WITH GRAVE
        "\xE1\xBA\x82"     => "\xE1\xBA\x83",             # LATIN CAPITAL LETTER W WITH ACUTE
        "\xE1\xBA\x84"     => "\xE1\xBA\x85",             # LATIN CAPITAL LETTER W WITH DIAERESIS
        "\xE1\xBA\x86"     => "\xE1\xBA\x87",             # LATIN CAPITAL LETTER W WITH DOT ABOVE
        "\xE1\xBA\x88"     => "\xE1\xBA\x89",             # LATIN CAPITAL LETTER W WITH DOT BELOW
        "\xE1\xBA\x8A"     => "\xE1\xBA\x8B",             # LATIN CAPITAL LETTER X WITH DOT ABOVE
        "\xE1\xBA\x8C"     => "\xE1\xBA\x8D",             # LATIN CAPITAL LETTER X WITH DIAERESIS
        "\xE1\xBA\x8E"     => "\xE1\xBA\x8F",             # LATIN CAPITAL LETTER Y WITH DOT ABOVE
        "\xE1\xBA\x90"     => "\xE1\xBA\x91",             # LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
        "\xE1\xBA\x92"     => "\xE1\xBA\x93",             # LATIN CAPITAL LETTER Z WITH DOT BELOW
        "\xE1\xBA\x94"     => "\xE1\xBA\x95",             # LATIN CAPITAL LETTER Z WITH LINE BELOW
        "\xE1\xBA\x96"     => "\x68\xCC\xB1",             # LATIN SMALL LETTER H WITH LINE BELOW
        "\xE1\xBA\x97"     => "\x74\xCC\x88",             # LATIN SMALL LETTER T WITH DIAERESIS
        "\xE1\xBA\x98"     => "\x77\xCC\x8A",             # LATIN SMALL LETTER W WITH RING ABOVE
        "\xE1\xBA\x99"     => "\x79\xCC\x8A",             # LATIN SMALL LETTER Y WITH RING ABOVE
        "\xE1\xBA\x9A"     => "\x61\xCA\xBE",             # LATIN SMALL LETTER A WITH RIGHT HALF RING
        "\xE1\xBA\x9B"     => "\xE1\xB9\xA1",             # LATIN SMALL LETTER LONG S WITH DOT ABOVE
        "\xE1\xBA\x9E"     => "\x73\x73",                 # LATIN CAPITAL LETTER SHARP S
        "\xE1\xBA\xA0"     => "\xE1\xBA\xA1",             # LATIN CAPITAL LETTER A WITH DOT BELOW
        "\xE1\xBA\xA2"     => "\xE1\xBA\xA3",             # LATIN CAPITAL LETTER A WITH HOOK ABOVE
        "\xE1\xBA\xA4"     => "\xE1\xBA\xA5",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBA\xA6"     => "\xE1\xBA\xA7",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBA\xA8"     => "\xE1\xBA\xA9",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBA\xAA"     => "\xE1\xBA\xAB",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
        "\xE1\xBA\xAC"     => "\xE1\xBA\xAD",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBA\xAE"     => "\xE1\xBA\xAF",             # LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
        "\xE1\xBA\xB0"     => "\xE1\xBA\xB1",             # LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
        "\xE1\xBA\xB2"     => "\xE1\xBA\xB3",             # LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
        "\xE1\xBA\xB4"     => "\xE1\xBA\xB5",             # LATIN CAPITAL LETTER A WITH BREVE AND TILDE
        "\xE1\xBA\xB6"     => "\xE1\xBA\xB7",             # LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
        "\xE1\xBA\xB8"     => "\xE1\xBA\xB9",             # LATIN CAPITAL LETTER E WITH DOT BELOW
        "\xE1\xBA\xBA"     => "\xE1\xBA\xBB",             # LATIN CAPITAL LETTER E WITH HOOK ABOVE
        "\xE1\xBA\xBC"     => "\xE1\xBA\xBD",             # LATIN CAPITAL LETTER E WITH TILDE
        "\xE1\xBA\xBE"     => "\xE1\xBA\xBF",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBB\x80"     => "\xE1\xBB\x81",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBB\x82"     => "\xE1\xBB\x83",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBB\x84"     => "\xE1\xBB\x85",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
        "\xE1\xBB\x86"     => "\xE1\xBB\x87",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBB\x88"     => "\xE1\xBB\x89",             # LATIN CAPITAL LETTER I WITH HOOK ABOVE
        "\xE1\xBB\x8A"     => "\xE1\xBB\x8B",             # LATIN CAPITAL LETTER I WITH DOT BELOW
        "\xE1\xBB\x8C"     => "\xE1\xBB\x8D",             # LATIN CAPITAL LETTER O WITH DOT BELOW
        "\xE1\xBB\x8E"     => "\xE1\xBB\x8F",             # LATIN CAPITAL LETTER O WITH HOOK ABOVE
        "\xE1\xBB\x90"     => "\xE1\xBB\x91",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBB\x92"     => "\xE1\xBB\x93",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBB\x94"     => "\xE1\xBB\x95",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBB\x96"     => "\xE1\xBB\x97",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
        "\xE1\xBB\x98"     => "\xE1\xBB\x99",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBB\x9A"     => "\xE1\xBB\x9B",             # LATIN CAPITAL LETTER O WITH HORN AND ACUTE
        "\xE1\xBB\x9C"     => "\xE1\xBB\x9D",             # LATIN CAPITAL LETTER O WITH HORN AND GRAVE
        "\xE1\xBB\x9E"     => "\xE1\xBB\x9F",             # LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
        "\xE1\xBB\xA0"     => "\xE1\xBB\xA1",             # LATIN CAPITAL LETTER O WITH HORN AND TILDE
        "\xE1\xBB\xA2"     => "\xE1\xBB\xA3",             # LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
        "\xE1\xBB\xA4"     => "\xE1\xBB\xA5",             # LATIN CAPITAL LETTER U WITH DOT BELOW
        "\xE1\xBB\xA6"     => "\xE1\xBB\xA7",             # LATIN CAPITAL LETTER U WITH HOOK ABOVE
        "\xE1\xBB\xA8"     => "\xE1\xBB\xA9",             # LATIN CAPITAL LETTER U WITH HORN AND ACUTE
        "\xE1\xBB\xAA"     => "\xE1\xBB\xAB",             # LATIN CAPITAL LETTER U WITH HORN AND GRAVE
        "\xE1\xBB\xAC"     => "\xE1\xBB\xAD",             # LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
        "\xE1\xBB\xAE"     => "\xE1\xBB\xAF",             # LATIN CAPITAL LETTER U WITH HORN AND TILDE
        "\xE1\xBB\xB0"     => "\xE1\xBB\xB1",             # LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
        "\xE1\xBB\xB2"     => "\xE1\xBB\xB3",             # LATIN CAPITAL LETTER Y WITH GRAVE
        "\xE1\xBB\xB4"     => "\xE1\xBB\xB5",             # LATIN CAPITAL LETTER Y WITH DOT BELOW
        "\xE1\xBB\xB6"     => "\xE1\xBB\xB7",             # LATIN CAPITAL LETTER Y WITH HOOK ABOVE
        "\xE1\xBB\xB8"     => "\xE1\xBB\xB9",             # LATIN CAPITAL LETTER Y WITH TILDE
        "\xE1\xBB\xBA"     => "\xE1\xBB\xBB",             # LATIN CAPITAL LETTER MIDDLE-WELSH LL
        "\xE1\xBB\xBC"     => "\xE1\xBB\xBD",             # LATIN CAPITAL LETTER MIDDLE-WELSH V
        "\xE1\xBB\xBE"     => "\xE1\xBB\xBF",             # LATIN CAPITAL LETTER Y WITH LOOP
        "\xE1\xBC\x88"     => "\xE1\xBC\x80",             # GREEK CAPITAL LETTER ALPHA WITH PSILI
        "\xE1\xBC\x89"     => "\xE1\xBC\x81",             # GREEK CAPITAL LETTER ALPHA WITH DASIA
        "\xE1\xBC\x8A"     => "\xE1\xBC\x82",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
        "\xE1\xBC\x8B"     => "\xE1\xBC\x83",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
        "\xE1\xBC\x8C"     => "\xE1\xBC\x84",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
        "\xE1\xBC\x8D"     => "\xE1\xBC\x85",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
        "\xE1\xBC\x8E"     => "\xE1\xBC\x86",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\x8F"     => "\xE1\xBC\x87",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
        "\xE1\xBC\x98"     => "\xE1\xBC\x90",             # GREEK CAPITAL LETTER EPSILON WITH PSILI
        "\xE1\xBC\x99"     => "\xE1\xBC\x91",             # GREEK CAPITAL LETTER EPSILON WITH DASIA
        "\xE1\xBC\x9A"     => "\xE1\xBC\x92",             # GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
        "\xE1\xBC\x9B"     => "\xE1\xBC\x93",             # GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
        "\xE1\xBC\x9C"     => "\xE1\xBC\x94",             # GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
        "\xE1\xBC\x9D"     => "\xE1\xBC\x95",             # GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
        "\xE1\xBC\xA8"     => "\xE1\xBC\xA0",             # GREEK CAPITAL LETTER ETA WITH PSILI
        "\xE1\xBC\xA9"     => "\xE1\xBC\xA1",             # GREEK CAPITAL LETTER ETA WITH DASIA
        "\xE1\xBC\xAA"     => "\xE1\xBC\xA2",             # GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
        "\xE1\xBC\xAB"     => "\xE1\xBC\xA3",             # GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
        "\xE1\xBC\xAC"     => "\xE1\xBC\xA4",             # GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
        "\xE1\xBC\xAD"     => "\xE1\xBC\xA5",             # GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
        "\xE1\xBC\xAE"     => "\xE1\xBC\xA6",             # GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\xAF"     => "\xE1\xBC\xA7",             # GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
        "\xE1\xBC\xB8"     => "\xE1\xBC\xB0",             # GREEK CAPITAL LETTER IOTA WITH PSILI
        "\xE1\xBC\xB9"     => "\xE1\xBC\xB1",             # GREEK CAPITAL LETTER IOTA WITH DASIA
        "\xE1\xBC\xBA"     => "\xE1\xBC\xB2",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
        "\xE1\xBC\xBB"     => "\xE1\xBC\xB3",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
        "\xE1\xBC\xBC"     => "\xE1\xBC\xB4",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
        "\xE1\xBC\xBD"     => "\xE1\xBC\xB5",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
        "\xE1\xBC\xBE"     => "\xE1\xBC\xB6",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\xBF"     => "\xE1\xBC\xB7",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
        "\xE1\xBD\x88"     => "\xE1\xBD\x80",             # GREEK CAPITAL LETTER OMICRON WITH PSILI
        "\xE1\xBD\x89"     => "\xE1\xBD\x81",             # GREEK CAPITAL LETTER OMICRON WITH DASIA
        "\xE1\xBD\x8A"     => "\xE1\xBD\x82",             # GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
        "\xE1\xBD\x8B"     => "\xE1\xBD\x83",             # GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
        "\xE1\xBD\x8C"     => "\xE1\xBD\x84",             # GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
        "\xE1\xBD\x8D"     => "\xE1\xBD\x85",             # GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
        "\xE1\xBD\x90"     => "\xCF\x85\xCC\x93",         # GREEK SMALL LETTER UPSILON WITH PSILI
        "\xE1\xBD\x92"     => "\xCF\x85\xCC\x93\xCC\x80", # GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
        "\xE1\xBD\x94"     => "\xCF\x85\xCC\x93\xCC\x81", # GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
        "\xE1\xBD\x96"     => "\xCF\x85\xCC\x93\xCD\x82", # GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
        "\xE1\xBD\x99"     => "\xE1\xBD\x91",             # GREEK CAPITAL LETTER UPSILON WITH DASIA
        "\xE1\xBD\x9B"     => "\xE1\xBD\x93",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
        "\xE1\xBD\x9D"     => "\xE1\xBD\x95",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
        "\xE1\xBD\x9F"     => "\xE1\xBD\x97",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
        "\xE1\xBD\xA8"     => "\xE1\xBD\xA0",             # GREEK CAPITAL LETTER OMEGA WITH PSILI
        "\xE1\xBD\xA9"     => "\xE1\xBD\xA1",             # GREEK CAPITAL LETTER OMEGA WITH DASIA
        "\xE1\xBD\xAA"     => "\xE1\xBD\xA2",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
        "\xE1\xBD\xAB"     => "\xE1\xBD\xA3",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
        "\xE1\xBD\xAC"     => "\xE1\xBD\xA4",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
        "\xE1\xBD\xAD"     => "\xE1\xBD\xA5",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
        "\xE1\xBD\xAE"     => "\xE1\xBD\xA6",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
        "\xE1\xBD\xAF"     => "\xE1\xBD\xA7",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
        "\xE1\xBE\x80"     => "\xE1\xBC\x80\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\x81"     => "\xE1\xBC\x81\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\x82"     => "\xE1\xBC\x82\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x83"     => "\xE1\xBC\x83\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x84"     => "\xE1\xBC\x84\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x85"     => "\xE1\xBC\x85\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x86"     => "\xE1\xBC\x86\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x87"     => "\xE1\xBC\x87\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x88"     => "\xE1\xBC\x80\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\x89"     => "\xE1\xBC\x81\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8A"     => "\xE1\xBC\x82\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8B"     => "\xE1\xBC\x83\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8C"     => "\xE1\xBC\x84\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8D"     => "\xE1\xBC\x85\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8E"     => "\xE1\xBC\x86\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x8F"     => "\xE1\xBC\x87\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x90"     => "\xE1\xBC\xA0\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\x91"     => "\xE1\xBC\xA1\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\x92"     => "\xE1\xBC\xA2\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x93"     => "\xE1\xBC\xA3\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x94"     => "\xE1\xBC\xA4\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x95"     => "\xE1\xBC\xA5\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x96"     => "\xE1\xBC\xA6\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x97"     => "\xE1\xBC\xA7\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x98"     => "\xE1\xBC\xA0\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\x99"     => "\xE1\xBC\xA1\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9A"     => "\xE1\xBC\xA2\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9B"     => "\xE1\xBC\xA3\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9C"     => "\xE1\xBC\xA4\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9D"     => "\xE1\xBC\xA5\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9E"     => "\xE1\xBC\xA6\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x9F"     => "\xE1\xBC\xA7\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xA0"     => "\xE1\xBD\xA0\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\xA1"     => "\xE1\xBD\xA1\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA2"     => "\xE1\xBD\xA2\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA3"     => "\xE1\xBD\xA3\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA4"     => "\xE1\xBD\xA4\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA5"     => "\xE1\xBD\xA5\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA6"     => "\xE1\xBD\xA6\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xA7"     => "\xE1\xBD\xA7\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xA8"     => "\xE1\xBD\xA0\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\xA9"     => "\xE1\xBD\xA1\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAA"     => "\xE1\xBD\xA2\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAB"     => "\xE1\xBD\xA3\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAC"     => "\xE1\xBD\xA4\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAD"     => "\xE1\xBD\xA5\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAE"     => "\xE1\xBD\xA6\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xAF"     => "\xE1\xBD\xA7\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xB2"     => "\xE1\xBD\xB0\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xB3"     => "\xCE\xB1\xCE\xB9",         # GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
        "\xE1\xBE\xB4"     => "\xCE\xAC\xCE\xB9",         # GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xB6"     => "\xCE\xB1\xCD\x82",         # GREEK SMALL LETTER ALPHA WITH PERISPOMENI
        "\xE1\xBE\xB7"     => "\xCE\xB1\xCD\x82\xCE\xB9", # GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xB8"     => "\xE1\xBE\xB0",             # GREEK CAPITAL LETTER ALPHA WITH VRACHY
        "\xE1\xBE\xB9"     => "\xE1\xBE\xB1",             # GREEK CAPITAL LETTER ALPHA WITH MACRON
        "\xE1\xBE\xBA"     => "\xE1\xBD\xB0",             # GREEK CAPITAL LETTER ALPHA WITH VARIA
        "\xE1\xBE\xBB"     => "\xE1\xBD\xB1",             # GREEK CAPITAL LETTER ALPHA WITH OXIA
        "\xE1\xBE\xBC"     => "\xCE\xB1\xCE\xB9",         # GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
        "\xE1\xBE\xBE"     => "\xCE\xB9",                 # GREEK PROSGEGRAMMENI
        "\xE1\xBF\x82"     => "\xE1\xBD\xB4\xCE\xB9",     # GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBF\x83"     => "\xCE\xB7\xCE\xB9",         # GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
        "\xE1\xBF\x84"     => "\xCE\xAE\xCE\xB9",         # GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBF\x86"     => "\xCE\xB7\xCD\x82",         # GREEK SMALL LETTER ETA WITH PERISPOMENI
        "\xE1\xBF\x87"     => "\xCE\xB7\xCD\x82\xCE\xB9", # GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBF\x88"     => "\xE1\xBD\xB2",             # GREEK CAPITAL LETTER EPSILON WITH VARIA
        "\xE1\xBF\x89"     => "\xE1\xBD\xB3",             # GREEK CAPITAL LETTER EPSILON WITH OXIA
        "\xE1\xBF\x8A"     => "\xE1\xBD\xB4",             # GREEK CAPITAL LETTER ETA WITH VARIA
        "\xE1\xBF\x8B"     => "\xE1\xBD\xB5",             # GREEK CAPITAL LETTER ETA WITH OXIA
        "\xE1\xBF\x8C"     => "\xCE\xB7\xCE\xB9",         # GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
        "\xE1\xBF\x92"     => "\xCE\xB9\xCC\x88\xCC\x80", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
        "\xE1\xBF\x93"     => "\xCE\xB9\xCC\x88\xCC\x81", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
        "\xE1\xBF\x96"     => "\xCE\xB9\xCD\x82",         # GREEK SMALL LETTER IOTA WITH PERISPOMENI
        "\xE1\xBF\x97"     => "\xCE\xB9\xCC\x88\xCD\x82", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
        "\xE1\xBF\x98"     => "\xE1\xBF\x90",             # GREEK CAPITAL LETTER IOTA WITH VRACHY
        "\xE1\xBF\x99"     => "\xE1\xBF\x91",             # GREEK CAPITAL LETTER IOTA WITH MACRON
        "\xE1\xBF\x9A"     => "\xE1\xBD\xB6",             # GREEK CAPITAL LETTER IOTA WITH VARIA
        "\xE1\xBF\x9B"     => "\xE1\xBD\xB7",             # GREEK CAPITAL LETTER IOTA WITH OXIA
        "\xE1\xBF\xA2"     => "\xCF\x85\xCC\x88\xCC\x80", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
        "\xE1\xBF\xA3"     => "\xCF\x85\xCC\x88\xCC\x81", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
        "\xE1\xBF\xA4"     => "\xCF\x81\xCC\x93",         # GREEK SMALL LETTER RHO WITH PSILI
        "\xE1\xBF\xA6"     => "\xCF\x85\xCD\x82",         # GREEK SMALL LETTER UPSILON WITH PERISPOMENI
        "\xE1\xBF\xA7"     => "\xCF\x85\xCC\x88\xCD\x82", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
        "\xE1\xBF\xA8"     => "\xE1\xBF\xA0",             # GREEK CAPITAL LETTER UPSILON WITH VRACHY
        "\xE1\xBF\xA9"     => "\xE1\xBF\xA1",             # GREEK CAPITAL LETTER UPSILON WITH MACRON
        "\xE1\xBF\xAA"     => "\xE1\xBD\xBA",             # GREEK CAPITAL LETTER UPSILON WITH VARIA
        "\xE1\xBF\xAB"     => "\xE1\xBD\xBB",             # GREEK CAPITAL LETTER UPSILON WITH OXIA
        "\xE1\xBF\xAC"     => "\xE1\xBF\xA5",             # GREEK CAPITAL LETTER RHO WITH DASIA
        "\xE1\xBF\xB2"     => "\xE1\xBD\xBC\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBF\xB3"     => "\xCF\x89\xCE\xB9",         # GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
        "\xE1\xBF\xB4"     => "\xCF\x8E\xCE\xB9",         # GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBF\xB6"     => "\xCF\x89\xCD\x82",         # GREEK SMALL LETTER OMEGA WITH PERISPOMENI
        "\xE1\xBF\xB7"     => "\xCF\x89\xCD\x82\xCE\xB9", # GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBF\xB8"     => "\xE1\xBD\xB8",             # GREEK CAPITAL LETTER OMICRON WITH VARIA
        "\xE1\xBF\xB9"     => "\xE1\xBD\xB9",             # GREEK CAPITAL LETTER OMICRON WITH OXIA
        "\xE1\xBF\xBA"     => "\xE1\xBD\xBC",             # GREEK CAPITAL LETTER OMEGA WITH VARIA
        "\xE1\xBF\xBB"     => "\xE1\xBD\xBD",             # GREEK CAPITAL LETTER OMEGA WITH OXIA
        "\xE1\xBF\xBC"     => "\xCF\x89\xCE\xB9",         # GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
        "\xE2\x84\xA6"     => "\xCF\x89",                 # OHM SIGN
        "\xE2\x84\xAA"     => "\x6B",                     # KELVIN SIGN
        "\xE2\x84\xAB"     => "\xC3\xA5",                 # ANGSTROM SIGN
        "\xE2\x84\xB2"     => "\xE2\x85\x8E",             # TURNED CAPITAL F
        "\xE2\x85\xA0"     => "\xE2\x85\xB0",             # ROMAN NUMERAL ONE
        "\xE2\x85\xA1"     => "\xE2\x85\xB1",             # ROMAN NUMERAL TWO
        "\xE2\x85\xA2"     => "\xE2\x85\xB2",             # ROMAN NUMERAL THREE
        "\xE2\x85\xA3"     => "\xE2\x85\xB3",             # ROMAN NUMERAL FOUR
        "\xE2\x85\xA4"     => "\xE2\x85\xB4",             # ROMAN NUMERAL FIVE
        "\xE2\x85\xA5"     => "\xE2\x85\xB5",             # ROMAN NUMERAL SIX
        "\xE2\x85\xA6"     => "\xE2\x85\xB6",             # ROMAN NUMERAL SEVEN
        "\xE2\x85\xA7"     => "\xE2\x85\xB7",             # ROMAN NUMERAL EIGHT
        "\xE2\x85\xA8"     => "\xE2\x85\xB8",             # ROMAN NUMERAL NINE
        "\xE2\x85\xA9"     => "\xE2\x85\xB9",             # ROMAN NUMERAL TEN
        "\xE2\x85\xAA"     => "\xE2\x85\xBA",             # ROMAN NUMERAL ELEVEN
        "\xE2\x85\xAB"     => "\xE2\x85\xBB",             # ROMAN NUMERAL TWELVE
        "\xE2\x85\xAC"     => "\xE2\x85\xBC",             # ROMAN NUMERAL FIFTY
        "\xE2\x85\xAD"     => "\xE2\x85\xBD",             # ROMAN NUMERAL ONE HUNDRED
        "\xE2\x85\xAE"     => "\xE2\x85\xBE",             # ROMAN NUMERAL FIVE HUNDRED
        "\xE2\x85\xAF"     => "\xE2\x85\xBF",             # ROMAN NUMERAL ONE THOUSAND
        "\xE2\x86\x83"     => "\xE2\x86\x84",             # ROMAN NUMERAL REVERSED ONE HUNDRED
        "\xE2\x92\xB6"     => "\xE2\x93\x90",             # CIRCLED LATIN CAPITAL LETTER A
        "\xE2\x92\xB7"     => "\xE2\x93\x91",             # CIRCLED LATIN CAPITAL LETTER B
        "\xE2\x92\xB8"     => "\xE2\x93\x92",             # CIRCLED LATIN CAPITAL LETTER C
        "\xE2\x92\xB9"     => "\xE2\x93\x93",             # CIRCLED LATIN CAPITAL LETTER D
        "\xE2\x92\xBA"     => "\xE2\x93\x94",             # CIRCLED LATIN CAPITAL LETTER E
        "\xE2\x92\xBB"     => "\xE2\x93\x95",             # CIRCLED LATIN CAPITAL LETTER F
        "\xE2\x92\xBC"     => "\xE2\x93\x96",             # CIRCLED LATIN CAPITAL LETTER G
        "\xE2\x92\xBD"     => "\xE2\x93\x97",             # CIRCLED LATIN CAPITAL LETTER H
        "\xE2\x92\xBE"     => "\xE2\x93\x98",             # CIRCLED LATIN CAPITAL LETTER I
        "\xE2\x92\xBF"     => "\xE2\x93\x99",             # CIRCLED LATIN CAPITAL LETTER J
        "\xE2\x93\x80"     => "\xE2\x93\x9A",             # CIRCLED LATIN CAPITAL LETTER K
        "\xE2\x93\x81"     => "\xE2\x93\x9B",             # CIRCLED LATIN CAPITAL LETTER L
        "\xE2\x93\x82"     => "\xE2\x93\x9C",             # CIRCLED LATIN CAPITAL LETTER M
        "\xE2\x93\x83"     => "\xE2\x93\x9D",             # CIRCLED LATIN CAPITAL LETTER N
        "\xE2\x93\x84"     => "\xE2\x93\x9E",             # CIRCLED LATIN CAPITAL LETTER O
        "\xE2\x93\x85"     => "\xE2\x93\x9F",             # CIRCLED LATIN CAPITAL LETTER P
        "\xE2\x93\x86"     => "\xE2\x93\xA0",             # CIRCLED LATIN CAPITAL LETTER Q
        "\xE2\x93\x87"     => "\xE2\x93\xA1",             # CIRCLED LATIN CAPITAL LETTER R
        "\xE2\x93\x88"     => "\xE2\x93\xA2",             # CIRCLED LATIN CAPITAL LETTER S
        "\xE2\x93\x89"     => "\xE2\x93\xA3",             # CIRCLED LATIN CAPITAL LETTER T
        "\xE2\x93\x8A"     => "\xE2\x93\xA4",             # CIRCLED LATIN CAPITAL LETTER U
        "\xE2\x93\x8B"     => "\xE2\x93\xA5",             # CIRCLED LATIN CAPITAL LETTER V
        "\xE2\x93\x8C"     => "\xE2\x93\xA6",             # CIRCLED LATIN CAPITAL LETTER W
        "\xE2\x93\x8D"     => "\xE2\x93\xA7",             # CIRCLED LATIN CAPITAL LETTER X
        "\xE2\x93\x8E"     => "\xE2\x93\xA8",             # CIRCLED LATIN CAPITAL LETTER Y
        "\xE2\x93\x8F"     => "\xE2\x93\xA9",             # CIRCLED LATIN CAPITAL LETTER Z
        "\xE2\xB0\x80"     => "\xE2\xB0\xB0",             # GLAGOLITIC CAPITAL LETTER AZU
        "\xE2\xB0\x81"     => "\xE2\xB0\xB1",             # GLAGOLITIC CAPITAL LETTER BUKY
        "\xE2\xB0\x82"     => "\xE2\xB0\xB2",             # GLAGOLITIC CAPITAL LETTER VEDE
        "\xE2\xB0\x83"     => "\xE2\xB0\xB3",             # GLAGOLITIC CAPITAL LETTER GLAGOLI
        "\xE2\xB0\x84"     => "\xE2\xB0\xB4",             # GLAGOLITIC CAPITAL LETTER DOBRO
        "\xE2\xB0\x85"     => "\xE2\xB0\xB5",             # GLAGOLITIC CAPITAL LETTER YESTU
        "\xE2\xB0\x86"     => "\xE2\xB0\xB6",             # GLAGOLITIC CAPITAL LETTER ZHIVETE
        "\xE2\xB0\x87"     => "\xE2\xB0\xB7",             # GLAGOLITIC CAPITAL LETTER DZELO
        "\xE2\xB0\x88"     => "\xE2\xB0\xB8",             # GLAGOLITIC CAPITAL LETTER ZEMLJA
        "\xE2\xB0\x89"     => "\xE2\xB0\xB9",             # GLAGOLITIC CAPITAL LETTER IZHE
        "\xE2\xB0\x8A"     => "\xE2\xB0\xBA",             # GLAGOLITIC CAPITAL LETTER INITIAL IZHE
        "\xE2\xB0\x8B"     => "\xE2\xB0\xBB",             # GLAGOLITIC CAPITAL LETTER I
        "\xE2\xB0\x8C"     => "\xE2\xB0\xBC",             # GLAGOLITIC CAPITAL LETTER DJERVI
        "\xE2\xB0\x8D"     => "\xE2\xB0\xBD",             # GLAGOLITIC CAPITAL LETTER KAKO
        "\xE2\xB0\x8E"     => "\xE2\xB0\xBE",             # GLAGOLITIC CAPITAL LETTER LJUDIJE
        "\xE2\xB0\x8F"     => "\xE2\xB0\xBF",             # GLAGOLITIC CAPITAL LETTER MYSLITE
        "\xE2\xB0\x90"     => "\xE2\xB1\x80",             # GLAGOLITIC CAPITAL LETTER NASHI
        "\xE2\xB0\x91"     => "\xE2\xB1\x81",             # GLAGOLITIC CAPITAL LETTER ONU
        "\xE2\xB0\x92"     => "\xE2\xB1\x82",             # GLAGOLITIC CAPITAL LETTER POKOJI
        "\xE2\xB0\x93"     => "\xE2\xB1\x83",             # GLAGOLITIC CAPITAL LETTER RITSI
        "\xE2\xB0\x94"     => "\xE2\xB1\x84",             # GLAGOLITIC CAPITAL LETTER SLOVO
        "\xE2\xB0\x95"     => "\xE2\xB1\x85",             # GLAGOLITIC CAPITAL LETTER TVRIDO
        "\xE2\xB0\x96"     => "\xE2\xB1\x86",             # GLAGOLITIC CAPITAL LETTER UKU
        "\xE2\xB0\x97"     => "\xE2\xB1\x87",             # GLAGOLITIC CAPITAL LETTER FRITU
        "\xE2\xB0\x98"     => "\xE2\xB1\x88",             # GLAGOLITIC CAPITAL LETTER HERU
        "\xE2\xB0\x99"     => "\xE2\xB1\x89",             # GLAGOLITIC CAPITAL LETTER OTU
        "\xE2\xB0\x9A"     => "\xE2\xB1\x8A",             # GLAGOLITIC CAPITAL LETTER PE
        "\xE2\xB0\x9B"     => "\xE2\xB1\x8B",             # GLAGOLITIC CAPITAL LETTER SHTA
        "\xE2\xB0\x9C"     => "\xE2\xB1\x8C",             # GLAGOLITIC CAPITAL LETTER TSI
        "\xE2\xB0\x9D"     => "\xE2\xB1\x8D",             # GLAGOLITIC CAPITAL LETTER CHRIVI
        "\xE2\xB0\x9E"     => "\xE2\xB1\x8E",             # GLAGOLITIC CAPITAL LETTER SHA
        "\xE2\xB0\x9F"     => "\xE2\xB1\x8F",             # GLAGOLITIC CAPITAL LETTER YERU
        "\xE2\xB0\xA0"     => "\xE2\xB1\x90",             # GLAGOLITIC CAPITAL LETTER YERI
        "\xE2\xB0\xA1"     => "\xE2\xB1\x91",             # GLAGOLITIC CAPITAL LETTER YATI
        "\xE2\xB0\xA2"     => "\xE2\xB1\x92",             # GLAGOLITIC CAPITAL LETTER SPIDERY HA
        "\xE2\xB0\xA3"     => "\xE2\xB1\x93",             # GLAGOLITIC CAPITAL LETTER YU
        "\xE2\xB0\xA4"     => "\xE2\xB1\x94",             # GLAGOLITIC CAPITAL LETTER SMALL YUS
        "\xE2\xB0\xA5"     => "\xE2\xB1\x95",             # GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL
        "\xE2\xB0\xA6"     => "\xE2\xB1\x96",             # GLAGOLITIC CAPITAL LETTER YO
        "\xE2\xB0\xA7"     => "\xE2\xB1\x97",             # GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS
        "\xE2\xB0\xA8"     => "\xE2\xB1\x98",             # GLAGOLITIC CAPITAL LETTER BIG YUS
        "\xE2\xB0\xA9"     => "\xE2\xB1\x99",             # GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS
        "\xE2\xB0\xAA"     => "\xE2\xB1\x9A",             # GLAGOLITIC CAPITAL LETTER FITA
        "\xE2\xB0\xAB"     => "\xE2\xB1\x9B",             # GLAGOLITIC CAPITAL LETTER IZHITSA
        "\xE2\xB0\xAC"     => "\xE2\xB1\x9C",             # GLAGOLITIC CAPITAL LETTER SHTAPIC
        "\xE2\xB0\xAD"     => "\xE2\xB1\x9D",             # GLAGOLITIC CAPITAL LETTER TROKUTASTI A
        "\xE2\xB0\xAE"     => "\xE2\xB1\x9E",             # GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE
        "\xE2\xB1\xA0"     => "\xE2\xB1\xA1",             # LATIN CAPITAL LETTER L WITH DOUBLE BAR
        "\xE2\xB1\xA2"     => "\xC9\xAB",                 # LATIN CAPITAL LETTER L WITH MIDDLE TILDE
        "\xE2\xB1\xA3"     => "\xE1\xB5\xBD",             # LATIN CAPITAL LETTER P WITH STROKE
        "\xE2\xB1\xA4"     => "\xC9\xBD",                 # LATIN CAPITAL LETTER R WITH TAIL
        "\xE2\xB1\xA7"     => "\xE2\xB1\xA8",             # LATIN CAPITAL LETTER H WITH DESCENDER
        "\xE2\xB1\xA9"     => "\xE2\xB1\xAA",             # LATIN CAPITAL LETTER K WITH DESCENDER
        "\xE2\xB1\xAB"     => "\xE2\xB1\xAC",             # LATIN CAPITAL LETTER Z WITH DESCENDER
        "\xE2\xB1\xAD"     => "\xC9\x91",                 # LATIN CAPITAL LETTER ALPHA
        "\xE2\xB1\xAE"     => "\xC9\xB1",                 # LATIN CAPITAL LETTER M WITH HOOK
        "\xE2\xB1\xAF"     => "\xC9\x90",                 # LATIN CAPITAL LETTER TURNED A
        "\xE2\xB1\xB0"     => "\xC9\x92",                 # LATIN CAPITAL LETTER TURNED ALPHA
        "\xE2\xB1\xB2"     => "\xE2\xB1\xB3",             # LATIN CAPITAL LETTER W WITH HOOK
        "\xE2\xB1\xB5"     => "\xE2\xB1\xB6",             # LATIN CAPITAL LETTER HALF H
        "\xE2\xB1\xBE"     => "\xC8\xBF",                 # LATIN CAPITAL LETTER S WITH SWASH TAIL
        "\xE2\xB1\xBF"     => "\xC9\x80",                 # LATIN CAPITAL LETTER Z WITH SWASH TAIL
        "\xE2\xB2\x80"     => "\xE2\xB2\x81",             # COPTIC CAPITAL LETTER ALFA
        "\xE2\xB2\x82"     => "\xE2\xB2\x83",             # COPTIC CAPITAL LETTER VIDA
        "\xE2\xB2\x84"     => "\xE2\xB2\x85",             # COPTIC CAPITAL LETTER GAMMA
        "\xE2\xB2\x86"     => "\xE2\xB2\x87",             # COPTIC CAPITAL LETTER DALDA
        "\xE2\xB2\x88"     => "\xE2\xB2\x89",             # COPTIC CAPITAL LETTER EIE
        "\xE2\xB2\x8A"     => "\xE2\xB2\x8B",             # COPTIC CAPITAL LETTER SOU
        "\xE2\xB2\x8C"     => "\xE2\xB2\x8D",             # COPTIC CAPITAL LETTER ZATA
        "\xE2\xB2\x8E"     => "\xE2\xB2\x8F",             # COPTIC CAPITAL LETTER HATE
        "\xE2\xB2\x90"     => "\xE2\xB2\x91",             # COPTIC CAPITAL LETTER THETHE
        "\xE2\xB2\x92"     => "\xE2\xB2\x93",             # COPTIC CAPITAL LETTER IAUDA
        "\xE2\xB2\x94"     => "\xE2\xB2\x95",             # COPTIC CAPITAL LETTER KAPA
        "\xE2\xB2\x96"     => "\xE2\xB2\x97",             # COPTIC CAPITAL LETTER LAULA
        "\xE2\xB2\x98"     => "\xE2\xB2\x99",             # COPTIC CAPITAL LETTER MI
        "\xE2\xB2\x9A"     => "\xE2\xB2\x9B",             # COPTIC CAPITAL LETTER NI
        "\xE2\xB2\x9C"     => "\xE2\xB2\x9D",             # COPTIC CAPITAL LETTER KSI
        "\xE2\xB2\x9E"     => "\xE2\xB2\x9F",             # COPTIC CAPITAL LETTER O
        "\xE2\xB2\xA0"     => "\xE2\xB2\xA1",             # COPTIC CAPITAL LETTER PI
        "\xE2\xB2\xA2"     => "\xE2\xB2\xA3",             # COPTIC CAPITAL LETTER RO
        "\xE2\xB2\xA4"     => "\xE2\xB2\xA5",             # COPTIC CAPITAL LETTER SIMA
        "\xE2\xB2\xA6"     => "\xE2\xB2\xA7",             # COPTIC CAPITAL LETTER TAU
        "\xE2\xB2\xA8"     => "\xE2\xB2\xA9",             # COPTIC CAPITAL LETTER UA
        "\xE2\xB2\xAA"     => "\xE2\xB2\xAB",             # COPTIC CAPITAL LETTER FI
        "\xE2\xB2\xAC"     => "\xE2\xB2\xAD",             # COPTIC CAPITAL LETTER KHI
        "\xE2\xB2\xAE"     => "\xE2\xB2\xAF",             # COPTIC CAPITAL LETTER PSI
        "\xE2\xB2\xB0"     => "\xE2\xB2\xB1",             # COPTIC CAPITAL LETTER OOU
        "\xE2\xB2\xB2"     => "\xE2\xB2\xB3",             # COPTIC CAPITAL LETTER DIALECT-P ALEF
        "\xE2\xB2\xB4"     => "\xE2\xB2\xB5",             # COPTIC CAPITAL LETTER OLD COPTIC AIN
        "\xE2\xB2\xB6"     => "\xE2\xB2\xB7",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
        "\xE2\xB2\xB8"     => "\xE2\xB2\xB9",             # COPTIC CAPITAL LETTER DIALECT-P KAPA
        "\xE2\xB2\xBA"     => "\xE2\xB2\xBB",             # COPTIC CAPITAL LETTER DIALECT-P NI
        "\xE2\xB2\xBC"     => "\xE2\xB2\xBD",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
        "\xE2\xB2\xBE"     => "\xE2\xB2\xBF",             # COPTIC CAPITAL LETTER OLD COPTIC OOU
        "\xE2\xB3\x80"     => "\xE2\xB3\x81",             # COPTIC CAPITAL LETTER SAMPI
        "\xE2\xB3\x82"     => "\xE2\xB3\x83",             # COPTIC CAPITAL LETTER CROSSED SHEI
        "\xE2\xB3\x84"     => "\xE2\xB3\x85",             # COPTIC CAPITAL LETTER OLD COPTIC SHEI
        "\xE2\xB3\x86"     => "\xE2\xB3\x87",             # COPTIC CAPITAL LETTER OLD COPTIC ESH
        "\xE2\xB3\x88"     => "\xE2\xB3\x89",             # COPTIC CAPITAL LETTER AKHMIMIC KHEI
        "\xE2\xB3\x8A"     => "\xE2\xB3\x8B",             # COPTIC CAPITAL LETTER DIALECT-P HORI
        "\xE2\xB3\x8C"     => "\xE2\xB3\x8D",             # COPTIC CAPITAL LETTER OLD COPTIC HORI
        "\xE2\xB3\x8E"     => "\xE2\xB3\x8F",             # COPTIC CAPITAL LETTER OLD COPTIC HA
        "\xE2\xB3\x90"     => "\xE2\xB3\x91",             # COPTIC CAPITAL LETTER L-SHAPED HA
        "\xE2\xB3\x92"     => "\xE2\xB3\x93",             # COPTIC CAPITAL LETTER OLD COPTIC HEI
        "\xE2\xB3\x94"     => "\xE2\xB3\x95",             # COPTIC CAPITAL LETTER OLD COPTIC HAT
        "\xE2\xB3\x96"     => "\xE2\xB3\x97",             # COPTIC CAPITAL LETTER OLD COPTIC GANGIA
        "\xE2\xB3\x98"     => "\xE2\xB3\x99",             # COPTIC CAPITAL LETTER OLD COPTIC DJA
        "\xE2\xB3\x9A"     => "\xE2\xB3\x9B",             # COPTIC CAPITAL LETTER OLD COPTIC SHIMA
        "\xE2\xB3\x9C"     => "\xE2\xB3\x9D",             # COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
        "\xE2\xB3\x9E"     => "\xE2\xB3\x9F",             # COPTIC CAPITAL LETTER OLD NUBIAN NGI
        "\xE2\xB3\xA0"     => "\xE2\xB3\xA1",             # COPTIC CAPITAL LETTER OLD NUBIAN NYI
        "\xE2\xB3\xA2"     => "\xE2\xB3\xA3",             # COPTIC CAPITAL LETTER OLD NUBIAN WAU
        "\xE2\xB3\xAB"     => "\xE2\xB3\xAC",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
        "\xE2\xB3\xAD"     => "\xE2\xB3\xAE",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
        "\xE2\xB3\xB2"     => "\xE2\xB3\xB3",             # COPTIC CAPITAL LETTER BOHAIRIC KHEI
        "\xEA\x99\x80"     => "\xEA\x99\x81",             # CYRILLIC CAPITAL LETTER ZEMLYA
        "\xEA\x99\x82"     => "\xEA\x99\x83",             # CYRILLIC CAPITAL LETTER DZELO
        "\xEA\x99\x84"     => "\xEA\x99\x85",             # CYRILLIC CAPITAL LETTER REVERSED DZE
        "\xEA\x99\x86"     => "\xEA\x99\x87",             # CYRILLIC CAPITAL LETTER IOTA
        "\xEA\x99\x88"     => "\xEA\x99\x89",             # CYRILLIC CAPITAL LETTER DJERV
        "\xEA\x99\x8A"     => "\xEA\x99\x8B",             # CYRILLIC CAPITAL LETTER MONOGRAPH UK
        "\xEA\x99\x8C"     => "\xEA\x99\x8D",             # CYRILLIC CAPITAL LETTER BROAD OMEGA
        "\xEA\x99\x8E"     => "\xEA\x99\x8F",             # CYRILLIC CAPITAL LETTER NEUTRAL YER
        "\xEA\x99\x90"     => "\xEA\x99\x91",             # CYRILLIC CAPITAL LETTER YERU WITH BACK YER
        "\xEA\x99\x92"     => "\xEA\x99\x93",             # CYRILLIC CAPITAL LETTER IOTIFIED YAT
        "\xEA\x99\x94"     => "\xEA\x99\x95",             # CYRILLIC CAPITAL LETTER REVERSED YU
        "\xEA\x99\x96"     => "\xEA\x99\x97",             # CYRILLIC CAPITAL LETTER IOTIFIED A
        "\xEA\x99\x98"     => "\xEA\x99\x99",             # CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
        "\xEA\x99\x9A"     => "\xEA\x99\x9B",             # CYRILLIC CAPITAL LETTER BLENDED YUS
        "\xEA\x99\x9C"     => "\xEA\x99\x9D",             # CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
        "\xEA\x99\x9E"     => "\xEA\x99\x9F",             # CYRILLIC CAPITAL LETTER YN
        "\xEA\x99\xA0"     => "\xEA\x99\xA1",             # CYRILLIC CAPITAL LETTER REVERSED TSE
        "\xEA\x99\xA2"     => "\xEA\x99\xA3",             # CYRILLIC CAPITAL LETTER SOFT DE
        "\xEA\x99\xA4"     => "\xEA\x99\xA5",             # CYRILLIC CAPITAL LETTER SOFT EL
        "\xEA\x99\xA6"     => "\xEA\x99\xA7",             # CYRILLIC CAPITAL LETTER SOFT EM
        "\xEA\x99\xA8"     => "\xEA\x99\xA9",             # CYRILLIC CAPITAL LETTER MONOCULAR O
        "\xEA\x99\xAA"     => "\xEA\x99\xAB",             # CYRILLIC CAPITAL LETTER BINOCULAR O
        "\xEA\x99\xAC"     => "\xEA\x99\xAD",             # CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
        "\xEA\x9A\x80"     => "\xEA\x9A\x81",             # CYRILLIC CAPITAL LETTER DWE
        "\xEA\x9A\x82"     => "\xEA\x9A\x83",             # CYRILLIC CAPITAL LETTER DZWE
        "\xEA\x9A\x84"     => "\xEA\x9A\x85",             # CYRILLIC CAPITAL LETTER ZHWE
        "\xEA\x9A\x86"     => "\xEA\x9A\x87",             # CYRILLIC CAPITAL LETTER CCHE
        "\xEA\x9A\x88"     => "\xEA\x9A\x89",             # CYRILLIC CAPITAL LETTER DZZE
        "\xEA\x9A\x8A"     => "\xEA\x9A\x8B",             # CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
        "\xEA\x9A\x8C"     => "\xEA\x9A\x8D",             # CYRILLIC CAPITAL LETTER TWE
        "\xEA\x9A\x8E"     => "\xEA\x9A\x8F",             # CYRILLIC CAPITAL LETTER TSWE
        "\xEA\x9A\x90"     => "\xEA\x9A\x91",             # CYRILLIC CAPITAL LETTER TSSE
        "\xEA\x9A\x92"     => "\xEA\x9A\x93",             # CYRILLIC CAPITAL LETTER TCHE
        "\xEA\x9A\x94"     => "\xEA\x9A\x95",             # CYRILLIC CAPITAL LETTER HWE
        "\xEA\x9A\x96"     => "\xEA\x9A\x97",             # CYRILLIC CAPITAL LETTER SHWE
        "\xEA\x9A\x98"     => "\xEA\x9A\x99",             # CYRILLIC CAPITAL LETTER DOUBLE O
        "\xEA\x9A\x9A"     => "\xEA\x9A\x9B",             # CYRILLIC CAPITAL LETTER CROSSED O
        "\xEA\x9C\xA2"     => "\xEA\x9C\xA3",             # LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
        "\xEA\x9C\xA4"     => "\xEA\x9C\xA5",             # LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
        "\xEA\x9C\xA6"     => "\xEA\x9C\xA7",             # LATIN CAPITAL LETTER HENG
        "\xEA\x9C\xA8"     => "\xEA\x9C\xA9",             # LATIN CAPITAL LETTER TZ
        "\xEA\x9C\xAA"     => "\xEA\x9C\xAB",             # LATIN CAPITAL LETTER TRESILLO
        "\xEA\x9C\xAC"     => "\xEA\x9C\xAD",             # LATIN CAPITAL LETTER CUATRILLO
        "\xEA\x9C\xAE"     => "\xEA\x9C\xAF",             # LATIN CAPITAL LETTER CUATRILLO WITH COMMA
        "\xEA\x9C\xB2"     => "\xEA\x9C\xB3",             # LATIN CAPITAL LETTER AA
        "\xEA\x9C\xB4"     => "\xEA\x9C\xB5",             # LATIN CAPITAL LETTER AO
        "\xEA\x9C\xB6"     => "\xEA\x9C\xB7",             # LATIN CAPITAL LETTER AU
        "\xEA\x9C\xB8"     => "\xEA\x9C\xB9",             # LATIN CAPITAL LETTER AV
        "\xEA\x9C\xBA"     => "\xEA\x9C\xBB",             # LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
        "\xEA\x9C\xBC"     => "\xEA\x9C\xBD",             # LATIN CAPITAL LETTER AY
        "\xEA\x9C\xBE"     => "\xEA\x9C\xBF",             # LATIN CAPITAL LETTER REVERSED C WITH DOT
        "\xEA\x9D\x80"     => "\xEA\x9D\x81",             # LATIN CAPITAL LETTER K WITH STROKE
        "\xEA\x9D\x82"     => "\xEA\x9D\x83",             # LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
        "\xEA\x9D\x84"     => "\xEA\x9D\x85",             # LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
        "\xEA\x9D\x86"     => "\xEA\x9D\x87",             # LATIN CAPITAL LETTER BROKEN L
        "\xEA\x9D\x88"     => "\xEA\x9D\x89",             # LATIN CAPITAL LETTER L WITH HIGH STROKE
        "\xEA\x9D\x8A"     => "\xEA\x9D\x8B",             # LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
        "\xEA\x9D\x8C"     => "\xEA\x9D\x8D",             # LATIN CAPITAL LETTER O WITH LOOP
        "\xEA\x9D\x8E"     => "\xEA\x9D\x8F",             # LATIN CAPITAL LETTER OO
        "\xEA\x9D\x90"     => "\xEA\x9D\x91",             # LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\x92"     => "\xEA\x9D\x93",             # LATIN CAPITAL LETTER P WITH FLOURISH
        "\xEA\x9D\x94"     => "\xEA\x9D\x95",             # LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
        "\xEA\x9D\x96"     => "\xEA\x9D\x97",             # LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\x98"     => "\xEA\x9D\x99",             # LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
        "\xEA\x9D\x9A"     => "\xEA\x9D\x9B",             # LATIN CAPITAL LETTER R ROTUNDA
        "\xEA\x9D\x9C"     => "\xEA\x9D\x9D",             # LATIN CAPITAL LETTER RUM ROTUNDA
        "\xEA\x9D\x9E"     => "\xEA\x9D\x9F",             # LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
        "\xEA\x9D\xA0"     => "\xEA\x9D\xA1",             # LATIN CAPITAL LETTER VY
        "\xEA\x9D\xA2"     => "\xEA\x9D\xA3",             # LATIN CAPITAL LETTER VISIGOTHIC Z
        "\xEA\x9D\xA4"     => "\xEA\x9D\xA5",             # LATIN CAPITAL LETTER THORN WITH STROKE
        "\xEA\x9D\xA6"     => "\xEA\x9D\xA7",             # LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\xA8"     => "\xEA\x9D\xA9",             # LATIN CAPITAL LETTER VEND
        "\xEA\x9D\xAA"     => "\xEA\x9D\xAB",             # LATIN CAPITAL LETTER ET
        "\xEA\x9D\xAC"     => "\xEA\x9D\xAD",             # LATIN CAPITAL LETTER IS
        "\xEA\x9D\xAE"     => "\xEA\x9D\xAF",             # LATIN CAPITAL LETTER CON
        "\xEA\x9D\xB9"     => "\xEA\x9D\xBA",             # LATIN CAPITAL LETTER INSULAR D
        "\xEA\x9D\xBB"     => "\xEA\x9D\xBC",             # LATIN CAPITAL LETTER INSULAR F
        "\xEA\x9D\xBD"     => "\xE1\xB5\xB9",             # LATIN CAPITAL LETTER INSULAR G
        "\xEA\x9D\xBE"     => "\xEA\x9D\xBF",             # LATIN CAPITAL LETTER TURNED INSULAR G
        "\xEA\x9E\x80"     => "\xEA\x9E\x81",             # LATIN CAPITAL LETTER TURNED L
        "\xEA\x9E\x82"     => "\xEA\x9E\x83",             # LATIN CAPITAL LETTER INSULAR R
        "\xEA\x9E\x84"     => "\xEA\x9E\x85",             # LATIN CAPITAL LETTER INSULAR S
        "\xEA\x9E\x86"     => "\xEA\x9E\x87",             # LATIN CAPITAL LETTER INSULAR T
        "\xEA\x9E\x8B"     => "\xEA\x9E\x8C",             # LATIN CAPITAL LETTER SALTILLO
        "\xEA\x9E\x8D"     => "\xC9\xA5",                 # LATIN CAPITAL LETTER TURNED H
        "\xEA\x9E\x90"     => "\xEA\x9E\x91",             # LATIN CAPITAL LETTER N WITH DESCENDER
        "\xEA\x9E\x92"     => "\xEA\x9E\x93",             # LATIN CAPITAL LETTER C WITH BAR
        "\xEA\x9E\x96"     => "\xEA\x9E\x97",             # LATIN CAPITAL LETTER B WITH FLOURISH
        "\xEA\x9E\x98"     => "\xEA\x9E\x99",             # LATIN CAPITAL LETTER F WITH STROKE
        "\xEA\x9E\x9A"     => "\xEA\x9E\x9B",             # LATIN CAPITAL LETTER VOLAPUK AE
        "\xEA\x9E\x9C"     => "\xEA\x9E\x9D",             # LATIN CAPITAL LETTER VOLAPUK OE
        "\xEA\x9E\x9E"     => "\xEA\x9E\x9F",             # LATIN CAPITAL LETTER VOLAPUK UE
        "\xEA\x9E\xA0"     => "\xEA\x9E\xA1",             # LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
        "\xEA\x9E\xA2"     => "\xEA\x9E\xA3",             # LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
        "\xEA\x9E\xA4"     => "\xEA\x9E\xA5",             # LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
        "\xEA\x9E\xA6"     => "\xEA\x9E\xA7",             # LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
        "\xEA\x9E\xA8"     => "\xEA\x9E\xA9",             # LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
        "\xEA\x9E\xAA"     => "\xC9\xA6",                 # LATIN CAPITAL LETTER H WITH HOOK
        "\xEA\x9E\xAB"     => "\xC9\x9C",                 # LATIN CAPITAL LETTER REVERSED OPEN E
        "\xEA\x9E\xAC"     => "\xC9\xA1",                 # LATIN CAPITAL LETTER SCRIPT G
        "\xEA\x9E\xAD"     => "\xC9\xAC",                 # LATIN CAPITAL LETTER L WITH BELT
        "\xEA\x9E\xAE"     => "\xC9\xAA",                 # LATIN CAPITAL LETTER SMALL CAPITAL I
        "\xEA\x9E\xB0"     => "\xCA\x9E",                 # LATIN CAPITAL LETTER TURNED K
        "\xEA\x9E\xB1"     => "\xCA\x87",                 # LATIN CAPITAL LETTER TURNED T
        "\xEA\x9E\xB2"     => "\xCA\x9D",                 # LATIN CAPITAL LETTER J WITH CROSSED-TAIL
        "\xEA\x9E\xB3"     => "\xEA\xAD\x93",             # LATIN CAPITAL LETTER CHI
        "\xEA\x9E\xB4"     => "\xEA\x9E\xB5",             # LATIN CAPITAL LETTER BETA
        "\xEA\x9E\xB6"     => "\xEA\x9E\xB7",             # LATIN CAPITAL LETTER OMEGA
        "\xEA\x9E\xB8"     => "\xEA\x9E\xB9",             # LATIN CAPITAL LETTER U WITH STROKE
        "\xEA\x9E\xBA"     => "\xEA\x9E\xBB",             # LATIN CAPITAL LETTER GLOTTAL A
        "\xEA\x9E\xBC"     => "\xEA\x9E\xBD",             # LATIN CAPITAL LETTER GLOTTAL I
        "\xEA\x9E\xBE"     => "\xEA\x9E\xBF",             # LATIN CAPITAL LETTER GLOTTAL U
        "\xEA\x9F\x82"     => "\xEA\x9F\x83",             # LATIN CAPITAL LETTER ANGLICANA W
        "\xEA\x9F\x84"     => "\xEA\x9E\x94",             # LATIN CAPITAL LETTER C WITH PALATAL HOOK
        "\xEA\x9F\x85"     => "\xCA\x82",                 # LATIN CAPITAL LETTER S WITH HOOK
        "\xEA\x9F\x86"     => "\xE1\xB6\x8E",             # LATIN CAPITAL LETTER Z WITH PALATAL HOOK
        "\xEA\xAD\xB0"     => "\xE1\x8E\xA0",             # CHEROKEE SMALL LETTER A
        "\xEA\xAD\xB1"     => "\xE1\x8E\xA1",             # CHEROKEE SMALL LETTER E
        "\xEA\xAD\xB2"     => "\xE1\x8E\xA2",             # CHEROKEE SMALL LETTER I
        "\xEA\xAD\xB3"     => "\xE1\x8E\xA3",             # CHEROKEE SMALL LETTER O
        "\xEA\xAD\xB4"     => "\xE1\x8E\xA4",             # CHEROKEE SMALL LETTER U
        "\xEA\xAD\xB5"     => "\xE1\x8E\xA5",             # CHEROKEE SMALL LETTER V
        "\xEA\xAD\xB6"     => "\xE1\x8E\xA6",             # CHEROKEE SMALL LETTER GA
        "\xEA\xAD\xB7"     => "\xE1\x8E\xA7",             # CHEROKEE SMALL LETTER KA
        "\xEA\xAD\xB8"     => "\xE1\x8E\xA8",             # CHEROKEE SMALL LETTER GE
        "\xEA\xAD\xB9"     => "\xE1\x8E\xA9",             # CHEROKEE SMALL LETTER GI
        "\xEA\xAD\xBA"     => "\xE1\x8E\xAA",             # CHEROKEE SMALL LETTER GO
        "\xEA\xAD\xBB"     => "\xE1\x8E\xAB",             # CHEROKEE SMALL LETTER GU
        "\xEA\xAD\xBC"     => "\xE1\x8E\xAC",             # CHEROKEE SMALL LETTER GV
        "\xEA\xAD\xBD"     => "\xE1\x8E\xAD",             # CHEROKEE SMALL LETTER HA
        "\xEA\xAD\xBE"     => "\xE1\x8E\xAE",             # CHEROKEE SMALL LETTER HE
        "\xEA\xAD\xBF"     => "\xE1\x8E\xAF",             # CHEROKEE SMALL LETTER HI
        "\xEA\xAE\x80"     => "\xE1\x8E\xB0",             # CHEROKEE SMALL LETTER HO
        "\xEA\xAE\x81"     => "\xE1\x8E\xB1",             # CHEROKEE SMALL LETTER HU
        "\xEA\xAE\x82"     => "\xE1\x8E\xB2",             # CHEROKEE SMALL LETTER HV
        "\xEA\xAE\x83"     => "\xE1\x8E\xB3",             # CHEROKEE SMALL LETTER LA
        "\xEA\xAE\x84"     => "\xE1\x8E\xB4",             # CHEROKEE SMALL LETTER LE
        "\xEA\xAE\x85"     => "\xE1\x8E\xB5",             # CHEROKEE SMALL LETTER LI
        "\xEA\xAE\x86"     => "\xE1\x8E\xB6",             # CHEROKEE SMALL LETTER LO
        "\xEA\xAE\x87"     => "\xE1\x8E\xB7",             # CHEROKEE SMALL LETTER LU
        "\xEA\xAE\x88"     => "\xE1\x8E\xB8",             # CHEROKEE SMALL LETTER LV
        "\xEA\xAE\x89"     => "\xE1\x8E\xB9",             # CHEROKEE SMALL LETTER MA
        "\xEA\xAE\x8A"     => "\xE1\x8E\xBA",             # CHEROKEE SMALL LETTER ME
        "\xEA\xAE\x8B"     => "\xE1\x8E\xBB",             # CHEROKEE SMALL LETTER MI
        "\xEA\xAE\x8C"     => "\xE1\x8E\xBC",             # CHEROKEE SMALL LETTER MO
        "\xEA\xAE\x8D"     => "\xE1\x8E\xBD",             # CHEROKEE SMALL LETTER MU
        "\xEA\xAE\x8E"     => "\xE1\x8E\xBE",             # CHEROKEE SMALL LETTER NA
        "\xEA\xAE\x8F"     => "\xE1\x8E\xBF",             # CHEROKEE SMALL LETTER HNA
        "\xEA\xAE\x90"     => "\xE1\x8F\x80",             # CHEROKEE SMALL LETTER NAH
        "\xEA\xAE\x91"     => "\xE1\x8F\x81",             # CHEROKEE SMALL LETTER NE
        "\xEA\xAE\x92"     => "\xE1\x8F\x82",             # CHEROKEE SMALL LETTER NI
        "\xEA\xAE\x93"     => "\xE1\x8F\x83",             # CHEROKEE SMALL LETTER NO
        "\xEA\xAE\x94"     => "\xE1\x8F\x84",             # CHEROKEE SMALL LETTER NU
        "\xEA\xAE\x95"     => "\xE1\x8F\x85",             # CHEROKEE SMALL LETTER NV
        "\xEA\xAE\x96"     => "\xE1\x8F\x86",             # CHEROKEE SMALL LETTER QUA
        "\xEA\xAE\x97"     => "\xE1\x8F\x87",             # CHEROKEE SMALL LETTER QUE
        "\xEA\xAE\x98"     => "\xE1\x8F\x88",             # CHEROKEE SMALL LETTER QUI
        "\xEA\xAE\x99"     => "\xE1\x8F\x89",             # CHEROKEE SMALL LETTER QUO
        "\xEA\xAE\x9A"     => "\xE1\x8F\x8A",             # CHEROKEE SMALL LETTER QUU
        "\xEA\xAE\x9B"     => "\xE1\x8F\x8B",             # CHEROKEE SMALL LETTER QUV
        "\xEA\xAE\x9C"     => "\xE1\x8F\x8C",             # CHEROKEE SMALL LETTER SA
        "\xEA\xAE\x9D"     => "\xE1\x8F\x8D",             # CHEROKEE SMALL LETTER S
        "\xEA\xAE\x9E"     => "\xE1\x8F\x8E",             # CHEROKEE SMALL LETTER SE
        "\xEA\xAE\x9F"     => "\xE1\x8F\x8F",             # CHEROKEE SMALL LETTER SI
        "\xEA\xAE\xA0"     => "\xE1\x8F\x90",             # CHEROKEE SMALL LETTER SO
        "\xEA\xAE\xA1"     => "\xE1\x8F\x91",             # CHEROKEE SMALL LETTER SU
        "\xEA\xAE\xA2"     => "\xE1\x8F\x92",             # CHEROKEE SMALL LETTER SV
        "\xEA\xAE\xA3"     => "\xE1\x8F\x93",             # CHEROKEE SMALL LETTER DA
        "\xEA\xAE\xA4"     => "\xE1\x8F\x94",             # CHEROKEE SMALL LETTER TA
        "\xEA\xAE\xA5"     => "\xE1\x8F\x95",             # CHEROKEE SMALL LETTER DE
        "\xEA\xAE\xA6"     => "\xE1\x8F\x96",             # CHEROKEE SMALL LETTER TE
        "\xEA\xAE\xA7"     => "\xE1\x8F\x97",             # CHEROKEE SMALL LETTER DI
        "\xEA\xAE\xA8"     => "\xE1\x8F\x98",             # CHEROKEE SMALL LETTER TI
        "\xEA\xAE\xA9"     => "\xE1\x8F\x99",             # CHEROKEE SMALL LETTER DO
        "\xEA\xAE\xAA"     => "\xE1\x8F\x9A",             # CHEROKEE SMALL LETTER DU
        "\xEA\xAE\xAB"     => "\xE1\x8F\x9B",             # CHEROKEE SMALL LETTER DV
        "\xEA\xAE\xAC"     => "\xE1\x8F\x9C",             # CHEROKEE SMALL LETTER DLA
        "\xEA\xAE\xAD"     => "\xE1\x8F\x9D",             # CHEROKEE SMALL LETTER TLA
        "\xEA\xAE\xAE"     => "\xE1\x8F\x9E",             # CHEROKEE SMALL LETTER TLE
        "\xEA\xAE\xAF"     => "\xE1\x8F\x9F",             # CHEROKEE SMALL LETTER TLI
        "\xEA\xAE\xB0"     => "\xE1\x8F\xA0",             # CHEROKEE SMALL LETTER TLO
        "\xEA\xAE\xB1"     => "\xE1\x8F\xA1",             # CHEROKEE SMALL LETTER TLU
        "\xEA\xAE\xB2"     => "\xE1\x8F\xA2",             # CHEROKEE SMALL LETTER TLV
        "\xEA\xAE\xB3"     => "\xE1\x8F\xA3",             # CHEROKEE SMALL LETTER TSA
        "\xEA\xAE\xB4"     => "\xE1\x8F\xA4",             # CHEROKEE SMALL LETTER TSE
        "\xEA\xAE\xB5"     => "\xE1\x8F\xA5",             # CHEROKEE SMALL LETTER TSI
        "\xEA\xAE\xB6"     => "\xE1\x8F\xA6",             # CHEROKEE SMALL LETTER TSO
        "\xEA\xAE\xB7"     => "\xE1\x8F\xA7",             # CHEROKEE SMALL LETTER TSU
        "\xEA\xAE\xB8"     => "\xE1\x8F\xA8",             # CHEROKEE SMALL LETTER TSV
        "\xEA\xAE\xB9"     => "\xE1\x8F\xA9",             # CHEROKEE SMALL LETTER WA
        "\xEA\xAE\xBA"     => "\xE1\x8F\xAA",             # CHEROKEE SMALL LETTER WE
        "\xEA\xAE\xBB"     => "\xE1\x8F\xAB",             # CHEROKEE SMALL LETTER WI
        "\xEA\xAE\xBC"     => "\xE1\x8F\xAC",             # CHEROKEE SMALL LETTER WO
        "\xEA\xAE\xBD"     => "\xE1\x8F\xAD",             # CHEROKEE SMALL LETTER WU
        "\xEA\xAE\xBE"     => "\xE1\x8F\xAE",             # CHEROKEE SMALL LETTER WV
        "\xEA\xAE\xBF"     => "\xE1\x8F\xAF",             # CHEROKEE SMALL LETTER YA
        "\xEF\xAC\x80"     => "\x66\x66",                 # LATIN SMALL LIGATURE FF
        "\xEF\xAC\x81"     => "\x66\x69",                 # LATIN SMALL LIGATURE FI
        "\xEF\xAC\x82"     => "\x66\x6C",                 # LATIN SMALL LIGATURE FL
        "\xEF\xAC\x83"     => "\x66\x66\x69",             # LATIN SMALL LIGATURE FFI
        "\xEF\xAC\x84"     => "\x66\x66\x6C",             # LATIN SMALL LIGATURE FFL
        "\xEF\xAC\x85"     => "\x73\x74",                 # LATIN SMALL LIGATURE LONG S T
        "\xEF\xAC\x86"     => "\x73\x74",                 # LATIN SMALL LIGATURE ST
        "\xEF\xAC\x93"     => "\xD5\xB4\xD5\xB6",         # ARMENIAN SMALL LIGATURE MEN NOW
        "\xEF\xAC\x94"     => "\xD5\xB4\xD5\xA5",         # ARMENIAN SMALL LIGATURE MEN ECH
        "\xEF\xAC\x95"     => "\xD5\xB4\xD5\xAB",         # ARMENIAN SMALL LIGATURE MEN INI
        "\xEF\xAC\x96"     => "\xD5\xBE\xD5\xB6",         # ARMENIAN SMALL LIGATURE VEW NOW
        "\xEF\xAC\x97"     => "\xD5\xB4\xD5\xAD",         # ARMENIAN SMALL LIGATURE MEN XEH
        "\xEF\xBC\xA1"     => "\xEF\xBD\x81",             # FULLWIDTH LATIN CAPITAL LETTER A
        "\xEF\xBC\xA2"     => "\xEF\xBD\x82",             # FULLWIDTH LATIN CAPITAL LETTER B
        "\xEF\xBC\xA3"     => "\xEF\xBD\x83",             # FULLWIDTH LATIN CAPITAL LETTER C
        "\xEF\xBC\xA4"     => "\xEF\xBD\x84",             # FULLWIDTH LATIN CAPITAL LETTER D
        "\xEF\xBC\xA5"     => "\xEF\xBD\x85",             # FULLWIDTH LATIN CAPITAL LETTER E
        "\xEF\xBC\xA6"     => "\xEF\xBD\x86",             # FULLWIDTH LATIN CAPITAL LETTER F
        "\xEF\xBC\xA7"     => "\xEF\xBD\x87",             # FULLWIDTH LATIN CAPITAL LETTER G
        "\xEF\xBC\xA8"     => "\xEF\xBD\x88",             # FULLWIDTH LATIN CAPITAL LETTER H
        "\xEF\xBC\xA9"     => "\xEF\xBD\x89",             # FULLWIDTH LATIN CAPITAL LETTER I
        "\xEF\xBC\xAA"     => "\xEF\xBD\x8A",             # FULLWIDTH LATIN CAPITAL LETTER J
        "\xEF\xBC\xAB"     => "\xEF\xBD\x8B",             # FULLWIDTH LATIN CAPITAL LETTER K
        "\xEF\xBC\xAC"     => "\xEF\xBD\x8C",             # FULLWIDTH LATIN CAPITAL LETTER L
        "\xEF\xBC\xAD"     => "\xEF\xBD\x8D",             # FULLWIDTH LATIN CAPITAL LETTER M
        "\xEF\xBC\xAE"     => "\xEF\xBD\x8E",             # FULLWIDTH LATIN CAPITAL LETTER N
        "\xEF\xBC\xAF"     => "\xEF\xBD\x8F",             # FULLWIDTH LATIN CAPITAL LETTER O
        "\xEF\xBC\xB0"     => "\xEF\xBD\x90",             # FULLWIDTH LATIN CAPITAL LETTER P
        "\xEF\xBC\xB1"     => "\xEF\xBD\x91",             # FULLWIDTH LATIN CAPITAL LETTER Q
        "\xEF\xBC\xB2"     => "\xEF\xBD\x92",             # FULLWIDTH LATIN CAPITAL LETTER R
        "\xEF\xBC\xB3"     => "\xEF\xBD\x93",             # FULLWIDTH LATIN CAPITAL LETTER S
        "\xEF\xBC\xB4"     => "\xEF\xBD\x94",             # FULLWIDTH LATIN CAPITAL LETTER T
        "\xEF\xBC\xB5"     => "\xEF\xBD\x95",             # FULLWIDTH LATIN CAPITAL LETTER U
        "\xEF\xBC\xB6"     => "\xEF\xBD\x96",             # FULLWIDTH LATIN CAPITAL LETTER V
        "\xEF\xBC\xB7"     => "\xEF\xBD\x97",             # FULLWIDTH LATIN CAPITAL LETTER W
        "\xEF\xBC\xB8"     => "\xEF\xBD\x98",             # FULLWIDTH LATIN CAPITAL LETTER X
        "\xEF\xBC\xB9"     => "\xEF\xBD\x99",             # FULLWIDTH LATIN CAPITAL LETTER Y
        "\xEF\xBC\xBA"     => "\xEF\xBD\x9A",             # FULLWIDTH LATIN CAPITAL LETTER Z
        "\xF0\x90\x90\x80" => "\xF0\x90\x90\xA8",         # DESERET CAPITAL LETTER LONG I
        "\xF0\x90\x90\x81" => "\xF0\x90\x90\xA9",         # DESERET CAPITAL LETTER LONG E
        "\xF0\x90\x90\x82" => "\xF0\x90\x90\xAA",         # DESERET CAPITAL LETTER LONG A
        "\xF0\x90\x90\x83" => "\xF0\x90\x90\xAB",         # DESERET CAPITAL LETTER LONG AH
        "\xF0\x90\x90\x84" => "\xF0\x90\x90\xAC",         # DESERET CAPITAL LETTER LONG O
        "\xF0\x90\x90\x85" => "\xF0\x90\x90\xAD",         # DESERET CAPITAL LETTER LONG OO
        "\xF0\x90\x90\x86" => "\xF0\x90\x90\xAE",         # DESERET CAPITAL LETTER SHORT I
        "\xF0\x90\x90\x87" => "\xF0\x90\x90\xAF",         # DESERET CAPITAL LETTER SHORT E
        "\xF0\x90\x90\x88" => "\xF0\x90\x90\xB0",         # DESERET CAPITAL LETTER SHORT A
        "\xF0\x90\x90\x89" => "\xF0\x90\x90\xB1",         # DESERET CAPITAL LETTER SHORT AH
        "\xF0\x90\x90\x8A" => "\xF0\x90\x90\xB2",         # DESERET CAPITAL LETTER SHORT O
        "\xF0\x90\x90\x8B" => "\xF0\x90\x90\xB3",         # DESERET CAPITAL LETTER SHORT OO
        "\xF0\x90\x90\x8C" => "\xF0\x90\x90\xB4",         # DESERET CAPITAL LETTER AY
        "\xF0\x90\x90\x8D" => "\xF0\x90\x90\xB5",         # DESERET CAPITAL LETTER OW
        "\xF0\x90\x90\x8E" => "\xF0\x90\x90\xB6",         # DESERET CAPITAL LETTER WU
        "\xF0\x90\x90\x8F" => "\xF0\x90\x90\xB7",         # DESERET CAPITAL LETTER YEE
        "\xF0\x90\x90\x90" => "\xF0\x90\x90\xB8",         # DESERET CAPITAL LETTER H
        "\xF0\x90\x90\x91" => "\xF0\x90\x90\xB9",         # DESERET CAPITAL LETTER PEE
        "\xF0\x90\x90\x92" => "\xF0\x90\x90\xBA",         # DESERET CAPITAL LETTER BEE
        "\xF0\x90\x90\x93" => "\xF0\x90\x90\xBB",         # DESERET CAPITAL LETTER TEE
        "\xF0\x90\x90\x94" => "\xF0\x90\x90\xBC",         # DESERET CAPITAL LETTER DEE
        "\xF0\x90\x90\x95" => "\xF0\x90\x90\xBD",         # DESERET CAPITAL LETTER CHEE
        "\xF0\x90\x90\x96" => "\xF0\x90\x90\xBE",         # DESERET CAPITAL LETTER JEE
        "\xF0\x90\x90\x97" => "\xF0\x90\x90\xBF",         # DESERET CAPITAL LETTER KAY
        "\xF0\x90\x90\x98" => "\xF0\x90\x91\x80",         # DESERET CAPITAL LETTER GAY
        "\xF0\x90\x90\x99" => "\xF0\x90\x91\x81",         # DESERET CAPITAL LETTER EF
        "\xF0\x90\x90\x9A" => "\xF0\x90\x91\x82",         # DESERET CAPITAL LETTER VEE
        "\xF0\x90\x90\x9B" => "\xF0\x90\x91\x83",         # DESERET CAPITAL LETTER ETH
        "\xF0\x90\x90\x9C" => "\xF0\x90\x91\x84",         # DESERET CAPITAL LETTER THEE
        "\xF0\x90\x90\x9D" => "\xF0\x90\x91\x85",         # DESERET CAPITAL LETTER ES
        "\xF0\x90\x90\x9E" => "\xF0\x90\x91\x86",         # DESERET CAPITAL LETTER ZEE
        "\xF0\x90\x90\x9F" => "\xF0\x90\x91\x87",         # DESERET CAPITAL LETTER ESH
        "\xF0\x90\x90\xA0" => "\xF0\x90\x91\x88",         # DESERET CAPITAL LETTER ZHEE
        "\xF0\x90\x90\xA1" => "\xF0\x90\x91\x89",         # DESERET CAPITAL LETTER ER
        "\xF0\x90\x90\xA2" => "\xF0\x90\x91\x8A",         # DESERET CAPITAL LETTER EL
        "\xF0\x90\x90\xA3" => "\xF0\x90\x91\x8B",         # DESERET CAPITAL LETTER EM
        "\xF0\x90\x90\xA4" => "\xF0\x90\x91\x8C",         # DESERET CAPITAL LETTER EN
        "\xF0\x90\x90\xA5" => "\xF0\x90\x91\x8D",         # DESERET CAPITAL LETTER ENG
        "\xF0\x90\x90\xA6" => "\xF0\x90\x91\x8E",         # DESERET CAPITAL LETTER OI
        "\xF0\x90\x90\xA7" => "\xF0\x90\x91\x8F",         # DESERET CAPITAL LETTER EW
        "\xF0\x90\x92\xB0" => "\xF0\x90\x93\x98",         # OSAGE CAPITAL LETTER A
        "\xF0\x90\x92\xB1" => "\xF0\x90\x93\x99",         # OSAGE CAPITAL LETTER AI
        "\xF0\x90\x92\xB2" => "\xF0\x90\x93\x9A",         # OSAGE CAPITAL LETTER AIN
        "\xF0\x90\x92\xB3" => "\xF0\x90\x93\x9B",         # OSAGE CAPITAL LETTER AH
        "\xF0\x90\x92\xB4" => "\xF0\x90\x93\x9C",         # OSAGE CAPITAL LETTER BRA
        "\xF0\x90\x92\xB5" => "\xF0\x90\x93\x9D",         # OSAGE CAPITAL LETTER CHA
        "\xF0\x90\x92\xB6" => "\xF0\x90\x93\x9E",         # OSAGE CAPITAL LETTER EHCHA
        "\xF0\x90\x92\xB7" => "\xF0\x90\x93\x9F",         # OSAGE CAPITAL LETTER E
        "\xF0\x90\x92\xB8" => "\xF0\x90\x93\xA0",         # OSAGE CAPITAL LETTER EIN
        "\xF0\x90\x92\xB9" => "\xF0\x90\x93\xA1",         # OSAGE CAPITAL LETTER HA
        "\xF0\x90\x92\xBA" => "\xF0\x90\x93\xA2",         # OSAGE CAPITAL LETTER HYA
        "\xF0\x90\x92\xBB" => "\xF0\x90\x93\xA3",         # OSAGE CAPITAL LETTER I
        "\xF0\x90\x92\xBC" => "\xF0\x90\x93\xA4",         # OSAGE CAPITAL LETTER KA
        "\xF0\x90\x92\xBD" => "\xF0\x90\x93\xA5",         # OSAGE CAPITAL LETTER EHKA
        "\xF0\x90\x92\xBE" => "\xF0\x90\x93\xA6",         # OSAGE CAPITAL LETTER KYA
        "\xF0\x90\x92\xBF" => "\xF0\x90\x93\xA7",         # OSAGE CAPITAL LETTER LA
        "\xF0\x90\x93\x80" => "\xF0\x90\x93\xA8",         # OSAGE CAPITAL LETTER MA
        "\xF0\x90\x93\x81" => "\xF0\x90\x93\xA9",         # OSAGE CAPITAL LETTER NA
        "\xF0\x90\x93\x82" => "\xF0\x90\x93\xAA",         # OSAGE CAPITAL LETTER O
        "\xF0\x90\x93\x83" => "\xF0\x90\x93\xAB",         # OSAGE CAPITAL LETTER OIN
        "\xF0\x90\x93\x84" => "\xF0\x90\x93\xAC",         # OSAGE CAPITAL LETTER PA
        "\xF0\x90\x93\x85" => "\xF0\x90\x93\xAD",         # OSAGE CAPITAL LETTER EHPA
        "\xF0\x90\x93\x86" => "\xF0\x90\x93\xAE",         # OSAGE CAPITAL LETTER SA
        "\xF0\x90\x93\x87" => "\xF0\x90\x93\xAF",         # OSAGE CAPITAL LETTER SHA
        "\xF0\x90\x93\x88" => "\xF0\x90\x93\xB0",         # OSAGE CAPITAL LETTER TA
        "\xF0\x90\x93\x89" => "\xF0\x90\x93\xB1",         # OSAGE CAPITAL LETTER EHTA
        "\xF0\x90\x93\x8A" => "\xF0\x90\x93\xB2",         # OSAGE CAPITAL LETTER TSA
        "\xF0\x90\x93\x8B" => "\xF0\x90\x93\xB3",         # OSAGE CAPITAL LETTER EHTSA
        "\xF0\x90\x93\x8C" => "\xF0\x90\x93\xB4",         # OSAGE CAPITAL LETTER TSHA
        "\xF0\x90\x93\x8D" => "\xF0\x90\x93\xB5",         # OSAGE CAPITAL LETTER DHA
        "\xF0\x90\x93\x8E" => "\xF0\x90\x93\xB6",         # OSAGE CAPITAL LETTER U
        "\xF0\x90\x93\x8F" => "\xF0\x90\x93\xB7",         # OSAGE CAPITAL LETTER WA
        "\xF0\x90\x93\x90" => "\xF0\x90\x93\xB8",         # OSAGE CAPITAL LETTER KHA
        "\xF0\x90\x93\x91" => "\xF0\x90\x93\xB9",         # OSAGE CAPITAL LETTER GHA
        "\xF0\x90\x93\x92" => "\xF0\x90\x93\xBA",         # OSAGE CAPITAL LETTER ZA
        "\xF0\x90\x93\x93" => "\xF0\x90\x93\xBB",         # OSAGE CAPITAL LETTER ZHA
        "\xF0\x90\xB2\x80" => "\xF0\x90\xB3\x80",         # OLD HUNGARIAN CAPITAL LETTER A
        "\xF0\x90\xB2\x81" => "\xF0\x90\xB3\x81",         # OLD HUNGARIAN CAPITAL LETTER AA
        "\xF0\x90\xB2\x82" => "\xF0\x90\xB3\x82",         # OLD HUNGARIAN CAPITAL LETTER EB
        "\xF0\x90\xB2\x83" => "\xF0\x90\xB3\x83",         # OLD HUNGARIAN CAPITAL LETTER AMB
        "\xF0\x90\xB2\x84" => "\xF0\x90\xB3\x84",         # OLD HUNGARIAN CAPITAL LETTER EC
        "\xF0\x90\xB2\x85" => "\xF0\x90\xB3\x85",         # OLD HUNGARIAN CAPITAL LETTER ENC
        "\xF0\x90\xB2\x86" => "\xF0\x90\xB3\x86",         # OLD HUNGARIAN CAPITAL LETTER ECS
        "\xF0\x90\xB2\x87" => "\xF0\x90\xB3\x87",         # OLD HUNGARIAN CAPITAL LETTER ED
        "\xF0\x90\xB2\x88" => "\xF0\x90\xB3\x88",         # OLD HUNGARIAN CAPITAL LETTER AND
        "\xF0\x90\xB2\x89" => "\xF0\x90\xB3\x89",         # OLD HUNGARIAN CAPITAL LETTER E
        "\xF0\x90\xB2\x8A" => "\xF0\x90\xB3\x8A",         # OLD HUNGARIAN CAPITAL LETTER CLOSE E
        "\xF0\x90\xB2\x8B" => "\xF0\x90\xB3\x8B",         # OLD HUNGARIAN CAPITAL LETTER EE
        "\xF0\x90\xB2\x8C" => "\xF0\x90\xB3\x8C",         # OLD HUNGARIAN CAPITAL LETTER EF
        "\xF0\x90\xB2\x8D" => "\xF0\x90\xB3\x8D",         # OLD HUNGARIAN CAPITAL LETTER EG
        "\xF0\x90\xB2\x8E" => "\xF0\x90\xB3\x8E",         # OLD HUNGARIAN CAPITAL LETTER EGY
        "\xF0\x90\xB2\x8F" => "\xF0\x90\xB3\x8F",         # OLD HUNGARIAN CAPITAL LETTER EH
        "\xF0\x90\xB2\x90" => "\xF0\x90\xB3\x90",         # OLD HUNGARIAN CAPITAL LETTER I
        "\xF0\x90\xB2\x91" => "\xF0\x90\xB3\x91",         # OLD HUNGARIAN CAPITAL LETTER II
        "\xF0\x90\xB2\x92" => "\xF0\x90\xB3\x92",         # OLD HUNGARIAN CAPITAL LETTER EJ
        "\xF0\x90\xB2\x93" => "\xF0\x90\xB3\x93",         # OLD HUNGARIAN CAPITAL LETTER EK
        "\xF0\x90\xB2\x94" => "\xF0\x90\xB3\x94",         # OLD HUNGARIAN CAPITAL LETTER AK
        "\xF0\x90\xB2\x95" => "\xF0\x90\xB3\x95",         # OLD HUNGARIAN CAPITAL LETTER UNK
        "\xF0\x90\xB2\x96" => "\xF0\x90\xB3\x96",         # OLD HUNGARIAN CAPITAL LETTER EL
        "\xF0\x90\xB2\x97" => "\xF0\x90\xB3\x97",         # OLD HUNGARIAN CAPITAL LETTER ELY
        "\xF0\x90\xB2\x98" => "\xF0\x90\xB3\x98",         # OLD HUNGARIAN CAPITAL LETTER EM
        "\xF0\x90\xB2\x99" => "\xF0\x90\xB3\x99",         # OLD HUNGARIAN CAPITAL LETTER EN
        "\xF0\x90\xB2\x9A" => "\xF0\x90\xB3\x9A",         # OLD HUNGARIAN CAPITAL LETTER ENY
        "\xF0\x90\xB2\x9B" => "\xF0\x90\xB3\x9B",         # OLD HUNGARIAN CAPITAL LETTER O
        "\xF0\x90\xB2\x9C" => "\xF0\x90\xB3\x9C",         # OLD HUNGARIAN CAPITAL LETTER OO
        "\xF0\x90\xB2\x9D" => "\xF0\x90\xB3\x9D",         # OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG OE
        "\xF0\x90\xB2\x9E" => "\xF0\x90\xB3\x9E",         # OLD HUNGARIAN CAPITAL LETTER RUDIMENTA OE
        "\xF0\x90\xB2\x9F" => "\xF0\x90\xB3\x9F",         # OLD HUNGARIAN CAPITAL LETTER OEE
        "\xF0\x90\xB2\xA0" => "\xF0\x90\xB3\xA0",         # OLD HUNGARIAN CAPITAL LETTER EP
        "\xF0\x90\xB2\xA1" => "\xF0\x90\xB3\xA1",         # OLD HUNGARIAN CAPITAL LETTER EMP
        "\xF0\x90\xB2\xA2" => "\xF0\x90\xB3\xA2",         # OLD HUNGARIAN CAPITAL LETTER ER
        "\xF0\x90\xB2\xA3" => "\xF0\x90\xB3\xA3",         # OLD HUNGARIAN CAPITAL LETTER SHORT ER
        "\xF0\x90\xB2\xA4" => "\xF0\x90\xB3\xA4",         # OLD HUNGARIAN CAPITAL LETTER ES
        "\xF0\x90\xB2\xA5" => "\xF0\x90\xB3\xA5",         # OLD HUNGARIAN CAPITAL LETTER ESZ
        "\xF0\x90\xB2\xA6" => "\xF0\x90\xB3\xA6",         # OLD HUNGARIAN CAPITAL LETTER ET
        "\xF0\x90\xB2\xA7" => "\xF0\x90\xB3\xA7",         # OLD HUNGARIAN CAPITAL LETTER ENT
        "\xF0\x90\xB2\xA8" => "\xF0\x90\xB3\xA8",         # OLD HUNGARIAN CAPITAL LETTER ETY
        "\xF0\x90\xB2\xA9" => "\xF0\x90\xB3\xA9",         # OLD HUNGARIAN CAPITAL LETTER ECH
        "\xF0\x90\xB2\xAA" => "\xF0\x90\xB3\xAA",         # OLD HUNGARIAN CAPITAL LETTER U
        "\xF0\x90\xB2\xAB" => "\xF0\x90\xB3\xAB",         # OLD HUNGARIAN CAPITAL LETTER UU
        "\xF0\x90\xB2\xAC" => "\xF0\x90\xB3\xAC",         # OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG UE
        "\xF0\x90\xB2\xAD" => "\xF0\x90\xB3\xAD",         # OLD HUNGARIAN CAPITAL LETTER RUDIMENTA UE
        "\xF0\x90\xB2\xAE" => "\xF0\x90\xB3\xAE",         # OLD HUNGARIAN CAPITAL LETTER EV
        "\xF0\x90\xB2\xAF" => "\xF0\x90\xB3\xAF",         # OLD HUNGARIAN CAPITAL LETTER EZ
        "\xF0\x90\xB2\xB0" => "\xF0\x90\xB3\xB0",         # OLD HUNGARIAN CAPITAL LETTER EZS
        "\xF0\x90\xB2\xB1" => "\xF0\x90\xB3\xB1",         # OLD HUNGARIAN CAPITAL LETTER ENT-SHAPED SIGN
        "\xF0\x90\xB2\xB2" => "\xF0\x90\xB3\xB2",         # OLD HUNGARIAN CAPITAL LETTER US
        "\xF0\x91\xA2\xA0" => "\xF0\x91\xA3\x80",         # WARANG CITI CAPITAL LETTER NGAA
        "\xF0\x91\xA2\xA1" => "\xF0\x91\xA3\x81",         # WARANG CITI CAPITAL LETTER A
        "\xF0\x91\xA2\xA2" => "\xF0\x91\xA3\x82",         # WARANG CITI CAPITAL LETTER WI
        "\xF0\x91\xA2\xA3" => "\xF0\x91\xA3\x83",         # WARANG CITI CAPITAL LETTER YU
        "\xF0\x91\xA2\xA4" => "\xF0\x91\xA3\x84",         # WARANG CITI CAPITAL LETTER YA
        "\xF0\x91\xA2\xA5" => "\xF0\x91\xA3\x85",         # WARANG CITI CAPITAL LETTER YO
        "\xF0\x91\xA2\xA6" => "\xF0\x91\xA3\x86",         # WARANG CITI CAPITAL LETTER II
        "\xF0\x91\xA2\xA7" => "\xF0\x91\xA3\x87",         # WARANG CITI CAPITAL LETTER UU
        "\xF0\x91\xA2\xA8" => "\xF0\x91\xA3\x88",         # WARANG CITI CAPITAL LETTER E
        "\xF0\x91\xA2\xA9" => "\xF0\x91\xA3\x89",         # WARANG CITI CAPITAL LETTER O
        "\xF0\x91\xA2\xAA" => "\xF0\x91\xA3\x8A",         # WARANG CITI CAPITAL LETTER ANG
        "\xF0\x91\xA2\xAB" => "\xF0\x91\xA3\x8B",         # WARANG CITI CAPITAL LETTER GA
        "\xF0\x91\xA2\xAC" => "\xF0\x91\xA3\x8C",         # WARANG CITI CAPITAL LETTER KO
        "\xF0\x91\xA2\xAD" => "\xF0\x91\xA3\x8D",         # WARANG CITI CAPITAL LETTER ENY
        "\xF0\x91\xA2\xAE" => "\xF0\x91\xA3\x8E",         # WARANG CITI CAPITAL LETTER YUJ
        "\xF0\x91\xA2\xAF" => "\xF0\x91\xA3\x8F",         # WARANG CITI CAPITAL LETTER UC
        "\xF0\x91\xA2\xB0" => "\xF0\x91\xA3\x90",         # WARANG CITI CAPITAL LETTER ENN
        "\xF0\x91\xA2\xB1" => "\xF0\x91\xA3\x91",         # WARANG CITI CAPITAL LETTER ODD
        "\xF0\x91\xA2\xB2" => "\xF0\x91\xA3\x92",         # WARANG CITI CAPITAL LETTER TTE
        "\xF0\x91\xA2\xB3" => "\xF0\x91\xA3\x93",         # WARANG CITI CAPITAL LETTER NUNG
        "\xF0\x91\xA2\xB4" => "\xF0\x91\xA3\x94",         # WARANG CITI CAPITAL LETTER DA
        "\xF0\x91\xA2\xB5" => "\xF0\x91\xA3\x95",         # WARANG CITI CAPITAL LETTER AT
        "\xF0\x91\xA2\xB6" => "\xF0\x91\xA3\x96",         # WARANG CITI CAPITAL LETTER AM
        "\xF0\x91\xA2\xB7" => "\xF0\x91\xA3\x97",         # WARANG CITI CAPITAL LETTER BU
        "\xF0\x91\xA2\xB8" => "\xF0\x91\xA3\x98",         # WARANG CITI CAPITAL LETTER PU
        "\xF0\x91\xA2\xB9" => "\xF0\x91\xA3\x99",         # WARANG CITI CAPITAL LETTER HIYO
        "\xF0\x91\xA2\xBA" => "\xF0\x91\xA3\x9A",         # WARANG CITI CAPITAL LETTER HOLO
        "\xF0\x91\xA2\xBB" => "\xF0\x91\xA3\x9B",         # WARANG CITI CAPITAL LETTER HORR
        "\xF0\x91\xA2\xBC" => "\xF0\x91\xA3\x9C",         # WARANG CITI CAPITAL LETTER HAR
        "\xF0\x91\xA2\xBD" => "\xF0\x91\xA3\x9D",         # WARANG CITI CAPITAL LETTER SSUU
        "\xF0\x91\xA2\xBE" => "\xF0\x91\xA3\x9E",         # WARANG CITI CAPITAL LETTER SII
        "\xF0\x91\xA2\xBF" => "\xF0\x91\xA3\x9F",         # WARANG CITI CAPITAL LETTER VIYO
        "\xF0\x96\xB9\x80" => "\xF0\x96\xB9\xA0",         # MEDEFAIDRIN CAPITAL LETTER M
        "\xF0\x96\xB9\x81" => "\xF0\x96\xB9\xA1",         # MEDEFAIDRIN CAPITAL LETTER S
        "\xF0\x96\xB9\x82" => "\xF0\x96\xB9\xA2",         # MEDEFAIDRIN CAPITAL LETTER V
        "\xF0\x96\xB9\x83" => "\xF0\x96\xB9\xA3",         # MEDEFAIDRIN CAPITAL LETTER W
        "\xF0\x96\xB9\x84" => "\xF0\x96\xB9\xA4",         # MEDEFAIDRIN CAPITAL LETTER ATIU
        "\xF0\x96\xB9\x85" => "\xF0\x96\xB9\xA5",         # MEDEFAIDRIN CAPITAL LETTER Z
        "\xF0\x96\xB9\x86" => "\xF0\x96\xB9\xA6",         # MEDEFAIDRIN CAPITAL LETTER KP
        "\xF0\x96\xB9\x87" => "\xF0\x96\xB9\xA7",         # MEDEFAIDRIN CAPITAL LETTER P
        "\xF0\x96\xB9\x88" => "\xF0\x96\xB9\xA8",         # MEDEFAIDRIN CAPITAL LETTER T
        "\xF0\x96\xB9\x89" => "\xF0\x96\xB9\xA9",         # MEDEFAIDRIN CAPITAL LETTER G
        "\xF0\x96\xB9\x8A" => "\xF0\x96\xB9\xAA",         # MEDEFAIDRIN CAPITAL LETTER F
        "\xF0\x96\xB9\x8B" => "\xF0\x96\xB9\xAB",         # MEDEFAIDRIN CAPITAL LETTER I
        "\xF0\x96\xB9\x8C" => "\xF0\x96\xB9\xAC",         # MEDEFAIDRIN CAPITAL LETTER K
        "\xF0\x96\xB9\x8D" => "\xF0\x96\xB9\xAD",         # MEDEFAIDRIN CAPITAL LETTER A
        "\xF0\x96\xB9\x8E" => "\xF0\x96\xB9\xAE",         # MEDEFAIDRIN CAPITAL LETTER J
        "\xF0\x96\xB9\x8F" => "\xF0\x96\xB9\xAF",         # MEDEFAIDRIN CAPITAL LETTER E
        "\xF0\x96\xB9\x90" => "\xF0\x96\xB9\xB0",         # MEDEFAIDRIN CAPITAL LETTER B
        "\xF0\x96\xB9\x91" => "\xF0\x96\xB9\xB1",         # MEDEFAIDRIN CAPITAL LETTER C
        "\xF0\x96\xB9\x92" => "\xF0\x96\xB9\xB2",         # MEDEFAIDRIN CAPITAL LETTER U
        "\xF0\x96\xB9\x93" => "\xF0\x96\xB9\xB3",         # MEDEFAIDRIN CAPITAL LETTER YU
        "\xF0\x96\xB9\x94" => "\xF0\x96\xB9\xB4",         # MEDEFAIDRIN CAPITAL LETTER L
        "\xF0\x96\xB9\x95" => "\xF0\x96\xB9\xB5",         # MEDEFAIDRIN CAPITAL LETTER Q
        "\xF0\x96\xB9\x96" => "\xF0\x96\xB9\xB6",         # MEDEFAIDRIN CAPITAL LETTER HP
        "\xF0\x96\xB9\x97" => "\xF0\x96\xB9\xB7",         # MEDEFAIDRIN CAPITAL LETTER NY
        "\xF0\x96\xB9\x98" => "\xF0\x96\xB9\xB8",         # MEDEFAIDRIN CAPITAL LETTER X
        "\xF0\x96\xB9\x99" => "\xF0\x96\xB9\xB9",         # MEDEFAIDRIN CAPITAL LETTER D
        "\xF0\x96\xB9\x9A" => "\xF0\x96\xB9\xBA",         # MEDEFAIDRIN CAPITAL LETTER OE
        "\xF0\x96\xB9\x9B" => "\xF0\x96\xB9\xBB",         # MEDEFAIDRIN CAPITAL LETTER N
        "\xF0\x96\xB9\x9C" => "\xF0\x96\xB9\xBC",         # MEDEFAIDRIN CAPITAL LETTER R
        "\xF0\x96\xB9\x9D" => "\xF0\x96\xB9\xBD",         # MEDEFAIDRIN CAPITAL LETTER O
        "\xF0\x96\xB9\x9E" => "\xF0\x96\xB9\xBE",         # MEDEFAIDRIN CAPITAL LETTER AI
        "\xF0\x96\xB9\x9F" => "\xF0\x96\xB9\xBF",         # MEDEFAIDRIN CAPITAL LETTER Y
        "\xF0\x9E\xA4\x80" => "\xF0\x9E\xA4\xA2",         # ADLAM CAPITAL LETTER ALIF
        "\xF0\x9E\xA4\x81" => "\xF0\x9E\xA4\xA3",         # ADLAM CAPITAL LETTER DAALI
        "\xF0\x9E\xA4\x82" => "\xF0\x9E\xA4\xA4",         # ADLAM CAPITAL LETTER LAAM
        "\xF0\x9E\xA4\x83" => "\xF0\x9E\xA4\xA5",         # ADLAM CAPITAL LETTER MIIM
        "\xF0\x9E\xA4\x84" => "\xF0\x9E\xA4\xA6",         # ADLAM CAPITAL LETTER BA
        "\xF0\x9E\xA4\x85" => "\xF0\x9E\xA4\xA7",         # ADLAM CAPITAL LETTER SINNYIIYHE
        "\xF0\x9E\xA4\x86" => "\xF0\x9E\xA4\xA8",         # ADLAM CAPITAL LETTER PE
        "\xF0\x9E\xA4\x87" => "\xF0\x9E\xA4\xA9",         # ADLAM CAPITAL LETTER BHE
        "\xF0\x9E\xA4\x88" => "\xF0\x9E\xA4\xAA",         # ADLAM CAPITAL LETTER RA
        "\xF0\x9E\xA4\x89" => "\xF0\x9E\xA4\xAB",         # ADLAM CAPITAL LETTER E
        "\xF0\x9E\xA4\x8A" => "\xF0\x9E\xA4\xAC",         # ADLAM CAPITAL LETTER FA
        "\xF0\x9E\xA4\x8B" => "\xF0\x9E\xA4\xAD",         # ADLAM CAPITAL LETTER I
        "\xF0\x9E\xA4\x8C" => "\xF0\x9E\xA4\xAE",         # ADLAM CAPITAL LETTER O
        "\xF0\x9E\xA4\x8D" => "\xF0\x9E\xA4\xAF",         # ADLAM CAPITAL LETTER DHA
        "\xF0\x9E\xA4\x8E" => "\xF0\x9E\xA4\xB0",         # ADLAM CAPITAL LETTER YHE
        "\xF0\x9E\xA4\x8F" => "\xF0\x9E\xA4\xB1",         # ADLAM CAPITAL LETTER WAW
        "\xF0\x9E\xA4\x90" => "\xF0\x9E\xA4\xB2",         # ADLAM CAPITAL LETTER NUN
        "\xF0\x9E\xA4\x91" => "\xF0\x9E\xA4\xB3",         # ADLAM CAPITAL LETTER KAF
        "\xF0\x9E\xA4\x92" => "\xF0\x9E\xA4\xB4",         # ADLAM CAPITAL LETTER YA
        "\xF0\x9E\xA4\x93" => "\xF0\x9E\xA4\xB5",         # ADLAM CAPITAL LETTER U
        "\xF0\x9E\xA4\x94" => "\xF0\x9E\xA4\xB6",         # ADLAM CAPITAL LETTER JIIM
        "\xF0\x9E\xA4\x95" => "\xF0\x9E\xA4\xB7",         # ADLAM CAPITAL LETTER CHI
        "\xF0\x9E\xA4\x96" => "\xF0\x9E\xA4\xB8",         # ADLAM CAPITAL LETTER HA
        "\xF0\x9E\xA4\x97" => "\xF0\x9E\xA4\xB9",         # ADLAM CAPITAL LETTER QAAF
        "\xF0\x9E\xA4\x98" => "\xF0\x9E\xA4\xBA",         # ADLAM CAPITAL LETTER GA
        "\xF0\x9E\xA4\x99" => "\xF0\x9E\xA4\xBB",         # ADLAM CAPITAL LETTER NYA
        "\xF0\x9E\xA4\x9A" => "\xF0\x9E\xA4\xBC",         # ADLAM CAPITAL LETTER TU
        "\xF0\x9E\xA4\x9B" => "\xF0\x9E\xA4\xBD",         # ADLAM CAPITAL LETTER NHA
        "\xF0\x9E\xA4\x9C" => "\xF0\x9E\xA4\xBE",         # ADLAM CAPITAL LETTER VA
        "\xF0\x9E\xA4\x9D" => "\xF0\x9E\xA4\xBF",         # ADLAM CAPITAL LETTER KHA
        "\xF0\x9E\xA4\x9E" => "\xF0\x9E\xA5\x80",         # ADLAM CAPITAL LETTER GBE
        "\xF0\x9E\xA4\x9F" => "\xF0\x9E\xA5\x81",         # ADLAM CAPITAL LETTER ZAL
        "\xF0\x9E\xA4\xA0" => "\xF0\x9E\xA5\x82",         # ADLAM CAPITAL LETTER KPO
        "\xF0\x9E\xA4\xA1" => "\xF0\x9E\xA5\x83",         # ADLAM CAPITAL LETTER SHA
    );
}

#endif
#if DIST_OLDUTF8
elsif (__PACKAGE__ =~ / \b Eoldutf8 \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x7F],
               [0xF5..0xFF], # malformed octet
             ],
        2 => [ [0xC0..0xDF],[0x80..0xBF],
             ],
        3 => [ [0xE0..0xEF],[0x80..0xBF],[0x80..0xBF],
             ],
        4 => [ [0xF0..0xF4],[0x80..0xBF],[0x80..0xBF],[0x80..0xBF],
             ],
    );

    # CaseFolding-12.0.0.txt
    # Date: 2019-01-22, 08:18:22 GMT
    # c 2019 UnicodeR, Inc.
    # Unicode and the Unicode Logo are registered trademarks of Unicode, Inc. in the U.S. and other countries.
    # For terms of use, see http://www.unicode.org/terms_of_use.html
    #
    # Unicode Character Database
    #   For documentation, see http://www.unicode.org/reports/tr44/

    # you can use "make_CaseFolding.pl" to update this hash

    %fc = (
        "\x41"             => "\x61",                     # LATIN CAPITAL LETTER A
        "\x42"             => "\x62",                     # LATIN CAPITAL LETTER B
        "\x43"             => "\x63",                     # LATIN CAPITAL LETTER C
        "\x44"             => "\x64",                     # LATIN CAPITAL LETTER D
        "\x45"             => "\x65",                     # LATIN CAPITAL LETTER E
        "\x46"             => "\x66",                     # LATIN CAPITAL LETTER F
        "\x47"             => "\x67",                     # LATIN CAPITAL LETTER G
        "\x48"             => "\x68",                     # LATIN CAPITAL LETTER H
        "\x49"             => "\x69",                     # LATIN CAPITAL LETTER I
        "\x4A"             => "\x6A",                     # LATIN CAPITAL LETTER J
        "\x4B"             => "\x6B",                     # LATIN CAPITAL LETTER K
        "\x4C"             => "\x6C",                     # LATIN CAPITAL LETTER L
        "\x4D"             => "\x6D",                     # LATIN CAPITAL LETTER M
        "\x4E"             => "\x6E",                     # LATIN CAPITAL LETTER N
        "\x4F"             => "\x6F",                     # LATIN CAPITAL LETTER O
        "\x50"             => "\x70",                     # LATIN CAPITAL LETTER P
        "\x51"             => "\x71",                     # LATIN CAPITAL LETTER Q
        "\x52"             => "\x72",                     # LATIN CAPITAL LETTER R
        "\x53"             => "\x73",                     # LATIN CAPITAL LETTER S
        "\x54"             => "\x74",                     # LATIN CAPITAL LETTER T
        "\x55"             => "\x75",                     # LATIN CAPITAL LETTER U
        "\x56"             => "\x76",                     # LATIN CAPITAL LETTER V
        "\x57"             => "\x77",                     # LATIN CAPITAL LETTER W
        "\x58"             => "\x78",                     # LATIN CAPITAL LETTER X
        "\x59"             => "\x79",                     # LATIN CAPITAL LETTER Y
        "\x5A"             => "\x7A",                     # LATIN CAPITAL LETTER Z
        "\xC2\xB5"         => "\xCE\xBC",                 # MICRO SIGN
        "\xC3\x80"         => "\xC3\xA0",                 # LATIN CAPITAL LETTER A WITH GRAVE
        "\xC3\x81"         => "\xC3\xA1",                 # LATIN CAPITAL LETTER A WITH ACUTE
        "\xC3\x82"         => "\xC3\xA2",                 # LATIN CAPITAL LETTER A WITH CIRCUMFLEX
        "\xC3\x83"         => "\xC3\xA3",                 # LATIN CAPITAL LETTER A WITH TILDE
        "\xC3\x84"         => "\xC3\xA4",                 # LATIN CAPITAL LETTER A WITH DIAERESIS
        "\xC3\x85"         => "\xC3\xA5",                 # LATIN CAPITAL LETTER A WITH RING ABOVE
        "\xC3\x86"         => "\xC3\xA6",                 # LATIN CAPITAL LETTER AE
        "\xC3\x87"         => "\xC3\xA7",                 # LATIN CAPITAL LETTER C WITH CEDILLA
        "\xC3\x88"         => "\xC3\xA8",                 # LATIN CAPITAL LETTER E WITH GRAVE
        "\xC3\x89"         => "\xC3\xA9",                 # LATIN CAPITAL LETTER E WITH ACUTE
        "\xC3\x8A"         => "\xC3\xAA",                 # LATIN CAPITAL LETTER E WITH CIRCUMFLEX
        "\xC3\x8B"         => "\xC3\xAB",                 # LATIN CAPITAL LETTER E WITH DIAERESIS
        "\xC3\x8C"         => "\xC3\xAC",                 # LATIN CAPITAL LETTER I WITH GRAVE
        "\xC3\x8D"         => "\xC3\xAD",                 # LATIN CAPITAL LETTER I WITH ACUTE
        "\xC3\x8E"         => "\xC3\xAE",                 # LATIN CAPITAL LETTER I WITH CIRCUMFLEX
        "\xC3\x8F"         => "\xC3\xAF",                 # LATIN CAPITAL LETTER I WITH DIAERESIS
        "\xC3\x90"         => "\xC3\xB0",                 # LATIN CAPITAL LETTER ETH
        "\xC3\x91"         => "\xC3\xB1",                 # LATIN CAPITAL LETTER N WITH TILDE
        "\xC3\x92"         => "\xC3\xB2",                 # LATIN CAPITAL LETTER O WITH GRAVE
        "\xC3\x93"         => "\xC3\xB3",                 # LATIN CAPITAL LETTER O WITH ACUTE
        "\xC3\x94"         => "\xC3\xB4",                 # LATIN CAPITAL LETTER O WITH CIRCUMFLEX
        "\xC3\x95"         => "\xC3\xB5",                 # LATIN CAPITAL LETTER O WITH TILDE
        "\xC3\x96"         => "\xC3\xB6",                 # LATIN CAPITAL LETTER O WITH DIAERESIS
        "\xC3\x98"         => "\xC3\xB8",                 # LATIN CAPITAL LETTER O WITH STROKE
        "\xC3\x99"         => "\xC3\xB9",                 # LATIN CAPITAL LETTER U WITH GRAVE
        "\xC3\x9A"         => "\xC3\xBA",                 # LATIN CAPITAL LETTER U WITH ACUTE
        "\xC3\x9B"         => "\xC3\xBB",                 # LATIN CAPITAL LETTER U WITH CIRCUMFLEX
        "\xC3\x9C"         => "\xC3\xBC",                 # LATIN CAPITAL LETTER U WITH DIAERESIS
        "\xC3\x9D"         => "\xC3\xBD",                 # LATIN CAPITAL LETTER Y WITH ACUTE
        "\xC3\x9E"         => "\xC3\xBE",                 # LATIN CAPITAL LETTER THORN
        "\xC3\x9F"         => "\x73\x73",                 # LATIN SMALL LETTER SHARP S
        "\xC4\x80"         => "\xC4\x81",                 # LATIN CAPITAL LETTER A WITH MACRON
        "\xC4\x82"         => "\xC4\x83",                 # LATIN CAPITAL LETTER A WITH BREVE
        "\xC4\x84"         => "\xC4\x85",                 # LATIN CAPITAL LETTER A WITH OGONEK
        "\xC4\x86"         => "\xC4\x87",                 # LATIN CAPITAL LETTER C WITH ACUTE
        "\xC4\x88"         => "\xC4\x89",                 # LATIN CAPITAL LETTER C WITH CIRCUMFLEX
        "\xC4\x8A"         => "\xC4\x8B",                 # LATIN CAPITAL LETTER C WITH DOT ABOVE
        "\xC4\x8C"         => "\xC4\x8D",                 # LATIN CAPITAL LETTER C WITH CARON
        "\xC4\x8E"         => "\xC4\x8F",                 # LATIN CAPITAL LETTER D WITH CARON
        "\xC4\x90"         => "\xC4\x91",                 # LATIN CAPITAL LETTER D WITH STROKE
        "\xC4\x92"         => "\xC4\x93",                 # LATIN CAPITAL LETTER E WITH MACRON
        "\xC4\x94"         => "\xC4\x95",                 # LATIN CAPITAL LETTER E WITH BREVE
        "\xC4\x96"         => "\xC4\x97",                 # LATIN CAPITAL LETTER E WITH DOT ABOVE
        "\xC4\x98"         => "\xC4\x99",                 # LATIN CAPITAL LETTER E WITH OGONEK
        "\xC4\x9A"         => "\xC4\x9B",                 # LATIN CAPITAL LETTER E WITH CARON
        "\xC4\x9C"         => "\xC4\x9D",                 # LATIN CAPITAL LETTER G WITH CIRCUMFLEX
        "\xC4\x9E"         => "\xC4\x9F",                 # LATIN CAPITAL LETTER G WITH BREVE
        "\xC4\xA0"         => "\xC4\xA1",                 # LATIN CAPITAL LETTER G WITH DOT ABOVE
        "\xC4\xA2"         => "\xC4\xA3",                 # LATIN CAPITAL LETTER G WITH CEDILLA
        "\xC4\xA4"         => "\xC4\xA5",                 # LATIN CAPITAL LETTER H WITH CIRCUMFLEX
        "\xC4\xA6"         => "\xC4\xA7",                 # LATIN CAPITAL LETTER H WITH STROKE
        "\xC4\xA8"         => "\xC4\xA9",                 # LATIN CAPITAL LETTER I WITH TILDE
        "\xC4\xAA"         => "\xC4\xAB",                 # LATIN CAPITAL LETTER I WITH MACRON
        "\xC4\xAC"         => "\xC4\xAD",                 # LATIN CAPITAL LETTER I WITH BREVE
        "\xC4\xAE"         => "\xC4\xAF",                 # LATIN CAPITAL LETTER I WITH OGONEK
        "\xC4\xB0"         => "\x69\xCC\x87",             # LATIN CAPITAL LETTER I WITH DOT ABOVE
        "\xC4\xB2"         => "\xC4\xB3",                 # LATIN CAPITAL LIGATURE IJ
        "\xC4\xB4"         => "\xC4\xB5",                 # LATIN CAPITAL LETTER J WITH CIRCUMFLEX
        "\xC4\xB6"         => "\xC4\xB7",                 # LATIN CAPITAL LETTER K WITH CEDILLA
        "\xC4\xB9"         => "\xC4\xBA",                 # LATIN CAPITAL LETTER L WITH ACUTE
        "\xC4\xBB"         => "\xC4\xBC",                 # LATIN CAPITAL LETTER L WITH CEDILLA
        "\xC4\xBD"         => "\xC4\xBE",                 # LATIN CAPITAL LETTER L WITH CARON
        "\xC4\xBF"         => "\xC5\x80",                 # LATIN CAPITAL LETTER L WITH MIDDLE DOT
        "\xC5\x81"         => "\xC5\x82",                 # LATIN CAPITAL LETTER L WITH STROKE
        "\xC5\x83"         => "\xC5\x84",                 # LATIN CAPITAL LETTER N WITH ACUTE
        "\xC5\x85"         => "\xC5\x86",                 # LATIN CAPITAL LETTER N WITH CEDILLA
        "\xC5\x87"         => "\xC5\x88",                 # LATIN CAPITAL LETTER N WITH CARON
        "\xC5\x89"         => "\xCA\xBC\x6E",             # LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
        "\xC5\x8A"         => "\xC5\x8B",                 # LATIN CAPITAL LETTER ENG
        "\xC5\x8C"         => "\xC5\x8D",                 # LATIN CAPITAL LETTER O WITH MACRON
        "\xC5\x8E"         => "\xC5\x8F",                 # LATIN CAPITAL LETTER O WITH BREVE
        "\xC5\x90"         => "\xC5\x91",                 # LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
        "\xC5\x92"         => "\xC5\x93",                 # LATIN CAPITAL LIGATURE OE
        "\xC5\x94"         => "\xC5\x95",                 # LATIN CAPITAL LETTER R WITH ACUTE
        "\xC5\x96"         => "\xC5\x97",                 # LATIN CAPITAL LETTER R WITH CEDILLA
        "\xC5\x98"         => "\xC5\x99",                 # LATIN CAPITAL LETTER R WITH CARON
        "\xC5\x9A"         => "\xC5\x9B",                 # LATIN CAPITAL LETTER S WITH ACUTE
        "\xC5\x9C"         => "\xC5\x9D",                 # LATIN CAPITAL LETTER S WITH CIRCUMFLEX
        "\xC5\x9E"         => "\xC5\x9F",                 # LATIN CAPITAL LETTER S WITH CEDILLA
        "\xC5\xA0"         => "\xC5\xA1",                 # LATIN CAPITAL LETTER S WITH CARON
        "\xC5\xA2"         => "\xC5\xA3",                 # LATIN CAPITAL LETTER T WITH CEDILLA
        "\xC5\xA4"         => "\xC5\xA5",                 # LATIN CAPITAL LETTER T WITH CARON
        "\xC5\xA6"         => "\xC5\xA7",                 # LATIN CAPITAL LETTER T WITH STROKE
        "\xC5\xA8"         => "\xC5\xA9",                 # LATIN CAPITAL LETTER U WITH TILDE
        "\xC5\xAA"         => "\xC5\xAB",                 # LATIN CAPITAL LETTER U WITH MACRON
        "\xC5\xAC"         => "\xC5\xAD",                 # LATIN CAPITAL LETTER U WITH BREVE
        "\xC5\xAE"         => "\xC5\xAF",                 # LATIN CAPITAL LETTER U WITH RING ABOVE
        "\xC5\xB0"         => "\xC5\xB1",                 # LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
        "\xC5\xB2"         => "\xC5\xB3",                 # LATIN CAPITAL LETTER U WITH OGONEK
        "\xC5\xB4"         => "\xC5\xB5",                 # LATIN CAPITAL LETTER W WITH CIRCUMFLEX
        "\xC5\xB6"         => "\xC5\xB7",                 # LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
        "\xC5\xB8"         => "\xC3\xBF",                 # LATIN CAPITAL LETTER Y WITH DIAERESIS
        "\xC5\xB9"         => "\xC5\xBA",                 # LATIN CAPITAL LETTER Z WITH ACUTE
        "\xC5\xBB"         => "\xC5\xBC",                 # LATIN CAPITAL LETTER Z WITH DOT ABOVE
        "\xC5\xBD"         => "\xC5\xBE",                 # LATIN CAPITAL LETTER Z WITH CARON
        "\xC5\xBF"         => "\x73",                     # LATIN SMALL LETTER LONG S
        "\xC6\x81"         => "\xC9\x93",                 # LATIN CAPITAL LETTER B WITH HOOK
        "\xC6\x82"         => "\xC6\x83",                 # LATIN CAPITAL LETTER B WITH TOPBAR
        "\xC6\x84"         => "\xC6\x85",                 # LATIN CAPITAL LETTER TONE SIX
        "\xC6\x86"         => "\xC9\x94",                 # LATIN CAPITAL LETTER OPEN O
        "\xC6\x87"         => "\xC6\x88",                 # LATIN CAPITAL LETTER C WITH HOOK
        "\xC6\x89"         => "\xC9\x96",                 # LATIN CAPITAL LETTER AFRICAN D
        "\xC6\x8A"         => "\xC9\x97",                 # LATIN CAPITAL LETTER D WITH HOOK
        "\xC6\x8B"         => "\xC6\x8C",                 # LATIN CAPITAL LETTER D WITH TOPBAR
        "\xC6\x8E"         => "\xC7\x9D",                 # LATIN CAPITAL LETTER REVERSED E
        "\xC6\x8F"         => "\xC9\x99",                 # LATIN CAPITAL LETTER SCHWA
        "\xC6\x90"         => "\xC9\x9B",                 # LATIN CAPITAL LETTER OPEN E
        "\xC6\x91"         => "\xC6\x92",                 # LATIN CAPITAL LETTER F WITH HOOK
        "\xC6\x93"         => "\xC9\xA0",                 # LATIN CAPITAL LETTER G WITH HOOK
        "\xC6\x94"         => "\xC9\xA3",                 # LATIN CAPITAL LETTER GAMMA
        "\xC6\x96"         => "\xC9\xA9",                 # LATIN CAPITAL LETTER IOTA
        "\xC6\x97"         => "\xC9\xA8",                 # LATIN CAPITAL LETTER I WITH STROKE
        "\xC6\x98"         => "\xC6\x99",                 # LATIN CAPITAL LETTER K WITH HOOK
        "\xC6\x9C"         => "\xC9\xAF",                 # LATIN CAPITAL LETTER TURNED M
        "\xC6\x9D"         => "\xC9\xB2",                 # LATIN CAPITAL LETTER N WITH LEFT HOOK
        "\xC6\x9F"         => "\xC9\xB5",                 # LATIN CAPITAL LETTER O WITH MIDDLE TILDE
        "\xC6\xA0"         => "\xC6\xA1",                 # LATIN CAPITAL LETTER O WITH HORN
        "\xC6\xA2"         => "\xC6\xA3",                 # LATIN CAPITAL LETTER OI
        "\xC6\xA4"         => "\xC6\xA5",                 # LATIN CAPITAL LETTER P WITH HOOK
        "\xC6\xA6"         => "\xCA\x80",                 # LATIN LETTER YR
        "\xC6\xA7"         => "\xC6\xA8",                 # LATIN CAPITAL LETTER TONE TWO
        "\xC6\xA9"         => "\xCA\x83",                 # LATIN CAPITAL LETTER ESH
        "\xC6\xAC"         => "\xC6\xAD",                 # LATIN CAPITAL LETTER T WITH HOOK
        "\xC6\xAE"         => "\xCA\x88",                 # LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
        "\xC6\xAF"         => "\xC6\xB0",                 # LATIN CAPITAL LETTER U WITH HORN
        "\xC6\xB1"         => "\xCA\x8A",                 # LATIN CAPITAL LETTER UPSILON
        "\xC6\xB2"         => "\xCA\x8B",                 # LATIN CAPITAL LETTER V WITH HOOK
        "\xC6\xB3"         => "\xC6\xB4",                 # LATIN CAPITAL LETTER Y WITH HOOK
        "\xC6\xB5"         => "\xC6\xB6",                 # LATIN CAPITAL LETTER Z WITH STROKE
        "\xC6\xB7"         => "\xCA\x92",                 # LATIN CAPITAL LETTER EZH
        "\xC6\xB8"         => "\xC6\xB9",                 # LATIN CAPITAL LETTER EZH REVERSED
        "\xC6\xBC"         => "\xC6\xBD",                 # LATIN CAPITAL LETTER TONE FIVE
        "\xC7\x84"         => "\xC7\x86",                 # LATIN CAPITAL LETTER DZ WITH CARON
        "\xC7\x85"         => "\xC7\x86",                 # LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
        "\xC7\x87"         => "\xC7\x89",                 # LATIN CAPITAL LETTER LJ
        "\xC7\x88"         => "\xC7\x89",                 # LATIN CAPITAL LETTER L WITH SMALL LETTER J
        "\xC7\x8A"         => "\xC7\x8C",                 # LATIN CAPITAL LETTER NJ
        "\xC7\x8B"         => "\xC7\x8C",                 # LATIN CAPITAL LETTER N WITH SMALL LETTER J
        "\xC7\x8D"         => "\xC7\x8E",                 # LATIN CAPITAL LETTER A WITH CARON
        "\xC7\x8F"         => "\xC7\x90",                 # LATIN CAPITAL LETTER I WITH CARON
        "\xC7\x91"         => "\xC7\x92",                 # LATIN CAPITAL LETTER O WITH CARON
        "\xC7\x93"         => "\xC7\x94",                 # LATIN CAPITAL LETTER U WITH CARON
        "\xC7\x95"         => "\xC7\x96",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
        "\xC7\x97"         => "\xC7\x98",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
        "\xC7\x99"         => "\xC7\x9A",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
        "\xC7\x9B"         => "\xC7\x9C",                 # LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
        "\xC7\x9E"         => "\xC7\x9F",                 # LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
        "\xC7\xA0"         => "\xC7\xA1",                 # LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
        "\xC7\xA2"         => "\xC7\xA3",                 # LATIN CAPITAL LETTER AE WITH MACRON
        "\xC7\xA4"         => "\xC7\xA5",                 # LATIN CAPITAL LETTER G WITH STROKE
        "\xC7\xA6"         => "\xC7\xA7",                 # LATIN CAPITAL LETTER G WITH CARON
        "\xC7\xA8"         => "\xC7\xA9",                 # LATIN CAPITAL LETTER K WITH CARON
        "\xC7\xAA"         => "\xC7\xAB",                 # LATIN CAPITAL LETTER O WITH OGONEK
        "\xC7\xAC"         => "\xC7\xAD",                 # LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
        "\xC7\xAE"         => "\xC7\xAF",                 # LATIN CAPITAL LETTER EZH WITH CARON
        "\xC7\xB0"         => "\x6A\xCC\x8C",             # LATIN SMALL LETTER J WITH CARON
        "\xC7\xB1"         => "\xC7\xB3",                 # LATIN CAPITAL LETTER DZ
        "\xC7\xB2"         => "\xC7\xB3",                 # LATIN CAPITAL LETTER D WITH SMALL LETTER Z
        "\xC7\xB4"         => "\xC7\xB5",                 # LATIN CAPITAL LETTER G WITH ACUTE
        "\xC7\xB6"         => "\xC6\x95",                 # LATIN CAPITAL LETTER HWAIR
        "\xC7\xB7"         => "\xC6\xBF",                 # LATIN CAPITAL LETTER WYNN
        "\xC7\xB8"         => "\xC7\xB9",                 # LATIN CAPITAL LETTER N WITH GRAVE
        "\xC7\xBA"         => "\xC7\xBB",                 # LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
        "\xC7\xBC"         => "\xC7\xBD",                 # LATIN CAPITAL LETTER AE WITH ACUTE
        "\xC7\xBE"         => "\xC7\xBF",                 # LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
        "\xC8\x80"         => "\xC8\x81",                 # LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
        "\xC8\x82"         => "\xC8\x83",                 # LATIN CAPITAL LETTER A WITH INVERTED BREVE
        "\xC8\x84"         => "\xC8\x85",                 # LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
        "\xC8\x86"         => "\xC8\x87",                 # LATIN CAPITAL LETTER E WITH INVERTED BREVE
        "\xC8\x88"         => "\xC8\x89",                 # LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
        "\xC8\x8A"         => "\xC8\x8B",                 # LATIN CAPITAL LETTER I WITH INVERTED BREVE
        "\xC8\x8C"         => "\xC8\x8D",                 # LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
        "\xC8\x8E"         => "\xC8\x8F",                 # LATIN CAPITAL LETTER O WITH INVERTED BREVE
        "\xC8\x90"         => "\xC8\x91",                 # LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
        "\xC8\x92"         => "\xC8\x93",                 # LATIN CAPITAL LETTER R WITH INVERTED BREVE
        "\xC8\x94"         => "\xC8\x95",                 # LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
        "\xC8\x96"         => "\xC8\x97",                 # LATIN CAPITAL LETTER U WITH INVERTED BREVE
        "\xC8\x98"         => "\xC8\x99",                 # LATIN CAPITAL LETTER S WITH COMMA BELOW
        "\xC8\x9A"         => "\xC8\x9B",                 # LATIN CAPITAL LETTER T WITH COMMA BELOW
        "\xC8\x9C"         => "\xC8\x9D",                 # LATIN CAPITAL LETTER YOGH
        "\xC8\x9E"         => "\xC8\x9F",                 # LATIN CAPITAL LETTER H WITH CARON
        "\xC8\xA0"         => "\xC6\x9E",                 # LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
        "\xC8\xA2"         => "\xC8\xA3",                 # LATIN CAPITAL LETTER OU
        "\xC8\xA4"         => "\xC8\xA5",                 # LATIN CAPITAL LETTER Z WITH HOOK
        "\xC8\xA6"         => "\xC8\xA7",                 # LATIN CAPITAL LETTER A WITH DOT ABOVE
        "\xC8\xA8"         => "\xC8\xA9",                 # LATIN CAPITAL LETTER E WITH CEDILLA
        "\xC8\xAA"         => "\xC8\xAB",                 # LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
        "\xC8\xAC"         => "\xC8\xAD",                 # LATIN CAPITAL LETTER O WITH TILDE AND MACRON
        "\xC8\xAE"         => "\xC8\xAF",                 # LATIN CAPITAL LETTER O WITH DOT ABOVE
        "\xC8\xB0"         => "\xC8\xB1",                 # LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
        "\xC8\xB2"         => "\xC8\xB3",                 # LATIN CAPITAL LETTER Y WITH MACRON
        "\xC8\xBA"         => "\xE2\xB1\xA5",             # LATIN CAPITAL LETTER A WITH STROKE
        "\xC8\xBB"         => "\xC8\xBC",                 # LATIN CAPITAL LETTER C WITH STROKE
        "\xC8\xBD"         => "\xC6\x9A",                 # LATIN CAPITAL LETTER L WITH BAR
        "\xC8\xBE"         => "\xE2\xB1\xA6",             # LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
        "\xC9\x81"         => "\xC9\x82",                 # LATIN CAPITAL LETTER GLOTTAL STOP
        "\xC9\x83"         => "\xC6\x80",                 # LATIN CAPITAL LETTER B WITH STROKE
        "\xC9\x84"         => "\xCA\x89",                 # LATIN CAPITAL LETTER U BAR
        "\xC9\x85"         => "\xCA\x8C",                 # LATIN CAPITAL LETTER TURNED V
        "\xC9\x86"         => "\xC9\x87",                 # LATIN CAPITAL LETTER E WITH STROKE
        "\xC9\x88"         => "\xC9\x89",                 # LATIN CAPITAL LETTER J WITH STROKE
        "\xC9\x8A"         => "\xC9\x8B",                 # LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
        "\xC9\x8C"         => "\xC9\x8D",                 # LATIN CAPITAL LETTER R WITH STROKE
        "\xC9\x8E"         => "\xC9\x8F",                 # LATIN CAPITAL LETTER Y WITH STROKE
        "\xCD\x85"         => "\xCE\xB9",                 # COMBINING GREEK YPOGEGRAMMENI
        "\xCD\xB0"         => "\xCD\xB1",                 # GREEK CAPITAL LETTER HETA
        "\xCD\xB2"         => "\xCD\xB3",                 # GREEK CAPITAL LETTER ARCHAIC SAMPI
        "\xCD\xB6"         => "\xCD\xB7",                 # GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
        "\xCD\xBF"         => "\xCF\xB3",                 # GREEK CAPITAL LETTER YOT
        "\xCE\x86"         => "\xCE\xAC",                 # GREEK CAPITAL LETTER ALPHA WITH TONOS
        "\xCE\x88"         => "\xCE\xAD",                 # GREEK CAPITAL LETTER EPSILON WITH TONOS
        "\xCE\x89"         => "\xCE\xAE",                 # GREEK CAPITAL LETTER ETA WITH TONOS
        "\xCE\x8A"         => "\xCE\xAF",                 # GREEK CAPITAL LETTER IOTA WITH TONOS
        "\xCE\x8C"         => "\xCF\x8C",                 # GREEK CAPITAL LETTER OMICRON WITH TONOS
        "\xCE\x8E"         => "\xCF\x8D",                 # GREEK CAPITAL LETTER UPSILON WITH TONOS
        "\xCE\x8F"         => "\xCF\x8E",                 # GREEK CAPITAL LETTER OMEGA WITH TONOS
        "\xCE\x90"         => "\xCE\xB9\xCC\x88\xCC\x81", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
        "\xCE\x91"         => "\xCE\xB1",                 # GREEK CAPITAL LETTER ALPHA
        "\xCE\x92"         => "\xCE\xB2",                 # GREEK CAPITAL LETTER BETA
        "\xCE\x93"         => "\xCE\xB3",                 # GREEK CAPITAL LETTER GAMMA
        "\xCE\x94"         => "\xCE\xB4",                 # GREEK CAPITAL LETTER DELTA
        "\xCE\x95"         => "\xCE\xB5",                 # GREEK CAPITAL LETTER EPSILON
        "\xCE\x96"         => "\xCE\xB6",                 # GREEK CAPITAL LETTER ZETA
        "\xCE\x97"         => "\xCE\xB7",                 # GREEK CAPITAL LETTER ETA
        "\xCE\x98"         => "\xCE\xB8",                 # GREEK CAPITAL LETTER THETA
        "\xCE\x99"         => "\xCE\xB9",                 # GREEK CAPITAL LETTER IOTA
        "\xCE\x9A"         => "\xCE\xBA",                 # GREEK CAPITAL LETTER KAPPA
        "\xCE\x9B"         => "\xCE\xBB",                 # GREEK CAPITAL LETTER LAMDA
        "\xCE\x9C"         => "\xCE\xBC",                 # GREEK CAPITAL LETTER MU
        "\xCE\x9D"         => "\xCE\xBD",                 # GREEK CAPITAL LETTER NU
        "\xCE\x9E"         => "\xCE\xBE",                 # GREEK CAPITAL LETTER XI
        "\xCE\x9F"         => "\xCE\xBF",                 # GREEK CAPITAL LETTER OMICRON
        "\xCE\xA0"         => "\xCF\x80",                 # GREEK CAPITAL LETTER PI
        "\xCE\xA1"         => "\xCF\x81",                 # GREEK CAPITAL LETTER RHO
        "\xCE\xA3"         => "\xCF\x83",                 # GREEK CAPITAL LETTER SIGMA
        "\xCE\xA4"         => "\xCF\x84",                 # GREEK CAPITAL LETTER TAU
        "\xCE\xA5"         => "\xCF\x85",                 # GREEK CAPITAL LETTER UPSILON
        "\xCE\xA6"         => "\xCF\x86",                 # GREEK CAPITAL LETTER PHI
        "\xCE\xA7"         => "\xCF\x87",                 # GREEK CAPITAL LETTER CHI
        "\xCE\xA8"         => "\xCF\x88",                 # GREEK CAPITAL LETTER PSI
        "\xCE\xA9"         => "\xCF\x89",                 # GREEK CAPITAL LETTER OMEGA
        "\xCE\xAA"         => "\xCF\x8A",                 # GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
        "\xCE\xAB"         => "\xCF\x8B",                 # GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
        "\xCE\xB0"         => "\xCF\x85\xCC\x88\xCC\x81", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
        "\xCF\x82"         => "\xCF\x83",                 # GREEK SMALL LETTER FINAL SIGMA
        "\xCF\x8F"         => "\xCF\x97",                 # GREEK CAPITAL KAI SYMBOL
        "\xCF\x90"         => "\xCE\xB2",                 # GREEK BETA SYMBOL
        "\xCF\x91"         => "\xCE\xB8",                 # GREEK THETA SYMBOL
        "\xCF\x95"         => "\xCF\x86",                 # GREEK PHI SYMBOL
        "\xCF\x96"         => "\xCF\x80",                 # GREEK PI SYMBOL
        "\xCF\x98"         => "\xCF\x99",                 # GREEK LETTER ARCHAIC KOPPA
        "\xCF\x9A"         => "\xCF\x9B",                 # GREEK LETTER STIGMA
        "\xCF\x9C"         => "\xCF\x9D",                 # GREEK LETTER DIGAMMA
        "\xCF\x9E"         => "\xCF\x9F",                 # GREEK LETTER KOPPA
        "\xCF\xA0"         => "\xCF\xA1",                 # GREEK LETTER SAMPI
        "\xCF\xA2"         => "\xCF\xA3",                 # COPTIC CAPITAL LETTER SHEI
        "\xCF\xA4"         => "\xCF\xA5",                 # COPTIC CAPITAL LETTER FEI
        "\xCF\xA6"         => "\xCF\xA7",                 # COPTIC CAPITAL LETTER KHEI
        "\xCF\xA8"         => "\xCF\xA9",                 # COPTIC CAPITAL LETTER HORI
        "\xCF\xAA"         => "\xCF\xAB",                 # COPTIC CAPITAL LETTER GANGIA
        "\xCF\xAC"         => "\xCF\xAD",                 # COPTIC CAPITAL LETTER SHIMA
        "\xCF\xAE"         => "\xCF\xAF",                 # COPTIC CAPITAL LETTER DEI
        "\xCF\xB0"         => "\xCE\xBA",                 # GREEK KAPPA SYMBOL
        "\xCF\xB1"         => "\xCF\x81",                 # GREEK RHO SYMBOL
        "\xCF\xB4"         => "\xCE\xB8",                 # GREEK CAPITAL THETA SYMBOL
        "\xCF\xB5"         => "\xCE\xB5",                 # GREEK LUNATE EPSILON SYMBOL
        "\xCF\xB7"         => "\xCF\xB8",                 # GREEK CAPITAL LETTER SHO
        "\xCF\xB9"         => "\xCF\xB2",                 # GREEK CAPITAL LUNATE SIGMA SYMBOL
        "\xCF\xBA"         => "\xCF\xBB",                 # GREEK CAPITAL LETTER SAN
        "\xCF\xBD"         => "\xCD\xBB",                 # GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
        "\xCF\xBE"         => "\xCD\xBC",                 # GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
        "\xCF\xBF"         => "\xCD\xBD",                 # GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
        "\xD0\x80"         => "\xD1\x90",                 # CYRILLIC CAPITAL LETTER IE WITH GRAVE
        "\xD0\x81"         => "\xD1\x91",                 # CYRILLIC CAPITAL LETTER IO
        "\xD0\x82"         => "\xD1\x92",                 # CYRILLIC CAPITAL LETTER DJE
        "\xD0\x83"         => "\xD1\x93",                 # CYRILLIC CAPITAL LETTER GJE
        "\xD0\x84"         => "\xD1\x94",                 # CYRILLIC CAPITAL LETTER UKRAINIAN IE
        "\xD0\x85"         => "\xD1\x95",                 # CYRILLIC CAPITAL LETTER DZE
        "\xD0\x86"         => "\xD1\x96",                 # CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
        "\xD0\x87"         => "\xD1\x97",                 # CYRILLIC CAPITAL LETTER YI
        "\xD0\x88"         => "\xD1\x98",                 # CYRILLIC CAPITAL LETTER JE
        "\xD0\x89"         => "\xD1\x99",                 # CYRILLIC CAPITAL LETTER LJE
        "\xD0\x8A"         => "\xD1\x9A",                 # CYRILLIC CAPITAL LETTER NJE
        "\xD0\x8B"         => "\xD1\x9B",                 # CYRILLIC CAPITAL LETTER TSHE
        "\xD0\x8C"         => "\xD1\x9C",                 # CYRILLIC CAPITAL LETTER KJE
        "\xD0\x8D"         => "\xD1\x9D",                 # CYRILLIC CAPITAL LETTER I WITH GRAVE
        "\xD0\x8E"         => "\xD1\x9E",                 # CYRILLIC CAPITAL LETTER SHORT U
        "\xD0\x8F"         => "\xD1\x9F",                 # CYRILLIC CAPITAL LETTER DZHE
        "\xD0\x90"         => "\xD0\xB0",                 # CYRILLIC CAPITAL LETTER A
        "\xD0\x91"         => "\xD0\xB1",                 # CYRILLIC CAPITAL LETTER BE
        "\xD0\x92"         => "\xD0\xB2",                 # CYRILLIC CAPITAL LETTER VE
        "\xD0\x93"         => "\xD0\xB3",                 # CYRILLIC CAPITAL LETTER GHE
        "\xD0\x94"         => "\xD0\xB4",                 # CYRILLIC CAPITAL LETTER DE
        "\xD0\x95"         => "\xD0\xB5",                 # CYRILLIC CAPITAL LETTER IE
        "\xD0\x96"         => "\xD0\xB6",                 # CYRILLIC CAPITAL LETTER ZHE
        "\xD0\x97"         => "\xD0\xB7",                 # CYRILLIC CAPITAL LETTER ZE
        "\xD0\x98"         => "\xD0\xB8",                 # CYRILLIC CAPITAL LETTER I
        "\xD0\x99"         => "\xD0\xB9",                 # CYRILLIC CAPITAL LETTER SHORT I
        "\xD0\x9A"         => "\xD0\xBA",                 # CYRILLIC CAPITAL LETTER KA
        "\xD0\x9B"         => "\xD0\xBB",                 # CYRILLIC CAPITAL LETTER EL
        "\xD0\x9C"         => "\xD0\xBC",                 # CYRILLIC CAPITAL LETTER EM
        "\xD0\x9D"         => "\xD0\xBD",                 # CYRILLIC CAPITAL LETTER EN
        "\xD0\x9E"         => "\xD0\xBE",                 # CYRILLIC CAPITAL LETTER O
        "\xD0\x9F"         => "\xD0\xBF",                 # CYRILLIC CAPITAL LETTER PE
        "\xD0\xA0"         => "\xD1\x80",                 # CYRILLIC CAPITAL LETTER ER
        "\xD0\xA1"         => "\xD1\x81",                 # CYRILLIC CAPITAL LETTER ES
        "\xD0\xA2"         => "\xD1\x82",                 # CYRILLIC CAPITAL LETTER TE
        "\xD0\xA3"         => "\xD1\x83",                 # CYRILLIC CAPITAL LETTER U
        "\xD0\xA4"         => "\xD1\x84",                 # CYRILLIC CAPITAL LETTER EF
        "\xD0\xA5"         => "\xD1\x85",                 # CYRILLIC CAPITAL LETTER HA
        "\xD0\xA6"         => "\xD1\x86",                 # CYRILLIC CAPITAL LETTER TSE
        "\xD0\xA7"         => "\xD1\x87",                 # CYRILLIC CAPITAL LETTER CHE
        "\xD0\xA8"         => "\xD1\x88",                 # CYRILLIC CAPITAL LETTER SHA
        "\xD0\xA9"         => "\xD1\x89",                 # CYRILLIC CAPITAL LETTER SHCHA
        "\xD0\xAA"         => "\xD1\x8A",                 # CYRILLIC CAPITAL LETTER HARD SIGN
        "\xD0\xAB"         => "\xD1\x8B",                 # CYRILLIC CAPITAL LETTER YERU
        "\xD0\xAC"         => "\xD1\x8C",                 # CYRILLIC CAPITAL LETTER SOFT SIGN
        "\xD0\xAD"         => "\xD1\x8D",                 # CYRILLIC CAPITAL LETTER E
        "\xD0\xAE"         => "\xD1\x8E",                 # CYRILLIC CAPITAL LETTER YU
        "\xD0\xAF"         => "\xD1\x8F",                 # CYRILLIC CAPITAL LETTER YA
        "\xD1\xA0"         => "\xD1\xA1",                 # CYRILLIC CAPITAL LETTER OMEGA
        "\xD1\xA2"         => "\xD1\xA3",                 # CYRILLIC CAPITAL LETTER YAT
        "\xD1\xA4"         => "\xD1\xA5",                 # CYRILLIC CAPITAL LETTER IOTIFIED E
        "\xD1\xA6"         => "\xD1\xA7",                 # CYRILLIC CAPITAL LETTER LITTLE YUS
        "\xD1\xA8"         => "\xD1\xA9",                 # CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
        "\xD1\xAA"         => "\xD1\xAB",                 # CYRILLIC CAPITAL LETTER BIG YUS
        "\xD1\xAC"         => "\xD1\xAD",                 # CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
        "\xD1\xAE"         => "\xD1\xAF",                 # CYRILLIC CAPITAL LETTER KSI
        "\xD1\xB0"         => "\xD1\xB1",                 # CYRILLIC CAPITAL LETTER PSI
        "\xD1\xB2"         => "\xD1\xB3",                 # CYRILLIC CAPITAL LETTER FITA
        "\xD1\xB4"         => "\xD1\xB5",                 # CYRILLIC CAPITAL LETTER IZHITSA
        "\xD1\xB6"         => "\xD1\xB7",                 # CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
        "\xD1\xB8"         => "\xD1\xB9",                 # CYRILLIC CAPITAL LETTER UK
        "\xD1\xBA"         => "\xD1\xBB",                 # CYRILLIC CAPITAL LETTER ROUND OMEGA
        "\xD1\xBC"         => "\xD1\xBD",                 # CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
        "\xD1\xBE"         => "\xD1\xBF",                 # CYRILLIC CAPITAL LETTER OT
        "\xD2\x80"         => "\xD2\x81",                 # CYRILLIC CAPITAL LETTER KOPPA
        "\xD2\x8A"         => "\xD2\x8B",                 # CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
        "\xD2\x8C"         => "\xD2\x8D",                 # CYRILLIC CAPITAL LETTER SEMISOFT SIGN
        "\xD2\x8E"         => "\xD2\x8F",                 # CYRILLIC CAPITAL LETTER ER WITH TICK
        "\xD2\x90"         => "\xD2\x91",                 # CYRILLIC CAPITAL LETTER GHE WITH UPTURN
        "\xD2\x92"         => "\xD2\x93",                 # CYRILLIC CAPITAL LETTER GHE WITH STROKE
        "\xD2\x94"         => "\xD2\x95",                 # CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
        "\xD2\x96"         => "\xD2\x97",                 # CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
        "\xD2\x98"         => "\xD2\x99",                 # CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
        "\xD2\x9A"         => "\xD2\x9B",                 # CYRILLIC CAPITAL LETTER KA WITH DESCENDER
        "\xD2\x9C"         => "\xD2\x9D",                 # CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
        "\xD2\x9E"         => "\xD2\x9F",                 # CYRILLIC CAPITAL LETTER KA WITH STROKE
        "\xD2\xA0"         => "\xD2\xA1",                 # CYRILLIC CAPITAL LETTER BASHKIR KA
        "\xD2\xA2"         => "\xD2\xA3",                 # CYRILLIC CAPITAL LETTER EN WITH DESCENDER
        "\xD2\xA4"         => "\xD2\xA5",                 # CYRILLIC CAPITAL LIGATURE EN GHE
        "\xD2\xA6"         => "\xD2\xA7",                 # CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
        "\xD2\xA8"         => "\xD2\xA9",                 # CYRILLIC CAPITAL LETTER ABKHASIAN HA
        "\xD2\xAA"         => "\xD2\xAB",                 # CYRILLIC CAPITAL LETTER ES WITH DESCENDER
        "\xD2\xAC"         => "\xD2\xAD",                 # CYRILLIC CAPITAL LETTER TE WITH DESCENDER
        "\xD2\xAE"         => "\xD2\xAF",                 # CYRILLIC CAPITAL LETTER STRAIGHT U
        "\xD2\xB0"         => "\xD2\xB1",                 # CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
        "\xD2\xB2"         => "\xD2\xB3",                 # CYRILLIC CAPITAL LETTER HA WITH DESCENDER
        "\xD2\xB4"         => "\xD2\xB5",                 # CYRILLIC CAPITAL LIGATURE TE TSE
        "\xD2\xB6"         => "\xD2\xB7",                 # CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
        "\xD2\xB8"         => "\xD2\xB9",                 # CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
        "\xD2\xBA"         => "\xD2\xBB",                 # CYRILLIC CAPITAL LETTER SHHA
        "\xD2\xBC"         => "\xD2\xBD",                 # CYRILLIC CAPITAL LETTER ABKHASIAN CHE
        "\xD2\xBE"         => "\xD2\xBF",                 # CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
        "\xD3\x80"         => "\xD3\x8F",                 # CYRILLIC LETTER PALOCHKA
        "\xD3\x81"         => "\xD3\x82",                 # CYRILLIC CAPITAL LETTER ZHE WITH BREVE
        "\xD3\x83"         => "\xD3\x84",                 # CYRILLIC CAPITAL LETTER KA WITH HOOK
        "\xD3\x85"         => "\xD3\x86",                 # CYRILLIC CAPITAL LETTER EL WITH TAIL
        "\xD3\x87"         => "\xD3\x88",                 # CYRILLIC CAPITAL LETTER EN WITH HOOK
        "\xD3\x89"         => "\xD3\x8A",                 # CYRILLIC CAPITAL LETTER EN WITH TAIL
        "\xD3\x8B"         => "\xD3\x8C",                 # CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
        "\xD3\x8D"         => "\xD3\x8E",                 # CYRILLIC CAPITAL LETTER EM WITH TAIL
        "\xD3\x90"         => "\xD3\x91",                 # CYRILLIC CAPITAL LETTER A WITH BREVE
        "\xD3\x92"         => "\xD3\x93",                 # CYRILLIC CAPITAL LETTER A WITH DIAERESIS
        "\xD3\x94"         => "\xD3\x95",                 # CYRILLIC CAPITAL LIGATURE A IE
        "\xD3\x96"         => "\xD3\x97",                 # CYRILLIC CAPITAL LETTER IE WITH BREVE
        "\xD3\x98"         => "\xD3\x99",                 # CYRILLIC CAPITAL LETTER SCHWA
        "\xD3\x9A"         => "\xD3\x9B",                 # CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
        "\xD3\x9C"         => "\xD3\x9D",                 # CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
        "\xD3\x9E"         => "\xD3\x9F",                 # CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
        "\xD3\xA0"         => "\xD3\xA1",                 # CYRILLIC CAPITAL LETTER ABKHASIAN DZE
        "\xD3\xA2"         => "\xD3\xA3",                 # CYRILLIC CAPITAL LETTER I WITH MACRON
        "\xD3\xA4"         => "\xD3\xA5",                 # CYRILLIC CAPITAL LETTER I WITH DIAERESIS
        "\xD3\xA6"         => "\xD3\xA7",                 # CYRILLIC CAPITAL LETTER O WITH DIAERESIS
        "\xD3\xA8"         => "\xD3\xA9",                 # CYRILLIC CAPITAL LETTER BARRED O
        "\xD3\xAA"         => "\xD3\xAB",                 # CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
        "\xD3\xAC"         => "\xD3\xAD",                 # CYRILLIC CAPITAL LETTER E WITH DIAERESIS
        "\xD3\xAE"         => "\xD3\xAF",                 # CYRILLIC CAPITAL LETTER U WITH MACRON
        "\xD3\xB0"         => "\xD3\xB1",                 # CYRILLIC CAPITAL LETTER U WITH DIAERESIS
        "\xD3\xB2"         => "\xD3\xB3",                 # CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
        "\xD3\xB4"         => "\xD3\xB5",                 # CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
        "\xD3\xB6"         => "\xD3\xB7",                 # CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
        "\xD3\xB8"         => "\xD3\xB9",                 # CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
        "\xD3\xBA"         => "\xD3\xBB",                 # CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
        "\xD3\xBC"         => "\xD3\xBD",                 # CYRILLIC CAPITAL LETTER HA WITH HOOK
        "\xD3\xBE"         => "\xD3\xBF",                 # CYRILLIC CAPITAL LETTER HA WITH STROKE
        "\xD4\x80"         => "\xD4\x81",                 # CYRILLIC CAPITAL LETTER KOMI DE
        "\xD4\x82"         => "\xD4\x83",                 # CYRILLIC CAPITAL LETTER KOMI DJE
        "\xD4\x84"         => "\xD4\x85",                 # CYRILLIC CAPITAL LETTER KOMI ZJE
        "\xD4\x86"         => "\xD4\x87",                 # CYRILLIC CAPITAL LETTER KOMI DZJE
        "\xD4\x88"         => "\xD4\x89",                 # CYRILLIC CAPITAL LETTER KOMI LJE
        "\xD4\x8A"         => "\xD4\x8B",                 # CYRILLIC CAPITAL LETTER KOMI NJE
        "\xD4\x8C"         => "\xD4\x8D",                 # CYRILLIC CAPITAL LETTER KOMI SJE
        "\xD4\x8E"         => "\xD4\x8F",                 # CYRILLIC CAPITAL LETTER KOMI TJE
        "\xD4\x90"         => "\xD4\x91",                 # CYRILLIC CAPITAL LETTER REVERSED ZE
        "\xD4\x92"         => "\xD4\x93",                 # CYRILLIC CAPITAL LETTER EL WITH HOOK
        "\xD4\x94"         => "\xD4\x95",                 # CYRILLIC CAPITAL LETTER LHA
        "\xD4\x96"         => "\xD4\x97",                 # CYRILLIC CAPITAL LETTER RHA
        "\xD4\x98"         => "\xD4\x99",                 # CYRILLIC CAPITAL LETTER YAE
        "\xD4\x9A"         => "\xD4\x9B",                 # CYRILLIC CAPITAL LETTER QA
        "\xD4\x9C"         => "\xD4\x9D",                 # CYRILLIC CAPITAL LETTER WE
        "\xD4\x9E"         => "\xD4\x9F",                 # CYRILLIC CAPITAL LETTER ALEUT KA
        "\xD4\xA0"         => "\xD4\xA1",                 # CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
        "\xD4\xA2"         => "\xD4\xA3",                 # CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
        "\xD4\xA4"         => "\xD4\xA5",                 # CYRILLIC CAPITAL LETTER PE WITH DESCENDER
        "\xD4\xA6"         => "\xD4\xA7",                 # CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
        "\xD4\xA8"         => "\xD4\xA9",                 # CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
        "\xD4\xAA"         => "\xD4\xAB",                 # CYRILLIC CAPITAL LETTER DZZHE
        "\xD4\xAC"         => "\xD4\xAD",                 # CYRILLIC CAPITAL LETTER DCHE
        "\xD4\xAE"         => "\xD4\xAF",                 # CYRILLIC CAPITAL LETTER EL WITH DESCENDER
        "\xD4\xB1"         => "\xD5\xA1",                 # ARMENIAN CAPITAL LETTER AYB
        "\xD4\xB2"         => "\xD5\xA2",                 # ARMENIAN CAPITAL LETTER BEN
        "\xD4\xB3"         => "\xD5\xA3",                 # ARMENIAN CAPITAL LETTER GIM
        "\xD4\xB4"         => "\xD5\xA4",                 # ARMENIAN CAPITAL LETTER DA
        "\xD4\xB5"         => "\xD5\xA5",                 # ARMENIAN CAPITAL LETTER ECH
        "\xD4\xB6"         => "\xD5\xA6",                 # ARMENIAN CAPITAL LETTER ZA
        "\xD4\xB7"         => "\xD5\xA7",                 # ARMENIAN CAPITAL LETTER EH
        "\xD4\xB8"         => "\xD5\xA8",                 # ARMENIAN CAPITAL LETTER ET
        "\xD4\xB9"         => "\xD5\xA9",                 # ARMENIAN CAPITAL LETTER TO
        "\xD4\xBA"         => "\xD5\xAA",                 # ARMENIAN CAPITAL LETTER ZHE
        "\xD4\xBB"         => "\xD5\xAB",                 # ARMENIAN CAPITAL LETTER INI
        "\xD4\xBC"         => "\xD5\xAC",                 # ARMENIAN CAPITAL LETTER LIWN
        "\xD4\xBD"         => "\xD5\xAD",                 # ARMENIAN CAPITAL LETTER XEH
        "\xD4\xBE"         => "\xD5\xAE",                 # ARMENIAN CAPITAL LETTER CA
        "\xD4\xBF"         => "\xD5\xAF",                 # ARMENIAN CAPITAL LETTER KEN
        "\xD5\x80"         => "\xD5\xB0",                 # ARMENIAN CAPITAL LETTER HO
        "\xD5\x81"         => "\xD5\xB1",                 # ARMENIAN CAPITAL LETTER JA
        "\xD5\x82"         => "\xD5\xB2",                 # ARMENIAN CAPITAL LETTER GHAD
        "\xD5\x83"         => "\xD5\xB3",                 # ARMENIAN CAPITAL LETTER CHEH
        "\xD5\x84"         => "\xD5\xB4",                 # ARMENIAN CAPITAL LETTER MEN
        "\xD5\x85"         => "\xD5\xB5",                 # ARMENIAN CAPITAL LETTER YI
        "\xD5\x86"         => "\xD5\xB6",                 # ARMENIAN CAPITAL LETTER NOW
        "\xD5\x87"         => "\xD5\xB7",                 # ARMENIAN CAPITAL LETTER SHA
        "\xD5\x88"         => "\xD5\xB8",                 # ARMENIAN CAPITAL LETTER VO
        "\xD5\x89"         => "\xD5\xB9",                 # ARMENIAN CAPITAL LETTER CHA
        "\xD5\x8A"         => "\xD5\xBA",                 # ARMENIAN CAPITAL LETTER PEH
        "\xD5\x8B"         => "\xD5\xBB",                 # ARMENIAN CAPITAL LETTER JHEH
        "\xD5\x8C"         => "\xD5\xBC",                 # ARMENIAN CAPITAL LETTER RA
        "\xD5\x8D"         => "\xD5\xBD",                 # ARMENIAN CAPITAL LETTER SEH
        "\xD5\x8E"         => "\xD5\xBE",                 # ARMENIAN CAPITAL LETTER VEW
        "\xD5\x8F"         => "\xD5\xBF",                 # ARMENIAN CAPITAL LETTER TIWN
        "\xD5\x90"         => "\xD6\x80",                 # ARMENIAN CAPITAL LETTER REH
        "\xD5\x91"         => "\xD6\x81",                 # ARMENIAN CAPITAL LETTER CO
        "\xD5\x92"         => "\xD6\x82",                 # ARMENIAN CAPITAL LETTER YIWN
        "\xD5\x93"         => "\xD6\x83",                 # ARMENIAN CAPITAL LETTER PIWR
        "\xD5\x94"         => "\xD6\x84",                 # ARMENIAN CAPITAL LETTER KEH
        "\xD5\x95"         => "\xD6\x85",                 # ARMENIAN CAPITAL LETTER OH
        "\xD5\x96"         => "\xD6\x86",                 # ARMENIAN CAPITAL LETTER FEH
        "\xD6\x87"         => "\xD5\xA5\xD6\x82",         # ARMENIAN SMALL LIGATURE ECH YIWN
        "\xE1\x82\xA0"     => "\xE2\xB4\x80",             # GEORGIAN CAPITAL LETTER AN
        "\xE1\x82\xA1"     => "\xE2\xB4\x81",             # GEORGIAN CAPITAL LETTER BAN
        "\xE1\x82\xA2"     => "\xE2\xB4\x82",             # GEORGIAN CAPITAL LETTER GAN
        "\xE1\x82\xA3"     => "\xE2\xB4\x83",             # GEORGIAN CAPITAL LETTER DON
        "\xE1\x82\xA4"     => "\xE2\xB4\x84",             # GEORGIAN CAPITAL LETTER EN
        "\xE1\x82\xA5"     => "\xE2\xB4\x85",             # GEORGIAN CAPITAL LETTER VIN
        "\xE1\x82\xA6"     => "\xE2\xB4\x86",             # GEORGIAN CAPITAL LETTER ZEN
        "\xE1\x82\xA7"     => "\xE2\xB4\x87",             # GEORGIAN CAPITAL LETTER TAN
        "\xE1\x82\xA8"     => "\xE2\xB4\x88",             # GEORGIAN CAPITAL LETTER IN
        "\xE1\x82\xA9"     => "\xE2\xB4\x89",             # GEORGIAN CAPITAL LETTER KAN
        "\xE1\x82\xAA"     => "\xE2\xB4\x8A",             # GEORGIAN CAPITAL LETTER LAS
        "\xE1\x82\xAB"     => "\xE2\xB4\x8B",             # GEORGIAN CAPITAL LETTER MAN
        "\xE1\x82\xAC"     => "\xE2\xB4\x8C",             # GEORGIAN CAPITAL LETTER NAR
        "\xE1\x82\xAD"     => "\xE2\xB4\x8D",             # GEORGIAN CAPITAL LETTER ON
        "\xE1\x82\xAE"     => "\xE2\xB4\x8E",             # GEORGIAN CAPITAL LETTER PAR
        "\xE1\x82\xAF"     => "\xE2\xB4\x8F",             # GEORGIAN CAPITAL LETTER ZHAR
        "\xE1\x82\xB0"     => "\xE2\xB4\x90",             # GEORGIAN CAPITAL LETTER RAE
        "\xE1\x82\xB1"     => "\xE2\xB4\x91",             # GEORGIAN CAPITAL LETTER SAN
        "\xE1\x82\xB2"     => "\xE2\xB4\x92",             # GEORGIAN CAPITAL LETTER TAR
        "\xE1\x82\xB3"     => "\xE2\xB4\x93",             # GEORGIAN CAPITAL LETTER UN
        "\xE1\x82\xB4"     => "\xE2\xB4\x94",             # GEORGIAN CAPITAL LETTER PHAR
        "\xE1\x82\xB5"     => "\xE2\xB4\x95",             # GEORGIAN CAPITAL LETTER KHAR
        "\xE1\x82\xB6"     => "\xE2\xB4\x96",             # GEORGIAN CAPITAL LETTER GHAN
        "\xE1\x82\xB7"     => "\xE2\xB4\x97",             # GEORGIAN CAPITAL LETTER QAR
        "\xE1\x82\xB8"     => "\xE2\xB4\x98",             # GEORGIAN CAPITAL LETTER SHIN
        "\xE1\x82\xB9"     => "\xE2\xB4\x99",             # GEORGIAN CAPITAL LETTER CHIN
        "\xE1\x82\xBA"     => "\xE2\xB4\x9A",             # GEORGIAN CAPITAL LETTER CAN
        "\xE1\x82\xBB"     => "\xE2\xB4\x9B",             # GEORGIAN CAPITAL LETTER JIL
        "\xE1\x82\xBC"     => "\xE2\xB4\x9C",             # GEORGIAN CAPITAL LETTER CIL
        "\xE1\x82\xBD"     => "\xE2\xB4\x9D",             # GEORGIAN CAPITAL LETTER CHAR
        "\xE1\x82\xBE"     => "\xE2\xB4\x9E",             # GEORGIAN CAPITAL LETTER XAN
        "\xE1\x82\xBF"     => "\xE2\xB4\x9F",             # GEORGIAN CAPITAL LETTER JHAN
        "\xE1\x83\x80"     => "\xE2\xB4\xA0",             # GEORGIAN CAPITAL LETTER HAE
        "\xE1\x83\x81"     => "\xE2\xB4\xA1",             # GEORGIAN CAPITAL LETTER HE
        "\xE1\x83\x82"     => "\xE2\xB4\xA2",             # GEORGIAN CAPITAL LETTER HIE
        "\xE1\x83\x83"     => "\xE2\xB4\xA3",             # GEORGIAN CAPITAL LETTER WE
        "\xE1\x83\x84"     => "\xE2\xB4\xA4",             # GEORGIAN CAPITAL LETTER HAR
        "\xE1\x83\x85"     => "\xE2\xB4\xA5",             # GEORGIAN CAPITAL LETTER HOE
        "\xE1\x83\x87"     => "\xE2\xB4\xA7",             # GEORGIAN CAPITAL LETTER YN
        "\xE1\x83\x8D"     => "\xE2\xB4\xAD",             # GEORGIAN CAPITAL LETTER AEN
        "\xE1\x8F\xB8"     => "\xE1\x8F\xB0",             # CHEROKEE SMALL LETTER YE
        "\xE1\x8F\xB9"     => "\xE1\x8F\xB1",             # CHEROKEE SMALL LETTER YI
        "\xE1\x8F\xBA"     => "\xE1\x8F\xB2",             # CHEROKEE SMALL LETTER YO
        "\xE1\x8F\xBB"     => "\xE1\x8F\xB3",             # CHEROKEE SMALL LETTER YU
        "\xE1\x8F\xBC"     => "\xE1\x8F\xB4",             # CHEROKEE SMALL LETTER YV
        "\xE1\x8F\xBD"     => "\xE1\x8F\xB5",             # CHEROKEE SMALL LETTER MV
        "\xE1\xB2\x80"     => "\xD0\xB2",                 # CYRILLIC SMALL LETTER ROUNDED VE
        "\xE1\xB2\x81"     => "\xD0\xB4",                 # CYRILLIC SMALL LETTER LONG-LEGGED DE
        "\xE1\xB2\x82"     => "\xD0\xBE",                 # CYRILLIC SMALL LETTER NARROW O
        "\xE1\xB2\x83"     => "\xD1\x81",                 # CYRILLIC SMALL LETTER WIDE ES
        "\xE1\xB2\x84"     => "\xD1\x82",                 # CYRILLIC SMALL LETTER TALL TE
        "\xE1\xB2\x85"     => "\xD1\x82",                 # CYRILLIC SMALL LETTER THREE-LEGGED TE
        "\xE1\xB2\x86"     => "\xD1\x8A",                 # CYRILLIC SMALL LETTER TALL HARD SIGN
        "\xE1\xB2\x87"     => "\xD1\xA3",                 # CYRILLIC SMALL LETTER TALL YAT
        "\xE1\xB2\x88"     => "\xEA\x99\x8B",             # CYRILLIC SMALL LETTER UNBLENDED UK
        "\xE1\xB2\x90"     => "\xE1\x83\x90",             # GEORGIAN MTAVRULI CAPITAL LETTER AN
        "\xE1\xB2\x91"     => "\xE1\x83\x91",             # GEORGIAN MTAVRULI CAPITAL LETTER BAN
        "\xE1\xB2\x92"     => "\xE1\x83\x92",             # GEORGIAN MTAVRULI CAPITAL LETTER GAN
        "\xE1\xB2\x93"     => "\xE1\x83\x93",             # GEORGIAN MTAVRULI CAPITAL LETTER DON
        "\xE1\xB2\x94"     => "\xE1\x83\x94",             # GEORGIAN MTAVRULI CAPITAL LETTER EN
        "\xE1\xB2\x95"     => "\xE1\x83\x95",             # GEORGIAN MTAVRULI CAPITAL LETTER VIN
        "\xE1\xB2\x96"     => "\xE1\x83\x96",             # GEORGIAN MTAVRULI CAPITAL LETTER ZEN
        "\xE1\xB2\x97"     => "\xE1\x83\x97",             # GEORGIAN MTAVRULI CAPITAL LETTER TAN
        "\xE1\xB2\x98"     => "\xE1\x83\x98",             # GEORGIAN MTAVRULI CAPITAL LETTER IN
        "\xE1\xB2\x99"     => "\xE1\x83\x99",             # GEORGIAN MTAVRULI CAPITAL LETTER KAN
        "\xE1\xB2\x9A"     => "\xE1\x83\x9A",             # GEORGIAN MTAVRULI CAPITAL LETTER LAS
        "\xE1\xB2\x9B"     => "\xE1\x83\x9B",             # GEORGIAN MTAVRULI CAPITAL LETTER MAN
        "\xE1\xB2\x9C"     => "\xE1\x83\x9C",             # GEORGIAN MTAVRULI CAPITAL LETTER NAR
        "\xE1\xB2\x9D"     => "\xE1\x83\x9D",             # GEORGIAN MTAVRULI CAPITAL LETTER ON
        "\xE1\xB2\x9E"     => "\xE1\x83\x9E",             # GEORGIAN MTAVRULI CAPITAL LETTER PAR
        "\xE1\xB2\x9F"     => "\xE1\x83\x9F",             # GEORGIAN MTAVRULI CAPITAL LETTER ZHAR
        "\xE1\xB2\xA0"     => "\xE1\x83\xA0",             # GEORGIAN MTAVRULI CAPITAL LETTER RAE
        "\xE1\xB2\xA1"     => "\xE1\x83\xA1",             # GEORGIAN MTAVRULI CAPITAL LETTER SAN
        "\xE1\xB2\xA2"     => "\xE1\x83\xA2",             # GEORGIAN MTAVRULI CAPITAL LETTER TAR
        "\xE1\xB2\xA3"     => "\xE1\x83\xA3",             # GEORGIAN MTAVRULI CAPITAL LETTER UN
        "\xE1\xB2\xA4"     => "\xE1\x83\xA4",             # GEORGIAN MTAVRULI CAPITAL LETTER PHAR
        "\xE1\xB2\xA5"     => "\xE1\x83\xA5",             # GEORGIAN MTAVRULI CAPITAL LETTER KHAR
        "\xE1\xB2\xA6"     => "\xE1\x83\xA6",             # GEORGIAN MTAVRULI CAPITAL LETTER GHAN
        "\xE1\xB2\xA7"     => "\xE1\x83\xA7",             # GEORGIAN MTAVRULI CAPITAL LETTER QAR
        "\xE1\xB2\xA8"     => "\xE1\x83\xA8",             # GEORGIAN MTAVRULI CAPITAL LETTER SHIN
        "\xE1\xB2\xA9"     => "\xE1\x83\xA9",             # GEORGIAN MTAVRULI CAPITAL LETTER CHIN
        "\xE1\xB2\xAA"     => "\xE1\x83\xAA",             # GEORGIAN MTAVRULI CAPITAL LETTER CAN
        "\xE1\xB2\xAB"     => "\xE1\x83\xAB",             # GEORGIAN MTAVRULI CAPITAL LETTER JIL
        "\xE1\xB2\xAC"     => "\xE1\x83\xAC",             # GEORGIAN MTAVRULI CAPITAL LETTER CIL
        "\xE1\xB2\xAD"     => "\xE1\x83\xAD",             # GEORGIAN MTAVRULI CAPITAL LETTER CHAR
        "\xE1\xB2\xAE"     => "\xE1\x83\xAE",             # GEORGIAN MTAVRULI CAPITAL LETTER XAN
        "\xE1\xB2\xAF"     => "\xE1\x83\xAF",             # GEORGIAN MTAVRULI CAPITAL LETTER JHAN
        "\xE1\xB2\xB0"     => "\xE1\x83\xB0",             # GEORGIAN MTAVRULI CAPITAL LETTER HAE
        "\xE1\xB2\xB1"     => "\xE1\x83\xB1",             # GEORGIAN MTAVRULI CAPITAL LETTER HE
        "\xE1\xB2\xB2"     => "\xE1\x83\xB2",             # GEORGIAN MTAVRULI CAPITAL LETTER HIE
        "\xE1\xB2\xB3"     => "\xE1\x83\xB3",             # GEORGIAN MTAVRULI CAPITAL LETTER WE
        "\xE1\xB2\xB4"     => "\xE1\x83\xB4",             # GEORGIAN MTAVRULI CAPITAL LETTER HAR
        "\xE1\xB2\xB5"     => "\xE1\x83\xB5",             # GEORGIAN MTAVRULI CAPITAL LETTER HOE
        "\xE1\xB2\xB6"     => "\xE1\x83\xB6",             # GEORGIAN MTAVRULI CAPITAL LETTER FI
        "\xE1\xB2\xB7"     => "\xE1\x83\xB7",             # GEORGIAN MTAVRULI CAPITAL LETTER YN
        "\xE1\xB2\xB8"     => "\xE1\x83\xB8",             # GEORGIAN MTAVRULI CAPITAL LETTER ELIFI
        "\xE1\xB2\xB9"     => "\xE1\x83\xB9",             # GEORGIAN MTAVRULI CAPITAL LETTER TURNED GAN
        "\xE1\xB2\xBA"     => "\xE1\x83\xBA",             # GEORGIAN MTAVRULI CAPITAL LETTER AIN
        "\xE1\xB2\xBD"     => "\xE1\x83\xBD",             # GEORGIAN MTAVRULI CAPITAL LETTER AEN
        "\xE1\xB2\xBE"     => "\xE1\x83\xBE",             # GEORGIAN MTAVRULI CAPITAL LETTER HARD SIGN
        "\xE1\xB2\xBF"     => "\xE1\x83\xBF",             # GEORGIAN MTAVRULI CAPITAL LETTER LABIAL SIGN
        "\xE1\xB8\x80"     => "\xE1\xB8\x81",             # LATIN CAPITAL LETTER A WITH RING BELOW
        "\xE1\xB8\x82"     => "\xE1\xB8\x83",             # LATIN CAPITAL LETTER B WITH DOT ABOVE
        "\xE1\xB8\x84"     => "\xE1\xB8\x85",             # LATIN CAPITAL LETTER B WITH DOT BELOW
        "\xE1\xB8\x86"     => "\xE1\xB8\x87",             # LATIN CAPITAL LETTER B WITH LINE BELOW
        "\xE1\xB8\x88"     => "\xE1\xB8\x89",             # LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
        "\xE1\xB8\x8A"     => "\xE1\xB8\x8B",             # LATIN CAPITAL LETTER D WITH DOT ABOVE
        "\xE1\xB8\x8C"     => "\xE1\xB8\x8D",             # LATIN CAPITAL LETTER D WITH DOT BELOW
        "\xE1\xB8\x8E"     => "\xE1\xB8\x8F",             # LATIN CAPITAL LETTER D WITH LINE BELOW
        "\xE1\xB8\x90"     => "\xE1\xB8\x91",             # LATIN CAPITAL LETTER D WITH CEDILLA
        "\xE1\xB8\x92"     => "\xE1\xB8\x93",             # LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
        "\xE1\xB8\x94"     => "\xE1\xB8\x95",             # LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
        "\xE1\xB8\x96"     => "\xE1\xB8\x97",             # LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
        "\xE1\xB8\x98"     => "\xE1\xB8\x99",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
        "\xE1\xB8\x9A"     => "\xE1\xB8\x9B",             # LATIN CAPITAL LETTER E WITH TILDE BELOW
        "\xE1\xB8\x9C"     => "\xE1\xB8\x9D",             # LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
        "\xE1\xB8\x9E"     => "\xE1\xB8\x9F",             # LATIN CAPITAL LETTER F WITH DOT ABOVE
        "\xE1\xB8\xA0"     => "\xE1\xB8\xA1",             # LATIN CAPITAL LETTER G WITH MACRON
        "\xE1\xB8\xA2"     => "\xE1\xB8\xA3",             # LATIN CAPITAL LETTER H WITH DOT ABOVE
        "\xE1\xB8\xA4"     => "\xE1\xB8\xA5",             # LATIN CAPITAL LETTER H WITH DOT BELOW
        "\xE1\xB8\xA6"     => "\xE1\xB8\xA7",             # LATIN CAPITAL LETTER H WITH DIAERESIS
        "\xE1\xB8\xA8"     => "\xE1\xB8\xA9",             # LATIN CAPITAL LETTER H WITH CEDILLA
        "\xE1\xB8\xAA"     => "\xE1\xB8\xAB",             # LATIN CAPITAL LETTER H WITH BREVE BELOW
        "\xE1\xB8\xAC"     => "\xE1\xB8\xAD",             # LATIN CAPITAL LETTER I WITH TILDE BELOW
        "\xE1\xB8\xAE"     => "\xE1\xB8\xAF",             # LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
        "\xE1\xB8\xB0"     => "\xE1\xB8\xB1",             # LATIN CAPITAL LETTER K WITH ACUTE
        "\xE1\xB8\xB2"     => "\xE1\xB8\xB3",             # LATIN CAPITAL LETTER K WITH DOT BELOW
        "\xE1\xB8\xB4"     => "\xE1\xB8\xB5",             # LATIN CAPITAL LETTER K WITH LINE BELOW
        "\xE1\xB8\xB6"     => "\xE1\xB8\xB7",             # LATIN CAPITAL LETTER L WITH DOT BELOW
        "\xE1\xB8\xB8"     => "\xE1\xB8\xB9",             # LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
        "\xE1\xB8\xBA"     => "\xE1\xB8\xBB",             # LATIN CAPITAL LETTER L WITH LINE BELOW
        "\xE1\xB8\xBC"     => "\xE1\xB8\xBD",             # LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
        "\xE1\xB8\xBE"     => "\xE1\xB8\xBF",             # LATIN CAPITAL LETTER M WITH ACUTE
        "\xE1\xB9\x80"     => "\xE1\xB9\x81",             # LATIN CAPITAL LETTER M WITH DOT ABOVE
        "\xE1\xB9\x82"     => "\xE1\xB9\x83",             # LATIN CAPITAL LETTER M WITH DOT BELOW
        "\xE1\xB9\x84"     => "\xE1\xB9\x85",             # LATIN CAPITAL LETTER N WITH DOT ABOVE
        "\xE1\xB9\x86"     => "\xE1\xB9\x87",             # LATIN CAPITAL LETTER N WITH DOT BELOW
        "\xE1\xB9\x88"     => "\xE1\xB9\x89",             # LATIN CAPITAL LETTER N WITH LINE BELOW
        "\xE1\xB9\x8A"     => "\xE1\xB9\x8B",             # LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
        "\xE1\xB9\x8C"     => "\xE1\xB9\x8D",             # LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
        "\xE1\xB9\x8E"     => "\xE1\xB9\x8F",             # LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
        "\xE1\xB9\x90"     => "\xE1\xB9\x91",             # LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
        "\xE1\xB9\x92"     => "\xE1\xB9\x93",             # LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
        "\xE1\xB9\x94"     => "\xE1\xB9\x95",             # LATIN CAPITAL LETTER P WITH ACUTE
        "\xE1\xB9\x96"     => "\xE1\xB9\x97",             # LATIN CAPITAL LETTER P WITH DOT ABOVE
        "\xE1\xB9\x98"     => "\xE1\xB9\x99",             # LATIN CAPITAL LETTER R WITH DOT ABOVE
        "\xE1\xB9\x9A"     => "\xE1\xB9\x9B",             # LATIN CAPITAL LETTER R WITH DOT BELOW
        "\xE1\xB9\x9C"     => "\xE1\xB9\x9D",             # LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
        "\xE1\xB9\x9E"     => "\xE1\xB9\x9F",             # LATIN CAPITAL LETTER R WITH LINE BELOW
        "\xE1\xB9\xA0"     => "\xE1\xB9\xA1",             # LATIN CAPITAL LETTER S WITH DOT ABOVE
        "\xE1\xB9\xA2"     => "\xE1\xB9\xA3",             # LATIN CAPITAL LETTER S WITH DOT BELOW
        "\xE1\xB9\xA4"     => "\xE1\xB9\xA5",             # LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
        "\xE1\xB9\xA6"     => "\xE1\xB9\xA7",             # LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
        "\xE1\xB9\xA8"     => "\xE1\xB9\xA9",             # LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
        "\xE1\xB9\xAA"     => "\xE1\xB9\xAB",             # LATIN CAPITAL LETTER T WITH DOT ABOVE
        "\xE1\xB9\xAC"     => "\xE1\xB9\xAD",             # LATIN CAPITAL LETTER T WITH DOT BELOW
        "\xE1\xB9\xAE"     => "\xE1\xB9\xAF",             # LATIN CAPITAL LETTER T WITH LINE BELOW
        "\xE1\xB9\xB0"     => "\xE1\xB9\xB1",             # LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
        "\xE1\xB9\xB2"     => "\xE1\xB9\xB3",             # LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
        "\xE1\xB9\xB4"     => "\xE1\xB9\xB5",             # LATIN CAPITAL LETTER U WITH TILDE BELOW
        "\xE1\xB9\xB6"     => "\xE1\xB9\xB7",             # LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
        "\xE1\xB9\xB8"     => "\xE1\xB9\xB9",             # LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
        "\xE1\xB9\xBA"     => "\xE1\xB9\xBB",             # LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
        "\xE1\xB9\xBC"     => "\xE1\xB9\xBD",             # LATIN CAPITAL LETTER V WITH TILDE
        "\xE1\xB9\xBE"     => "\xE1\xB9\xBF",             # LATIN CAPITAL LETTER V WITH DOT BELOW
        "\xE1\xBA\x80"     => "\xE1\xBA\x81",             # LATIN CAPITAL LETTER W WITH GRAVE
        "\xE1\xBA\x82"     => "\xE1\xBA\x83",             # LATIN CAPITAL LETTER W WITH ACUTE
        "\xE1\xBA\x84"     => "\xE1\xBA\x85",             # LATIN CAPITAL LETTER W WITH DIAERESIS
        "\xE1\xBA\x86"     => "\xE1\xBA\x87",             # LATIN CAPITAL LETTER W WITH DOT ABOVE
        "\xE1\xBA\x88"     => "\xE1\xBA\x89",             # LATIN CAPITAL LETTER W WITH DOT BELOW
        "\xE1\xBA\x8A"     => "\xE1\xBA\x8B",             # LATIN CAPITAL LETTER X WITH DOT ABOVE
        "\xE1\xBA\x8C"     => "\xE1\xBA\x8D",             # LATIN CAPITAL LETTER X WITH DIAERESIS
        "\xE1\xBA\x8E"     => "\xE1\xBA\x8F",             # LATIN CAPITAL LETTER Y WITH DOT ABOVE
        "\xE1\xBA\x90"     => "\xE1\xBA\x91",             # LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
        "\xE1\xBA\x92"     => "\xE1\xBA\x93",             # LATIN CAPITAL LETTER Z WITH DOT BELOW
        "\xE1\xBA\x94"     => "\xE1\xBA\x95",             # LATIN CAPITAL LETTER Z WITH LINE BELOW
        "\xE1\xBA\x96"     => "\x68\xCC\xB1",             # LATIN SMALL LETTER H WITH LINE BELOW
        "\xE1\xBA\x97"     => "\x74\xCC\x88",             # LATIN SMALL LETTER T WITH DIAERESIS
        "\xE1\xBA\x98"     => "\x77\xCC\x8A",             # LATIN SMALL LETTER W WITH RING ABOVE
        "\xE1\xBA\x99"     => "\x79\xCC\x8A",             # LATIN SMALL LETTER Y WITH RING ABOVE
        "\xE1\xBA\x9A"     => "\x61\xCA\xBE",             # LATIN SMALL LETTER A WITH RIGHT HALF RING
        "\xE1\xBA\x9B"     => "\xE1\xB9\xA1",             # LATIN SMALL LETTER LONG S WITH DOT ABOVE
        "\xE1\xBA\x9E"     => "\x73\x73",                 # LATIN CAPITAL LETTER SHARP S
        "\xE1\xBA\xA0"     => "\xE1\xBA\xA1",             # LATIN CAPITAL LETTER A WITH DOT BELOW
        "\xE1\xBA\xA2"     => "\xE1\xBA\xA3",             # LATIN CAPITAL LETTER A WITH HOOK ABOVE
        "\xE1\xBA\xA4"     => "\xE1\xBA\xA5",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBA\xA6"     => "\xE1\xBA\xA7",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBA\xA8"     => "\xE1\xBA\xA9",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBA\xAA"     => "\xE1\xBA\xAB",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
        "\xE1\xBA\xAC"     => "\xE1\xBA\xAD",             # LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBA\xAE"     => "\xE1\xBA\xAF",             # LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
        "\xE1\xBA\xB0"     => "\xE1\xBA\xB1",             # LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
        "\xE1\xBA\xB2"     => "\xE1\xBA\xB3",             # LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
        "\xE1\xBA\xB4"     => "\xE1\xBA\xB5",             # LATIN CAPITAL LETTER A WITH BREVE AND TILDE
        "\xE1\xBA\xB6"     => "\xE1\xBA\xB7",             # LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
        "\xE1\xBA\xB8"     => "\xE1\xBA\xB9",             # LATIN CAPITAL LETTER E WITH DOT BELOW
        "\xE1\xBA\xBA"     => "\xE1\xBA\xBB",             # LATIN CAPITAL LETTER E WITH HOOK ABOVE
        "\xE1\xBA\xBC"     => "\xE1\xBA\xBD",             # LATIN CAPITAL LETTER E WITH TILDE
        "\xE1\xBA\xBE"     => "\xE1\xBA\xBF",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBB\x80"     => "\xE1\xBB\x81",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBB\x82"     => "\xE1\xBB\x83",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBB\x84"     => "\xE1\xBB\x85",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
        "\xE1\xBB\x86"     => "\xE1\xBB\x87",             # LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBB\x88"     => "\xE1\xBB\x89",             # LATIN CAPITAL LETTER I WITH HOOK ABOVE
        "\xE1\xBB\x8A"     => "\xE1\xBB\x8B",             # LATIN CAPITAL LETTER I WITH DOT BELOW
        "\xE1\xBB\x8C"     => "\xE1\xBB\x8D",             # LATIN CAPITAL LETTER O WITH DOT BELOW
        "\xE1\xBB\x8E"     => "\xE1\xBB\x8F",             # LATIN CAPITAL LETTER O WITH HOOK ABOVE
        "\xE1\xBB\x90"     => "\xE1\xBB\x91",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
        "\xE1\xBB\x92"     => "\xE1\xBB\x93",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
        "\xE1\xBB\x94"     => "\xE1\xBB\x95",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
        "\xE1\xBB\x96"     => "\xE1\xBB\x97",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
        "\xE1\xBB\x98"     => "\xE1\xBB\x99",             # LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
        "\xE1\xBB\x9A"     => "\xE1\xBB\x9B",             # LATIN CAPITAL LETTER O WITH HORN AND ACUTE
        "\xE1\xBB\x9C"     => "\xE1\xBB\x9D",             # LATIN CAPITAL LETTER O WITH HORN AND GRAVE
        "\xE1\xBB\x9E"     => "\xE1\xBB\x9F",             # LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
        "\xE1\xBB\xA0"     => "\xE1\xBB\xA1",             # LATIN CAPITAL LETTER O WITH HORN AND TILDE
        "\xE1\xBB\xA2"     => "\xE1\xBB\xA3",             # LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
        "\xE1\xBB\xA4"     => "\xE1\xBB\xA5",             # LATIN CAPITAL LETTER U WITH DOT BELOW
        "\xE1\xBB\xA6"     => "\xE1\xBB\xA7",             # LATIN CAPITAL LETTER U WITH HOOK ABOVE
        "\xE1\xBB\xA8"     => "\xE1\xBB\xA9",             # LATIN CAPITAL LETTER U WITH HORN AND ACUTE
        "\xE1\xBB\xAA"     => "\xE1\xBB\xAB",             # LATIN CAPITAL LETTER U WITH HORN AND GRAVE
        "\xE1\xBB\xAC"     => "\xE1\xBB\xAD",             # LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
        "\xE1\xBB\xAE"     => "\xE1\xBB\xAF",             # LATIN CAPITAL LETTER U WITH HORN AND TILDE
        "\xE1\xBB\xB0"     => "\xE1\xBB\xB1",             # LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
        "\xE1\xBB\xB2"     => "\xE1\xBB\xB3",             # LATIN CAPITAL LETTER Y WITH GRAVE
        "\xE1\xBB\xB4"     => "\xE1\xBB\xB5",             # LATIN CAPITAL LETTER Y WITH DOT BELOW
        "\xE1\xBB\xB6"     => "\xE1\xBB\xB7",             # LATIN CAPITAL LETTER Y WITH HOOK ABOVE
        "\xE1\xBB\xB8"     => "\xE1\xBB\xB9",             # LATIN CAPITAL LETTER Y WITH TILDE
        "\xE1\xBB\xBA"     => "\xE1\xBB\xBB",             # LATIN CAPITAL LETTER MIDDLE-WELSH LL
        "\xE1\xBB\xBC"     => "\xE1\xBB\xBD",             # LATIN CAPITAL LETTER MIDDLE-WELSH V
        "\xE1\xBB\xBE"     => "\xE1\xBB\xBF",             # LATIN CAPITAL LETTER Y WITH LOOP
        "\xE1\xBC\x88"     => "\xE1\xBC\x80",             # GREEK CAPITAL LETTER ALPHA WITH PSILI
        "\xE1\xBC\x89"     => "\xE1\xBC\x81",             # GREEK CAPITAL LETTER ALPHA WITH DASIA
        "\xE1\xBC\x8A"     => "\xE1\xBC\x82",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
        "\xE1\xBC\x8B"     => "\xE1\xBC\x83",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
        "\xE1\xBC\x8C"     => "\xE1\xBC\x84",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
        "\xE1\xBC\x8D"     => "\xE1\xBC\x85",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
        "\xE1\xBC\x8E"     => "\xE1\xBC\x86",             # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\x8F"     => "\xE1\xBC\x87",             # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
        "\xE1\xBC\x98"     => "\xE1\xBC\x90",             # GREEK CAPITAL LETTER EPSILON WITH PSILI
        "\xE1\xBC\x99"     => "\xE1\xBC\x91",             # GREEK CAPITAL LETTER EPSILON WITH DASIA
        "\xE1\xBC\x9A"     => "\xE1\xBC\x92",             # GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
        "\xE1\xBC\x9B"     => "\xE1\xBC\x93",             # GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
        "\xE1\xBC\x9C"     => "\xE1\xBC\x94",             # GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
        "\xE1\xBC\x9D"     => "\xE1\xBC\x95",             # GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
        "\xE1\xBC\xA8"     => "\xE1\xBC\xA0",             # GREEK CAPITAL LETTER ETA WITH PSILI
        "\xE1\xBC\xA9"     => "\xE1\xBC\xA1",             # GREEK CAPITAL LETTER ETA WITH DASIA
        "\xE1\xBC\xAA"     => "\xE1\xBC\xA2",             # GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
        "\xE1\xBC\xAB"     => "\xE1\xBC\xA3",             # GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
        "\xE1\xBC\xAC"     => "\xE1\xBC\xA4",             # GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
        "\xE1\xBC\xAD"     => "\xE1\xBC\xA5",             # GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
        "\xE1\xBC\xAE"     => "\xE1\xBC\xA6",             # GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\xAF"     => "\xE1\xBC\xA7",             # GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
        "\xE1\xBC\xB8"     => "\xE1\xBC\xB0",             # GREEK CAPITAL LETTER IOTA WITH PSILI
        "\xE1\xBC\xB9"     => "\xE1\xBC\xB1",             # GREEK CAPITAL LETTER IOTA WITH DASIA
        "\xE1\xBC\xBA"     => "\xE1\xBC\xB2",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
        "\xE1\xBC\xBB"     => "\xE1\xBC\xB3",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
        "\xE1\xBC\xBC"     => "\xE1\xBC\xB4",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
        "\xE1\xBC\xBD"     => "\xE1\xBC\xB5",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
        "\xE1\xBC\xBE"     => "\xE1\xBC\xB6",             # GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
        "\xE1\xBC\xBF"     => "\xE1\xBC\xB7",             # GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
        "\xE1\xBD\x88"     => "\xE1\xBD\x80",             # GREEK CAPITAL LETTER OMICRON WITH PSILI
        "\xE1\xBD\x89"     => "\xE1\xBD\x81",             # GREEK CAPITAL LETTER OMICRON WITH DASIA
        "\xE1\xBD\x8A"     => "\xE1\xBD\x82",             # GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
        "\xE1\xBD\x8B"     => "\xE1\xBD\x83",             # GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
        "\xE1\xBD\x8C"     => "\xE1\xBD\x84",             # GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
        "\xE1\xBD\x8D"     => "\xE1\xBD\x85",             # GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
        "\xE1\xBD\x90"     => "\xCF\x85\xCC\x93",         # GREEK SMALL LETTER UPSILON WITH PSILI
        "\xE1\xBD\x92"     => "\xCF\x85\xCC\x93\xCC\x80", # GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
        "\xE1\xBD\x94"     => "\xCF\x85\xCC\x93\xCC\x81", # GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
        "\xE1\xBD\x96"     => "\xCF\x85\xCC\x93\xCD\x82", # GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
        "\xE1\xBD\x99"     => "\xE1\xBD\x91",             # GREEK CAPITAL LETTER UPSILON WITH DASIA
        "\xE1\xBD\x9B"     => "\xE1\xBD\x93",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
        "\xE1\xBD\x9D"     => "\xE1\xBD\x95",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
        "\xE1\xBD\x9F"     => "\xE1\xBD\x97",             # GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
        "\xE1\xBD\xA8"     => "\xE1\xBD\xA0",             # GREEK CAPITAL LETTER OMEGA WITH PSILI
        "\xE1\xBD\xA9"     => "\xE1\xBD\xA1",             # GREEK CAPITAL LETTER OMEGA WITH DASIA
        "\xE1\xBD\xAA"     => "\xE1\xBD\xA2",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
        "\xE1\xBD\xAB"     => "\xE1\xBD\xA3",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
        "\xE1\xBD\xAC"     => "\xE1\xBD\xA4",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
        "\xE1\xBD\xAD"     => "\xE1\xBD\xA5",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
        "\xE1\xBD\xAE"     => "\xE1\xBD\xA6",             # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
        "\xE1\xBD\xAF"     => "\xE1\xBD\xA7",             # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
        "\xE1\xBE\x80"     => "\xE1\xBC\x80\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\x81"     => "\xE1\xBC\x81\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\x82"     => "\xE1\xBC\x82\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x83"     => "\xE1\xBC\x83\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x84"     => "\xE1\xBC\x84\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x85"     => "\xE1\xBC\x85\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x86"     => "\xE1\xBC\x86\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x87"     => "\xE1\xBC\x87\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x88"     => "\xE1\xBC\x80\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\x89"     => "\xE1\xBC\x81\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8A"     => "\xE1\xBC\x82\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8B"     => "\xE1\xBC\x83\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8C"     => "\xE1\xBC\x84\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8D"     => "\xE1\xBC\x85\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x8E"     => "\xE1\xBC\x86\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x8F"     => "\xE1\xBC\x87\xCE\xB9",     # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x90"     => "\xE1\xBC\xA0\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\x91"     => "\xE1\xBC\xA1\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\x92"     => "\xE1\xBC\xA2\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x93"     => "\xE1\xBC\xA3\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\x94"     => "\xE1\xBC\xA4\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x95"     => "\xE1\xBC\xA5\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\x96"     => "\xE1\xBC\xA6\xCE\xB9",     # GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x97"     => "\xE1\xBC\xA7\xCE\xB9",     # GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\x98"     => "\xE1\xBC\xA0\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\x99"     => "\xE1\xBC\xA1\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9A"     => "\xE1\xBC\xA2\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9B"     => "\xE1\xBC\xA3\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9C"     => "\xE1\xBC\xA4\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9D"     => "\xE1\xBC\xA5\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\x9E"     => "\xE1\xBC\xA6\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\x9F"     => "\xE1\xBC\xA7\xCE\xB9",     # GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xA0"     => "\xE1\xBD\xA0\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
        "\xE1\xBE\xA1"     => "\xE1\xBD\xA1\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA2"     => "\xE1\xBD\xA2\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA3"     => "\xE1\xBD\xA3\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA4"     => "\xE1\xBD\xA4\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA5"     => "\xE1\xBD\xA5\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xA6"     => "\xE1\xBD\xA6\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xA7"     => "\xE1\xBD\xA7\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xA8"     => "\xE1\xBD\xA0\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
        "\xE1\xBE\xA9"     => "\xE1\xBD\xA1\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAA"     => "\xE1\xBD\xA2\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAB"     => "\xE1\xBD\xA3\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAC"     => "\xE1\xBD\xA4\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAD"     => "\xE1\xBD\xA5\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
        "\xE1\xBE\xAE"     => "\xE1\xBD\xA6\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xAF"     => "\xE1\xBD\xA7\xCE\xB9",     # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
        "\xE1\xBE\xB2"     => "\xE1\xBD\xB0\xCE\xB9",     # GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBE\xB3"     => "\xCE\xB1\xCE\xB9",         # GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
        "\xE1\xBE\xB4"     => "\xCE\xAC\xCE\xB9",         # GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBE\xB6"     => "\xCE\xB1\xCD\x82",         # GREEK SMALL LETTER ALPHA WITH PERISPOMENI
        "\xE1\xBE\xB7"     => "\xCE\xB1\xCD\x82\xCE\xB9", # GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBE\xB8"     => "\xE1\xBE\xB0",             # GREEK CAPITAL LETTER ALPHA WITH VRACHY
        "\xE1\xBE\xB9"     => "\xE1\xBE\xB1",             # GREEK CAPITAL LETTER ALPHA WITH MACRON
        "\xE1\xBE\xBA"     => "\xE1\xBD\xB0",             # GREEK CAPITAL LETTER ALPHA WITH VARIA
        "\xE1\xBE\xBB"     => "\xE1\xBD\xB1",             # GREEK CAPITAL LETTER ALPHA WITH OXIA
        "\xE1\xBE\xBC"     => "\xCE\xB1\xCE\xB9",         # GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
        "\xE1\xBE\xBE"     => "\xCE\xB9",                 # GREEK PROSGEGRAMMENI
        "\xE1\xBF\x82"     => "\xE1\xBD\xB4\xCE\xB9",     # GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBF\x83"     => "\xCE\xB7\xCE\xB9",         # GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
        "\xE1\xBF\x84"     => "\xCE\xAE\xCE\xB9",         # GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBF\x86"     => "\xCE\xB7\xCD\x82",         # GREEK SMALL LETTER ETA WITH PERISPOMENI
        "\xE1\xBF\x87"     => "\xCE\xB7\xCD\x82\xCE\xB9", # GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBF\x88"     => "\xE1\xBD\xB2",             # GREEK CAPITAL LETTER EPSILON WITH VARIA
        "\xE1\xBF\x89"     => "\xE1\xBD\xB3",             # GREEK CAPITAL LETTER EPSILON WITH OXIA
        "\xE1\xBF\x8A"     => "\xE1\xBD\xB4",             # GREEK CAPITAL LETTER ETA WITH VARIA
        "\xE1\xBF\x8B"     => "\xE1\xBD\xB5",             # GREEK CAPITAL LETTER ETA WITH OXIA
        "\xE1\xBF\x8C"     => "\xCE\xB7\xCE\xB9",         # GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
        "\xE1\xBF\x92"     => "\xCE\xB9\xCC\x88\xCC\x80", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
        "\xE1\xBF\x93"     => "\xCE\xB9\xCC\x88\xCC\x81", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
        "\xE1\xBF\x96"     => "\xCE\xB9\xCD\x82",         # GREEK SMALL LETTER IOTA WITH PERISPOMENI
        "\xE1\xBF\x97"     => "\xCE\xB9\xCC\x88\xCD\x82", # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
        "\xE1\xBF\x98"     => "\xE1\xBF\x90",             # GREEK CAPITAL LETTER IOTA WITH VRACHY
        "\xE1\xBF\x99"     => "\xE1\xBF\x91",             # GREEK CAPITAL LETTER IOTA WITH MACRON
        "\xE1\xBF\x9A"     => "\xE1\xBD\xB6",             # GREEK CAPITAL LETTER IOTA WITH VARIA
        "\xE1\xBF\x9B"     => "\xE1\xBD\xB7",             # GREEK CAPITAL LETTER IOTA WITH OXIA
        "\xE1\xBF\xA2"     => "\xCF\x85\xCC\x88\xCC\x80", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
        "\xE1\xBF\xA3"     => "\xCF\x85\xCC\x88\xCC\x81", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
        "\xE1\xBF\xA4"     => "\xCF\x81\xCC\x93",         # GREEK SMALL LETTER RHO WITH PSILI
        "\xE1\xBF\xA6"     => "\xCF\x85\xCD\x82",         # GREEK SMALL LETTER UPSILON WITH PERISPOMENI
        "\xE1\xBF\xA7"     => "\xCF\x85\xCC\x88\xCD\x82", # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
        "\xE1\xBF\xA8"     => "\xE1\xBF\xA0",             # GREEK CAPITAL LETTER UPSILON WITH VRACHY
        "\xE1\xBF\xA9"     => "\xE1\xBF\xA1",             # GREEK CAPITAL LETTER UPSILON WITH MACRON
        "\xE1\xBF\xAA"     => "\xE1\xBD\xBA",             # GREEK CAPITAL LETTER UPSILON WITH VARIA
        "\xE1\xBF\xAB"     => "\xE1\xBD\xBB",             # GREEK CAPITAL LETTER UPSILON WITH OXIA
        "\xE1\xBF\xAC"     => "\xE1\xBF\xA5",             # GREEK CAPITAL LETTER RHO WITH DASIA
        "\xE1\xBF\xB2"     => "\xE1\xBD\xBC\xCE\xB9",     # GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
        "\xE1\xBF\xB3"     => "\xCF\x89\xCE\xB9",         # GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
        "\xE1\xBF\xB4"     => "\xCF\x8E\xCE\xB9",         # GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
        "\xE1\xBF\xB6"     => "\xCF\x89\xCD\x82",         # GREEK SMALL LETTER OMEGA WITH PERISPOMENI
        "\xE1\xBF\xB7"     => "\xCF\x89\xCD\x82\xCE\xB9", # GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
        "\xE1\xBF\xB8"     => "\xE1\xBD\xB8",             # GREEK CAPITAL LETTER OMICRON WITH VARIA
        "\xE1\xBF\xB9"     => "\xE1\xBD\xB9",             # GREEK CAPITAL LETTER OMICRON WITH OXIA
        "\xE1\xBF\xBA"     => "\xE1\xBD\xBC",             # GREEK CAPITAL LETTER OMEGA WITH VARIA
        "\xE1\xBF\xBB"     => "\xE1\xBD\xBD",             # GREEK CAPITAL LETTER OMEGA WITH OXIA
        "\xE1\xBF\xBC"     => "\xCF\x89\xCE\xB9",         # GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
        "\xE2\x84\xA6"     => "\xCF\x89",                 # OHM SIGN
        "\xE2\x84\xAA"     => "\x6B",                     # KELVIN SIGN
        "\xE2\x84\xAB"     => "\xC3\xA5",                 # ANGSTROM SIGN
        "\xE2\x84\xB2"     => "\xE2\x85\x8E",             # TURNED CAPITAL F
        "\xE2\x85\xA0"     => "\xE2\x85\xB0",             # ROMAN NUMERAL ONE
        "\xE2\x85\xA1"     => "\xE2\x85\xB1",             # ROMAN NUMERAL TWO
        "\xE2\x85\xA2"     => "\xE2\x85\xB2",             # ROMAN NUMERAL THREE
        "\xE2\x85\xA3"     => "\xE2\x85\xB3",             # ROMAN NUMERAL FOUR
        "\xE2\x85\xA4"     => "\xE2\x85\xB4",             # ROMAN NUMERAL FIVE
        "\xE2\x85\xA5"     => "\xE2\x85\xB5",             # ROMAN NUMERAL SIX
        "\xE2\x85\xA6"     => "\xE2\x85\xB6",             # ROMAN NUMERAL SEVEN
        "\xE2\x85\xA7"     => "\xE2\x85\xB7",             # ROMAN NUMERAL EIGHT
        "\xE2\x85\xA8"     => "\xE2\x85\xB8",             # ROMAN NUMERAL NINE
        "\xE2\x85\xA9"     => "\xE2\x85\xB9",             # ROMAN NUMERAL TEN
        "\xE2\x85\xAA"     => "\xE2\x85\xBA",             # ROMAN NUMERAL ELEVEN
        "\xE2\x85\xAB"     => "\xE2\x85\xBB",             # ROMAN NUMERAL TWELVE
        "\xE2\x85\xAC"     => "\xE2\x85\xBC",             # ROMAN NUMERAL FIFTY
        "\xE2\x85\xAD"     => "\xE2\x85\xBD",             # ROMAN NUMERAL ONE HUNDRED
        "\xE2\x85\xAE"     => "\xE2\x85\xBE",             # ROMAN NUMERAL FIVE HUNDRED
        "\xE2\x85\xAF"     => "\xE2\x85\xBF",             # ROMAN NUMERAL ONE THOUSAND
        "\xE2\x86\x83"     => "\xE2\x86\x84",             # ROMAN NUMERAL REVERSED ONE HUNDRED
        "\xE2\x92\xB6"     => "\xE2\x93\x90",             # CIRCLED LATIN CAPITAL LETTER A
        "\xE2\x92\xB7"     => "\xE2\x93\x91",             # CIRCLED LATIN CAPITAL LETTER B
        "\xE2\x92\xB8"     => "\xE2\x93\x92",             # CIRCLED LATIN CAPITAL LETTER C
        "\xE2\x92\xB9"     => "\xE2\x93\x93",             # CIRCLED LATIN CAPITAL LETTER D
        "\xE2\x92\xBA"     => "\xE2\x93\x94",             # CIRCLED LATIN CAPITAL LETTER E
        "\xE2\x92\xBB"     => "\xE2\x93\x95",             # CIRCLED LATIN CAPITAL LETTER F
        "\xE2\x92\xBC"     => "\xE2\x93\x96",             # CIRCLED LATIN CAPITAL LETTER G
        "\xE2\x92\xBD"     => "\xE2\x93\x97",             # CIRCLED LATIN CAPITAL LETTER H
        "\xE2\x92\xBE"     => "\xE2\x93\x98",             # CIRCLED LATIN CAPITAL LETTER I
        "\xE2\x92\xBF"     => "\xE2\x93\x99",             # CIRCLED LATIN CAPITAL LETTER J
        "\xE2\x93\x80"     => "\xE2\x93\x9A",             # CIRCLED LATIN CAPITAL LETTER K
        "\xE2\x93\x81"     => "\xE2\x93\x9B",             # CIRCLED LATIN CAPITAL LETTER L
        "\xE2\x93\x82"     => "\xE2\x93\x9C",             # CIRCLED LATIN CAPITAL LETTER M
        "\xE2\x93\x83"     => "\xE2\x93\x9D",             # CIRCLED LATIN CAPITAL LETTER N
        "\xE2\x93\x84"     => "\xE2\x93\x9E",             # CIRCLED LATIN CAPITAL LETTER O
        "\xE2\x93\x85"     => "\xE2\x93\x9F",             # CIRCLED LATIN CAPITAL LETTER P
        "\xE2\x93\x86"     => "\xE2\x93\xA0",             # CIRCLED LATIN CAPITAL LETTER Q
        "\xE2\x93\x87"     => "\xE2\x93\xA1",             # CIRCLED LATIN CAPITAL LETTER R
        "\xE2\x93\x88"     => "\xE2\x93\xA2",             # CIRCLED LATIN CAPITAL LETTER S
        "\xE2\x93\x89"     => "\xE2\x93\xA3",             # CIRCLED LATIN CAPITAL LETTER T
        "\xE2\x93\x8A"     => "\xE2\x93\xA4",             # CIRCLED LATIN CAPITAL LETTER U
        "\xE2\x93\x8B"     => "\xE2\x93\xA5",             # CIRCLED LATIN CAPITAL LETTER V
        "\xE2\x93\x8C"     => "\xE2\x93\xA6",             # CIRCLED LATIN CAPITAL LETTER W
        "\xE2\x93\x8D"     => "\xE2\x93\xA7",             # CIRCLED LATIN CAPITAL LETTER X
        "\xE2\x93\x8E"     => "\xE2\x93\xA8",             # CIRCLED LATIN CAPITAL LETTER Y
        "\xE2\x93\x8F"     => "\xE2\x93\xA9",             # CIRCLED LATIN CAPITAL LETTER Z
        "\xE2\xB0\x80"     => "\xE2\xB0\xB0",             # GLAGOLITIC CAPITAL LETTER AZU
        "\xE2\xB0\x81"     => "\xE2\xB0\xB1",             # GLAGOLITIC CAPITAL LETTER BUKY
        "\xE2\xB0\x82"     => "\xE2\xB0\xB2",             # GLAGOLITIC CAPITAL LETTER VEDE
        "\xE2\xB0\x83"     => "\xE2\xB0\xB3",             # GLAGOLITIC CAPITAL LETTER GLAGOLI
        "\xE2\xB0\x84"     => "\xE2\xB0\xB4",             # GLAGOLITIC CAPITAL LETTER DOBRO
        "\xE2\xB0\x85"     => "\xE2\xB0\xB5",             # GLAGOLITIC CAPITAL LETTER YESTU
        "\xE2\xB0\x86"     => "\xE2\xB0\xB6",             # GLAGOLITIC CAPITAL LETTER ZHIVETE
        "\xE2\xB0\x87"     => "\xE2\xB0\xB7",             # GLAGOLITIC CAPITAL LETTER DZELO
        "\xE2\xB0\x88"     => "\xE2\xB0\xB8",             # GLAGOLITIC CAPITAL LETTER ZEMLJA
        "\xE2\xB0\x89"     => "\xE2\xB0\xB9",             # GLAGOLITIC CAPITAL LETTER IZHE
        "\xE2\xB0\x8A"     => "\xE2\xB0\xBA",             # GLAGOLITIC CAPITAL LETTER INITIAL IZHE
        "\xE2\xB0\x8B"     => "\xE2\xB0\xBB",             # GLAGOLITIC CAPITAL LETTER I
        "\xE2\xB0\x8C"     => "\xE2\xB0\xBC",             # GLAGOLITIC CAPITAL LETTER DJERVI
        "\xE2\xB0\x8D"     => "\xE2\xB0\xBD",             # GLAGOLITIC CAPITAL LETTER KAKO
        "\xE2\xB0\x8E"     => "\xE2\xB0\xBE",             # GLAGOLITIC CAPITAL LETTER LJUDIJE
        "\xE2\xB0\x8F"     => "\xE2\xB0\xBF",             # GLAGOLITIC CAPITAL LETTER MYSLITE
        "\xE2\xB0\x90"     => "\xE2\xB1\x80",             # GLAGOLITIC CAPITAL LETTER NASHI
        "\xE2\xB0\x91"     => "\xE2\xB1\x81",             # GLAGOLITIC CAPITAL LETTER ONU
        "\xE2\xB0\x92"     => "\xE2\xB1\x82",             # GLAGOLITIC CAPITAL LETTER POKOJI
        "\xE2\xB0\x93"     => "\xE2\xB1\x83",             # GLAGOLITIC CAPITAL LETTER RITSI
        "\xE2\xB0\x94"     => "\xE2\xB1\x84",             # GLAGOLITIC CAPITAL LETTER SLOVO
        "\xE2\xB0\x95"     => "\xE2\xB1\x85",             # GLAGOLITIC CAPITAL LETTER TVRIDO
        "\xE2\xB0\x96"     => "\xE2\xB1\x86",             # GLAGOLITIC CAPITAL LETTER UKU
        "\xE2\xB0\x97"     => "\xE2\xB1\x87",             # GLAGOLITIC CAPITAL LETTER FRITU
        "\xE2\xB0\x98"     => "\xE2\xB1\x88",             # GLAGOLITIC CAPITAL LETTER HERU
        "\xE2\xB0\x99"     => "\xE2\xB1\x89",             # GLAGOLITIC CAPITAL LETTER OTU
        "\xE2\xB0\x9A"     => "\xE2\xB1\x8A",             # GLAGOLITIC CAPITAL LETTER PE
        "\xE2\xB0\x9B"     => "\xE2\xB1\x8B",             # GLAGOLITIC CAPITAL LETTER SHTA
        "\xE2\xB0\x9C"     => "\xE2\xB1\x8C",             # GLAGOLITIC CAPITAL LETTER TSI
        "\xE2\xB0\x9D"     => "\xE2\xB1\x8D",             # GLAGOLITIC CAPITAL LETTER CHRIVI
        "\xE2\xB0\x9E"     => "\xE2\xB1\x8E",             # GLAGOLITIC CAPITAL LETTER SHA
        "\xE2\xB0\x9F"     => "\xE2\xB1\x8F",             # GLAGOLITIC CAPITAL LETTER YERU
        "\xE2\xB0\xA0"     => "\xE2\xB1\x90",             # GLAGOLITIC CAPITAL LETTER YERI
        "\xE2\xB0\xA1"     => "\xE2\xB1\x91",             # GLAGOLITIC CAPITAL LETTER YATI
        "\xE2\xB0\xA2"     => "\xE2\xB1\x92",             # GLAGOLITIC CAPITAL LETTER SPIDERY HA
        "\xE2\xB0\xA3"     => "\xE2\xB1\x93",             # GLAGOLITIC CAPITAL LETTER YU
        "\xE2\xB0\xA4"     => "\xE2\xB1\x94",             # GLAGOLITIC CAPITAL LETTER SMALL YUS
        "\xE2\xB0\xA5"     => "\xE2\xB1\x95",             # GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL
        "\xE2\xB0\xA6"     => "\xE2\xB1\x96",             # GLAGOLITIC CAPITAL LETTER YO
        "\xE2\xB0\xA7"     => "\xE2\xB1\x97",             # GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS
        "\xE2\xB0\xA8"     => "\xE2\xB1\x98",             # GLAGOLITIC CAPITAL LETTER BIG YUS
        "\xE2\xB0\xA9"     => "\xE2\xB1\x99",             # GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS
        "\xE2\xB0\xAA"     => "\xE2\xB1\x9A",             # GLAGOLITIC CAPITAL LETTER FITA
        "\xE2\xB0\xAB"     => "\xE2\xB1\x9B",             # GLAGOLITIC CAPITAL LETTER IZHITSA
        "\xE2\xB0\xAC"     => "\xE2\xB1\x9C",             # GLAGOLITIC CAPITAL LETTER SHTAPIC
        "\xE2\xB0\xAD"     => "\xE2\xB1\x9D",             # GLAGOLITIC CAPITAL LETTER TROKUTASTI A
        "\xE2\xB0\xAE"     => "\xE2\xB1\x9E",             # GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE
        "\xE2\xB1\xA0"     => "\xE2\xB1\xA1",             # LATIN CAPITAL LETTER L WITH DOUBLE BAR
        "\xE2\xB1\xA2"     => "\xC9\xAB",                 # LATIN CAPITAL LETTER L WITH MIDDLE TILDE
        "\xE2\xB1\xA3"     => "\xE1\xB5\xBD",             # LATIN CAPITAL LETTER P WITH STROKE
        "\xE2\xB1\xA4"     => "\xC9\xBD",                 # LATIN CAPITAL LETTER R WITH TAIL
        "\xE2\xB1\xA7"     => "\xE2\xB1\xA8",             # LATIN CAPITAL LETTER H WITH DESCENDER
        "\xE2\xB1\xA9"     => "\xE2\xB1\xAA",             # LATIN CAPITAL LETTER K WITH DESCENDER
        "\xE2\xB1\xAB"     => "\xE2\xB1\xAC",             # LATIN CAPITAL LETTER Z WITH DESCENDER
        "\xE2\xB1\xAD"     => "\xC9\x91",                 # LATIN CAPITAL LETTER ALPHA
        "\xE2\xB1\xAE"     => "\xC9\xB1",                 # LATIN CAPITAL LETTER M WITH HOOK
        "\xE2\xB1\xAF"     => "\xC9\x90",                 # LATIN CAPITAL LETTER TURNED A
        "\xE2\xB1\xB0"     => "\xC9\x92",                 # LATIN CAPITAL LETTER TURNED ALPHA
        "\xE2\xB1\xB2"     => "\xE2\xB1\xB3",             # LATIN CAPITAL LETTER W WITH HOOK
        "\xE2\xB1\xB5"     => "\xE2\xB1\xB6",             # LATIN CAPITAL LETTER HALF H
        "\xE2\xB1\xBE"     => "\xC8\xBF",                 # LATIN CAPITAL LETTER S WITH SWASH TAIL
        "\xE2\xB1\xBF"     => "\xC9\x80",                 # LATIN CAPITAL LETTER Z WITH SWASH TAIL
        "\xE2\xB2\x80"     => "\xE2\xB2\x81",             # COPTIC CAPITAL LETTER ALFA
        "\xE2\xB2\x82"     => "\xE2\xB2\x83",             # COPTIC CAPITAL LETTER VIDA
        "\xE2\xB2\x84"     => "\xE2\xB2\x85",             # COPTIC CAPITAL LETTER GAMMA
        "\xE2\xB2\x86"     => "\xE2\xB2\x87",             # COPTIC CAPITAL LETTER DALDA
        "\xE2\xB2\x88"     => "\xE2\xB2\x89",             # COPTIC CAPITAL LETTER EIE
        "\xE2\xB2\x8A"     => "\xE2\xB2\x8B",             # COPTIC CAPITAL LETTER SOU
        "\xE2\xB2\x8C"     => "\xE2\xB2\x8D",             # COPTIC CAPITAL LETTER ZATA
        "\xE2\xB2\x8E"     => "\xE2\xB2\x8F",             # COPTIC CAPITAL LETTER HATE
        "\xE2\xB2\x90"     => "\xE2\xB2\x91",             # COPTIC CAPITAL LETTER THETHE
        "\xE2\xB2\x92"     => "\xE2\xB2\x93",             # COPTIC CAPITAL LETTER IAUDA
        "\xE2\xB2\x94"     => "\xE2\xB2\x95",             # COPTIC CAPITAL LETTER KAPA
        "\xE2\xB2\x96"     => "\xE2\xB2\x97",             # COPTIC CAPITAL LETTER LAULA
        "\xE2\xB2\x98"     => "\xE2\xB2\x99",             # COPTIC CAPITAL LETTER MI
        "\xE2\xB2\x9A"     => "\xE2\xB2\x9B",             # COPTIC CAPITAL LETTER NI
        "\xE2\xB2\x9C"     => "\xE2\xB2\x9D",             # COPTIC CAPITAL LETTER KSI
        "\xE2\xB2\x9E"     => "\xE2\xB2\x9F",             # COPTIC CAPITAL LETTER O
        "\xE2\xB2\xA0"     => "\xE2\xB2\xA1",             # COPTIC CAPITAL LETTER PI
        "\xE2\xB2\xA2"     => "\xE2\xB2\xA3",             # COPTIC CAPITAL LETTER RO
        "\xE2\xB2\xA4"     => "\xE2\xB2\xA5",             # COPTIC CAPITAL LETTER SIMA
        "\xE2\xB2\xA6"     => "\xE2\xB2\xA7",             # COPTIC CAPITAL LETTER TAU
        "\xE2\xB2\xA8"     => "\xE2\xB2\xA9",             # COPTIC CAPITAL LETTER UA
        "\xE2\xB2\xAA"     => "\xE2\xB2\xAB",             # COPTIC CAPITAL LETTER FI
        "\xE2\xB2\xAC"     => "\xE2\xB2\xAD",             # COPTIC CAPITAL LETTER KHI
        "\xE2\xB2\xAE"     => "\xE2\xB2\xAF",             # COPTIC CAPITAL LETTER PSI
        "\xE2\xB2\xB0"     => "\xE2\xB2\xB1",             # COPTIC CAPITAL LETTER OOU
        "\xE2\xB2\xB2"     => "\xE2\xB2\xB3",             # COPTIC CAPITAL LETTER DIALECT-P ALEF
        "\xE2\xB2\xB4"     => "\xE2\xB2\xB5",             # COPTIC CAPITAL LETTER OLD COPTIC AIN
        "\xE2\xB2\xB6"     => "\xE2\xB2\xB7",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
        "\xE2\xB2\xB8"     => "\xE2\xB2\xB9",             # COPTIC CAPITAL LETTER DIALECT-P KAPA
        "\xE2\xB2\xBA"     => "\xE2\xB2\xBB",             # COPTIC CAPITAL LETTER DIALECT-P NI
        "\xE2\xB2\xBC"     => "\xE2\xB2\xBD",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
        "\xE2\xB2\xBE"     => "\xE2\xB2\xBF",             # COPTIC CAPITAL LETTER OLD COPTIC OOU
        "\xE2\xB3\x80"     => "\xE2\xB3\x81",             # COPTIC CAPITAL LETTER SAMPI
        "\xE2\xB3\x82"     => "\xE2\xB3\x83",             # COPTIC CAPITAL LETTER CROSSED SHEI
        "\xE2\xB3\x84"     => "\xE2\xB3\x85",             # COPTIC CAPITAL LETTER OLD COPTIC SHEI
        "\xE2\xB3\x86"     => "\xE2\xB3\x87",             # COPTIC CAPITAL LETTER OLD COPTIC ESH
        "\xE2\xB3\x88"     => "\xE2\xB3\x89",             # COPTIC CAPITAL LETTER AKHMIMIC KHEI
        "\xE2\xB3\x8A"     => "\xE2\xB3\x8B",             # COPTIC CAPITAL LETTER DIALECT-P HORI
        "\xE2\xB3\x8C"     => "\xE2\xB3\x8D",             # COPTIC CAPITAL LETTER OLD COPTIC HORI
        "\xE2\xB3\x8E"     => "\xE2\xB3\x8F",             # COPTIC CAPITAL LETTER OLD COPTIC HA
        "\xE2\xB3\x90"     => "\xE2\xB3\x91",             # COPTIC CAPITAL LETTER L-SHAPED HA
        "\xE2\xB3\x92"     => "\xE2\xB3\x93",             # COPTIC CAPITAL LETTER OLD COPTIC HEI
        "\xE2\xB3\x94"     => "\xE2\xB3\x95",             # COPTIC CAPITAL LETTER OLD COPTIC HAT
        "\xE2\xB3\x96"     => "\xE2\xB3\x97",             # COPTIC CAPITAL LETTER OLD COPTIC GANGIA
        "\xE2\xB3\x98"     => "\xE2\xB3\x99",             # COPTIC CAPITAL LETTER OLD COPTIC DJA
        "\xE2\xB3\x9A"     => "\xE2\xB3\x9B",             # COPTIC CAPITAL LETTER OLD COPTIC SHIMA
        "\xE2\xB3\x9C"     => "\xE2\xB3\x9D",             # COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
        "\xE2\xB3\x9E"     => "\xE2\xB3\x9F",             # COPTIC CAPITAL LETTER OLD NUBIAN NGI
        "\xE2\xB3\xA0"     => "\xE2\xB3\xA1",             # COPTIC CAPITAL LETTER OLD NUBIAN NYI
        "\xE2\xB3\xA2"     => "\xE2\xB3\xA3",             # COPTIC CAPITAL LETTER OLD NUBIAN WAU
        "\xE2\xB3\xAB"     => "\xE2\xB3\xAC",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
        "\xE2\xB3\xAD"     => "\xE2\xB3\xAE",             # COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
        "\xE2\xB3\xB2"     => "\xE2\xB3\xB3",             # COPTIC CAPITAL LETTER BOHAIRIC KHEI
        "\xEA\x99\x80"     => "\xEA\x99\x81",             # CYRILLIC CAPITAL LETTER ZEMLYA
        "\xEA\x99\x82"     => "\xEA\x99\x83",             # CYRILLIC CAPITAL LETTER DZELO
        "\xEA\x99\x84"     => "\xEA\x99\x85",             # CYRILLIC CAPITAL LETTER REVERSED DZE
        "\xEA\x99\x86"     => "\xEA\x99\x87",             # CYRILLIC CAPITAL LETTER IOTA
        "\xEA\x99\x88"     => "\xEA\x99\x89",             # CYRILLIC CAPITAL LETTER DJERV
        "\xEA\x99\x8A"     => "\xEA\x99\x8B",             # CYRILLIC CAPITAL LETTER MONOGRAPH UK
        "\xEA\x99\x8C"     => "\xEA\x99\x8D",             # CYRILLIC CAPITAL LETTER BROAD OMEGA
        "\xEA\x99\x8E"     => "\xEA\x99\x8F",             # CYRILLIC CAPITAL LETTER NEUTRAL YER
        "\xEA\x99\x90"     => "\xEA\x99\x91",             # CYRILLIC CAPITAL LETTER YERU WITH BACK YER
        "\xEA\x99\x92"     => "\xEA\x99\x93",             # CYRILLIC CAPITAL LETTER IOTIFIED YAT
        "\xEA\x99\x94"     => "\xEA\x99\x95",             # CYRILLIC CAPITAL LETTER REVERSED YU
        "\xEA\x99\x96"     => "\xEA\x99\x97",             # CYRILLIC CAPITAL LETTER IOTIFIED A
        "\xEA\x99\x98"     => "\xEA\x99\x99",             # CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
        "\xEA\x99\x9A"     => "\xEA\x99\x9B",             # CYRILLIC CAPITAL LETTER BLENDED YUS
        "\xEA\x99\x9C"     => "\xEA\x99\x9D",             # CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
        "\xEA\x99\x9E"     => "\xEA\x99\x9F",             # CYRILLIC CAPITAL LETTER YN
        "\xEA\x99\xA0"     => "\xEA\x99\xA1",             # CYRILLIC CAPITAL LETTER REVERSED TSE
        "\xEA\x99\xA2"     => "\xEA\x99\xA3",             # CYRILLIC CAPITAL LETTER SOFT DE
        "\xEA\x99\xA4"     => "\xEA\x99\xA5",             # CYRILLIC CAPITAL LETTER SOFT EL
        "\xEA\x99\xA6"     => "\xEA\x99\xA7",             # CYRILLIC CAPITAL LETTER SOFT EM
        "\xEA\x99\xA8"     => "\xEA\x99\xA9",             # CYRILLIC CAPITAL LETTER MONOCULAR O
        "\xEA\x99\xAA"     => "\xEA\x99\xAB",             # CYRILLIC CAPITAL LETTER BINOCULAR O
        "\xEA\x99\xAC"     => "\xEA\x99\xAD",             # CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
        "\xEA\x9A\x80"     => "\xEA\x9A\x81",             # CYRILLIC CAPITAL LETTER DWE
        "\xEA\x9A\x82"     => "\xEA\x9A\x83",             # CYRILLIC CAPITAL LETTER DZWE
        "\xEA\x9A\x84"     => "\xEA\x9A\x85",             # CYRILLIC CAPITAL LETTER ZHWE
        "\xEA\x9A\x86"     => "\xEA\x9A\x87",             # CYRILLIC CAPITAL LETTER CCHE
        "\xEA\x9A\x88"     => "\xEA\x9A\x89",             # CYRILLIC CAPITAL LETTER DZZE
        "\xEA\x9A\x8A"     => "\xEA\x9A\x8B",             # CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
        "\xEA\x9A\x8C"     => "\xEA\x9A\x8D",             # CYRILLIC CAPITAL LETTER TWE
        "\xEA\x9A\x8E"     => "\xEA\x9A\x8F",             # CYRILLIC CAPITAL LETTER TSWE
        "\xEA\x9A\x90"     => "\xEA\x9A\x91",             # CYRILLIC CAPITAL LETTER TSSE
        "\xEA\x9A\x92"     => "\xEA\x9A\x93",             # CYRILLIC CAPITAL LETTER TCHE
        "\xEA\x9A\x94"     => "\xEA\x9A\x95",             # CYRILLIC CAPITAL LETTER HWE
        "\xEA\x9A\x96"     => "\xEA\x9A\x97",             # CYRILLIC CAPITAL LETTER SHWE
        "\xEA\x9A\x98"     => "\xEA\x9A\x99",             # CYRILLIC CAPITAL LETTER DOUBLE O
        "\xEA\x9A\x9A"     => "\xEA\x9A\x9B",             # CYRILLIC CAPITAL LETTER CROSSED O
        "\xEA\x9C\xA2"     => "\xEA\x9C\xA3",             # LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
        "\xEA\x9C\xA4"     => "\xEA\x9C\xA5",             # LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
        "\xEA\x9C\xA6"     => "\xEA\x9C\xA7",             # LATIN CAPITAL LETTER HENG
        "\xEA\x9C\xA8"     => "\xEA\x9C\xA9",             # LATIN CAPITAL LETTER TZ
        "\xEA\x9C\xAA"     => "\xEA\x9C\xAB",             # LATIN CAPITAL LETTER TRESILLO
        "\xEA\x9C\xAC"     => "\xEA\x9C\xAD",             # LATIN CAPITAL LETTER CUATRILLO
        "\xEA\x9C\xAE"     => "\xEA\x9C\xAF",             # LATIN CAPITAL LETTER CUATRILLO WITH COMMA
        "\xEA\x9C\xB2"     => "\xEA\x9C\xB3",             # LATIN CAPITAL LETTER AA
        "\xEA\x9C\xB4"     => "\xEA\x9C\xB5",             # LATIN CAPITAL LETTER AO
        "\xEA\x9C\xB6"     => "\xEA\x9C\xB7",             # LATIN CAPITAL LETTER AU
        "\xEA\x9C\xB8"     => "\xEA\x9C\xB9",             # LATIN CAPITAL LETTER AV
        "\xEA\x9C\xBA"     => "\xEA\x9C\xBB",             # LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
        "\xEA\x9C\xBC"     => "\xEA\x9C\xBD",             # LATIN CAPITAL LETTER AY
        "\xEA\x9C\xBE"     => "\xEA\x9C\xBF",             # LATIN CAPITAL LETTER REVERSED C WITH DOT
        "\xEA\x9D\x80"     => "\xEA\x9D\x81",             # LATIN CAPITAL LETTER K WITH STROKE
        "\xEA\x9D\x82"     => "\xEA\x9D\x83",             # LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
        "\xEA\x9D\x84"     => "\xEA\x9D\x85",             # LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
        "\xEA\x9D\x86"     => "\xEA\x9D\x87",             # LATIN CAPITAL LETTER BROKEN L
        "\xEA\x9D\x88"     => "\xEA\x9D\x89",             # LATIN CAPITAL LETTER L WITH HIGH STROKE
        "\xEA\x9D\x8A"     => "\xEA\x9D\x8B",             # LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
        "\xEA\x9D\x8C"     => "\xEA\x9D\x8D",             # LATIN CAPITAL LETTER O WITH LOOP
        "\xEA\x9D\x8E"     => "\xEA\x9D\x8F",             # LATIN CAPITAL LETTER OO
        "\xEA\x9D\x90"     => "\xEA\x9D\x91",             # LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\x92"     => "\xEA\x9D\x93",             # LATIN CAPITAL LETTER P WITH FLOURISH
        "\xEA\x9D\x94"     => "\xEA\x9D\x95",             # LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
        "\xEA\x9D\x96"     => "\xEA\x9D\x97",             # LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\x98"     => "\xEA\x9D\x99",             # LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
        "\xEA\x9D\x9A"     => "\xEA\x9D\x9B",             # LATIN CAPITAL LETTER R ROTUNDA
        "\xEA\x9D\x9C"     => "\xEA\x9D\x9D",             # LATIN CAPITAL LETTER RUM ROTUNDA
        "\xEA\x9D\x9E"     => "\xEA\x9D\x9F",             # LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
        "\xEA\x9D\xA0"     => "\xEA\x9D\xA1",             # LATIN CAPITAL LETTER VY
        "\xEA\x9D\xA2"     => "\xEA\x9D\xA3",             # LATIN CAPITAL LETTER VISIGOTHIC Z
        "\xEA\x9D\xA4"     => "\xEA\x9D\xA5",             # LATIN CAPITAL LETTER THORN WITH STROKE
        "\xEA\x9D\xA6"     => "\xEA\x9D\xA7",             # LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
        "\xEA\x9D\xA8"     => "\xEA\x9D\xA9",             # LATIN CAPITAL LETTER VEND
        "\xEA\x9D\xAA"     => "\xEA\x9D\xAB",             # LATIN CAPITAL LETTER ET
        "\xEA\x9D\xAC"     => "\xEA\x9D\xAD",             # LATIN CAPITAL LETTER IS
        "\xEA\x9D\xAE"     => "\xEA\x9D\xAF",             # LATIN CAPITAL LETTER CON
        "\xEA\x9D\xB9"     => "\xEA\x9D\xBA",             # LATIN CAPITAL LETTER INSULAR D
        "\xEA\x9D\xBB"     => "\xEA\x9D\xBC",             # LATIN CAPITAL LETTER INSULAR F
        "\xEA\x9D\xBD"     => "\xE1\xB5\xB9",             # LATIN CAPITAL LETTER INSULAR G
        "\xEA\x9D\xBE"     => "\xEA\x9D\xBF",             # LATIN CAPITAL LETTER TURNED INSULAR G
        "\xEA\x9E\x80"     => "\xEA\x9E\x81",             # LATIN CAPITAL LETTER TURNED L
        "\xEA\x9E\x82"     => "\xEA\x9E\x83",             # LATIN CAPITAL LETTER INSULAR R
        "\xEA\x9E\x84"     => "\xEA\x9E\x85",             # LATIN CAPITAL LETTER INSULAR S
        "\xEA\x9E\x86"     => "\xEA\x9E\x87",             # LATIN CAPITAL LETTER INSULAR T
        "\xEA\x9E\x8B"     => "\xEA\x9E\x8C",             # LATIN CAPITAL LETTER SALTILLO
        "\xEA\x9E\x8D"     => "\xC9\xA5",                 # LATIN CAPITAL LETTER TURNED H
        "\xEA\x9E\x90"     => "\xEA\x9E\x91",             # LATIN CAPITAL LETTER N WITH DESCENDER
        "\xEA\x9E\x92"     => "\xEA\x9E\x93",             # LATIN CAPITAL LETTER C WITH BAR
        "\xEA\x9E\x96"     => "\xEA\x9E\x97",             # LATIN CAPITAL LETTER B WITH FLOURISH
        "\xEA\x9E\x98"     => "\xEA\x9E\x99",             # LATIN CAPITAL LETTER F WITH STROKE
        "\xEA\x9E\x9A"     => "\xEA\x9E\x9B",             # LATIN CAPITAL LETTER VOLAPUK AE
        "\xEA\x9E\x9C"     => "\xEA\x9E\x9D",             # LATIN CAPITAL LETTER VOLAPUK OE
        "\xEA\x9E\x9E"     => "\xEA\x9E\x9F",             # LATIN CAPITAL LETTER VOLAPUK UE
        "\xEA\x9E\xA0"     => "\xEA\x9E\xA1",             # LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
        "\xEA\x9E\xA2"     => "\xEA\x9E\xA3",             # LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
        "\xEA\x9E\xA4"     => "\xEA\x9E\xA5",             # LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
        "\xEA\x9E\xA6"     => "\xEA\x9E\xA7",             # LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
        "\xEA\x9E\xA8"     => "\xEA\x9E\xA9",             # LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
        "\xEA\x9E\xAA"     => "\xC9\xA6",                 # LATIN CAPITAL LETTER H WITH HOOK
        "\xEA\x9E\xAB"     => "\xC9\x9C",                 # LATIN CAPITAL LETTER REVERSED OPEN E
        "\xEA\x9E\xAC"     => "\xC9\xA1",                 # LATIN CAPITAL LETTER SCRIPT G
        "\xEA\x9E\xAD"     => "\xC9\xAC",                 # LATIN CAPITAL LETTER L WITH BELT
        "\xEA\x9E\xAE"     => "\xC9\xAA",                 # LATIN CAPITAL LETTER SMALL CAPITAL I
        "\xEA\x9E\xB0"     => "\xCA\x9E",                 # LATIN CAPITAL LETTER TURNED K
        "\xEA\x9E\xB1"     => "\xCA\x87",                 # LATIN CAPITAL LETTER TURNED T
        "\xEA\x9E\xB2"     => "\xCA\x9D",                 # LATIN CAPITAL LETTER J WITH CROSSED-TAIL
        "\xEA\x9E\xB3"     => "\xEA\xAD\x93",             # LATIN CAPITAL LETTER CHI
        "\xEA\x9E\xB4"     => "\xEA\x9E\xB5",             # LATIN CAPITAL LETTER BETA
        "\xEA\x9E\xB6"     => "\xEA\x9E\xB7",             # LATIN CAPITAL LETTER OMEGA
        "\xEA\x9E\xB8"     => "\xEA\x9E\xB9",             # LATIN CAPITAL LETTER U WITH STROKE
        "\xEA\x9E\xBA"     => "\xEA\x9E\xBB",             # LATIN CAPITAL LETTER GLOTTAL A
        "\xEA\x9E\xBC"     => "\xEA\x9E\xBD",             # LATIN CAPITAL LETTER GLOTTAL I
        "\xEA\x9E\xBE"     => "\xEA\x9E\xBF",             # LATIN CAPITAL LETTER GLOTTAL U
        "\xEA\x9F\x82"     => "\xEA\x9F\x83",             # LATIN CAPITAL LETTER ANGLICANA W
        "\xEA\x9F\x84"     => "\xEA\x9E\x94",             # LATIN CAPITAL LETTER C WITH PALATAL HOOK
        "\xEA\x9F\x85"     => "\xCA\x82",                 # LATIN CAPITAL LETTER S WITH HOOK
        "\xEA\x9F\x86"     => "\xE1\xB6\x8E",             # LATIN CAPITAL LETTER Z WITH PALATAL HOOK
        "\xEA\xAD\xB0"     => "\xE1\x8E\xA0",             # CHEROKEE SMALL LETTER A
        "\xEA\xAD\xB1"     => "\xE1\x8E\xA1",             # CHEROKEE SMALL LETTER E
        "\xEA\xAD\xB2"     => "\xE1\x8E\xA2",             # CHEROKEE SMALL LETTER I
        "\xEA\xAD\xB3"     => "\xE1\x8E\xA3",             # CHEROKEE SMALL LETTER O
        "\xEA\xAD\xB4"     => "\xE1\x8E\xA4",             # CHEROKEE SMALL LETTER U
        "\xEA\xAD\xB5"     => "\xE1\x8E\xA5",             # CHEROKEE SMALL LETTER V
        "\xEA\xAD\xB6"     => "\xE1\x8E\xA6",             # CHEROKEE SMALL LETTER GA
        "\xEA\xAD\xB7"     => "\xE1\x8E\xA7",             # CHEROKEE SMALL LETTER KA
        "\xEA\xAD\xB8"     => "\xE1\x8E\xA8",             # CHEROKEE SMALL LETTER GE
        "\xEA\xAD\xB9"     => "\xE1\x8E\xA9",             # CHEROKEE SMALL LETTER GI
        "\xEA\xAD\xBA"     => "\xE1\x8E\xAA",             # CHEROKEE SMALL LETTER GO
        "\xEA\xAD\xBB"     => "\xE1\x8E\xAB",             # CHEROKEE SMALL LETTER GU
        "\xEA\xAD\xBC"     => "\xE1\x8E\xAC",             # CHEROKEE SMALL LETTER GV
        "\xEA\xAD\xBD"     => "\xE1\x8E\xAD",             # CHEROKEE SMALL LETTER HA
        "\xEA\xAD\xBE"     => "\xE1\x8E\xAE",             # CHEROKEE SMALL LETTER HE
        "\xEA\xAD\xBF"     => "\xE1\x8E\xAF",             # CHEROKEE SMALL LETTER HI
        "\xEA\xAE\x80"     => "\xE1\x8E\xB0",             # CHEROKEE SMALL LETTER HO
        "\xEA\xAE\x81"     => "\xE1\x8E\xB1",             # CHEROKEE SMALL LETTER HU
        "\xEA\xAE\x82"     => "\xE1\x8E\xB2",             # CHEROKEE SMALL LETTER HV
        "\xEA\xAE\x83"     => "\xE1\x8E\xB3",             # CHEROKEE SMALL LETTER LA
        "\xEA\xAE\x84"     => "\xE1\x8E\xB4",             # CHEROKEE SMALL LETTER LE
        "\xEA\xAE\x85"     => "\xE1\x8E\xB5",             # CHEROKEE SMALL LETTER LI
        "\xEA\xAE\x86"     => "\xE1\x8E\xB6",             # CHEROKEE SMALL LETTER LO
        "\xEA\xAE\x87"     => "\xE1\x8E\xB7",             # CHEROKEE SMALL LETTER LU
        "\xEA\xAE\x88"     => "\xE1\x8E\xB8",             # CHEROKEE SMALL LETTER LV
        "\xEA\xAE\x89"     => "\xE1\x8E\xB9",             # CHEROKEE SMALL LETTER MA
        "\xEA\xAE\x8A"     => "\xE1\x8E\xBA",             # CHEROKEE SMALL LETTER ME
        "\xEA\xAE\x8B"     => "\xE1\x8E\xBB",             # CHEROKEE SMALL LETTER MI
        "\xEA\xAE\x8C"     => "\xE1\x8E\xBC",             # CHEROKEE SMALL LETTER MO
        "\xEA\xAE\x8D"     => "\xE1\x8E\xBD",             # CHEROKEE SMALL LETTER MU
        "\xEA\xAE\x8E"     => "\xE1\x8E\xBE",             # CHEROKEE SMALL LETTER NA
        "\xEA\xAE\x8F"     => "\xE1\x8E\xBF",             # CHEROKEE SMALL LETTER HNA
        "\xEA\xAE\x90"     => "\xE1\x8F\x80",             # CHEROKEE SMALL LETTER NAH
        "\xEA\xAE\x91"     => "\xE1\x8F\x81",             # CHEROKEE SMALL LETTER NE
        "\xEA\xAE\x92"     => "\xE1\x8F\x82",             # CHEROKEE SMALL LETTER NI
        "\xEA\xAE\x93"     => "\xE1\x8F\x83",             # CHEROKEE SMALL LETTER NO
        "\xEA\xAE\x94"     => "\xE1\x8F\x84",             # CHEROKEE SMALL LETTER NU
        "\xEA\xAE\x95"     => "\xE1\x8F\x85",             # CHEROKEE SMALL LETTER NV
        "\xEA\xAE\x96"     => "\xE1\x8F\x86",             # CHEROKEE SMALL LETTER QUA
        "\xEA\xAE\x97"     => "\xE1\x8F\x87",             # CHEROKEE SMALL LETTER QUE
        "\xEA\xAE\x98"     => "\xE1\x8F\x88",             # CHEROKEE SMALL LETTER QUI
        "\xEA\xAE\x99"     => "\xE1\x8F\x89",             # CHEROKEE SMALL LETTER QUO
        "\xEA\xAE\x9A"     => "\xE1\x8F\x8A",             # CHEROKEE SMALL LETTER QUU
        "\xEA\xAE\x9B"     => "\xE1\x8F\x8B",             # CHEROKEE SMALL LETTER QUV
        "\xEA\xAE\x9C"     => "\xE1\x8F\x8C",             # CHEROKEE SMALL LETTER SA
        "\xEA\xAE\x9D"     => "\xE1\x8F\x8D",             # CHEROKEE SMALL LETTER S
        "\xEA\xAE\x9E"     => "\xE1\x8F\x8E",             # CHEROKEE SMALL LETTER SE
        "\xEA\xAE\x9F"     => "\xE1\x8F\x8F",             # CHEROKEE SMALL LETTER SI
        "\xEA\xAE\xA0"     => "\xE1\x8F\x90",             # CHEROKEE SMALL LETTER SO
        "\xEA\xAE\xA1"     => "\xE1\x8F\x91",             # CHEROKEE SMALL LETTER SU
        "\xEA\xAE\xA2"     => "\xE1\x8F\x92",             # CHEROKEE SMALL LETTER SV
        "\xEA\xAE\xA3"     => "\xE1\x8F\x93",             # CHEROKEE SMALL LETTER DA
        "\xEA\xAE\xA4"     => "\xE1\x8F\x94",             # CHEROKEE SMALL LETTER TA
        "\xEA\xAE\xA5"     => "\xE1\x8F\x95",             # CHEROKEE SMALL LETTER DE
        "\xEA\xAE\xA6"     => "\xE1\x8F\x96",             # CHEROKEE SMALL LETTER TE
        "\xEA\xAE\xA7"     => "\xE1\x8F\x97",             # CHEROKEE SMALL LETTER DI
        "\xEA\xAE\xA8"     => "\xE1\x8F\x98",             # CHEROKEE SMALL LETTER TI
        "\xEA\xAE\xA9"     => "\xE1\x8F\x99",             # CHEROKEE SMALL LETTER DO
        "\xEA\xAE\xAA"     => "\xE1\x8F\x9A",             # CHEROKEE SMALL LETTER DU
        "\xEA\xAE\xAB"     => "\xE1\x8F\x9B",             # CHEROKEE SMALL LETTER DV
        "\xEA\xAE\xAC"     => "\xE1\x8F\x9C",             # CHEROKEE SMALL LETTER DLA
        "\xEA\xAE\xAD"     => "\xE1\x8F\x9D",             # CHEROKEE SMALL LETTER TLA
        "\xEA\xAE\xAE"     => "\xE1\x8F\x9E",             # CHEROKEE SMALL LETTER TLE
        "\xEA\xAE\xAF"     => "\xE1\x8F\x9F",             # CHEROKEE SMALL LETTER TLI
        "\xEA\xAE\xB0"     => "\xE1\x8F\xA0",             # CHEROKEE SMALL LETTER TLO
        "\xEA\xAE\xB1"     => "\xE1\x8F\xA1",             # CHEROKEE SMALL LETTER TLU
        "\xEA\xAE\xB2"     => "\xE1\x8F\xA2",             # CHEROKEE SMALL LETTER TLV
        "\xEA\xAE\xB3"     => "\xE1\x8F\xA3",             # CHEROKEE SMALL LETTER TSA
        "\xEA\xAE\xB4"     => "\xE1\x8F\xA4",             # CHEROKEE SMALL LETTER TSE
        "\xEA\xAE\xB5"     => "\xE1\x8F\xA5",             # CHEROKEE SMALL LETTER TSI
        "\xEA\xAE\xB6"     => "\xE1\x8F\xA6",             # CHEROKEE SMALL LETTER TSO
        "\xEA\xAE\xB7"     => "\xE1\x8F\xA7",             # CHEROKEE SMALL LETTER TSU
        "\xEA\xAE\xB8"     => "\xE1\x8F\xA8",             # CHEROKEE SMALL LETTER TSV
        "\xEA\xAE\xB9"     => "\xE1\x8F\xA9",             # CHEROKEE SMALL LETTER WA
        "\xEA\xAE\xBA"     => "\xE1\x8F\xAA",             # CHEROKEE SMALL LETTER WE
        "\xEA\xAE\xBB"     => "\xE1\x8F\xAB",             # CHEROKEE SMALL LETTER WI
        "\xEA\xAE\xBC"     => "\xE1\x8F\xAC",             # CHEROKEE SMALL LETTER WO
        "\xEA\xAE\xBD"     => "\xE1\x8F\xAD",             # CHEROKEE SMALL LETTER WU
        "\xEA\xAE\xBE"     => "\xE1\x8F\xAE",             # CHEROKEE SMALL LETTER WV
        "\xEA\xAE\xBF"     => "\xE1\x8F\xAF",             # CHEROKEE SMALL LETTER YA
        "\xEF\xAC\x80"     => "\x66\x66",                 # LATIN SMALL LIGATURE FF
        "\xEF\xAC\x81"     => "\x66\x69",                 # LATIN SMALL LIGATURE FI
        "\xEF\xAC\x82"     => "\x66\x6C",                 # LATIN SMALL LIGATURE FL
        "\xEF\xAC\x83"     => "\x66\x66\x69",             # LATIN SMALL LIGATURE FFI
        "\xEF\xAC\x84"     => "\x66\x66\x6C",             # LATIN SMALL LIGATURE FFL
        "\xEF\xAC\x85"     => "\x73\x74",                 # LATIN SMALL LIGATURE LONG S T
        "\xEF\xAC\x86"     => "\x73\x74",                 # LATIN SMALL LIGATURE ST
        "\xEF\xAC\x93"     => "\xD5\xB4\xD5\xB6",         # ARMENIAN SMALL LIGATURE MEN NOW
        "\xEF\xAC\x94"     => "\xD5\xB4\xD5\xA5",         # ARMENIAN SMALL LIGATURE MEN ECH
        "\xEF\xAC\x95"     => "\xD5\xB4\xD5\xAB",         # ARMENIAN SMALL LIGATURE MEN INI
        "\xEF\xAC\x96"     => "\xD5\xBE\xD5\xB6",         # ARMENIAN SMALL LIGATURE VEW NOW
        "\xEF\xAC\x97"     => "\xD5\xB4\xD5\xAD",         # ARMENIAN SMALL LIGATURE MEN XEH
        "\xEF\xBC\xA1"     => "\xEF\xBD\x81",             # FULLWIDTH LATIN CAPITAL LETTER A
        "\xEF\xBC\xA2"     => "\xEF\xBD\x82",             # FULLWIDTH LATIN CAPITAL LETTER B
        "\xEF\xBC\xA3"     => "\xEF\xBD\x83",             # FULLWIDTH LATIN CAPITAL LETTER C
        "\xEF\xBC\xA4"     => "\xEF\xBD\x84",             # FULLWIDTH LATIN CAPITAL LETTER D
        "\xEF\xBC\xA5"     => "\xEF\xBD\x85",             # FULLWIDTH LATIN CAPITAL LETTER E
        "\xEF\xBC\xA6"     => "\xEF\xBD\x86",             # FULLWIDTH LATIN CAPITAL LETTER F
        "\xEF\xBC\xA7"     => "\xEF\xBD\x87",             # FULLWIDTH LATIN CAPITAL LETTER G
        "\xEF\xBC\xA8"     => "\xEF\xBD\x88",             # FULLWIDTH LATIN CAPITAL LETTER H
        "\xEF\xBC\xA9"     => "\xEF\xBD\x89",             # FULLWIDTH LATIN CAPITAL LETTER I
        "\xEF\xBC\xAA"     => "\xEF\xBD\x8A",             # FULLWIDTH LATIN CAPITAL LETTER J
        "\xEF\xBC\xAB"     => "\xEF\xBD\x8B",             # FULLWIDTH LATIN CAPITAL LETTER K
        "\xEF\xBC\xAC"     => "\xEF\xBD\x8C",             # FULLWIDTH LATIN CAPITAL LETTER L
        "\xEF\xBC\xAD"     => "\xEF\xBD\x8D",             # FULLWIDTH LATIN CAPITAL LETTER M
        "\xEF\xBC\xAE"     => "\xEF\xBD\x8E",             # FULLWIDTH LATIN CAPITAL LETTER N
        "\xEF\xBC\xAF"     => "\xEF\xBD\x8F",             # FULLWIDTH LATIN CAPITAL LETTER O
        "\xEF\xBC\xB0"     => "\xEF\xBD\x90",             # FULLWIDTH LATIN CAPITAL LETTER P
        "\xEF\xBC\xB1"     => "\xEF\xBD\x91",             # FULLWIDTH LATIN CAPITAL LETTER Q
        "\xEF\xBC\xB2"     => "\xEF\xBD\x92",             # FULLWIDTH LATIN CAPITAL LETTER R
        "\xEF\xBC\xB3"     => "\xEF\xBD\x93",             # FULLWIDTH LATIN CAPITAL LETTER S
        "\xEF\xBC\xB4"     => "\xEF\xBD\x94",             # FULLWIDTH LATIN CAPITAL LETTER T
        "\xEF\xBC\xB5"     => "\xEF\xBD\x95",             # FULLWIDTH LATIN CAPITAL LETTER U
        "\xEF\xBC\xB6"     => "\xEF\xBD\x96",             # FULLWIDTH LATIN CAPITAL LETTER V
        "\xEF\xBC\xB7"     => "\xEF\xBD\x97",             # FULLWIDTH LATIN CAPITAL LETTER W
        "\xEF\xBC\xB8"     => "\xEF\xBD\x98",             # FULLWIDTH LATIN CAPITAL LETTER X
        "\xEF\xBC\xB9"     => "\xEF\xBD\x99",             # FULLWIDTH LATIN CAPITAL LETTER Y
        "\xEF\xBC\xBA"     => "\xEF\xBD\x9A",             # FULLWIDTH LATIN CAPITAL LETTER Z
        "\xF0\x90\x90\x80" => "\xF0\x90\x90\xA8",         # DESERET CAPITAL LETTER LONG I
        "\xF0\x90\x90\x81" => "\xF0\x90\x90\xA9",         # DESERET CAPITAL LETTER LONG E
        "\xF0\x90\x90\x82" => "\xF0\x90\x90\xAA",         # DESERET CAPITAL LETTER LONG A
        "\xF0\x90\x90\x83" => "\xF0\x90\x90\xAB",         # DESERET CAPITAL LETTER LONG AH
        "\xF0\x90\x90\x84" => "\xF0\x90\x90\xAC",         # DESERET CAPITAL LETTER LONG O
        "\xF0\x90\x90\x85" => "\xF0\x90\x90\xAD",         # DESERET CAPITAL LETTER LONG OO
        "\xF0\x90\x90\x86" => "\xF0\x90\x90\xAE",         # DESERET CAPITAL LETTER SHORT I
        "\xF0\x90\x90\x87" => "\xF0\x90\x90\xAF",         # DESERET CAPITAL LETTER SHORT E
        "\xF0\x90\x90\x88" => "\xF0\x90\x90\xB0",         # DESERET CAPITAL LETTER SHORT A
        "\xF0\x90\x90\x89" => "\xF0\x90\x90\xB1",         # DESERET CAPITAL LETTER SHORT AH
        "\xF0\x90\x90\x8A" => "\xF0\x90\x90\xB2",         # DESERET CAPITAL LETTER SHORT O
        "\xF0\x90\x90\x8B" => "\xF0\x90\x90\xB3",         # DESERET CAPITAL LETTER SHORT OO
        "\xF0\x90\x90\x8C" => "\xF0\x90\x90\xB4",         # DESERET CAPITAL LETTER AY
        "\xF0\x90\x90\x8D" => "\xF0\x90\x90\xB5",         # DESERET CAPITAL LETTER OW
        "\xF0\x90\x90\x8E" => "\xF0\x90\x90\xB6",         # DESERET CAPITAL LETTER WU
        "\xF0\x90\x90\x8F" => "\xF0\x90\x90\xB7",         # DESERET CAPITAL LETTER YEE
        "\xF0\x90\x90\x90" => "\xF0\x90\x90\xB8",         # DESERET CAPITAL LETTER H
        "\xF0\x90\x90\x91" => "\xF0\x90\x90\xB9",         # DESERET CAPITAL LETTER PEE
        "\xF0\x90\x90\x92" => "\xF0\x90\x90\xBA",         # DESERET CAPITAL LETTER BEE
        "\xF0\x90\x90\x93" => "\xF0\x90\x90\xBB",         # DESERET CAPITAL LETTER TEE
        "\xF0\x90\x90\x94" => "\xF0\x90\x90\xBC",         # DESERET CAPITAL LETTER DEE
        "\xF0\x90\x90\x95" => "\xF0\x90\x90\xBD",         # DESERET CAPITAL LETTER CHEE
        "\xF0\x90\x90\x96" => "\xF0\x90\x90\xBE",         # DESERET CAPITAL LETTER JEE
        "\xF0\x90\x90\x97" => "\xF0\x90\x90\xBF",         # DESERET CAPITAL LETTER KAY
        "\xF0\x90\x90\x98" => "\xF0\x90\x91\x80",         # DESERET CAPITAL LETTER GAY
        "\xF0\x90\x90\x99" => "\xF0\x90\x91\x81",         # DESERET CAPITAL LETTER EF
        "\xF0\x90\x90\x9A" => "\xF0\x90\x91\x82",         # DESERET CAPITAL LETTER VEE
        "\xF0\x90\x90\x9B" => "\xF0\x90\x91\x83",         # DESERET CAPITAL LETTER ETH
        "\xF0\x90\x90\x9C" => "\xF0\x90\x91\x84",         # DESERET CAPITAL LETTER THEE
        "\xF0\x90\x90\x9D" => "\xF0\x90\x91\x85",         # DESERET CAPITAL LETTER ES
        "\xF0\x90\x90\x9E" => "\xF0\x90\x91\x86",         # DESERET CAPITAL LETTER ZEE
        "\xF0\x90\x90\x9F" => "\xF0\x90\x91\x87",         # DESERET CAPITAL LETTER ESH
        "\xF0\x90\x90\xA0" => "\xF0\x90\x91\x88",         # DESERET CAPITAL LETTER ZHEE
        "\xF0\x90\x90\xA1" => "\xF0\x90\x91\x89",         # DESERET CAPITAL LETTER ER
        "\xF0\x90\x90\xA2" => "\xF0\x90\x91\x8A",         # DESERET CAPITAL LETTER EL
        "\xF0\x90\x90\xA3" => "\xF0\x90\x91\x8B",         # DESERET CAPITAL LETTER EM
        "\xF0\x90\x90\xA4" => "\xF0\x90\x91\x8C",         # DESERET CAPITAL LETTER EN
        "\xF0\x90\x90\xA5" => "\xF0\x90\x91\x8D",         # DESERET CAPITAL LETTER ENG
        "\xF0\x90\x90\xA6" => "\xF0\x90\x91\x8E",         # DESERET CAPITAL LETTER OI
        "\xF0\x90\x90\xA7" => "\xF0\x90\x91\x8F",         # DESERET CAPITAL LETTER EW
        "\xF0\x90\x92\xB0" => "\xF0\x90\x93\x98",         # OSAGE CAPITAL LETTER A
        "\xF0\x90\x92\xB1" => "\xF0\x90\x93\x99",         # OSAGE CAPITAL LETTER AI
        "\xF0\x90\x92\xB2" => "\xF0\x90\x93\x9A",         # OSAGE CAPITAL LETTER AIN
        "\xF0\x90\x92\xB3" => "\xF0\x90\x93\x9B",         # OSAGE CAPITAL LETTER AH
        "\xF0\x90\x92\xB4" => "\xF0\x90\x93\x9C",         # OSAGE CAPITAL LETTER BRA
        "\xF0\x90\x92\xB5" => "\xF0\x90\x93\x9D",         # OSAGE CAPITAL LETTER CHA
        "\xF0\x90\x92\xB6" => "\xF0\x90\x93\x9E",         # OSAGE CAPITAL LETTER EHCHA
        "\xF0\x90\x92\xB7" => "\xF0\x90\x93\x9F",         # OSAGE CAPITAL LETTER E
        "\xF0\x90\x92\xB8" => "\xF0\x90\x93\xA0",         # OSAGE CAPITAL LETTER EIN
        "\xF0\x90\x92\xB9" => "\xF0\x90\x93\xA1",         # OSAGE CAPITAL LETTER HA
        "\xF0\x90\x92\xBA" => "\xF0\x90\x93\xA2",         # OSAGE CAPITAL LETTER HYA
        "\xF0\x90\x92\xBB" => "\xF0\x90\x93\xA3",         # OSAGE CAPITAL LETTER I
        "\xF0\x90\x92\xBC" => "\xF0\x90\x93\xA4",         # OSAGE CAPITAL LETTER KA
        "\xF0\x90\x92\xBD" => "\xF0\x90\x93\xA5",         # OSAGE CAPITAL LETTER EHKA
        "\xF0\x90\x92\xBE" => "\xF0\x90\x93\xA6",         # OSAGE CAPITAL LETTER KYA
        "\xF0\x90\x92\xBF" => "\xF0\x90\x93\xA7",         # OSAGE CAPITAL LETTER LA
        "\xF0\x90\x93\x80" => "\xF0\x90\x93\xA8",         # OSAGE CAPITAL LETTER MA
        "\xF0\x90\x93\x81" => "\xF0\x90\x93\xA9",         # OSAGE CAPITAL LETTER NA
        "\xF0\x90\x93\x82" => "\xF0\x90\x93\xAA",         # OSAGE CAPITAL LETTER O
        "\xF0\x90\x93\x83" => "\xF0\x90\x93\xAB",         # OSAGE CAPITAL LETTER OIN
        "\xF0\x90\x93\x84" => "\xF0\x90\x93\xAC",         # OSAGE CAPITAL LETTER PA
        "\xF0\x90\x93\x85" => "\xF0\x90\x93\xAD",         # OSAGE CAPITAL LETTER EHPA
        "\xF0\x90\x93\x86" => "\xF0\x90\x93\xAE",         # OSAGE CAPITAL LETTER SA
        "\xF0\x90\x93\x87" => "\xF0\x90\x93\xAF",         # OSAGE CAPITAL LETTER SHA
        "\xF0\x90\x93\x88" => "\xF0\x90\x93\xB0",         # OSAGE CAPITAL LETTER TA
        "\xF0\x90\x93\x89" => "\xF0\x90\x93\xB1",         # OSAGE CAPITAL LETTER EHTA
        "\xF0\x90\x93\x8A" => "\xF0\x90\x93\xB2",         # OSAGE CAPITAL LETTER TSA
        "\xF0\x90\x93\x8B" => "\xF0\x90\x93\xB3",         # OSAGE CAPITAL LETTER EHTSA
        "\xF0\x90\x93\x8C" => "\xF0\x90\x93\xB4",         # OSAGE CAPITAL LETTER TSHA
        "\xF0\x90\x93\x8D" => "\xF0\x90\x93\xB5",         # OSAGE CAPITAL LETTER DHA
        "\xF0\x90\x93\x8E" => "\xF0\x90\x93\xB6",         # OSAGE CAPITAL LETTER U
        "\xF0\x90\x93\x8F" => "\xF0\x90\x93\xB7",         # OSAGE CAPITAL LETTER WA
        "\xF0\x90\x93\x90" => "\xF0\x90\x93\xB8",         # OSAGE CAPITAL LETTER KHA
        "\xF0\x90\x93\x91" => "\xF0\x90\x93\xB9",         # OSAGE CAPITAL LETTER GHA
        "\xF0\x90\x93\x92" => "\xF0\x90\x93\xBA",         # OSAGE CAPITAL LETTER ZA
        "\xF0\x90\x93\x93" => "\xF0\x90\x93\xBB",         # OSAGE CAPITAL LETTER ZHA
        "\xF0\x90\xB2\x80" => "\xF0\x90\xB3\x80",         # OLD HUNGARIAN CAPITAL LETTER A
        "\xF0\x90\xB2\x81" => "\xF0\x90\xB3\x81",         # OLD HUNGARIAN CAPITAL LETTER AA
        "\xF0\x90\xB2\x82" => "\xF0\x90\xB3\x82",         # OLD HUNGARIAN CAPITAL LETTER EB
        "\xF0\x90\xB2\x83" => "\xF0\x90\xB3\x83",         # OLD HUNGARIAN CAPITAL LETTER AMB
        "\xF0\x90\xB2\x84" => "\xF0\x90\xB3\x84",         # OLD HUNGARIAN CAPITAL LETTER EC
        "\xF0\x90\xB2\x85" => "\xF0\x90\xB3\x85",         # OLD HUNGARIAN CAPITAL LETTER ENC
        "\xF0\x90\xB2\x86" => "\xF0\x90\xB3\x86",         # OLD HUNGARIAN CAPITAL LETTER ECS
        "\xF0\x90\xB2\x87" => "\xF0\x90\xB3\x87",         # OLD HUNGARIAN CAPITAL LETTER ED
        "\xF0\x90\xB2\x88" => "\xF0\x90\xB3\x88",         # OLD HUNGARIAN CAPITAL LETTER AND
        "\xF0\x90\xB2\x89" => "\xF0\x90\xB3\x89",         # OLD HUNGARIAN CAPITAL LETTER E
        "\xF0\x90\xB2\x8A" => "\xF0\x90\xB3\x8A",         # OLD HUNGARIAN CAPITAL LETTER CLOSE E
        "\xF0\x90\xB2\x8B" => "\xF0\x90\xB3\x8B",         # OLD HUNGARIAN CAPITAL LETTER EE
        "\xF0\x90\xB2\x8C" => "\xF0\x90\xB3\x8C",         # OLD HUNGARIAN CAPITAL LETTER EF
        "\xF0\x90\xB2\x8D" => "\xF0\x90\xB3\x8D",         # OLD HUNGARIAN CAPITAL LETTER EG
        "\xF0\x90\xB2\x8E" => "\xF0\x90\xB3\x8E",         # OLD HUNGARIAN CAPITAL LETTER EGY
        "\xF0\x90\xB2\x8F" => "\xF0\x90\xB3\x8F",         # OLD HUNGARIAN CAPITAL LETTER EH
        "\xF0\x90\xB2\x90" => "\xF0\x90\xB3\x90",         # OLD HUNGARIAN CAPITAL LETTER I
        "\xF0\x90\xB2\x91" => "\xF0\x90\xB3\x91",         # OLD HUNGARIAN CAPITAL LETTER II
        "\xF0\x90\xB2\x92" => "\xF0\x90\xB3\x92",         # OLD HUNGARIAN CAPITAL LETTER EJ
        "\xF0\x90\xB2\x93" => "\xF0\x90\xB3\x93",         # OLD HUNGARIAN CAPITAL LETTER EK
        "\xF0\x90\xB2\x94" => "\xF0\x90\xB3\x94",         # OLD HUNGARIAN CAPITAL LETTER AK
        "\xF0\x90\xB2\x95" => "\xF0\x90\xB3\x95",         # OLD HUNGARIAN CAPITAL LETTER UNK
        "\xF0\x90\xB2\x96" => "\xF0\x90\xB3\x96",         # OLD HUNGARIAN CAPITAL LETTER EL
        "\xF0\x90\xB2\x97" => "\xF0\x90\xB3\x97",         # OLD HUNGARIAN CAPITAL LETTER ELY
        "\xF0\x90\xB2\x98" => "\xF0\x90\xB3\x98",         # OLD HUNGARIAN CAPITAL LETTER EM
        "\xF0\x90\xB2\x99" => "\xF0\x90\xB3\x99",         # OLD HUNGARIAN CAPITAL LETTER EN
        "\xF0\x90\xB2\x9A" => "\xF0\x90\xB3\x9A",         # OLD HUNGARIAN CAPITAL LETTER ENY
        "\xF0\x90\xB2\x9B" => "\xF0\x90\xB3\x9B",         # OLD HUNGARIAN CAPITAL LETTER O
        "\xF0\x90\xB2\x9C" => "\xF0\x90\xB3\x9C",         # OLD HUNGARIAN CAPITAL LETTER OO
        "\xF0\x90\xB2\x9D" => "\xF0\x90\xB3\x9D",         # OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG OE
        "\xF0\x90\xB2\x9E" => "\xF0\x90\xB3\x9E",         # OLD HUNGARIAN CAPITAL LETTER RUDIMENTA OE
        "\xF0\x90\xB2\x9F" => "\xF0\x90\xB3\x9F",         # OLD HUNGARIAN CAPITAL LETTER OEE
        "\xF0\x90\xB2\xA0" => "\xF0\x90\xB3\xA0",         # OLD HUNGARIAN CAPITAL LETTER EP
        "\xF0\x90\xB2\xA1" => "\xF0\x90\xB3\xA1",         # OLD HUNGARIAN CAPITAL LETTER EMP
        "\xF0\x90\xB2\xA2" => "\xF0\x90\xB3\xA2",         # OLD HUNGARIAN CAPITAL LETTER ER
        "\xF0\x90\xB2\xA3" => "\xF0\x90\xB3\xA3",         # OLD HUNGARIAN CAPITAL LETTER SHORT ER
        "\xF0\x90\xB2\xA4" => "\xF0\x90\xB3\xA4",         # OLD HUNGARIAN CAPITAL LETTER ES
        "\xF0\x90\xB2\xA5" => "\xF0\x90\xB3\xA5",         # OLD HUNGARIAN CAPITAL LETTER ESZ
        "\xF0\x90\xB2\xA6" => "\xF0\x90\xB3\xA6",         # OLD HUNGARIAN CAPITAL LETTER ET
        "\xF0\x90\xB2\xA7" => "\xF0\x90\xB3\xA7",         # OLD HUNGARIAN CAPITAL LETTER ENT
        "\xF0\x90\xB2\xA8" => "\xF0\x90\xB3\xA8",         # OLD HUNGARIAN CAPITAL LETTER ETY
        "\xF0\x90\xB2\xA9" => "\xF0\x90\xB3\xA9",         # OLD HUNGARIAN CAPITAL LETTER ECH
        "\xF0\x90\xB2\xAA" => "\xF0\x90\xB3\xAA",         # OLD HUNGARIAN CAPITAL LETTER U
        "\xF0\x90\xB2\xAB" => "\xF0\x90\xB3\xAB",         # OLD HUNGARIAN CAPITAL LETTER UU
        "\xF0\x90\xB2\xAC" => "\xF0\x90\xB3\xAC",         # OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG UE
        "\xF0\x90\xB2\xAD" => "\xF0\x90\xB3\xAD",         # OLD HUNGARIAN CAPITAL LETTER RUDIMENTA UE
        "\xF0\x90\xB2\xAE" => "\xF0\x90\xB3\xAE",         # OLD HUNGARIAN CAPITAL LETTER EV
        "\xF0\x90\xB2\xAF" => "\xF0\x90\xB3\xAF",         # OLD HUNGARIAN CAPITAL LETTER EZ
        "\xF0\x90\xB2\xB0" => "\xF0\x90\xB3\xB0",         # OLD HUNGARIAN CAPITAL LETTER EZS
        "\xF0\x90\xB2\xB1" => "\xF0\x90\xB3\xB1",         # OLD HUNGARIAN CAPITAL LETTER ENT-SHAPED SIGN
        "\xF0\x90\xB2\xB2" => "\xF0\x90\xB3\xB2",         # OLD HUNGARIAN CAPITAL LETTER US
        "\xF0\x91\xA2\xA0" => "\xF0\x91\xA3\x80",         # WARANG CITI CAPITAL LETTER NGAA
        "\xF0\x91\xA2\xA1" => "\xF0\x91\xA3\x81",         # WARANG CITI CAPITAL LETTER A
        "\xF0\x91\xA2\xA2" => "\xF0\x91\xA3\x82",         # WARANG CITI CAPITAL LETTER WI
        "\xF0\x91\xA2\xA3" => "\xF0\x91\xA3\x83",         # WARANG CITI CAPITAL LETTER YU
        "\xF0\x91\xA2\xA4" => "\xF0\x91\xA3\x84",         # WARANG CITI CAPITAL LETTER YA
        "\xF0\x91\xA2\xA5" => "\xF0\x91\xA3\x85",         # WARANG CITI CAPITAL LETTER YO
        "\xF0\x91\xA2\xA6" => "\xF0\x91\xA3\x86",         # WARANG CITI CAPITAL LETTER II
        "\xF0\x91\xA2\xA7" => "\xF0\x91\xA3\x87",         # WARANG CITI CAPITAL LETTER UU
        "\xF0\x91\xA2\xA8" => "\xF0\x91\xA3\x88",         # WARANG CITI CAPITAL LETTER E
        "\xF0\x91\xA2\xA9" => "\xF0\x91\xA3\x89",         # WARANG CITI CAPITAL LETTER O
        "\xF0\x91\xA2\xAA" => "\xF0\x91\xA3\x8A",         # WARANG CITI CAPITAL LETTER ANG
        "\xF0\x91\xA2\xAB" => "\xF0\x91\xA3\x8B",         # WARANG CITI CAPITAL LETTER GA
        "\xF0\x91\xA2\xAC" => "\xF0\x91\xA3\x8C",         # WARANG CITI CAPITAL LETTER KO
        "\xF0\x91\xA2\xAD" => "\xF0\x91\xA3\x8D",         # WARANG CITI CAPITAL LETTER ENY
        "\xF0\x91\xA2\xAE" => "\xF0\x91\xA3\x8E",         # WARANG CITI CAPITAL LETTER YUJ
        "\xF0\x91\xA2\xAF" => "\xF0\x91\xA3\x8F",         # WARANG CITI CAPITAL LETTER UC
        "\xF0\x91\xA2\xB0" => "\xF0\x91\xA3\x90",         # WARANG CITI CAPITAL LETTER ENN
        "\xF0\x91\xA2\xB1" => "\xF0\x91\xA3\x91",         # WARANG CITI CAPITAL LETTER ODD
        "\xF0\x91\xA2\xB2" => "\xF0\x91\xA3\x92",         # WARANG CITI CAPITAL LETTER TTE
        "\xF0\x91\xA2\xB3" => "\xF0\x91\xA3\x93",         # WARANG CITI CAPITAL LETTER NUNG
        "\xF0\x91\xA2\xB4" => "\xF0\x91\xA3\x94",         # WARANG CITI CAPITAL LETTER DA
        "\xF0\x91\xA2\xB5" => "\xF0\x91\xA3\x95",         # WARANG CITI CAPITAL LETTER AT
        "\xF0\x91\xA2\xB6" => "\xF0\x91\xA3\x96",         # WARANG CITI CAPITAL LETTER AM
        "\xF0\x91\xA2\xB7" => "\xF0\x91\xA3\x97",         # WARANG CITI CAPITAL LETTER BU
        "\xF0\x91\xA2\xB8" => "\xF0\x91\xA3\x98",         # WARANG CITI CAPITAL LETTER PU
        "\xF0\x91\xA2\xB9" => "\xF0\x91\xA3\x99",         # WARANG CITI CAPITAL LETTER HIYO
        "\xF0\x91\xA2\xBA" => "\xF0\x91\xA3\x9A",         # WARANG CITI CAPITAL LETTER HOLO
        "\xF0\x91\xA2\xBB" => "\xF0\x91\xA3\x9B",         # WARANG CITI CAPITAL LETTER HORR
        "\xF0\x91\xA2\xBC" => "\xF0\x91\xA3\x9C",         # WARANG CITI CAPITAL LETTER HAR
        "\xF0\x91\xA2\xBD" => "\xF0\x91\xA3\x9D",         # WARANG CITI CAPITAL LETTER SSUU
        "\xF0\x91\xA2\xBE" => "\xF0\x91\xA3\x9E",         # WARANG CITI CAPITAL LETTER SII
        "\xF0\x91\xA2\xBF" => "\xF0\x91\xA3\x9F",         # WARANG CITI CAPITAL LETTER VIYO
        "\xF0\x96\xB9\x80" => "\xF0\x96\xB9\xA0",         # MEDEFAIDRIN CAPITAL LETTER M
        "\xF0\x96\xB9\x81" => "\xF0\x96\xB9\xA1",         # MEDEFAIDRIN CAPITAL LETTER S
        "\xF0\x96\xB9\x82" => "\xF0\x96\xB9\xA2",         # MEDEFAIDRIN CAPITAL LETTER V
        "\xF0\x96\xB9\x83" => "\xF0\x96\xB9\xA3",         # MEDEFAIDRIN CAPITAL LETTER W
        "\xF0\x96\xB9\x84" => "\xF0\x96\xB9\xA4",         # MEDEFAIDRIN CAPITAL LETTER ATIU
        "\xF0\x96\xB9\x85" => "\xF0\x96\xB9\xA5",         # MEDEFAIDRIN CAPITAL LETTER Z
        "\xF0\x96\xB9\x86" => "\xF0\x96\xB9\xA6",         # MEDEFAIDRIN CAPITAL LETTER KP
        "\xF0\x96\xB9\x87" => "\xF0\x96\xB9\xA7",         # MEDEFAIDRIN CAPITAL LETTER P
        "\xF0\x96\xB9\x88" => "\xF0\x96\xB9\xA8",         # MEDEFAIDRIN CAPITAL LETTER T
        "\xF0\x96\xB9\x89" => "\xF0\x96\xB9\xA9",         # MEDEFAIDRIN CAPITAL LETTER G
        "\xF0\x96\xB9\x8A" => "\xF0\x96\xB9\xAA",         # MEDEFAIDRIN CAPITAL LETTER F
        "\xF0\x96\xB9\x8B" => "\xF0\x96\xB9\xAB",         # MEDEFAIDRIN CAPITAL LETTER I
        "\xF0\x96\xB9\x8C" => "\xF0\x96\xB9\xAC",         # MEDEFAIDRIN CAPITAL LETTER K
        "\xF0\x96\xB9\x8D" => "\xF0\x96\xB9\xAD",         # MEDEFAIDRIN CAPITAL LETTER A
        "\xF0\x96\xB9\x8E" => "\xF0\x96\xB9\xAE",         # MEDEFAIDRIN CAPITAL LETTER J
        "\xF0\x96\xB9\x8F" => "\xF0\x96\xB9\xAF",         # MEDEFAIDRIN CAPITAL LETTER E
        "\xF0\x96\xB9\x90" => "\xF0\x96\xB9\xB0",         # MEDEFAIDRIN CAPITAL LETTER B
        "\xF0\x96\xB9\x91" => "\xF0\x96\xB9\xB1",         # MEDEFAIDRIN CAPITAL LETTER C
        "\xF0\x96\xB9\x92" => "\xF0\x96\xB9\xB2",         # MEDEFAIDRIN CAPITAL LETTER U
        "\xF0\x96\xB9\x93" => "\xF0\x96\xB9\xB3",         # MEDEFAIDRIN CAPITAL LETTER YU
        "\xF0\x96\xB9\x94" => "\xF0\x96\xB9\xB4",         # MEDEFAIDRIN CAPITAL LETTER L
        "\xF0\x96\xB9\x95" => "\xF0\x96\xB9\xB5",         # MEDEFAIDRIN CAPITAL LETTER Q
        "\xF0\x96\xB9\x96" => "\xF0\x96\xB9\xB6",         # MEDEFAIDRIN CAPITAL LETTER HP
        "\xF0\x96\xB9\x97" => "\xF0\x96\xB9\xB7",         # MEDEFAIDRIN CAPITAL LETTER NY
        "\xF0\x96\xB9\x98" => "\xF0\x96\xB9\xB8",         # MEDEFAIDRIN CAPITAL LETTER X
        "\xF0\x96\xB9\x99" => "\xF0\x96\xB9\xB9",         # MEDEFAIDRIN CAPITAL LETTER D
        "\xF0\x96\xB9\x9A" => "\xF0\x96\xB9\xBA",         # MEDEFAIDRIN CAPITAL LETTER OE
        "\xF0\x96\xB9\x9B" => "\xF0\x96\xB9\xBB",         # MEDEFAIDRIN CAPITAL LETTER N
        "\xF0\x96\xB9\x9C" => "\xF0\x96\xB9\xBC",         # MEDEFAIDRIN CAPITAL LETTER R
        "\xF0\x96\xB9\x9D" => "\xF0\x96\xB9\xBD",         # MEDEFAIDRIN CAPITAL LETTER O
        "\xF0\x96\xB9\x9E" => "\xF0\x96\xB9\xBE",         # MEDEFAIDRIN CAPITAL LETTER AI
        "\xF0\x96\xB9\x9F" => "\xF0\x96\xB9\xBF",         # MEDEFAIDRIN CAPITAL LETTER Y
        "\xF0\x9E\xA4\x80" => "\xF0\x9E\xA4\xA2",         # ADLAM CAPITAL LETTER ALIF
        "\xF0\x9E\xA4\x81" => "\xF0\x9E\xA4\xA3",         # ADLAM CAPITAL LETTER DAALI
        "\xF0\x9E\xA4\x82" => "\xF0\x9E\xA4\xA4",         # ADLAM CAPITAL LETTER LAAM
        "\xF0\x9E\xA4\x83" => "\xF0\x9E\xA4\xA5",         # ADLAM CAPITAL LETTER MIIM
        "\xF0\x9E\xA4\x84" => "\xF0\x9E\xA4\xA6",         # ADLAM CAPITAL LETTER BA
        "\xF0\x9E\xA4\x85" => "\xF0\x9E\xA4\xA7",         # ADLAM CAPITAL LETTER SINNYIIYHE
        "\xF0\x9E\xA4\x86" => "\xF0\x9E\xA4\xA8",         # ADLAM CAPITAL LETTER PE
        "\xF0\x9E\xA4\x87" => "\xF0\x9E\xA4\xA9",         # ADLAM CAPITAL LETTER BHE
        "\xF0\x9E\xA4\x88" => "\xF0\x9E\xA4\xAA",         # ADLAM CAPITAL LETTER RA
        "\xF0\x9E\xA4\x89" => "\xF0\x9E\xA4\xAB",         # ADLAM CAPITAL LETTER E
        "\xF0\x9E\xA4\x8A" => "\xF0\x9E\xA4\xAC",         # ADLAM CAPITAL LETTER FA
        "\xF0\x9E\xA4\x8B" => "\xF0\x9E\xA4\xAD",         # ADLAM CAPITAL LETTER I
        "\xF0\x9E\xA4\x8C" => "\xF0\x9E\xA4\xAE",         # ADLAM CAPITAL LETTER O
        "\xF0\x9E\xA4\x8D" => "\xF0\x9E\xA4\xAF",         # ADLAM CAPITAL LETTER DHA
        "\xF0\x9E\xA4\x8E" => "\xF0\x9E\xA4\xB0",         # ADLAM CAPITAL LETTER YHE
        "\xF0\x9E\xA4\x8F" => "\xF0\x9E\xA4\xB1",         # ADLAM CAPITAL LETTER WAW
        "\xF0\x9E\xA4\x90" => "\xF0\x9E\xA4\xB2",         # ADLAM CAPITAL LETTER NUN
        "\xF0\x9E\xA4\x91" => "\xF0\x9E\xA4\xB3",         # ADLAM CAPITAL LETTER KAF
        "\xF0\x9E\xA4\x92" => "\xF0\x9E\xA4\xB4",         # ADLAM CAPITAL LETTER YA
        "\xF0\x9E\xA4\x93" => "\xF0\x9E\xA4\xB5",         # ADLAM CAPITAL LETTER U
        "\xF0\x9E\xA4\x94" => "\xF0\x9E\xA4\xB6",         # ADLAM CAPITAL LETTER JIIM
        "\xF0\x9E\xA4\x95" => "\xF0\x9E\xA4\xB7",         # ADLAM CAPITAL LETTER CHI
        "\xF0\x9E\xA4\x96" => "\xF0\x9E\xA4\xB8",         # ADLAM CAPITAL LETTER HA
        "\xF0\x9E\xA4\x97" => "\xF0\x9E\xA4\xB9",         # ADLAM CAPITAL LETTER QAAF
        "\xF0\x9E\xA4\x98" => "\xF0\x9E\xA4\xBA",         # ADLAM CAPITAL LETTER GA
        "\xF0\x9E\xA4\x99" => "\xF0\x9E\xA4\xBB",         # ADLAM CAPITAL LETTER NYA
        "\xF0\x9E\xA4\x9A" => "\xF0\x9E\xA4\xBC",         # ADLAM CAPITAL LETTER TU
        "\xF0\x9E\xA4\x9B" => "\xF0\x9E\xA4\xBD",         # ADLAM CAPITAL LETTER NHA
        "\xF0\x9E\xA4\x9C" => "\xF0\x9E\xA4\xBE",         # ADLAM CAPITAL LETTER VA
        "\xF0\x9E\xA4\x9D" => "\xF0\x9E\xA4\xBF",         # ADLAM CAPITAL LETTER KHA
        "\xF0\x9E\xA4\x9E" => "\xF0\x9E\xA5\x80",         # ADLAM CAPITAL LETTER GBE
        "\xF0\x9E\xA4\x9F" => "\xF0\x9E\xA5\x81",         # ADLAM CAPITAL LETTER ZAL
        "\xF0\x9E\xA4\xA0" => "\xF0\x9E\xA5\x82",         # ADLAM CAPITAL LETTER KPO
        "\xF0\x9E\xA4\xA1" => "\xF0\x9E\xA5\x83",         # ADLAM CAPITAL LETTER SHA
    );
}

#endif
#if DIST_DIS10646JP
elsif (__PACKAGE__ =~ / \b Edis10646jp \z/oxms) {
    %range_tr = (
        1 => [ [0x00..0x1F],
               [0x80..0x9F],
             ],
        4 => [ [0x20..0x20],[0x40..0x4F],[0x20..0x7F],[0x20..0x7F],
               [0x20..0x20],[0x40..0x4F],[0x20..0x7F],[0xA0..0xFF],
               [0x20..0x20],[0x40..0x4F],[0xA0..0xFF],[0x20..0x7F],
               [0x20..0x20],[0x40..0x4F],[0xA0..0xFF],[0xA0..0xFF],
             ],
    );
}

#endif
else {
    croak "Don't know my package name '@{[__PACKAGE__]}'";
}

#
# @ARGV wildcard globbing
#
sub import {

    if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
        my @argv = ();
        for (@ARGV) {

            # has space
            if (/\A (?:$q_char)*? [ ] /oxms) {
                if (my @glob = Esjis::glob(qq{"$_"})) {
                    push @argv, @glob;
                }
                else {
                    push @argv, $_;
                }
            }

            # has wildcard metachar
            elsif (/\A (?:$q_char)*? [*?] /oxms) {
                if (my @glob = Esjis::glob($_)) {
                    push @argv, @glob;
                }
                else {
                    push @argv, $_;
                }
            }

            # no wildcard globbing
            else {
                push @argv, $_;
            }
        }
        @ARGV = @argv;
    }

    *Char::ord           = \&Sjis::ord;
    *Char::ord_          = \&Sjis::ord_;
    *Char::reverse       = \&Sjis::reverse;
    *Char::getc          = \&Sjis::getc;
    *Char::length        = \&Sjis::length;
    *Char::substr        = \&Sjis::substr;
    *Char::index         = \&Sjis::index;
    *Char::rindex        = \&Sjis::rindex;
    *Char::eval          = \&Sjis::eval;
    *Char::escape        = \&Sjis::escape;
    *Char::escape_token  = \&Sjis::escape_token;
    *Char::escape_script = \&Sjis::escape_script;
}

# P.230 Care with Prototypes
# in Chapter 6: Subroutines
# of ISBN 0-596-00027-8 Programming Perl Third Edition.
#
# If you aren't careful, you can get yourself into trouble with prototypes.
# But if you are careful, you can do a lot of neat things with them. This is
# all very powerful, of course, and should only be used in moderation to make
# the world a better place.

# P.332 Care with Prototypes
# in Chapter 7: Subroutines
# of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
#
# If you aren't careful, you can get yourself into trouble with prototypes.
# But if you are careful, you can do a lot of neat things with them. This is
# all very powerful, of course, and should only be used in moderation to make
# the world a better place.

#
# Prototypes of subroutines
#
sub unimport {}
sub Esjis::split(;$$$);
sub Esjis::tr($$$$;$);
sub Esjis::chop(@);
sub Esjis::index($$;$);
sub Esjis::rindex($$;$);
sub Esjis::lcfirst(@);
sub Esjis::lcfirst_();
sub Esjis::lc(@);
sub Esjis::lc_();
sub Esjis::ucfirst(@);
sub Esjis::ucfirst_();
sub Esjis::uc(@);
sub Esjis::uc_();
sub Esjis::fc(@);
sub Esjis::fc_();
sub Esjis::ignorecase;
sub Esjis::classic_character_class;
sub Esjis::capture;
sub Esjis::chr(;$);
sub Esjis::chr_();
#if ESCAPE_SECOND_OCTET
sub Esjis::filetest;
sub Esjis::r(;*@);
sub Esjis::w(;*@);
sub Esjis::x(;*@);
sub Esjis::o(;*@);
sub Esjis::R(;*@);
sub Esjis::W(;*@);
sub Esjis::X(;*@);
sub Esjis::O(;*@);
sub Esjis::e(;*@);
sub Esjis::z(;*@);
sub Esjis::s(;*@);
sub Esjis::f(;*@);
sub Esjis::d(;*@);
sub Esjis::l(;*@);
sub Esjis::p(;*@);
sub Esjis::S(;*@);
sub Esjis::b(;*@);
sub Esjis::c(;*@);
sub Esjis::u(;*@);
sub Esjis::g(;*@);
sub Esjis::k(;*@);
sub Esjis::T(;*@);
sub Esjis::B(;*@);
sub Esjis::M(;*@);
sub Esjis::A(;*@);
sub Esjis::C(;*@);
sub Esjis::filetest_;
sub Esjis::r_();
sub Esjis::w_();
sub Esjis::x_();
sub Esjis::o_();
sub Esjis::R_();
sub Esjis::W_();
sub Esjis::X_();
sub Esjis::O_();
sub Esjis::e_();
sub Esjis::z_();
sub Esjis::s_();
sub Esjis::f_();
sub Esjis::d_();
sub Esjis::l_();
sub Esjis::p_();
sub Esjis::S_();
sub Esjis::b_();
sub Esjis::c_();
sub Esjis::u_();
sub Esjis::g_();
sub Esjis::k_();
sub Esjis::T_();
sub Esjis::B_();
sub Esjis::M_();
sub Esjis::A_();
sub Esjis::C_();
#endif
sub Esjis::glob($);
sub Esjis::glob_();
#if ESCAPE_SECOND_OCTET
sub Esjis::lstat(*);
sub Esjis::lstat_();
sub Esjis::opendir(*$);
sub Esjis::stat(*);
sub Esjis::stat_();
sub Esjis::unlink(@);
sub Esjis::chdir(;$);
sub Esjis::do($);
sub Esjis::require(;$);
sub Esjis::telldir(*);
#endif

sub Sjis::ord(;$);
sub Sjis::ord_();
sub Sjis::reverse(@);
sub Sjis::getc(;*@);
sub Sjis::length(;$);
sub Sjis::substr($$;$$);
sub Sjis::index($$;$);
sub Sjis::rindex($$;$);
sub Sjis::escape(;$);

#
# Regexp work
#
use vars qw(
    $re_a
    $re_t
    $re_n
    $re_r
);

#
# Character class
#
use vars qw(
    $dot
    $dot_s
    $eD
    $eS
    $eW
    $eH
    $eV
    $eR
    $eN
    $not_alnum
    $not_alpha
    $not_ascii
    $not_blank
    $not_cntrl
    $not_digit
    $not_graph
    $not_lower
    $not_lower_i
    $not_print
    $not_punct
    $not_space
    $not_upper
    $not_upper_i
    $not_word
    $not_xdigit
    $eb
    $eB
);

#if MULTIBYTE_ANCHORING
use vars qw(
    $anchor
    $matched
);
${Esjis::anchor} = qr{\G(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?}oxms;
#endif
#if DIST_GB18030

    # unless LONG_STRING_FOR_RE
    if (1) {
    }

#endif
#if DIST_EUCTW

    # unless LONG_STRING_FOR_RE
    if (1) {
    }

#endif
#if DIST_EUCJP

    # unless LONG_STRING_FOR_RE
    if (1) {
    }

#endif
#if DIST_INFORMIXV6ALS

    # unless LONG_STRING_FOR_RE
    if (1) {
    }

#endif
#if LONG_STRING_FOR_RE
my $q_char_SADAHIRO_Tomoyuki_2002_01_17 = '';

# Quantifiers
#   {n,m}  ---  Match at least n but not more than m times
#
# n and m are limited to non-negative integral values less than a
# preset limit defined when perl is built. This is usually 32766 on
# the most common platforms.
#
# The following code is an attempt to solve the above limitations
# in a multi-byte anchoring.

# avoid "Segmentation fault" and "Error: Parse exception"

# perl5101delta
# http://perldoc.perl.org/perl5101delta.html
# In 5.10.0, the * quantifier in patterns was sometimes treated as {0,32767}
# [RT #60034, #60464]. For example, this match would fail:
#   ("ab" x 32768) =~ /^(ab)*$/

# SEE ALSO
#
# Complex regular subexpression recursion limit
# http://www.perlmonks.org/?node_id=810857
#
# regexp iteration limits
# http://www.nntp.perl.org/group/perl.perl5.porters/2009/02/msg144065.html
#
# latest Perl won't match certain regexes more than 32768 characters long
# http://stackoverflow.com/questions/26226630/latest-perl-wont-match-certain-regexes-more-than-32768-characters-long
#
# Break through the limitations of regular expressions of Perl
# http://d.hatena.ne.jp/gfx/20110212/1297512479

if (($] >= 5.010001) or
    # ActivePerl 5.6 or later (include 5.10.0)
    (defined($ActivePerl::VERSION) and ($ActivePerl::VERSION > 800)) or
    (($^O eq 'MSWin32') and ($] =~ /\A 5\.006/oxms))
) {
    my $sbcs = ''; # Single Byte Character Set
    for my $range (@{ $range_tr{1} }) {
        $sbcs .= sprintf('\\x%02X-\\x%02X', $range->[0], $range->[-1]);
    }

    if (0) {
    }

#endif
#if DIST_GB18030
    # GB18030 encoding
    elsif (__PACKAGE__ =~ / \b Egb18030 \z/oxms) {
        ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17} = qr{.*?[^\x30-\x39\x81-\xFE](?>[\x30-\x39]|[\x81-\xFE][\x81-\xFE]|[\x81-\xFE][\x30-\x39][\x81-\xFE][\x30-\x39])*?}oxms;
        #                                                     ********************* octets not in multiple octet char (always char boundary)
        #                                                                             *********** 1 octet chars in multiple octet char
        #                                                                                         ********************** 2 octet chars
        #                                                                                                                ******************************************** 4 octet chars
    }

#endif
#if DIST_EUCTW
    # EUC-TW encoding
    elsif (__PACKAGE__ =~ / \b Eeuctw \z/oxms) {
        ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17} = qr{.*?[^\x8E\xA1-\xFE]     (?>            [\xA1-\xFE][\xA1-\xFE]|\x8E[\xA1-\xB0][\xA1-\xFE][\xA1-\xFE]       )*?}oxms;
        #                                                     **************** octets not in multiple octet char (always char boundary)
        #                                                                                         ********************** 2 octet chars
        #                                                                                                                ************************************* 4 octet chars
    }

#endif
#if DIST_EUCJP
    # EUC-JP encoding
    elsif (__PACKAGE__ =~ / \b Eeucjp \z/oxms) {
        ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17} = qr{.*?[^\x8E\x8F\xA1-\xFE] (?>            [\x8E\xA1-\xFE][\xA1-\xFE]|\x8F[\xA1-\xFE][\xA1-\xFE] )*?}oxms;
        #                                                     ******************** octets not in multiple octet char (always char boundary)
        #                                                                                         ************************** 2 octet chars
        #                                                                                                                    ************************** 3 octet chars
    }

#endif
#if DIST_INFORMIXV6ALS
    # INFORMIX V6 ALS encoding
    elsif (__PACKAGE__ =~ / \b Einformixv6als \z/oxms) {
        ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17} = qr{.*?[\x00-\x80\xA0-\xDF\xFF](?>[\x81-\x9F\xE0-\xFC][\x40-\x7E\x80-\xFC]|\xFD[\xA1-\xFE][\xA1-\xFE]|\xFE)*?}oxms;
        #                                                     ************************ octets not in multiple octet char (always char boundary)
        #                                                                                **************************************** 2 octet chars
        #                                                                                                                         ************************** 3 octet chars
        #                                                                                                                                                    **** malformed octet?
    }

#endif
#if LONG_STRING_FOR_RE
    # other encoding
    else {
        ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17} = qr{.*?[$sbcs](?:[^$sbcs][^$sbcs])*?}oxms;
        #                                                     ******* octets not in multiple octet char (always char boundary)
        #                                                               **************** 2 octet chars
    }

    ${Esjis::anchor_SADAHIRO_Tomoyuki_2002_01_17} =
    qr{\G(?(?=.{0,32766}\z)(?:[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?|(?(?=[$sbcs]+\z).*?|(?:${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17})))}oxms;
#   qr{
#      \G # (1), (2)
#        (? # (3)
#          (?=.{0,32766}\z) # (4)
#                          (?:[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?| # (5)
#                                                                                      (?(?=[$sbcs]+\z) # (6)
#                                                                                                      .*?| #(7)
#                                                                                                          (?:${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17}) # (8)
#                                                                                                                                           ))}oxms;

    # avoid: Complex regular subexpression recursion limit (32766) exceeded at here.
    local $^W = 0;
    local $SIG{__WARN__} = sub {};

    if (((('A' x 32768).'B') !~ / ${Esjis::anchor}                              B /oxms) and
        ((('A' x 32768).'B') =~ / ${Esjis::anchor_SADAHIRO_Tomoyuki_2002_01_17} B /oxms)
    ) {
        ${Esjis::anchor} = ${Esjis::anchor_SADAHIRO_Tomoyuki_2002_01_17};
    }
    else {
        undef ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17};
    }
}

# (1)
# P.128 Start of match (or end of previous match): \G
# P.130 Advanced Use of \G with Perl
# in Chapter3: Over view of Regular Expression Features and Flavors
# of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

# (2)
# P.255 Use leading anchors
# P.256 Expose ^ and \G at the front of expressions
# in Chapter6: Crafting an Efficient Expression
# of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

# (3)
# P.138 Conditional: (? if then| else)
# in Chapter3: Over view of Regular Expression Features and Flavors
# of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

# (4)
# perlre
# http://perldoc.perl.org/perlre.html
# The "*" quantifier is equivalent to {0,} , the "+" quantifier to {1,} ,
# and the "?" quantifier to {0,1}, ., n, and m are limited to non-negative
# integral values less than a preset limit defined when perl is built.
# This is usually 32766 on the most common platforms. The actual limit
# can be seen in the error message generated by code such as this:
#  $_ **= $_ , / {$_} / for 2 .. 42;

# (5)
# P.1023 Multiple-Byte Anchoring
# in Appendix W Perl Code Examples
# of ISBN 1-56592-224-7 CJKV Information Processing

# (6)
# if string has only SBCS (Single Byte Character Set)

# (7)
# then .*? (isn't limited to 32766)

# (8)
# else ShiftJIS::Regexp::Const (SADAHIRO Tomoyuki)
# http://homepage1.nifty.com/nomenclator/perl/shiftjis.htm#long
# http://search.cpan.org/~sadahiro/ShiftJIS-Regexp/
# $PadA  = '  (?:\A|                                           [\x00-\x80\xA0-\xDF])(?:[\x81-\x9F\xE0-\xFC]{2})*?';
# $PadG  = '\G(?:                                |[\x00-\xFF]*?[\x00-\x80\xA0-\xDF])(?:[\x81-\x9F\xE0-\xFC]{2})*?';
# $PadGA = '\G(?:\A|(?:[\x81-\x9F\xE0-\xFC]{2})+?|[\x00-\xFF]*?[\x00-\x80\xA0-\xDF] (?:[\x81-\x9F\xE0-\xFC]{2})*?)';

#endif
${Esjis::dot}         = qr{(?>[^\x81-\x9F\xE0-\xFC\x0A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::dot_s}       = qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eD}          = qr{(?>[^\x81-\x9F\xE0-\xFC0-9]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};

# Vertical tabs are now whitespace
# \s in a regex now matches a vertical tab in all circumstances.
# http://search.cpan.org/dist/perl-5.18.0/pod/perldelta.pod#Vertical_tabs_are_now_whitespace
# ${Esjis::eS}          = qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x0A    \x0C\x0D\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
# ${Esjis::eS}          = qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x0A\x0B\x0C\x0D\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eS}            = qr{(?>[^\x81-\x9F\xE0-\xFC\s]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};

${Esjis::eW}            = qr{(?>[^\x81-\x9F\xE0-\xFC0-9A-Z_a-z]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eH}            = qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eV}            = qr{(?>[^\x81-\x9F\xE0-\xFC\x0A\x0B\x0C\x0D]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eR}            = qr{(?>\x0D\x0A|[\x0A\x0D])};
${Esjis::eN}            = qr{(?>[^\x81-\x9F\xE0-\xFC\x0A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_alnum}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_alpha}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_ascii}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x00-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_blank}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x09\x20]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_cntrl}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x00-\x1F\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_digit}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_graph}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x21-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_lower}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_lower_i}   = qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])}; # Perl 5.16 compatible
# ${Esjis::not_lower_i} = qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};                   # older Perl compatible
${Esjis::not_print}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x20-\x7F]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_punct}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x21-\x2F\x3A-\x3F\x40\x5B-\x5F\x60\x7B-\x7E]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_space}     = qr{(?>[^\x81-\x9F\xE0-\xFC\s\x0B]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_upper}     = qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_upper_i}   = qr{(?>[^\x81-\x9F\xE0-\xFC\x41-\x5A\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])}; # Perl 5.16 compatible
# ${Esjis::not_upper_i} = qr{(?>[^\x81-\x9F\xE0-\xFC]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};                   # older Perl compatible
${Esjis::not_word}      = qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x5A\x5F\x61-\x7A]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::not_xdigit}    = qr{(?>[^\x81-\x9F\xE0-\xFC\x30-\x39\x41-\x46\x61-\x66]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])};
${Esjis::eb}            = qr{(?:\A(?=[0-9A-Z_a-z])|(?<=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF])(?=[0-9A-Z_a-z])|(?<=[0-9A-Z_a-z])(?=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF]|\z))};
${Esjis::eB}            = qr{(?:(?<=[0-9A-Z_a-z])(?=[0-9A-Z_a-z])|(?<=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF])(?=[\x00-\x2F\x40\x5B-\x5E\x60\x7B-\xFF]))};

# avoid: Name "Esjis::foo" used only once: possible typo at here.
${Esjis::dot}         = ${Esjis::dot};
${Esjis::dot_s}       = ${Esjis::dot_s};
${Esjis::eD}          = ${Esjis::eD};
${Esjis::eS}          = ${Esjis::eS};
${Esjis::eW}          = ${Esjis::eW};
${Esjis::eH}          = ${Esjis::eH};
${Esjis::eV}          = ${Esjis::eV};
${Esjis::eR}          = ${Esjis::eR};
${Esjis::eN}          = ${Esjis::eN};
${Esjis::not_alnum}   = ${Esjis::not_alnum};
${Esjis::not_alpha}   = ${Esjis::not_alpha};
${Esjis::not_ascii}   = ${Esjis::not_ascii};
${Esjis::not_blank}   = ${Esjis::not_blank};
${Esjis::not_cntrl}   = ${Esjis::not_cntrl};
${Esjis::not_digit}   = ${Esjis::not_digit};
${Esjis::not_graph}   = ${Esjis::not_graph};
${Esjis::not_lower}   = ${Esjis::not_lower};
${Esjis::not_lower_i} = ${Esjis::not_lower_i};
${Esjis::not_print}   = ${Esjis::not_print};
${Esjis::not_punct}   = ${Esjis::not_punct};
${Esjis::not_space}   = ${Esjis::not_space};
${Esjis::not_upper}   = ${Esjis::not_upper};
${Esjis::not_upper_i} = ${Esjis::not_upper_i};
${Esjis::not_word}    = ${Esjis::not_word};
${Esjis::not_xdigit}  = ${Esjis::not_xdigit};
${Esjis::eb}          = ${Esjis::eb};
${Esjis::eB}          = ${Esjis::eB};

#
# ShiftJIS split
#
sub Esjis::split(;$$$) {

    # P.794 29.2.161. split
    # in Chapter 29: Functions
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.951 split
    # in Chapter 27: Functions
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    my $pattern = $_[0];
    my $string  = $_[1];
    my $limit   = $_[2];

    # if $pattern is also omitted or is the literal space, " "
    if (not defined $pattern) {
        $pattern = ' ';
    }

    # if $string is omitted, the function splits the $_ string
    if (not defined $string) {
        if (defined $_) {
            $string = $_;
        }
        else {
            $string = '';
        }
    }

    my @split = ();

    # when string is empty
    if ($string eq '') {

        # resulting list value in list context
        if (wantarray) {
            return @split;
        }

        # count of substrings in scalar context
        else {
            carp "Use of implicit split to \@_ is deprecated" if $^W;
            @_ = @split;
            return scalar @_;
        }
    }

    # split's first argument is more consistently interpreted
    #
    # After some changes earlier in v5.17, split's behavior has been simplified:
    # if the PATTERN argument evaluates to a string containing one space, it is
    # treated the way that a literal string containing one space once was.
    # http://search.cpan.org/dist/perl-5.18.0/pod/perldelta.pod#split's_first_argument_is_more_consistently_interpreted

    # if $pattern is also omitted or is the literal space, " ", the function splits
    # on whitespace, /\s+/, after skipping any leading whitespace
    # (and so on)

    elsif ($pattern eq ' ') {
        if (not defined $limit) {
            return CORE::split(' ', $string);
        }
        else {
            return CORE::split(' ', $string, $limit);
        }
    }

#if LONG_STRING_FOR_RE
    local $q_char = $q_char;
    if (CORE::length($string) > 32766) {
        if ($string =~ /\A [\x00-\x7F]+ \z/oxms) {
            $q_char = qr{.}s;
        }
        elsif (defined ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17}) {
            $q_char = ${Esjis::q_char_SADAHIRO_Tomoyuki_2002_01_17};
        }
    }

#endif
    # if $limit is negative, it is treated as if an arbitrarily large $limit has been specified
    if ((not defined $limit) or ($limit <= 0)) {

        # a pattern capable of matching either the null string or something longer than the
        # null string will split the value of $string into separate characters wherever it
        # matches the null string between characters
        # (and so on)

        if ('' =~ / \A $pattern \z /xms) {
            my $last_subexpression_offsets = _last_subexpression_offsets($pattern);
            my $limit = scalar(() = $string =~ /($pattern)/oxmsg);

            # P.1024 Appendix W.10 Multibyte Processing
            # of ISBN 1-56592-224-7 CJKV Information Processing
            # (and so on)

            # the //m modifier is assumed when you split on the pattern /^/
            # (and so on)

            eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
            #                                                     V
            while ((--$limit > 0) and ($string =~ s/\A((?:$q_char)+?)$pattern//m)) {

                # if the $pattern contains parentheses, then the substring matched by each pair of parentheses
                # is included in the resulting list, interspersed with the fields that are ordinarily returned
                # (and so on)

                local $@;
                eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
                for (my $digit=1; $digit <= ($last_subexpression_offsets + 1); $digit++) {
                    push @split, CORE::eval('$' . $digit);
                }
            }
        }

        else {
            my $last_subexpression_offsets = _last_subexpression_offsets($pattern);

            eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
            #                                 V
            while ($string =~ s/\A((?:$q_char)*?)$pattern//m) {
                local $@;
                eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
                for (my $digit=1; $digit <= ($last_subexpression_offsets + 1); $digit++) {
                    push @split, CORE::eval('$' . $digit);
                }
            }
        }
    }

    elsif ($limit > 0) {
        if ('' =~ / \A $pattern \z /xms) {
            my $last_subexpression_offsets = _last_subexpression_offsets($pattern);
            while ((--$limit > 0) and (CORE::length($string) > 0)) {

                eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
                #                              V
                if ($string =~ s/\A((?:$q_char)+?)$pattern//m) {
                    local $@;
                    for (my $digit=1; $digit <= ($last_subexpression_offsets + 1); $digit++) {
                        push @split, CORE::eval('$' . $digit);
                    }
                }
            }
        }
        else {
            my $last_subexpression_offsets = _last_subexpression_offsets($pattern);
            while ((--$limit > 0) and (CORE::length($string) > 0)) {

                eval q{ no warnings }; # avoid: Complex regular subexpression recursion limit (32766) exceeded at ...
                #                              V
                if ($string =~ s/\A((?:$q_char)*?)$pattern//m) {
                    local $@;
                    for (my $digit=1; $digit <= ($last_subexpression_offsets + 1); $digit++) {
                        push @split, CORE::eval('$' . $digit);
                    }
                }
            }
        }
    }

    if (CORE::length($string) > 0) {
        push @split, $string;
    }

    # if $_[2] (NOT "$limit") is omitted or zero, trailing null fields are stripped from the result
    if ((not defined $_[2]) or ($_[2] == 0)) {
        while ((scalar(@split) >= 1) and ($split[-1] eq '')) {
            pop @split;
        }
    }

    # resulting list value in list context
    if (wantarray) {
        return @split;
    }

    # count of substrings in scalar context
    else {
        carp "Use of implicit split to \@_ is deprecated" if $^W;
        @_ = @split;
        return scalar @_;
    }
}

#
# get last subexpression offsets
#
sub _last_subexpression_offsets {
    my $pattern = $_[0];

    # remove comment
    $pattern =~ s/\(\?\# .*? \)//oxmsg;

    my $modifier = '';
    if ($pattern =~ /\(\?\^? ([\-A-Za-z]+) :/oxms) {
        $modifier = $1;
        $modifier =~ s/-[A-Za-z]*//;
    }

    # with /x modifier
    my @char = ();
    if ($modifier =~ /x/oxms) {
        @char = $pattern =~ /\G((?>
            [^\x81-\x9F\xE0-\xFC\\\#\[\(]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
            \\ $q_char      |
            \# (?>[^\n]*) $ |
            \[ (?>(?:[^\x81-\x9F\xE0-\xFC\\\]]|[\x81-\x9F\xE0-\xFC][\x00-\xFF]|\\\\|\\\]|$q_char)+) \] |
            \(\?            |
                $q_char
        ))/oxmsg;
    }

    # without /x modifier
    else {
        @char = $pattern =~ /\G((?>
            [^\x81-\x9F\xE0-\xFC\\\[\(]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
            \\ $q_char      |
            \[ (?>(?:[^\x81-\x9F\xE0-\xFC\\\]]|[\x81-\x9F\xE0-\xFC][\x00-\xFF]|\\\\|\\\]|$q_char)+) \] |
            \(\?            |
                $q_char
        ))/oxmsg;
    }

    return scalar grep { $_ eq '(' } @char;
}

#
# ShiftJIS transliteration (tr///)
#
sub Esjis::tr($$$$;$) {

    my $bind_operator   = $_[1];
    my $searchlist      = $_[2];
    my $replacementlist = $_[3];
    my $modifier        = $_[4] || '';

    if ($modifier =~ /r/oxms) {
        if ($bind_operator =~ / !~ /oxms) {
            croak "Using !~ with tr///r doesn't make sense";
        }
    }

    my @char            = $_[0] =~ /\G (?>$q_char) /oxmsg;
    my @searchlist      = _charlist_tr($searchlist);
    my @replacementlist = _charlist_tr($replacementlist);

    my %tr = ();
    for (my $i=0; $i <= $#searchlist; $i++) {
        if (not exists $tr{$searchlist[$i]}) {
            if (defined $replacementlist[$i] and ($replacementlist[$i] ne '')) {
                $tr{$searchlist[$i]} = $replacementlist[$i];
            }
            elsif ($modifier =~ /d/oxms) {
                $tr{$searchlist[$i]} = '';
            }
            elsif (defined $replacementlist[-1] and ($replacementlist[-1] ne '')) {
                $tr{$searchlist[$i]} = $replacementlist[-1];
            }
            else {
                $tr{$searchlist[$i]} = $searchlist[$i];
            }
        }
    }

    my $tr = 0;
    my $replaced = '';
    if ($modifier =~ /c/oxms) {
        while (defined(my $char = shift @char)) {
            if (not exists $tr{$char}) {
                if (defined $replacementlist[-1]) {
                    $replaced .= $replacementlist[-1];
                }
                $tr++;
                if ($modifier =~ /s/oxms) {
                    while (@char and (not exists $tr{$char[0]})) {
                        shift @char;
                        $tr++;
                    }
                }
            }
            else {
                $replaced .= $char;
            }
        }
    }
    else {
        while (defined(my $char = shift @char)) {
            if (exists $tr{$char}) {
                $replaced .= $tr{$char};
                $tr++;
                if ($modifier =~ /s/oxms) {
                    while (@char and (exists $tr{$char[0]}) and ($tr{$char[0]} eq $tr{$char})) {
                        shift @char;
                        $tr++;
                    }
                }
            }
            else {
                $replaced .= $char;
            }
        }
    }

    if ($modifier =~ /r/oxms) {
        return $replaced;
    }
    else {
        $_[0] = $replaced;
        if ($bind_operator =~ / !~ /oxms) {
            return not $tr;
        }
        else {
            return $tr;
        }
    }
}

#
# ShiftJIS chop
#
sub Esjis::chop(@) {

    my $chop;
    if (@_ == 0) {
        my @char = /\G (?>$q_char) /oxmsg;
        $chop = pop @char;
        $_ = join '', @char;
    }
    else {
        for (@_) {
            my @char = /\G (?>$q_char) /oxmsg;
            $chop = pop @char;
            $_ = join '', @char;
        }
    }
    return $chop;
}

#
# ShiftJIS index by octet
#
sub Esjis::index($$;$) {

    my($str,$substr,$position) = @_;
    $position ||= 0;
    my $pos = 0;

    while ($pos < CORE::length($str)) {
        if (CORE::substr($str,$pos,CORE::length($substr)) eq $substr) {
            if ($pos >= $position) {
                return $pos;
            }
        }
        if (CORE::substr($str,$pos) =~ /\A ($q_char) /oxms) {
            $pos += CORE::length($1);
        }
        else {
            $pos += 1;
        }
    }
    return -1;
}

#
# ShiftJIS reverse index
#
sub Esjis::rindex($$;$) {

    my($str,$substr,$position) = @_;
    $position ||= CORE::length($str) - 1;
    my $pos = 0;
    my $rindex = -1;

    while (($pos < CORE::length($str)) and ($pos <= $position)) {
        if (CORE::substr($str,$pos,CORE::length($substr)) eq $substr) {
            $rindex = $pos;
        }
        if (CORE::substr($str,$pos) =~ /\A ($q_char) /oxms) {
            $pos += CORE::length($1);
        }
        else {
            $pos += 1;
        }
    }
    return $rindex;
}

#
# ShiftJIS lower case first with parameter
#
sub Esjis::lcfirst(@) {
    if (@_) {
        my $s = shift @_;
        if (@_ and wantarray) {
            return Esjis::lc(CORE::substr($s,0,1)) . CORE::substr($s,1), @_;
        }
        else {
            return Esjis::lc(CORE::substr($s,0,1)) . CORE::substr($s,1);
        }
    }
    else {
        return Esjis::lc(CORE::substr($_,0,1)) . CORE::substr($_,1);
    }
}

#
# ShiftJIS lower case first without parameter
#
sub Esjis::lcfirst_() {
    return Esjis::lc(CORE::substr($_,0,1)) . CORE::substr($_,1);
}

#
# ShiftJIS lower case with parameter
#
sub Esjis::lc(@) {
    if (@_) {
        my $s = shift @_;
        if (@_ and wantarray) {
            return join('', map {defined($lc{$_}) ? $lc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg)), @_;
        }
        else {
            return join('', map {defined($lc{$_}) ? $lc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg));
        }
    }
    else {
        return Esjis::lc_();
    }
}

#
# ShiftJIS lower case without parameter
#
sub Esjis::lc_() {
    my $s = $_;
    return join '', map {defined($lc{$_}) ? $lc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg);
}

#
# ShiftJIS upper case first with parameter
#
sub Esjis::ucfirst(@) {
    if (@_) {
        my $s = shift @_;
        if (@_ and wantarray) {
            return Esjis::uc(CORE::substr($s,0,1)) . CORE::substr($s,1), @_;
        }
        else {
            return Esjis::uc(CORE::substr($s,0,1)) . CORE::substr($s,1);
        }
    }
    else {
        return Esjis::uc(CORE::substr($_,0,1)) . CORE::substr($_,1);
    }
}

#
# ShiftJIS upper case first without parameter
#
sub Esjis::ucfirst_() {
    return Esjis::uc(CORE::substr($_,0,1)) . CORE::substr($_,1);
}

#
# ShiftJIS upper case with parameter
#
sub Esjis::uc(@) {
    if (@_) {
        my $s = shift @_;
        if (@_ and wantarray) {
            return join('', map {defined($uc{$_}) ? $uc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg)), @_;
        }
        else {
            return join('', map {defined($uc{$_}) ? $uc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg));
        }
    }
    else {
        return Esjis::uc_();
    }
}

#
# ShiftJIS upper case without parameter
#
sub Esjis::uc_() {
    my $s = $_;
    return join '', map {defined($uc{$_}) ? $uc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg);
}

#
# ShiftJIS fold case with parameter
#
sub Esjis::fc(@) {
    if (@_) {
        my $s = shift @_;
        if (@_ and wantarray) {
            return join('', map {defined($fc{$_}) ? $fc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg)), @_;
        }
        else {
            return join('', map {defined($fc{$_}) ? $fc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg));
        }
    }
    else {
        return Esjis::fc_();
    }
}

#
# ShiftJIS fold case without parameter
#
sub Esjis::fc_() {
    my $s = $_;
    return join '', map {defined($fc{$_}) ? $fc{$_} : $_} ($s =~ /\G ($q_char) /oxmsg);
}

#
# ShiftJIS regexp capture
#
{
#if MULTIBYTE_ANCHORING
    # 10.3. Creating Persistent Private Variables
    # in Chapter 10. Subroutines
    # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.

    my $last_s_matched = 0;

#endif
    sub Esjis::capture {
#if MULTIBYTE_ANCHORING
        if ($last_s_matched and ($_[0] =~ /\A (?>[1-9][0-9]*) \z/oxms)) {
            return $_[0] + 1;
        }
#endif
        return $_[0];
    }
#if MULTIBYTE_ANCHORING

    # ShiftJIS mark last regexp matched
    sub Esjis::matched() {
        $last_s_matched = 0;
    }

    # ShiftJIS mark last s/// matched
    sub Esjis::s_matched() {
        $last_s_matched = 1;
    }

    # P.854 31.17. use re
    # in Chapter 31. Pragmatic Modules
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.1026 re
    # in Chapter 29. Pragmatic Modules
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    $Esjis::matched = qr/(?{Esjis::matched})/;
#endif
}

#
# ShiftJIS regexp ignore case modifier
#
sub Esjis::ignorecase {

    my @string = @_;
    my $metachar = qr/[\@\\|[\]{]/oxms;

    # ignore case of $scalar or @array
    for my $string (@string) {

        # split regexp
        my @char = $string =~ /\G (?>\[\^|\\$q_char|$q_char) /oxmsg;

        # unescape character
        for (my $i=0; $i <= $#char; $i++) {
            next if not defined $char[$i];

            # open character class [...]
            if ($char[$i] eq '[') {
                my $left = $i;

                # [] make die "unmatched [] in regexp ...\n"

                if ($char[$i+1] eq ']') {
                    $i++;
                }

                while (1) {
                    if (++$i > $#char) {
                        croak "Unmatched [] in regexp";
                    }
                    if ($char[$i] eq ']') {
                        my $right = $i;
                        my @charlist = charlist_qr(@char[$left+1..$right-1], 'i');

                        # escape character
                        for my $char (@charlist) {
                            if (0) {
                            }

#if ESCAPE_SECOND_OCTET
                            # do not use quotemeta here
                            elsif ($char =~ /\A ([\x80-\xFF].*) ($metachar) \z/oxms) {
                                $char = $1 . '\\' . $2;
                            }
#endif
                            elsif ($char =~ /\A [.|)] \z/oxms) {
                                $char = '\\' . $char;
                            }
                        }

                        # [...]
                        splice @char, $left, $right-$left+1, '(?:' . join('|', @charlist) . ')';

                        $i = $left;
                        last;
                    }
                }
            }

            # open character class [^...]
            elsif ($char[$i] eq '[^') {
                my $left = $i;

                # [^] make die "unmatched [] in regexp ...\n"

                if ($char[$i+1] eq ']') {
                    $i++;
                }

                while (1) {
                    if (++$i > $#char) {
                        croak "Unmatched [] in regexp";
                    }
                    if ($char[$i] eq ']') {
                        my $right = $i;
                        my @charlist = charlist_not_qr(@char[$left+1..$right-1], 'i');

                        # escape character
                        for my $char (@charlist) {
                            if (0) {
                            }

#if ESCAPE_SECOND_OCTET
                            # do not use quotemeta here
                            elsif ($char =~ /\A ([\x80-\xFF].*) ($metachar) \z/oxms) {
                                $char = $1 . '\\' . $2;
                            }
#endif
                            elsif ($char =~ /\A [.|)] \z/oxms) {
                                $char = '\\' . $char;
                            }
                        }

                        # [^...]
                        splice @char, $left, $right-$left+1, '(?!' . join('|', @charlist) . ")(?:$your_char)";

                        $i = $left;
                        last;
                    }
                }
            }

            # rewrite classic character class or escape character
            elsif (my $char = classic_character_class($char[$i])) {
                $char[$i] = $char;
            }

            # with /i modifier
            elsif ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) {
                my $uc = Esjis::uc($char[$i]);
                my $fc = Esjis::fc($char[$i]);
                if ($uc ne $fc) {
                    if (CORE::length($fc) == 1) {
                        $char[$i] = '['   . $uc       . $fc . ']';
                    }
                    else {
                        $char[$i] = '(?:' . $uc . '|' . $fc . ')';
                    }
                }
            }
        }

        # characterize
        for (my $i=0; $i <= $#char; $i++) {
            next if not defined $char[$i];

            if (0) {
            }

#if ESCAPE_SECOND_OCTET
            # escape last octet of multiple-octet
            elsif ($char[$i] =~ /\A ([\x80-\xFF].*) ($metachar) \z/oxms) {
                $char[$i] = $1 . '\\' . $2;
            }

#endif
            # quote character before ? + * {
            elsif (($i >= 1) and ($char[$i] =~ /\A [\?\+\*\{] \z/oxms)) {
                if ($char[$i-1] !~ /\A [\x00-\xFF] \z/oxms) {
                    $char[$i-1] = '(?:' . $char[$i-1] . ')';
                }
            }
        }

        $string = join '', @char;
    }

    # make regexp string
    return @string;
}

#
# classic character class ( \D \S \W \d \s \w \C \X \H \V \h \v \R \N \b \B )
#
sub Esjis::classic_character_class {
    my($char) = @_;

    return {
        '\D' => '${Esjis::eD}',
        '\S' => '${Esjis::eS}',
        '\W' => '${Esjis::eW}',
        '\d' => '[0-9]',

        # Before Perl 5.6, \s only matched the five whitespace characters
        # tab, newline, form-feed, carriage return, and the space character
        # itself, which, taken together, is the character class [\t\n\f\r ].

        # Vertical tabs are now whitespace
        # \s in a regex now matches a vertical tab in all circumstances.
        # http://search.cpan.org/dist/perl-5.18.0/pod/perldelta.pod#Vertical_tabs_are_now_whitespace
        #            \t  \n  \v  \f  \r space
        # '\s' => '[\x09\x0A    \x0C\x0D\x20]',
        # '\s' => '[\x09\x0A\x0B\x0C\x0D\x20]',
        '\s'   => '\s',

        '\w' => '[0-9A-Z_a-z]',
        '\C' => '[\x00-\xFF]',
        '\X' => 'X',

        # \h \v \H \V

        # P.114 Character Class Shortcuts
        # in Chapter 7: In the World of Regular Expressions
        # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

        # P.357 13.2.3 Whitespace
        # in Chapter 13: perlrecharclass: Perl Regular Expression Character Classes
        # of ISBN-13: 978-1-906966-02-7 The Perl Language Reference Manual (for Perl version 5.12.1)
        #
        # 0x00009   CHARACTER TABULATION  h s
        # 0x0000a         LINE FEED (LF)   vs
        # 0x0000b        LINE TABULATION   v
        # 0x0000c         FORM FEED (FF)   vs
        # 0x0000d   CARRIAGE RETURN (CR)   vs
        # 0x00020                  SPACE  h s

        # P.196 Table 5-9. Alphanumeric regex metasymbols
        # in Chapter 5. Pattern Matching
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # (and so on)

        '\H' => '${Esjis::eH}',
        '\V' => '${Esjis::eV}',
        '\h' => '[\x09\x20]',
        '\v' => '[\x0A\x0B\x0C\x0D]',
        '\R' => '${Esjis::eR}',

        # \N
        #
        # http://perldoc.perl.org/perlre.html
        # Character Classes and other Special Escapes
        # Any character but \n (experimental). Not affected by /s modifier

        '\N' => '${Esjis::eN}',

        # \b \B

        # P.180 Boundaries: The \b and \B Assertions
        # in Chapter 5: Pattern Matching
        # of ISBN 0-596-00027-8 Programming Perl Third Edition.

        # P.219 Boundaries: The \b and \B Assertions
        # in Chapter 5: Pattern Matching
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # \b really means (?:(?<=\w)(?!\w)|(?<!\w)(?=\w))
        #           or (?:(?<=\A|\W)(?=\w)|(?<=\w)(?=\W|\z))
        '\b' => '${Esjis::eb}',

        # \B really means (?:(?<=\w)(?=\w)|(?<!\w)(?!\w))
        #              or (?:(?<=\w)(?=\w)|(?<=\W)(?=\W))
        '\B' => '${Esjis::eB}',

    }->{$char} || '';
}

#
# prepare ShiftJIS characters per length
#

# 1 octet characters
my @chars1 = ();
sub chars1 {
    if (@chars1) {
        return @chars1;
    }
    if (exists $range_tr{1}) {
        my @ranges = @{ $range_tr{1} };
        while (my @range = splice(@ranges,0,1)) {
            for my $oct0 (@{$range[0]}) {
                push @chars1, pack 'C', $oct0;
            }
        }
    }
    return @chars1;
}

# 2 octets characters
my @chars2 = ();
sub chars2 {
    if (@chars2) {
        return @chars2;
    }
    if (exists $range_tr{2}) {
        my @ranges = @{ $range_tr{2} };
        while (my @range = splice(@ranges,0,2)) {
            for my $oct0 (@{$range[0]}) {
                for my $oct1 (@{$range[1]}) {
                    push @chars2, pack 'CC', $oct0,$oct1;
                }
            }
        }
    }
    return @chars2;
}

# 3 octets characters
my @chars3 = ();
sub chars3 {
    if (@chars3) {
        return @chars3;
    }
    if (exists $range_tr{3}) {
        my @ranges = @{ $range_tr{3} };
        while (my @range = splice(@ranges,0,3)) {
            for my $oct0 (@{$range[0]}) {
                for my $oct1 (@{$range[1]}) {
                    for my $oct2 (@{$range[2]}) {
                        push @chars3, pack 'CCC', $oct0,$oct1,$oct2;
                    }
                }
            }
        }
    }
    return @chars3;
}

# 4 octets characters
my @chars4 = ();
sub chars4 {
    if (@chars4) {
        return @chars4;
    }
    if (exists $range_tr{4}) {
        my @ranges = @{ $range_tr{4} };
        while (my @range = splice(@ranges,0,4)) {
            for my $oct0 (@{$range[0]}) {
                for my $oct1 (@{$range[1]}) {
                    for my $oct2 (@{$range[2]}) {
                        for my $oct3 (@{$range[3]}) {
                            push @chars4, pack 'CCCC', $oct0,$oct1,$oct2,$oct3;
                        }
                    }
                }
            }
        }
    }
    return @chars4;
}

#
# ShiftJIS open character list for tr
#
sub _charlist_tr {

    local $_ = shift @_;

    # unescape character
    my @char = ();
    while (not /\G \z/oxmsgc) {
        if (/\G (\\0?55|\\x2[Dd]|\\-) /oxmsgc) {
            push @char, '\-';
        }
        elsif (/\G \\ ([0-7]{2,3}) /oxmsgc) {
            push @char, CORE::chr(oct $1);
        }
        elsif (/\G \\x ([0-9A-Fa-f]{1,2}) /oxmsgc) {
            push @char, CORE::chr(hex $1);
        }
        elsif (/\G \\c ([\x40-\x5F]) /oxmsgc) {
            push @char, CORE::chr(CORE::ord($1) & 0x1F);
        }
        elsif (/\G (\\ [0nrtfbae]) /oxmsgc) {
            push @char, {
                '\0' => "\0",
                '\n' => "\n",
                '\r' => "\r",
                '\t' => "\t",
                '\f' => "\f",
                '\b' => "\x08", # \b means backspace in character class
                '\a' => "\a",
                '\e' => "\e",
            }->{$1};
        }
        elsif (/\G \\ ($q_char) /oxmsgc) {
            push @char, $1;
        }
        elsif (/\G ($q_char) /oxmsgc) {
            push @char, $1;
        }
    }

    # join separated multiple-octet
    @char = join('',@char) =~ /\G (?>\\-|$q_char) /oxmsg;

    # unescape '-'
    my @i = ();
    for my $i (0 .. $#char) {
        if ($char[$i] eq '\-') {
            $char[$i] = '-';
        }
        elsif ($char[$i] eq '-') {
            if ((0 < $i) and ($i < $#char)) {
                push @i, $i;
            }
        }
    }

    # open character list (reverse for splice)
    for my $i (CORE::reverse @i) {
        my @range = ();

        # range error
        if ((CORE::length($char[$i-1]) > CORE::length($char[$i+1])) or ($char[$i-1] gt $char[$i+1])) {
            croak "Invalid tr/// range \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
        }

        # range of multiple-octet code
        if (CORE::length($char[$i-1]) == 1) {
            if (CORE::length($char[$i+1]) == 1) {
                push @range, grep {($char[$i-1] le $_) and ($_ le $char[$i+1])} chars1();
            }
            elsif (CORE::length($char[$i+1]) == 2) {
                push @range, grep {$char[$i-1] le $_}                           chars1();
                push @range, grep {$_ le $char[$i+1]}                           chars2();
            }
            elsif (CORE::length($char[$i+1]) == 3) {
                push @range, grep {$char[$i-1] le $_}                           chars1();
                push @range,                                                    chars2();
                push @range, grep {$_ le $char[$i+1]}                           chars3();
            }
            elsif (CORE::length($char[$i+1]) == 4) {
                push @range, grep {$char[$i-1] le $_}                           chars1();
                push @range,                                                    chars2();
                push @range,                                                    chars3();
                push @range, grep {$_ le $char[$i+1]}                           chars4();
            }
            else {
                croak "Invalid tr/// range (over 4octets) \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
            }
        }
        elsif (CORE::length($char[$i-1]) == 2) {
            if (CORE::length($char[$i+1]) == 2) {
                push @range, grep {($char[$i-1] le $_) and ($_ le $char[$i+1])} chars2();
            }
            elsif (CORE::length($char[$i+1]) == 3) {
                push @range, grep {$char[$i-1] le $_}                           chars2();
                push @range, grep {$_ le $char[$i+1]}                           chars3();
            }
            elsif (CORE::length($char[$i+1]) == 4) {
                push @range, grep {$char[$i-1] le $_}                           chars2();
                push @range,                                                    chars3();
                push @range, grep {$_ le $char[$i+1]}                           chars4();
            }
            else {
                croak "Invalid tr/// range (over 4octets) \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
            }
        }
        elsif (CORE::length($char[$i-1]) == 3) {
            if (CORE::length($char[$i+1]) == 3) {
                push @range, grep {($char[$i-1] le $_) and ($_ le $char[$i+1])} chars3();
            }
            elsif (CORE::length($char[$i+1]) == 4) {
                push @range, grep {$char[$i-1] le $_}                           chars3();
                push @range, grep {$_ le $char[$i+1]}                           chars4();
            }
            else {
                croak "Invalid tr/// range (over 4octets) \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
            }
        }
        elsif (CORE::length($char[$i-1]) == 4) {
            if (CORE::length($char[$i+1]) == 4) {
                push @range, grep {($char[$i-1] le $_) and ($_ le $char[$i+1])} chars4();
            }
            else {
                croak "Invalid tr/// range (over 4octets) \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
            }
        }
        else {
            croak "Invalid tr/// range (over 4octets) \"\\x" . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]) . '"';
        }

        splice @char, $i-1, 3, @range;
    }

    return @char;
}

#
# ShiftJIS open character class
#
sub _cc {
    if (scalar(@_) == 0) {
        die __FILE__, ": subroutine cc got no parameter.\n";
    }
    elsif (scalar(@_) == 1) {
        return sprintf('\x%02X',$_[0]);
    }
    elsif (scalar(@_) == 2) {
        if ($_[0] > $_[1]) {
            die __FILE__, ": subroutine cc got \$_[0] > \$_[1] parameters).\n";
        }
        elsif ($_[0] == $_[1]) {
            return sprintf('\x%02X',$_[0]);
        }
        elsif (($_[0]+1) == $_[1]) {
            return sprintf('[\\x%02X\\x%02X]',$_[0],$_[1]);
        }
        else {
            return sprintf('[\\x%02X-\\x%02X]',$_[0],$_[1]);
        }
    }
    else {
        die __FILE__, ": subroutine cc got 3 or more parameters (@{[scalar(@_)]} parameters).\n";
    }
}

#
# ShiftJIS octet range
#
sub _octets {
    my $length = shift @_;

    if ($length == 1) {
        my($a1) = unpack 'C', $_[0];
        my($z1) = unpack 'C', $_[1];

        if ($a1 > $z1) {
            croak 'Invalid [] range in regexp (CORE::ord(A) > CORE::ord(B)) ' . '\x' . unpack('H*',$a1) . '-\x' . unpack('H*',$z1);
        }

        if ($a1 == $z1) {
            return sprintf('\x%02X',$a1);
        }
        elsif (($a1+1) == $z1) {
            return sprintf('\x%02X\x%02X',$a1,$z1);
        }
        else {
            return sprintf('\x%02X-\x%02X',$a1,$z1);
        }
    }
#if MULTIBYTE_ENCODING
    elsif ($length == 2) {
        my($a1,$a2) = unpack 'CC', $_[0];
        my($z1,$z2) = unpack 'CC', $_[1];
        my($A1,$A2) = unpack 'CC', $_[2];
        my($Z1,$Z2) = unpack 'CC', $_[3];

        if ($a1 == $z1) {
            return (
            #   11111111   222222222222
            #        A          A   Z
                _cc($a1) . _cc($a2,$z2), # a2-z2
            );
        }
        elsif (($a1+1) == $z1) {
            return (
            #   11111111111   222222222222
            #        A  Z          A   Z
                _cc($a1)    . _cc($a2,$Z2), # a2-
                _cc(   $z1) . _cc($A2,$z2), #   -z2
            );
        }
        else {
            return (
            #   1111111111111111   222222222222
            #        A     Z            A   Z
                _cc($a1)         . _cc($a2,$Z2), # a2-
                _cc($a1+1,$z1-1) . _cc($A2,$Z2), #   -
                _cc(      $z1)   . _cc($A2,$z2), #   -z2
            );
        }
    }
    elsif ($length == 3) {
        my($a1,$a2,$a3) = unpack 'CCC', $_[0];
        my($z1,$z2,$z3) = unpack 'CCC', $_[1];
        my($A1,$A2,$A3) = unpack 'CCC', $_[2];
        my($Z1,$Z2,$Z3) = unpack 'CCC', $_[3];

        if ($a1 == $z1) {
            if ($a2 == $z2) {
                return (
                #   11111111   22222222   333333333333
                #        A          A          A   Z
                    _cc($a1) . _cc($a2) . _cc($a3,$z3), # a3-z3
                );
            }
            elsif (($a2+1) == $z2) {
                return (
                #   11111111   22222222222   333333333333
                #        A          A  Z          A   Z
                    _cc($a1) . _cc($a2)    . _cc($a3,$Z3), # a3-
                    _cc($a1) . _cc(   $z2) . _cc($A3,$z3), #   -z3
                );
            }
            else {
                return (
                #   11111111   2222222222222222   333333333333
                #        A          A     Z            A   Z
                    _cc($a1) . _cc($a2)         . _cc($a3,$Z3), # a3-
                    _cc($a1) . _cc($a2+1,$z2-1) . _cc($A3,$Z3), #   -
                    _cc($a1) . _cc(      $z2)   . _cc($A3,$z3), #   -z3
                );
            }
        }
        elsif (($a1+1) == $z1) {
            return (
            #   11111111111   22222222222222   333333333333
            #        A  Z          A     Z          A   Z
                _cc($a1)    . _cc($a2)       . _cc($a3,$Z3), # a3-
                _cc($a1)    . _cc($a2+1,$Z2) . _cc($A3,$Z3), #   -
                _cc(   $z1) . _cc($A2,$z2-1) . _cc($A3,$Z3), #   -
                _cc(   $z1) . _cc(    $z2)   . _cc($A3,$z3), #   -z3
            );
        }
        else {
            return (
            #   1111111111111111   22222222222222   333333333333
            #        A     Z            A     Z          A   Z
                _cc($a1)         . _cc($a2)       . _cc($a3,$Z3), # a3-
                _cc($a1)         . _cc($a2+1,$Z2) . _cc($A3,$Z3), #   -
                _cc($a1+1,$z1-1) . _cc($A2,$Z2)   . _cc($A3,$Z3), #   -
                _cc(      $z1)   . _cc($A2,$z2-1) . _cc($A3,$Z3), #   -
                _cc(      $z1)   . _cc(    $z2)   . _cc($A3,$z3), #   -z3
            );
        }
    }
    elsif ($length == 4) {
        my($a1,$a2,$a3,$a4) = unpack 'CCCC', $_[0];
        my($z1,$z2,$z3,$z4) = unpack 'CCCC', $_[1];
        my($A1,$A2,$A3,$A4) = unpack 'CCCC', $_[0];
        my($Z1,$Z2,$Z3,$Z4) = unpack 'CCCC', $_[1];

        if ($a1 == $z1) {
            if ($a2 == $z2) {
                if ($a3 == $z3) {
                    return (
                    #   11111111   22222222   33333333   444444444444
                    #        A          A          A          A   Z
                        _cc($a1) . _cc($a2) . _cc($a3) . _cc($a4,$z4), # a4-z4
                    );
                }
                elsif (($a3+1) == $z3) {
                    return (
                    #   11111111   22222222   33333333333   444444444444
                    #        A          A          A  Z          A   Z
                        _cc($a1) . _cc($a2) . _cc($a3)    . _cc($a4,$Z4), # a4-
                        _cc($a1) . _cc($a2) . _cc(   $z3) . _cc($A4,$z4), #   -z4
                    );
                }
                else {
                    return (
                    #   11111111   22222222   3333333333333333   444444444444
                    #        A          A          A     Z            A   Z
                        _cc($a1) . _cc($a2) . _cc($a3)         . _cc($a4,$Z4), # a4-
                        _cc($a1) . _cc($a2) . _cc($a3+1,$z3-1) . _cc($A4,$Z4), #   -
                        _cc($a1) . _cc($a2) . _cc(      $z3)   . _cc($A4,$z4), #   -z4
                    );
                }
            }
            elsif (($a2+1) == $z2) {
                return (
                #   11111111   22222222222   33333333333333   444444444444
                #        A          A  Z          A     Z          A   Z
                    _cc($a1) . _cc($a2)    . _cc($a3)       . _cc($a4,$Z4), # a4-
                    _cc($a1) . _cc($a2)    . _cc($a3+1,$Z3) . _cc($A4,$Z4), #   -
                    _cc($a1) . _cc(   $z2) . _cc($A3,$z3-1) . _cc($A4,$Z4), #   -
                    _cc($a1) . _cc(   $z2) . _cc(    $z3)   . _cc($A4,$z4), #   -z4
                );
            }
            else {
                return (
                #   11111111   2222222222222222   33333333333333   444444444444
                #        A          A     Z            A     Z          A   Z
                    _cc($a1) . _cc($a2)         . _cc($a3)       . _cc($a4,$Z4), # a4-
                    _cc($a1) . _cc($a2)         . _cc($a3+1,$Z3) . _cc($A4,$Z4), #   -
                    _cc($a1) . _cc($a2+1,$z2-1) . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                    _cc($a1) . _cc(      $z2)   . _cc($A3,$z3-1) . _cc($A4,$Z4), #   -
                    _cc($a1) . _cc(      $z2)   . _cc(    $z3)   . _cc($A4,$z4), #   -z4
                );
            }
        }
        elsif (($a1+1) == $z1) {
            return (
            #   11111111111   22222222222222   33333333333333   444444444444
            #        A  Z          A     Z          A     Z          A   Z
                _cc($a1)    . _cc($a2)       . _cc($a3)       . _cc($a4,$Z4), # a4-
                _cc($a1)    . _cc($a2)       . _cc($a3+1,$Z3) . _cc($A4,$Z4), #   -
                _cc($a1)    . _cc($a2+1,$Z2) . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                _cc(   $z1) . _cc($A2,$z2-1) . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                _cc(   $z1) . _cc(    $z2)   . _cc($A3,$z3-1) . _cc($A4,$Z4), #   -
                _cc(   $z1) . _cc(    $z2)   . _cc(    $z3)   . _cc($A4,$z4), #   -z4
            );
        }
        else {
            return (
            #   1111111111111111   22222222222222   33333333333333   444444444444
            #        A     Z            A     Z          A     Z          A   Z
                _cc($a1)         . _cc($a2)       . _cc($a3)       . _cc($a4,$Z4), # a4-
                _cc($a1)         . _cc($a2)       . _cc($a3+1,$Z3) . _cc($A4,$Z4), #   -
                _cc($a1)         . _cc($a2+1,$Z2) . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                _cc($a1+1,$z1-1) . _cc($A2,$Z2)   . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                _cc(      $z1)   . _cc($A2,$z2-1) . _cc($A3,$Z3)   . _cc($A4,$Z4), #   -
                _cc(      $z1)   . _cc(    $z2)   . _cc($A3,$z3-1) . _cc($A4,$Z4), #   -
                _cc(      $z1)   . _cc(    $z2)   . _cc(    $z3)   . _cc($A4,$z4), #   -z4
            );
        }
    }
#endif
    else {
        die __FILE__, ": subroutine _octets got invalid length ($length).\n";
    }
}

#
# ShiftJIS range regexp
#
sub _range_regexp {
    my($length,$first,$last) = @_;

    my @range_regexp = ();
    if (not exists $range_tr{$length}) {
        return @range_regexp;
    }

    my @ranges = @{ $range_tr{$length} };
    while (my @range = splice(@ranges,0,$length)) {
        my $min = '';
        my $max = '';
        for (my $i=0; $i < $length; $i++) {
            $min .= pack 'C', $range[$i][0];
            $max .= pack 'C', $range[$i][-1];
        }

# min___max
#            FIRST_____________LAST
#       (nothing)

        if ($max lt $first) {
        }

#            **********
#       min_________max
#            FIRST_____________LAST
#            **********

        elsif (($min le $first) and ($first le $max) and ($max le $last)) {
            push @range_regexp, _octets($length,$first,$max,$min,$max);
        }

#            **********************
#            min________________max
#            FIRST_____________LAST
#            **********************

        elsif (($min eq $first) and ($max eq $last)) {
            push @range_regexp, _octets($length,$first,$last,$min,$max);
        }

#                   *********
#                   min___max
#            FIRST_____________LAST
#                   *********

        elsif (($first le $min) and ($max le $last)) {
            push @range_regexp, _octets($length,$min,$max,$min,$max);
        }

#            **********************
#       min__________________________max
#            FIRST_____________LAST
#            **********************

        elsif (($min le $first) and ($last le $max)) {
            push @range_regexp, _octets($length,$first,$last,$min,$max);
        }

#                         *********
#                         min________max
#            FIRST_____________LAST
#                         *********

        elsif (($first le $min) and ($min le $last) and ($last le $max)) {
            push @range_regexp, _octets($length,$min,$last,$min,$max);
        }

#                                    min___max
#            FIRST_____________LAST
#                              (nothing)

        elsif ($last lt $min) {
        }

        else {
            die __FILE__, ": subroutine _range_regexp panic.\n";
        }
    }

    return @range_regexp;
}

#
# ShiftJIS open character list for qr and not qr
#
sub _charlist {

    my $modifier = pop @_;
    my @char = @_;

    my $ignorecase = ($modifier =~ /i/oxms) ? 1 : 0;

    # unescape character
    for (my $i=0; $i <= $#char; $i++) {

        # escape - to ...
        if ($char[$i] eq '-') {
            if ((0 < $i) and ($i < $#char)) {
                $char[$i] = '...';
            }
        }

        # octal escape sequence
        elsif ($char[$i] =~ /\A \\o \{ ([0-7]+) \} \z/oxms) {
            $char[$i] = octchr($1);
        }

        # hexadecimal escape sequence
        elsif ($char[$i] =~ /\A \\x \{ ([0-9A-Fa-f]+) \} \z/oxms) {
            $char[$i] = hexchr($1);
        }

        # \b{...}      --> b\{...}
        # \B{...}      --> B\{...}
        # \N{CHARNAME} --> N\{CHARNAME}
        # \p{PROPERTY} --> p\{PROPERTY}
        # \P{PROPERTY} --> P\{PROPERTY}
        elsif ($char[$i] =~ /\A \\ ([bBNpP]) ( \{ ([^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} ) \z/oxms) {
            $char[$i] = $1 . '\\' . $2;
        }

        # \p, \P, \X --> p, P, X
        elsif ($char[$i] =~ /\A \\ ( [pPX] ) \z/oxms) {
            $char[$i] = $1;
        }

        elsif ($char[$i] =~ /\A \\ ([0-7]{2,3}) \z/oxms) {
            $char[$i] = CORE::chr oct $1;
        }
        elsif ($char[$i] =~ /\A \\x ([0-9A-Fa-f]{1,2}) \z/oxms) {
            $char[$i] = CORE::chr hex $1;
        }
        elsif ($char[$i] =~ /\A \\c ([\x40-\x5F]) \z/oxms) {
            $char[$i] = CORE::chr(CORE::ord($1) & 0x1F);
        }
        elsif ($char[$i] =~ /\A (\\ [0nrtfbaedswDSWHVhvR]) \z/oxms) {
            $char[$i] = {
                '\0' => "\0",
                '\n' => "\n",
                '\r' => "\r",
                '\t' => "\t",
                '\f' => "\f",
                '\b' => "\x08", # \b means backspace in character class
                '\a' => "\a",
                '\e' => "\e",
                '\d' => '[0-9]',

                # Vertical tabs are now whitespace
                # \s in a regex now matches a vertical tab in all circumstances.
                # http://search.cpan.org/dist/perl-5.18.0/pod/perldelta.pod#Vertical_tabs_are_now_whitespace
                #            \t  \n  \v  \f  \r space
                # '\s' => '[\x09\x0A    \x0C\x0D\x20]',
                # '\s' => '[\x09\x0A\x0B\x0C\x0D\x20]',
                '\s'   => '\s',

                '\w' => '[0-9A-Z_a-z]',
                '\D' => '${Esjis::eD}',
                '\S' => '${Esjis::eS}',
                '\W' => '${Esjis::eW}',

                '\H' => '${Esjis::eH}',
                '\V' => '${Esjis::eV}',
                '\h' => '[\x09\x20]',
                '\v' => '[\x0A\x0B\x0C\x0D]',
                '\R' => '${Esjis::eR}',

            }->{$1};
        }

        # POSIX-style character classes
        elsif ($ignorecase and ($char[$i] =~ /\A ( \[\: \^? (?:lower|upper) :\] ) \z/oxms)) {
            $char[$i] = {

                '[:lower:]'   => '[\x41-\x5A\x61-\x7A]',
                '[:upper:]'   => '[\x41-\x5A\x61-\x7A]',
                '[:^lower:]'  => '${Esjis::not_lower_i}',
                '[:^upper:]'  => '${Esjis::not_upper_i}',

            }->{$1};
        }
        elsif ($char[$i] =~ /\A ( \[\: \^? (?:alnum|alpha|ascii|blank|cntrl|digit|graph|lower|print|punct|space|upper|word|xdigit) :\] ) \z/oxms) {
            $char[$i] = {

                '[:alnum:]'   => '[\x30-\x39\x41-\x5A\x61-\x7A]',
                '[:alpha:]'   => '[\x41-\x5A\x61-\x7A]',
                '[:ascii:]'   => '[\x00-\x7F]',
                '[:blank:]'   => '[\x09\x20]',
                '[:cntrl:]'   => '[\x00-\x1F\x7F]',
                '[:digit:]'   => '[\x30-\x39]',
                '[:graph:]'   => '[\x21-\x7F]',
                '[:lower:]'   => '[\x61-\x7A]',
                '[:print:]'   => '[\x20-\x7F]',
                '[:punct:]'   => '[\x21-\x2F\x3A-\x3F\x40\x5B-\x5F\x60\x7B-\x7E]',

                # P.174 POSIX-Style Character Classes
                # in Chapter 5: Pattern Matching
                # of ISBN 0-596-00027-8 Programming Perl Third Edition.

                # P.311 11.2.4 Character Classes and other Special Escapes
                # in Chapter 11: perlre: Perl regular expressions
                # of ISBN-13: 978-1-906966-02-7 The Perl Language Reference Manual (for Perl version 5.12.1)

                # P.210 POSIX-Style Character Classes
                # in Chapter 5: Pattern Matching
                # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

                '[:space:]'   => '[\s\x0B]', # "\s" plus vertical tab ("\cK")

                '[:upper:]'   => '[\x41-\x5A]',
                '[:word:]'    => '[\x30-\x39\x41-\x5A\x5F\x61-\x7A]',
                '[:xdigit:]'  => '[\x30-\x39\x41-\x46\x61-\x66]',
                '[:^alnum:]'  => '${Esjis::not_alnum}',
                '[:^alpha:]'  => '${Esjis::not_alpha}',
                '[:^ascii:]'  => '${Esjis::not_ascii}',
                '[:^blank:]'  => '${Esjis::not_blank}',
                '[:^cntrl:]'  => '${Esjis::not_cntrl}',
                '[:^digit:]'  => '${Esjis::not_digit}',
                '[:^graph:]'  => '${Esjis::not_graph}',
                '[:^lower:]'  => '${Esjis::not_lower}',
                '[:^print:]'  => '${Esjis::not_print}',
                '[:^punct:]'  => '${Esjis::not_punct}',
                '[:^space:]'  => '${Esjis::not_space}',
                '[:^upper:]'  => '${Esjis::not_upper}',
                '[:^word:]'   => '${Esjis::not_word}',
                '[:^xdigit:]' => '${Esjis::not_xdigit}',

            }->{$1};
        }
        elsif ($char[$i] =~ /\A \\ ($q_char) \z/oxms) {
            $char[$i] = $1;
        }
    }

    # open character list
    my @singleoctet   = ();
    my @multipleoctet = ();
    for (my $i=0; $i <= $#char; ) {

        # escaped -
        if (defined($char[$i+1]) and ($char[$i+1] eq '...')) {
            $i += 1;
            next;
        }

        # make range regexp
        elsif ($char[$i] eq '...') {

            # range error
            if (CORE::length($char[$i-1]) > CORE::length($char[$i+1])) {
                croak 'Invalid [] range in regexp (length(A) > length(B)) ' . '\x' . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]);
            }
            elsif (CORE::length($char[$i-1]) == CORE::length($char[$i+1])) {
                if ($char[$i-1] gt $char[$i+1]) {
                    croak 'Invalid [] range in regexp (CORE::ord(A) > CORE::ord(B)) ' . '\x' . unpack('H*',$char[$i-1]) . '-\x' . unpack('H*',$char[$i+1]);
                }
            }

            # make range regexp per length
            for my $length (CORE::length($char[$i-1]) .. CORE::length($char[$i+1])) {
                my @regexp = ();

                # is first and last
                if (($length == CORE::length($char[$i-1])) and ($length == CORE::length($char[$i+1]))) {
                    push @regexp, _range_regexp($length, $char[$i-1], $char[$i+1]);
                }

                # is first
                elsif ($length == CORE::length($char[$i-1])) {
                    push @regexp, _range_regexp($length, $char[$i-1], "\xFF" x $length);
                }

                # is inside in first and last
                elsif ((CORE::length($char[$i-1]) < $length) and ($length < CORE::length($char[$i+1]))) {
                    push @regexp, _range_regexp($length, "\x00" x $length, "\xFF" x $length);
                }

                # is last
                elsif ($length == CORE::length($char[$i+1])) {
                    push @regexp, _range_regexp($length, "\x00" x $length, $char[$i+1]);
                }

                else {
                    die __FILE__, ": subroutine make_regexp panic.\n";
                }

                if ($length == 1) {
                    push @singleoctet, @regexp;
                }
                else {
                    push @multipleoctet, @regexp;
                }
            }

            $i += 2;
        }

        # with /i modifier
        elsif ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) {
            if ($modifier =~ /i/oxms) {
                my $uc = Esjis::uc($char[$i]);
                my $fc = Esjis::fc($char[$i]);
                if ($uc ne $fc) {
                    if (CORE::length($fc) == 1) {
                        push @singleoctet, $uc, $fc;
                    }
                    else {
                        push @singleoctet,   $uc;
                        push @multipleoctet, $fc;
                    }
                }
                else {
                    push @singleoctet, $char[$i];
                }
            }
            else {
                push @singleoctet, $char[$i];
            }
            $i += 1;
        }

        # single character of single octet code
        elsif ($char[$i] =~ /\A (?: \\h ) \z/oxms) {
            push @singleoctet, "\t", "\x20";
            $i += 1;
        }
        elsif ($char[$i] =~ /\A (?: \\v ) \z/oxms) {
            push @singleoctet, "\x0A", "\x0B", "\x0C", "\x0D";
            $i += 1;
        }
        elsif ($char[$i] =~ /\A (?: \\d | \\s | \\w ) \z/oxms) {
            push @singleoctet, $char[$i];
            $i += 1;
        }

        # single character of multiple-octet code
        else {
            push @multipleoctet, $char[$i];
            $i += 1;
        }
    }

    # quote metachar
    for (@singleoctet) {
        if ($_ eq '...') {
            $_ = '-';
        }
        elsif (/\A \n \z/oxms) {
            $_ = '\n';
        }
        elsif (/\A \r \z/oxms) {
            $_ = '\r';
        }
        elsif (/\A ([\x00-\x20\x7F-\xFF]) \z/oxms) {
            $_ = sprintf('\x%02X', CORE::ord $1);
        }
        elsif (/\A [\x00-\xFF] \z/oxms) {
            $_ = quotemeta $_;
        }
    }
#if ESCAPE_SECOND_OCTET
    for (@multipleoctet) {
        if (/\A ([\x80-\xFF].*) ([\x00-\xFF]) \z/oxms) {
            $_ = $1 . quotemeta $2;
        }
    }
#endif

    # return character list
    return \@singleoctet, \@multipleoctet;
}

#
# ShiftJIS octal escape sequence
#
sub octchr {
    my($octdigit) = @_;

    my @binary = ();
    for my $octal (split(//,$octdigit)) {
        push @binary, {
            '0' => '000',
            '1' => '001',
            '2' => '010',
            '3' => '011',
            '4' => '100',
            '5' => '101',
            '6' => '110',
            '7' => '111',
        }->{$octal};
    }
    my $binary = join '', @binary;

    my $octchr = {
        #                1234567
        1 => pack('B*', "0000000$binary"),
        2 => pack('B*', "000000$binary"),
        3 => pack('B*', "00000$binary"),
        4 => pack('B*', "0000$binary"),
        5 => pack('B*', "000$binary"),
        6 => pack('B*', "00$binary"),
        7 => pack('B*', "0$binary"),
        0 => pack('B*', "$binary"),

    }->{CORE::length($binary) % 8};

    return $octchr;
}

#
# ShiftJIS hexadecimal escape sequence
#
sub hexchr {
    my($hexdigit) = @_;

    my $hexchr = {
        1 => pack('H*', "0$hexdigit"),
        0 => pack('H*', "$hexdigit"),

    }->{CORE::length($_[0]) % 2};

    return $hexchr;
}

#
# ShiftJIS open character list for qr
#
sub charlist_qr {

    my $modifier = pop @_;
    my @char = @_;

    my($singleoctet, $multipleoctet) = _charlist(@char, $modifier);
    my @singleoctet   = @$singleoctet;
    my @multipleoctet = @$multipleoctet;

    # return character list
    if (scalar(@singleoctet) >= 1) {

        # with /i modifier
        if ($modifier =~ m/i/oxms) {
            my %singleoctet_ignorecase = ();
            for (@singleoctet) {
                while (s/ \A \\x(..) - \\x(..) //oxms or s/ \A \\x((..)) //oxms) {
                    for my $ord (hex($1) .. hex($2)) {
                        my $char = CORE::chr($ord);
                        my $uc = Esjis::uc($char);
                        my $fc = Esjis::fc($char);
                        if ($uc eq $fc) {
                            $singleoctet_ignorecase{unpack 'C*', $char} = 1;
                        }
                        else {
                            if (CORE::length($fc) == 1) {
                                $singleoctet_ignorecase{unpack 'C*', $uc} = 1;
                                $singleoctet_ignorecase{unpack 'C*', $fc} = 1;
                            }
                            else {
                                $singleoctet_ignorecase{unpack 'C*', $uc} = 1;
                                push @multipleoctet, join '', map {sprintf('\x%02X',$_)} unpack 'C*', $fc;
                            }
                        }
                    }
                }
                if ($_ ne '') {
                    $singleoctet_ignorecase{unpack 'C*', $_} = 1;
                }
            }
            my $i = 0;
            my @singleoctet_ignorecase = ();
            for my $ord (0 .. 255) {
                if (exists $singleoctet_ignorecase{$ord}) {
                    push @{$singleoctet_ignorecase[$i]}, $ord;
                }
                else {
                    $i++;
                }
            }
            @singleoctet = ();
            for my $range (@singleoctet_ignorecase) {
                if (ref $range) {
                    if (scalar(@{$range}) == 1) {
                        push @singleoctet, sprintf('\x%02X', @{$range}[0]);
                    }
                    elsif (scalar(@{$range}) == 2) {
                        push @singleoctet, sprintf('\x%02X\x%02X', @{$range}[0], @{$range}[-1]);
                    }
                    else {
                        push @singleoctet, sprintf('\x%02X-\x%02X', @{$range}[0], @{$range}[-1]);
                    }
                }
            }
        }

        my $not_anchor = '';
#if MULTIBYTE_ENCODING
        $not_anchor = '(?![\x81-\x9F\xE0-\xFC])';
#endif

        push @multipleoctet, join('', $not_anchor, '[', @singleoctet, ']' );
    }
    if (scalar(@multipleoctet) >= 2) {
        return '(?:' . join('|', @multipleoctet) . ')';
    }
    else {
        return $multipleoctet[0];
    }
}

#
# ShiftJIS open character list for not qr
#
sub charlist_not_qr {

    my $modifier = pop @_;
    my @char = @_;

    my($singleoctet, $multipleoctet) = _charlist(@char, $modifier);
    my @singleoctet   = @$singleoctet;
    my @multipleoctet = @$multipleoctet;

    # with /i modifier
    if ($modifier =~ m/i/oxms) {
        my %singleoctet_ignorecase = ();
        for (@singleoctet) {
            while (s/ \A \\x(..) - \\x(..) //oxms or s/ \A \\x((..)) //oxms) {
                for my $ord (hex($1) .. hex($2)) {
                    my $char = CORE::chr($ord);
                    my $uc = Esjis::uc($char);
                    my $fc = Esjis::fc($char);
                    if ($uc eq $fc) {
                        $singleoctet_ignorecase{unpack 'C*', $char} = 1;
                    }
                    else {
                        if (CORE::length($fc) == 1) {
                            $singleoctet_ignorecase{unpack 'C*', $uc} = 1;
                            $singleoctet_ignorecase{unpack 'C*', $fc} = 1;
                        }
                        else {
                            $singleoctet_ignorecase{unpack 'C*', $uc} = 1;
                            push @multipleoctet, join '', map {sprintf('\x%02X',$_)} unpack 'C*', $fc;
                        }
                    }
                }
            }
            if ($_ ne '') {
                $singleoctet_ignorecase{unpack 'C*', $_} = 1;
            }
        }
        my $i = 0;
        my @singleoctet_ignorecase = ();
        for my $ord (0 .. 255) {
            if (exists $singleoctet_ignorecase{$ord}) {
                push @{$singleoctet_ignorecase[$i]}, $ord;
            }
            else {
                $i++;
            }
        }
        @singleoctet = ();
        for my $range (@singleoctet_ignorecase) {
            if (ref $range) {
                if (scalar(@{$range}) == 1) {
                    push @singleoctet, sprintf('\x%02X', @{$range}[0]);
                }
                elsif (scalar(@{$range}) == 2) {
                    push @singleoctet, sprintf('\x%02X\x%02X', @{$range}[0], @{$range}[-1]);
                }
                else {
                    push @singleoctet, sprintf('\x%02X-\x%02X', @{$range}[0], @{$range}[-1]);
                }
            }
        }
    }

    # return character list
    if (scalar(@multipleoctet) >= 1) {
        if (scalar(@singleoctet) >= 1) {

            # any character other than multiple-octet and single octet character class
            return '(?!' . join('|', @multipleoctet) . ')(?:[^\x81-\x9F\xE0-\xFC' . join('', @singleoctet) . ']|[\x81-\x9F\xE0-\xFC][\x00-\xFF])';
        }
        else {

            # any character other than multiple-octet character class
            return '(?!' . join('|', @multipleoctet) . ")(?:$your_char)";
        }
    }
    else {
        if (scalar(@singleoctet) >= 1) {

            # any character other than single octet character class
            return                                      '(?:[^\x81-\x9F\xE0-\xFC' . join('', @singleoctet) . ']|[\x81-\x9F\xE0-\xFC][\x00-\xFF])';
        }
        else {

            # any character
            return                                      "(?:$your_char)";
        }
    }
}

#
# open file in read mode
#
sub _open_r {
    my(undef,$file) = @_;
    use Fcntl qw(O_RDONLY);
    return CORE::sysopen($_[0], $file, &O_RDONLY);
}

#
# open file in append mode
#
sub _open_a {
    my(undef,$file) = @_;
    use Fcntl qw(O_WRONLY O_APPEND O_CREAT);
    return CORE::sysopen($_[0], $file, &O_WRONLY|&O_APPEND|&O_CREAT);
}

#
# safe system
#
sub _systemx {

    # P.707 29.2.33. exec
    # in Chapter 29: Functions
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.
    #
    # Be aware that in older releases of Perl, exec (and system) did not flush
    # your output buffer, so you needed to enable command buffering by setting $|
    # on one or more filehandles to avoid lost output in the case of exec, or
    # misordererd output in the case of system. This situation was largely remedied
    # in the 5.6 release of Perl. (So, 5.005 release not yet.)

    # P.855 exec
    # in Chapter 27: Functions
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
    #
    # In very old release of Perl (before v5.6), exec (and system) did not flush
    # your output buffer, so you needed to enable command buffering by setting $|
    # on one or more filehandles to avoid lost output with exec or misordered
    # output with system.

    $| = 1;

    # P.565 23.1.2. Cleaning Up Your Environment
    # in Chapter 23: Security
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.656 Cleaning Up Your Environment
    # in Chapter 20: Security
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # local $ENV{'PATH'} = '.';
    local @ENV{qw(IFS CDPATH ENV BASH_ENV)}; # Make %ENV safer

    # P.707 29.2.33. exec
    # in Chapter 29: Functions
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.
    #
    # As we mentioned earlier, exec treats a discrete list of arguments as an
    # indication that it should bypass shell processing. However, there is one
    # place where you might still get tripped up. The exec call (and system, too)
    # will not distinguish between a single scalar argument and an array containing
    # only one element.
    #
    #     @args = ("echo surprise");  # just one element in list
    #     exec @args                  # still subject to shell escapes
    #         or die "exec: $!";      #   because @args == 1
    #
    # To avoid this, you can use the PATHNAME syntax, explicitly duplicating the
    # first argument as the pathname, which forces the rest of the arguments to be
    # interpreted as a list, even if there is only one of them:
    #
    #     exec { $args[0] } @args  # safe even with one-argument list
    #         or die "can't exec @args: $!";

    # P.855 exec
    # in Chapter 27: Functions
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
    #
    # As we mentioned earlier, exec treats a discrete list of arguments as a
    # directive to bypass shell processing. However, there is one place where
    # you might still get tripped up. The exec call (and system, too) cannot
    # distinguish between a single scalar argument and an array containing
    # only one element.
    #
    #     @args = ("echo surprise");  # just one element in list
    #     exec @args                  # still subject to shell escapes
    #         || die "exec: $!";      #   because @args == 1
    #
    # To avoid this, use the PATHNAME syntax, explicitly duplicating the first
    # argument as the pathname, which forces the rest of the arguments to be
    # interpreted as a list, even if there is only one of them:
    #
    #     exec { $args[0] } @args  # safe even with one-argument list
    #         || die "can't exec @args: $!";

    return CORE::system { $_[0] } @_; # safe even with one-argument list
}

#
# ShiftJIS order to character (with parameter)
#
sub Esjis::chr(;$) {

    my $c = @_ ? $_[0] : $_;

    if ($c == 0x00) {
        return "\x00";
    }
    else {
        my @chr = ();
        while ($c > 0) {
            unshift @chr, ($c % 0x100);
            $c = int($c / 0x100);
        }
        return pack 'C*', @chr;
    }
}

#
# ShiftJIS order to character (without parameter)
#
sub Esjis::chr_() {

    my $c = $_;

    if ($c == 0x00) {
        return "\x00";
    }
    else {
        my @chr = ();
        while ($c > 0) {
            unshift @chr, ($c % 0x100);
            $c = int($c / 0x100);
        }
        return pack 'C*', @chr;
    }
}

#if ESCAPE_SECOND_OCTET
#
# ShiftJIS stacked file test expr
#
sub Esjis::filetest {

    my $file     = pop @_;
    my $filetest = substr(pop @_, 1);

    unless (CORE::eval qq{Esjis::$filetest(\$file)}) {
        return '';
    }
    for my $filetest (CORE::reverse @_) {
        unless (CORE::eval qq{ $filetest _ }) {
            return '';
        }
    }
    return 1;
}

#
# ShiftJIS file test -r expr
#
sub Esjis::r(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -r (Esjis::r)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-r _,@_) : -r _;
    }

    # P.908 32.39. Symbol
    # in Chapter 32: Standard Modules
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.326 Prototypes
    # in Chapter 7: Subroutines
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # (and so on)

    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-r $fh,@_) : -r $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-r _,@_) : -r _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-r _,@_) : -r _;
        }
        else {

            # Even if ${^WIN32_SLOPPY_STAT} is set to a true value, Esjis::*()
            # on Windows opens the file for the path which has 5c at end.
            # (and so on)

            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $r = -r $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($r,@_) : $r;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -w expr
#
sub Esjis::w(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -w (Esjis::w)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-w _,@_) : -w _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-w $fh,@_) : -w $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-w _,@_) : -w _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-w _,@_) : -w _;
        }
        else {
            my $fh = gensym();
            if (_open_a($fh, $_)) {
                my $w = -w $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($w,@_) : $w;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -x expr
#
sub Esjis::x(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -x (Esjis::x)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-x _,@_) : -x _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-x $fh,@_) : -x $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-x _,@_) : -x _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-x _,@_) : -x _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $dummy_for_underline_cache = -x $fh;
                close($fh) or die "Can't close file: $_: $!";
            }

            # filename is not .COM .EXE .BAT .CMD
            return wantarray ? ('',@_) : '';
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -o expr
#
sub Esjis::o(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -o (Esjis::o)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-o _,@_) : -o _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-o $fh,@_) : -o $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-o _,@_) : -o _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-o _,@_) : -o _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $o = -o $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($o,@_) : $o;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -R expr
#
sub Esjis::R(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -R (Esjis::R)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-R _,@_) : -R _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-R $fh,@_) : -R $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-R _,@_) : -R _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-R _,@_) : -R _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $R = -R $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($R,@_) : $R;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -W expr
#
sub Esjis::W(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -W (Esjis::W)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-W _,@_) : -W _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-W $fh,@_) : -W $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-W _,@_) : -W _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-W _,@_) : -W _;
        }
        else {
            my $fh = gensym();
            if (_open_a($fh, $_)) {
                my $W = -W $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($W,@_) : $W;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -X expr
#
sub Esjis::X(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -X (Esjis::X)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-X _,@_) : -X _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-X $fh,@_) : -X $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-X _,@_) : -X _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-X _,@_) : -X _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $dummy_for_underline_cache = -X $fh;
                close($fh) or die "Can't close file: $_: $!";
            }

            # filename is not .COM .EXE .BAT .CMD
            return wantarray ? ('',@_) : '';
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -O expr
#
sub Esjis::O(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -O (Esjis::O)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-O _,@_) : -O _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-O $fh,@_) : -O $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-O _,@_) : -O _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-O _,@_) : -O _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $O = -O $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($O,@_) : $O;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -e expr
#
sub Esjis::e(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -e (Esjis::e)' if @_ and not wantarray;

    local $^W = 0;
    local $SIG{__WARN__} = sub {}; # telldir() attempted on invalid dirhandle at here

    my $fh = qualify_to_ref $_;
    if ($_ eq '_') {
        return wantarray ? (-e _,@_) : -e _;
    }

    # return false if directory handle
    elsif (defined Esjis::telldir($fh)) {
        return wantarray ? ('',@_) : '';
    }

    # return true if file handle
    elsif (defined fileno $fh) {
        return wantarray ? (1,@_) : 1;
    }

    elsif (-e $_) {
        return wantarray ? (1,@_) : 1;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (1,@_) : 1;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $e = -e $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($e,@_) : $e;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -z expr
#
sub Esjis::z(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -z (Esjis::z)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-z _,@_) : -z _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-z $fh,@_) : -z $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-z _,@_) : -z _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-z _,@_) : -z _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $z = -z $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($z,@_) : $z;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -s expr
#
sub Esjis::s(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -s (Esjis::s)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-s _,@_) : -s _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-s $fh,@_) : -s $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-s _,@_) : -s _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-s _,@_) : -s _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $s = -s $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($s,@_) : $s;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -f expr
#
sub Esjis::f(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -f (Esjis::f)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-f _,@_) : -f _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-f $fh,@_) : -f $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-f _,@_) : -f _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? ('',@_) : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $f = -f $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($f,@_) : $f;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -d expr
#
sub Esjis::d(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -d (Esjis::d)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-d _,@_) : -d _;
    }

    # return false if file handle or directory handle
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? ('',@_) : '';
    }
    elsif (-e $_) {
        return wantarray ? (-d _,@_) : -d _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        return wantarray ? (-d "$_/.",@_) : -d "$_/.";
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -l expr
#
sub Esjis::l(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -l (Esjis::l)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-l _,@_) : -l _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-l $fh,@_) : -l $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-l _,@_) : -l _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-l _,@_) : -l _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $l = -l $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($l,@_) : $l;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -p expr
#
sub Esjis::p(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -p (Esjis::p)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-p _,@_) : -p _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-p $fh,@_) : -p $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-p _,@_) : -p _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-p _,@_) : -p _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $p = -p $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($p,@_) : $p;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -S expr
#
sub Esjis::S(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -S (Esjis::S)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-S _,@_) : -S _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-S $fh,@_) : -S $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-S _,@_) : -S _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-S _,@_) : -S _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $S = -S $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($S,@_) : $S;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -b expr
#
sub Esjis::b(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -b (Esjis::b)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-b _,@_) : -b _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-b $fh,@_) : -b $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-b _,@_) : -b _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-b _,@_) : -b _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $b = -b $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($b,@_) : $b;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -c expr
#
sub Esjis::c(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -c (Esjis::c)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-c _,@_) : -c _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-c $fh,@_) : -c $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-c _,@_) : -c _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-c _,@_) : -c _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $c = -c $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($c,@_) : $c;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -u expr
#
sub Esjis::u(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -u (Esjis::u)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-u _,@_) : -u _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-u $fh,@_) : -u $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-u _,@_) : -u _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-u _,@_) : -u _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $u = -u $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($u,@_) : $u;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -g expr
#
sub Esjis::g(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -g (Esjis::g)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-g _,@_) : -g _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-g $fh,@_) : -g $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-g _,@_) : -g _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-g _,@_) : -g _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $g = -g $fh;
                close($fh) or die "Can't close file: $_: $!";
                return wantarray ? ($g,@_) : $g;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -k expr
#
sub Esjis::k(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -k (Esjis::k)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? ('',@_) : '';
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? ('',@_) : '';
    }
    elsif ($] =~ /^5\.008/oxms) {
        return wantarray ? ('',@_) : '';
    }
    return wantarray ? ($_,@_) : $_;
}

#
# ShiftJIS file test -T expr
#
sub Esjis::T(;*@) {

    local $_ = shift if @_;

    # Use of croak without parentheses makes die on Strawberry Perl 5.008 and 5.010, like:
    #     croak 'Too many arguments for -T (Esjis::T)';
    # Must be used by parentheses like:
    #     croak('Too many arguments for -T (Esjis::T)');

    if (@_ and not wantarray) {
        croak('Too many arguments for -T (Esjis::T)');
    }

    my $T = 1;

    my $fh = qualify_to_ref $_;
    if (defined fileno $fh) {

        local $SIG{__WARN__} = sub {}; # telldir() attempted on invalid dirhandle at here
        if (defined Esjis::telldir($fh)) {
            return wantarray ? (undef,@_) : undef;
        }

        # P.813 29.2.176. tell
        # in Chapter 29: Functions
        # of ISBN 0-596-00027-8 Programming Perl Third Edition.

        # P.970 tell
        # in Chapter 27: Functions
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # (and so on)

        my $systell = sysseek $fh, 0, 1;

        if (sysread $fh, my $block, 512) {

            # P.163 Binary file check in Little Perl Parlor 16
            # of Book No. T1008901080816 ZASSHI 08901-8 UNIX MAGAZINE 1993 Aug VOL8#8
            # (and so on)

            if ($block =~ /[\000\377]/oxms) {
                $T = '';
            }
            elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
                $T = '';
            }
        }

        # 0 byte or eof
        else {
            $T = 1;
        }

        my $dummy_for_underline_cache = -T $fh;
        sysseek $fh, $systell, 0;
    }
    else {
        if (-d $_ or -d "$_/.") {
            return wantarray ? (undef,@_) : undef;
        }

        $fh = gensym();
        if (_open_r($fh, $_)) {
        }
        else {
            return wantarray ? (undef,@_) : undef;
        }
        if (sysread $fh, my $block, 512) {
            if ($block =~ /[\000\377]/oxms) {
                $T = '';
            }
            elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
                $T = '';
            }
        }

        # 0 byte or eof
        else {
            $T = 1;
        }
        my $dummy_for_underline_cache = -T $fh;
        close($fh) or die "Can't close file: $_: $!";
    }

    return wantarray ? ($T,@_) : $T;
}

#
# ShiftJIS file test -B expr
#
sub Esjis::B(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -B (Esjis::B)' if @_ and not wantarray;
    my $B = '';

    my $fh = qualify_to_ref $_;
    if (defined fileno $fh) {

        local $SIG{__WARN__} = sub {}; # telldir() attempted on invalid dirhandle at here
        if (defined Esjis::telldir($fh)) {
            return wantarray ? (undef,@_) : undef;
        }

        my $systell = sysseek $fh, 0, 1;

        if (sysread $fh, my $block, 512) {
            if ($block =~ /[\000\377]/oxms) {
                $B = 1;
            }
            elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
                $B = 1;
            }
        }

        # 0 byte or eof
        else {
            $B = 1;
        }

        my $dummy_for_underline_cache = -B $fh;
        sysseek $fh, $systell, 0;
    }
    else {
        if (-d $_ or -d "$_/.") {
            return wantarray ? (undef,@_) : undef;
        }

        $fh = gensym();
        if (_open_r($fh, $_)) {
        }
        else {
            return wantarray ? (undef,@_) : undef;
        }
        if (sysread $fh, my $block, 512) {
            if ($block =~ /[\000\377]/oxms) {
                $B = 1;
            }
            elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
                $B = 1;
            }
        }

        # 0 byte or eof
        else {
            $B = 1;
        }
        my $dummy_for_underline_cache = -B $fh;
        close($fh) or die "Can't close file: $_: $!";
    }

    return wantarray ? ($B,@_) : $B;
}

#
# ShiftJIS file test -M expr
#
sub Esjis::M(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -M (Esjis::M)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-M _,@_) : -M _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-M $fh,@_) : -M $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-M _,@_) : -M _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-M _,@_) : -M _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $M = ($^T - $mtime) / (24*60*60);
                return wantarray ? ($M,@_) : $M;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -A expr
#
sub Esjis::A(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -A (Esjis::A)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-A _,@_) : -A _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-A $fh,@_) : -A $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-A _,@_) : -A _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-A _,@_) : -A _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $A = ($^T - $atime) / (24*60*60);
                return wantarray ? ($A,@_) : $A;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS file test -C expr
#
sub Esjis::C(;*@) {

    local $_ = shift if @_;
    croak 'Too many arguments for -C (Esjis::C)' if @_ and not wantarray;

    if ($_ eq '_') {
        return wantarray ? (-C _,@_) : -C _;
    }
    elsif (defined fileno(my $fh = qualify_to_ref $_)) {
        return wantarray ? (-C $fh,@_) : -C $fh;
    }
    elsif (-e $_) {
        return wantarray ? (-C _,@_) : -C _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return wantarray ? (-C _,@_) : -C _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $C = ($^T - $ctime) / (24*60*60);
                return wantarray ? ($C,@_) : $C;
            }
        }
    }
    return wantarray ? (undef,@_) : undef;
}

#
# ShiftJIS stacked file test $_
#
sub Esjis::filetest_ {

    my $filetest = substr(pop @_, 1);

    unless (CORE::eval qq{Esjis::${filetest}_}) {
        return '';
    }
    for my $filetest (CORE::reverse @_) {
        unless (CORE::eval qq{ $filetest _ }) {
            return '';
        }
    }
    return 1;
}

#
# ShiftJIS file test -r $_
#
sub Esjis::r_() {

    if (-e $_) {
        return -r _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -r _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $r = -r $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $r ? 1 : '';
            }
        }
    }

# 10.10. Returning Failure
# in Chapter 10. Subroutines
# of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
# (and so on)

# 2010-01-26 The difference of "return;" and "return undef;"
# http://d.hatena.ne.jp/gfx/20100126/1264474754
#
# "Perl Best Practices" recommends to use "return;"*1 to return nothing, but
# it might be wrong in some cases. If you use this idiom for those functions
# which are expected to return a scalar value, e.g. searching functions, the
# user of those functions will be surprised at what they return in list
# context, an empty list - note that many functions and all the methods
# evaluate their arguments in list context. You'd better to use "return undef;"
# for such scalar functions.
#
#     sub search_something {
#         my($arg) = @_;
#         # search_something...
#         if(defined $found){
#             return $found;
#         }
#         return; # XXX: you'd better to "return undef;"
#     }
#
#     # ...
#
#     # you'll get what you want, but ...
#     my $something = search_something($source);
#
#     # you won't get what you want here.
#     # @_ for doit() is (-foo => $opt), not (undef, -foo => $opt).
#     $obj->doit(search_something($source), -option=> $optval);
#
#     # you have to use the "scalar" operator in such a case.
#     $obj->doit(scalar search_something($source), ...);
#
# *1: it returns an empty list in list context, or returns undef in scalar
#     context
#
# (and so on)

    return undef;
}

#
# ShiftJIS file test -w $_
#
sub Esjis::w_() {

    if (-e $_) {
        return -w _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -w _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_a($fh, $_)) {
                my $w = -w $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $w ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -x $_
#
sub Esjis::x_() {

    if (-e $_) {
        return -x _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -x _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $dummy_for_underline_cache = -x $fh;
                close($fh) or die "Can't close file: $_: $!";
            }

            # filename is not .COM .EXE .BAT .CMD
            return '';
        }
    }
    return undef;
}

#
# ShiftJIS file test -o $_
#
sub Esjis::o_() {

    if (-e $_) {
        return -o _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -o _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $o = -o $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $o ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -R $_
#
sub Esjis::R_() {

    if (-e $_) {
        return -R _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -R _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $R = -R $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $R ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -W $_
#
sub Esjis::W_() {

    if (-e $_) {
        return -W _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -W _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_a($fh, $_)) {
                my $W = -W $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $W ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -X $_
#
sub Esjis::X_() {

    if (-e $_) {
        return -X _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -X _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $dummy_for_underline_cache = -X $fh;
                close($fh) or die "Can't close file: $_: $!";
            }

            # filename is not .COM .EXE .BAT .CMD
            return '';
        }
    }
    return undef;
}

#
# ShiftJIS file test -O $_
#
sub Esjis::O_() {

    if (-e $_) {
        return -O _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -O _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $O = -O $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $O ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -e $_
#
sub Esjis::e_() {

    if (-e $_) {
        return 1;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return 1;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $e = -e $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $e ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -z $_
#
sub Esjis::z_() {

    if (-e $_) {
        return -z _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -z _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $z = -z $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $z ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -s $_
#
sub Esjis::s_() {

    if (-e $_) {
        return -s _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -s _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $s = -s $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $s;
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -f $_
#
sub Esjis::f_() {

    if (-e $_) {
        return -f _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $f = -f $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $f ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -d $_
#
sub Esjis::d_() {

    if (-e $_) {
        return -d _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        return -d "$_/." ? 1 : '';
    }
    return undef;
}

#
# ShiftJIS file test -l $_
#
sub Esjis::l_() {

    if (-e $_) {
        return -l _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -l _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $l = -l $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $l ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -p $_
#
sub Esjis::p_() {

    if (-e $_) {
        return -p _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -p _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $p = -p $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $p ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -S $_
#
sub Esjis::S_() {

    if (-e $_) {
        return -S _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -S _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $S = -S $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $S ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -b $_
#
sub Esjis::b_() {

    if (-e $_) {
        return -b _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -b _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $b = -b $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $b ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -c $_
#
sub Esjis::c_() {

    if (-e $_) {
        return -c _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -c _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $c = -c $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $c ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -u $_
#
sub Esjis::u_() {

    if (-e $_) {
        return -u _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -u _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $u = -u $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $u ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -g $_
#
sub Esjis::g_() {

    if (-e $_) {
        return -g _ ? 1 : '';
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -g _ ? 1 : '';
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my $g = -g $fh;
                close($fh) or die "Can't close file: $_: $!";
                return $g ? 1 : '';
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -k $_
#
sub Esjis::k_() {

    if ($] =~ /^5\.008/oxms) {
        return wantarray ? ('',@_) : '';
    }
    return wantarray ? ($_,@_) : $_;
}

#
# ShiftJIS file test -T $_
#
sub Esjis::T_() {

    my $T = 1;

    if (-d $_ or -d "$_/.") {
        return undef;
    }
    my $fh = gensym();
    if (_open_r($fh, $_)) {
    }
    else {
        return undef;
    }

    if (sysread $fh, my $block, 512) {
        if ($block =~ /[\000\377]/oxms) {
            $T = '';
        }
        elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
            $T = '';
        }
    }

    # 0 byte or eof
    else {
        $T = 1;
    }
    my $dummy_for_underline_cache = -T $fh;
    close($fh) or die "Can't close file: $_: $!";

    return $T;
}

#
# ShiftJIS file test -B $_
#
sub Esjis::B_() {

    my $B = '';

    if (-d $_ or -d "$_/.") {
        return undef;
    }
    my $fh = gensym();
    if (_open_r($fh, $_)) {
    }
    else {
        return undef;
    }

    if (sysread $fh, my $block, 512) {
        if ($block =~ /[\000\377]/oxms) {
            $B = 1;
        }
        elsif (($block =~ tr/\000-\007\013\016-\032\034-\037\377//) * 10 > CORE::length $block) {
            $B = 1;
        }
    }

    # 0 byte or eof
    else {
        $B = 1;
    }
    my $dummy_for_underline_cache = -B $fh;
    close($fh) or die "Can't close file: $_: $!";

    return $B;
}

#
# ShiftJIS file test -M $_
#
sub Esjis::M_() {

    if (-e $_) {
        return -M _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -M _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $M = ($^T - $mtime) / (24*60*60);
                return $M;
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -A $_
#
sub Esjis::A_() {

    if (-e $_) {
        return -A _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -A _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $A = ($^T - $atime) / (24*60*60);
                return $A;
            }
        }
    }
    return undef;
}

#
# ShiftJIS file test -C $_
#
sub Esjis::C_() {

    if (-e $_) {
        return -C _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        if (-d "$_/.") {
            return -C _;
        }
        else {
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = CORE::stat $fh;
                close($fh) or die "Can't close file: $_: $!";
                my $C = ($^T - $ctime) / (24*60*60);
                return $C;
            }
        }
    }
    return undef;
}

#endif /* #if ESCAPE_SECOND_OCTET */
#
# ShiftJIS path globbing (with parameter)
#
sub Esjis::glob($) {

    if (wantarray) {
        my @glob = _DOS_like_glob(@_);
        for my $glob (@glob) {
            $glob =~ s{ \A (?:\./)+ }{}oxms;
        }
        return @glob;
    }
    else {
        my $glob = _DOS_like_glob(@_);
        $glob =~ s{ \A (?:\./)+ }{}oxms;
        return $glob;
    }
}

#
# ShiftJIS path globbing (without parameter)
#
sub Esjis::glob_() {

    if (wantarray) {
        my @glob = _DOS_like_glob();
        for my $glob (@glob) {
            $glob =~ s{ \A (?:\./)+ }{}oxms;
        }
        return @glob;
    }
    else {
        my $glob = _DOS_like_glob();
        $glob =~ s{ \A (?:\./)+ }{}oxms;
        return $glob;
    }
}

#
# ShiftJIS path globbing via File::DosGlob 1.10
#
# Often I confuse "_dosglob" and "_doglob".
# So, I renamed "_dosglob" to "_DOS_like_glob".
#
my %iter;
my %entries;
sub _DOS_like_glob {

    # context (keyed by second cxix argument provided by core)
    my($expr,$cxix) = @_;

    # glob without args defaults to $_
    $expr = $_ if not defined $expr;

    # represents the current user's home directory
    #
    # 7.3. Expanding Tildes in Filenames
    # in Chapter 7. File Access
    # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
    #
    # and File::HomeDir, File::HomeDir::Windows module

    # DOS-like system
    if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
        $expr =~ s{ \A ~ (?= [^/\\] ) }
                  { my_home_MSWin32() }oxmse;
    }

#if MACPERL
    # Mac OS system
    elsif ($^O eq 'MacOS') {
        if ($expr =~ / \A ~ /oxms) {
            $expr =~ s{ \A ~ (?= [^/:] ) }
                      { my_home_MacOS() }oxmse;
        }
    }

#endif
    # UNIX-like system
    else {
        $expr =~ s{ \A ~ ( (?:[^\x81-\x9F\xE0-\xFC/]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])* ) }
                  { $1 ? (CORE::eval(q{(getpwnam($1))[7]})||my_home()) : my_home() }oxmse;
    }

    # assume global context if not provided one
    $cxix = '_G_' if not defined $cxix;
    $iter{$cxix} = 0 if not exists $iter{$cxix};

    # if we're just beginning, do it all first
    if ($iter{$cxix} == 0) {
#if MACPERL
        if ($^O eq 'MacOS') {

            # first, take care of updirs and trailing colons
            my @expr = _canonpath_MacOS(_parse_line($expr));

            # expand volume names
            @expr = _expand_volume_MacOS(@expr);

            $entries{$cxix} = (@expr) ? [ map { _unescape_MacOS($_) } _do_glob_MacOS(1,@expr) ] : [()];
        }
        else {
#endif
            $entries{$cxix} = [ _do_glob(1, _parse_line($expr)) ];
#if MACPERL
        }
#endif
    }

    # chuck it all out, quick or slow
    if (wantarray) {
        delete $iter{$cxix};
        return @{delete $entries{$cxix}};
    }
    else {
        if ($iter{$cxix} = scalar @{$entries{$cxix}}) {
            return shift @{$entries{$cxix}};
        }
        else {
            # return undef for EOL
            delete $iter{$cxix};
            delete $entries{$cxix};
            return undef;
        }
    }
}

#
# ShiftJIS path globbing subroutine
#
sub _do_glob {

    my($cond,@expr) = @_;
    my @glob = ();
    my $fix_drive_relative_paths = 0;

OUTER:
    for my $expr (@expr) {
        next OUTER if not defined $expr;
        next OUTER if $expr eq '';

        my @matched = ();
        my @globdir = ();
        my $head    = '.';
        my $pathsep = '/';
        my $tail;

        # if argument is within quotes strip em and do no globbing
        if ($expr =~ /\A " ((?:$q_char)*?) " \z/oxms) {
            $expr = $1;
            if ($cond eq 'd') {
                if (Esjis::d $expr) {
                    push @glob, $expr;
                }
            }
            else {
                if (Esjis::e $expr) {
                    push @glob, $expr;
                }
            }
            next OUTER;
        }

        # wildcards with a drive prefix such as h:*.pm must be changed
        # to h:./*.pm to expand correctly
        if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
            if ($expr =~ s# \A ((?:[A-Za-z]:)?) ([^\x81-\x9F\xE0-\xFC/\\]|[\x81-\x9F\xE0-\xFC][\x00-\xFF]) #$1./$2#oxms) {
                $fix_drive_relative_paths = 1;
            }
        }

        if (($head, $tail) = _parse_path($expr,$pathsep)) {
            if ($tail eq '') {
                push @glob, $expr;
                next OUTER;
            }
            if ($head =~ / \A (?:$q_char)*? [*?] /oxms) {
                if (@globdir = _do_glob('d', $head)) {
                    push @glob, _do_glob($cond, map {"$_$pathsep$tail"} @globdir);
                    next OUTER;
                }
            }
            if ($head eq '' or $head =~ /\A [A-Za-z]: \z/oxms) {
                $head .= $pathsep;
            }
            $expr = $tail;
        }

        # If file component has no wildcards, we can avoid opendir
        if ($expr !~ / \A (?:$q_char)*? [*?] /oxms) {
            if ($head eq '.') {
                $head = '';
            }
            if ($head ne '' and ($head =~ / \G ($q_char) /oxmsg)[-1] ne $pathsep) {
                $head .= $pathsep;
            }
            $head .= $expr;
            if ($cond eq 'd') {
                if (Esjis::d $head) {
                    push @glob, $head;
                }
            }
            else {
                if (Esjis::e $head) {
                    push @glob, $head;
                }
            }
            next OUTER;
        }
        Esjis::opendir(*DIR, $head) or next OUTER;
        my @leaf = readdir DIR;
        closedir DIR;

        if ($head eq '.') {
            $head = '';
        }
        if ($head ne '' and ($head =~ / \G ($q_char) /oxmsg)[-1] ne $pathsep) {
            $head .= $pathsep;
        }

        my $pattern = '';
        while ($expr =~ / \G ($q_char) /oxgc) {
            my $char = $1;

            # 6.9. Matching Shell Globs as Regular Expressions
            # in Chapter 6. Pattern Matching
            # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
            # (and so on)

            if ($char eq '*') {
                $pattern .= "(?:$your_char)*",
            }
            elsif ($char eq '?') {
                $pattern .= "(?:$your_char)?",  # DOS style
#               $pattern .= "(?:$your_char)",   # UNIX style
            }
            elsif ((my $fc = Esjis::fc($char)) ne $char) {
                $pattern .= $fc;
            }
            else {
                $pattern .= quotemeta $char;
            }
        }
        my $matchsub = sub { Esjis::fc($_[0]) =~ /\A $pattern \z/xms };

#       if ($@) {
#           print STDERR "$0: $@\n";
#           next OUTER;
#       }

INNER:
        for my $leaf (@leaf) {
            if ($leaf eq '.' or $leaf eq '..') {
                next INNER;
            }
            if ($cond eq 'd' and not Esjis::d "$head$leaf") {
                next INNER;
            }

            if (&$matchsub($leaf)) {
                push @matched, "$head$leaf";
                next INNER;
            }

            # [DOS compatibility special case]
            # Failed, add a trailing dot and try again, but only...

            if (Esjis::index($leaf,'.') == -1 and   # if name does not have a dot in it *and*
                CORE::length($leaf) <= 8 and        # name is shorter than or equal to 8 chars *and*
                Esjis::index($pattern,'\\.') != -1  # pattern has a dot.
            ) {
                if (&$matchsub("$leaf.")) {
                    push @matched, "$head$leaf";
                    next INNER;
                }
            }
        }
        if (@matched) {
            push @glob, @matched;
        }
    }
    if ($fix_drive_relative_paths) {
        for my $glob (@glob) {
            $glob =~ s# \A ([A-Za-z]:) \./ #$1#oxms;
        }
    }
    return @glob;
}

#
# ShiftJIS parse line
#
sub _parse_line {

    my($line) = @_;

    $line .= ' ';
    my @piece = ();
    while ($line =~ /
        " ( (?>(?: [^\x81-\x9F\xE0-\xFC"]  |[\x81-\x9F\xE0-\xFC][\x00-\xFF] )* ) ) " (?>\s+) |
          ( (?>(?: [^\x81-\x9F\xE0-\xFC"\s]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] )* ) )   (?>\s+)
        /oxmsg
    ) {
        push @piece, defined($1) ? $1 : $2;
    }
    return @piece;
}

#
# ShiftJIS parse path
#
sub _parse_path {

    my($path,$pathsep) = @_;

    $path .= '/';
    my @subpath = ();
    while ($path =~ /
        ((?: [^\x81-\x9F\xE0-\xFC\/\\]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] )+?) [\/\\]
        /oxmsg
    ) {
        push @subpath, $1;
    }

    my $tail = pop @subpath;
    my $head = join $pathsep, @subpath;
    return $head, $tail;
}

#if MACPERL
#
# ShiftJIS path globbing on Mac OS
#
sub _do_glob_MacOS {

    my($cond,@expr) = @_;
    my @glob = ();

OUTER_MACOS:
    for my $expr (@expr) {
        next OUTER_MACOS if not defined $expr;
        next OUTER_MACOS if $expr eq '';

        my @matched = ();
        my @globdir = ();
        my $head    = ':';
        my $unesc_head = $head;
        my $pathsep = ':';
        my $tail;

        # if $expr is within quotes strip em and do no globbing
        if ($expr =~ /\A " ((?:$q_char)*?) " \z/oxms) {
            $expr = $1;

            # $expr may contain escaped metachars '\*', '\?', and '\'
            $expr = _unescape_MacOS($expr);

            if ($cond eq 'd') {
                if (Esjis::d $expr) {
                    push @glob, $expr;
                }
            }
            else {
                if (Esjis::e $expr) {
                    push @glob, $expr;
                }
            }
            next OUTER_MACOS;
        }

        # note: $1 is not greedy
        if (($head,$pathsep,$tail) = $expr =~ /\A ((?:$q_char)*?) (:+) ((?:[^\x81-\x9F\xE0-\xFC:]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?) \z/oxms) {
            if ($tail eq '') {
                push @glob, $expr;
                next OUTER_MACOS;
            }
            if (_hasmeta_MacOS($head)) {
                if (@globdir = _do_glob_MacOS('d', $head)) {
                    push @glob, _do_glob_MacOS($cond, map {"$_$pathsep$tail"} @globdir);
                    next OUTER_MACOS;
                }
            }
            $head .= $pathsep;

            # unescape $head for file operations
            $unesc_head = _unescape_MacOS($head);

            $expr = $tail;
        }

        # If file component has no wildcards, we can avoid opendir
        if (not _hasmeta_MacOS($expr)) {
            if ($head eq ':') {
                $unesc_head = $head = '';
            }
            $head .= $expr;

            # unescape $head and $expr for file operations
            $unesc_head .= _unescape_MacOS($expr);

            if ($cond eq 'd') {
                if (Esjis::d $unesc_head) {
                    push @glob, $head;
                }
            }
            else {
                if (Esjis::e $unesc_head) {
                    push @glob, $head;
                }
            }
            next OUTER_MACOS;
        }
        Esjis::opendir(*DIR, $head) or next OUTER_MACOS;
        my @leaf = readdir DIR;
        closedir DIR;

        my $pattern = _quotemeta_MacOS($expr);
        my $matchsub = sub { Esjis::fc($_[0]) =~ /\A $pattern \z/xms };

#       if ($@) {
#           print STDERR "$0: $@\n";
#           next OUTER_MACOS;
#       }

INNER_MACOS:
        for my $leaf (@leaf) {
            if ($leaf eq '.' or $leaf eq '..') {
                next INNER_MACOS;
            }
            if ($cond eq 'd' and not Esjis::d qq{$unesc_head$leaf}) {
                next INNER_MACOS;
            }

            if (&$matchsub($leaf)) {
                if (($unesc_head eq ':') and (Esjis::f qq{$unesc_head$leaf})) {
                }
                else {
                    $leaf = $unesc_head . $leaf;
                }

                # On Mac OS, the two glob metachars '*' and '?' and the escape
                # char '\' are valid characters for file and directory names.
                # We have to escape and treat them specially.
                push @matched, _escape_MacOS($leaf);
                next INNER_MACOS;
            }
        }
        if (@matched) {
            push @glob, @matched;
        }
    }
    return @glob;
}

#
# _expand_volume_MacOS() will only be used on Mac OS (OS9 or older):
# Takes an array of original patterns as argument and returns an array of
# possibly modified patterns. Each original pattern is processed like
# that:
# + If there's a volume name in the pattern, we push a separate pattern
#   for each mounted volume that matches (with '*', '?', and '\' escaped).
# + If there's no volume name in the original pattern, it is pushed
#   unchanged.
# Note that the returned array of patterns may be empty.
#
sub _expand_volume_MacOS {

    CORE::eval q{ CORE::require MacPerl; };
    croak "Can't require MacPerl;" if $@;

    my @volume_glob = @_;
    my @expand_volume = ();
    for my $volume_glob (@volume_glob) {

        # volume name in pattern
        if ($volume_glob =~ /\A ((?:[^\x81-\x9F\xE0-\xFC:]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])+:) (.*) \z/oxms) {
            my $pattern = _quotemeta_MacOS($1);
            my $tail = $2;

            for my $volume (map { MacPerl::MakePath($_) } MacPerl::Volumes()) {
                if ($volume =~ /\A $pattern \z/xmsi) {

                    # On Mac OS, the two glob metachars '*' and '?' and the
                    # escape char '\' are valid characters for volume names.
                    # We have to escape and treat them specially.
                    push @expand_volume, _escape_MacOS($volume) . $tail;
                }
            }
        }

        # no volume name in pattern
        else {
            push @expand_volume, $volume_glob;
        }
    }
    return @expand_volume;
}

#
# _canonpath_MacOS() will only be used on Mac OS (OS9 or older):
# Resolves any updirs in the pattern. Removes a single trailing colon
# from the pattern, unless it's a volume name pattern like "*HD:"
#
sub _canonpath_MacOS {
    my(@expr) = @_;

    for my $expr (@expr) {

        # resolve any updirs, e.g. "*HD:t?p::a*" -> "*HD:a*"
        1 while ($expr =~ s/\A ((?:[^\x81-\x9F\xE0-\xFC:]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])*?) : (?:[^\x81-\x9F\xE0-\xFC:]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])+ :: ((?:$q_char)*?) \z/$1:$2/oxms);

        # remove a single trailing colon, e.g. ":*:" -> ":*"
        $expr =~ s/ : ((?:[^\x81-\x9F\xE0-\xFC:]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])+) : \z/:$1/oxms;
    }
    return @expr;
}

#
# _escape_MacOS() will only be used on Mac OS (OS9 or older):
# Escape metachars '*', '?', and '\' of arguments.
#
sub _escape_MacOS {
    my($expr) = @_;

    # escape regex metachars but not '\' and glob chars '*', '?'
    my $escape = '';
    while ($expr =~ / \G ($q_char) /oxmsgc) {
        my $char = $1;
        if ($char =~ /\A [*?\\] \z/oxms) {
            $escape .= '\\' . $char;
        }
        else {
            $escape .= $char;
        }
    }
    return $escape;
}

#
# _unescape_MacOS() will only be used on Mac OS (OS9 or older):
# Unescapes a list of arguments which may contain escaped
# metachars '*', '?', and '\'.
#
sub _unescape_MacOS {
    my($expr) = @_;

    my $unescape = '';
    while ($expr =~ / \G (\\[*?\\] | $q_char) /oxmsgc) {
        my $char = $1;
        if ($char =~ /\A \\([*?\\]) \z/oxms) {
            $unescape .= $1;
        }
        else {
            $unescape .= $char;
        }
    }
    return $unescape;
}

#
# _hasmeta_MacOS() will only be used on Mac OS (OS9 or older):
#
sub _hasmeta_MacOS {
    my($expr) = @_;

    # if a '*' or '?' is preceded by an odd count of '\', temporary delete
    # it (and its preceding backslashes), i.e. don't treat '\*' and '\?' as
    # wildcards
    while ($expr =~ / \G (\\[*?\\] | $q_char) /oxgc) {
        my $char = $1;
        if ($char eq '*') {
            return 1;
        }
        elsif ($char eq '?') {
            return 1;
        }
    }
    return 0;
}

#
# _quotemeta_MacOS() will only be used on Mac OS (OS9 or older):
#
sub _quotemeta_MacOS {
    my($expr) = @_;

    # escape regex metachars but not '\' and glob chars '*', '?'
    my $quotemeta = '';
    while ($expr =~ / \G (\\[*?\\] | $q_char) /oxgc) {
        my $char = $1;
        if ($char =~ /\A \\[*?\\] \z/oxms) {
            $quotemeta .= $char;
        }
        elsif ($char eq '*') {
            $quotemeta .= "(?:$your_char)*",
        }
        elsif ($char eq '?') {
            $quotemeta .= "(?:$your_char)?",  # DOS style
#           $quotemeta .= "(?:$your_char)",   # UNIX style
        }
        elsif ((my $fc = Esjis::fc($char)) ne $char) {
            $quotemeta .= $fc;
        }
        else {
            $quotemeta .= quotemeta $char;
        }
    }
    return $quotemeta;
}

#endif
#
# via File::HomeDir::Windows 1.00
#
sub my_home_MSWin32 {

    # A lot of unix people and unix-derived tools rely on
    # the ability to overload HOME. We will support it too
    # so that they can replace raw HOME calls with File::HomeDir.
    if (exists $ENV{'HOME'} and $ENV{'HOME'}) {
        return $ENV{'HOME'};
    }

    # Do we have a user profile?
    elsif (exists $ENV{'USERPROFILE'} and $ENV{'USERPROFILE'}) {
        return $ENV{'USERPROFILE'};
    }

    # Some Windows use something like $ENV{'HOME'}
    elsif (exists $ENV{'HOMEDRIVE'} and exists $ENV{'HOMEPATH'} and $ENV{'HOMEDRIVE'} and $ENV{'HOMEPATH'}) {
        return join '', $ENV{'HOMEDRIVE'}, $ENV{'HOMEPATH'};
    }

    return undef;
}

#if MACPERL
#
# via File::HomeDir::MacOS9 1.00
#
sub my_home_MacOS {

    # Try for $ENV{'HOME'} if we have it
    if (defined $ENV{'HOME'}) {
        return $ENV{'HOME'};
    }

    ### DESPERATION SETS IN

    # We could use the desktop
    SCOPE: {
        local $@;
        CORE::eval {
            # Find the desktop via Mac::Files
            local $SIG{'__DIE__'} = '';
            CORE::require Mac::Files;
            my $home = Mac::Files::FindFolder(
                Mac::Files::kOnSystemDisk(),
                Mac::Files::kDesktopFolderType(),
            );
            return $home if $home and Esjis::d($home);
        };
    }

    # Desperation on any platform
    SCOPE: {
        # On some platforms getpwuid dies if called at all
        local $SIG{'__DIE__'} = '';
        my $home = CORE::eval q{ (getpwuid($<))[7] };
        return $home if $home and Esjis::d($home);
    }

    croak "Could not locate current user's home directory";
}

#endif
#
# via File::HomeDir::Unix 1.00
#
sub my_home {
    my $home;

    if (exists $ENV{'HOME'} and defined $ENV{'HOME'}) {
        $home = $ENV{'HOME'};
    }

    # This is from the original code, but I'm guessing
    # it means "login directory" and exists on some Unixes.
    elsif (exists $ENV{'LOGDIR'} and $ENV{'LOGDIR'}) {
        $home = $ENV{'LOGDIR'};
    }

    ### More-desperate methods

    # Light desperation on any (Unixish) platform
    else {
        $home = CORE::eval q{ (getpwuid($<))[7] };
    }

    # On Unix in general, a non-existant home means "no home"
    # For example, "nobody"-like users might use /nonexistant
    if (defined $home and ! Esjis::d($home)) {
        $home = undef;
    }
    return $home;
}

#if ESCAPE_SECOND_OCTET
#
# ShiftJIS file lstat (with parameter)
#
sub Esjis::lstat(*) {

    local $_ = shift if @_;

    if (-e $_) {
        return CORE::lstat _;
    }
    elsif (_MSWin32_5Cended_path($_)) {

        # Even if ${^WIN32_SLOPPY_STAT} is set to a true value, Esjis::lstat()
        # on Windows opens the file for the path which has 5c at end.
        # (and so on)

        local *MUST_BE_BAREWORD_AT_HERE;
        if (CORE::open(MUST_BE_BAREWORD_AT_HERE, $_)) {
            if (wantarray) {
                my @stat = CORE::stat MUST_BE_BAREWORD_AT_HERE; # not CORE::lstat
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return @stat;
            }
            else {
                my $stat = CORE::stat MUST_BE_BAREWORD_AT_HERE; # not CORE::lstat
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return $stat;
            }
        }
    }
    return wantarray ? () : undef;
}

#
# ShiftJIS file lstat (without parameter)
#
sub Esjis::lstat_() {

    if (-e $_) {
        return CORE::lstat _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        local *MUST_BE_BAREWORD_AT_HERE;
        if (CORE::open(MUST_BE_BAREWORD_AT_HERE, $_)) {
            if (wantarray) {
                my @stat = CORE::stat MUST_BE_BAREWORD_AT_HERE; # not CORE::lstat
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return @stat;
            }
            else {
                my $stat = CORE::stat MUST_BE_BAREWORD_AT_HERE; # not CORE::lstat
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return $stat;
            }
        }
    }
    return wantarray ? () : undef;
}

#
# ShiftJIS path opendir
#
sub Esjis::opendir(*$) {

    my $dh = qualify_to_ref $_[0];
    if (CORE::opendir $dh, $_[1]) {
        return 1;
    }
    elsif (_MSWin32_5Cended_path($_[1])) {
        if (CORE::opendir $dh, "$_[1]/.") {
            return 1;
        }
    }
    return undef;
}

#
# ShiftJIS file stat (with parameter)
#
sub Esjis::stat(*) {

    local $_ = shift if @_;

    my $fh = qualify_to_ref $_;
    if (defined fileno $fh) {
        return CORE::stat $fh;
    }
    elsif (-e $_) {
        return CORE::stat _;
    }
    elsif (_MSWin32_5Cended_path($_)) {

        # Even if ${^WIN32_SLOPPY_STAT} is set to a true value, Esjis::stat()
        # on Windows opens the file for the path which has 5c at end.
        # (and so on)

        local *MUST_BE_BAREWORD_AT_HERE;
        if (CORE::open(MUST_BE_BAREWORD_AT_HERE, $_)) {
            if (wantarray) {
                my @stat = CORE::stat MUST_BE_BAREWORD_AT_HERE;
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return @stat;
            }
            else {
                my $stat = CORE::stat MUST_BE_BAREWORD_AT_HERE;
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return $stat;
            }
        }
    }
    return wantarray ? () : undef;
}

#
# ShiftJIS file stat (without parameter)
#
sub Esjis::stat_() {

    my $fh = qualify_to_ref $_;
    if (defined fileno $fh) {
        return CORE::stat $fh;
    }
    elsif (-e $_) {
        return CORE::stat _;
    }
    elsif (_MSWin32_5Cended_path($_)) {
        local *MUST_BE_BAREWORD_AT_HERE;
        if (CORE::open(MUST_BE_BAREWORD_AT_HERE, $_)) {
            if (wantarray) {
                my @stat = CORE::stat MUST_BE_BAREWORD_AT_HERE;
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return @stat;
            }
            else {
                my $stat = CORE::stat MUST_BE_BAREWORD_AT_HERE;
                close(MUST_BE_BAREWORD_AT_HERE) or die "Can't close file: $_: $!";
                return $stat;
            }
        }
    }
    return wantarray ? () : undef;
}

#
# ShiftJIS path unlink
#
sub Esjis::unlink(@) {

    local @_ = ($_) unless @_;

    my $unlink = 0;
    for (@_) {
        if (CORE::unlink) {
            $unlink++;
        }
        elsif (Esjis::d($_)) {
        }
        elsif (_MSWin32_5Cended_path($_)) {
            my @char = /\G (?>$q_char) /oxmsg;
            my $file = join '', map {{'/' => '\\'}->{$_} || $_} @char;
            if ($file =~ / \A (?:$q_char)*? [ ] /oxms) {
                $file = qq{"$file"};
            }
            my $fh = gensym();
            if (_open_r($fh, $_)) {
                close($fh) or die "Can't close file: $_: $!";

                # cmd.exe on Windows NT, Windows 2000, Windows XP, Windows Vista, Windows 7, Windows 8, Windows 8.1, Windows 10 or later
                if ((defined $ENV{'OS'}) and ($ENV{'OS'} eq 'Windows_NT')) {
                    CORE::system 'DEL', '/F', $file, '2>NUL';
                }

                # Win95Cmd.exe on any Windows (when SET PERL5SHELL=Win95Cmd.exe /c, `var` returns "Windows 2000")
                elsif (qx{ver} =~ /\b(?:Windows 2000)\b/oms) {
                    CORE::system 'DEL', '/F', $file, '2>NUL';
                }

                # COMMAND.COM on Windows 95, Windows 98, Windows 98 Second Edition, Windows Millennium Edition
                # command.com can not "2>NUL"
                else {
                    CORE::system 'ATTRIB', '-R', $file; # clears Read-only file attribute
                    CORE::system 'DEL',          $file;
                }

                if (_open_r($fh, $_)) {
                    close($fh) or die "Can't close file: $_: $!";
                }
                else {
                    $unlink++;
                }
            }
        }
    }
    return $unlink;
}

#
# ShiftJIS chdir
#
sub Esjis::chdir(;$) {

    if (@_ == 0) {
        return CORE::chdir;
    }

    my($dir) = @_;

    if (_MSWin32_5Cended_path($dir)) {
        if (not Esjis::d $dir) {
            return 0;
        }

        if ($] =~ /^5\.005/oxms) {
            return CORE::chdir $dir;
        }
        elsif (($] =~ /^(?:5\.006|5\.008000)/oxms) and ($^O eq 'MSWin32')) {
            local $@;
            my $chdir = CORE::eval q{
                CORE::require 'jacode.pl';

                # P.676 ${^WIDE_SYSTEM_CALLS}
                # in Chapter 28: Special Names
                # of ISBN 0-596-00027-8 Programming Perl Third Edition.

                # P.790 ${^WIDE_SYSTEM_CALLS}
                # in Chapter 25: Special Names
                # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

                local ${^WIDE_SYSTEM_CALLS} = 1;
                return CORE::chdir jcode::utf8($dir,'sjis');
            };
            if (not $@) {
                return $chdir;
            }
        }

        # old idea (Win32 module required)
        elsif (0) {
            local $@;
            my $shortdir = '';
            my $chdir = CORE::eval q{
                use Win32;
                $shortdir = Win32::GetShortPathName($dir);
                if ($shortdir ne $dir) {
                    return CORE::chdir $shortdir;
                }
                else {
                    return 0;
                }
            };
            if ($@) {
                my @char = $dir =~ /\G (?>$q_char) /oxmsg;
                while ($char[-1] eq "\x5C") {
                    pop @char;
                }
                $dir = join '', @char;
                croak "Perl$] can't chdir to $dir (chr(0x5C) ended path), Win32.pm module may help you";
            }
            elsif ($shortdir eq $dir) {
                my @char = $dir =~ /\G (?>$q_char) /oxmsg;
                while ($char[-1] eq "\x5C") {
                    pop @char;
                }
                $dir = join '', @char;
                croak "Perl$] can't chdir to $dir (chr(0x5C) ended path)";
            }
            return $chdir;
        }

        # rejected idea ...
        elsif (0) {

            # MSDN SetCurrentDirectory function
            # http://msdn.microsoft.com/ja-jp/library/windows/desktop/aa365530(v=vs.85).aspx
            #
            # Data Execution Prevention (DEP)
            # http://vlaurie.com/computers2/Articles/dep.htm
            #
            # Learning x86 assembler with Perl -- Shibuya.pm#11
            # http://developer.cybozu.co.jp/takesako/2009/06/perl-x86-shibuy.html
            #
            # Introduction to Win32::API programming in Perl
            # http://d.hatena.ne.jp/TAKESAKO/20090324/1237879559
            #
            # DynaLoader - Dynamically load C libraries into Perl code
            # http://perldoc.perl.org/DynaLoader.html
            #
            # Basic knowledge of DynaLoader
            # http://blog.64p.org/entry/20090313/1236934042

            if (($] =~ /^5\.006/oxms)                     and
                ($^O eq 'MSWin32')                        and
                ($ENV{'PROCESSOR_ARCHITECTURE'} eq 'x86') and
                CORE::eval(q{CORE::require 'Dyna'.'Loader'})
            ) {
                my $x86 = join('',

                    # PUSH Iv
                    "\x68", pack('P', "$dir\\\0"),

                    # MOV eAX, Iv
                    "\xb8", pack('L',
                        *{'Dyna'.'Loader::dl_find_symbol'}{'CODE'}->(
                            *{'Dyna'.'Loader::dl_load_file'}{'CODE'}->("$ENV{'SystemRoot'}\\system32\\kernel32.dll"),
                            'SetCurrentDirectoryA'
                        )
                    ),

                    # CALL eAX
                    "\xff\xd0",

                    # RETN
                    "\xc3",
                );
                *{'Dyna'.'Loader::dl_install_xsub'}{'CODE'}->('_SetCurrentDirectoryA', unpack('L', pack 'P', $x86));
                _SetCurrentDirectoryA();
                chomp(my $chdir = qx{chdir});
                if (Esjis::fc($chdir) eq Esjis::fc($dir)) {
                    return 1;
                }
                else {
                    return 0;
                }
            }
        }

# COMMAND.COM's unhelpful tips:
# Displays a list of files and subdirectories in a directory.
# http://www.lagmonster.org/docs/DOS7/z-dir.html
#
# Syntax:
#
#   DIR [drive:] [path] [filename] [/Switches]
#
#   /Z Long file names are not displayed in the file listing
#
#  Limitations
#   The undocumented /Z switch (no long names) would appear to
#   have been not fully developed and has a couple of problems:
#
#  1. It will only work if:
#   There is no path specified (ie. for the current directory in
#   the current drive)
#   The path is specified as the root directory of any drive
#   (eg. C:\, D:\, etc.)
#   The path is specified as the current directory of any drive
#   by using the drive letter only (eg. C:, D:, etc.)
#   The path is specified as the parent directory using the ..
#   notation (eg. DIR .. /Z)
#   Any other syntax results in a "File Not Found" error message.
#
#  2. The /Z switch is compatable with the /S switch to show
#   subdirectories (as long as the above rules are followed) and
#   all the files are shown with short names only. The
#   subdirectories are also shown with short names only. However,
#   the header for each subdirectory after the first level gives
#   the subdirectory's long name.
#
#  3. The /Z switch is also compatable with the /B switch to give
#   a simple list of files with short names only. When used with
#   the /S switch as well, all files are listed with their full
#   paths. The file names themselves are all in short form, and
#   the path of those files in the current directory are in short
#   form, but the paths of any files in subdirectories are in
#   long filename form.

        my $shortdir = '';
        my $i = 0;
        my @subdir = ();
        while ($dir =~ / \G ($q_char) /oxgc) {
            my $char = $1;
            if (($char eq '\\') or ($char eq '/')) {
                $i++;
                $subdir[$i] = $char;
                $i++;
            }
            else {
                $subdir[$i] .= $char;
            }
        }
        if (($subdir[-1] eq '\\') or ($subdir[-1] eq '/')) {
            pop @subdir;
        }

        # P.504 PERL5SHELL (Microsoft ports only)
        # in Chapter 19: The Command-Line Interface
        # of ISBN 0-596-00027-8 Programming Perl Third Edition.

        # P.597 PERL5SHELL (Microsoft ports only)
        # in Chapter 17: The Command-Line Interface
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # Win95Cmd.exe on any Windows (when SET PERL5SHELL=Win95Cmd.exe /c, `var` returns "Windows 2000")
        # cmd.exe on Windows NT, Windows 2000
        if (qx{ver} =~ /\b(?:Windows NT|Windows 2000)\b/oms) {
            chomp(my @dirx = grep /<DIR>/oxms, qx{dir /ad /x "$dir*" 2>NUL});
            for my $dirx (sort { CORE::length($a) <=> CORE::length($b) } @dirx) {
                if (Esjis::fc(CORE::substr $dirx,-CORE::length($subdir[-1]),CORE::length($subdir[-1])) eq Esjis::fc($subdir[-1])) {

                    # short file name (8dot3name) here-----vv
                    my $shortleafdir = CORE::substr $dirx, 39, 8+1+3;
                    $shortleafdir =~ s/ [ ]+ \z//oxms;
                    $shortdir = join '', @subdir[0..$#subdir-1], $shortleafdir;
                    last;
                }
            }
        }

        # an idea (not so portable, only Windows 2000 or later)
        elsif (0) {
            chomp($shortdir = qx{for %I in ("$dir") do \@echo %~sI 2>NUL});
        }

        # cmd.exe on Windows XP, Windows Vista, Windows 7, Windows 8, Windows 8.1, Windows 10 or later
        elsif ((defined $ENV{'OS'}) and ($ENV{'OS'} eq 'Windows_NT')) {
            chomp(my @dirx = grep /<DIR>/oxms, qx{dir /ad /x "$dir*" 2>NUL});
            for my $dirx (sort { CORE::length($a) <=> CORE::length($b) } @dirx) {
                if (Esjis::fc(CORE::substr $dirx,-CORE::length($subdir[-1]),CORE::length($subdir[-1])) eq Esjis::fc($subdir[-1])) {

                    # short file name (8dot3name) here-----vv
                    my $shortleafdir = CORE::substr $dirx, 36, 8+1+3;
                    $shortleafdir =~ s/ [ ]+ \z//oxms;
                    $shortdir = join '', @subdir[0..$#subdir-1], $shortleafdir;
                    last;
                }
            }
        }

        # COMMAND.COM on Windows 95, Windows 98, Windows 98 Second Edition, Windows Millennium Edition
        else {
            chomp(my @dirx = grep /<DIR>/oxms, qx{dir /ad "$dir*"});
            for my $dirx (sort { CORE::length($a) <=> CORE::length($b) } @dirx) {
                if (Esjis::fc(CORE::substr $dirx,-CORE::length($subdir[-1]),CORE::length($subdir[-1])) eq Esjis::fc($subdir[-1])) {

                    # short file name (8dot3name) here-----v
                    my $shortleafdir = CORE::substr $dirx, 0, 8+1+3;
                    CORE::substr($shortleafdir,8,1) = '.';
                    $shortleafdir =~ s/ \. [ ]+ \z//oxms;
                    $shortdir = join '', @subdir[0..$#subdir-1], $shortleafdir;
                    last;
                }
            }
        }

        if ($shortdir eq '') {
            return 0;
        }
        elsif (Esjis::fc($shortdir) eq Esjis::fc($dir)) {
            return 0;
        }
        return CORE::chdir $shortdir;
    }
    else {
        return CORE::chdir $dir;
    }
}

#
# ShiftJIS chr(0x5C) ended path on MSWin32
#
sub _MSWin32_5Cended_path {

    if ((@_ >= 1) and ($_[0] ne '')) {
        if ($^O =~ /\A (?: MSWin32 | NetWare | symbian | dos ) \z/oxms) {
            my @char = $_[0] =~ /\G (?>$q_char) /oxmsg;
            if ($char[-1] =~ / \x5C \z/oxms) {
                return 1;
            }
        }
    }
    return undef;
}

#
# do ShiftJIS file
#
sub Esjis::do($) {

    my($filename) = @_;

    my $realfilename;
    my $result;
ITER_DO:
    {
        for my $prefix (@INC) {
            if ($^O eq 'MacOS') {
                $realfilename = "$prefix$filename";
            }
            else {
                $realfilename = "$prefix/$filename";
            }

            if (Esjis::f($realfilename)) {

                my $script = '';

                if (Esjis::e("$realfilename.e")) {
                    my $e_mtime      = (Esjis::stat("$realfilename.e"))[9];
                    my $mtime        = (Esjis::stat($realfilename))[9];
                    my $module_mtime = (Esjis::stat(__FILE__))[9];
                    if (($e_mtime < $mtime) or ($mtime < $module_mtime)) {
                        Esjis::unlink "$realfilename.e";
                    }
                }

                if (Esjis::e("$realfilename.e")) {
                    my $fh = gensym();
                    if (_open_r($fh, "$realfilename.e")) {
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpSetFLock("$realfilename.e");
                            };
                        }
                        elsif (exists $ENV{'CHAR_NONBLOCK'}) {

                            # P.419 File Locking
                            # in Chapter 16: Interprocess Communication
                            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

                            # P.524 File Locking
                            # in Chapter 15: Interprocess Communication
                            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

                            # (and so on)

                            CORE::eval q{ flock($fh, LOCK_SH | LOCK_NB) };
                            if ($@) {
                                carp "Can't immediately read-lock the file: $realfilename.e";
                            }
                        }
                        else {
                            CORE::eval q{ flock($fh, LOCK_SH) };
                        }
                        local $/ = undef; # slurp mode
                        $script = <$fh>;
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpRstFLock("$realfilename.e");
                            };
                        }
                        close($fh) or die "Can't close file: $realfilename.e: $!";
                    }
                }
                else {
                    my $fh = gensym();
                    if (_open_r($fh, $realfilename)) {
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpSetFLock($realfilename);
                            };
                        }
                        elsif (exists $ENV{'CHAR_NONBLOCK'}) {
                            CORE::eval q{ flock($fh, LOCK_SH | LOCK_NB) };
                            if ($@) {
                                carp "Can't immediately read-lock the file: $realfilename";
                            }
                        }
                        else {
                            CORE::eval q{ flock($fh, LOCK_SH) };
                        }
                        local $/ = undef; # slurp mode
                        $script = <$fh>;
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpRstFLock($realfilename);
                            };
                        }
                        close($fh) or die "Can't close file: $realfilename.e: $!";
                    }

                    if ($script =~ /^ (?>\s*) use (?>\s+) Sjis (?>\s*) ([^\x81-\x9F\xE0-\xFC;]*) ; (?>\s*) \n? $/oxms) {
                        CORE::require Sjis;
                        $script = Sjis::escape_script($script);
                        my $fh = gensym();
                        open($fh, ">$realfilename.e") or die __FILE__, ": Can't write open file: $realfilename.e\n";
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpSetFLock("$realfilename.e");
                            };
                        }
                        elsif (exists $ENV{'CHAR_NONBLOCK'}) {
                            CORE::eval q{ flock($fh, LOCK_EX | LOCK_NB) };
                            if ($@) {
                                carp "Can't immediately write-lock the file: $realfilename.e";
                            }
                        }
                        else {
                            CORE::eval q{ flock($fh, LOCK_EX) };
                        }
                        CORE::eval q{ truncate($fh, 0) };
                        seek($fh, 0, 0) or croak "Can't seek file: $realfilename.e";
                        print {$fh} $script;
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpRstFLock("$realfilename.e");
                            };
                        }
                        close($fh) or die "Can't close file: $realfilename.e: $!";
                    }
                }

                {
                    no strict;
                    $result = scalar CORE::eval $script;
                }
                last ITER_DO;
            }
        }
    }

    if ($@) {
        $INC{$filename} = undef;
        return undef;
    }
    elsif (not $result) {
        return undef;
    }
    else {
        $INC{$filename} = $realfilename;
        return $result;
    }
}

#
# require ShiftJIS file
#

# require
# in Chapter 3: Functions
# of ISBN 1-56592-149-6 Programming Perl, Second Edition.
#
# sub require {
#     my($filename) = @_;
#     return 1 if $INC{$filename};
#     my($realfilename, $result);
#     ITER: {
#         foreach $prefix (@INC) {
#             $realfilename = "$prefix/$filename";
#             if (-f $realfilename) {
#                 $result = CORE::eval `cat $realfilename`;
#                 last ITER;
#             }
#         }
#         die "Can't find $filename in \@INC";
#     }
#     die $@ if $@;
#     die "$filename did not return true value" unless $result;
#     $INC{$filename} = $realfilename;
#     return $result;
# }

# require
# in Chapter 9: perlfunc: Perl builtin functions
# of ISBN-13: 978-1-906966-02-7 The Perl Language Reference Manual (for Perl version 5.12.1)
#
# sub require {
#     my($filename) = @_;
#     if (exists $INC{$filename}) {
#         return 1 if $INC{$filename};
#         die "Compilation failed in require";
#     }
#     my($realfilename, $result);
#     ITER: {
#         foreach $prefix (@INC) {
#             $realfilename = "$prefix/$filename";
#             if (-f $realfilename) {
#                 $INC{$filename} = $realfilename;
#                 $result = do $realfilename;
#                 last ITER;
#             }
#         }
#         die "Can't find $filename in \@INC";
#     }
#     if ($@) {
#         $INC{$filename} = undef;
#         die $@;
#     }
#     elsif (!$result) {
#         delete $INC{$filename};
#         die "$filename did not return true value";
#     }
#     else {
#         return $result;
#     }
# }

sub Esjis::require(;$) {

    local $_ = shift if @_;

    if (exists $INC{$_}) {
        return 1 if $INC{$_};
        croak "Compilation failed in require: $_";
    }

    # jcode.pl
    # ftp://ftp.iij.ad.jp/pub/IIJ/dist/utashiro/perl/

    # jacode.pl
    # http://search.cpan.org/dist/jacode/

    if (/ \b (?: jcode\.pl | jacode(?>[0-9]*)\.pl ) \z /oxms) {
        return CORE::require($_);
    }

    my $realfilename;
    my $result;
ITER_REQUIRE:
    {
        for my $prefix (@INC) {
            if ($^O eq 'MacOS') {
                $realfilename = "$prefix$_";
            }
            else {
                $realfilename = "$prefix/$_";
            }

            if (Esjis::f($realfilename)) {
                $INC{$_} = $realfilename;

                my $script = '';

                if (Esjis::e("$realfilename.e")) {
                    my $e_mtime      = (Esjis::stat("$realfilename.e"))[9];
                    my $mtime        = (Esjis::stat($realfilename))[9];
                    my $module_mtime = (Esjis::stat(__FILE__))[9];
                    if (($e_mtime < $mtime) or ($mtime < $module_mtime)) {
                        Esjis::unlink "$realfilename.e";
                    }
                }

                if (Esjis::e("$realfilename.e")) {
                    my $fh = gensym();
                    _open_r($fh, "$realfilename.e") or croak "Can't open file: $realfilename.e";
                    if ($^O eq 'MacOS') {
                        CORE::eval q{
                            CORE::require Mac::Files;
                            Mac::Files::FSpSetFLock("$realfilename.e");
                        };
                    }
                    elsif (exists $ENV{'CHAR_NONBLOCK'}) {
                        CORE::eval q{ flock($fh, LOCK_SH | LOCK_NB) };
                        if ($@) {
                            carp "Can't immediately read-lock the file: $realfilename.e";
                        }
                    }
                    else {
                        CORE::eval q{ flock($fh, LOCK_SH) };
                    }
                    local $/ = undef; # slurp mode
                    $script = <$fh>;
                    if ($^O eq 'MacOS') {
                        CORE::eval q{
                            CORE::require Mac::Files;
                            Mac::Files::FSpRstFLock("$realfilename.e");
                        };
                    }
                    close($fh) or croak "Can't close file: $realfilename: $!";
                }
                else {
                    my $fh = gensym();
                    _open_r($fh, $realfilename) or croak "Can't open file: $realfilename";
                    if ($^O eq 'MacOS') {
                        CORE::eval q{
                            CORE::require Mac::Files;
                            Mac::Files::FSpSetFLock($realfilename);
                        };
                    }
                    elsif (exists $ENV{'CHAR_NONBLOCK'}) {
                        CORE::eval q{ flock($fh, LOCK_SH | LOCK_NB) };
                        if ($@) {
                            carp "Can't immediately read-lock the file: $realfilename";
                        }
                    }
                    else {
                        CORE::eval q{ flock($fh, LOCK_SH) };
                    }
                    local $/ = undef; # slurp mode
                    $script = <$fh>;
                    if ($^O eq 'MacOS') {
                        CORE::eval q{
                            CORE::require Mac::Files;
                            Mac::Files::FSpRstFLock($realfilename);
                        };
                    }
                    close($fh) or croak "Can't close file: $realfilename: $!";

                    if ($script =~ /^ (?>\s*) use (?>\s+) Sjis (?>\s*) ([^\x81-\x9F\xE0-\xFC;]*) ; (?>\s*) \n? $/oxms) {
                        CORE::require Sjis;
                        $script = Sjis::escape_script($script);
                        my $fh = gensym();
                        open($fh, ">$realfilename.e") or croak "Can't write open file: $realfilename.e";
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpSetFLock("$realfilename.e");
                            };
                        }
                        elsif (exists $ENV{'CHAR_NONBLOCK'}) {
                            CORE::eval q{ flock($fh, LOCK_EX | LOCK_NB) };
                            if ($@) {
                                carp "Can't immediately write-lock the file: $realfilename.e";
                            }
                        }
                        else {
                            CORE::eval q{ flock($fh, LOCK_EX) };
                        }
                        CORE::eval q{ truncate($fh, 0) };
                        seek($fh, 0, 0) or croak "Can't seek file: $realfilename.e";
                        print {$fh} $script;
                        if ($^O eq 'MacOS') {
                            CORE::eval q{
                                CORE::require Mac::Files;
                                Mac::Files::FSpRstFLock("$realfilename.e");
                            };
                        }
                        close($fh) or croak "Can't close file: $realfilename: $!";
                    }
                }

                {
                    no strict;
                    $result = scalar CORE::eval $script;
                }
                last ITER_REQUIRE;
            }
        }
        croak "Can't find $_ in \@INC";
    }

    if ($@) {
        $INC{$_} = undef;
        croak $@;
    }
    elsif (not $result) {
        delete $INC{$_};
        croak "$_ did not return true value";
    }
    else {
        return $result;
    }
}

#
# ShiftJIS telldir avoid warning
#
sub Esjis::telldir(*) {

    local $^W = 0;

    return CORE::telldir $_[0];
}

#endif /* #if ESCAPE_SECOND_OCTET */
#
# ${^PREMATCH}, $PREMATCH, $` the string preceding what was matched
#
sub Esjis::PREMATCH {
#if MULTIBYTE_ANCHORING
    if (defined($&)) {
        if (defined($1) and (CORE::substr($&,-CORE::length($1),CORE::length($1)) eq $1)) {
            return CORE::substr($&,0,CORE::length($&)-CORE::length($1));
        }
        else {
            croak 'Use of "$`", $PREMATCH, and ${^PREMATCH} need to /( capture all )/ in regexp';
        }
    }
    else {
        return '';
    }
#endif
    return $`;
}

#
# ${^MATCH}, $MATCH, $& the string that matched
#
sub Esjis::MATCH {
#if MULTIBYTE_ANCHORING
    if (defined($&)) {
        if (defined($1)) {
            return $1;
        }
        else {
            croak 'Use of "$&", $MATCH, and ${^MATCH} need to /( capture all )/ in regexp';
        }
    }
    else {
        return '';
    }
#endif
    return $&;
}

#
# ${^POSTMATCH}, $POSTMATCH, $' the string following what was matched
#
sub Esjis::POSTMATCH {
    return $';
}

#
# ShiftJIS character to order (with parameter)
#
sub Sjis::ord(;$) {

    local $_ = shift if @_;

    if (/\A ($q_char) /oxms) {
        my @ord = unpack 'C*', $1;
        my $ord = 0;
        while (my $o = shift @ord) {
            $ord = $ord * 0x100 + $o;
        }
        return $ord;
    }
    else {
        return CORE::ord $_;
    }
}

#
# ShiftJIS character to order (without parameter)
#
sub Sjis::ord_() {

    if (/\A ($q_char) /oxms) {
        my @ord = unpack 'C*', $1;
        my $ord = 0;
        while (my $o = shift @ord) {
            $ord = $ord * 0x100 + $o;
        }
        return $ord;
    }
    else {
        return CORE::ord $_;
    }
}

#
# ShiftJIS reverse
#
sub Sjis::reverse(@) {

    if (wantarray) {
        return CORE::reverse @_;
    }
    else {

        # One of us once cornered Larry in an elevator and asked him what
        # problem he was solving with this, but he looked as far off into
        # the distance as he could in an elevator and said, "It seemed like
        # a good idea at the time."

        return join '', CORE::reverse(join('',@_) =~ /\G ($q_char) /oxmsg);
    }
}

#
# ShiftJIS getc (with parameter, without parameter)
#
sub Sjis::getc(;*@) {

    my($package) = caller;
    my $fh = @_ ? qualify_to_ref(shift,$package) : \*STDIN;
    croak 'Too many arguments for Sjis::getc' if @_ and not wantarray;

    my @length = sort { $a <=> $b } keys %range_tr;
    my $getc = '';
    for my $length ($length[0] .. $length[-1]) {
        $getc .= CORE::getc($fh);
        if (exists $range_tr{CORE::length($getc)}) {
            if ($getc =~ /\A ${Esjis::dot_s} \z/oxms) {
                return wantarray ? ($getc,@_) : $getc;
            }
        }
    }
    return wantarray ? ($getc,@_) : $getc;
}

#
# ShiftJIS length by character
#
sub Sjis::length(;$) {

    local $_ = shift if @_;

    local @_ = /\G ($q_char) /oxmsg;
    return scalar @_;
}

#
# ShiftJIS substr by character
#
BEGIN {

    # P.232 The lvalue Attribute
    # in Chapter 6: Subroutines
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.336 The lvalue Attribute
    # in Chapter 7: Subroutines
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # P.144 8.4 Lvalue subroutines
    # in Chapter 8: perlsub: Perl subroutines
    # of ISBN-13: 978-1-906966-02-7 The Perl Language Reference Manual (for Perl version 5.12.1)

    CORE::eval sprintf(<<'END', ($] >= 5.014000) ? ':lvalue' : '');
    #                       vv----------------------*******
    sub Sjis::substr($$;$$) %s {

        my @char = $_[0] =~ /\G (?>$q_char) /oxmsg;

        # If the substring is beyond either end of the string, substr() returns the undefined
        # value and produces a warning. When used as an lvalue, specifying a substring that
        # is entirely outside the string raises an exception.
        # http://perldoc.perl.org/functions/substr.html

        # A return with no argument returns the scalar value undef in scalar context,
        # an empty list () in list context, and (naturally) nothing at all in void
        # context.

        my $offset = $_[1];
        if (($offset > scalar(@char)) or ($offset < (-1 * scalar(@char)))) {
            return;
        }

        # substr($string,$offset,$length,$replacement)
        if (@_ == 4) {
            my(undef,undef,$length,$replacement) = @_;
            my $substr = join '', splice(@char, $offset, $length, $replacement);
            $_[0] = join '', @char;

            # return $substr; this doesn't work, don't say "return"
            $substr;
        }

        # substr($string,$offset,$length)
        elsif (@_ == 3) {
            my(undef,undef,$length) = @_;
            my $octet_offset = 0;
            my $octet_length = 0;
            if ($offset == 0) {
                $octet_offset = 0;
            }
            elsif ($offset > 0) {
                local $SIG{__WARN__} = sub {}; # avoid: Use of uninitialized value in join or string at here
                $octet_offset =      CORE::length(join '', @char[0..$offset-1]);
            }
            else {
                local $SIG{__WARN__} = sub {}; # avoid: Use of uninitialized value in join or string at here
                $octet_offset = -1 * CORE::length(join '', @char[$#char+$offset+1..$#char]);
            }
            if ($length == 0) {
                $octet_length = 0;
            }
            elsif ($length > 0) {
                local $SIG{__WARN__} = sub {}; # avoid: Use of uninitialized value in join or string at here
                $octet_length =      CORE::length(join '', @char[$offset..$offset+$length-1]);
            }
            else {
                local $SIG{__WARN__} = sub {}; # avoid: Use of uninitialized value in join or string at here
                $octet_length = -1 * CORE::length(join '', @char[$#char+$length+1..$#char]);
            }
            CORE::substr($_[0], $octet_offset, $octet_length);
        }

        # substr($string,$offset)
        else {
            my $octet_offset = 0;
            if ($offset == 0) {
                $octet_offset = 0;
            }
            elsif ($offset > 0) {
                $octet_offset =      CORE::length(join '', @char[0..$offset-1]);
            }
            else {
                $octet_offset = -1 * CORE::length(join '', @char[$#char+$offset+1..$#char]);
            }
            CORE::substr($_[0], $octet_offset);
        }
    }
END
}

#
# ShiftJIS index by character
#
sub Sjis::index($$;$) {

    my $index;
    if (@_ == 3) {
        $index = Esjis::index($_[0], $_[1], CORE::length(Sjis::substr($_[0], 0, $_[2])));
    }
    else {
        $index = Esjis::index($_[0], $_[1]);
    }

    if ($index == -1) {
        return -1;
    }
    else {
        return Sjis::length(CORE::substr $_[0], 0, $index);
    }
}

#
# ShiftJIS rindex by character
#
sub Sjis::rindex($$;$) {

    my $rindex;
    if (@_ == 3) {
        $rindex = Esjis::rindex($_[0], $_[1], CORE::length(Sjis::substr($_[0], 0, $_[2])));
    }
    else {
        $rindex = Esjis::rindex($_[0], $_[1]);
    }

    if ($rindex == -1) {
        return -1;
    }
    else {
        return Sjis::length(CORE::substr $_[0], 0, $rindex);
    }
}

# when 'm//', '/' means regexp match 'm//' and '?' means regexp match '??'
# when 'div', '/' means division operator and '?' means conditional operator (condition ? then : else)
use vars qw($slash); $slash = 'm//';

# ord() to ord() or Sjis::ord()
my $function_ord = 'ord';

# ord to ord or Sjis::ord_
my $function_ord_ = 'ord';

# reverse to reverse or Sjis::reverse
my $function_reverse = 'reverse';

# getc to getc or Sjis::getc
my $function_getc = 'getc';

# P.1023 Appendix W.9 Multibyte Anchoring
# of ISBN 1-56592-224-7 CJKV Information Processing

my $anchor = '';
#if MULTIBYTE_ANCHORING
$anchor = q{${Esjis::anchor}};
#endif

use vars qw($nest);

# regexp of nested parens in qqXX

# P.340 Matching Nested Constructs with Embedded Code
# in Chapter 7: Perl
# of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

my $qq_paren   = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\\()] |
                           \(  (?{$nest++}) |
                           \)  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                    \\ [^\x81-\x9F\xE0-\xFCc] |
                    \\c[\x40-\x5F] |
#if MULTIBYTE_ENCODING
                    \\ [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                       [\x00-\xFF]
                 }xms;

my $qq_brace   = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\\{}] |
                           \{  (?{$nest++}) |
                           \}  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                    \\ [^\x81-\x9F\xE0-\xFCc] |
                    \\c[\x40-\x5F] |
#if MULTIBYTE_ENCODING
                    \\ [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                       [\x00-\xFF]
                 }xms;

my $qq_bracket = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\\\[\]] |
                           \[  (?{$nest++}) |
                           \]  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                    \\ [^\x81-\x9F\xE0-\xFCc] |
                    \\c[\x40-\x5F] |
#if MULTIBYTE_ENCODING
                    \\ [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                       [\x00-\xFF]
                 }xms;

my $qq_angle   = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\\<>] |
                           \<  (?{$nest++}) |
                           \>  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                    \\ [^\x81-\x9F\xE0-\xFCc] |
                    \\c[\x40-\x5F] |
#if MULTIBYTE_ENCODING
                    \\ [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                       [\x00-\xFF]
                 }xms;

my $qq_scalar  = qr{(?: \{ (?:$qq_brace)*? \} |
                       (?: ::)? (?:
                             (?> [a-zA-Z_][a-zA-Z_0-9]* (?: ::[a-zA-Z_][a-zA-Z_0-9]*)* )
                                                    (?>(?:                                   \[ (?: \$\[ | \$\] | $qq_char )*? \] |           \{ (?:$qq_brace)*? \} )*)
                                      (?>(?: (?: -> )? (?: [\$\@\%\&\*]\* | \$\#\* | [\@\%]? \[ (?: \$\[ | \$\] | $qq_char )*? \] | [\@\%\*]? \{ (?:$qq_brace)*? \} ) )*)
                   ))
                 }xms;

my $qq_variable = qr{(?: \{ (?:$qq_brace)*? \}                    |
                        (?: ::)? (?:
                              (?>[0-9]+)                          |
                              [^\x81-\x9F\xE0-\xFCa-zA-Z_0-9\[\]] |
                              ^[A-Z]                              |
                              (?> [a-zA-Z_][a-zA-Z_0-9]* (?: ::[a-zA-Z_][a-zA-Z_0-9]*)* )
                                                     (?>(?:                                   \[ (?: \$\[ | \$\] | $qq_char )*? \] |           \{ (?:$qq_brace)*? \} )*)
                                       (?>(?: (?: -> )? (?: [\$\@\%\&\*]\* | \$\#\* | [\@\%]? \[ (?: \$\[ | \$\] | $qq_char )*? \] | [\@\%\*]? \{ (?:$qq_brace)*? \} ) )*)
                    ))
                  }xms;

my $qq_substr  = qr{(?> Char::substr | Sjis::substr | CORE::substr | substr ) (?>\s*) \( $qq_paren \)
                 }xms;

# regexp of nested parens in qXX
my $q_paren    = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC()] |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                             \(  (?{$nest++}) |
                             \)  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
                       [\x00-\xFF]
                 }xms;

my $q_brace    = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\{\}] |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                             \{  (?{$nest++}) |
                             \}  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
                       [\x00-\xFF]
                 }xms;

my $q_bracket  = qr{(?{local $nest=0}) (?>(?:
                       [^\x81-\x9F\xE0-\xFC\[\]] |
#if MULTIBYTE_ENCODING
                       [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                             \[  (?{$nest++}) |
                             \]  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
                    [\x00-\xFF]
                 }xms;

my $q_angle    = qr{(?{local $nest=0}) (?>(?:
                    [^\x81-\x9F\xE0-\xFC<>] |
#if MULTIBYTE_ENCODING
                    [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
#endif
                             \<  (?{$nest++}) |
                             \>  (?(?{$nest>0})(?{$nest--})|(?!)))*) (?(?{$nest!=0})(?!)) |
                    [\x00-\xFF]
                 }xms;

my $matched     = '';
my $s_matched   = '';
#if MULTIBYTE_ANCHORING
$matched        = q{$Esjis::matched};
$s_matched      = q{ Esjis::s_matched();};
#endif

my $tr_variable   = '';   # variable of tr///
my $sub_variable  = '';   # variable of s///
my $bind_operator = '';   # =~ or !~

my @heredoc = ();         # here document
my @heredoc_delimiter = ();
my $here_script = '';     # here script

#
# escape ShiftJIS script
#
sub Sjis::escape(;$) {
    local($_) = $_[0] if @_;

    # P.359 The Study Function
    # in Chapter 7: Perl
    # of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

    study $_; # Yes, I studied study yesterday.

    # while all script

    # 6.14. Matching from Where the Last Pattern Left Off
    # in Chapter 6. Pattern Matching
    # of ISBN 0-596-00313-7 Perl Cookbook, 2nd Edition.
    # (and so on)

    # one member of Tag-team
    #
    # P.128 Start of match (or end of previous match): \G
    # P.130 Advanced Use of \G with Perl
    # in Chapter 3: Overview of Regular Expression Features and Flavors
    # P.255 Use leading anchors
    # P.256 Expose ^ and \G at the front expressions
    # in Chapter 6: Crafting an Efficient Expression
    # P.315 "Tag-team" matching with /gc
    # in Chapter 7: Perl
    # of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

    my $e_script = '';
    while (not /\G \z/oxgc) { # member
        $e_script .= Sjis::escape_token();
    }

    return $e_script;
}

#
# escape ShiftJIS token of script
#
sub Sjis::escape_token {

# \n output here document

    my $ignore_modules = join('|', qw(
        utf8
        bytes
        charnames
        I18N::Japanese
        I18N::Collate
        I18N::JExt
        File::DosGlob
        Wild
        Wildcard
        Japanese
    ));

    # another member of Tag-team
    #
    # P.315 "Tag-team" matching with /gc
    # in Chapter 7: Perl
    # of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

    if (/\G ( \n ) /oxgc) { # another member (and so on)
        my $heredoc = '';
        if (scalar(@heredoc_delimiter) >= 1) {
            $slash = 'm//';

            $heredoc = join '', @heredoc;
            @heredoc = ();

            # skip here document
            for my $heredoc_delimiter (@heredoc_delimiter) {
                /\G .*? \n $heredoc_delimiter \n/xmsgc;
            }
            @heredoc_delimiter = ();

            $here_script = '';
        }
        return "\n" . $heredoc;
    }

# ignore space, comment
    elsif (/\G ((?>\s+)|\#.*) /oxgc) { return $1; }

# if (, elsif (, unless (, while (, until (, given (, and when (

    # given, when

    # P.225 The given Statement
    # in Chapter 15: Smart Matching and given-when
    # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

    # P.133 The given Statement
    # in Chapter 4: Statements and Declarations
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    elsif (/\G ( (?: if | elsif | unless | while | until | given | when ) (?>\s*) \( ) /oxgc) {
        $slash = 'm//';
        return $1;
    }

# scalar variable ($scalar = ...) =~ tr///;
# scalar variable ($scalar = ...) =~ s///;

    # state

    # P.68 Persistent, Private Variables
    # in Chapter 4: Subroutines
    # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

    # P.160 Persistent Lexically Scoped Variables: state
    # in Chapter 4: Statements and Declarations
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # (and so on)

    elsif (/\G ( \( (?>\s*) (?: local \b | my \b | our \b | state \b )? (?>\s*) \$ $qq_scalar ) /oxgc) {
        my $e_string = e_string($1);

        if (/\G ( (?>\s*) = $qq_paren \) ) ( (?>\s*) (?: =~ | !~ ) (?>\s*) ) (?= (?: tr | y ) \b ) /oxgc) {
            $tr_variable = $e_string . e_string($1);
            $bind_operator = $2;
            $slash = 'm//';
            return '';
        }
        elsif (/\G ( (?>\s*) = $qq_paren \) ) ( (?>\s*) (?: =~ | !~ ) (?>\s*) ) (?= s \b ) /oxgc) {
            $sub_variable = $e_string . e_string($1);
            $bind_operator = $2;
            $slash = 'm//';
            return '';
        }
        else {
            $slash = 'div';
            return $e_string;
        }
    }

# $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> Esjis::PREMATCH()
    elsif (/\G ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  \b | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  ) /oxmsgc) {
        $slash = 'div';
        return q{Esjis::PREMATCH()};
    }

# $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> Esjis::MATCH()
    elsif (/\G ( \$& | \$\{&\} | \$ (?>\s*) MATCH     \b | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) /oxmsgc) {
        $slash = 'div';
        return q{Esjis::MATCH()};
    }

# $', ${'} --> $', ${'}
    elsif (/\G ( \$' | \$\{'\}                                                                                                     ) /oxmsgc) {
        $slash = 'div';
        return $1;
    }

# $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> Esjis::POSTMATCH()
    elsif (/\G (                 \$ (?>\s*) POSTMATCH \b | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) /oxmsgc) {
        $slash = 'div';
        return q{Esjis::POSTMATCH()};
    }

# scalar variable $scalar =~ tr///;
# scalar variable $scalar =~ s///;
# substr() =~ tr///;
# substr() =~ s///;
    elsif (/\G ( \$ $qq_scalar | $qq_substr ) /oxgc) {
        my $scalar = e_string($1);

        if (/\G (    (?>\s*) (?: =~ | !~ ) (?>\s*) ) (?= (?: tr | y ) \b ) /oxgc) {
            $tr_variable = $scalar;
            $bind_operator = $1;
            $slash = 'm//';
            return '';
        }
        elsif (/\G ( (?>\s*) (?: =~ | !~ ) (?>\s*) ) (?= s            \b ) /oxgc) {
            $sub_variable = $scalar;
            $bind_operator = $1;
            $slash = 'm//';
            return '';
        }
        else {
            $slash = 'div';
            return $scalar;
        }
    }

    # end of statement
    elsif (/\G ( [,;] ) /oxgc) {
        $slash = 'm//';

        # clear tr/// variable
        $tr_variable  = '';

        # clear s/// variable
        $sub_variable  = '';

        $bind_operator = '';

        return $1;
    }

# bareword
    elsif (/\G ( \{ (?>\s*) (?: tr | index | rindex | reverse ) (?>\s*) \} ) /oxmsgc) {
        return $1;
    }

# $0 --> $0
    elsif (/\G ( \$ 0 ) /oxmsgc) {
        $slash = 'div';
        return $1;
    }
    elsif (/\G ( \$ \{ (?>\s*) 0 (?>\s*) \} ) /oxmsgc) {
        $slash = 'div';
        return $1;
    }

# $$ --> $$
    elsif (/\G ( \$ \$ ) (?![\w\{]) /oxmsgc) {
        $slash = 'div';
        return $1;
    }

# $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
# $1, $2, $3 --> $1, $2, $3 otherwise
    elsif (/\G \$ ((?>[1-9][0-9]*)) /oxmsgc) {
        $slash = 'div';
        return e_capture($1);
    }
    elsif (/\G \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} /oxmsgc) {
        $slash = 'div';
        return e_capture($1);
    }

# $$foo[ ... ] --> $ $foo->[ ... ]
    elsif (/\G \$ ( \$ (?> [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ .+? \] ) /oxmsgc) {
        $slash = 'div';
        return e_capture($1.'->'.$2);
    }

# $$foo{ ... } --> $ $foo->{ ... }
    elsif (/\G \$ ( \$ (?> [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ .+? \} ) /oxmsgc) {
        $slash = 'div';
        return e_capture($1.'->'.$2);
    }

# $$foo
    elsif (/\G \$ ( \$ (?> [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) /oxmsgc) {
        $slash = 'div';
        return e_capture($1);
    }

# ${ foo }
    elsif (/\G \$ (?>\s*) \{ ( (?>\s*) (?> [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* ) (?>\s*) ) \} /oxmsgc) {
        $slash = 'div';
        return '${' . $1 . '}';
    }

# ${ ... }
    elsif (/\G \$ (?>\s*) \{ (?>\s*) ( $qq_brace ) (?>\s*) \} /oxmsgc) {
        $slash = 'div';
        return e_capture($1);
    }

# variable or function
    #                  $ @ % & *     $ #
    elsif (/\G ( (?: [\$\@\%\&\*] | \$\# | -> | \b sub \b) (?>\s*) (?: split | chop | index | rindex | lc | uc | fc | chr | ord | reverse | getc | tr | y | q | qq | qx | qw | m | s | qr | glob | lstat | opendir | stat | unlink | chdir ) ) \b /oxmsgc) {
        $slash = 'div';
        return $1;
    }
    #                $ $ $ $ $ $ $ $ $ $ $ $ $ $
    #                $ @ # \ ' " / ? ( ) [ ] < >
    elsif (/\G ( \$[\$\@\#\\\'\"\/\?\(\)\[\]\<\>] ) /oxmsgc) {
        $slash = 'div';
        return $1;
    }

# while (<FILEHANDLE>)
    elsif (/\G \b (while (?>\s*) \( (?>\s*) <[\$]?[A-Za-z_][A-Za-z_0-9]*> (?>\s*) \)) \b /oxgc) {
        return $1;
    }

# while (<WILDCARD>) --- glob

    # avoid "Error: Runtime exception" of perl version 5.005_03

    elsif (/\G \b while (?>\s*) \( (?>\s*) < ((?:[^\x81-\x9F\xE0-\xFC>\0\a\e\f\n\r\t]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])+?) > (?>\s*) \) \b /oxgc) {
        return 'while ($_ = Esjis::glob("' . $1 . '"))';
    }

# while (glob)
    elsif (/\G \b while (?>\s*) \( (?>\s*) glob (?>\s*) \) /oxgc) {
        return 'while ($_ = Esjis::glob_)';
    }

# while (glob(WILDCARD))
    elsif (/\G \b while (?>\s*) \( (?>\s*) glob \b /oxgc) {
        return 'while ($_ = Esjis::glob';
    }

# doit if, doit unless, doit while, doit until, doit for, doit when
    elsif (/\G \b ( if | unless | while | until | for | when ) \b /oxgc) { $slash = 'm//'; return $1; }

# subroutines of package Esjis
    elsif (/\G \b (CORE:: | ->(>?\s*) (?: atan2 | [a-z]{2,})) \b       /oxgc) { $slash = 'm//'; return $1;                  }
    elsif (/\G \b Char::eval       (?= (?>\s*) \{ )                    /oxgc) { $slash = 'm//'; return 'eval';              }
    elsif (/\G \b Sjis::eval       (?= (?>\s*) \{ )                    /oxgc) { $slash = 'm//'; return 'eval';              }
    elsif (/\G \b Char::eval    \b (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'eval Char::escape'; }
    elsif (/\G \b Sjis::eval    \b (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'eval Sjis::escape'; }
    elsif (/\G \b bytes::substr \b (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'substr';            }
    elsif (/\G \b chop \b          (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Esjis::chop';       }
    elsif (/\G \b bytes::index \b  (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'index';             }
    elsif (/\G \b Char::index \b   (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Char::index';       }
    elsif (/\G \b Sjis::index \b   (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Sjis::index';       }
    elsif (/\G \b index \b         (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Esjis::index';      }
    elsif (/\G \b bytes::rindex \b (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'rindex';            }
    elsif (/\G \b Char::rindex \b  (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Char::rindex';      }
    elsif (/\G \b Sjis::rindex \b  (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Sjis::rindex';      }
    elsif (/\G \b rindex \b        (?! (?>\s*) => )                    /oxgc) { $slash = 'm//'; return 'Esjis::rindex';     }
    elsif (/\G \b lc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::lc';         }
    elsif (/\G \b lcfirst (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::lcfirst';    }
    elsif (/\G \b uc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::uc';         }
    elsif (/\G \b ucfirst (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::ucfirst';    }
    elsif (/\G \b fc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::fc';         }
#if ESCAPE_SECOND_OCTET

    # stacked file test operators

    # P.179 File Test Operators
    # in Chapter 12: File Tests
    # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

    # P.106 Named Unary and File Test Operators
    # in Chapter 3: Unary and Binary Operators
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # (and so on)

    elsif (/\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                                                             (?>\s*) (\") ((?:$qq_char)+?)             (\") /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('',  $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\#) ((?:$qq_char)+?)             (\#) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\() ((?:$qq_paren)+?)            (\)) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\{) ((?:$qq_brace)+?)            (\}) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\[) ((?:$qq_bracket)+?)          (\]) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\<) ((?:$qq_angle)+?)            (\>) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\S) ((?:$qq_char)+?)             (\2) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; }

    elsif (/\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                                                             (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)    (\') /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('',  $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)    (\#) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)   (\)) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)   (\}) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?) (\]) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)   (\>) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\S) ((?:\\\2|\\\\|$q_char)+?)    (\2) /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; }

    elsif (/\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+)) (\$ (?> \w+ (?: ::\w+)* ) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ) )*) /oxgc)
                                                                                                                   { $slash = 'm//'; return "Esjis::filetest(qw($1),$2)"; }
    elsif (/\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+)) \( ((?:$qq_paren)*?) \)   /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1),$2)"; }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) (?= [a-z]+)                                       /oxgc) { $slash = 'm//'; return "Esjis::filetest qw($1),";    }
    elsif (/\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) ((?>\w+))                                         /oxgc) { $slash = 'm//'; return "Esjis::filetest(qw($1),$2)"; }

    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC])            (?>\s*) (\") ((?:$qq_char)+?)                (\") /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('',  $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\#) ((?:$qq_char)+?)                (\#) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\() ((?:$qq_paren)+?)               (\)) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\{) ((?:$qq_brace)+?)               (\}) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\[) ((?:$qq_bracket)+?)             (\]) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\<) ((?:$qq_angle)+?)               (\>) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\S) ((?:$qq_char)+?)                (\2) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; }

    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC])            (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)       (\') /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('',  $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)       (\#) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)      (\)) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)      (\}) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?)    (\]) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)      (\>) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\S) ((?:\\\2|\\\\|$q_char)+?)       (\2) /oxgc) { $slash = 'm//'; return "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; }

    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s*) (\$ (?> \w+ (?: ::\w+)* ) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ) )*) /oxgc)
                                                                                                                   { $slash = 'm//'; return "Esjis::$1($2)";             }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s*) \( ((?:$qq_paren)*?) \)                              /oxgc) { $slash = 'm//'; return "Esjis::$1($2)";             }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?= (?>\s+) [a-z]+)                                          /oxgc) { $slash = 'm//'; return "Esjis::$1";                 }
    elsif (/\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) ((?>\w+))                                            /oxgc) { $slash = 'm//'; return "Esjis::$1(::"."$2)";        }
    elsif (/\G -(t)                            (?>\s+) ((?>\w+))                                            /oxgc) { $slash = 'm//'; return "-t $2";                     }
    elsif (/\G \b lstat         (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(])                                /oxgc) { $slash = 'm//'; return 'Esjis::lstat';              }
    elsif (/\G \b stat          (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(])                                /oxgc) { $slash = 'm//'; return 'Esjis::stat';               }
#endif /* #if ESCAPE_SECOND_OCTET */

    # "-s '' ..." means file test "-s 'filename' ..." (not means "- s/// ...")
    elsif (/\G -s                                          (?>\s*) (\") ((?:$qq_char)+?)               (\") /oxgc) { $slash = 'm//'; return '-s ' . e_qq('',  $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\#) ((?:$qq_char)+?)               (\#) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\() ((?:$qq_paren)+?)              (\)) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\{) ((?:$qq_brace)+?)              (\}) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\[) ((?:$qq_bracket)+?)            (\]) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\<) ((?:$qq_angle)+?)              (\>) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }
    elsif (/\G -s                               (?>\s+) qq (?>\s*) (\S) ((?:$qq_char)+?)               (\1) /oxgc) { $slash = 'm//'; return '-s ' . e_qq('qq',$1,$3,$2); }

    elsif (/\G -s                                          (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)      (\') /oxgc) { $slash = 'm//'; return '-s ' . e_q ('',  $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)      (\#) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)     (\)) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)     (\}) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?)   (\]) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)     (\>) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }
    elsif (/\G -s                               (?>\s+) q  (?>\s*) (\S) ((?:\\\1|\\\\|$q_char)+?)      (\1) /oxgc) { $slash = 'm//'; return '-s ' . e_q ('q', $1,$3,$2); }

    elsif (/\G -s                               (?>\s*) (\$ (?> \w+ (?: ::\w+)* ) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ) )*) /oxgc)
                                                                                                                   { $slash = 'm//'; return "-s $1";   }
    elsif (/\G -s                               (?>\s*) \( ((?:$qq_paren)*?) \)                             /oxgc) { $slash = 'm//'; return "-s ($1)"; }
    elsif (/\G -s                               (?= (?>\s+) [a-z]+)                                         /oxgc) { $slash = 'm//'; return '-s';      }
    elsif (/\G -s                               (?>\s+) ((?>\w+))                                           /oxgc) { $slash = 'm//'; return "-s $1";   }

    elsif (/\G \b bytes::length (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'length';                   }
    elsif (/\G \b bytes::chr    (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'chr';                      }
    elsif (/\G \b chr           (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::chr';               }
    elsif (/\G \b bytes::ord    (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'div'; return 'ord';                      }
    elsif (/\G \b ord           (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'div'; return $function_ord;              }
    elsif (/\G \b glob          (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $slash = 'm//'; return 'Esjis::glob';              }
    elsif (/\G \b lc \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::lc_';               }
    elsif (/\G \b lcfirst \b       (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::lcfirst_';          }
    elsif (/\G \b uc \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::uc_';               }
    elsif (/\G \b ucfirst \b       (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::ucfirst_';          }
    elsif (/\G \b fc \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::fc_';               }
#if ESCAPE_SECOND_OCTET
    elsif (/\G \b lstat \b         (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::lstat_';            }
    elsif (/\G \b stat \b          (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::stat_';             }
    elsif (/\G    (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                     \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return "Esjis::filetest_(qw($1))"; }
    elsif (/\G    -([rwxoRWXOezsfdlpSbcugkTBMAC])
                     \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return "Esjis::${1}_";             }

#endif
    elsif (/\G    -s \b            (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return '-s ';                      }

    elsif (/\G \b bytes::length \b (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'length';                   }
    elsif (/\G \b bytes::chr \b    (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'chr';                      }
    elsif (/\G \b chr \b           (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::chr_';              }
    elsif (/\G \b bytes::ord \b    (?! (?>\s*) => )                          /oxgc) { $slash = 'div'; return 'ord';                      }
    elsif (/\G \b ord \b           (?! (?>\s*) => )                          /oxgc) { $slash = 'div'; return $function_ord_;             }
    elsif (/\G \b glob \b          (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return 'Esjis::glob_';             }
    elsif (/\G \b reverse \b       (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return $function_reverse;          }
    elsif (/\G \b getc \b          (?! (?>\s*) => )                          /oxgc) { $slash = 'm//'; return $function_getc;             }
#if ESCAPE_SECOND_OCTET
    elsif (/\G \b opendir ((?>\s*) \( (?>\s*)) (?=[A-Za-z_])                 /oxgc) { $slash = 'm//'; return "Esjis::opendir$1*";        }
    elsif (/\G \b opendir ((?>\s+))            (?=[A-Za-z_])                 /oxgc) { $slash = 'm//'; return "Esjis::opendir$1*";        }
    elsif (/\G \b unlink \b       (?! (?>\s*) => )                           /oxgc) { $slash = 'm//'; return 'Esjis::unlink';            }

# chdir
    elsif (/\G \b (chdir) \b       (?! (?>\s*) => ) /oxgc) {
        $slash = 'm//';

        my $e = 'Esjis::chdir';

        while (/\G ( (?>\s+) | \( | \#.* ) /oxgc) {
            $e .= $1;
        }

# end of chdir
        if    (/\G (?= [,;\)\}\]] )          /oxgc) { return $e;                 }

# chdir scalar value
        elsif (/\G ( [\$\@\&\*] $qq_scalar ) /oxgc) { return $e . e_string($1);  }

# chdir qq//
        elsif (/\G \b (qq) \b /oxgc) {
            if (/\G (\#) ((?:$qq_char)*?) (\#) /oxgc)                        { return $e . e_chdir('qq',$1,$3,$2);   } # qq# #  --> qr # #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                         /oxgc) { $e .= $1; }
                    elsif (/\G (\()          ((?:$qq_paren)*?)   (\)) /oxgc) { return $e . e_chdir('qq',$1,$3,$2);   } # qq ( ) --> qr ( )
                    elsif (/\G (\{)          ((?:$qq_brace)*?)   (\}) /oxgc) { return $e . e_chdir('qq',$1,$3,$2);   } # qq { } --> qr { }
                    elsif (/\G (\[)          ((?:$qq_bracket)*?) (\]) /oxgc) { return $e . e_chdir('qq',$1,$3,$2);   } # qq [ ] --> qr [ ]
                    elsif (/\G (\<)          ((?:$qq_angle)*?)   (\>) /oxgc) { return $e . e_chdir('qq',$1,$3,$2);   } # qq < > --> qr < >
                    elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) /oxgc) { return $e . e_chdir('qq','{','}',$2); } # qq | | --> qr { }
                    elsif (/\G (\S)          ((?:$qq_char)*?)    (\1) /oxgc) { return $e . e_chdir('qq',$1,$3,$2);   } # qq * * --> qr * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# chdir q//
        elsif (/\G \b (q) \b /oxgc) {
            if (/\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc)                    { return $e . e_chdir_q('q',$1,$3,$2);   } # q# #  --> qr # #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                              /oxgc) { $e .= $1; }
                    elsif (/\G (\() ((?:\\\\|\\\)|\\\(|$q_paren)*?)   (\)) /oxgc) { return $e . e_chdir_q('q',$1,$3,$2);   } # q ( ) --> qr ( )
                    elsif (/\G (\{) ((?:\\\\|\\\}|\\\{|$q_brace)*?)   (\}) /oxgc) { return $e . e_chdir_q('q',$1,$3,$2);   } # q { } --> qr { }
                    elsif (/\G (\[) ((?:\\\\|\\\]|\\\[|$q_bracket)*?) (\]) /oxgc) { return $e . e_chdir_q('q',$1,$3,$2);   } # q [ ] --> qr [ ]
                    elsif (/\G (\<) ((?:\\\\|\\\>|\\\<|$q_angle)*?)   (\>) /oxgc) { return $e . e_chdir_q('q',$1,$3,$2);   } # q < > --> qr < >
                    elsif (/\G ([*\-:?\\^|])       ((?:$q_char)*?)    (\1) /oxgc) { return $e . e_chdir_q('q','{','}',$2); } # q | | --> qr { }
                    elsif (/\G (\S) ((?:\\\\|\\\1|     $q_char)*?)    (\1) /oxgc) { return $e . e_chdir_q('q',$1,$3,$2);   } # q * * --> qr * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# chdir ''
        elsif (/\G (\') /oxgc) {
            my $q_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)    /oxgc) { $q_string .= $1; }
                elsif (/\G (\\\')    /oxgc) { $q_string .= $1; }
                elsif (/\G \'        /oxgc)                                       { return $e . e_chdir_q('',"'","'",$q_string); }
                elsif (/\G ($q_char) /oxgc) { $q_string .= $1; }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }

# chdir ""
        elsif (/\G (\") /oxgc) {
            my $qq_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)    /oxgc) { $qq_string .= $1; }
                elsif (/\G (\\\")    /oxgc) { $qq_string .= $1; }
                elsif (/\G \"        /oxgc)                                       { return $e . e_chdir('','"','"',$qq_string); }
                elsif (/\G ($q_char) /oxgc) { $qq_string .= $1; }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

#endif
# split
    elsif (/\G \b (split) \b (?! (?>\s*) => ) /oxgc) {
        $slash = 'm//';

        my $e = '';
        while (/\G ( (?>\s+) | \( | \#.* ) /oxgc) {
            $e .= $1;
        }

# end of split
        if    (/\G (?= [,;\)\}\]] )          /oxgc) { return 'Esjis::split' . $e;                 }

# split scalar value
        elsif (/\G ( [\$\@\&\*] $qq_scalar ) /oxgc) { return 'Esjis::split' . $e . e_string($1);  }

# split literal space
        elsif (/\G \b qq           (\#) [ ] (\#) /oxgc) { return 'Esjis::split' . $e . qq  {qq$1 $2}; }
        elsif (/\G \b qq ((?>\s*)) (\() [ ] (\)) /oxgc) { return 'Esjis::split' . $e . qq{$1qq$2 $3}; }
        elsif (/\G \b qq ((?>\s*)) (\{) [ ] (\}) /oxgc) { return 'Esjis::split' . $e . qq{$1qq$2 $3}; }
        elsif (/\G \b qq ((?>\s*)) (\[) [ ] (\]) /oxgc) { return 'Esjis::split' . $e . qq{$1qq$2 $3}; }
        elsif (/\G \b qq ((?>\s*)) (\<) [ ] (\>) /oxgc) { return 'Esjis::split' . $e . qq{$1qq$2 $3}; }
        elsif (/\G \b qq ((?>\s*)) (\S) [ ] (\2) /oxgc) { return 'Esjis::split' . $e . qq{$1qq$2 $3}; }
        elsif (/\G \b q            (\#) [ ] (\#) /oxgc) { return 'Esjis::split' . $e . qq   {q$1 $2}; }
        elsif (/\G \b q  ((?>\s*)) (\() [ ] (\)) /oxgc) { return 'Esjis::split' . $e . qq {$1q$2 $3}; }
        elsif (/\G \b q  ((?>\s*)) (\{) [ ] (\}) /oxgc) { return 'Esjis::split' . $e . qq {$1q$2 $3}; }
        elsif (/\G \b q  ((?>\s*)) (\[) [ ] (\]) /oxgc) { return 'Esjis::split' . $e . qq {$1q$2 $3}; }
        elsif (/\G \b q  ((?>\s*)) (\<) [ ] (\>) /oxgc) { return 'Esjis::split' . $e . qq {$1q$2 $3}; }
        elsif (/\G \b q  ((?>\s*)) (\S) [ ] (\2) /oxgc) { return 'Esjis::split' . $e . qq {$1q$2 $3}; }
        elsif (/\G                    ' [ ] '    /oxgc) { return 'Esjis::split' . $e . qq     {' '};  }
        elsif (/\G                    " [ ] "    /oxgc) { return 'Esjis::split' . $e . qq     {" "};  }

# split qq//
        elsif (/\G \b (qq) \b /oxgc) {
            if (/\G (\#) ((?:$qq_char)*?) (\#) /oxgc)                        { return e_split($e.'qr',$1,$3,$2,'');   } # qq# #  --> qr # #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                         /oxgc) { $e .= $1; }
                    elsif (/\G (\()          ((?:$qq_paren)*?)   (\)) /oxgc) { return e_split($e.'qr',$1,$3,$2,'');   } # qq ( ) --> qr ( )
                    elsif (/\G (\{)          ((?:$qq_brace)*?)   (\}) /oxgc) { return e_split($e.'qr',$1,$3,$2,'');   } # qq { } --> qr { }
                    elsif (/\G (\[)          ((?:$qq_bracket)*?) (\]) /oxgc) { return e_split($e.'qr',$1,$3,$2,'');   } # qq [ ] --> qr [ ]
                    elsif (/\G (\<)          ((?:$qq_angle)*?)   (\>) /oxgc) { return e_split($e.'qr',$1,$3,$2,'');   } # qq < > --> qr < >
                    elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) /oxgc) { return e_split($e.'qr','{','}',$2,''); } # qq | | --> qr { }
                    elsif (/\G (\S)          ((?:$qq_char)*?)    (\1) /oxgc) { return e_split($e.'qr',$1,$3,$2,'');   } # qq * * --> qr * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# split qr//
        elsif (/\G \b (qr) \b /oxgc) {
            if (/\G (\#) ((?:$qq_char)*?) (\#) ([imosxpadlunbB]*) /oxgc)                        { return e_split  ($e.'qr',$1,$3,$2,$4);   } # qr# #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                                            /oxgc) { $e .= $1; }
                    elsif (/\G (\()          ((?:$qq_paren)*?)   (\)) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # qr ( )
                    elsif (/\G (\{)          ((?:$qq_brace)*?)   (\}) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # qr { }
                    elsif (/\G (\[)          ((?:$qq_bracket)*?) (\]) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # qr [ ]
                    elsif (/\G (\<)          ((?:$qq_angle)*?)   (\>) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # qr < >
                    elsif (/\G (\')          ((?:$qq_char)*?)    (\') ([imosxpadlunbB]*) /oxgc) { return e_split_q($e.'qr',$1, $3, $2,$4); } # qr ' '
                    elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr','{','}',$2,$4); } # qr | | --> qr { }
                    elsif (/\G (\S)          ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # qr * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# split q//
        elsif (/\G \b (q) \b /oxgc) {
            if (/\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc)                    { return e_split_q($e.'qr',$1,$3,$2,'');   } # q# #  --> qr # #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                              /oxgc) { $e .= $1; }
                    elsif (/\G (\() ((?:\\\\|\\\)|\\\(|$q_paren)*?)   (\)) /oxgc) { return e_split_q($e.'qr',$1,$3,$2,'');   } # q ( ) --> qr ( )
                    elsif (/\G (\{) ((?:\\\\|\\\}|\\\{|$q_brace)*?)   (\}) /oxgc) { return e_split_q($e.'qr',$1,$3,$2,'');   } # q { } --> qr { }
                    elsif (/\G (\[) ((?:\\\\|\\\]|\\\[|$q_bracket)*?) (\]) /oxgc) { return e_split_q($e.'qr',$1,$3,$2,'');   } # q [ ] --> qr [ ]
                    elsif (/\G (\<) ((?:\\\\|\\\>|\\\<|$q_angle)*?)   (\>) /oxgc) { return e_split_q($e.'qr',$1,$3,$2,'');   } # q < > --> qr < >
                    elsif (/\G ([*\-:?\\^|])       ((?:$q_char)*?)    (\1) /oxgc) { return e_split_q($e.'qr','{','}',$2,''); } # q | | --> qr { }
                    elsif (/\G (\S) ((?:\\\\|\\\1|     $q_char)*?)    (\1) /oxgc) { return e_split_q($e.'qr',$1,$3,$2,'');   } # q * * --> qr * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# split m//
        elsif (/\G \b (m) \b /oxgc) {
            if (/\G (\#) ((?:$qq_char)*?) (\#) ([cgimosxpadlunbB]*) /oxgc)                        { return e_split  ($e.'qr',$1,$3,$2,$4);   } # m# #  --> qr # #
            else {
                while (not /\G \z/oxgc) {
                    if    (/\G ((?>\s+)|\#.*)                                              /oxgc) { $e .= $1; }
                    elsif (/\G (\()          ((?:$qq_paren)*?)   (\)) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # m ( ) --> qr ( )
                    elsif (/\G (\{)          ((?:$qq_brace)*?)   (\}) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # m { } --> qr { }
                    elsif (/\G (\[)          ((?:$qq_bracket)*?) (\]) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # m [ ] --> qr [ ]
                    elsif (/\G (\<)          ((?:$qq_angle)*?)   (\>) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # m < > --> qr < >
                    elsif (/\G (\')          ((?:$qq_char)*?)    (\') ([cgimosxpadlunbB]*) /oxgc) { return e_split_q($e.'qr',$1, $3, $2,$4); } # m ' ' --> qr ' '
                    elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr','{','}',$2,$4); } # m | | --> qr { }
                    elsif (/\G (\S)          ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { return e_split  ($e.'qr',$1, $3, $2,$4); } # m * * --> qr * *
                }
                die __FILE__, ": Search pattern not terminated\n";
            }
        }

# split ''
        elsif (/\G (\') /oxgc) {
            my $q_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)    /oxgc) { $q_string .= $1; }
                elsif (/\G (\\\')    /oxgc) { $q_string .= $1; }                               # splitqr'' --> split qr''
                elsif (/\G \'        /oxgc)                                                    { return e_split_q($e.q{ qr},"'","'",$q_string,''); } # ' ' --> qr ' '
                elsif (/\G ($q_char) /oxgc) { $q_string .= $1; }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }

# split ""
        elsif (/\G (\") /oxgc) {
            my $qq_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)    /oxgc) { $qq_string .= $1; }
                elsif (/\G (\\\")    /oxgc) { $qq_string .= $1; }                              # splitqr"" --> split qr""
                elsif (/\G \"        /oxgc)                                                    { return e_split($e.q{ qr},'"','"',$qq_string,''); } # " " --> qr " "
                elsif (/\G ($q_char) /oxgc) { $qq_string .= $1; }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }

# split //
        elsif (/\G (\/) /oxgc) {
            my $regexp = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)                  /oxgc) { $regexp .= $1; }
                elsif (/\G (\\\/)                  /oxgc) { $regexp .= $1; }                   # splitqr// --> split qr//
                elsif (/\G \/ ([cgimosxpadlunbB]*) /oxgc)                                      { return e_split($e.q{ qr}, '/','/',$regexp,$1); } # / / --> qr / /
                elsif (/\G ($q_char)               /oxgc) { $regexp .= $1; }
            }
            die __FILE__, ": Search pattern not terminated\n";
        }
    }

# tr/// or y///

    # about [cdsrbB]* (/B modifier)
    #
    # P.559 appendix C
    # of ISBN 4-89052-384-7 Programming perl
    # (Japanese title is: Perl puroguramingu)

    elsif (/\G \b ( tr | y ) \b /oxgc) {
        my $ope = $1;

        #        $1   $2               $3   $4               $5   $6
        if (/\G (\#) ((?:$qq_char)*?) (\#) ((?:$qq_char)*?) (\#) ([cdsrbB]*) /oxgc) { # tr# # #
            my @tr = ($tr_variable,$2);
            return e_tr(@tr,'',$4,$6);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)              /oxgc) { $e .= $1; }
                elsif (/\G (\() ((?:$qq_paren)*?) (\)) /oxgc) {
                    my @tr = ($tr_variable,$2);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                            /oxgc) { $e .= $1; }
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr ( ) ( )
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr ( ) { }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr ( ) [ ]
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr ( ) < >
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr ( ) * *
                    }
                    die __FILE__, ": Transliteration replacement not terminated\n";
                }
                elsif (/\G (\{) ((?:$qq_brace)*?) (\}) /oxgc) {
                    my @tr = ($tr_variable,$2);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                            /oxgc) { $e .= $1; }
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr { } ( )
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr { } { }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr { } [ ]
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr { } < >
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr { } * *
                    }
                    die __FILE__, ": Transliteration replacement not terminated\n";
                }
                elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) {
                    my @tr = ($tr_variable,$2);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                            /oxgc) { $e .= $1; }
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr [ ] ( )
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr [ ] { }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr [ ] [ ]
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr [ ] < >
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr [ ] * *
                    }
                    die __FILE__, ": Transliteration replacement not terminated\n";
                }
                elsif (/\G (\<) ((?:$qq_angle)*?) (\>) /oxgc) {
                    my @tr = ($tr_variable,$2);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                            /oxgc) { $e .= $1; }
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr < > ( )
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr < > { }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr < > [ ]
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr < > < >
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cdsrbB]*) /oxgc) { return e_tr(@tr,$e,$2,$4); } # tr < > * *
                    }
                    die __FILE__, ": Transliteration replacement not terminated\n";
                }
                #           $1   $2               $3   $4               $5   $6
                elsif (/\G (\S) ((?:$qq_char)*?) (\1) ((?:$qq_char)*?) (\1) ([cdsrbB]*) /oxgc) { # tr * * *
                    my @tr = ($tr_variable,$2);
                    return e_tr(@tr,'',$4,$6);
                }
            }
            die __FILE__, ": Transliteration pattern not terminated\n";
        }
    }

# qq//
    elsif (/\G \b (qq) \b /oxgc) {
        my $ope = $1;

#       if (/\G (\#) ((?:$qq_char)*?) (\#) /oxgc) { return e_qq($ope,$1,$3,$2); } # qq# #
        if (/\G (\#) /oxgc) {                                                     # qq# #
            my $qq_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)     /oxgc) { $qq_string .= $1;                     }
                elsif (/\G (\\\#)     /oxgc) { $qq_string .= $1;                     }
                elsif (/\G (\#)       /oxgc) { return e_qq($ope,'#','#',$qq_string); }
                elsif (/\G ($qq_char) /oxgc) { $qq_string .= $1;                     }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }

        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)              /oxgc) { $e .= $1; }

#               elsif (/\G (\() ((?:$qq_paren)*?) (\)) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qq ( )
                elsif (/\G (\() /oxgc) {                                                           # qq ( )
                    my $qq_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\\\))     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\()       /oxgc) { $qq_string .= $1; $nest++;                 }
                        elsif (/\G (\))       /oxgc) {
                            if (--$nest == 0)        { return $e . e_qq($ope,'(',')',$qq_string); }
                            else                     { $qq_string .= $1;                          }
                        }
                        elsif (/\G ($qq_char) /oxgc) { $qq_string .= $1;                          }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\{) ((?:$qq_brace)*?) (\}) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qq { }
                elsif (/\G (\{) /oxgc) {                                                           # qq { }
                    my $qq_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\\\})     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\{)       /oxgc) { $qq_string .= $1; $nest++;                 }
                        elsif (/\G (\})       /oxgc) {
                            if (--$nest == 0)        { return $e . e_qq($ope,'{','}',$qq_string); }
                            else                     { $qq_string .= $1;                          }
                        }
                        elsif (/\G ($qq_char) /oxgc) { $qq_string .= $1;                          }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qq [ ]
                elsif (/\G (\[) /oxgc) {                                                             # qq [ ]
                    my $qq_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\\\])     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\[)       /oxgc) { $qq_string .= $1; $nest++;                 }
                        elsif (/\G (\])       /oxgc) {
                            if (--$nest == 0)        { return $e . e_qq($ope,'[',']',$qq_string); }
                            else                     { $qq_string .= $1;                          }
                        }
                        elsif (/\G ($qq_char) /oxgc) { $qq_string .= $1;                          }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\<) ((?:$qq_angle)*?) (\>) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qq < >
                elsif (/\G (\<) /oxgc) {                                                           # qq < >
                    my $qq_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\\\>)     /oxgc) { $qq_string .= $1;                          }
                        elsif (/\G (\<)       /oxgc) { $qq_string .= $1; $nest++;                 }
                        elsif (/\G (\>)       /oxgc) {
                            if (--$nest == 0)        { return $e . e_qq($ope,'<','>',$qq_string); }
                            else                     { $qq_string .= $1;                          }
                        }
                        elsif (/\G ($qq_char) /oxgc) { $qq_string .= $1;                          }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\S) ((?:$qq_char)*?) (\1) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qq * *
                elsif (/\G (\S) /oxgc) {                                                          # qq * *
                    my $delimiter = $1;
                    my $qq_string = '';
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)             /oxgc) { $qq_string .= $1;                                        }
                        elsif (/\G (\\\Q$delimiter\E) /oxgc) { $qq_string .= $1;                                        }
                        elsif (/\G (\Q$delimiter\E)   /oxgc) { return $e . e_qq($ope,$delimiter,$delimiter,$qq_string); }
                        elsif (/\G ($qq_char)         /oxgc) { $qq_string .= $1;                                        }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

# qr//
    elsif (/\G \b (qr) \b /oxgc) {
        my $ope = $1;
        if (/\G (\#) ((?:$qq_char)*?) (\#) ([imosxpadlunbB]*) /oxgc) { # qr# # #
            return e_qr($ope,$1,$3,$2,$4);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)                                            /oxgc) { $e .= $1; }
                elsif (/\G (\()          ((?:$qq_paren)*?)   (\)) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # qr ( )
                elsif (/\G (\{)          ((?:$qq_brace)*?)   (\}) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # qr { }
                elsif (/\G (\[)          ((?:$qq_bracket)*?) (\]) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # qr [ ]
                elsif (/\G (\<)          ((?:$qq_angle)*?)   (\>) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # qr < >
                elsif (/\G (\')          ((?:$qq_char)*?)    (\') ([imosxpadlunbB]*) /oxgc) { return $e . e_qr_q($ope,$1, $3, $2,$4); } # qr ' '
                elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,'{','}',$2,$4); } # qr | | --> qr { }
                elsif (/\G (\S)          ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # qr * *
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

# qw//
    elsif (/\G \b (qw) \b /oxgc) {
        my $ope = $1;
        if (/\G (\#) (.*?) (\#) /oxmsgc) { # qw# #
            return e_qw($ope,$1,$3,$2);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)                        /oxgc)   { $e .= $1; }

                elsif (/\G (\()          ([^(]*?)           (\)) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw ( )
                elsif (/\G (\()          ((?:$q_paren)*?)   (\)) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw ( )

                elsif (/\G (\{)          ([^{]*?)           (\}) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw { }
                elsif (/\G (\{)          ((?:$q_brace)*?)   (\}) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw { }

                elsif (/\G (\[)          ([^[]*?)           (\]) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw [ ]
                elsif (/\G (\[)          ((?:$q_bracket)*?) (\]) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw [ ]

                elsif (/\G (\<)          ([^<]*?)           (\>) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw < >
                elsif (/\G (\<)          ((?:$q_angle)*?)   (\>) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw < >

                elsif (/\G ([\x21-\x3F]) (.*?)              (\1) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw * *
                elsif (/\G (\S)          ((?:$q_char)*?)    (\1) /oxmsgc) { return $e . e_qw($ope,$1,$3,$2); } # qw * *
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

# qx//
    elsif (/\G \b (qx) \b /oxgc) {
        my $ope = $1;
        if (/\G (\#) ((?:$qq_char)*?) (\#) /oxgc) { # qx# #
            return e_qq($ope,$1,$3,$2);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)                /oxgc) { $e .= $1; }
                elsif (/\G (\() ((?:$qq_paren)*?)   (\)) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qx ( )
                elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qx { }
                elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qx [ ]
                elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qx < >
                elsif (/\G (\') ((?:$qq_char)*?)    (\') /oxgc) { return $e . e_q ($ope,$1,$3,$2); } # qx ' '
                elsif (/\G (\S) ((?:$qq_char)*?)    (\1) /oxgc) { return $e . e_qq($ope,$1,$3,$2); } # qx * *
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

# q//
    elsif (/\G \b (q) \b /oxgc) {
        my $ope = $1;

#       if (/\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc) { return e_q($ope,$1,$3,$2); } # q# #

        # avoid "Error: Runtime exception" of perl version 5.005_03
        # (and so on)

        if (/\G (\#) /oxgc) {                                                             # q# #
            my $q_string = '';
            while (not /\G \z/oxgc) {
                if    (/\G (\\\\)    /oxgc) { $q_string .= $1;                    }
                elsif (/\G (\\\#)    /oxgc) { $q_string .= $1;                    }
                elsif (/\G (\#)      /oxgc) { return e_q($ope,'#','#',$q_string); }
                elsif (/\G ($q_char) /oxgc) { $q_string .= $1;                    }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }

        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)                       /oxgc) { $e .= $1; }

#               elsif (/\G (\() ((?:\\\)|\\\\|$q_paren)*?) (\)) /oxgc) { return $e . e_q($ope,$1,$3,$2); } # q ( )
                elsif (/\G (\() /oxgc) {                                                                   # q ( )
                    my $q_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\))    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\()    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\()      /oxgc) { $q_string .= $1; $nest++;                }
                        elsif (/\G (\))      /oxgc) {
                            if (--$nest == 0)       { return $e . e_q($ope,'(',')',$q_string); }
                            else                    { $q_string .= $1;                         }
                        }
                        elsif (/\G ($q_char) /oxgc) { $q_string .= $1;                         }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\{) ((?:\\\}|\\\\|$q_brace)*?) (\}) /oxgc) { return $e . e_q($ope,$1,$3,$2); } # q { }
                elsif (/\G (\{) /oxgc) {                                                                   # q { }
                    my $q_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\})    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\{)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\{)      /oxgc) { $q_string .= $1; $nest++;                }
                        elsif (/\G (\})      /oxgc) {
                            if (--$nest == 0)       { return $e . e_q($ope,'{','}',$q_string); }
                            else                    { $q_string .= $1;                         }
                        }
                        elsif (/\G ($q_char) /oxgc) { $q_string .= $1;                         }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\[) ((?:\\\]|\\\\|$q_bracket)*?) (\]) /oxgc) { return $e . e_q($ope,$1,$3,$2); } # q [ ]
                elsif (/\G (\[) /oxgc) {                                                                     # q [ ]
                    my $q_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\])    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\[)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\[)      /oxgc) { $q_string .= $1; $nest++;                }
                        elsif (/\G (\])      /oxgc) {
                            if (--$nest == 0)       { return $e . e_q($ope,'[',']',$q_string); }
                            else                    { $q_string .= $1;                         }
                        }
                        elsif (/\G ($q_char) /oxgc) { $q_string .= $1;                         }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\<) ((?:\\\>|\\\\|$q_angle)*?) (\>) /oxgc) { return $e . e_q($ope,$1,$3,$2); } # q < >
                elsif (/\G (\<) /oxgc) {                                                                   # q < >
                    my $q_string = '';
                    local $nest = 1;
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\>)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\\\<)    /oxgc) { $q_string .= $1;                         }
                        elsif (/\G (\<)      /oxgc) { $q_string .= $1; $nest++;                }
                        elsif (/\G (\>)      /oxgc) {
                            if (--$nest == 0)       { return $e . e_q($ope,'<','>',$q_string); }
                            else                    { $q_string .= $1;                         }
                        }
                        elsif (/\G ($q_char) /oxgc) { $q_string .= $1;                         }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }

#               elsif (/\G (\S) ((?:\\\1|\\\\|$q_char)*?) (\1) /oxgc) { return $e . e_q($ope,$1,$3,$2); } # q * *
                elsif (/\G (\S) /oxgc) {                                                                  # q * *
                    my $delimiter = $1;
                    my $q_string = '';
                    while (not /\G \z/oxgc) {
                        if    (/\G (\\\\)             /oxgc) { $q_string .= $1;                                       }
                        elsif (/\G (\\\Q$delimiter\E) /oxgc) { $q_string .= $1;                                       }
                        elsif (/\G (\Q$delimiter\E)   /oxgc) { return $e . e_q($ope,$delimiter,$delimiter,$q_string); }
                        elsif (/\G ($q_char)          /oxgc) { $q_string .= $1;                                       }
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }
            die __FILE__, ": Can't find string terminator anywhere before EOF\n";
        }
    }

# m//
    elsif (/\G \b (m) \b /oxgc) {
        my $ope = $1;
        if (/\G (\#) ((?:$qq_char)*?) (\#) ([cgimosxpadlunbB]*) /oxgc) { # m# #
            return e_qr($ope,$1,$3,$2,$4);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if    (/\G ((?>\s+)|\#.*)                                             /oxgc) { $e .= $1; }
                elsif (/\G (\()         ((?:$qq_paren)*?)   (\)) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m ( )
                elsif (/\G (\{)         ((?:$qq_brace)*?)   (\}) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m { }
                elsif (/\G (\[)         ((?:$qq_bracket)*?) (\]) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m [ ]
                elsif (/\G (\<)         ((?:$qq_angle)*?)   (\>) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m < >
                elsif (/\G (\?)         ((?:$qq_char)*?)    (\?) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m ? ?
                elsif (/\G (\')         ((?:$qq_char)*?)    (\') ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr_q($ope,$1, $3, $2,$4); } # m ' '
                elsif (/\G ([*\-:\\^|]) ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,'{','}',$2,$4); } # m | | --> m { }
                elsif (/\G (\S)         ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { return $e . e_qr  ($ope,$1, $3, $2,$4); } # m * *
            }
            die __FILE__, ": Search pattern not terminated\n";
        }
    }

# s///

    # about [cegimosxpradlunbB]* (/cg modifier)
    #
    # P.67 Pattern-Matching Operators
    # of ISBN 0-596-00241-6 Perl in a Nutshell, Second Edition.

    elsif (/\G \b (s) \b /oxgc) {
        my $ope = $1;

        #        $1   $2               $3   $4               $5   $6
        if (/\G (\#) ((?:$qq_char)*?) (\#) ((?:$qq_char)*?) (\#) ([cegimosxpradlunbB]*) /oxgc) { # s# # #
            return e_sub($sub_variable,$1,$2,$3,$3,$4,$5,$6);
        }
        else {
            my $e = '';
            while (not /\G \z/oxgc) {
                if (/\G ((?>\s+)|\#.*) /oxgc) { $e .= $1; }
                elsif (/\G (\() ((?:$qq_paren)*?) (\)) /oxgc) {
                    my @s = ($1,$2,$3);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                                       /oxgc) { $e .= $1; }
                        #           $1   $2                  $3   $4
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\') ((?:$qq_char)*?)    (\') ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\$) ((?:$qq_char)*?)    (\$) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\:) ((?:$qq_char)*?)    (\:) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\@) ((?:$qq_char)*?)    (\@) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                    }
                    die __FILE__, ": Substitution replacement not terminated\n";
                }
                elsif (/\G (\{) ((?:$qq_brace)*?) (\}) /oxgc) {
                    my @s = ($1,$2,$3);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                                       /oxgc) { $e .= $1; }
                        #           $1   $2                  $3   $4
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\') ((?:$qq_char)*?)    (\') ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\$) ((?:$qq_char)*?)    (\$) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\:) ((?:$qq_char)*?)    (\:) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\@) ((?:$qq_char)*?)    (\@) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                    }
                    die __FILE__, ": Substitution replacement not terminated\n";
                }
                elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) {
                    my @s = ($1,$2,$3);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                                       /oxgc) { $e .= $1; }
                        #           $1   $2                  $3   $4
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\') ((?:$qq_char)*?)    (\') ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\$) ((?:$qq_char)*?)    (\$) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                    }
                    die __FILE__, ": Substitution replacement not terminated\n";
                }
                elsif (/\G (\<) ((?:$qq_angle)*?) (\>) /oxgc) {
                    my @s = ($1,$2,$3);
                    while (not /\G \z/oxgc) {
                        if    (/\G ((?>\s+)|\#.*)                                       /oxgc) { $e .= $1; }
                        #           $1   $2                  $3   $4
                        elsif (/\G (\() ((?:$qq_paren)*?)   (\)) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\{) ((?:$qq_brace)*?)   (\}) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\[) ((?:$qq_bracket)*?) (\]) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\<) ((?:$qq_angle)*?)   (\>) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\') ((?:$qq_char)*?)    (\') ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\$) ((?:$qq_char)*?)    (\$) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\:) ((?:$qq_char)*?)    (\:) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\@) ((?:$qq_char)*?)    (\@) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                        elsif (/\G (\S) ((?:$qq_char)*?)    (\1) ([cegimosxpradlunbB]*) /oxgc) { return e_sub($sub_variable,@s,$1,$2,$3,$4); }
                    }
                    die __FILE__, ": Substitution replacement not terminated\n";
                }
                #           $1   $2               $3   $4               $5   $6
                elsif (/\G (\') ((?:$qq_char)*?) (\') ((?:$qq_char)*?) (\') ([cegimosxpradlunbB]*) /oxgc) {
                    return e_sub($sub_variable,$1,$2,$3,$3,$4,$5,$6);
                }
                #           $1            $2               $3   $4               $5   $6
                elsif (/\G ([*\-:?\\^|]) ((?:$qq_char)*?) (\1) ((?:$qq_char)*?) (\1) ([cegimosxpradlunbB]*) /oxgc) {
                    return e_sub($sub_variable,'{',$2,'}','{',$4,'}',$6); # s | | | --> s { } { }
                }
                #           $1   $2               $3   $4               $5   $6
                elsif (/\G (\$) ((?:$qq_char)*?) (\1) ((?:$qq_char)*?) (\1) ([cegimosxpradlunbB]*) /oxgc) {
                    return e_sub($sub_variable,$1,$2,$3,$3,$4,$5,$6);
                }
                #           $1   $2               $3   $4               $5   $6
                elsif (/\G (\S) ((?:$qq_char)*?) (\1) ((?:$qq_char)*?) (\1) ([cegimosxpradlunbB]*) /oxgc) {
                    return e_sub($sub_variable,$1,$2,$3,$3,$4,$5,$6);
                }
            }
            die __FILE__, ": Substitution pattern not terminated\n";
        }
    }

#if ESCAPE_SECOND_OCTET
# do
    elsif (/\G \b do (?= (?>\s*) \{ )                                                         /oxmsgc) { return 'do';                }
    elsif (/\G \b do (?= (?>\s+) (?: q | qq | qx ) \b)                                        /oxmsgc) { return 'Esjis::do';         }
    elsif (/\G \b do (?= (?>\s+) (?>\w+))                                                     /oxmsgc) { return 'do';                }
    elsif (/\G \b do (?= (?>\s*) \$ (?> \w+ (?: ::\w+)* ) \( )                                /oxmsgc) { return 'do';                }
    elsif (/\G \b do \b                                                                       /oxmsgc) { return 'Esjis::do';         }

#endif
# require ignore module
    elsif (/\G \b require ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [#\n])                  /oxmsgc) { return "# require$1$2";     }
    elsif (/\G \b require ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [^\x81-\x9F\xE0-\xFC#]) /oxmsgc) { return "# require$1\n$2";   }
    elsif (/\G \b require ((?>\s+) (?:$ignore_modules)) \b                                    /oxmsgc) { return "# require$1";       }

#if ESCAPE_SECOND_OCTET
# require version number
    elsif (/\G \b require (?>\s+) ((?>0[0-7_]*))                    (?>\s*) ;                 /oxmsgc) { return "require $1;";       }
    elsif (/\G \b require (?>\s+) ((?>[1-9][0-9_]*(?:\.[0-9_]+)*))  (?>\s*) ;                 /oxmsgc) { return "require $1;";       }
    elsif (/\G \b require (?>\s+) ((?>v[0-9][0-9_]*(?:\.[0-9_]+)*)) (?>\s*) ;                 /oxmsgc) { return "require $1;";       }

# require bare package name
    elsif (/\G \b require (?>\s+) ((?>[A-Za-z_]\w* (?: :: [A-Za-z_]\w*)*)) (?>\s*) ;          /oxmsgc) { return "require $1;";       }

# require else
    elsif (/\G \b require                                       (?>\s*) ;                     /oxmsgc) { return 'Esjis::require;';   }
    elsif (/\G \b require \b                                                                  /oxmsgc) { return 'Esjis::require';    }

#endif
# use strict; --> use strict; no strict qw(refs);
    elsif (/\G \b use ((?>\s+) strict .*? ;) ([ \t]* [#\n])                                   /oxmsgc) { return "use$1 no strict qw(refs);$2";   }
    elsif (/\G \b use ((?>\s+) strict .*? ;) ([ \t]* [^\x81-\x9F\xE0-\xFC#])                  /oxmsgc) { return "use$1 no strict qw(refs);\n$2"; }
    elsif (/\G \b use ((?>\s+) strict) \b                                                     /oxmsgc) { return "use$1; no strict qw(refs)";     }

# use 5.12.0; --> use 5.12.0; no strict qw(refs);
    elsif (/\G \b use (?>\s+) ((?>([1-9][0-9_]*)(?:\.([0-9_]+))*))  (?>\s*) ; /oxmsgc) {
        if (($2 >= 6) or (($2 == 5) and ($3 ge '012'))) {
            return "use $1; no strict qw(refs);";
        }
        else {
            return "use $1;";
        }
    }
    elsif (/\G \b use (?>\s+) ((?>v([0-9][0-9_]*)(?:\.([0-9_]+))*)) (?>\s*) ; /oxmsgc) {
        if (($2 >= 6) or (($2 == 5) and ($3 >= 12))) {
            return "use $1; no strict qw(refs);";
        }
        else {
            return "use $1;";
        }
    }

# ignore use module
    elsif (/\G \b use ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [#\n])                  /oxmsgc) { return "# use$1$2";         }
    elsif (/\G \b use ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [^\x81-\x9F\xE0-\xFC#]) /oxmsgc) { return "# use$1\n$2";       }
    elsif (/\G \b use ((?>\s+) (?:$ignore_modules)) \b                                    /oxmsgc) { return "# use$1";           }

# ignore no module
    elsif (/\G \b no  ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [#\n])                  /oxmsgc) { return "# no$1$2";          }
    elsif (/\G \b no  ((?>\s+) (?:$ignore_modules) .*? ;) ([ \t]* [^\x81-\x9F\xE0-\xFC#]) /oxmsgc) { return "# no$1\n$2";        }
    elsif (/\G \b no  ((?>\s+) (?:$ignore_modules)) \b                                    /oxmsgc) { return "# no$1";            }

#if ESCAPE_SECOND_OCTET
# use without import
    elsif (/\G \b use (?>\s+) ((?>0[0-7_]*))                                                        (?>\s*) ; /oxmsgc) { return "use $1;";           }
    elsif (/\G \b use (?>\s+) ((?>[1-9][0-9_]*(?:\.[0-9_]+)*))                                      (?>\s*) ; /oxmsgc) { return "use $1;";           }
    elsif (/\G \b use (?>\s+) ((?>v[0-9][0-9_]*(?:\.[0-9_]+)*))                                     (?>\s*) ; /oxmsgc) { return "use $1;";           }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*)            (\()          (?>\s*) \) (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\()          (?>\s*) \) (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\{)          (?>\s*) \} (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\[)          (?>\s*) \] (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\<)          (?>\s*) \> (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) ([\x21-\x3F]) (?>\s*) \2 (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\S)          (?>\s*) \2 (?>\s*) ; /oxmsgc) { return e_use_noimport($1);  }

# use with import no parameter
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*))                                             (?>\s*) ; /oxmsgc) { return e_use_noparam($1);   }

# use with import parameters
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\()    [^\x81-\x9F\xE0-\xFC)]* \)) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\')    [^\x81-\x9F\xE0-\xFC']* \') (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\")    [^\x81-\x9F\xE0-\xFC"]* \") (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\()    [^\x81-\x9F\xE0-\xFC)]* \)) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\{)    (?:$q_char)*?           \}) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\[)    (?:$q_char)*?           \]) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\<)    [^\x81-\x9F\xE0-\xFC>]* \>) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) ([\x21-\x3F]) .*?               \3) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }
    elsif (/\G \b use (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\S)    (?:$q_char)*?           \3) (?>\s*) ; /oxmsgc) { return e_use($1,$2); }

# no without unimport
    elsif (/\G \b no  (?>\s+) ((?>0[0-7_]*))                                                        (?>\s*) ; /oxmsgc) { return "no $1;";            }
    elsif (/\G \b no  (?>\s+) ((?>[1-9][0-9_]*(?:\.[0-9_]+)*))                                      (?>\s*) ; /oxmsgc) { return "no $1;";            }
    elsif (/\G \b no  (?>\s+) ((?>v[0-9][0-9_]*(?:\.[0-9_]+)*))                                     (?>\s*) ; /oxmsgc) { return "no $1;";            }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*)            (\()          (?>\s*) \) (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\()          (?>\s*) \) (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\{)          (?>\s*) \} (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\[)          (?>\s*) \] (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\<)          (?>\s*) \> (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) ([\x21-\x3F]) (?>\s*) \2 (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) qw (?>\s*) (\S)          (?>\s*) \2 (?>\s*) ; /oxmsgc) { return e_no_nounimport($1); }

# no with unimport no parameter
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*))                                             (?>\s*) ; /oxmsgc) { return e_no_noparam($1);    }

# no with unimport parameters
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\()    [^\x81-\x9F\xE0-\xFC)]* \)) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\')    [^\x81-\x9F\xE0-\xFC']* \') (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s*) (                          (\")    [^\x81-\x9F\xE0-\xFC"]* \") (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\()    [^\x81-\x9F\xE0-\xFC)]* \)) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\{)    (?:$q_char)*?           \}) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\[)    (?:$q_char)*?           \]) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\<)    [^\x81-\x9F\xE0-\xFC>]* \>) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) ([\x21-\x3F]) .*?               \3) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }
    elsif (/\G \b no  (?>\s+) ((?>[A-Z]\w*(?: ::\w+)*)) (?>\s+) ((?: q | qq | qw ) (?>\s*) (\S)    (?:$q_char)*?           \3) (?>\s*) ; /oxmsgc) { return e_no($1,$2);  }

#endif /* #if ESCAPE_SECOND_OCTET */
# use else
    elsif (/\G \b use \b /oxmsgc) { return "use"; }

# use else
    elsif (/\G \b no  \b /oxmsgc) { return "no";  }

# ''
    elsif (/\G (?<![\w\$\@\%\&\*]) (\') /oxgc) {
        my $q_string = '';
        while (not /\G \z/oxgc) {
            if    (/\G (\\\\)                  /oxgc) { $q_string .= $1;                   }
            elsif (/\G (\\\')                  /oxgc) { $q_string .= $1;                   }
            elsif (/\G \'                      /oxgc) { return e_q('', "'","'",$q_string); }
            elsif (/\G ($q_char)               /oxgc) { $q_string .= $1;                   }
        }
        die __FILE__, ": Can't find string terminator anywhere before EOF\n";
    }

# ""
    elsif (/\G (\") /oxgc) {
        my $qq_string = '';
        while (not /\G \z/oxgc) {
            if    (/\G (\\\\)                  /oxgc) { $qq_string .= $1;                    }
            elsif (/\G (\\\")                  /oxgc) { $qq_string .= $1;                    }
            elsif (/\G \"                      /oxgc) { return e_qq('', '"','"',$qq_string); }
            elsif (/\G ($q_char)               /oxgc) { $qq_string .= $1;                    }
        }
        die __FILE__, ": Can't find string terminator anywhere before EOF\n";
    }

# ``
    elsif (/\G (\`) /oxgc) {
        my $qx_string = '';
        while (not /\G \z/oxgc) {
            if    (/\G (\\\\)                  /oxgc) { $qx_string .= $1;                    }
            elsif (/\G (\\\`)                  /oxgc) { $qx_string .= $1;                    }
            elsif (/\G \`                      /oxgc) { return e_qq('', '`','`',$qx_string); }
            elsif (/\G ($q_char)               /oxgc) { $qx_string .= $1;                    }
        }
        die __FILE__, ": Can't find string terminator anywhere before EOF\n";
    }

# //   --- not divide operator (num / num), not defined-or
    elsif (($slash eq 'm//') and /\G (\/) /oxgc) {
        my $regexp = '';
        while (not /\G \z/oxgc) {
            if    (/\G (\\\\)                  /oxgc) { $regexp .= $1;                       }
            elsif (/\G (\\\/)                  /oxgc) { $regexp .= $1;                       }
            elsif (/\G \/ ([cgimosxpadlunbB]*) /oxgc) { return e_qr('', '/','/',$regexp,$1); }
            elsif (/\G ($q_char)               /oxgc) { $regexp .= $1;                       }
        }
        die __FILE__, ": Search pattern not terminated\n";
    }

# ??   --- not conditional operator (condition ? then : else)
    elsif (($slash eq 'm//') and /\G (\?) /oxgc) {
        my $regexp = '';
        while (not /\G \z/oxgc) {
            if    (/\G (\\\\)                  /oxgc) { $regexp .= $1;                       }
            elsif (/\G (\\\?)                  /oxgc) { $regexp .= $1;                       }
            elsif (/\G \? ([cgimosxpadlunbB]*) /oxgc) { return e_qr('m','?','?',$regexp,$1); }
            elsif (/\G ($q_char)               /oxgc) { $regexp .= $1;                       }
        }
        die __FILE__, ": Search pattern not terminated\n";
    }

# <<>> (a safer ARGV)
    elsif (/\G ( <<>> ) /oxgc)                         { $slash = 'm//'; return $1;          }

# << (bit shift)   --- not here document
    elsif (/\G ( << (?>\s*) ) (?= [0-9\$\@\&] ) /oxgc) { $slash = 'm//'; return $1;          }

# <<~'HEREDOC'
    elsif (/\G ( <<~ [\t ]* '([a-zA-Z_0-9]*)' ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
            my $heredoc = $1;
            my $indent  = $2;
            $heredoc =~ s{^$indent}{}msg; # no /ox
            push @heredoc, $heredoc . qq{\n$delimiter\n};
            push @heredoc_delimiter, qq{\\s*$delimiter};
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return qq{<<'$delimiter'};
    }

# <<~\HEREDOC

    # P.66 2.6.6. "Here" Documents
    # in Chapter 2: Bits and Pieces
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.73 "Here" Documents
    # in Chapter 2: Bits and Pieces
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    elsif (/\G ( <<~ \\([a-zA-Z_0-9]+) ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
            my $heredoc = $1;
            my $indent  = $2;
            $heredoc =~ s{^$indent}{}msg; # no /ox
            push @heredoc, $heredoc . qq{\n$delimiter\n};
            push @heredoc_delimiter, qq{\\s*$delimiter};
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return qq{<<\\$delimiter};
    }

# <<~"HEREDOC"
    elsif (/\G ( <<~ [\t ]* "([a-zA-Z_0-9]*)" ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
            my $heredoc = $1;
            my $indent  = $2;
            $heredoc =~ s{^$indent}{}msg; # no /ox
            push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
            push @heredoc_delimiter, qq{\\s*$delimiter};
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return qq{<<"$delimiter"};
    }

# <<~HEREDOC
    elsif (/\G ( <<~ ([a-zA-Z_0-9]+) ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
            my $heredoc = $1;
            my $indent  = $2;
            $heredoc =~ s{^$indent}{}msg; # no /ox
            push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
            push @heredoc_delimiter, qq{\\s*$delimiter};
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return qq{<<$delimiter};
    }

# <<~`HEREDOC`
    elsif (/\G ( <<~ [\t ]* `([a-zA-Z_0-9]*)` ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
            my $heredoc = $1;
            my $indent  = $2;
            $heredoc =~ s{^$indent}{}msg; # no /ox
            push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
            push @heredoc_delimiter, qq{\\s*$delimiter};
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return qq{<<`$delimiter`};
    }

# <<'HEREDOC'
    elsif (/\G ( << '([a-zA-Z_0-9]*)' ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
            push @heredoc, $1 . qq{\n$delimiter\n};
            push @heredoc_delimiter, $delimiter;
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return $here_quote;
    }

# <<\HEREDOC

    # P.66 2.6.6. "Here" Documents
    # in Chapter 2: Bits and Pieces
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.73 "Here" Documents
    # in Chapter 2: Bits and Pieces
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    elsif (/\G ( << \\([a-zA-Z_0-9]+) ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
            push @heredoc, $1 . qq{\n$delimiter\n};
            push @heredoc_delimiter, $delimiter;
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return $here_quote;
    }

# <<"HEREDOC"
    elsif (/\G ( << "([a-zA-Z_0-9]*)" ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
            push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
            push @heredoc_delimiter, $delimiter;
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return $here_quote;
    }

# <<HEREDOC
    elsif (/\G ( << ([a-zA-Z_0-9]+) ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
            push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
            push @heredoc_delimiter, $delimiter;
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return $here_quote;
    }

# <<`HEREDOC`
    elsif (/\G ( << `([a-zA-Z_0-9]*)` ) /oxgc) {
        $slash = 'm//';
        my $here_quote = $1;
        my $delimiter  = $2;

        # get here document
        if ($here_script eq '') {
            $here_script = CORE::substr $_, pos $_;
            $here_script =~ s/.*?\n//oxm;
        }
        if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
            push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
            push @heredoc_delimiter, $delimiter;
        }
        else {
            die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
        }
        return $here_quote;
    }

# <<= <=> <= < operator
    elsif (/\G ( <<= | <=> | <= | < ) (?= (?>\s*) [A-Za-z_0-9'"`\$\@\&\*\(\+\-] )/oxgc) {
        return $1;
    }

# <FILEHANDLE>
    elsif (/\G (<[\$]?[A-Za-z_][A-Za-z_0-9]*>) /oxgc) {
        return $1;
    }

# <WILDCARD> --- glob

    # avoid "Error: Runtime exception" of perl version 5.005_03

    elsif (/\G < ((?:[^\x81-\x9F\xE0-\xFC>\0\a\e\f\n\r\t]|[\x81-\x9F\xE0-\xFC][\x00-\xFF])+?) > /oxgc) {
        return 'Esjis::glob("' . $1 . '")';
    }

# __DATA__
    elsif (/\G ^ ( __DATA__ \n .*) \z /oxmsgc) { return $1; }

# __END__
    elsif (/\G ^ ( __END__  \n .*) \z /oxmsgc) { return $1; }

# \cD Control-D

    # P.68 2.6.8. Other Literal Tokens
    # in Chapter 2: Bits and Pieces
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.76 Other Literal Tokens
    # in Chapter 2: Bits and Pieces
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    elsif (/\G   ( \cD         .*) \z /oxmsgc) { return $1; }

# \cZ Control-Z
    elsif (/\G   ( \cZ         .*) \z /oxmsgc) { return $1; }

    # any operator before div
    elsif (/\G (
            -- | \+\+ |
            [\)\}\]]

            ) /oxgc) { $slash = 'div'; return $1; }

    # yada-yada or triple-dot operator
    elsif (/\G (
            \.\.\.

            ) /oxgc) { $slash = 'm//'; return q{die('Unimplemented')}; }

    # any operator before m//

    # //, //= (defined-or)

    # P.164 Logical Operators
    # in Chapter 10: More Control Structures
    # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

    # P.119 C-Style Logical (Short-Circuit) Operators
    # in Chapter 3: Unary and Binary Operators
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # (and so on)

    # ~~

    # P.221 The Smart Match Operator
    # in Chapter 15: Smart Matching and given-when
    # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

    # P.112 Smartmatch Operator
    # in Chapter 3: Unary and Binary Operators
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    # (and so on)

    elsif (/\G ((?>

            !~~ | !~ | != | ! |
            %= | % |
            &&= | && | &= | &\.= | &\. | & |
            -= | -> | - |
            :(?>\s*)= |
            : |
            <<>> |
            <<= | <=> | <= | < |
            == | => | =~ | = |
            >>= | >> | >= | > |
            \*\*= | \*\* | \*= | \* |
            \+= | \+ |
            \.\. | \.= | \. |
            \/\/= | \/\/ |
            \/= | \/ |
            \? |
            \\ |
            \^= | \^\.= | \^\. | \^ |
            \b x= |
            \|\|= | \|\| | \|= | \|\.= | \|\. | \| |
            ~~ | ~\. | ~ |
            \b(?: and | cmp | eq | ge | gt | le | lt | ne | not | or | xor | x )\b |
            \b(?: print )\b |

            [,;\(\{\[]

            )) /oxgc) { $slash = 'm//'; return $1; }

    # other any character
    elsif (/\G ($q_char) /oxgc) { $slash = 'div'; return $1; }

    # system error
    else {
        die __FILE__, ": Oops, this shouldn't happen!\n";
    }
}

# escape ShiftJIS string
sub e_string {
    my($string) = @_;
    my $e_string = '';

    local $slash = 'm//';

    # P.1024 Appendix W.10 Multibyte Processing
    # of ISBN 1-56592-224-7 CJKV Information Processing
    # (and so on)

    my @char = $string =~ / \G (?>[^\x81-\x9F\xE0-\xFC\\]|\\$q_char|$q_char) /oxmsg;

    # without { ... }
    if (not (grep(/\A \{ \z/xms, @char) and grep(/\A \} \z/xms, @char))) {
        if ($string !~ /<</oxms) {
            return $string;
        }
    }

E_STRING_LOOP:
    while ($string !~ /\G \z/oxgc) {
        if (0) {
        }

# $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> @{[Esjis::PREMATCH()]}
        elsif ($string =~ /\G ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  \b | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  ) /oxmsgc) {
            $e_string .= q{Esjis::PREMATCH()};
            $slash = 'div';
        }

# $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> @{[Esjis::MATCH()]}
        elsif ($string =~ /\G ( \$& | \$\{&\} | \$ (?>\s*) MATCH     \b | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) /oxmsgc) {
            $e_string .= q{Esjis::MATCH()};
            $slash = 'div';
        }

# $', ${'} --> $', ${'}
        elsif ($string =~ /\G ( \$' | \$\{'\}                                                                                                     ) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }

# $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> @{[Esjis::POSTMATCH()]}
        elsif ($string =~ /\G (                 \$ (?>\s*) POSTMATCH \b | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) /oxmsgc) {
            $e_string .= q{Esjis::POSTMATCH()};
            $slash = 'div';
        }

# bareword
        elsif ($string =~ /\G ( \{ (?>\s*) (?: tr | index | rindex | reverse ) (?>\s*) \} ) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }

# $0 --> $0
        elsif ($string =~ /\G ( \$ 0 ) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }
        elsif ($string =~ /\G ( \$ \{ (?>\s*) 0 (?>\s*) \} ) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }

# $$ --> $$
        elsif ($string =~ /\G ( \$ \$ ) (?![\w\{]) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }

# $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
# $1, $2, $3 --> $1, $2, $3 otherwise
        elsif ($string =~ /\G \$ ((?>[1-9][0-9]*)) /oxmsgc) {
            $e_string .= e_capture($1);
            $slash = 'div';
        }
        elsif ($string =~ /\G \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} /oxmsgc) {
            $e_string .= e_capture($1);
            $slash = 'div';
        }

# $$foo[ ... ] --> $ $foo->[ ... ]
        elsif ($string =~ /\G \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ .+? \] ) /oxmsgc) {
            $e_string .= e_capture($1.'->'.$2);
            $slash = 'div';
        }

# $$foo{ ... } --> $ $foo->{ ... }
        elsif ($string =~ /\G \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ .+? \} ) /oxmsgc) {
            $e_string .= e_capture($1.'->'.$2);
            $slash = 'div';
        }

# $$foo
        elsif ($string =~ /\G \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) /oxmsgc) {
            $e_string .= e_capture($1);
            $slash = 'div';
        }

# ${ foo }
        elsif ($string =~ /\G \$ (?>\s*) \{ ((?> \s* [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* \s* )) \} /oxmsgc) {
            $e_string .= '${' . $1 . '}';
            $slash = 'div';
        }

# ${ ... }
        elsif ($string =~ /\G \$ (?>\s*) \{ (?>\s*) ( $qq_brace ) (?>\s*) \} /oxmsgc) {
            $e_string .= e_capture($1);
            $slash = 'div';
        }

# variable or function
        #                             $ @ % & *     $ #
        elsif ($string =~ /\G ( (?: [\$\@\%\&\*] | \$\# | -> | \b sub \b) (?>\s*) (?: split | chop | index | rindex | lc | uc | fc | chr | ord | reverse | getc | tr | y | q | qq | qx | qw | m | s | qr | glob | lstat | opendir | stat | unlink | chdir ) ) \b /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }
        #                           $ $ $ $ $ $ $ $ $ $ $ $ $ $
        #                           $ @ # \ ' " / ? ( ) [ ] < >
        elsif ($string =~ /\G ( \$[\$\@\#\\\'\"\/\?\(\)\[\]\<\>] ) /oxmsgc) {
            $e_string .= $1;
            $slash = 'div';
        }

# subroutines of package Esjis
        elsif ($string =~ /\G \b (CORE:: | ->(>?\s*) (?: atan2 | [a-z]{2,})) \b       /oxgc) { $e_string .= $1;                  $slash = 'm//'; }
        elsif ($string =~ /\G \b Char::eval       (?= (?>\s*) \{ )                    /oxgc) { $e_string .= 'eval';              $slash = 'm//'; }
        elsif ($string =~ /\G \b Sjis::eval       (?= (?>\s*) \{ )                    /oxgc) { $e_string .= 'eval';              $slash = 'm//'; }
        elsif ($string =~ /\G \b Char::eval \b                                        /oxgc) { $e_string .= 'eval Char::escape'; $slash = 'm//'; }
        elsif ($string =~ /\G \b Sjis::eval \b                                        /oxgc) { $e_string .= 'eval Sjis::escape'; $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::substr \b                                     /oxgc) { $e_string .= 'substr';            $slash = 'm//'; }
        elsif ($string =~ /\G \b chop \b                                              /oxgc) { $e_string .= 'Esjis::chop';       $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::index \b                                      /oxgc) { $e_string .= 'index';             $slash = 'm//'; }
        elsif ($string =~ /\G \b Char::index \b                                       /oxgc) { $e_string .= 'Char::index';       $slash = 'm//'; }
        elsif ($string =~ /\G \b Sjis::index \b                                       /oxgc) { $e_string .= 'Sjis::index';       $slash = 'm//'; }
        elsif ($string =~ /\G \b index \b                                             /oxgc) { $e_string .= 'Esjis::index';      $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::rindex \b                                     /oxgc) { $e_string .= 'rindex';            $slash = 'm//'; }
        elsif ($string =~ /\G \b Char::rindex \b                                      /oxgc) { $e_string .= 'Char::rindex';      $slash = 'm//'; }
        elsif ($string =~ /\G \b Sjis::rindex \b                                      /oxgc) { $e_string .= 'Sjis::rindex';      $slash = 'm//'; }
        elsif ($string =~ /\G \b rindex \b                                            /oxgc) { $e_string .= 'Esjis::rindex';     $slash = 'm//'; }
        elsif ($string =~ /\G \b lc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::lc';         $slash = 'm//'; }
        elsif ($string =~ /\G \b lcfirst (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::lcfirst';    $slash = 'm//'; }
        elsif ($string =~ /\G \b uc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::uc';         $slash = 'm//'; }
        elsif ($string =~ /\G \b ucfirst (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::ucfirst';    $slash = 'm//'; }
        elsif ($string =~ /\G \b fc      (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::fc';         $slash = 'm//'; }
#if ESCAPE_SECOND_OCTET
        elsif ($string =~ /\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                                                                            (?>\s*) (\") ((?:$qq_char)+?)             (\") /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('',  $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\#) ((?:$qq_char)+?)             (\#) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\() ((?:$qq_paren)+?)            (\)) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\{) ((?:$qq_brace)+?)            (\}) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\[) ((?:$qq_bracket)+?)          (\]) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\<) ((?:$qq_angle)+?)            (\>) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) qq (?>\s*) (\S) ((?:$qq_char)+?)             (\2) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }

        elsif ($string =~ /\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                                                                            (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)    (\') /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('',  $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)    (\#) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)   (\)) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)   (\}) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?) (\]) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)   (\>) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) q  (?>\s*) (\S) ((?:\\\2|\\\\|$q_char)+?)    (\2) /oxgc) { $e_string .= "Esjis::filetest(qw($1)," . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }

        elsif ($string =~ /\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+)) (\$ (?> \w+ (?: ::\w+)*) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ) )*) /oxgc)
                                                                                                                                  { $e_string .= "Esjis::filetest(qw($1),$2)"; $slash = 'm//'; }
        elsif ($string =~ /\G (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+)) \( ((?:$qq_paren)*?) \)   /oxgc) { $e_string .= "Esjis::filetest(qw($1),$2)"; $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) (?= [a-z]+)                                       /oxgc) { $e_string .= "Esjis::filetest qw($1),";    $slash = 'm//'; }
        elsif ($string =~ /\G ((?:-[rwxoRWXOezfdlpSbcugkTB](?>\s+)){2,}) ((?>\w+))                                         /oxgc) { $e_string .= "Esjis::filetest(qw($1),$2)"; $slash = 'm//'; }

        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC])            (?>\s*) (\") ((?:$qq_char)+?)                (\") /oxgc) { $e_string .= "Esjis::$1(" . e_qq('',  $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\#) ((?:$qq_char)+?)                (\#) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\() ((?:$qq_paren)+?)               (\)) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\{) ((?:$qq_brace)+?)               (\}) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\[) ((?:$qq_bracket)+?)             (\]) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\<) ((?:$qq_angle)+?)               (\>) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) qq (?>\s*) (\S) ((?:$qq_char)+?)                (\2) /oxgc) { $e_string .= "Esjis::$1(" . e_qq('qq',$2,$4,$3) . ")"; $slash = 'm//'; }

        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC])            (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)       (\') /oxgc) { $e_string .= "Esjis::$1(" . e_q ('',  $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)       (\#) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)      (\)) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)      (\}) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?)    (\]) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)      (\>) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) q  (?>\s*) (\S) ((?:\\\2|\\\\|$q_char)+?)       (\2) /oxgc) { $e_string .= "Esjis::$1(" . e_q ('q', $2,$4,$3) . ")"; $slash = 'm//'; }

        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s*) (\$ (?> \w+ (?: ::\w+)*) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ) )*) /oxgc)
                                                                                                                                  { $e_string .= "Esjis::$1($2)";      $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s*) \( ((?:$qq_paren)*?) \)                              /oxgc) { $e_string .= "Esjis::$1($2)";      $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?= (?>\s+) [a-z]+)                                          /oxgc) { $e_string .= "Esjis::$1";          $slash = 'm//'; }
        elsif ($string =~ /\G -([rwxoRWXOezsfdlpSbcugkTBMAC]) (?>\s+) ((?>\w+))                                            /oxgc) { $e_string .= "Esjis::$1(::"."$2)"; $slash = 'm//'; }
        elsif ($string =~ /\G -(t)                            (?>\s+) ((?>\w+))                                            /oxgc) { $e_string .= "-t $2";              $slash = 'm//'; }
        elsif ($string =~ /\G \b lstat         (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(])                                /oxgc) { $e_string .= 'Esjis::lstat';       $slash = 'm//'; }
        elsif ($string =~ /\G \b stat          (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(])                                /oxgc) { $e_string .= 'Esjis::stat';        $slash = 'm//'; }
#endif /* #if ESCAPE_SECOND_OCTET */

        # "-s '' ..." means file test "-s 'filename' ..." (not means "- s/// ...")
        elsif ($string =~ /\G -s                                         (?>\s*) (\") ((?:$qq_char)+?)                (\") /oxgc) { $e_string .= '-s ' . e_qq('',  $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\#) ((?:$qq_char)+?)                (\#) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\() ((?:$qq_paren)+?)               (\)) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\{) ((?:$qq_brace)+?)               (\}) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\[) ((?:$qq_bracket)+?)             (\]) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\<) ((?:$qq_angle)+?)               (\>) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) qq (?>\s*) (\S) ((?:$qq_char)+?)                (\1) /oxgc) { $e_string .= '-s ' . e_qq('qq',$1,$3,$2); $slash = 'm//'; }

        elsif ($string =~ /\G -s                                         (?>\s*) (\') ((?:\\\'|\\\\|$q_char)+?)       (\') /oxgc) { $e_string .= '-s ' . e_q ('',  $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\#) ((?:\\\#|\\\\|$q_char)+?)       (\#) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\() ((?:\\\)|\\\\|$q_paren)+?)      (\)) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\{) ((?:\\\}|\\\\|$q_brace)+?)      (\}) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\[) ((?:\\\]|\\\\|$q_bracket)+?)    (\]) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\<) ((?:\\\>|\\\\|$q_angle)+?)      (\>) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) q  (?>\s*) (\S) ((?:\\\1|\\\\|$q_char)+?)       (\1) /oxgc) { $e_string .= '-s ' . e_q ('q', $1,$3,$2); $slash = 'm//'; }

        elsif ($string =~ /\G -s                              (?>\s*) (\$ (?> \w+ (?: ::\w+)*) (?: (?: ->)? (?: [\$\@\%\&\*]\* | \$\#\* | \( (?:$qq_paren)*? \) | [\@\%\*]? \{ (?:$qq_brace)+? \} | [\@\%]? \[ (?:$qq_bracket)+? \] ))*) /oxgc)
                                                                                                                                  { $e_string .= "-s $1";   $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s*) \( ((?:$qq_paren)*?) \)                              /oxgc) { $e_string .= "-s ($1)"; $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?= (?>\s+) [a-z]+)                                          /oxgc) { $e_string .= '-s';      $slash = 'm//'; }
        elsif ($string =~ /\G -s                              (?>\s+) ((?>\w+))                                            /oxgc) { $e_string .= "-s $1";   $slash = 'm//'; }

        elsif ($string =~ /\G \b bytes::length (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'length';               $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::chr    (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'chr';                  $slash = 'm//'; }
        elsif ($string =~ /\G \b chr           (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::chr';           $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::ord    (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'ord';                  $slash = 'div'; }
        elsif ($string =~ /\G \b ord           (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= $function_ord;          $slash = 'div'; }
        elsif ($string =~ /\G \b glob          (?= (?>\s+)[A-Za-z_]|(?>\s*)['"`\$\@\&\*\(]) /oxgc) { $e_string .= 'Esjis::glob';          $slash = 'm//'; }
        elsif ($string =~ /\G \b lc \b                                                      /oxgc) { $e_string .= 'Esjis::lc_';               $slash = 'm//'; }
        elsif ($string =~ /\G \b lcfirst \b                                                 /oxgc) { $e_string .= 'Esjis::lcfirst_';          $slash = 'm//'; }
        elsif ($string =~ /\G \b uc \b                                                      /oxgc) { $e_string .= 'Esjis::uc_';               $slash = 'm//'; }
        elsif ($string =~ /\G \b ucfirst \b                                                 /oxgc) { $e_string .= 'Esjis::ucfirst_';          $slash = 'm//'; }
        elsif ($string =~ /\G \b fc \b                                                      /oxgc) { $e_string .= 'Esjis::fc_';               $slash = 'm//'; }
#if ESCAPE_SECOND_OCTET
        elsif ($string =~ /\G \b lstat \b                                                   /oxgc) { $e_string .= 'Esjis::lstat_';            $slash = 'm//'; }
        elsif ($string =~ /\G \b stat \b                                                    /oxgc) { $e_string .= 'Esjis::stat_';             $slash = 'm//'; }
        elsif ($string =~ /\G    (-[rwxoRWXOezfdlpSbcugkTB] (?>(?:\s+ -[rwxoRWXOezfdlpSbcugkTB])+))
                                                                 \b                         /oxgc) { $e_string .= "Esjis::filetest_(qw($1))"; $slash = 'm//'; }
        elsif ($string =~ /\G    -([rwxoRWXOezsfdlpSbcugkTBMAC]) \b                         /oxgc) { $e_string .= "Esjis::${1}_";             $slash = 'm//'; }
#endif
        elsif ($string =~ /\G    -s                              \b                         /oxgc) { $e_string .= '-s ';                      $slash = 'm//'; }

        elsif ($string =~ /\G \b bytes::length \b                                           /oxgc) { $e_string .= 'length';                   $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::chr \b                                              /oxgc) { $e_string .= 'chr';                      $slash = 'm//'; }
        elsif ($string =~ /\G \b chr \b                                                     /oxgc) { $e_string .= 'Esjis::chr_';              $slash = 'm//'; }
        elsif ($string =~ /\G \b bytes::ord \b                                              /oxgc) { $e_string .= 'ord';                      $slash = 'div'; }
        elsif ($string =~ /\G \b ord \b                                                     /oxgc) { $e_string .= $function_ord_;             $slash = 'div'; }
        elsif ($string =~ /\G \b glob \b                                                    /oxgc) { $e_string .= 'Esjis::glob_';             $slash = 'm//'; }
        elsif ($string =~ /\G \b reverse \b                                                 /oxgc) { $e_string .= $function_reverse;          $slash = 'm//'; }
        elsif ($string =~ /\G \b getc \b                                                    /oxgc) { $e_string .= $function_getc;             $slash = 'm//'; }
#if ESCAPE_SECOND_OCTET
        elsif ($string =~ /\G \b opendir ((?>\s*) \( (?>\s*)) (?=[A-Za-z_])                 /oxgc) { $e_string .= "Esjis::opendir$1*";        $slash = 'm//'; }
        elsif ($string =~ /\G \b opendir ((?>\s+))            (?=[A-Za-z_])                 /oxgc) { $e_string .= "Esjis::opendir$1*";        $slash = 'm//'; }
        elsif ($string =~ /\G \b unlink \b                                                  /oxgc) { $e_string .= 'Esjis::unlink';            $slash = 'm//'; }

# chdir
        elsif ($string =~ /\G \b (chdir) \b (?! (?>\s*) => ) /oxgc) {
            $slash = 'm//';

            $e_string .= 'Esjis::chdir';

            while ($string =~ /\G ( (?>\s+) | \( | \#.* ) /oxgc) {
                $e_string .= $1;
            }

# end of chdir
            if    ($string =~ /\G (?= [,;\)\}\]] )          /oxgc) { return $e_string;                               }

# chdir scalar value
            elsif ($string =~ /\G ( [\$\@\&\*] $qq_scalar ) /oxgc) { $e_string .= e_string($1);  next E_STRING_LOOP; }

# chdir qq//
            elsif ($string =~ /\G \b (qq) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) /oxgc)                             { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq# #  --> qr # #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                              /oxgc) { $e_string .= $1; }
                        elsif ($string =~ /\G (\()          ((?:$qq_paren)*?)   (\))      /oxgc) { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq ( ) --> qr ( )
                        elsif ($string =~ /\G (\{)          ((?:$qq_brace)*?)   (\})      /oxgc) { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq { } --> qr { }
                        elsif ($string =~ /\G (\[)          ((?:$qq_bracket)*?) (\])      /oxgc) { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq [ ] --> qr [ ]
                        elsif ($string =~ /\G (\<)          ((?:$qq_angle)*?)   (\>)      /oxgc) { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq < > --> qr < >
                        elsif ($string =~ /\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1)      /oxgc) { $e_string .= e_chdir('qq','{','}',$2); next E_STRING_LOOP; } # qq | | --> qr { }
                        elsif ($string =~ /\G (\S)          ((?:$qq_char)*?)    (\1)      /oxgc) { $e_string .= e_chdir('qq',$1,$3,$2);   next E_STRING_LOOP; } # qq * * --> qr * *
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }

# chdir q//
            elsif ($string =~ /\G \b (q) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc)                    { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q# #  --> qr # #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                              /oxgc) { $e_string .= $1; }
                        elsif ($string =~ /\G (\() ((?:\\\\|\\\)|\\\(|$q_paren)*?)   (\)) /oxgc) { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q ( ) --> qr ( )
                        elsif ($string =~ /\G (\{) ((?:\\\\|\\\}|\\\{|$q_brace)*?)   (\}) /oxgc) { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q { } --> qr { }
                        elsif ($string =~ /\G (\[) ((?:\\\\|\\\]|\\\[|$q_bracket)*?) (\]) /oxgc) { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q [ ] --> qr [ ]
                        elsif ($string =~ /\G (\<) ((?:\\\\|\\\>|\\\<|$q_angle)*?)   (\>) /oxgc) { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q < > --> qr < >
                        elsif ($string =~ /\G ([*\-:?\\^|])       ((?:$q_char)*?)    (\1) /oxgc) { $e_string .= e_chdir_q('q','{','}',$2); next E_STRING_LOOP; } # q | | --> qr { }
                        elsif ($string =~ /\G (\S) ((?:\\\\|\\\1|     $q_char)*?)    (\1) /oxgc) { $e_string .= e_chdir_q('q',$1,$3,$2);   next E_STRING_LOOP; } # q * * --> qr * *
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }

# chdir ''
            elsif ($string =~ /\G (\') /oxgc) {
                my $q_string = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G (\\\\)    /oxgc) { $q_string .= $1; }
                    elsif ($string =~ /\G (\\\')    /oxgc) { $q_string .= $1; }
                    elsif ($string =~ /\G \'        /oxgc)                                       { $e_string .= e_chdir_q('',"'","'",$q_string); next E_STRING_LOOP; }
                    elsif ($string =~ /\G ($q_char) /oxgc) { $q_string .= $1; }
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }

# chdir ""
            elsif ($string =~ /\G (\") /oxgc) {
                my $qq_string = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G (\\\\)    /oxgc) { $qq_string .= $1; }
                    elsif ($string =~ /\G (\\\")    /oxgc) { $qq_string .= $1; }
                    elsif ($string =~ /\G \"        /oxgc)                                       { $e_string .= e_chdir('','"','"',$qq_string); next E_STRING_LOOP; }
                    elsif ($string =~ /\G ($q_char) /oxgc) { $qq_string .= $1; }
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

#endif
# split
        elsif ($string =~ /\G \b (split) \b (?! (?>\s*) => ) /oxgc) {
            $slash = 'm//';

            my $e = '';
            while ($string =~ /\G ( (?>\s+) | \( | \#.* ) /oxgc) {
                $e .= $1;
            }

# end of split
            if    ($string =~ /\G (?= [,;\)\}\]] )          /oxgc) { return 'Esjis::split' . $e;                                           }

# split scalar value
            elsif ($string =~ /\G ( [\$\@\&\*] $qq_scalar ) /oxgc) { $e_string .= 'Esjis::split' . $e . e_string($1);  next E_STRING_LOOP; }

# split literal space
            elsif ($string =~ /\G \b qq           (\#) [ ] (\#) /oxgc) { $e_string .= 'Esjis::split' . $e . qq  {qq$1 $2}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b qq ((?>\s*)) (\() [ ] (\)) /oxgc) { $e_string .= 'Esjis::split' . $e . qq{$1qq$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b qq ((?>\s*)) (\{) [ ] (\}) /oxgc) { $e_string .= 'Esjis::split' . $e . qq{$1qq$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b qq ((?>\s*)) (\[) [ ] (\]) /oxgc) { $e_string .= 'Esjis::split' . $e . qq{$1qq$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b qq ((?>\s*)) (\<) [ ] (\>) /oxgc) { $e_string .= 'Esjis::split' . $e . qq{$1qq$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b qq ((?>\s*)) (\S) [ ] (\2) /oxgc) { $e_string .= 'Esjis::split' . $e . qq{$1qq$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q            (\#) [ ] (\#) /oxgc) { $e_string .= 'Esjis::split' . $e . qq   {q$1 $2}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q  ((?>\s*)) (\() [ ] (\)) /oxgc) { $e_string .= 'Esjis::split' . $e . qq {$1q$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q  ((?>\s*)) (\{) [ ] (\}) /oxgc) { $e_string .= 'Esjis::split' . $e . qq {$1q$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q  ((?>\s*)) (\[) [ ] (\]) /oxgc) { $e_string .= 'Esjis::split' . $e . qq {$1q$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q  ((?>\s*)) (\<) [ ] (\>) /oxgc) { $e_string .= 'Esjis::split' . $e . qq {$1q$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G \b q  ((?>\s*)) (\S) [ ] (\2) /oxgc) { $e_string .= 'Esjis::split' . $e . qq {$1q$2 $3}; next E_STRING_LOOP; }
            elsif ($string =~ /\G                    ' [ ] '    /oxgc) { $e_string .= 'Esjis::split' . $e . qq     {' '};  next E_STRING_LOOP; }
            elsif ($string =~ /\G                    " [ ] "    /oxgc) { $e_string .= 'Esjis::split' . $e . qq     {" "};  next E_STRING_LOOP; }

# split qq//
            elsif ($string =~ /\G \b (qq) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) /oxgc)                        { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq# #  --> qr # #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                         /oxgc) { $e_string .= $e . $1; }
                        elsif ($string =~ /\G (\()          ((?:$qq_paren)*?)   (\)) /oxgc) { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq ( ) --> qr ( )
                        elsif ($string =~ /\G (\{)          ((?:$qq_brace)*?)   (\}) /oxgc) { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq { } --> qr { }
                        elsif ($string =~ /\G (\[)          ((?:$qq_bracket)*?) (\]) /oxgc) { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq [ ] --> qr [ ]
                        elsif ($string =~ /\G (\<)          ((?:$qq_angle)*?)   (\>) /oxgc) { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq < > --> qr < >
                        elsif ($string =~ /\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) /oxgc) { $e_string .= e_split($e.'qr','{','}',$2,''); next E_STRING_LOOP; } # qq | | --> qr { }
                        elsif ($string =~ /\G (\S)          ((?:$qq_char)*?)    (\1) /oxgc) { $e_string .= e_split($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # qq * * --> qr * *
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }

# split qr//
            elsif ($string =~ /\G \b (qr) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) ([imosxpadlunbB]*) /oxgc)                        { $e_string .= e_split  ($e.'qr',$1,$3,$2,$4);   next E_STRING_LOOP; } # qr# #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                                            /oxgc) { $e_string .= $e . $1; }
                        elsif ($string =~ /\G (\()          ((?:$qq_paren)*?)   (\)) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr ( )
                        elsif ($string =~ /\G (\{)          ((?:$qq_brace)*?)   (\}) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr { }
                        elsif ($string =~ /\G (\[)          ((?:$qq_bracket)*?) (\]) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr [ ]
                        elsif ($string =~ /\G (\<)          ((?:$qq_angle)*?)   (\>) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr < >
                        elsif ($string =~ /\G (\')          ((?:$qq_char)*?)    (\') ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split_q($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr ' '
                        elsif ($string =~ /\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr','{','}',$2,$4); next E_STRING_LOOP; } # qr | | --> qr { }
                        elsif ($string =~ /\G (\S)          ((?:$qq_char)*?)    (\1) ([imosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # qr * *
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }

# split q//
            elsif ($string =~ /\G \b (q) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc)                    { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q# #  --> qr # #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                              /oxgc) { $e_string .= $e . $1; }
                        elsif ($string =~ /\G (\() ((?:\\\\|\\\)|\\\(|$q_paren)*?)   (\)) /oxgc) { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q ( ) --> qr ( )
                        elsif ($string =~ /\G (\{) ((?:\\\\|\\\}|\\\{|$q_brace)*?)   (\}) /oxgc) { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q { } --> qr { }
                        elsif ($string =~ /\G (\[) ((?:\\\\|\\\]|\\\[|$q_bracket)*?) (\]) /oxgc) { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q [ ] --> qr [ ]
                        elsif ($string =~ /\G (\<) ((?:\\\\|\\\>|\\\<|$q_angle)*?)   (\>) /oxgc) { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q < > --> qr < >
                        elsif ($string =~ /\G ([*\-:?\\^|])       ((?:$q_char)*?)    (\1) /oxgc) { $e_string .= e_split_q($e.'qr','{','}',$2,''); next E_STRING_LOOP; } # q | | --> qr { }
                        elsif ($string =~ /\G (\S) ((?:\\\\|\\\1|     $q_char)*?)    (\1) /oxgc) { $e_string .= e_split_q($e.'qr',$1,$3,$2,'');   next E_STRING_LOOP; } # q * * --> qr * *
                    }
                    die __FILE__, ": Can't find string terminator anywhere before EOF\n";
                }
            }

# split m//
            elsif ($string =~ /\G \b (m) \b /oxgc) {
                if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) ([cgimosxpadlunbB]*) /oxgc)                        { $e_string .= e_split  ($e.'qr',$1,$3,$2,$4);   next E_STRING_LOOP; } # m# #  --> qr # #
                else {
                    while ($string !~ /\G \z/oxgc) {
                        if    ($string =~ /\G ((?>\s+)|\#.*)                                              /oxgc) { $e_string .= $e . $1; }
                        elsif ($string =~ /\G (\()          ((?:$qq_paren)*?)   (\)) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m ( ) --> qr ( )
                        elsif ($string =~ /\G (\{)          ((?:$qq_brace)*?)   (\}) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m { } --> qr { }
                        elsif ($string =~ /\G (\[)          ((?:$qq_bracket)*?) (\]) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m [ ] --> qr [ ]
                        elsif ($string =~ /\G (\<)          ((?:$qq_angle)*?)   (\>) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m < > --> qr < >
                        elsif ($string =~ /\G (\')          ((?:$qq_char)*?)    (\') ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split_q($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m ' ' --> qr ' '
                        elsif ($string =~ /\G ([*\-:?\\^|]) ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr','{','}',$2,$4); next E_STRING_LOOP; } # m | | --> qr { }
                        elsif ($string =~ /\G (\S)          ((?:$qq_char)*?)    (\1) ([cgimosxpadlunbB]*) /oxgc) { $e_string .= e_split  ($e.'qr',$1, $3, $2,$4); next E_STRING_LOOP; } # m * * --> qr * *
                    }
                    die __FILE__, ": Search pattern not terminated\n";
                }
            }

# split ''
            elsif ($string =~ /\G (\') /oxgc) {
                my $q_string = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G (\\\\)    /oxgc) { $q_string .= $1; }
                    elsif ($string =~ /\G (\\\')    /oxgc) { $q_string .= $1; } # splitqr'' --> split qr''
                    elsif ($string =~ /\G \'        /oxgc)                      { $e_string .= e_split_q($e.q{ qr},"'","'",$q_string,''); next E_STRING_LOOP; } # ' ' --> qr ' '
                    elsif ($string =~ /\G ($q_char) /oxgc) { $q_string .= $1; }
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }

# split ""
            elsif ($string =~ /\G (\") /oxgc) {
                my $qq_string = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G (\\\\)    /oxgc) { $qq_string .= $1; }
                    elsif ($string =~ /\G (\\\")    /oxgc) { $qq_string .= $1; } # splitqr"" --> split qr""
                    elsif ($string =~ /\G \"        /oxgc)                       { $e_string .= e_split($e.q{ qr},'"','"',$qq_string,''); next E_STRING_LOOP; } # " " --> qr " "
                    elsif ($string =~ /\G ($q_char) /oxgc) { $qq_string .= $1; }
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }

# split //
            elsif ($string =~ /\G (\/) /oxgc) {
                my $regexp = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G (\\\\)                  /oxgc) { $regexp .= $1; }
                    elsif ($string =~ /\G (\\\/)                  /oxgc) { $regexp .= $1; } # splitqr// --> split qr//
                    elsif ($string =~ /\G \/ ([cgimosxpadlunbB]*) /oxgc)                    { $e_string .= e_split($e.q{ qr}, '/','/',$regexp,$1); next E_STRING_LOOP; } # / / --> qr / /
                    elsif ($string =~ /\G ($q_char)               /oxgc) { $regexp .= $1; }
                }
                die __FILE__, ": Search pattern not terminated\n";
            }
        }

# qq//
        elsif ($string =~ /\G \b (qq) \b /oxgc) {
            my $ope = $1;
            if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) /oxgc) { # qq# #
                $e_string .= e_qq($ope,$1,$3,$2);
            }
            else {
                my $e = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G ((?>\s+)|\#.*)                /oxgc) { $e .= $1; }
                    elsif ($string =~ /\G (\() ((?:$qq_paren)*?)   (\)) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qq ( )
                    elsif ($string =~ /\G (\{) ((?:$qq_brace)*?)   (\}) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qq { }
                    elsif ($string =~ /\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qq [ ]
                    elsif ($string =~ /\G (\<) ((?:$qq_angle)*?)   (\>) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qq < >
                    elsif ($string =~ /\G (\S) ((?:$qq_char)*?)    (\1) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qq * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# qx//
        elsif ($string =~ /\G \b (qx) \b /oxgc) {
            my $ope = $1;
            if ($string =~ /\G (\#) ((?:$qq_char)*?) (\#) /oxgc) { # qx# #
                $e_string .= e_qq($ope,$1,$3,$2);
            }
            else {
                my $e = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G ((?>\s+)|\#.*)                /oxgc) { $e .= $1; }
                    elsif ($string =~ /\G (\() ((?:$qq_paren)*?)   (\)) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qx ( )
                    elsif ($string =~ /\G (\{) ((?:$qq_brace)*?)   (\}) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qx { }
                    elsif ($string =~ /\G (\[) ((?:$qq_bracket)*?) (\]) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qx [ ]
                    elsif ($string =~ /\G (\<) ((?:$qq_angle)*?)   (\>) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qx < >
                    elsif ($string =~ /\G (\') ((?:$qq_char)*?)    (\') /oxgc) { $e_string .= $e . e_q ($ope,$1,$3,$2); next E_STRING_LOOP; } # qx ' '
                    elsif ($string =~ /\G (\S) ((?:$qq_char)*?)    (\1) /oxgc) { $e_string .= $e . e_qq($ope,$1,$3,$2); next E_STRING_LOOP; } # qx * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# q//
        elsif ($string =~ /\G \b (q) \b /oxgc) {
            my $ope = $1;
            if ($string =~ /\G (\#) ((?:\\\#|\\\\|$q_char)*?) (\#) /oxgc) { # q# #
                $e_string .= e_q($ope,$1,$3,$2);
            }
            else {
                my $e = '';
                while ($string !~ /\G \z/oxgc) {
                    if    ($string =~ /\G ((?>\s+)|\#.*)                              /oxgc) { $e .= $1; }
                    elsif ($string =~ /\G (\() ((?:\\\\|\\\)|\\\(|$q_paren)*?)   (\)) /oxgc) { $e_string .= $e . e_q($ope,$1,$3,$2); next E_STRING_LOOP; } # q ( )
                    elsif ($string =~ /\G (\{) ((?:\\\\|\\\}|\\\{|$q_brace)*?)   (\}) /oxgc) { $e_string .= $e . e_q($ope,$1,$3,$2); next E_STRING_LOOP; } # q { }
                    elsif ($string =~ /\G (\[) ((?:\\\\|\\\]|\\\[|$q_bracket)*?) (\]) /oxgc) { $e_string .= $e . e_q($ope,$1,$3,$2); next E_STRING_LOOP; } # q [ ]
                    elsif ($string =~ /\G (\<) ((?:\\\\|\\\>|\\\<|$q_angle)*?)   (\>) /oxgc) { $e_string .= $e . e_q($ope,$1,$3,$2); next E_STRING_LOOP; } # q < >
                    elsif ($string =~ /\G (\S) ((?:\\\\|\\\1|     $q_char)*?)    (\1) /oxgc) { $e_string .= $e . e_q($ope,$1,$3,$2); next E_STRING_LOOP; } # q * *
                }
                die __FILE__, ": Can't find string terminator anywhere before EOF\n";
            }
        }

# ''
        elsif ($string =~ /\G (?<![\w\$\@\%\&\*]) (\') ((?:\\\'|\\\\|$q_char)*?) (\')           /oxgc) { $e_string .= e_q('',$1,$3,$2);  }

# ""
        elsif ($string =~ /\G (\") ((?:$qq_char)*?) (\")                                        /oxgc) { $e_string .= e_qq('',$1,$3,$2); }

# ``
        elsif ($string =~ /\G (\`) ((?:$qq_char)*?) (\`)                                        /oxgc) { $e_string .= e_qq('',$1,$3,$2); }

# <<>> (a safer ARGV)
        elsif ($string =~ /\G ( <<>> )                                                          /oxgc) { $e_string .= $1;                }

# <<= <=> <= < operator
        elsif ($string =~ /\G ( <<= | <=> | <= | < ) (?= (?>\s*) [A-Za-z_0-9'"`\$\@\&\*\(\+\-] )/oxgc) { $e_string .= $1;                }

# <FILEHANDLE>
        elsif ($string =~ /\G (<[\$]?[A-Za-z_][A-Za-z_0-9]*>)                                   /oxgc) { $e_string .= $1;                }

# <WILDCARD>   --- glob
        elsif ($string =~ /\G < ((?:$q_char)+?) > /oxgc) {
            $e_string .= 'Esjis::glob("' . $1 . '")';
        }

# << (bit shift)   --- not here document
        elsif ($string =~ /\G ( << (?>\s*) ) (?= [0-9\$\@\&] ) /oxgc) {
            $slash = 'm//';
            $e_string .= $1;
        }

# <<~'HEREDOC'
        elsif ($string =~ /\G ( <<~ [\t ]* '([a-zA-Z_0-9]*)' ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
                my $heredoc = $1;
                my $indent  = $2;
                $heredoc =~ s{^$indent}{}msg; # no /ox
                push @heredoc, $heredoc . qq{\n$delimiter\n};
                push @heredoc_delimiter, qq{\\s*$delimiter};
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= qq{<<'$delimiter'};
        }

# <<~\HEREDOC
        elsif ($string =~ /\G ( <<~ \\([a-zA-Z_0-9]+) ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
                my $heredoc = $1;
                my $indent  = $2;
                $heredoc =~ s{^$indent}{}msg; # no /ox
                push @heredoc, $heredoc . qq{\n$delimiter\n};
                push @heredoc_delimiter, qq{\\s*$delimiter};
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= qq{<<\\$delimiter};
        }

# <<~"HEREDOC"
        elsif ($string =~ /\G ( <<~ [\t ]* "([a-zA-Z_0-9]*)" ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
                my $heredoc = $1;
                my $indent  = $2;
                $heredoc =~ s{^$indent}{}msg; # no /ox
                push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
                push @heredoc_delimiter, qq{\\s*$delimiter};
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= qq{<<"$delimiter"};
        }

# <<~HEREDOC
        elsif ($string =~ /\G ( <<~ ([a-zA-Z_0-9]+) ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
                my $heredoc = $1;
                my $indent  = $2;
                $heredoc =~ s{^$indent}{}msg; # no /ox
                push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
                push @heredoc_delimiter, qq{\\s*$delimiter};
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= qq{<<$delimiter};
        }

# <<~`HEREDOC`
        elsif ($string =~ /\G ( <<~ [\t ]* `([a-zA-Z_0-9]*)` ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n ([\t ]*) $delimiter \n //xms) {
                my $heredoc = $1;
                my $indent  = $2;
                $heredoc =~ s{^$indent}{}msg; # no /ox
                push @heredoc, e_heredoc($heredoc) . qq{\n$delimiter\n};
                push @heredoc_delimiter, qq{\\s*$delimiter};
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= qq{<<`$delimiter`};
        }

# <<'HEREDOC'
        elsif ($string =~ /\G ( << '([a-zA-Z_0-9]*)' ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
                push @heredoc, $1 . qq{\n$delimiter\n};
                push @heredoc_delimiter, $delimiter;
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= $here_quote;
        }

# <<\HEREDOC
        elsif ($string =~ /\G ( << \\([a-zA-Z_0-9]+) ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
                push @heredoc, $1 . qq{\n$delimiter\n};
                push @heredoc_delimiter, $delimiter;
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= $here_quote;
        }

# <<"HEREDOC"
        elsif ($string =~ /\G ( << "([a-zA-Z_0-9]*)" ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
                push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
                push @heredoc_delimiter, $delimiter;
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= $here_quote;
        }

# <<HEREDOC
        elsif ($string =~ /\G ( << ([a-zA-Z_0-9]+) ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
                push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
                push @heredoc_delimiter, $delimiter;
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= $here_quote;
        }

# <<`HEREDOC`
        elsif ($string =~ /\G ( << `([a-zA-Z_0-9]*)` ) /oxgc) {
            $slash = 'm//';
            my $here_quote = $1;
            my $delimiter  = $2;

            # get here document
            if ($here_script eq '') {
                $here_script = CORE::substr $_, pos $_;
                $here_script =~ s/.*?\n//oxm;
            }
            if ($here_script =~ s/\A (.*?) \n $delimiter \n //xms) {
                push @heredoc, e_heredoc($1) . qq{\n$delimiter\n};
                push @heredoc_delimiter, $delimiter;
            }
            else {
                die __FILE__, ": Can't find string terminator $delimiter anywhere before EOF\n";
            }
            $e_string .= $here_quote;
        }

        # any operator before div
        elsif ($string =~ /\G (
            -- | \+\+ |
            [\)\}\]]

            ) /oxgc) { $slash = 'div'; $e_string .= $1; }

        # yada-yada or triple-dot operator
        elsif ($string =~ /\G (
            \.\.\.

            ) /oxgc) { $slash = 'm//'; $e_string .= q{die('Unimplemented')}; }

        # any operator before m//
        elsif ($string =~ /\G ((?>

            !~~ | !~ | != | ! |
            %= | % |
            &&= | && | &= | &\.= | &\. | & |
            -= | -> | - |
            :(?>\s*)= |
            : |
            <<>> |
            <<= | <=> | <= | < |
            == | => | =~ | = |
            >>= | >> | >= | > |
            \*\*= | \*\* | \*= | \* |
            \+= | \+ |
            \.\. | \.= | \. |
            \/\/= | \/\/ |
            \/= | \/ |
            \? |
            \\ |
            \^= | \^\.= | \^\. | \^ |
            \b x= |
            \|\|= | \|\| | \|= | \|\.= | \|\. | \| |
            ~~ | ~\. | ~ |
            \b(?: and | cmp | eq | ge | gt | le | lt | ne | not | or | xor | x )\b |
            \b(?: print )\b |

            [,;\(\{\[]

            )) /oxgc) { $slash = 'm//'; $e_string .= $1; }

        # other any character
        elsif ($string =~ /\G ($q_char) /oxgc) { $e_string .= $1; }

        # system error
        else {
            die __FILE__, ": Oops, this shouldn't happen!\n";
        }
    }

    return $e_string;
}

#
# character class
#
sub character_class {
    my($char,$modifier) = @_;

    if ($char eq '.') {
        if ($modifier =~ /s/) {
            return '${Esjis::dot_s}';
        }
        else {
            return '${Esjis::dot}';
        }
    }
    else {
        return Esjis::classic_character_class($char);
    }
}

#
# escape capture ($1, $2, $3, ...)
#
sub e_capture {

#if MULTIBYTE_ANCHORING
    return join '', '${Esjis::capture(', $_[0], ')}';
#endif
    return join '', '${',                $_[0],  '}';
}

#
# escape transliteration (tr/// or y///)
#
sub e_tr {
    my($variable,$charclass,$e,$charclass2,$modifier) = @_;
    my $e_tr = '';
    $modifier ||= '';

    $slash = 'div';

    # quote character class 1
    $charclass  = q_tr($charclass);

    # quote character class 2
    $charclass2 = q_tr($charclass2);

    # /b /B modifier
    if ($modifier =~ tr/bB//d) {
        if ($variable eq '') {
            $e_tr = qq{tr$charclass$e$charclass2$modifier};
        }
        else {
            $e_tr = qq{$variable${bind_operator}tr$charclass$e$charclass2$modifier};
        }
    }
    else {
        if ($variable eq '') {
            $e_tr = qq{Esjis::tr(\$_,' =~ ',$charclass,$e$charclass2,'$modifier')};
        }
        else {
            $e_tr = qq{Esjis::tr($variable,'$bind_operator',$charclass,$e$charclass2,'$modifier')};
        }
    }

    # clear tr/// variable
    $tr_variable = '';
    $bind_operator = '';

    return $e_tr;
}

#
# quote for escape transliteration (tr/// or y///)
#
sub q_tr {
    my($charclass) = @_;

    # quote character class
    if ($charclass !~ /'/oxms) {
        return e_q('',  "'", "'", $charclass); # --> q' '
    }
    elsif ($charclass !~ /\//oxms) {
        return e_q('q',  '/', '/', $charclass); # --> q/ /
    }
    elsif ($charclass !~ /\#/oxms) {
        return e_q('q',  '#', '#', $charclass); # --> q# #
    }
    elsif ($charclass !~ /[\<\>]/oxms) {
        return e_q('q', '<', '>', $charclass); # --> q< >
    }
    elsif ($charclass !~ /[\(\)]/oxms) {
        return e_q('q', '(', ')', $charclass); # --> q( )
    }
    elsif ($charclass !~ /[\{\}]/oxms) {
        return e_q('q', '{', '}', $charclass); # --> q{ }
    }
    else {
        for my $char (qw( ! " $ % & * + . : = ? @ ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
            if ($charclass !~ /\Q$char\E/xms) {
                return e_q('q', $char, $char, $charclass);
            }
        }
    }

    return e_q('q', '{', '}', $charclass);
}

#
# escape q string (q//, '')
#
sub e_q {
    my($ope,$delimiter,$end_delimiter,$string) = @_;

    $slash = 'div';

#if ESCAPE_SECOND_OCTET
    my @char = $string =~ / \G (?>$q_char) /oxmsg;
    for (my $i=0; $i <= $#char; $i++) {

        # escape last octet of multiple-octet
        if ($char[$i] =~ /\A ([\x80-\xFF].*) (\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }
        elsif (defined($char[$i+1]) and ($char[$i+1] eq '\\') and ($char[$i] =~ /\A ([\x80-\xFF].*) (\\) \z/xms)) {
            $char[$i] = $1 . '\\' . $2;
        }
    }
    if (defined($char[-1]) and ($char[-1] =~ /\A ([\x80-\xFF].*) (\\) \z/xms)) {
        $char[-1] = $1 . '\\' . $2;
    }

    return join '', $ope, $delimiter, @char,   $end_delimiter;
#endif
    return join '', $ope, $delimiter, $string, $end_delimiter;
}

#
# escape qq string (qq//, "", qx//, ``)
#
sub e_qq {
    my($ope,$delimiter,$end_delimiter,$string) = @_;

    $slash = 'div';

    my $left_e  = 0;
    my $right_e = 0;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\$]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \\x\{ (?>[0-9A-Fa-f]+) \}            |
        \\o\{ (?>[0-7]+)       \}            |
        \\N\{ (?>[^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} |
        \\ $q_char                           |
        \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  |
        \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     |
                        \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} |
        \$ (?>\s* [0-9]+)                    |
        \$ (?>\s*) \{ (?>\s* [0-9]+ \s*) \}  |
        \$ \$ (?![\w\{])                     |
        \$ (?>\s*) \$ (?>\s*) $qq_variable   |
           $q_char
    ))/oxmsg;

    for (my $i=0; $i <= $#char; $i++) {

        # "\L\u" --> "\u\L"
        if (($char[$i] eq '\L') and ($char[$i+1] eq '\u')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # "\U\l" --> "\l\U"
        elsif (($char[$i] eq '\U') and ($char[$i+1] eq '\l')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # octal escape sequence
        elsif ($char[$i] =~ /\A \\o \{ ([0-7]+) \} \z/oxms) {
            $char[$i] = Esjis::octchr($1);
        }

        # hexadecimal escape sequence
        elsif ($char[$i] =~ /\A \\x \{ ([0-9A-Fa-f]+) \} \z/oxms) {
            $char[$i] = Esjis::hexchr($1);
        }

        # \N{CHARNAME} --> N{CHARNAME}
        elsif ($char[$i] =~ /\A \\ ( N\{ ([^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} ) \z/oxms) {
            $char[$i] = $1;
        }

        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        # my $metachar = qr/[\@\\\|]/oxms; # '|' is for qx//, ``, open(), and system()
        # variable $delimiter and $end_delimiter can be ''
        elsif ($char[$i] =~ /\A ([\x80-\xFF].*) ([\@\\\|$delimiter$end_delimiter]) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # \F
        #
        # P.69 Table 2-6. Translation escapes
        # in Chapter 2: Bits and Pieces
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.
        # (and so on)

        # \u \l \U \L \F \Q \E
        elsif ($char[$i] =~ /\A ([<>]) \z/oxms) {
            if ($right_e < $left_e) {
                $char[$i] = '\\' . $char[$i];
            }
        }
        elsif ($char[$i] eq '\u') {

            # "STRING @{[ LIST EXPR ]} MORE STRING"

            # P.257 Other Tricks You Can Do with Hard References
            # in Chapter 8: References
            # of ISBN 0-596-00027-8 Programming Perl Third Edition.

            # P.353 Other Tricks You Can Do with Hard References
            # in Chapter 8: References
            # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

            # (and so on)

            $char[$i] = '@{[Esjis::ucfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\l') {
            $char[$i] = '@{[Esjis::lcfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\U') {
            $char[$i] = '@{[Esjis::uc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\L') {
            $char[$i] = '@{[Esjis::lc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\F') {
            $char[$i] = '@{[Esjis::fc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\Q') {
            $char[$i] = '@{[CORE::quotemeta qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\E') {
            if ($right_e < $left_e) {
                $char[$i] = '>]}';
                $right_e++;
            }
            else {
                $char[$i] = '';
            }
        }
        elsif ($char[$i] eq '\Q') {
            while (1) {
                if (++$i > $#char) {
                    last;
                }
                if ($char[$i] eq '\E') {
                    last;
                }
            }
        }
        elsif ($char[$i] eq '\E') {
        }

        # $0 --> $0
        elsif ($char[$i] =~ /\A \$ 0 \z/oxms) {
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) 0 (?>\s*) \} \z/oxms) {
        }

        # $$ --> $$
        elsif ($char[$i] =~ /\A \$\$ \z/oxms) {
        }

        # $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
        # $1, $2, $3 --> $1, $2, $3 otherwise
        elsif ($char[$i] =~ /\A \$ ((?>[1-9][0-9]*)) \z/oxms) {
            $char[$i] = e_capture($1);
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
            $char[$i] = e_capture($1);
        }

        # $$foo[ ... ] --> $ $foo->[ ... ]
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ (?:$qq_bracket)*? \] ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
        }

        # $$foo{ ... } --> $ $foo->{ ... }
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ (?:$qq_brace)*? \} ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
        }

        # $$foo
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) \z/oxms) {
            $char[$i] = e_capture($1);
        }

        # $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> Esjis::PREMATCH()
        elsif ($char[$i] =~ /\A ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH (?>\s*) \}  | \$ (?>\s*) \{\^PREMATCH\}  ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::PREMATCH()]}';
        }

        # $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> Esjis::MATCH()
        elsif ($char[$i] =~ /\A ( \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::MATCH()]}';
        }

        # $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> Esjis::POSTMATCH()
        elsif ($char[$i] =~ /\A (                 \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::POSTMATCH()]}';
        }

        # ${ foo } --> ${ foo }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ (?> \s* [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* \s* ) \}                                \z/oxms) {
        }

        # ${ ... }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ( .+ ) \} \z/oxms) {
            $char[$i] = e_capture($1);
        }
    }

    # return string
    if ($left_e > $right_e) {
        return join '', $ope, $delimiter, @char, '>]}' x ($left_e - $right_e), $end_delimiter;
    }
    return     join '', $ope, $delimiter, @char,                               $end_delimiter;
}

#
# escape qw string (qw//)
#
sub e_qw {
    my($ope,$delimiter,$end_delimiter,$string) = @_;

    $slash = 'div';

    # choice again delimiter
    my %octet = map {$_ => 1} ($string =~ /\G ([\x00-\xFF]) /oxmsg);
    if (not $octet{$end_delimiter}) {
        return join '', $ope, $delimiter, $string, $end_delimiter;
    }
    elsif (not $octet{')'}) {
        return join '', $ope, '(',        $string, ')';
    }
    elsif (not $octet{'}'}) {
        return join '', $ope, '{',        $string, '}';
    }
    elsif (not $octet{']'}) {
        return join '', $ope, '[',        $string, ']';
    }
    elsif (not $octet{'>'}) {
        return join '', $ope, '<',        $string, '>';
    }
    else {
        for my $char (qw( ! " $ % & * + - . / : = ? @ ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
            if (not $octet{$char}) {
                return join '', $ope,      $char, $string, $char;
            }
        }
    }

    # qw/AAA BBB C'CC/ --> ('AAA', 'BBB', 'C\'CC')
    my @string = CORE::split(/\s+/, $string);
    for my $string (@string) {
        my @octet = $string =~ /\G ([\x00-\xFF]) /oxmsg;
        for my $octet (@octet) {
            if ($octet =~ /\A (['\\]) \z/oxms) {
                $octet = '\\' . $1;
            }
        }
        $string = join '', @octet;
    }
    return join '', '(', (join ', ', map { "'$_'" } @string), ')';
}

#
# escape here document (<<"HEREDOC", <<HEREDOC, <<`HEREDOC`, <<~"HEREDOC", <<~HEREDOC, <<~`HEREDOC`)
#
sub e_heredoc {
    my($string) = @_;

    $slash = 'm//';

    my $metachar = qr/[\@\\|]/oxms; # '|' is for <<`HEREDOC`

    my $left_e  = 0;
    my $right_e = 0;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\$]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \\x\{ (?>[0-9A-Fa-f]+) \}            |
        \\o\{ (?>[0-7]+)       \}            |
        \\N\{ (?>[^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} |
        \\ $q_char                           |
        \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  |
        \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     |
                        \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} |
        \$ (?>\s* [0-9]+)                    |
        \$ (?>\s*) \{ (?>\s* [0-9]+ \s*) \}  |
        \$ \$ (?![\w\{])                     |
        \$ (?>\s*) \$ (?>\s*) $qq_variable   |
           $q_char
    ))/oxmsg;

    for (my $i=0; $i <= $#char; $i++) {

        # "\L\u" --> "\u\L"
        if (($char[$i] eq '\L') and ($char[$i+1] eq '\u')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # "\U\l" --> "\l\U"
        elsif (($char[$i] eq '\U') and ($char[$i+1] eq '\l')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # octal escape sequence
        elsif ($char[$i] =~ /\A \\o \{ ([0-7]+) \} \z/oxms) {
            $char[$i] = Esjis::octchr($1);
        }

        # hexadecimal escape sequence
        elsif ($char[$i] =~ /\A \\x \{ ([0-9A-Fa-f]+) \} \z/oxms) {
            $char[$i] = Esjis::hexchr($1);
        }

        # \N{CHARNAME} --> N{CHARNAME}
        elsif ($char[$i] =~ /\A \\ ( N\{ ([^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} ) \z/oxms) {
            $char[$i] = $1;
        }

        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape character
        elsif ($char[$i] =~ /\A ([\x80-\xFF].*) ($metachar) \z/oxms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # \u \l \U \L \F \Q \E
        elsif ($char[$i] =~ /\A ([<>]) \z/oxms) {
            if ($right_e < $left_e) {
                $char[$i] = '\\' . $char[$i];
            }
        }
        elsif ($char[$i] eq '\u') {
            $char[$i] = '@{[Esjis::ucfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\l') {
            $char[$i] = '@{[Esjis::lcfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\U') {
            $char[$i] = '@{[Esjis::uc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\L') {
            $char[$i] = '@{[Esjis::lc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\F') {
            $char[$i] = '@{[Esjis::fc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\Q') {
            $char[$i] = '@{[CORE::quotemeta qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\E') {
            if ($right_e < $left_e) {
                $char[$i] = '>]}';
                $right_e++;
            }
            else {
                $char[$i] = '';
            }
        }
        elsif ($char[$i] eq '\Q') {
            while (1) {
                if (++$i > $#char) {
                    last;
                }
                if ($char[$i] eq '\E') {
                    last;
                }
            }
        }
        elsif ($char[$i] eq '\E') {
        }

        # $0 --> $0
        elsif ($char[$i] =~ /\A \$ 0 \z/oxms) {
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) 0 (?>\s*) \} \z/oxms) {
        }

        # $$ --> $$
        elsif ($char[$i] =~ /\A \$\$ \z/oxms) {
        }

        # $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
        # $1, $2, $3 --> $1, $2, $3 otherwise
        elsif ($char[$i] =~ /\A \$ ((?>[1-9][0-9]*)) \z/oxms) {
            $char[$i] = e_capture($1);
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
            $char[$i] = e_capture($1);
        }

        # $$foo[ ... ] --> $ $foo->[ ... ]
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ (?:$qq_bracket)*? \] ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
        }

        # $$foo{ ... } --> $ $foo->{ ... }
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ (?:$qq_brace)*? \} ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
        }

        # $$foo
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) \z/oxms) {
            $char[$i] = e_capture($1);
        }

        # $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> Esjis::PREMATCH()
        elsif ($char[$i] =~ /\A ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::PREMATCH()]}';
        }

        # $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> Esjis::MATCH()
        elsif ($char[$i] =~ /\A ( \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::MATCH()]}';
        }

        # $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> Esjis::POSTMATCH()
        elsif ($char[$i] =~ /\A (                 \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) \z/oxmsgc) {
            $char[$i] = '@{[Esjis::POSTMATCH()]}';
        }

        # ${ foo } --> ${ foo }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ (?> \s* [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* \s* ) \}                                \z/oxms) {
        }

        # ${ ... }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ( .+ ) \} \z/oxms) {
            $char[$i] = e_capture($1);
        }
    }

    # return string
    if ($left_e > $right_e) {
        return join '', @char, '>]}' x ($left_e - $right_e);
    }
    return     join '', @char;
}

#
# escape regexp (m//, qr//)
#
sub e_qr {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;
    $modifier ||= '';

    $modifier =~ tr/p//d;
    if ($modifier =~ /([adlu])/oxms) {
        my $line = 0;
        for (my $i=0; my($package,$filename,$use_line,$subroutine) = caller($i); $i++) {
            if ($filename ne __FILE__) {
                $line = $use_line + (CORE::substr($_,0,pos($_)) =~ tr/\n//) + 1;
                last;
            }
        }
        die qq{Unsupported modifier "$1" used at line $line.\n};
    }

    $slash = 'div';

    # literal null string pattern
    if ($string eq '') {
        $modifier =~ tr/bB//d;
        $modifier =~ tr/i//d;
        return join '', $ope, $delimiter, $end_delimiter, $modifier;
    }

    # /b /B modifier
    elsif ($modifier =~ tr/bB//d) {

        # choice again delimiter
        if ($delimiter =~ / [\@:] /oxms) {
            my @char = $string =~ /\G ([\x00-\xFF]) /oxmsg;
            my %octet = map {$_ => 1} @char;
            if (not $octet{')'}) {
                $delimiter     = '(';
                $end_delimiter = ')';
            }
            elsif (not $octet{'}'}) {
                $delimiter     = '{';
                $end_delimiter = '}';
            }
            elsif (not $octet{']'}) {
                $delimiter     = '[';
                $end_delimiter = ']';
            }
            elsif (not $octet{'>'}) {
                $delimiter     = '<';
                $end_delimiter = '>';
            }
            else {
                for my $char (qw( ! " $ % & * + - . / = ? ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
                    if (not $octet{$char}) {
                        $delimiter     = $char;
                        $end_delimiter = $char;
                        last;
                    }
                }
            }
        }

        if (($ope =~ /\A m? \z/oxms) and ($delimiter eq '?')) {
            return join '', $ope, $delimiter,        $string,      $matched, $end_delimiter, $modifier;
        }
        else {
            return join '', $ope, $delimiter, '(?:', $string, ')', $matched, $end_delimiter, $modifier;
        }
    }

    my $ignorecase = ($modifier =~ /i/oxms) ? 1 : 0;
    my $metachar = qr/[\@\\|[\]{^]/oxms;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\$\@\[\(]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \\x   (?>[0-9A-Fa-f]{1,2}) |
        \\    (?>[0-7]{2,3})       |
        \\c   [\x40-\x5F]          |
        \\x\{ (?>[0-9A-Fa-f]+) \}  |
        \\o\{ (?>[0-7]+)       \}  |
        \\[bBNpP]\{ (?>[^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} |
        \\  $q_char                |
        \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  |
        \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     |
                        \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} |
        [\$\@] $qq_variable        |
        \$ (?>\s* [0-9]+)          |
        \$ (?>\s*) \{ (?>\s* [0-9]+ \s*) \}  |
        \$ \$ (?![\w\{])           |
        \$ (?>\s*) \$ (?>\s*) $qq_variable   |
        \[\^                       |
        \[\:   (?>[a-z]+) :\]      |
        \[\:\^ (?>[a-z]+) :\]      |
        \(\?                       |
            $q_char
    ))/oxmsg;

    # choice again delimiter
    if ($delimiter =~ / [\@:] /oxms) {
        my %octet = map {$_ => 1} @char;
        if (not $octet{')'}) {
            $delimiter     = '(';
            $end_delimiter = ')';
        }
        elsif (not $octet{'}'}) {
            $delimiter     = '{';
            $end_delimiter = '}';
        }
        elsif (not $octet{']'}) {
            $delimiter     = '[';
            $end_delimiter = ']';
        }
        elsif (not $octet{'>'}) {
            $delimiter     = '<';
            $end_delimiter = '>';
        }
        else {
            for my $char (qw( ! " $ % & * + - . / = ? ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
                if (not $octet{$char}) {
                    $delimiter     = $char;
                    $end_delimiter = $char;
                    last;
                }
            }
        }
    }

    my $left_e  = 0;
    my $right_e = 0;
    for (my $i=0; $i <= $#char; $i++) {

        # "\L\u" --> "\u\L"
        if (($char[$i] eq '\L') and ($char[$i+1] eq '\u')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # "\U\l" --> "\l\U"
        elsif (($char[$i] eq '\U') and ($char[$i+1] eq '\l')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # octal escape sequence
        elsif ($char[$i] =~ /\A \\o \{ ([0-7]+) \} \z/oxms) {
            $char[$i] = Esjis::octchr($1);
        }

        # hexadecimal escape sequence
        elsif ($char[$i] =~ /\A \\x \{ ([0-9A-Fa-f]+) \} \z/oxms) {
            $char[$i] = Esjis::hexchr($1);
        }

        # \b{...}      --> b\{...}
        # \B{...}      --> B\{...}
        # \N{CHARNAME} --> N\{CHARNAME}
        # \p{PROPERTY} --> p\{PROPERTY}
        # \P{PROPERTY} --> P\{PROPERTY}
        elsif ($char[$i] =~ /\A \\ ([bBNpP]) ( \{ ([^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} ) \z/oxms) {
            $char[$i] = $1 . '\\' . $2;
        }

        # \p, \P, \X --> p, P, X
        elsif ($char[$i] =~ /\A \\ ( [pPX] ) \z/oxms) {
            $char[$i] = $1;
        }

        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        elsif ($char[$i] =~ /\A \\? ([\x80-\xFF].*) ($metachar|\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # join separated multiple-octet
        elsif ($char[$i] =~ /\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms) {
            if (   ($i+3 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, @char[$i+1..$i+3]) == 3) and (CORE::eval(sprintf '"%s%s%s%s"', @char[$i..$i+3]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 3;
            }
            elsif (($i+2 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, @char[$i+1..$i+2]) == 2) and (CORE::eval(sprintf '"%s%s%s"',   @char[$i..$i+2]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 2;
            }
            elsif (($i+1 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, $char[$i+1      ]) == 1) and (CORE::eval(sprintf '"%s%s"',     @char[$i..$i+1]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 1;
            }
        }

        # open character class [...]
        elsif ($char[$i] eq '[') {
            my $left = $i;

            # [] make die "Unmatched [] in regexp ...\n"
            # (and so on)

            if ($char[$i+1] eq ']') {
                $i++;
            }

            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [...]
                    if (grep(/\A [\$\@]/oxms,@char[$left+1..$right-1]) >= 1) {
                        splice @char, $left, $right-$left+1, sprintf(q{@{[Esjis::charlist_qr(%s,'%s')]}}, join(',', map {qq_stuff($delimiter,$end_delimiter,$_)} @char[$left+1..$right-1]), $modifier);
                    }
                    else {
                        splice @char, $left, $right-$left+1, Esjis::charlist_qr(@char[$left+1..$right-1], $modifier);
                    }

                    $i = $left;
                    last;
                }
            }
        }

        # open character class [^...]
        elsif ($char[$i] eq '[^') {
            my $left = $i;

            # [^] make die "Unmatched [] in regexp ...\n"
            # (and so on)

            if ($char[$i+1] eq ']') {
                $i++;
            }

            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [^...]
                    if (grep(/\A [\$\@]/oxms,@char[$left+1..$right-1]) >= 1) {
                        splice @char, $left, $right-$left+1, sprintf(q{@{[Esjis::charlist_not_qr(%s,'%s')]}}, join(',', map {qq_stuff($delimiter,$end_delimiter,$_)} @char[$left+1..$right-1]), $modifier);
                    }
                    else {
                        splice @char, $left, $right-$left+1, Esjis::charlist_not_qr(@char[$left+1..$right-1], $modifier);
                    }

                    $i = $left;
                    last;
                }
            }
        }

        # rewrite character class or escape character
        elsif (my $char = character_class($char[$i],$modifier)) {
            $char[$i] = $char;
        }

        # /i modifier
        elsif ($ignorecase and ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) and (Esjis::uc($char[$i]) ne Esjis::fc($char[$i]))) {
            if (CORE::length(Esjis::fc($char[$i])) == 1) {
                $char[$i] = '['   . Esjis::uc($char[$i])       . Esjis::fc($char[$i]) . ']';
            }
            else {
                $char[$i] = '(?:' . Esjis::uc($char[$i]) . '|' . Esjis::fc($char[$i]) . ')';
            }
        }

        # \u \l \U \L \F \Q \E
        elsif ($char[$i] =~ /\A [<>] \z/oxms) {
            if ($right_e < $left_e) {
                $char[$i] = '\\' . $char[$i];
            }
        }
        elsif ($char[$i] eq '\u') {
            $char[$i] = '@{[Esjis::ucfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\l') {
            $char[$i] = '@{[Esjis::lcfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\U') {
            $char[$i] = '@{[Esjis::uc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\L') {
            $char[$i] = '@{[Esjis::lc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\F') {
            $char[$i] = '@{[Esjis::fc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\Q') {
            $char[$i] = '@{[CORE::quotemeta qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\E') {
            if ($right_e < $left_e) {
                $char[$i] = '>]}';
                $right_e++;
            }
            else {
                $char[$i] = '';
            }
        }
        elsif ($char[$i] eq '\Q') {
            while (1) {
                if (++$i > $#char) {
                    last;
                }
                if ($char[$i] eq '\E') {
                    last;
                }
            }
        }
        elsif ($char[$i] eq '\E') {
        }

        # $0 --> $0
        elsif ($char[$i] =~ /\A \$ 0 \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) 0 (?>\s*) \} \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$ --> $$
        elsif ($char[$i] =~ /\A \$\$ \z/oxms) {
        }

        # $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
        # $1, $2, $3 --> $1, $2, $3 otherwise
        elsif ($char[$i] =~ /\A \$ ((?>[1-9][0-9]*)) \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo[ ... ] --> $ $foo->[ ... ]
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ (?:$qq_bracket)*? \] ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo{ ... } --> $ $foo->{ ... }
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ (?:$qq_brace)*? \} ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> Esjis::PREMATCH()
        elsif ($char[$i] =~ /\A ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::PREMATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::PREMATCH()]}';
            }
        }

        # $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> Esjis::MATCH()
        elsif ($char[$i] =~ /\A ( \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::MATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::MATCH()]}';
            }
        }

        # $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> Esjis::POSTMATCH()
        elsif ($char[$i] =~ /\A (                 \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::POSTMATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::POSTMATCH()]}';
            }
        }

        # ${ foo }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ((?> \s* [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* \s* )) \}                              \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # ${ ... }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ( .+ ) \} \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $scalar or @array
        elsif ($char[$i] =~ /\A [\$\@].+ /oxms) {
            $char[$i] = e_string($char[$i]);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # quote character before ? + * {
        elsif (($i >= 1) and ($char[$i] =~ /\A [\?\+\*\{] \z/oxms)) {
            if ($char[$i-1] =~ /\A (?:[\x00-\xFF]|\\[0-7]{2,3}|\\x[0-9-A-Fa-f]{1,2}) \z/oxms) {
            }
            elsif (($ope =~ /\A m? \z/oxms) and ($delimiter eq '?')) {
                my $char = $char[$i-1];
                if ($char[$i] eq '{') {
                    die __FILE__, qq{: "MULTIBYTE{n}" should be "(MULTIBYTE){n}" in m?? (and shift \$1,\$2,\$3,...) ($char){n}\n};
                }
                else {
                    die __FILE__, qq{: "MULTIBYTE$char[$i]" should be "(MULTIBYTE)$char[$i]" in m?? (and shift \$1,\$2,\$3,...) ($char)$char[$i]\n};
                }
            }
            else {
                $char[$i-1] = '(?:' . $char[$i-1] . ')';
            }
        }
    }

    # make regexp string
    $modifier =~ tr/i//d;
    if ($left_e > $right_e) {
        if (($ope =~ /\A m? \z/oxms) and ($delimiter eq '?')) {
            return join '', $ope, $delimiter, $anchor,        @char, '>]}' x ($left_e - $right_e),      $matched, $end_delimiter, $modifier;
        }
        else {
            return join '', $ope, $delimiter, $anchor, '(?:', @char, '>]}' x ($left_e - $right_e), ')', $matched, $end_delimiter, $modifier;
        }
    }
    if (($ope =~ /\A m? \z/oxms) and ($delimiter eq '?')) {
        return     join '', $ope, $delimiter, $anchor,        @char,                                    $matched, $end_delimiter, $modifier;
    }
    else {
        return     join '', $ope, $delimiter, $anchor, '(?:', @char,                               ')', $matched, $end_delimiter, $modifier;
    }
}

#
# double quote stuff
#
sub qq_stuff {
    my($delimiter,$end_delimiter,$stuff) = @_;

    # scalar variable or array variable
    if ($stuff =~ /\A [\$\@] /oxms) {
        return $stuff;
    }

    # quote by delimiter
    my %octet = map {$_ => 1} ($stuff =~ /\G ([\x00-\xFF]) /oxmsg);
    for my $char (qw( ! " $ % & * + - . / : = ? @ ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
        next if $char eq $delimiter;
        next if $char eq $end_delimiter;
        if (not $octet{$char}) {
            return join '', 'qq', $char, $stuff, $char;
        }
    }
    return join '', 'qq', '<', $stuff, '>';
}

#
# escape regexp (m'', qr'', and m''b, qr''b)
#
sub e_qr_q {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;
    $modifier ||= '';

    $modifier =~ tr/p//d;
    if ($modifier =~ /([adlu])/oxms) {
        my $line = 0;
        for (my $i=0; my($package,$filename,$use_line,$subroutine) = caller($i); $i++) {
            if ($filename ne __FILE__) {
                $line = $use_line + (CORE::substr($_,0,pos($_)) =~ tr/\n//) + 1;
                last;
            }
        }
        die qq{Unsupported modifier "$1" used at line $line.\n};
    }

    $slash = 'div';

    # literal null string pattern
    if ($string eq '') {
        $modifier =~ tr/bB//d;
        $modifier =~ tr/i//d;
        return join '', $ope, $delimiter, $end_delimiter, $modifier;
    }

    # with /b /B modifier
    elsif ($modifier =~ tr/bB//d) {
        return e_qr_qb($ope,$delimiter,$end_delimiter,$string,$modifier);
    }

    # without /b /B modifier
    else {
        return e_qr_qt($ope,$delimiter,$end_delimiter,$string,$modifier);
    }
}

#
# escape regexp (m'', qr'')
#
sub e_qr_qt {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;

    my $ignorecase = ($modifier =~ /i/oxms) ? 1 : 0;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\[\$\@\/] |
        [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \[\^                            |
        \[\:   (?>[a-z]+) \:\]          |
        \[\:\^ (?>[a-z]+) \:\]          |
        [\$\@\/]                        |
        \\     (?:$q_char)              |
               (?:$q_char)
    ))/oxmsg;

    # unescape character
    for (my $i=0; $i <= $#char; $i++) {
        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        elsif ($char[$i] =~ /\A ([\x80-\xFF].*) ([\\|\[\{\^]|\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # open character class [...]
        elsif ($char[$i] eq '[') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [...]
                    splice @char, $left, $right-$left+1, Esjis::charlist_qr(@char[$left+1..$right-1], $modifier);

                    $i = $left;
                    last;
                }
            }
        }

        # open character class [^...]
        elsif ($char[$i] eq '[^') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [^...]
                    splice @char, $left, $right-$left+1, Esjis::charlist_not_qr(@char[$left+1..$right-1], $modifier);

                    $i = $left;
                    last;
                }
            }
        }

        # escape $ @ / and \
        elsif ($char[$i] =~ /\A [\$\@\/\\] \z/oxms) {
            $char[$i] = '\\' . $char[$i];
        }

        # rewrite character class or escape character
        elsif (my $char = character_class($char[$i],$modifier)) {
            $char[$i] = $char;
        }

        # /i modifier
        elsif ($ignorecase and ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) and (Esjis::uc($char[$i]) ne Esjis::fc($char[$i]))) {
            if (CORE::length(Esjis::fc($char[$i])) == 1) {
                $char[$i] = '['   . Esjis::uc($char[$i])       . Esjis::fc($char[$i]) . ']';
            }
            else {
                $char[$i] = '(?:' . Esjis::uc($char[$i]) . '|' . Esjis::fc($char[$i]) . ')';
            }
        }

        # quote character before ? + * {
        elsif (($i >= 1) and ($char[$i] =~ /\A [\?\+\*\{] \z/oxms)) {
            if ($char[$i-1] =~ /\A [\x00-\xFF] \z/oxms) {
            }
            else {
                $char[$i-1] = '(?:' . $char[$i-1] . ')';
            }
        }
    }

    $delimiter     = '/';
    $end_delimiter = '/';

    $modifier =~ tr/i//d;
    return join '', $ope, $delimiter, $anchor, '(?:', @char, ')', $matched, $end_delimiter, $modifier;
}

#
# escape regexp (m''b, qr''b)
#
sub e_qr_qb {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;

    # split regexp
    my @char = $string =~ /\G ((?>[^\\]|\\\\|[\x00-\xFF])) /oxmsg;

    # unescape character
    for (my $i=0; $i <= $#char; $i++) {
        if (0) {
        }

        # remain \\
        elsif ($char[$i] eq '\\\\') {
        }

        # escape $ @ / and \
        elsif ($char[$i] =~ /\A [\$\@\/\\] \z/oxms) {
            $char[$i] = '\\' . $char[$i];
        }
    }

    $delimiter     = '/';
    $end_delimiter = '/';
    return join '', $ope, $delimiter, '(?:', @char, ')', $matched, $end_delimiter, $modifier;
}

#
# escape regexp (s/here//)
#
sub e_s1 {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;
    $modifier ||= '';

    $modifier =~ tr/p//d;
    if ($modifier =~ /([adlu])/oxms) {
        my $line = 0;
        for (my $i=0; my($package,$filename,$use_line,$subroutine) = caller($i); $i++) {
            if ($filename ne __FILE__) {
                $line = $use_line + (CORE::substr($_,0,pos($_)) =~ tr/\n//) + 1;
                last;
            }
        }
        die qq{Unsupported modifier "$1" used at line $line.\n};
    }

    $slash = 'div';

    # literal null string pattern
    if ($string eq '') {
        $modifier =~ tr/bB//d;
        $modifier =~ tr/i//d;
        return join '', $ope, $delimiter, $end_delimiter, $modifier;
    }

    # /b /B modifier
    elsif ($modifier =~ tr/bB//d) {

        # choice again delimiter
        if ($delimiter =~ / [\@:] /oxms) {
            my @char = $string =~ /\G ([\x00-\xFF]) /oxmsg;
            my %octet = map {$_ => 1} @char;
            if (not $octet{')'}) {
                $delimiter     = '(';
                $end_delimiter = ')';
            }
            elsif (not $octet{'}'}) {
                $delimiter     = '{';
                $end_delimiter = '}';
            }
            elsif (not $octet{']'}) {
                $delimiter     = '[';
                $end_delimiter = ']';
            }
            elsif (not $octet{'>'}) {
                $delimiter     = '<';
                $end_delimiter = '>';
            }
            else {
                for my $char (qw( ! " $ % & * + - . / = ? ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
                    if (not $octet{$char}) {
                        $delimiter     = $char;
                        $end_delimiter = $char;
                        last;
                    }
                }
            }
        }

        my $prematch = '';
#if MULTIBYTE_ANCHORING
        $prematch = q{(\G[\x00-\xFF]*?)};
#endif
        return join '', $ope, $delimiter, $prematch, '(?:', $string, ')', $matched, $end_delimiter, $modifier;
    }

    my $ignorecase = ($modifier =~ /i/oxms) ? 1 : 0;
    my $metachar = qr/[\@\\|[\]{^]/oxms;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\$\@\[\(]|[\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \\                               (?>[1-9][0-9]*)            |
        \\g (?>\s*)                      (?>[1-9][0-9]*)            |
        \\g (?>\s*) \{ (?>\s*)           (?>[1-9][0-9]*) (?>\s*) \} |
        \\g (?>\s*) \{ (?>\s*) - (?>\s*) (?>[1-9][0-9]*) (?>\s*) \} |
        \\x                              (?>[0-9A-Fa-f]{1,2})       |
        \\                               (?>[0-7]{2,3})             |
        \\c                              [\x40-\x5F]                |
        \\x\{                            (?>[0-9A-Fa-f]+)        \} |
        \\o\{                            (?>[0-7]+)              \} |
        \\[bBNpP]\{                      (?>[^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} |
        \\ $q_char                           |
        \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  |
        \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     |
                        \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} |
        [\$\@] $qq_variable                  |
        \$ (?>\s* [0-9]+)                    |
        \$ (?>\s*) \{ (?>\s* [0-9]+ \s*) \}  |
        \$ \$ (?![\w\{])                     |
        \$ (?>\s*) \$ (?>\s*) $qq_variable   |
        \[\^                                 |
        \[\:   (?>[a-z]+) :\]                |
        \[\:\^ (?>[a-z]+) :\]                |
        \(\?                                 |
            $q_char
    ))/oxmsg;

    # choice again delimiter
    if ($delimiter =~ / [\@:] /oxms) {
        my %octet = map {$_ => 1} @char;
        if (not $octet{')'}) {
            $delimiter     = '(';
            $end_delimiter = ')';
        }
        elsif (not $octet{'}'}) {
            $delimiter     = '{';
            $end_delimiter = '}';
        }
        elsif (not $octet{']'}) {
            $delimiter     = '[';
            $end_delimiter = ']';
        }
        elsif (not $octet{'>'}) {
            $delimiter     = '<';
            $end_delimiter = '>';
        }
        else {
            for my $char (qw( ! " $ % & * + - . / = ? ^ ` | ~ ), "\x00".."\x1F", "\x7F", "\xFF") {
                if (not $octet{$char}) {
                    $delimiter     = $char;
                    $end_delimiter = $char;
                    last;
                }
            }
        }
    }

    # count '('
    my $parens = grep { $_ eq '(' } @char;

    my $left_e  = 0;
    my $right_e = 0;
    for (my $i=0; $i <= $#char; $i++) {

        # "\L\u" --> "\u\L"
        if (($char[$i] eq '\L') and ($char[$i+1] eq '\u')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # "\U\l" --> "\l\U"
        elsif (($char[$i] eq '\U') and ($char[$i+1] eq '\l')) {
            @char[$i,$i+1] = @char[$i+1,$i];
        }

        # octal escape sequence
        elsif ($char[$i] =~ /\A \\o \{ ([0-7]+) \} \z/oxms) {
            $char[$i] = Esjis::octchr($1);
        }

        # hexadecimal escape sequence
        elsif ($char[$i] =~ /\A \\x \{ ([0-9A-Fa-f]+) \} \z/oxms) {
            $char[$i] = Esjis::hexchr($1);
        }

        # \b{...}      --> b\{...}
        # \B{...}      --> B\{...}
        # \N{CHARNAME} --> N\{CHARNAME}
        # \p{PROPERTY} --> p\{PROPERTY}
        # \P{PROPERTY} --> P\{PROPERTY}
        elsif ($char[$i] =~ /\A \\ ([bBNpP]) ( \{ ([^\x81-\x9F\xE0-\xFC0-9\}][^\x81-\x9F\xE0-\xFC\}]*) \} ) \z/oxms) {
            $char[$i] = $1 . '\\' . $2;
        }

        # \p, \P, \X --> p, P, X
        elsif ($char[$i] =~ /\A \\ ( [pPX] ) \z/oxms) {
            $char[$i] = $1;
        }

        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        elsif ($char[$i] =~ /\A \\? ([\x80-\xFF].*) ($metachar|\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # join separated multiple-octet
        elsif ($char[$i] =~ /\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms) {
            if (   ($i+3 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, @char[$i+1..$i+3]) == 3) and (CORE::eval(sprintf '"%s%s%s%s"', @char[$i..$i+3]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 3;
            }
            elsif (($i+2 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, @char[$i+1..$i+2]) == 2) and (CORE::eval(sprintf '"%s%s%s"',   @char[$i..$i+2]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 2;
            }
            elsif (($i+1 <= $#char) and (grep(/\A (?: \\ [0-7]{2,3} | \\x [0-9A-Fa-f]{1,2}) \z/oxms, $char[$i+1      ]) == 1) and (CORE::eval(sprintf '"%s%s"',     @char[$i..$i+1]) =~ /\A $q_char \z/oxms)) {
                $char[$i] .= join '', splice @char, $i+1, 1;
            }
        }

        # open character class [...]
        elsif ($char[$i] eq '[') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [...]
                    if (grep(/\A [\$\@]/oxms,@char[$left+1..$right-1]) >= 1) {
                        splice @char, $left, $right-$left+1, sprintf(q{@{[Esjis::charlist_qr(%s,'%s')]}}, join(',', map {qq_stuff($delimiter,$end_delimiter,$_)} @char[$left+1..$right-1]), $modifier);
                    }
                    else {
                        splice @char, $left, $right-$left+1, Esjis::charlist_qr(@char[$left+1..$right-1], $modifier);
                    }

                    $i = $left;
                    last;
                }
            }
        }

        # open character class [^...]
        elsif ($char[$i] eq '[^') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [^...]
                    if (grep(/\A [\$\@]/oxms,@char[$left+1..$right-1]) >= 1) {
                        splice @char, $left, $right-$left+1, sprintf(q{@{[Esjis::charlist_not_qr(%s,'%s')]}}, join(',', map {qq_stuff($delimiter,$end_delimiter,$_)} @char[$left+1..$right-1]), $modifier);
                    }
                    else {
                        splice @char, $left, $right-$left+1, Esjis::charlist_not_qr(@char[$left+1..$right-1], $modifier);
                    }

                    $i = $left;
                    last;
                }
            }
        }

        # rewrite character class or escape character
        elsif (my $char = character_class($char[$i],$modifier)) {
            $char[$i] = $char;
        }

        # /i modifier
        elsif ($ignorecase and ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) and (Esjis::uc($char[$i]) ne Esjis::fc($char[$i]))) {
            if (CORE::length(Esjis::fc($char[$i])) == 1) {
                $char[$i] = '['   . Esjis::uc($char[$i])       . Esjis::fc($char[$i]) . ']';
            }
            else {
                $char[$i] = '(?:' . Esjis::uc($char[$i]) . '|' . Esjis::fc($char[$i]) . ')';
            }
        }

        # \u \l \U \L \F \Q \E
        elsif ($char[$i] =~ /\A [<>] \z/oxms) {
            if ($right_e < $left_e) {
                $char[$i] = '\\' . $char[$i];
            }
        }
        elsif ($char[$i] eq '\u') {
            $char[$i] = '@{[Esjis::ucfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\l') {
            $char[$i] = '@{[Esjis::lcfirst qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\U') {
            $char[$i] = '@{[Esjis::uc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\L') {
            $char[$i] = '@{[Esjis::lc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\F') {
            $char[$i] = '@{[Esjis::fc qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\Q') {
            $char[$i] = '@{[CORE::quotemeta qq<';
            $left_e++;
        }
        elsif ($char[$i] eq '\E') {
            if ($right_e < $left_e) {
                $char[$i] = '>]}';
                $right_e++;
            }
            else {
                $char[$i] = '';
            }
        }
        elsif ($char[$i] eq '\Q') {
            while (1) {
                if (++$i > $#char) {
                    last;
                }
                if ($char[$i] eq '\E') {
                    last;
                }
            }
        }
        elsif ($char[$i] eq '\E') {
        }

        # \0 --> \0
        elsif ($char[$i] =~ /\A \\ (?>\s*) 0 \z/oxms) {
        }

        # \g{N}, \g{-N}

        # P.108 Using Simple Patterns
        # in Chapter 7: In the World of Regular Expressions
        # of ISBN 978-0-596-52010-6 Learning Perl, Fifth Edition

        # P.221 Capturing
        # in Chapter 5: Pattern Matching
        # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

        # \g{-1}, \g{-2}, \g{-3} --> \g{-1}, \g{-2}, \g{-3}
        elsif ($char[$i] =~ /\A \\g (?>\s*) \{ (?>\s*) - (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
        }

        # \g{1}, \g{2}, \g{3} --> \g{2}, \g{3}, \g{4} (only when multibyte anchoring is enable)
        elsif ($char[$i] =~ /\A \\g (?>\s*) \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
#if MULTIBYTE_ANCHORING
            if ($1 <= $parens) {
                $char[$i] = '\\g{' . ($1 + 1) . '}';
            }
#endif
        }

        # \g1, \g2, \g3 --> \g2, \g3, \g4 (only when multibyte anchoring is enable)
        elsif ($char[$i] =~ /\A \\g (?>\s*) ((?>[1-9][0-9]*)) \z/oxms) {
#if MULTIBYTE_ANCHORING
            if ($1 <= $parens) {
                $char[$i] = '\\g' . ($1 + 1);
            }
#endif
        }

        # \1, \2, \3 --> \2, \3, \4 (only when multibyte anchoring is enable)
        elsif ($char[$i] =~ /\A \\ (?>\s*) ((?>[1-9][0-9]*)) \z/oxms) {
#if MULTIBYTE_ANCHORING
            if ($1 <= $parens) {
                $char[$i] = '\\' . ($1 + 1);
            }
#endif
        }

        # $0 --> $0
        elsif ($char[$i] =~ /\A \$ 0 \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) 0 (?>\s*) \} \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$ --> $$
        elsif ($char[$i] =~ /\A \$\$ \z/oxms) {
        }

        # $1, $2, $3 --> $2, $3, $4 after s/// with multibyte anchoring
        # $1, $2, $3 --> $1, $2, $3 otherwise
        elsif ($char[$i] =~ /\A \$ ((?>[1-9][0-9]*)) \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }
        elsif ($char[$i] =~ /\A \$ \{ (?>\s*) ((?>[1-9][0-9]*)) (?>\s*) \} \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo[ ... ] --> $ $foo->[ ... ]
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \[ (?:$qq_bracket)*? \] ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo{ ... } --> $ $foo->{ ... }
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) ( \{ (?:$qq_brace)*? \} ) \z/oxms) {
            $char[$i] = e_capture($1.'->'.$2);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $$foo
        elsif ($char[$i] =~ /\A \$ ((?> \$ [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* )) \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $`, ${`}, $PREMATCH, ${PREMATCH}, ${^PREMATCH} --> Esjis::PREMATCH()
        elsif ($char[$i] =~ /\A ( \$` | \$\{`\} | \$ (?>\s*) PREMATCH  | \$ (?>\s*) \{ (?>\s*) PREMATCH  (?>\s*) \} | \$ (?>\s*) \{\^PREMATCH\}  ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::PREMATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::PREMATCH()]}';
            }
        }

        # $&, ${&}, $MATCH, ${MATCH}, ${^MATCH} --> Esjis::MATCH()
        elsif ($char[$i] =~ /\A ( \$& | \$\{&\} | \$ (?>\s*) MATCH     | \$ (?>\s*) \{ (?>\s*) MATCH     (?>\s*) \} | \$ (?>\s*) \{\^MATCH\}     ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::MATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::MATCH()]}';
            }
        }

        # $POSTMATCH, ${POSTMATCH}, ${^POSTMATCH} --> Esjis::POSTMATCH()
        elsif ($char[$i] =~ /\A (                 \$ (?>\s*) POSTMATCH | \$ (?>\s*) \{ (?>\s*) POSTMATCH (?>\s*) \} | \$ (?>\s*) \{\^POSTMATCH\} ) \z/oxmsgc) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(Esjis::POSTMATCH())]}';
            }
            else {
                $char[$i] = '@{[Esjis::POSTMATCH()]}';
            }
        }

        # ${ foo }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ((?> \s* [A-Za-z_][A-Za-z0-9_]*(?: ::[A-Za-z_][A-Za-z0-9_]*)* \s* )) \}                              \z/oxms) {
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # ${ ... }
        elsif ($char[$i] =~ /\A \$ (?>\s*) \{ ( .+ ) \} \z/oxms) {
            $char[$i] = e_capture($1);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # $scalar or @array
        elsif ($char[$i] =~ /\A [\$\@].+ /oxms) {
            $char[$i] = e_string($char[$i]);
            if ($ignorecase) {
                $char[$i] = '@{[Esjis::ignorecase(' . $char[$i] . ')]}';
            }
        }

        # quote character before ? + * {
        elsif (($i >= 1) and ($char[$i] =~ /\A [\?\+\*\{] \z/oxms)) {
            if ($char[$i-1] =~ /\A (?:[\x00-\xFF]|\\[0-7]{2,3}|\\x[0-9-A-Fa-f]{1,2}) \z/oxms) {
            }
            else {
                $char[$i-1] = '(?:' . $char[$i-1] . ')';
            }
        }
    }

    # make regexp string
    my $prematch = '';
#if MULTIBYTE_ANCHORING
    $prematch = "($anchor)";
#endif
    $modifier =~ tr/i//d;
    if ($left_e > $right_e) {
        return join '', $ope, $delimiter, $prematch, '(?:', @char, '>]}' x ($left_e - $right_e), ')', $matched, $end_delimiter, $modifier;
    }
    return     join '', $ope, $delimiter, $prematch, '(?:', @char,                               ')', $matched, $end_delimiter, $modifier;
}

#
# escape regexp (s'here'' or s'here''b)
#
sub e_s1_q {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;
    $modifier ||= '';

    $modifier =~ tr/p//d;
    if ($modifier =~ /([adlu])/oxms) {
        my $line = 0;
        for (my $i=0; my($package,$filename,$use_line,$subroutine) = caller($i); $i++) {
            if ($filename ne __FILE__) {
                $line = $use_line + (CORE::substr($_,0,pos($_)) =~ tr/\n//) + 1;
                last;
            }
        }
        die qq{Unsupported modifier "$1" used at line $line.\n};
    }

    $slash = 'div';

    # literal null string pattern
    if ($string eq '') {
        $modifier =~ tr/bB//d;
        $modifier =~ tr/i//d;
        return join '', $ope, $delimiter, $end_delimiter, $modifier;
    }

    # with /b /B modifier
    elsif ($modifier =~ tr/bB//d) {
        return e_s1_qb($ope,$delimiter,$end_delimiter,$string,$modifier);
    }

    # without /b /B modifier
    else {
        return e_s1_qt($ope,$delimiter,$end_delimiter,$string,$modifier);
    }
}

#
# escape regexp (s'here'')
#
sub e_s1_qt {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;

    my $ignorecase = ($modifier =~ /i/oxms) ? 1 : 0;

    # split regexp
    my @char = $string =~ /\G((?>
        [^\x81-\x9F\xE0-\xFC\\\[\$\@\/] |
        [\x81-\x9F\xE0-\xFC][\x00-\xFF] |
        \[\^                            |
        \[\:   (?>[a-z]+) \:\]          |
        \[\:\^ (?>[a-z]+) \:\]          |
        [\$\@\/]                        |
        \\     (?:$q_char)              |
               (?:$q_char)
    ))/oxmsg;

    # unescape character
    for (my $i=0; $i <= $#char; $i++) {
        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        elsif ($char[$i] =~ /\A ([\x80-\xFF].*) ([\\|\[\{\^]|\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # open character class [...]
        elsif ($char[$i] eq '[') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [...]
                    splice @char, $left, $right-$left+1, Esjis::charlist_qr(@char[$left+1..$right-1], $modifier);

                    $i = $left;
                    last;
                }
            }
        }

        # open character class [^...]
        elsif ($char[$i] eq '[^') {
            my $left = $i;
            if ($char[$i+1] eq ']') {
                $i++;
            }
            while (1) {
                if (++$i > $#char) {
                    die __FILE__, ": Unmatched [] in regexp\n";
                }
                if ($char[$i] eq ']') {
                    my $right = $i;

                    # [^...]
                    splice @char, $left, $right-$left+1, Esjis::charlist_not_qr(@char[$left+1..$right-1], $modifier);

                    $i = $left;
                    last;
                }
            }
        }

        # escape $ @ / and \
        elsif ($char[$i] =~ /\A [\$\@\/\\] \z/oxms) {
            $char[$i] = '\\' . $char[$i];
        }

        # rewrite character class or escape character
        elsif (my $char = character_class($char[$i],$modifier)) {
            $char[$i] = $char;
        }

        # /i modifier
        elsif ($ignorecase and ($char[$i] =~ /\A [\x00-\xFF] \z/oxms) and (Esjis::uc($char[$i]) ne Esjis::fc($char[$i]))) {
            if (CORE::length(Esjis::fc($char[$i])) == 1) {
                $char[$i] = '['   . Esjis::uc($char[$i])       . Esjis::fc($char[$i]) . ']';
            }
            else {
                $char[$i] = '(?:' . Esjis::uc($char[$i]) . '|' . Esjis::fc($char[$i]) . ')';
            }
        }

        # quote character before ? + * {
        elsif (($i >= 1) and ($char[$i] =~ /\A [\?\+\*\{] \z/oxms)) {
            if ($char[$i-1] =~ /\A [\x00-\xFF] \z/oxms) {
            }
            else {
                $char[$i-1] = '(?:' . $char[$i-1] . ')';
            }
        }
    }

    $modifier =~ tr/i//d;
    $delimiter     = '/';
    $end_delimiter = '/';
    my $prematch = '';
#if MULTIBYTE_ANCHORING
    $prematch = "($anchor)";
#endif
    return join '', $ope, $delimiter, $prematch, '(?:', @char, ')', $matched, $end_delimiter, $modifier;
}

#
# escape regexp (s'here''b)
#
sub e_s1_qb {
    my($ope,$delimiter,$end_delimiter,$string,$modifier) = @_;

    # split regexp
    my @char = $string =~ /\G (?>[^\\]|\\\\|[\x00-\xFF]) /oxmsg;

    # unescape character
    for (my $i=0; $i <= $#char; $i++) {
        if (0) {
        }

        # remain \\
        elsif ($char[$i] eq '\\\\') {
        }

        # escape $ @ / and \
        elsif ($char[$i] =~ /\A [\$\@\/\\] \z/oxms) {
            $char[$i] = '\\' . $char[$i];
        }
    }

    $delimiter     = '/';
    $end_delimiter = '/';
    my $prematch = '';
#if MULTIBYTE_ANCHORING
    $prematch = q{(\G[\x00-\xFF]*?)};
#endif
    return join '', $ope, $delimiter, $prematch, '(?:', @char, ')', $matched, $end_delimiter, $modifier;
}

#
# escape regexp (s''here')
#
sub e_s2_q {
    my($ope,$delimiter,$end_delimiter,$string) = @_;

    $slash = 'div';

    my @char = $string =~ / \G (?>[^\x81-\x9F\xE0-\xFC\\]|\\\\|$q_char) /oxmsg;
    for (my $i=0; $i <= $#char; $i++) {
        if (0) {
        }

#if ESCAPE_SECOND_OCTET
        # escape last octet of multiple-octet
        elsif ($char[$i] =~ /\A ([\x80-\xFF].*) (\Q$delimiter\E|\Q$end_delimiter\E) \z/xms) {
            $char[$i] = $1 . '\\' . $2;
        }
        elsif (defined($char[$i+1]) and ($char[$i+1] eq '\\') and ($char[$i] =~ /\A ([\x80-\xFF].*) (\\) \z/xms)) {
            $char[$i] = $1 . '\\' . $2;
        }

#endif
        # not escape \\
        elsif ($char[$i] =~ /\A \\\\ \z/oxms) {
        }

        # escape $ @ / and \
        elsif ($char[$i] =~ /\A [\$\@\/\\] \z/oxms) {
            $char[$i] = '\\' . $char[$i];
        }
    }
#if ESCAPE_SECOND_OCTET
    if (defined($char[-1]) and ($char[-1] =~ /\A ([\x80-\xFF].*) (\\) \z/xms)) {
        $char[-1] = $1 . '\\' . $2;
    }
#endif

    return join '', $ope, $delimiter, @char,   $end_delimiter;
}

#
# escape regexp (s/here/and here/modifier)
#
sub e_sub {
    my($variable,$delimiter1,$pattern,$end_delimiter1,$delimiter2,$replacement,$end_delimiter2,$modifier) = @_;
    $modifier ||= '';

    $modifier =~ tr/p//d;
    if ($modifier =~ /([adlu])/oxms) {
        my $line = 0;
        for (my $i=0; my($package,$filename,$use_line,$subroutine) = caller($i); $i++) {
            if ($filename ne __FILE__) {
                $line = $use_line + (CORE::substr($_,0,pos($_)) =~ tr/\n//) + 1;
                last;
            }
        }
        die qq{Unsupported modifier "$1" used at line $line.\n};
    }

    if ($variable eq '') {
        $variable      = '$_';
        $bind_operator = ' =~ ';
    }

    $slash = 'div';

    # P.128 Start of match (or end of previous match): \G
    # P.130 Advanced Use of \G with Perl
    # in Chapter 3: Overview of Regular Expression Features and Flavors
    # P.312 Iterative Matching: Scalar Context, with /g
    # in Chapter 7: Perl
    # of ISBN 0-596-00289-0 Mastering Regular Expressions, Second edition

    # P.181 Where You Left Off: The \G Assertion
    # in Chapter 5: Pattern Matching
    # of ISBN 0-596-00027-8 Programming Perl Third Edition.

    # P.220 Where You Left Off: The \G Assertion
    # in Chapter 5: Pattern Matching
    # of ISBN 978-0-596-00492-7 Programming Perl 4th Edition.

    my $e_modifier = $modifier =~ tr/e//d;
    my $r_modifier = $modifier =~ tr/r//d;

    my $my = '';
    if ($variable =~ s/\A \( (?>\s*) ( (?>(?: local \b | my \b | our \b | state \b )?) .+ ) \) \z/$1/oxms) {
        $my = $variable;
        $variable =~ s/ (?: local \b | my \b | our \b | state \b ) (?>\s*) //oxms;
        $variable =~ s/ = .+ \z//oxms;
    }

    (my $variable_basename = $variable) =~ s/ [\[\{].* \z//oxms;
    $variable_basename =~ s/ \s+ \z//oxms;

    # quote replacement string
    my $e_replacement = '';
    if ($e_modifier >= 1) {
        $e_replacement = e_qq('', '', '', $replacement);
        $e_modifier--;
    }
    else {
        if ($delimiter2 eq "'") {
            $e_replacement = e_s2_q('qq', '/',         '/',             $replacement);
        }
        else {
            $e_replacement = e_qq  ('qq', $delimiter2, $end_delimiter2, $replacement);
        }
    }

    my $sub = '';

    # with /r
    if ($r_modifier) {
        if (0) {
        }

#if MULTIBYTE_ANCHORING
        # s///gr with multibyte anchoring
        elsif ($modifier =~ /g/oxms) {
            $sub = sprintf(
                #                               1                                                2   3                                  4   5
                q<CORE::eval{local $Esjis::re_t=%s; local $Esjis::re_a=''; while($Esjis::re_t =~ %s){%s local $^W=0; local $Esjis::re_r=%s; %s$Esjis::re_t="$Esjis::re_a${1}$Esjis::re_r$'"; pos($Esjis::re_t)=length "$Esjis::re_a${1}$Esjis::re_r"; $Esjis::re_a=substr($Esjis::re_t,0,pos($Esjis::re_t)); } return $Esjis::re_t}>,

                $variable,                                                       #  1
                ($delimiter1 eq "'") ?                                           #  2
                e_s1_q('m', $delimiter1, $end_delimiter1, $pattern, $modifier) : #  :
                e_s1  ('m', $delimiter1, $end_delimiter1, $pattern, $modifier),  #  :
                $s_matched,                                                      #  3
                $e_replacement,                                                  #  4
                '$Esjis::re_r=CORE::eval $Esjis::re_r; ' x $e_modifier,          #  5
            );
        }

#endif
        # s///gr without multibyte anchoring
        elsif ($modifier =~ /g/oxms) {
            $sub = sprintf(
                #                               1                         2   3                                  4   5
                q<CORE::eval{local $Esjis::re_t=%s; while($Esjis::re_t =~ %s){%s local $^W=0; local $Esjis::re_r=%s; %s$Esjis::re_t="$`$Esjis::re_r$'"; pos($Esjis::re_t)=length "$`$Esjis::re_r"; } return $Esjis::re_t}>,

                $variable,                          