
# perl_fork_utils

### NAME

    Fork::Utils - set of usefull methods to work with processes and signals on Linux platform

### SYNOPSIS

```perl
    use Fork::Utils qw/ safe_exec /;
    use POSIX ();

    my $sig_action = sub { printf("SIG%s was received\n", $_[0]); };

    $SIG{TERM} = $SIG{INT} = $SIG{QUIT} = $SIG{ALRM} = $sig_action;

    alarm(1);

    my $result = safe_exec(
        args => [ @params ],
        code => sub {

            my @args = @_;

            my $pending_sigset = new POSIX::SigSet ();

            sleep(2);

            if ( POSIX::sigpending( $pending_sigset ) == -1 ) {
              die("sigpending error has occurred");
            }

            if ( $pending_sigset->ismember( POSIX::SIGTERM ) ) {
                printf("%s is pending\n", 'SIGTERM');
            }

            if ( $pending_sigset->ismember( POSIX::SIGINT ) ) {
                printf("%s is pending\n", 'SIGINT');
            }

            if ( $pending_sigset->ismember( POSIX::SIGQUIT ) ) {
                printf("%s is pending\n", 'SIGQUIT');
            }

            if ( $pending_sigset->ismember( POSIX::SIGALRM ) ) {
                printf("%s is pending\n", 'SIGALRM');
            }
        },
        sigset => [qw/ ALRM TERM INT QUIT /]
    );

    if (my $error = $@) {
        STDERR->print("Error: $error\n");
    }

    alarm(0);

    printf("Good bye\n")
```

    The possible output of program is shown below (just press Ctrl+c during the execution to get this certain output):

```text
    SIGINT is pending
    SIGALRM is pending
    SIGINT was received
    SIGALRM was received
    Good bye
```

### DESCRIPTION

    This package provides some methods that can be helpfull while working
    with sub-processes and signals.

##### safe_exec
    Gets a hash with arguments, one of them is code reference is required to
    be executed in safe context. "Safe context" means context which can't be
    accidently interrupted by some signals.

    This method receives list of signals required to be blocked while code
    execution. Once code is executed the original signal mask will be
    restored.

    Any signal (except KILL, STOP) can be blocked.

    The signal names can be taken from $Config{'sig_names'}.

    Returns a result of mentioned code reference as "$code->( @$args )".
    Be aware that in current implementation this method can't return the list.
    The return value looks like the one shown below:

```perl
        my $result = $code->( @$args );
```

    In case of any error in the executed code reference the standard $@
    variable will be set.

###### code

          it's a code reference to be executed in safe context

###### args

          it's an array reference of arguments required to be passed into C<code> (see above)

###### sigset

          it's an array reference of signal names to be blocked while executing the C<code> (see above)

###### replace_mask

          It's a flag, by default it's turned off.
  
          If it's off than passed signals will be added to the current signal mask,
          otherwise mask will be replaced with new one built with mentioned signals

### AUTHOR
    Chernenko Dmitiry cdn@cpan.org

### LICENSE
    This program is free software; you can redistribute it and/or modify it under the terms of the the Artistic License (2.0).
