package Text::Amuse::Utils;

use utf8;
use strict;
use warnings;

=head1 NAME

Text::Amuse::Output - Internal module for L<Text::Amuse> output

=head1 FUNCTIONS

=head2 language_mapping

Return an hashref with the ISO language codes to Babel ones.

=head2 get_latex_lang($iso)

Return the babel name of the ISO language code. If missing or invalid return 'english'.

=head2 has_babel_ldf($language_name)

Return true if the babel name passed has an .ldf file.

=head2 lang_code_is_rtl($iso)

Return true if the language is RTL.

=head2 language_code_locale_captions($iso)

If the language is aliased, return an hashref with a C<_base_> key
pointing to the base language, and the other keys should be use to set
the captions.

E.g.

  \setlocalecaption{english}{contents}{Table of Contents}

Requires babel 3.51 (2020-10-27)

=cut


# this comes from /opt/amusewiki-texlive/2021/texmf-dist/tex/generic/babel/locale/*/*tex

sub language_mapping {
    return {
            af => 'afrikaans',
            agq => 'aghem',
            ak => 'akan',
            am => 'amharic',
            ar => 'arabic',
            # ar => 'arabic-algeria',
            # ar => 'arabic-dz',
            # ar => 'arabic-ma',
            # ar => 'arabic-morocco',
            # ar => 'arabic-sy',
            # ar => 'arabic-syria',
            asa => 'asu',
            ast => 'asturian',
            as => 'assamese',
            az => 'azerbaijani',
            # az => 'azerbaijani-cyrillic',
            # az => 'azerbaijani-cyrl',
            # az => 'azerbaijani-latin',
            # az => 'azerbaijani-latn',
            bas => 'basaa',
            bem => 'bemba',
            bez => 'bena',
            be => 'belarusian',
            bg => 'bulgarian',
            bm => 'bambara',
            bn => 'bengali',
            bo => 'tibetan',
            brx => 'bodo',
            br => 'breton',
            bs => 'bosnian',
            # bs => 'bosnian-cyrillic',
            # bs => 'bosnian-cyrl',
            bs => 'bosnian-latin',
            # bs => 'bosnian-latn',
            ca => 'catalan',
            ce => 'chechen',
            cgg => 'chiga',
            chr => 'cherokee',
            ckb => 'centralkurdish',
            # ckb => 'centralkurdish-latin',
            # ckb => 'centralkurdish-latn',
            ckb => 'sorani',
            cop => 'coptic',
            cs => 'czech',
            cu => 'churchslavic',
            # cu => 'churchslavic-cyrs',
            # cu => 'churchslavic-glag',
            # cu => 'churchslavic-glagolitic',
            # cu => 'churchslavic-oldcyrillic',
            # cu => 'churchslavonic',
            cy => 'welsh',
            dav => 'taita',
            da => 'danish',
            de => 'austrian',
            de => 'german',
            # de => 'german-at',
            # de => 'german-austria',
            # de => 'german-austria-traditional',
            # de => 'german-ch',
            # de => 'german-switzerland',
            # de => 'german-switzerland-traditional',
            # de => 'german-traditional',
            # de => 'naustrian',
            # de => 'ngerman',
            # de => 'nswissgerman',
            # de => 'swissgerman',
            dje => 'zarma',
            dsb => 'lowersorbian',
            dsb => 'lsorbian',
            dua => 'duala',
            dyo => 'jolafonyi',
            dz => 'dzongkha',
            ebu => 'embu',
            ee => 'ewe',
            el => 'greek',
            # el => 'monotonicgreek',
            # el => 'polytonicgreek',
            # en => 'american',
            # en => 'australian',
            # en => 'british',
            # en => 'canadian',
            en => 'english',
            # en => 'english-au',
            # en => 'english-australia',
            # en => 'english-ca',
            # en => 'english-canada',
            # en => 'english-gb',
            # en => 'english-newzealand',
            # en => 'english-nz',
            # en => 'english-unitedkingdom',
            # en => 'english-unitedstates',
            # en => 'english-us',
            # en => 'newzealand',
            # en => 'ukenglish',
            # en => 'usenglish',
            eo => 'esperanto',
            # es => 'mexican',
            es => 'spanish',
            # es => 'spanish-mexico',
            # es => 'spanish-mx',
            et => 'estonian',
            eu => 'basque',
            ewo => 'ewondo',
            fa => 'persian',
            ff => 'fulah',
            fil => 'filipino',
            fi => 'finnish',
            fo => 'faroese',
            fr => 'french',
            # fr => 'french-be',
            # fr => 'french-belgium',
            # fr => 'french-ca',
            # fr => 'french-canada',
            # fr => 'french-ch',
            # fr => 'french-lu',
            # fr => 'french-luxembourg',
            # fr => 'french-switzerland',
            fur => 'friulian',
            fy => 'westernfrisian',
            ga => 'irish',
            gd => 'scottishgaelic',
            gl => 'galician',
            grc => 'ancientgreek',
            guz => 'gusii',
            gu => 'gujarati',
            gv => 'manx',
            haw => 'hawaiian',
            ha => 'hausa',
            # ha => 'hausa-gh',
            # ha => 'hausa-ghana',
            # ha => 'hausa-ne',
            # ha => 'hausa-niger',
            he => 'hebrew',
            hi => 'hindi',
            hr => 'croatian',
            hsb => 'uppersorbian',
            # hsb => 'usorbian',
            hu => 'hungarian',
            # hu => 'magyar',
            hy => 'armenian',
            ia => 'interlingua',
            id => 'indonesian',
            ig => 'igbo',
            ii => 'sichuanyi',
            is => 'icelandic',
            it => 'italian',
            ja => 'japanese',
            jgo => 'ngomba',
            jmc => 'machame',
            kab => 'kabyle',
            kam => 'kamba',
            ka => 'georgian',
            kde => 'makonde',
            kea => 'kabuverdianu',
            khq => 'koyrachiini',
            ki => 'kikuyu',
            kkj => 'kako',
            kk => 'kazakh',
            kln => 'kalenjin',
            kl => 'kalaallisut',
            kmr => 'kurmanji',
            # kmr => 'northernkurdish',
            # kmr => 'northernkurdish-arab',
            # kmr => 'northernkurdish-arabic',
            km => 'khmer',
            kn => 'kannada',
            kok => 'konkani',
            ko => 'korean',
            # ko => 'korean-han',
            # ko => 'korean-hani',
            ksb => 'shambala',
            ksf => 'bafia',
            ksh => 'colognian',
            ks => 'kashmiri',
            kw => 'cornish',
            ky => 'kyrgyz',
            lag => 'langi',
            # la => 'classiclatin',
            # la => 'ecclesiasticlatin',
            la => 'latin',
            # la => 'medievallatin',
            lb => 'luxembourgish',
            lg => 'ganda',
            lkt => 'lakota',
            ln => 'lingala',
            lo => 'lao',
            lrc => 'northernluri',
            lt => 'lithuanian',
            luo => 'luo',
            luy => 'luyia',
            lu => 'lubakatanga',
            lv => 'latvian',
            mas => 'masai',
            mer => 'meru',
            mfe => 'morisyen',
            mgh => 'makhuwameetto',
            mgo => 'meta',
            mg => 'malagasy',
            mk => 'macedonian',
            ml => 'malayalam',
            mn => 'mongolian',
            mr => 'marathi',
            ms => 'malay',
            # ms => 'malay-bn',
            # ms => 'malay-brunei',
            # ms => 'malay-sg',
            # ms => 'malay-singapore',
            mt => 'maltese',
            mua => 'mundang',
            my => 'burmese',
            mzn => 'mazanderani',
            naq => 'nama',
            nb => 'norwegianbokmal',
            nd => 'northndebele',
            ne => 'nepali',
            nl => 'dutch',
            nmg => 'kwasio',
            nnh => 'ngiemboon',
            # nn => 'norwegiannynorsk',
            nn => 'nynorsk',
            no => 'norsk',
            # no => 'norwegian',
            nus => 'nuer',
            nyn => 'nyankole',
            oc => 'occitan',
            om => 'oromo',
            or => 'oriya',
            os => 'ossetic',
            pa => 'punjabi',
            # pa => 'punjabi-arab',
            # pa => 'punjabi-arabic',
            # pa => 'punjabi-gurmukhi',
            # pa => 'punjabi-guru',
            pl => 'polish',
            pms => 'piedmontese',
            ps => 'pashto',
            # pt => 'brazilian',
            pt => 'portuguese',
            # pt => 'portuguese-br',
            # pt => 'portuguese-brazil',
            # pt => 'portuguese-portugal',
            # pt => 'portuguese-pt',
            qu => 'quechua',
            rm => 'romansh',
            rn => 'rundi',
            rof => 'rombo',
            ro => 'romanian',
            ru => 'russian',
            rwk => 'rwa',
            rw => 'kinyarwanda',
            sah => 'sakha',
            saq => 'samburu',
            sa => 'sanskrit',
            # sa => 'sanskrit-beng',
            # sa => 'sanskrit-bengali',
            # sa => 'sanskrit-deva',
            # sa => 'sanskrit-devanagari',
            # sa => 'sanskrit-gujarati',
            # sa => 'sanskrit-gujr',
            # sa => 'sanskrit-kannada',
            # sa => 'sanskrit-knda',
            # sa => 'sanskrit-malayalam',
            # sa => 'sanskrit-mlym',
            # sa => 'sanskrit-telu',
            # sa => 'sanskrit-telugu',
            sbp => 'sangu',
            seh => 'sena',
            ses => 'koyraborosenni',
            se => 'northernsami',
            # se => 'samin',
            sg => 'sango',
            shi => 'tachelhit',
            # shi => 'tachelhit-latin',
            # shi => 'tachelhit-latn',
            # shi => 'tachelhit-tfng',
            # shi => 'tachelhit-tifinagh',
            si => 'sinhala',
            sk => 'slovak',
            # sl => 'slovene',
            sl => 'slovenian',
            smn => 'inarisami',
            sn => 'shona',
            so => 'somali',
            sq => 'albanian',
            # sr => 'serbian',
            # sr => 'serbian-cyrillic',
            # sr => 'serbian-cyrillic-bosniaherzegovina',
            # sr => 'serbian-cyrillic-kosovo',
            # sr => 'serbian-cyrillic-montenegro',
            # sr => 'serbian-cyrl',
            # sr => 'serbian-cyrl-ba',
            # sr => 'serbian-cyrl-me',
            # sr => 'serbian-cyrl-xk',
            sr => 'serbian',
            # sr => 'serbian-latin-bosniaherzegovina',
            # sr => 'serbian-latin-kosovo',
            # sr => 'serbian-latin-montenegro',
            # sr => 'serbian-latn',
            # sr => 'serbian-latn-ba',
            # sr => 'serbian-latn-me',
            # sr => 'serbian-latn-xk',
            sv => 'swedish',
            sw => 'swahili',
            syr => 'syriac',
            ta => 'tamil',
            teo => 'teso',
            te => 'telugu',
            th => 'thai',
            ti => 'tigrinya',
            tk => 'turkmen',
            to => 'tongan',
            tr => 'turkish',
            twq => 'tasawaq',
            tzm => 'centralatlastamazight',
            ug => 'uyghur',
            uk => 'ukrainian',
            ur => 'urdu',
            uz => 'uzbek',
            # uz => 'uzbek-arab',
            # uz => 'uzbek-arabic',
            # uz => 'uzbek-cyrillic',
            # uz => 'uzbek-cyrl',
            # uz => 'uzbek-latin',
            # uz => 'uzbek-latn',
            vai => 'vai',
            # vai => 'vai-latin',
            # vai => 'vai-latn',
            # vai => 'vai-vai',
            # vai => 'vai-vaii',
            vi => 'vietnam',
            vi => 'vietnamese',
            vun => 'vunjo',
            wae => 'walser',
            xog => 'soga',
            yav => 'yangben',
            yi => 'yiddish',
            yo => 'yoruba',
            yue => 'cantonese',
            zgh => 'standardmoroccantamazight',
            # zh => 'chinese',
            # zh => 'chinese-hans',
            # zh => 'chinese-hans-hk',
            # zh => 'chinese-hans-mo',
            # zh => 'chinese-hans-sg',
            # zh => 'chinese-hant',
            # zh => 'chinese-hant-hk',
            # zh => 'chinese-hant-mo',
            zh => 'chinese',
            # zh => 'chinese-simplified-hongkongsarchina',
            # zh => 'chinese-simplified-macausarchina',
            # zh => 'chinese-simplified-singapore',
            # zh => 'chinese-traditional',
            # zh => 'chinese-traditional-hongkongsarchina',
            # zh => 'chinese-traditional-macausarchina',
            zu => 'zulu',

            # aliases, see below. cebuano and tagalog
            ceb => 'filipino',
            tl => 'filipino',
           };
}

# and this from the documentation, 2022-01-29

sub has_babel_ldf {
    my $lang = shift;
    my %langs = (
                 afrikaans        => 1,
                 azerbaijani      => 1,
                 basque           => 1,
                 breton           => 1,
                 bulgarian        => 1,
                 catalan          => 1,
                 croatian         => 1,
                 czech            => 1,
                 danish           => 1,
                 dutch            => 1,
                 english          => 1,
                 USenglish        => 1,
                 american         => 1,
                 UKenglish        => 1,
                 british          => 1,
                 canadian         => 1,
                 australian       => 1,
                 newzealand       => 1,
                 esperanto        => 1,
                 estonian         => 1,
                 finnish          => 1,
                 french           => 1,
                 francais         => 1,
                 canadien         => 1,
                 acadian          => 1,
                 galician         => 1,
                 austrian         => 1,
                 german           => 1,
                 germanb          => 1,
                 ngerman          => 1,
                 naustrian        => 1,
                 # greek            => 1,
                 polutonikogreek  => 1,
                 hebrew           => 1,
                 icelandic        => 1,
                 indonesian       => 1,
                 bahasa           => 1,
                 indon            => 1,
                 bahasai          => 1,
                 interlingua      => 1,
                 irish            => 1,
                 italian          => 1,
                 latin            => 1,
                 lowersorbian     => 1,
                 malay            => 1,
                 melayu           => 1,
                 bahasam          => 1,
                 samin            => 1,
                 norsk            => 1,
                 polish           => 1,
                 portuguese       => 1,
                 brazilian        => 1,
                 romanian         => 1,
                 russian          => 1,
                 scottish         => 1,
                 spanish          => 1,
                 slovak           => 1,
                 slovene          => 1,
                 swedish          => 1,
                 serbian          => 1,
                 turkish          => 1,
                 ukrainian        => 1,
                 uppersorbian     => 1,
                 welsh            => 1,
                );
    if (exists $langs{$lang}) {
        return $langs{$lang};
    }
    elsif (my $alias = language_mapping()->{$lang}) {
        if (exists $langs{$alias}) {
            return $langs{$alias};
        }
    }
    return 0;
}

sub get_latex_lang {
    my $lang = shift || 'en';
    return language_mapping()->{$lang} || 'english';
}

sub lang_code_is_rtl {
    my $lang = shift || 'en';
    my %rtl = (
               ar => 1,
               he => 1,
               fa => 1,
              );
    return $rtl{$lang};
}

sub language_code_locale_captions {
    my $code = shift;
    # usually we need only the doc
    my %aliases = (
                   ceb => {
                           _base_ => "filipino", # not used for the moment.
                           contents => "Talaan sa mga Sulod",
                          },
                   tl => {
                          _base_ => "filipino",
                          contents => "Talaan ng mga nilalaman",
                         },
                  );
    return $aliases{$code || ''};
}

1;
