package App::Zealc::Command::expand;

use 5.014000;
use strict;
use warnings;

our $VERSION = '0.000_001';

use App::Zealc '-command';

sub usage_desc { "%c expand %o pattern"}

sub opt_spec {
	(['family|f!', 'Include docset family in output']),
}

sub validate_args {
	my ($self, $opts, $args) = @_;
	my @args = @$args;
	$self->usage_error("No pattern specified") unless @args;
	$self->usage_error("Too many arguments") if @args > 1;
}

sub execute {
	my ($self, $opts, $args) = @_;
	my @results = sort { $a->name cmp $b->name } $self->app->zeal->query($args->[0]);
	exit 1 unless @results;
	say join "\n", map {
		my $family = $_->docset->family;
		$opts->{family} ? "$family:" . $_->name : $_->name
	} @results
}

1;
__END__

=encoding utf-8

=head1 NAME

App::Zealc::Command::expand - list all documents that match a pattern

=head1 SYNOPSIS

  zealc expand perlmo%
  # perlmod
  # perlmodinstall
  # perlmodlib
  # perlmodstyle

  zealc expand -f perl_os
  # perl:perldos
  # perl:perlvos

=head1 DESCRIPTION

The expand command lists all documents that match a case-insensitive
SQL LIKE pattern.

A SQL LIKE pattern is similar to a shell glob. The "%" character
matches zero or more characters (like "*" in a shell glob or ".*" in a
regex) and "_" matches exactly one character (like "?" in a shell glob
or "." in a regex). Matching is case-insensitive.

=head1 SEE ALSO

L<zealc>, L<Zeal>

=head1 AUTHOR

Marius Gavrilescu, E<lt>marius@ieval.roE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 by Marius Gavrilescu

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.20.1 or,
at your option, any later version of Perl 5 you may have available.


=cut
