Colorizable
===========

A Raku module for colorizing text using ANSI escape sequences. 

Synopsis
--------

    use Colorizable;

    my $string = "Hello" but Colorizable;

    given $string {
        say .colorize(blue);      # blue text
        say .colorize(blue, red); # blue text on red background
    }

Installation
------------

From source code:

    $ git clone https://gitlab.com/uzluisf/raku-colorized.git
    $ cd raku-colorized/
    $ zef install .

Description
-----------

The module allows to colorize text using ANSI escape sequences by composing the `Colorizable` role into a string. After doing this, the method `colorize` can be used to set the text color, background color and text effects (e.g., bold, italic, etc.):

    use Colorizable;

    my $planet = "Arrakis" but Colorizable;
    say $planet.colorize(:fg(blue), :bg(red), :mo(bold));

If composed into a class, `Colorizable` assumes the class implements the appropriate `Str` method:

    use Colorizable;

    class A does Colorizable {
    }

    say A.new.colorize(blue); #=> A<94065101711120>, in blue

    class B does Colorizable {
        method Str { 'Class B' }
    }

    say B.new.colorize(blue); #=> class B, in blue

Invocations to the `colorize` method are chainable:

    use Colorizable;

    my $text = "Bold red text on yellow background" but Colorizable;

    say $text
    .colorize(:fg(red))
    .colorize(:bg(yellow))
    .colorize(:mo(bold));

Although this might not be advisable, you can make regular strings colorizable by augmenting the `Str` class: 

    use Colorizable;
    use MONKEY-TYPING;

    # making strings colorizable
    augment class Str {
        also does Colorizable;
    }

    say "Bold red text on yellow background"
    .colorize(:fg(red))
    .colorize(:bg(yellow))
    .colorize(:mo(bold));

Note the use of the pragma [`MONKEY-TYPING`](https://docs.raku.org/language/pragmas#index-entry-MONKEY-TYPING__pragma).

Methods
-------

colorize
========

`method colorize($fg, $bg, $mo)`

Change color of string.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.colorize(yellow);          # yellow text
    say $str.colorize('yellow');        # yellow text
    say $str.colorize(blue, red, bold); # bold blue text in red background

**Parameters:**

  * `$fg` the string's foreground color

  * `$bg` the string's background color

  * `$mo` the string's mode

colorize
========

`method colorize(:fg(:$foreground), :bg(:$background), :mo(:$mode))`

Change color of string.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.colorize(:fg(yellow));           # yellow text
    say $str.colorize(:foreground('yellow')); # yellow text
    say $str.colorize(:fg(blue), :bg(red));   # blue text in red background

**Parameters:**

  * `$fg` the string's foreground color

  * `$bg` the string's background color

  * `$mo` the string's mode

colors
======

`method colors(--> List)`

Return list of available colors.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.colors.head(3).join(' ');

modes
=====

`method modes( --> List )`

Return list of available modes.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.modes.head(3).join(' ');

is-colorized
============

`method is-colorized( --> Bool )`

Return true if the string is colorized. Otherwise, false. A string is considered colorized if it has at least one element (foreground, background, or mode) set.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.is-colorized;

uncolorize
==========

`method uncolorize( --> Str )`

Return uncolorized string.

**Examples:**

    my $str = "Hello" does Colorizable;
    say $str.uncolorize;

color-samples
=============

`method color-samples()`

Display color samples.

**Examples:**

    my $str = "Hello" does Colorizable;
    $str.color-samples; # display color samples with string

To-Do
-----

  * Add tests

Acknowledgement
---------------

This module is inspired by the Ruby gem [colorize](https://github.com/fazibear/colorize).

Authors
-------

Luis F. Uceta

License
-------

You can use and distribute this module under the terms of the The Artistic License 2.0. See the LICENSE file included in this distribution for complete details.

The META6.json file of this distribution may be distributed and modified without restrictions or attribution.

