﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecommit/CodeCommitRequest.h>
#include <aws/codecommit/CodeCommit_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCommit {
namespace Model {

/**
 */
class GetCommentReactionsRequest : public CodeCommitRequest {
 public:
  AWS_CODECOMMIT_API GetCommentReactionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetCommentReactions"; }

  AWS_CODECOMMIT_API Aws::String SerializePayload() const override;

  AWS_CODECOMMIT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the comment for which you want to get reactions information.</p>
   */
  inline const Aws::String& GetCommentId() const { return m_commentId; }
  inline bool CommentIdHasBeenSet() const { return m_commentIdHasBeenSet; }
  template <typename CommentIdT = Aws::String>
  void SetCommentId(CommentIdT&& value) {
    m_commentIdHasBeenSet = true;
    m_commentId = std::forward<CommentIdT>(value);
  }
  template <typename CommentIdT = Aws::String>
  GetCommentReactionsRequest& WithCommentId(CommentIdT&& value) {
    SetCommentId(std::forward<CommentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional. The Amazon Resource Name (ARN) of the user or identity for which
   * you want to get reaction information.</p>
   */
  inline const Aws::String& GetReactionUserArn() const { return m_reactionUserArn; }
  inline bool ReactionUserArnHasBeenSet() const { return m_reactionUserArnHasBeenSet; }
  template <typename ReactionUserArnT = Aws::String>
  void SetReactionUserArn(ReactionUserArnT&& value) {
    m_reactionUserArnHasBeenSet = true;
    m_reactionUserArn = std::forward<ReactionUserArnT>(value);
  }
  template <typename ReactionUserArnT = Aws::String>
  GetCommentReactionsRequest& WithReactionUserArn(ReactionUserArnT&& value) {
    SetReactionUserArn(std::forward<ReactionUserArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An enumeration token that, when provided in a request, returns the next batch
   * of the results. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  GetCommentReactionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A non-zero, non-negative integer used to limit the number of returned
   * results. The default is the same as the allowed maximum, 1,000.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline GetCommentReactionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_commentId;

  Aws::String m_reactionUserArn;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_commentIdHasBeenSet = false;
  bool m_reactionUserArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCommit
}  // namespace Aws
