from enum import Enum

from pydantic import BaseModel
from sqlalchemy import Column, Integer, String, Double, ForeignKey, CheckConstraint
from sqlalchemy.orm import relationship

from ..base import Base


class HealthCheckMetricType(Enum):
    BOOL = 'bool'
    PERCENT = 'pct'
    INTEGER = 'int'
    FLOAT = 'float'


class HealthCheckFlag(Enum):
    GREEN = 'green'
    YELLOW = 'yellow'
    RED = 'red'
    NONE = ''


class HealthCheck(Base):
    __tablename__ = 'health_check'
    id = Column(Integer, primary_key=True, index=True)
    description = Column(String)
    metric_value = Column(Double)
    _metric_type = Column(name='metric_type', type_=String, nullable=False, default=HealthCheckMetricType.FLOAT.value)
    warning = Column(String)
    _flag = Column(name='flag', type_=String, nullable=False, default=HealthCheckFlag.GREEN.value)
    report_id = Column(Integer, ForeignKey('report.id'))
    report = relationship('Report', back_populates='health_check')

    __table_args__ = (
        CheckConstraint(_metric_type.in_([e.value for e in HealthCheckMetricType]), name='metric_type_check'),
        CheckConstraint(_flag.in_([e.value for e in HealthCheckFlag]), name='flag_check'),
    )

    @property
    def metric_type(self) -> HealthCheckMetricType:
        return HealthCheckMetricType(self._metric_type)

    @metric_type.setter
    def metric_type(self, enum_value: HealthCheckMetricType):
        self._metric_type = enum_value.value

    @property
    def flag(self) -> HealthCheckFlag:
        return HealthCheckFlag(self._flag)

    @flag.setter
    def flag(self, enum_value: HealthCheckFlag):
        self._flag = enum_value.value


class HealthCheckInDB(BaseModel):
    description: str
    metric_value: float
    metric_type: HealthCheckMetricType
    warning: str
    flag: HealthCheckFlag

    class Config:
        from_attributes = True


class HealthCheckCreate(BaseModel):
    description: str
    metric_value: float
    metric_type: HealthCheckMetricType
    warning: str
    flag: HealthCheckFlag
