#!/usr/bin/perl

=head1 NAME

  ez_moga.pl - Easy implementation of a very primitive multiobjective optimization algorithm

=head1 SYNOPSIS

  prompt% ./ez_moga.pl <population> <number of generations>

or

  prompt% perl p_peaks.pl <bits> <peaks> <population> <number of generations>

  Shows the values of the two floating-point components of the
  chromosome and finally the best value and fitness reached, which
  should be as close to 1 as possible.
  

=head1 DESCRIPTION  

A simple example of how to run an Evolutionary algorithm based on
Algorithm::Evolutionary. Tries to find the max of the bidimensional
Tide , and outputs the x and y coordinates, along with fitness. Best
fitness is close to 1. Around 50 generations should be enough, but
default is population and number of generations equal to 100.

=cut

use warnings;
use strict;

use Time::HiRes qw( gettimeofday tv_interval);

use lib qw(lib ../lib);
use Algorithm::Evolutionary qw( Individual::BitString Op::Easy_MO
				Op::Mutation Op::Crossover
				Fitness::ZDT1 );

use Algorithm::Evolutionary::Utils qw(entropy);

#----------------------------------------------------------#
my $popSize = shift || 128; #Population size
my $numGens = shift || 100; #Max number of generations
my $selection_rate = shift || 0.5;

#----------------------------------------------------------#
#Initial population
my @pop;
#Creamos $popSize individuos
my $bits_x_var = 8;
my $number_of_vars= 30;
my $bits = $bits_x_var * $number_of_vars;
for ( 0..$popSize ) {
  my $indi = Algorithm::Evolutionary::Individual::BitString->new( $bits );
  push( @pop, $indi );
}

#----------------------------------------------------------#
# Variation operators
my $m = Algorithm::Evolutionary::Op::Mutation->new( 1/$bits ); # Rate = 1
my $c = Algorithm::Evolutionary::Op::Crossover->new(2, 4 ); # Rate = 4

my $fitness = new  Algorithm::Evolutionary::Fitness::ZDT1 $bits_x_var;

#----------------------------------------------------------#
#Usamos estos operadores para definir una generación del algoritmo. Lo cual
# no es realmente necesario ya que este algoritmo define ambos operadores por
# defecto. Los parámetros son la función de fitness, la tasa de selección y los
# operadores de variación.
my $generation = Algorithm::Evolutionary::Op::Easy_MO->new( $fitness, 
							    $selection_rate, 
							    [$m, $c] ) ;

#Time
my $inicioTiempo = [gettimeofday()];

my $counter=0;
do {
  $generation->apply( \@pop );
  
  print "$counter : ", $pop[0]->asString(), "\n" ;
  $counter++;
} while( ($counter < $numGens) );


#----------------------------------------------------------#
#leemos el mejor resultado

#Mostramos los resultados obtenidos
print "Best is:\n\t ",$pop[0]->asString()," Fitness: ",$pop[0]->Fitness(),"\n";

print "\n\n\tTime: ", tv_interval( $inicioTiempo ) , "\n";

print "\n\tEvaluations: ", $fitness->evaluations(), "\n";

for ( my $p = 0; $p <= $#pop; $p ++ ) {
  print join( ",", @{$fitness->apply( $pop[$p] )}), "\n";
}

=head1 AUTHOR

Contributed by Pedro Castillo Valdivieso, modified by J. J. Merelo

=cut

=head1 Copyright
  
  This file is released under the GPL. See the LICENSE file included in this distribution,
  or go to http://www.fsf.org/licenses/gpl.txt

  CVS Info: $Date: 2009/11/17 19:19:41 $ 
  $Header: /cvsroot/opeal/Algorithm-Evolutionary/examples/ez_moga.pl,v 3.1 2009/11/17 19:19:41 jmerelo Exp $ 
  $Author: jmerelo $ 
  $Revision: 3.1 $
  $Name $

=cut
