package Games::ABC_Path::Solver::App;
$Games::ABC_Path::Solver::App::VERSION = '0.8.1';
use warnings;
use strict;


use parent 'Games::ABC_Path::Solver::Base';

use Getopt::Long qw/ GetOptionsFromArray /;
use Pod::Usage   qw/ pod2usage /;

use Games::ABC_Path::Solver::Board ();

sub _argv
{
    my $self = shift;

    if (@_)
    {
        $self->{_argv} = shift;
    }

    return $self->{_argv};
}

sub _init
{
    my ( $self, $args ) = @_;

    $self->_argv( [ @{ $args->{argv} } ] );

    return;
}

sub run
{
    my $self = shift;

    my $man          = 0;
    my $help         = 0;
    my $gen_template = 0;
    GetOptionsFromArray(
        $self->_argv,
        'help|h'          => \$help,
        man               => \$man,
        'gen-v1-template' => \$gen_template,
    ) or pod2usage(2);

    if ($help)
    {
        pod2usage(1);
    }
    elsif ($man)
    {
        pod2usage( -verbose => 2 );
    }
    elsif ($gen_template)
    {
        print <<'EOF';
ABC Path Solver Layout Version 1:
???????
?     ?
?     ?
?     ?
?   A ?
?     ?
???????
EOF
    }
    else
    {
        my $board_fn = shift( @{ $self->_argv } );

        if ( !defined($board_fn) )
        {
            die
"Filename not specified - usage: abc-path-solver.pl [filename]!";
        }

        my $solver = Games::ABC_Path::Solver::Board->input_from_file($board_fn);

        # Now let's do a neighbourhood inferring of the board.

        $solver->solve;

        foreach my $move ( @{ $solver->get_moves } )
        {
            print +( ' => ' x $move->get_depth() ), $move->get_text(), "\n";
        }

        print @{ $solver->get_successes_text_tables };
    }

    exit(0);
}


1;    # End of Games::ABC_Path::Solver::App

__END__

=pod

=encoding UTF-8

=head1 NAME

Games::ABC_Path::Solver::App - a class wrapping a command line app for
solving ABC Path

=head1 VERSION

version 0.8.1

=head1 SYNOPSIS

    #!/usr/bin/perl

    use strict;
    use warnings;

    use Games::ABC_Path::Solver::App;

    Games::ABC_Path::Solver::App->new({argv => \@ARGV, })->run;

=head1 FUNCTIONS

=head2 new

The constructor. Accepts a hash ref of named arguments. Currently only C<'argv'>
which should point to an array ref of command-line arguments.

=head2 run

Run the application based on the arguments in the constructor.

=head1 AUTHOR

Shlomi Fish, C<< <shlomif at cpan.org> >>

=head1 SEE ALSO

L<Games::ABC_Path::Solver> , L<Games::ABC_Path::Solver::Board> .

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/Games-ABC_Path-Solver>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=Games-ABC_Path-Solver>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/Games-ABC_Path-Solver>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/G/Games-ABC_Path-Solver>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=Games-ABC_Path-Solver>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=Games::ABC_Path::Solver>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-games-abc_path-solver at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=Games-ABC_Path-Solver>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/abc-path>

  git clone https://github.com/shlomif/abc-path

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/abc-path/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2025 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=cut
