package RackMan::Format::Bacula;

use strict;
use Carp;
use RackMan::File;


use constant {
    CONFIG_SECTION  => "format:bacula",
    DEFAULT_PATH    => "/etc/bacula/hosts",
};


#
# write()
# -----
sub write {
    my ($class, $args) = @_;

    my $rackdev = $args->{rackdev};
    my $rackman = $args->{rackman};
    my $name    = $rackdev->object_name;
    my $fqdn    = $rackdev->attributes->{FQDN} || $name;
    my ($short) = split /\./, $name;

    # fetch Bacula parameters
    my $password= $rackman->config->val(CONFIG_SECTION, "password", "***");
    my $catalog = $rackman->config->val(CONFIG_SECTION, "catalog", "MyCatalog");
    my $fdport  = $rackman->config->val(CONFIG_SECTION, "fdport");
    my $file_ret= $rackman->config->val(CONFIG_SECTION, "file_retention");
    my $job_ret = $rackman->config->val(CONFIG_SECTION, "job_retention");
    my $max_jobs= $rackman->config->val(CONFIG_SECTION, "max_jobs");
    my $bs_path = $rackman->config->val(CONFIG_SECTION, "write_bootstrap");

    # generate the config files
    my $config_client = RackMan::File->new;
    $config_client->name("$name-fd.conf");
    $config_client->add_content(
        "Client {\n",
        "    Name = $short-fd\n",
        "    Address = $fqdn\n",
        "    FDPort = $fdport\n"                    x!! $fdport,
        "    Catalog = $catalog\n",
        "    Password = \"$password\"\n",
        "    File Retention = $file_ret\n"          x!! $file_ret,
        "    Job Retention = $job_ret\n"            x!! $job_ret,
        "    Maximum Concurrent Jobs = $max_jobs\n" x!! $max_jobs,
        "}\n",
    );

    my $config_fileset = RackMan::File->new;
    my @files = $rackman->config->val(CONFIG_SECTION, "add_file");

    if (@files) {
        $config_fileset->name("$name.fileset.conf");
        $config_fileset->add_content(
            "FileSet {\n",
            "    Name = \"$short\"\n",
            "    Include {\n",
            "        Options {\n",
            "            signature = MD5\n",
            "        }\n",
        );
        $config_fileset->add_content(
            map "        File = $_\n", @files
        );
        $config_fileset->add_content(
            "    }\n",
            "}\n",
        );
    }

    my $config_job = RackMan::File->new;
    $config_job->name("$name.job.conf");
    $config_job->add_content(
        "Job {\n",
        "    Name = \"$short\"\n",
        "    JobDefs = \"Monthly\"\n",
        "    Client = $short-fd\n",
        "    FileSet = $short\n",
        "    Write Bootstrap = \"$bs_path\"\n"      x!! $bs_path,
        "    Maximum Concurrent Jobs = $max_jobs\n" x!! $max_jobs,
        "}\n",
    );

    # write the configuration files on disk
    my $path = $rackman->config->val(CONFIG_SECTION, "path", DEFAULT_PATH);
    my $scm = $rackman->get_scm({ path => $path });
    $scm->update;

    for my $file ($config_client, $config_fileset, $config_job) {
        next unless defined $file->name;
        $file->path($path);
        print "  + writing ", $file->fullpath, $/ if $args->{verbose};
        $file->write;
        $scm->add($file->name);
        $scm->commit($file->name, "generated by $class / $::PROGRAM v$::VERSION");
    }
}


__PACKAGE__

__END__

=pod

=head1 NAME

RackMan::Format::Bacula - Generate the Bacula config files for a given RackObject

=head1 SYNOPSIS

    use RackMan::Format::Bacula;

    RackMan::Format::Bacula->write({
        rackdev => $rackdev,  # a RackMan::Device instance
        rackman => $rackman,  # a RackMan instance
    });


=head1 DESCRIPTION

This module generates the Bacula configuration files for the given RackObject.


=head1 METHODS

=head2 write

Generate the file.

B<Arguments>

Arguments are expected as a hashref with the following keys:

=over

=item *

C<rackdev> - I<(mandatory)> a RackMan::Device instance

=item *

C<rackman> - I<(mandatory)> a RackMan instance

=item *

C<verbose> - I<(optional)> boolean, set to true to be verbose

=back


=head1 CONFIGURATION

This module gets its configuration from the C<[format:bacula]> section
of the main F<rack.conf>, with the following parameters:

=head2 path

Path of the directory to store the generated files.

=head2 catalog

Specify the name of the catalog resource to be used.
This parameter is required.

=head2 password

Specify the Bacula password.  This parameter is required.

=head2 fdport

Port number of the Bacula server.

=head2 file_retention

Bacula parameter C<File Retention>.

=head2 job_retention

Bacula parameter C<Job Retention>.

=head2 max_jobs

Bacula parameter C<Maximum Concurrent Jobs>.

=head2 write_bootstrap

Bacula parameter C<Write Bootstrap>.

=head2 add_file

Add the given path to the FileSet.

=head2 scm

Specify the SCM tool to use for versionning generated files.


=head1 AUTHOR

Sebastien Aperghis-Tramoni

=cut

