use strict;
use warnings;
use Test::More;
use Data::Dumper;

# Module usage test
use_ok('Net::SolarWinds::FileRotationBase');

# blank instance test
isa_ok( new Net::SolarWinds::FileRotationBase, 'Net::SolarWinds::FileRotationBase' );

# filename generated by isntance check
{
    my $fr = new Net::SolarWinds::FileRotationBase;

    # filename creation check
    cmp_ok( $fr->generate_filename, 'eq', './DefaultFile.log', 'default filename fetch' );
}

# single filename argument testing
{
    my $fr = new Net::SolarWinds::FileRotationBase( filename => 'test.txt', );
    isa_ok( $fr, 'Net::SolarWinds::FileRotationBase' );
    cmp_ok( $fr->generate_filename, 'eq', 'test.txt', 'filename fetch' );
}

# custom file rotation object creationc check
{
    my $fr = new Net::SolarWinds::FileRotationBase(
        basefilename => 'test',
        ext          => 'xxx',
        pathconncat  => '',
        folder       => '',
    );

    # filename creation check
    cmp_ok( $fr->generate_filename, 'eq', 'test.xxx', 'filename fetch' );

    # since we should have a fairly safe filename we can check and see if we
    # are able to generate a file handle
    my $fh = eval { $fr->get_fh };

    ok( !$@, 'should not explode when creating a local file handle' )
      or diag("error was $@");
}

# actual file rotation check
{

    # clean up temp files created, before we test
    unlink qw(text.xxx test.xxx.1 test.xxx.2 test.xxx.3);
    my $fr = new Net::SolarWinds::FileRotationBase(
        basefilename => 'test',
        ext          => 'xxx',
        pathconncat  => '',
        folder       => '',
        maxfilecount => 2,
    );

    # filename creation check
    cmp_ok( $fr->generate_filename, 'eq', 'test.xxx', 'filename fetch' );

    $fr->write_to_file("test 1\n");

    ok( -e 'test.xxx',        'file should exist' );
    ok( exists( $fr->{fh} ),  'file handle should exist' );
    ok( defined( $fr->{fh} ), 'file handle should be defined' );
    ok( $fr->close_fh,        'file handle should close without error' );
    ok( !exists( $fr->{fh} ), 'file handle should no longer exist!' )
      or diag Dumper($fr);

    #$DB::single=1;
    ok( $fr->rotate_files, 'file rotation check' );

    ok( -e 'test.xxx.1', 'rotated file should exist' );
    ok( !-e 'test.xxx',  'old file should not exist' );

    $fr->write_to_file("test 2\n");
    ok( -e 'test.xxx.1',   'rotated file should exist' );
    ok( -e 'test.xxx',     'current file should exist' );
    ok( !-e 'test.xxx.2',  'next rotated file not should exist' );
    ok( $fr->rotate_files, 'file rotation check' );

    ok( !-e 'test.xxx',   'old file should not exist' );
    ok( -e 'test.xxx.1',  'rotated file 1 should exist' );
    ok( -e 'test.xxx.2',  'rotated file 2 should exist' );
    ok( !-e 'test.xxx.3', 'rotated file 3 should not exist' );

    $fr->write_to_file("test 3\n");
    ok( -e 'test.xxx', 'file should exist' );

    ok( $fr->rotate_files, 'file rotation check' );

    ok( !-e 'test.xxx',   'old file should not exist' );
    ok( -e 'test.xxx.1',  'rotated file 1 should exist' );
    ok( -e 'test.xxx.2',  'rotated file 2 should exist' );
    ok( !-e 'test.xxx.3', 'rotated file 3 should not exist' );
}

{
    close(STDERR);
    unlink 'test.txt';
    my $fr = new Net::SolarWinds::FileRotationBase( filename => 'test.txt',mirror=>\*STDERR );
    isa_ok( $fr, 'Net::SolarWinds::FileRotationBase' );
    cmp_ok( $fr->generate_filename, 'eq', 'test.txt', 'filename fetch' );
    $fr->get_fh;
    warn('this is a test');
    $fr->get_fh->close;
    my $res=IO::File->new('test.txt')->getline;
    like($res,'/this is a test/','Should have dumped the warning from stderr to test.txt');
}
{
    close(STDERR);
    unlink 'test.txt';
    my $fr = new Net::SolarWinds::FileRotationBase( filename => 'test.txt');
    $fr->set_mirror(\*STDERR);
    isa_ok( $fr, 'Net::SolarWinds::FileRotationBase' );
    cmp_ok( $fr->generate_filename, 'eq', 'test.txt', 'filename fetch' );
    $fr->get_fh;
    warn('this is a test');
    $fr->get_fh->close;
    my $res=IO::File->new('test.txt')->getline;
    like($res,'/this is a test/','Should have dumped the warning from stderr to test.txt');
}
END {
  unlink 'test.xxx.1','test.xxx.2','test.xxx.3','test.txt';
}
done_testing;
