package App::Countdown;
$App::Countdown::VERSION = '0.8.3';
use 5.010;

use strict;
use warnings FATAL => 'all';

use DateTime::Format::Natural ();
use Time::HiRes qw(sleep time);
use POSIX qw();
use IO::Handle;
use Getopt::Long qw(2.36 GetOptionsFromArray);
use Pod::Usage;
use Carp;


sub new
{
    my $class = shift;

    my $self = bless {}, $class;

    $self->_init(@_);

    return $self;
}

sub _delay
{
    my $self = shift;

    if (@_)
    {
        $self->{_delay} = shift;
    }

    return $self->{_delay};
}

sub _end
{
    my $self = shift;

    if (@_)
    {
        $self->{_end} = shift;
    }

    return $self->{_end};
}

my $up_to_60_re = qr/[1-9]|[1-5][0-9]|0[0-9]?/;

sub _get_up_to_60_val
{
    my ($v) = @_;

    ( $v //= '' ) =~ s/\A0*//;

    return ( length($v) ? $v : 0 );
}

sub _calc_delay
{
    my ( $self, $delay_spec ) = @_;

    if ( my ( $n, $qualifier ) =
        $delay_spec =~ /\A((?:[1-9][0-9]*(?:\.\d*)?)|(?:0\.\d+))([mhs]?)\z/ )
    {
        return int(
            $n * (
                  $qualifier eq 'h' ? ( 60 * 60 )
                : $qualifier eq 'm' ? 60
                :                     1
            )
        );
    }
    elsif ( my ( $min, $sec ) =
        $delay_spec =~ /\A([1-9][0-9]*)m($up_to_60_re)s\z/ )
    {
        return $min * 60 + _get_up_to_60_val($sec);
    }
    elsif ( ( ( my $hour ), $min, $sec ) =
        $delay_spec =~
        /\A([1-9][0-9]*)h(?:($up_to_60_re)m)?(?:($up_to_60_re)s)?\z/ )
    {
        return ( ( $hour * 60 + _get_up_to_60_val($min) ) * 60 +
                _get_up_to_60_val($sec) );
    }
    else
    {
        die
"Invalid delay. Must be a positive and possibly fractional number, possibly followed by s, m, or h";
    }
}

sub _init
{
    my ( $self, $args ) = @_;

    my $argv = [ @{ $args->{argv} } ];

    my $help    = 0;
    my $man     = 0;
    my $version = 0;
    my $end_str;
    if (
        !(
            my $ret = GetOptionsFromArray(
                $argv,
                'help|h' => \$help,
                man      => \$man,
                version  => \$version,
                'to=s'   => \$end_str,
            )
        )
        )
    {
        die "GetOptions failed!";
    }

    if ($help)
    {
        pod2usage(1);
    }

    if ($man)
    {
        pod2usage( -verbose => 2 );
    }

    if ($version)
    {
        print "countdown version $App::Countdown::VERSION .\n";
        exit(0);
    }

    if ( defined $end_str )
    {
        my $parser = DateTime::Format::Natural->new(
            prefer_future => 1,
            time_zone     => 'local',
        );
        my $dt = $parser->parse_datetime($end_str);
        if ( not $parser->success )
        {
            die $parser->error;
        }
        $self->_end( $dt->epoch );
    }
    else
    {
        my $delay = shift(@$argv);

        if ( !defined $delay )
        {
            Carp::confess("You should pass a number of seconds.");
        }

        $self->_delay( $self->_calc_delay($delay) );
    }
    return;
}

sub _format
{
    my $delay = shift;
    return sprintf( "%d:%02d:%02d",
        POSIX::floor( $delay / 3600 ),
        POSIX::floor( $delay / 60 ) % 60,
        $delay % 60 );
}

sub _calc_end
{
    my ( $self, $start ) = @_;

    return defined( $self->_end ) ? $self->_end : ( $start + $self->_delay );
}

sub run
{
    my ($self) = @_;

    STDOUT->autoflush(1);

    my $start = time();
    my $end   = $self->_calc_end($start);

    my $delay = $end - $start;

    my $hms_tot = _format($delay);
    my $last_printed;
    while ( ( my $t = time() ) < $end )
    {
        my $new_to_print = POSIX::floor( $end - $t );
        if ( !defined($last_printed) or $new_to_print != $last_printed )
        {
            $last_printed = $new_to_print;
            my $hms = _format($new_to_print);
            print "Remaining $hms / $hms_tot ( $new_to_print/$delay )",
                ' ' x 10, "\r";
        }
        sleep(0.1);
    }

    return;
}

1;


1;    # End of App::Countdown

__END__

=pod

=encoding UTF-8

=head1 NAME

App::Countdown - wait some specified time while displaying the remaining time.

=head1 VERSION

version 0.8.3

=head1 SYNOPSIS

    use App::Countdown ();

    App::Countdown->new({ argv => [@ARGV] })->run();

=head1 DESCRIPTION

B<countdown> waits for a certain time to pass, in a similar fashion to the
UNIX sleep command, but unlike sleep, it displays the amount of time left to
sleep. I always found it frustrating that I've placed an alarm using
C<sleep $secs ; finish-client> and could not tell how much time left, so I
wrote B<countdown> for that.

=head1 USAGE

    countdown [number of seconds]
    countdown [minutes]m
    countdown [hours]h
    countdown [seconds]s
    countdown [minutes]m[seconds]s
    countdown [hours]h[minutes]m[seconds]s
    countdown --to "20:30"

=head1 SUBROUTINES/METHODS

=head2 new

A constructor. Accepts the argv named arguments.

=head2 run

Runs the program.

=head1 OPTIONS

    --man - displays the man page.
    --help - displays the help.
    --version - displays the version.

=head1 EXAMPLES

    $ countdown 30s # 30 seconds

    $ countdown 1m  # 1 minute

    $ countdown 100 # 100 seconds

    $ countdown 2h  # 2 hours

    $ countdown 2m30s # 2 minutes and 30 seconds.

    $ countdown 1h0m30s # 1 hour, 0 minutes and 30 seconds.

=head1 AUTHOR

Shlomi Fish, L<http://www.shlomifish.org/>, C<< <shlomif at cpan.org> >> .

=head1 ACKNOWLEDGEMENTS

=over 4

=item * Neil Bowers

Reporting a typo and a problem with the description not fitting on one line.

=back

=for :stopwords cpan testmatrix url bugtracker rt cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 SUPPORT

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<https://metacpan.org/release/App-Countdown>

=item *

RT: CPAN's Bug Tracker

The RT ( Request Tracker ) website is the default bug/issue tracking system for CPAN.

L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-Countdown>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/App-Countdown>

=item *

CPAN Testers

The CPAN Testers is a network of smoke testers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/A/App-Countdown>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=App-Countdown>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=App::Countdown>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests by email to C<bug-app-countdown at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/Public/Bug/Report.html?Queue=App-Countdown>. You will be automatically notified of any
progress on the request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/shlomif/app-countdown>

  git clone https://github.com/shlomif/App-Countdown

=head1 AUTHOR

Shlomi Fish <shlomif@cpan.org>

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
L<https://github.com/shlomif/app-countdown/issues>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2020 by Shlomi Fish.

This is free software, licensed under:

  The MIT (X11) License

=cut
