package WebService::ZombiesRun;
use v5.010;
use strict;
use warnings;
use autodie;
use Moo;
use JSON::Any;
use WWW::Mechanize;
use Data::Dumper;

our $DEBUG = 0;

use constant API_BASE => 'https://www.zombiesrungame.com/api/v3';

# ABSTRACT: Access run information from the Zombies, Run! game

our $VERSION = '0.01'; # VERSION: Generated by DZP::OurPkg:Version

has player => (is => 'rw');
has agent  => (is => 'lazy');
has json   => (is => 'lazy');

sub _build_agent { return WWW::Mechanize->new; }
sub _build_json  { return JSON::Any->new;      }


sub runs_raw {
    my ($self) = @_;

    return $self->_fetch_user_path("manifest")->{objects};
}


sub total_runs { my ($self) = @_; return $self->meta_raw->{total_count}; }


sub meta_raw { my ($self) = @_; return $self->_fetch_user_path("manifest")->{meta}; }

sub _fetch_user_path {
    my ($self, $path) = @_;

    my $user = $self->player;

    my $call = join("/", API_BASE, "user/$user", $path, "?format=json");

    warn "Calling $call\n" if $DEBUG;
    $self->agent->get($call);

    return $self->json->decode( $self->agent->content );
}


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

WebService::ZombiesRun - Access run information from the Zombies, Run! game

=head1 VERSION

version 0.01

=head1 SYNOPSIS

    use WebService::ZombiesRun;

    my $zombies = WebService::ZombiesRun->new( player => 'pjf' );

    my $runs = $zombies->runs_raw;

    foreach my $run (@$runs) {
       say "Ran $run->{distance} metres, burning $run->{energy} calories";
    }

=head1 DESCRIPTION

This module provides a thin interface that crawls the Zombies, Run!
website for your running stats.

As a thin interface, any method ending with C<_raw> is simply returning the
JSON as used internally by the Zombies, Run! website.
These structures may change.

=head1 METHODS

=head2 runs_raw

    my $runs = $zombies->runs_raw;

    foreach my $run (@$runs) {
       say "Ran $run->{distance} metres, burning $run->{energy} calories";
    }

Returns an array reference of run records. These are returned directly
translated from the JSON provided by the underlying server.

=head2 total_runs

    my $runs = $zombies->total_runs;

Returns the total runs completed by the player.

=head2 meta_raw

    my $metadata  = $zombies->meta_raw;

Returns the contents of the 'meta' data in the manfiest. At the time
of writing, this is returned as a hash in the following form, which is
directly interpreted from the JSON returned by the server:

    {
        'total_count' => 12,
        'offset'      => 0,
        'limit'       => 1000,
        'previous'    => undef,
        'next'        => undef
    }

=head1 SEE ALSO

=over

=item L<zombiesrun>

Cmdline tool for retrieving information. Also a great proof of concept
for using this module.

=item L<Zombies, Run!|https://www.zombiesrungame.com/>

The Zombies, Run! game.

=back

=for Pod::Coverage BUILD DEMOLISH

=head1 AUTHOR

Paul Fenwick <pjf@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Paul Fenwick.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
