package Class::Unload;
# ABSTRACT: Unload a class
$Class::Unload::VERSION = '0.11_02'; # TRIAL

$Class::Unload::VERSION = '0.1102';use warnings;
use strict;
no strict 'refs'; # we're fiddling with the symbol table

use Class::Inspector;
BEGIN { eval "use Hash::Util"; } # since 5.8.0


sub unload {
    my ($self, $class) = @_;

    return unless Class::Inspector->loaded( $class );

    if ($class =~ /\A(main|CORE|Internals|utf8|UNIVERSAL|PerlIO|re)\z/) {
        require Carp;
        Carp::carp("Cannot unload $class");
        return;
    }

    my $symtab = $class.'::';
    my ($was_locked, $was_readonly);
    if (defined $Hash::Util::VERSION) {
        if (Hash::Util::hash_locked %{$symtab}) {
            Hash::Util::unlock_hash %{$symtab};
            $was_locked++;
        }
    }
    elsif (Internals::SvREADONLY(%{$symtab})) {
        Internals::SvREADONLY(%{$symtab}, 0);
        $was_readonly++;
    }

    # Flush inheritance caches
    if (Internals::SvREADONLY(@{"$class\::ISA"})) {
        Internals::SvREADONLY(@{"$class\::ISA"}, 0);
    }
    @{$class . '::ISA'} = ();

    # Delete all symbols except other namespaces
    for my $symbol (keys %$symtab) {
        next if $symbol =~ /\A[^:]+::\z/;
        delete $symtab->{$symbol};
    }

    # Policy: could be restricted further, but perl5 cannot properly handle
    # restricted stashes yet. Avoid AUTOLOAD/DESTROY surprises and keep em unlocked.
    #if ($was_locked) {
    #    Hash::Util::lock_hash %{$symtab};
    #}
    #elsif ($was_readonly) {
    #    Internals::SvREADONLY(%{$symtab}, 1);
    #}

    my $inc_file = join( '/', split /(?:'|::)/, $class ) . '.pm';
    delete $INC{ $inc_file };

    if (Class::Inspector->loaded('Class::MOP')) {
        Class::MOP::remove_metaclass_by_name($class);
    }

    return 1;
}


1; # End of Class::Unload

__END__

=pod

=encoding UTF-8

=head1 NAME

Class::Unload - Unload a class

=head1 VERSION

version 0.11_02

=head1 SYNOPSIS

    use Class::Unload;
    use Class::Inspector;

    use Some::Class;

    Class::Unload->unload( 'Some::Class' );
    Class::Inspector->loaded( 'Some::Class' ); # Returns false

    require Some::Class; # Reloads the class

=head1 METHODS

=head2 unload $class

Unloads the given class by clearing out its symbol table and removing it
from %INC.  If it's a L<Moose> class, the metaclass is also removed.

Avoids unloading internal core classes, like main, CORE, Internals,
utf8, UNIVERSAL, PerlIO, re.

Handles restricted class (protected stashes) and ISA's.

=head1 SEE ALSO

L<Class::Inspector>

=head1 ACKNOWLEDGEMENTS

Thanks to Matt S. Trout, James Mastros and Uri Guttman for various tips
and pointers.

=head1 AUTHOR

Dagfinn Ilmari Mannsåker <ilmari@ilmari.org>;

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by Dagfinn Ilmari Mannsåker.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
