#! /usr/bin/env false

use v6.c;

use Hash::Merge;
use JSON::Fast;

unit module Dist::Helper::Meta;

#| Read a dist's C<META6.json> file, and return it interpreted as a hash. If
#| C<:$add-missing> is set, it well be merged into a complete, empty META6 hash,
#| ensuring there are no missing keys.
sub get-meta (
	Str:D $path = ".",
	Bool:D :$add-missing = False,
	--> Hash
) is export {
	my $meta6-file = $path.IO.add("META6.json");

	if (! $meta6-file.e) {
		die "No META6.json in {$path.IO.absolute}";
	}

	my %meta = from-json(slurp($meta6-file.path));

	return merge-hash(new-meta, %meta, :no-append-array) if $add-missing;

	%meta;
}

#| Get the base structure of a new META6 hash. This can then be used to create a
#| new C<META6.json> without missing keys.
sub new-meta (
	--> Hash
) is export {
	%(
		meta-version => 0,
		perl => "6.c",
		name => "",
		description => "",
		license => "",
		authors => [],
		tags => [],
		api => "*",
		version => "*",
		provides => %(),
		depends => [],
		resources => [],
	);
}

#| Save a dist's META6 hash to C<META6.json> in the given C<:$path>.
sub put-meta (
	:%meta!,
	:$path = ".",
	:$clobber = True
) is export {
	my $meta6 = $path.IO.add("META6.json").absolute;

	if ($meta6.IO.e && !$clobber) {
		die "Not clobbering {$meta6}";
	}

	# Manually sort arrays used in META6
	my @sortable-arrays = (
		"depends",
		"resources",
		"tags",
	);

	for @sortable-arrays -> $array {
		next if %meta{$array}:!exists;
		next if %meta{$array}.elems < 1;

		%meta{$array} = %meta{$array}.sort;
	}

	spurt($meta6, to-json(%meta, :sorted-keys))
}
